/* Josh Pieper, (c) 2000
   Robert Munafo, (c) 2001

   This file is distributed under the GPL, see file COPYING for details */


#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <pthread.h>
#include <ctype.h>
#ifndef WIN32
#include <unistd.h>
#else
#include <io.h>
#endif

#ifdef MCHECK
#include <mcheck.h>
#endif

#include "conf.h"
#include "lib.h"
#include "threads.h"

/* ----------------------------- realtime debugging ----------------------- */

/* Debug queue, really a circular array. I record numbers into this
 * array for debugging purposes, and dump it when there is a segfault. */
uint16 dq[DQ_SIZE];
uint16 dq_copy[DQ_SIZE];  /* This is for quickly copying the array
                           * to minimize overwrite by other threads
                           * when we are ready to dump */
int32 dqptr;
int32 dq_dumped;

uint16 dqt[DQT_CHUNK_SIZE * DQT_CHUNK_NUM];
int32 dqtp[DQT_CHUNK_NUM];
uint16 dqtt;

int32 dq_nth;

/* dq_init is called once when the program starts up */
void dq_init(void)
{
  int32 i;

  for(i=0; i<DQ_SIZE; i++) {
    dq[i] = 0;
  }
  for(i=0; i<DQT_CHUNK_SIZE * DQT_CHUNK_NUM; i++) {
    dqt[i] = 0;
  }
  for(i=0; i<DQT_CHUNK_NUM; i++) {
    dqtp[i] = 0;
  }
  dqptr = 0;
  dq_dumped = 0;
  dq_nth = 0;
}

void dq_hi_init(void)
{
  if(conf_get_int("auto_remove_segv_info")) {
    unlink("gnut-segfault-info");
  }
}

/* dq.start is called by each thread when it starts up */
void dq_start(char *thtype, uint16 dqinum)
{
  uint16 i;

  dq_nth++;
  /* printf("Thread %s %3i %i\n", thtype, DQ_TID, dq_nth); */

  for(i=0; i<DQT_CHUNK_SIZE; i++) {
    dqt[(DQ_TID * DQT_CHUNK_SIZE) + i] = 0;
  }

  dqi(dqinum);
}

void dql(uint32 x)
{
  uint32 y;

  y = (x >> 24) & 0x0fff;
  dqi(0x8000 | y);
  y = (x >> 12) & 0x0fff;
  dqi(0x8000 | y);
  y = x & 0x0fff;
  dqi(0x8000 | y);
}

void dqp(void *p)
{
  dql((uint32) p);
}

/* dq_end is called by a thread when it wants to exit normally */
void dq_end(void * retval)
{
  dq_nth--;
  /* printf(" exit     %3i %i\n", DQ_TID, dq_nth); */

  pthread_exit(retval);
}

#define DQ_DUMP_MAX 4

void dqd_dump1(FILE *dumpfile, uint16 bufsize, uint16 *buf, uint16 startloc)
{
  int32 i, j, k;

  k = 0; j = startloc;
  for(i=0; i<bufsize; i++) {
    fprintf(dumpfile, "%4hx", buf[j]);
    k++;
    if (k >= 16) {
      fprintf(dumpfile, "\n");
      k = 0;
    } else {
      fprintf(dumpfile, " ");
    }

    j++;
    if (j >= bufsize) {
      j = 0;
    }
  }
  if (k > 0) {
    fprintf(dumpfile, "\n");
  }
}

void dq_dump(char *msg)
{
  int32 i, j;
  uint16 *a;
  uint16 *b;
  FILE *dumpfile;

  if (dq_dumped >= DQ_DUMP_MAX) {
    return;
  }

  /* Flash-copy the global debug queue */
  j = dqptr;
  a = dq; b = dq_copy;
  for(i=0; i<DQ_SIZE; i++) {
    *b++ = *a++;
  }

  if (dq_dumped == 0) {
    /* First time */
    dumpfile = fopen("gnut-segfault-info", "w");
  } else {
    dumpfile = fopen("gnut-segfault-info", "a");
  }
  /* Dump the global debug queue */
  dqd_dump1(dumpfile, DQ_SIZE, dq_copy, j);
  fprintf(dumpfile, " - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -\n");

  /* Dump the thread-specific debug queue */
  dqd_dump1(dumpfile, DQT_CHUNK_SIZE, dqt + (DQ_TID * DQT_CHUNK_SIZE),
	    dqtp[DQ_TID]);
  fprintf(dumpfile, "thread %i %s\n", DQ_TID, msg);
  fprintf(dumpfile, " -----------------------------------------------------------------------------\n");

  fclose(dumpfile);

  dq_dumped++;
}

/* ----------------------------- memory management and leak detection ------ */

int _gnut_mprobe(void *d)
{
#ifdef MCHECK
  return mprobe(d);
#else
  return 0;
#endif
}

#define LEAKIDX 700
long leakers[LEAKIDX];
long leak_d[LEAKIDX];
long reset_count;
long reset_thres;

/* Init the tables used to keep track of leaks */
void init_leaks()
{
  int i;

  for(i=0; i<LEAKIDX; i++) {
    leakers[i] = 0;
    leak_d[i] = 0;
  }
  leakers[1] = 1;
  reset_count = 0;
  reset_thres = 100;
}

/* Reset delta counts for leak detection */
void reset_leaks()
{
  int i;

  for(i=0; i<LEAKIDX; i++) {
    leak_d[i] = 0;
  }
}

/* Find and report the memory-block-type that is currently believed to be
   the biggest suspect for leaking */
void tell_leaks()
{
  int i;
  int max, maxi, maxd;

  max = 0; maxi = 0; maxd = 0;
  for(i=0; i<LEAKIDX; i++) {
    if (leak_d[i] > maxd) {
      maxd = leak_d[i];
      max = leakers[i];
      maxi = i;
    }
  }
  printf("yma""loc(..., %d) == %d, delta %d\n", maxi, max, maxd);
}

/* Every time a block is allocated via yma.loc, this routine is called.
   It resets the delta counts periodically but with an exponentially
   lenghthening frequency. The effect of this is that the delta count
   will measure allocations that have taken place during the last 5%
   (approximately) of the history of this invocation of gnut. That's
   the most useful way to catch leaks, because it filters out short-term
   spikes as well at the big initial allocations associated with startup.
 */
void check_reset()
{
  reset_count++;
  if (reset_count > reset_thres) {
    reset_count = 0;
    reset_thres += (reset_thres / 20);
    reset_leaks();
  }
}

#if YTAGS
typedef struct {
  uint32 tn;
  uint32 usize;
  uint32 canarylo;
} ymheader;

typedef struct {
  uint32 canaryhi;
} ymfooter;

/* yprobe checks for overwriting past either end of a block. This is done
   using "canaries", which are 4-byte blocks placed just past either end of
   the block and filled with known values so that it can be detected if they
   are overwritten. Returns 1 if the block is OK, 0 otherwise.

   For efficiency, don't call yprobe directly, use the YPROBE macro instead.
   When YTAGS is 0 the macro is just a constant "1". */
int yprobe(void *a, int tracking_number)
{
  uint32 usiz;
  ymheader * p2;
  ymfooter * p3;
  int tn1;

  if (a) {
    dqi(0x017f);
    /* Calculate positions of end tag blocks */
    p2 = (ymheader *) (((uint8 *) (a)) - sizeof(ymheader));
    usiz = p2->usize;
    p3 = (ymfooter *) (((uint8 *) (a)) + usiz);

    /* Check canaries */
    tn1 = p2->tn;
    if (p2->canarylo != (tn1 ^ 0x55555555)) {
      printf("Wrote before beg of block alloc TN %d dealloc TN %d\n",
	     p3->canaryhi ^ 0xaaaaaaaa, tracking_number);
      dqi(0x0180);
      return 0;
    } else if (p3->canaryhi != (tn1 ^ 0xaaaaaaaa)) {
      printf("Wrote past end of block alloc TN %d dealloc TN %d\n",
	     tn1, tracking_number);
      dqi(0x0181);
      return 0;
    } else {
      dqi(0x0182);
      return 1;
    }
  }
  dqi(0x0183);
  return 0;
}
#endif

/* Leak-detection version of free(3) */
void yfre(void **a, int tracking_number)
{
#if YTAGS
  ymheader * p2;
  int tn1;
  int ypr;
#endif

  gd_s(5, "yfre a="); gd_p(5, a); gd_s(5, " *a="); gd_p(5, *a); gd_s(5, "\n");
  if (*a == 0) {
    if (tracking_number) {
      printf("Double-fr""ee error, case %d\n", tracking_number);
      printf("Please email gnut@mrob.com and include 'case %d' in subject\n",
	     tracking_number);
    }
    return;
  }

#if YTAGS
  ypr = YPROBE(*a, tracking_number)
  if (ypr) {
    p2 = (ymheader *) (((uint8 *) (*a)) - sizeof(ymheader));
    tn1 = p2->tn;
    leakers[tn1]--;
    leak_d[tn1]--;
    free((void *) p2);
  }

  /*  printf("yfre %d source %d\n", tracking_number, tn); */
#else
  if (*a) {
    free(*a);
  }
#endif

  *a=0;
}

void fre_v(void **x, int bugnum)
{
  yfre(x, bugnum);
}

void fre_strl(char ***x, int bugnum)
{
  yfre((void **) x, bugnum);
}

void fre_str(char **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

/* Leak-detection version of malloc(3) */
void *ymaloc(int size, int tracking_number)
{
  uint8 * ptr;
#if YTAGS
  ymheader * p2;
  ymfooter * p3;
  int tsiz;
#endif

#if YTAGS
  tsiz = sizeof(ymheader) + sizeof(ymfooter);
  ptr = (uint8 *) malloc(size + tsiz);
#else
  ptr = (uint8 *) malloc(size);
#endif

  if (!ptr) {
    if (tracking_number) {
      printf("Failed al""loc case %d (%d bytes)\n", tracking_number, size);
      printf("Please email gnut@mrob.com and include 'case %d' in subject\n",
	     tracking_number);
    } else {
      printf("Failed al""loc %d bytes\n", size);
    }
  }

#if YTAGS
  p2 = (ymheader *) ptr;
  ptr += sizeof(ymheader);
  p3 = (ymfooter *) (ptr + size);

  p2->tn = tracking_number;
  p2->usize = size;
  p2->canarylo = (tracking_number ^ 0x55555555);

  p3->canaryhi = (tracking_number ^ 0xaaaaaaaa);
#endif

  leakers[tracking_number]++;
  leak_d[tracking_number]++;

  check_reset();

  /*  printf("ymaloc src %d size %d\n", tracking_number, size + tsiz); */

  return ptr;
}

/* Leak-detection version of strdup(3) */
char *ystdup(const char *s, int tracking_number)
{
#if YTAGS
  char * rv;
  int len;

  len = strlen(s) + 1;

  rv = (char *) ymaloc(len, tracking_number);

  memcpy(rv, s, len);

  return rv;
#else
  return(strdup(s));
#endif
}

/* Leak-detection version of calloc(3) */
void * ycaloc(size_t nmemb, size_t size, int tracking_number)
{
#if YTAGS
  size_t ts;
  char * rv;
  int i;

  ts = nmemb * size;
  rv = ymaloc(ts, tracking_number);

  for(i=0; i<ts; i++) {
    rv[i] = 0;
  }

  return rv;
#else
  return calloc(nmemb, size);
#endif
}

#ifdef WIN32
char *path_separator = ";";
char *path_slash = "\\";
#else
char *path_separator = ":";
char *path_slash = "/";
#endif

/* -------------------- character type, case mapping, 8th bit, etc. ------- */

/* isblank is a GNU library extension but I can only use POSIX and ANSI C */
int g_isblank(char c) /* 0.4.27.c06 */
{
  return((c == ' ') || (c == '\t'));
}

/* ----------------------------- varstring -------------------------------- */

void fre_vs(varstring **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

varstring * vs_new(void)
{
  varstring * rv;

  rv = ymaloc(sizeof(varstring), 541);
  rv->vs_alloc = 10;
  rv->vs_data = ymaloc(rv->vs_alloc, 542);
  rv->vs_used = 0;

  return rv;
}

void vs_kill(varstring *vs)
{
  if(vs) {
    if(vs->vs_data) {
      fre_str(&(vs->vs_data), 543);
    }
    fre_vs(&vs, 544);
  }
}

/* ----------------------------- CRC -------------------------------------- */
/* CRC-16

The constant 0x1021 is based on the X25 standard CRC polynomial
x^16 + x^12 + x^5 + 1. The highest bit (x616) doesn't need to be
included; the other three bits (12, 5, 1) form the value 0x1021.

The following illustration is from http://www.io.com/~ritter/ARTS/CRCMYST.HTM

       Polynomial = x^5 + x^4 + x^2 + 1 = 110101

       dn   x^4         x^3     x^2         x^1     x^0
       v   +---+       +---+   +---+       +---+   +---+
   +<-XOR<-|Q D|<-XOR<-|Q D|<--|Q D|<-XOR<-|Q D|<--|Q D|<-+
   |       |   |   ^   |   |   |   |   ^   |   |   |   |  |
   |       +---+   |   +---+   +---+   |   +---+   +---+  |
   |               |                   |                  |
   +---------------+-------------------+------------------+

   FIGURE 2.  CRC Hardware

crc = 0xffff;
for each data bit {
  if (high bit of CRC ^ new data bit == 1) {
    crc = (crc << 1) ^ 0x1021;
  } else {
    crc = crc << 1;
  }
}

for CRC-32 (used in e.g. Ethernet) we use 0x04c11db7L, or in binary:
1 0000 0100 1100 0001 0001 1101 1011 0111

There is a long explanation at http://www.snippets.org/CRC.TXT

See also http://cell-relay.indiana.edu/cell-relay/publications/software/CRC/32bitCRC.c.html

If any of these URLs don't work, contact the author Robert Munafo
(www.mrob.com)

*/

#define CRC32_MASK 0x04C11DB7L;

uint32 crc32_conv_8[256];
int crc32_init = 0;

void crc32_start(uint32 *crc)
{
  int i, j;
  uint32 cctemp;
  
  *crc = 0xffffffffL;
  if (crc32_init == 0) {
    for(i=0; i<256; i++) {
      cctemp = ((uint32) i) << 24L;
      for(j=0; j<8; j++) {
	if (cctemp & 0x80000000L) {
	  cctemp = (cctemp << 1L) ^ CRC32_MASK;
	} else {
	  cctemp <<= 1L;
	}
      }
      crc32_conv_8[i] = cctemp;
    }
    crc32_init = 1;
  }
}

void crc32_add8(uint32 *crc, uchar data)
{
  uchar index;
  
  index = (*crc >> 24L) ^ data;
  *crc = (*crc << 8L) ^ crc32_conv_8[index];
}

uint32 crc32_string(char *data)
{
  uint32 rv;
  char *s;

  crc32_start(&rv);
  s = data;
  while(*s) {
    crc32_add8(&rv, *s);
    s++;
  }
  return rv;
}

/* CRC an arbitrary block of memory */
uint32 crc32_block(void * data, uint32 len)
{
  uint32 crc;
  uint32 i;
  uchar *k;

  /* Generate 32-bit CRC from key */
  crc32_start(&crc);
  k = (uchar *) data;
  for(i=0; i<len; i++) {
    crc32_add8(&crc, k[i]);
  }

  return crc;
}

/* ----------------------------- OS Interface ----------------------------- */
/* --- This section also includes anything that should be in the standard
   --- libraries, but isn't                                                 */

#ifndef PTHREADS_DRAFT4
  pthread_mutex_t _g_debug_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t mutex_mutex = PTHREAD_MUTEX_INITIALIZER;
#else
  pthread_mutex_t _g_debug_mutex;
  pthread_mutex_t mutex_mutex;
#endif

/* Because of unsolved bugs in gnut, I need to be able to clean up after
   dead threads, including possibly clearing a mutex that is locked by a
   dead thread. Since pthreads doesn't guarantee that a mutex can be cleared
   by another thread, I have to implement mutexes myself! */
int pmutex_lock(int *m)
{
  int rv;
  uint32 crc;
  uchar j;

  rv = 0;
  crc = ((uint32) pthread_self());
  j = 0;
  while(rv == 0) {
    int t;

    pthread_mutex_lock(&mutex_mutex);
    t = *m; *m = 1; /* test-and-set */
    pthread_mutex_unlock(&mutex_mutex);
    if (t == 0) {
      rv = 1;
    } else {
      int i, l;

      /* random holdoff */
      l = crc & 0xfff;
      for(i=0; i<l; i++) {
        crc32_add8(&crc, j);
        j++;
      }
    }
  }
  return rv;
}

char *expand_path(char *a)
{
  char *b;
  if (strncmp(a,"~/", 2)==0) {
#ifndef WIN32
    char *c;
    c = getenv("HOME");
#else
    char c[MAX_PATH];
	/*    GetWindowsDirectory(c, MAX_PATH); */
    /* I did this because too many people couldn't find
     * the .gnut* files under win32  -jp */
    c[0] = 0;  
#endif
    b = (char *)ymaloc(strlen(a)+strlen(c)+3, 321);
    b[0] = 0;
    if (strlen(c) > 1) {
      strcpy(b, c);
      strcat(b, path_slash);
    }
    strcat(b, &a[2]);
  } else {
    b=(char *) ymaloc(strlen(a)+3, 322);
    strcpy(b,a);
  }

  if (strlen(b) > 1) {
    if (b[strlen(b)-1] == path_slash[0]) {
      b[strlen(b)-1]=0;
    }
  }

  return b;   
}                                               

/* Counts the alphanumeric characters in a string. */
int strlen_an(char *s)
{
  int rv;

  for(rv=0; *s; s++) {
    if (isalnum(*s)) { /* 0.4.27.c07 */
      rv++;
    }
  }
  return rv;
}

/* Prints a string, but changes any nonprintable characters to '#' */
void print_ascii(uchar *s)
{
  while(*s) {
    if (isprint(*s)) {
      putchar(*s);
    } else {
      putchar('#');
    }
    s++;
  }
}

/* Like print_ascii, but includes a length limit option */
void print_asc_n(uchar *s, int len)
{
  int i;

  for(i=0; i<len; i++) {
    if (isprint(s[i])) {
      putchar(s[i]);
    } else {
      putchar('#');
    }
  }
}

/* Write a string to a socket */
int writestr(int sock, char *str)
{
  int rv;

  rv = write(sock, str, strlen(str));
  return(rv);
}

/* strcatlim is meant to be a replacement for strncat, which is not available
   on all machines (as several users have told me through email) */
void strcatlim(char *dest, char *src, int maxlen)
{
  int l1, l2, i;
  char *s;

  l1 = strlen(dest);
  l2 = strlen(src);
  if (l1 >= maxlen) {
    /* Don't catenate anything */
    return;
  } else if (l1 + l2 < maxlen) {
    /* All set */
  } else {
    /* Enforce l1 + l2 < maxlen */
    l2 = maxlen - l1 - 1;
  }

  s = dest + l1;
  for(i=0; i<l2; i++) {
    s[i] = src[i];
  }
  s[i] = 0;
}

/* Converts a string to lowercase */
void make_lc(char *str)
{
  char *s;

  for(s=str; *s; s++) {
    *s = tolower(*s); /* 0.4.27.c07 */
  }
}

/* Converts a string to all 7-bit characters (I use this because I use the
   high bits as flags while doing boolean AND matches) */
void make_7bit(char *str)
{
  uint8 *s;

  for(s = (uint8 *) str; *s; s++) {
    if (*s > 127) { /* tagok */
      /* %%% Eventually we want to implement a mapping from the 256. ISO
         8-bit characters onto a 7-bit set, so that diacriticals don't
         get lost. */
      *s = ' ';
    }
  }
}

/* 0.4.28.c15 */
void make_htmlsafe(char *str)
{
  char *s;

  for(s=str; *s; s++) {
    if (*s == '<') {
      *s = '(';
    } else if (*s == '>') {
      *s = ')';
    } else if (*s == '"') {
      *s = '\'';
    }
  }
}

/* 0.4.27.c14 */
/* Like strncmp, but also has an ignore-case option */
int gl_strncmp(char *a, char* b, int len, int igcase)
{
  int rv;
  uchar *pa;
  uchar *pb;
  uchar ca, cb;
  int i;

  pa = a; pb = b;

  rv = 0;
  if(pa && pb) {
    for(i=0; (i<len) && (rv == 0); i++) {
      ca = *pa; cb = *pb;
      if (igcase) {
	ca = tolower(ca);
	cb = tolower(cb);
      }
      if (ca == 0) {
        rv = -1;
      } else if (cb == 0) {
	rv = 1;
      } else if (ca > cb) {
	rv = 1;
      } else if (cb > ca) {
	rv = -1;
      }
      pa++; pb++;
    }
  }
  return rv;
}

/* 0.4.27.c11 */
/* Convert a string to a 32-bit UNSIGNED quantity, like strtol. atol is
   similar, but the manpage doesn't say whether it handles bith signed
   and unsigned (it could if they wrote it right). I don't want to take
   any chances with the library routines.

   The parameter p is optional (pass 0 if you don't want it), and will get
   set to the first character after the last digit, unless no digits were
   found. */
uint32 gl_stou32(char *s, char **p)
{
  char *s1;
  uint32 rv;
  int gotone;

  rv = 0;
  s1 = s;
  gotone = 0;

  while(isspace(*s)) {
    s++;
  }
  if (*s == '+') {
    s++;
  }
  while(isdigit(*s)) {
    uint32 d;

    gotone = 1;
    d = (*s - '0');
    rv = (rv * 10U) + d;
    s++;
  }
  if (p) {
    if (gotone) {
      *p = s;
    } else {
      *p = s1;
    }
  }
  return rv;
}

/* ----------------------------- debug messages and logging --------------- */

int gnut_lib_debug=0;

/* These gd_xxx routines are for debug printf's. There is one for each
   type of thing you might want to print, like strings, hex bytes,
   integers, etc. Originally printf's with multiple arguments were
   used, but I found some bugs in Josh's gnut that resulted from
   failure to pass the right parameters on the stack. I eliminated
   those bugs by removing all use of var_args routines including
   printf */
void gd_s(int level, char * str)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%s", str);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_02x(int level, uchar c)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%02x", c);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_i(int level, int32 x)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%i", x);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

/* 0.4.27.c11 Note, I did not flag the calls to gd_i that got changed to gd_u,
   you can find them just by searching the code for "gd_u" */
void gd_u(int level, uint32 x)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%u", x);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_x(int level, int32 x)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%x", x);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_li(int level, int32 x)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%li", (long int) x);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_lu(int level, uint32 x)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%lu", (long unsigned int) x);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_f64(int level, float64 x)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%g", x);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

void gd_p(int level, void *p)
{
  if (gnut_lib_debug >= level) {
    pthread_mutex_lock(&_g_debug_mutex);
#ifndef WIN32
    fflush(stdout);
#endif
    fprintf(stderr, "%p", p);
#ifndef WIN32
    fflush(stderr);
#endif
    pthread_mutex_unlock(&_g_debug_mutex);
  }
}

/* ----------------------------- unclassified ----------------------------- */

char *gnut_strdelimit(char *string, char *delim, char new_delim)
{
  char *c;
  
  for (c=string;*c;c++) {
    if (strchr(delim,*c)) *c=new_delim;
  }
  
  return string;
}

/* remove any trailing whitespace */
void gnut_strstrip(char *string)
{
  int i;

  for (i=(strlen(string)-1);i>0 && isspace(string[i]);i--) {
    string[i]=0;
  }
}

char *munge_time(int secs)
{
  /* these buffers should be 'large enough' */
  char *s;
  char buf[10];
  int days, hours, minutes, seconds;
  int fmt;
  
  s = ymaloc(40*sizeof(char), 323);
  days = hours = minutes = seconds = 0;
  strcpy(s,"");
  
  if (secs==0) {
    strcat(s,"0s");
    return s;
  }
  
  if (secs<0) {
    /* handle negatives correctly */
    strcat(s,"-");
    secs=-secs;
  }
  
  seconds=secs;
  
  if (seconds > 60) {
    /* how many minutes? */
    minutes=seconds / 60;
    seconds=seconds % 60;
  }
  
  if (minutes >= 60) {
    /* hours? */
    hours=minutes / 60;
    minutes = minutes %60;
  }
  
  if (hours >= 24) {
    /* and the days */
    days=hours/24;
    hours=hours % 24;
  }
  
  /* build the string */
  fmt = 0;
  if (days) {
    sprintf(buf, "%dd", days);
    strcat(s, buf);
    fmt = 1;
  }
  if (hours || fmt) {
    sprintf(buf, "%dh", hours);
    strcat(s, buf); 
    fmt = 1;
  }
  if (minutes || fmt) {
    sprintf(buf, "%dm", minutes);
    strcat(s, buf);
    fmt = 1;
  }
  if (seconds || fmt) {
    sprintf(buf, "%ds", seconds);
    strcat(s, buf);
    fmt = 1;
  }
  
  return s;
}

char *format_si(float64 x, char *s)
{
  /* According to GCMP, the prefixes following exa are defined from
   * letters of the Latin alphabet, starting from the last and moving
   * backwards. Since Latin has no J or W, the prefixes after Z and Y will
   * be X, V, and U; after that I don't know what to predict because the
   * letter T is already used for tera. 
   *
   * Refer to: "The Peta-Principle" by Jim Binder, NIST publictaion
   * SP811 "Guide for the Use of the International System of Units (SI)",
   * http://physics.nist.gov/Pubs/SP811/contents.html ,
   * "Etymology of Units" by PC Hariharan */
  static char *si_prefixes = "-kMGTPEZYXVU";
  char *p;
  char *rv;
  float64 k1000;

  k1000 = 1000.0;

  if (s) {
    rv = s;
  } else {
    rv = ymaloc(50*sizeof(char), 324);
  }

  if (x < k1000) {
    sprintf(rv, "%g", x);
    return rv;
  }

  p = si_prefixes;
  while ((x >= ((double) k1000)) && (*p)) {
    x = x / ((double) k1000);
    p++;
  }
  if (*p == 'k') {
    /* we only need one digit after the decimal for kilobytes */
    sprintf(rv, "%.2f%c", x, *p);
  } else {
    sprintf(rv, "%.3f%c", x, *p);
  }

  return(rv);
}

int trycmd(char *cmd, char *pat)
{
  FILE *pipe;
  char inbuf[100];
  int rv;

  gd_s(2, "trycmd '");
  gd_s(2, cmd);
  gd_s(2, "'\n");

  pipe = popen(cmd, "r");
  if (pipe == 0) {
    gd_s(2, "trycmd openpipe failed\n");
    return(0);
  }

  /* Read results of command. NOTE: If command not found, error goes to
   * STDERR and doesn't get read. Anything to STDOUT is taken as proof
   * that the command exists. */
  rv = 0;
  while (fgets(inbuf, 100, pipe)) {
    gd_s(3, "trycmd read '");
    gd_s(3, inbuf);
    gd_s(3, "'\n");
    if(strstr(inbuf, pat)) {
      rv = 1;
    }
  }
  pclose(pipe);
  return rv;
}

/* keyword_match does a boolean match test on a string (s1) using a
 * query string (q1). Both are standard C strings. The third parameter
 * "and" should be 0 for a boolean OR match, and 1 for a boolean AND
 * match. The parameter "ignorecase" should be nonzero if you want
 * it to ignore case. */
int keyword_match(char *q1, char *s1, int and, int ignorecase)
{
  char *a,*b;
  char *q2;
  char *s2;
  char *query;
  char *str;
  int result;
  int going;
  char *loc;

  gd_s(5, "ismatch_tok entering\n");

  q2 = s2 = 0;

  /* Copy query and str so make_lc and strtok doesn't trash them */
  q2 = ystdup(q1, 436); query = q2;
  s2 = ystdup(s1, 437); str = s2;

  if (ignorecase) {
    make_lc(str);
    make_lc(query);
  }

  /* First, break the query string into words ("tokens"). For example,
   * if the query is "john smith .jpg" the tokens will be "john",
   * "smith" and "jpg". */
#ifdef HAVE_STRTOK_R
  a = strtok_r(query, " \t", &b);
#else
  a = strtok(query, " \t");
#endif

  if (a == 0) {
    result = 0;
    goto km_exit1;
  }

  /* Check each token to see if it's in the string */
  if (and) {
    result = 1;
  } else {
    result = 0;
  }
  going = 1;
  while (a && going) {
    gd_s(5, "ismatch_tok checking for \"");
    gd_s(5, a);
    gd_s(5, "\" in \"");
    gd_s(5, str);
    gd_s(5, "\" \n");
    loc = strstr(str, a);
    if (loc == 0) {
      /* Not found */
      if (and) {
        result = 0;
        going = 0;
      }
    } else {
      /* Found at position loc */

      if (and) {
	int i;
        /* "Erase" the matched text, so that two occurrances of the
         * same keyword in query will require two occurrances of that
         * word in the string. Note this affects AND matches only, not
         * OR matches */
        for(i=0; a[i]; i++) {
	  if (loc[i]) {
            loc[i] = ' ';
          }
        }
      } else {
        result = 1;
        going = 0;
      }
    }
#ifdef HAVE_STRTOK_R
    a = strtok_r(0, " \t", &b);
#else
    a = strtok(0, " \t");
#endif
  }

 km_exit1: ;

  if (result == 0) {
    gd_s(5, "ismatch_tok returning failure\n");
  } else {
    gd_s(5, "ismatch_tok returning success\n");
  }

  /* Free up any copies we made */
  if (q2) {
    fre_str(&q2, 157);
  }
  if (s2) {
    fre_str(&s2, 158);
  }

  return result;
}
