/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pthread.h> /* 0.4.27.c16 */

#include "lib.h"
#include "list.h"
#include "queue.h"
#include "hash.h"

/* 0.4.27.c16 */
#ifndef PTHREADS_DRAFT4
 pthread_mutex_t lruh_mutex=PTHREAD_MUTEX_INITIALIZER;
#else
 pthread_mutex_t lruh_mutex;
#endif


void fre_gh(Gnut_Hash **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

void fre_crcp(crcptr **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

/*

host history word:

  8 hex digits (32-bit long int)
  defaults to all 0's
  each time a connection is attempted, shifts to the left then OR 0x08
  each time connection opens successfully, OR 0x04
  each time IP address is seen in a packet, low bit set to 1

     A C K I
     ^ ^ ^ ^
     | | | `- IP address was gleaned from packets (else it's from PONG)
     | | `--- deliberately killed because of dropped packets or transfer rate
     | `----- Our attempted connection was accepted
     `------- We attempted a connection

  mask with 0x66666666 to get just the connect history
  host deleted from database when history is nonzero AND history & 0x66666666 is zero

  connect-candidate score = weighted sum of the C bits: recentmost position =
  2^7 points, prev position = 2^6, and so on. A deliberate kill counts
  as a failed connection when choosing a new host, select highest
  score in database that isn't already in connection list

I want to figure out why my 'sent' stats are so much lower than
'received'. With 8 connections, keep a 8x8 array of stats showing how
much is being routed through each link of the crossbar, so to speak.
Also extra columns to log how many packets got knocked out pre-routing
and for which reasons (TTL 0, TTL bad, etc.)

*/

static int next_hashstart;

/* gnut.hash.foreach calls a callback function on each item in a hash.
   But if the function returns -1, it will stop going thrugh the hash
   immediately. */
int gnut_hash_foreach(Gnut_Hash *gh, int (*a)(void *, void *), void *data)
{
  int i, j, k;

  next_hashstart += (rand() & 0xff);
  j = next_hashstart % 256;
  
  for (i=0;i<256;i++) {
    k = (i + j) % 256;
    if (gnut_list_foreach(gh->list[k], a, data) == -1) {
      /* Iterator told us to stop walking through the list */
      return -1;
    }
  }
  return 0;
}

Gnut_Hash * gnut_hash_new(uchar (*hash_func)(void *),
			  int (*compare_func)(void *, void *))
{
  Gnut_Hash *gh;
  int i=0;

  gh=(Gnut_Hash *) ymaloc(sizeof(Gnut_Hash), 368);
  for (i=0;i<256;i++) {
    gh->list[i]=0;
  }
  gh->hash_func=hash_func;
  gh->compare_func=compare_func;
  
  return gh;
}

/* Returns bucket number of a data item in a hash */
int gnut_hash_bucketnum(Gnut_Hash *gh, void *data)
{
  int rv;
  rv = (*gh->hash_func)(data);
  return(rv);
}

/* Call this just before inserting a data item in the hash to find out how
   big its bucket is */
int gnut_hash_bucketsize(Gnut_Hash *gh, void *data)
{
  int rv;
  int a;

  rv = 0;
  if (gh) {
    a = (*gh->hash_func)(data);
    rv = gnut_list_size(gh->list[a]);
  }

  return rv;
}

/* Given a hash and a data item (which may or may not actually be in the hash)
   this returns the oldest data item in the bucket which the given data item
   belongs in. You can use this along with gnut_hash_bucketsize to limit
   the size of the hash. */
void * gnut_hash_bucketoldest(Gnut_Hash *gh, void *data)
{
  void * rv;
  int a;

  rv = 0;
  if (gh) {
    a = (*gh->hash_func)(data);
    /* hash routines use list.append to insert, that means the oldest item is
       the first item */
    if (gh->list[a]) {
      rv = gnut_list_head(gh->list[a]);
    }
  }

  return rv;
}

void gnut_hash_insert(Gnut_Hash *gh, void *data, int tracking_number)
{
  int a;

  if (!gh) {
    gd_s(0, "gnut_hash_insert gh="); gd_p(0, gh); gd_s(0, "\n data=");
    gd_p(0, data); gd_s(0, "\n");
    return;
  }

  a = (*gh->hash_func)(data);

  gd_s(5, "gnut_hash_insert hash returned: "); gd_i(5, a); gd_s(5, "\n");

  gh->list[a] = gnut_list_append(gh->list[a], data, tracking_number);
}

/* gnut_hash.find_raw searches every item in a hash using the compare
 * function, to find an item that compares equal to the given data.
 * It does not use the hash function to optimize */
void * gnut_hash_find_raw(Gnut_Hash *gh, void *data)
{
  int a;
  int ret;
  Gnut_List *tmp;

  if (!gh) {
    /* gd_s(0,"gh=%p data=%p\n"); */
    gd_s(5,"gnut_hash.find_raw gh == 0\n");  
    return 0;
  }

  for (a = 0; a < 256; a++) {
    gd_s(5, "gnut_hash.find_raw hash returned: "); gd_i(5, a);
    gd_s(5, "\n");

    for (tmp=gh->list[a]; tmp; tmp=tmp->next) {
      ret=(*gh->compare_func)(data,tmp->data);
      if (ret==0) {
	return tmp->data;
      }
    }
  }

  gd_s(5, "gnut_hash.find_raw not found\n");
  return 0;
}

/* gnut_hash.find looks for given data in the hash, and if found, returns
 * a pointer to the hash's version of that data. */
void * gnut_hash_find(Gnut_Hash *gh, void *data)
{
  int a;
  int ret;
  Gnut_List *tmp;

  dqi(0x013b);

  if (!gh) {
    gd_s(0, "gnut_hash.find gh="); gd_p(0, gh); gd_s(0, " data=");
    gd_p(0, data); gd_s(0, "\n");
    dqi(0x013c);
    return(0);
  }

  a = (*gh->hash_func)(data);
  dqi(0x013d);

  gd_s(5, "gnut_hash.find hash returned: "); gd_i(5, a); gd_s(5, "\n");

  for (tmp=gh->list[a]; tmp; tmp=tmp->next) {
    gd_s(5, "gnut_hash.find loop tmp == "); gd_i(5, ((int) tmp));
    gd_s(5, "\n");
    gnut_mprobe(tmp);
    ret = (*gh->compare_func)(data, tmp->data);
    if (ret==0) {
      gd_s(5, "gnut_hash.find loop return "); gd_i(5, ((int) (tmp->data)));
      gd_s(5, "\n");
      dqi(0x013e);
      return tmp->data;
    }
  }
  
  gd_s(3,"gnut_hash.find not found\n");
  dqi(0x013f);
  return 0;
}

/* uint32 gnut_hash.remove(Gnut_Hash *gh, void *data)
 *
 * removes the occurence of data (if any) in the hash. It does NOT try
 * to call xfr.ee on the data block. */
uint32 gnut_hash_remove(Gnut_Hash *gh, void *data)
{
  int a;

  if (!gh) {
    gd_s(0, "gnut_hash.remove gh="); gd_p(0, gh); gd_s(0, " data=");
    gd_p(0, data); gd_s(0, "\n");
    return 0;
  }

  gd_s(3, "gnut_hash.remove entering\n");

  a = (*gh->hash_func)(data);

  gh->list[a] = gnut_list_remove(gh->list[a], data, 3); /* %%% sometimes
                                              reports 'item not found' */

  return 0;
}

uint32 gnut_hash_fre(Gnut_Hash *gh)
{
  int i;

  gd_s(3, "gnut_hash_fre entering gh="); gd_p(3, gh); gd_s(3, "\n");

  if (!gh) {
    gd_s(0, "gnut_hash_fre  gh="); gd_p(0, gh); gd_s(0, "\n");
    return 0;
  }

  for (i=0;i<256;i++) {
    gh->list[i] = gnut_list_fre(gh->list[i]);
  }

  fre_gh(&gh, 410);
  return 0;
}

/*        LRU HASH ROUTINES
 *
 * An LRU hash is an associative array indexed by arbitrary blocks
 * of data (e.g. text strings, or 16-byte GUIDs) that have arbitrary
 * blocks of data as array elements. Furthermore, an LRU hash includes
 * automaic garbage collection -- when the hash has a certain maximum
 * number of elements, the oldest (least-recently-used) items will be
 * thrown away.
 *
 * Internally, the LRU hash is implemented with two simpler data
 * structures, a Gnut_Queue and a Gnut_Hash. The LRU hash key (an
 * arbitrary block of data) is CRC'd to generate a 32-bit hash key.
 * This key is used to store the data in the Gnut_Hash. Copies of the
 * 32-bit keys are also stored in a Gnut_List, which is used to figure
 * out which item is the next to get thrown away.
 */

/* The hash function just looks at the CRC, not the data pointer */
uchar lru_hashfunc(void * x)
{
  crcptr * cp;
  uint32 t;
  uchar rv;

  cp = (crcptr *) x;

  t = cp->crc;
  t ^= (t >> 16L);
  rv = t ^ (t >> 8L);

  return rv;
}

/* The compare function just compares the CRCs, not the data pointers
 * or their data */
int lru_compfunc(void * a, void * b)
{
  crcptr *acp;
  crcptr *bcp;

  acp = (crcptr *) a;
  bcp = (crcptr *) b;

  if (acp->crc == bcp->crc) {
    return 0;
  }
  return -1;
}

/* lru_init sets up an empty LRU hash. */
void lru_init(lru_hash * h, uint32 maxaloc)
{
  h->maxaloc = maxaloc;
  h->hash = gnut_hash_new(lru_hashfunc, lru_compfunc);
  h->queue = gnut_queue_new(364);
  h->alocnum = 0;
}

/* Do garbage-collection */
void lru_gc(lru_hash *h)
{
  crcptr * cp;

  /* Find and remove the last thing on the queue */
  cp = (crcptr *) gnut_queue_remove(h->queue);

  /* If found, remove it from the hash too, then delete its data */
  if (cp) {
    /* printf("gc "); print_ascii((char *) cp->ptr); printf("\n"); */
    gnut_hash_remove(h->hash, cp);
    if (cp->ptr) {
      fre_v(&(cp->ptr), 510);
    }
    fre_crcp(&cp, 411);
    h->alocnum--;
  }
}

/* Store a new piece of data into an lru_hash. If data already exists
 * under the given key, the new data will replace the old data.
 * NOTE: If you want it to al.locate and copy the data, use
 * lru.storecopy. */
int lru_store(lru_hash * h, void * key, uint32 keylen, void * data,
			   uint32 datalen)
{
  uint32 i;
  crcptr * cp;
  crcptr * other;
  int rv; /* 0.4.27.c16 */

  rv = 0; /* 0.4.27.c16 */

  /* Al.locate the crcptr block */
  cp = (crcptr *) ymaloc(sizeof(crcptr), 369);
  if (cp == 0) {
    printf("lru_store mem full\n");
    return 0;
  }

  /* Store CRC, data pointer and data length */
  cp->crc = crc32_block(key, keylen);
  cp->ptr = data;
  cp->datalen = datalen;

  /* printf("lru_store   %08lx %4i ", cp->crc, datalen);
     print_ascii((char *) data); printf("\n"); */

  pthread_mutex_lock(&lruh_mutex); /* 0.4.27.c16 */

  /* Now we need to check if this thing is already in the hash */
  dqi(0x014d);
  other = (crcptr *) gnut_hash_find(h->hash, (void *) cp);
  if (other) {
    /* An entry was found with the same key, so we have to replace it.
     * Purge its data, but keep its crcptr, and deal.locate the crcptr
     * we just created */
    if (other->ptr) {
      /* printf("lru_st fr.ee %08lx %4i ", cp->crc, other->datalen);
	 print_ascii((char *) other->ptr); printf("\n"); */
      fre_v(&(other->ptr), 412);
    }
    other->ptr = data;
    other->datalen = datalen;
    fre_crcp(&cp, 413);
    rv = 0; /* 0.4.27.c16 */
  } else {
    /* Add references to this crcptr block to the queue and the hash */
    gnut_hash_insert(h->hash, cp, 502);
    gnut_queue_insert(h->queue, cp, 509);

    /* Increment al.locate, and GC if necessary */
    (h->alocnum)++;
    for(i=0; i<3; i++) {
      if ((h->alocnum) > (h->maxaloc)) {
	lru_gc(h);
      }
    }
    rv = 1; /* 0.4.27.c16 */
  }

  pthread_mutex_unlock(&lruh_mutex); /* 0.4.27.c16 */
  return rv; /* 0.4.27.c16 */
}

/* Copy a piece of data and store it into an lru_hash. If data
 * already exists under the given key, the new data will replace the
 * old data.
 *
 * Returns nonzero result if the key is new, 0 if the key has been seen
 * before */
int lru_storecopy(lru_hash * h, void * key, uint32 keylen, void * data,
		  uint32 datalen, int tracking_number)
{
  void * dc;

  dc = ymaloc(datalen, tracking_number);
  if (dc == 0) {
    gd_s(0, "lru_storecopy mem full\n"); /* 0.4.27.c16  used to be a printf */
    return 0;
  }
  memcpy(dc, data, datalen);

  return lru_store(h, key, keylen, dc, datalen);
}

/* Get a piece of data from an LRU hash, wthout copying.
 * WARNING: subsequent calls to lru_store, perhaps by other threads,
 * can delete the data pointed to by the pointer returned by lru_get.
 * If multithreading is an issue and if you are going to actually do
 * anything with the data (other than just testing for its existence
 * or size) you should probably use lru_copy, or lock the mutex yourself
 * and pass 0 in the needlock parameter to lru_get */
void * lru_get(lru_hash * h, void * key, uint32 keylen, uint32 * datalen,
	       int needlock) /* 0.4.27.c16 */
{
  crcptr cp;
  crcptr * it;
  void * rv; /* 0.4.27.c16 */

  rv = 0; /* 0.4.27.c16 */

  /* Fill in our temp crcdata block */
  cp.crc = crc32_block(key, keylen);
  cp.ptr = 0;
  cp.datalen = 0;

  if (needlock) { /* 0.4.27.c16 */
    pthread_mutex_lock(&lruh_mutex);
  }

  /* look for it */
  dqi(0x014e);
  it = (crcptr *) gnut_hash_find(h->hash, &cp);
  if (it) {
    *datalen = it->datalen;
    rv = it->ptr; /* 0.4.27.c16 */
  } else {
    /* not found */
    *datalen = 0;
    rv = 0; /* 0.4.27.c16 */
  }

  if (needlock) { /* 0.4.27.c16 */
    pthread_mutex_unlock(&lruh_mutex);
  }
  return rv; /* 0.4.27.c16 */
}

/* Get a piece of data from an LRU hash, al.locate and copy it, and
 * return a pointer to the result. You are responsible for deal.locating
 * this copy yourself. You also have to test it before using or
 * deal.locating (in case nothing was found) */
void * lru_copy(lru_hash * h, void * key, uint32 keylen, uint32 * datalen)
{
  uint32 l2;
  void *it;
  void *rv;

  rv = 0; /* 0.4.27.c16 */

  pthread_mutex_unlock(&lruh_mutex); /* 0.4.27.c16 */

  it = lru_get(h, key, keylen, &l2, 0); /* 0.4.27.c16 */
  if (it) {
    rv = ymaloc(l2, 371);
    if (rv) {
      memcpy(rv, it, l2);
    } else {
      l2 = 0;
    }
    *datalen = l2;
  } else { /* 0.4.27.c16 */
    *datalen = 0;
    rv = 0; /* 0.4.27.c16 */
  }

  pthread_mutex_unlock(&lruh_mutex); /* 0.4.27.c16 */
  return rv; /* 0.4.27.c16 */
}
