/* Josh Pieper, (c) 2000 */

/* This file is distributed under the GPL, see file COPYING for details */

#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#ifndef WIN32
#include <sys/socket.h>
#include <netinet/in.h>
#include <unistd.h>
#include <sys/time.h>
#include <netdb.h>
#endif
#include <errno.h>
#include <string.h>
#include <time.h>
#include <pthread.h>
#include <regex.h>

#include "cli.h"
#include "http.h" /* 0.4.27.c12 */
#include "lib.h"
#include "threads.h"
#include "connection.h"
#include "queue.h"
#include "net.h"
#include "ui.h"
#include "protocol.h"
#include "route.h"
#include "share.h"
#include "conf.h"
#include "host.h"
#include "monitor.h"
#include "qry.h"
#include "cache.h"
#include "gnut.h"
#include "blacklist.h"

/* 0.4.27.c12 removed the prototype of gnut_http_result_append, include the
   header file instead */

/* this mutex protects changes to the connection list */
#ifndef PTHREADS_DRAFT4
  pthread_mutex_t gc_list_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t send_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t query_packet_mutex = PTHREAD_MUTEX_INITIALIZER;
  pthread_mutex_t qrepl_mutex = PTHREAD_MUTEX_INITIALIZER;
#else
  pthread_mutex_t gc_list_mutex;
  pthread_mutex_t send_mutex;
  pthread_mutex_t query_packet_mutex;
  pthread_mutex_t qrepl_mutex;
#endif

/* linked list of all connections */
Gnut_List *gc_list = 0;

uint32 con_num_queries = 0;
uint32 con_num_responses = 0;
uint32 con_num_messages = 0;
uint32 con_num_sent = 0;

float64 g_rcv_bytes = 0;
float64 g_snt_bytes = 0;

void fre_gc(gcs **x, int bugnum)
{
  yfre((void **) x, bugnum);
}

int gnut_connection_num()
{
  return gnut_list_size(gc_list);
}

int gnut_connection_net_totals(uint32 *messages, uint32 *sent,
  float64 *rcv_bytes, float64 *sbytes, float64 *blacklisted)
{
  *messages = con_num_messages;
  *sent = con_num_sent;
  
  gd_s(2, "rcv_bytes="); gd_f64(2, g_rcv_bytes); gd_s(2, " snt_bytes=");
  gd_f64(2, g_snt_bytes); gd_s(2, "\n");

  if (rcv_bytes) {
    *rcv_bytes = g_rcv_bytes;
  }
  if (sbytes) {
    *sbytes = g_snt_bytes;
  }
  *blacklisted = gnut_blacklist_totals();
  return 0;
}

int gnut_connection_find_ptr(gcs *gc)
{
  Gnut_List *gltmp;
  gcs *mgc;

  for (gltmp=gc_list;gltmp;gltmp=gltmp->next) {
    mgc=gltmp->data;
    if (mgc==gc) {
      return 0;
    }
  }
  return -1;
}

/* int gnut_connection_find(uchar ip[4], uint16 port)
 *
 * Tests to see if the given IP/port is already on the active
 * connections list
 *
 * returns 0 if a connection is found to the listed ip and port
 * returns -1 otherwise
 */
int gnut_connection_find(uchar ip[4], uint16 port)
{
  Gnut_List *gltmp;
  gcs *gc;

  for (gltmp=gc_list; gltmp; gltmp=gltmp->next) {
    gc = gltmp->data;
    if ((memcmp(ip, gc->ip.b, 4)==0) && (gc->port==port)) {
      return 0;
    }
  }

  return -1;
}

int send_to_one(gcs *gc, gnutella_packet *gpa)
{
  gnutella_packet *ngp;

  gd_s(3, "send_to.one entering (gc="); gd_p(3, gc); gd_s(3, ", gpa=");
  gd_p(3, gpa); gd_s(3, ")\n");

  pthread_mutex_lock(&(gc->out_q_mutex)); {
    gd_s(3, "send_to.one got mutex\n");

    gd_s(3, "send_to.one  adding to packet queue\n");
    ngp = gp_dup(gpa);
    if (gc->out_queue) {
      gnut_queue_insert(gc->out_queue, ngp, 506);
    } else {
      fre_gpa(&ngp, 553);
    }

    gd_s(3, "send_to.one rel mutex\n");
  } pthread_mutex_unlock(&gc->out_q_mutex);

  if (gnut_lib_debug>=2) {
    gd_s(2,"send_to.one I just sent this packet:\n");
    gp_print(gpa);
  }

  gd_s(3,"send_to.one ret success\n");
  return 0;
}

int send_to_all(gnutella_packet *gpa)
{
  Gnut_List *gltemp;
  gnutella_packet *ngp;
  gcs *gc;

  gd_s(3, "send_to.all entering\n");

  /*  pthread_mutex_lock(&gc_list_mutex); */

  for (gltemp=gc_list; gltemp; gltemp=gltemp->next) {
    gc = gltemp->data;

    if (gc->cstate==STATE_CONNECTED) {
      pthread_mutex_lock(&(gc->out_q_mutex)); {
	gd_s(3, "send_to.all got mutex\n");

	ngp = gp_dup(gpa);
	gnut_queue_insert(gc->out_queue, ngp, 507);

	gd_s(3, "send_to.all rel mutex\n");
      } pthread_mutex_unlock(&(gc->out_q_mutex));
    }
  }

  /*  pthread_mutex_unlock(&gc_list_mutex); */
  if (gnut_lib_debug>=2) {
    gd_s(2, "send_to.all I just sent this packet:\n");
    gp_print(gpa);
  }

  gd_s(3, "send_to.all ret success\n");
  return 0;
}
    
int send_to_all_except(gcs *gc, gnutella_packet *gpa, int no_drop)
{
  Gnut_List *gltemp;
  gnutella_packet *ngp;
  gcs *mgc;

  dqi(0x0173);

  gd_s(3, "send_to_all_except entering (gc="); gd_p(3, gc); gd_s(3, ", gpa=");
  gd_p(3, gpa); gd_s(3, ")\n");

  /*  pthread_mutex_lock(&gc_list_mutex); */

  gd_s(3,"done locking\n");

  for (gltemp=gc_list; gltemp; gltemp=gltemp->next) {
    gd_s(3, "send_to_all_except casting data\n");
    mgc = gltemp->data;
    gd_s(3, "send_to_all_except checking for exceptions mgc=");
    gd_p(3, mgc); gd_s(3, "\n");
    if (mgc!=gc && mgc->cstate==STATE_CONNECTED) {
      gd_s(3, "send_to_all_except adding to packet queue on mgc=");
      gd_p(3, mgc); gd_s(3, "\n");

      dqi(0x0174);
      pthread_mutex_lock(&(mgc->out_q_mutex)); {
	gd_s(3, "send_to_all_except mutex locked successfully\n");

	if (no_drop || (mgc->out_queue->size < BACKLOG_MAX)) {
	  gd_s(4, "send_to_all_except duplicating packet\n");
	  ngp = gp_dup(gpa);
	  gd_s(4, "send_to_all_except prepending to packet queue\n");
	  gnut_queue_insert(mgc->out_queue, ngp, 508);
	} else {
	  /* You can't kill connections just for getting backlogged, that
	   * would cause cable modem users to drop all connections to
	   * dialup users. Instead, we just 'drop' the packet by not
           * duplicating or enqueuing it. */

	  /* gd_s(1,"killing connection %p because of backlog\n",mgc); */
	  /* mgc->cstate=STATE_DEAD; */
	}
      } pthread_mutex_unlock(&(mgc->out_q_mutex));
    }
  }
  dqi(0x0175);

  if (gnut_lib_debug>=2) {
    gd_s(2, "send_to_all_except I just sent this packet:\n");
    gp_print(gpa);
  }

  /*  pthread_mutex_unlock(&gc_list_mutex); */
  gd_s(3, "send_to_all_except returning success\n");
  return 0;
}

/* gcs * gnut_conn.ection.new()
 *
 * Al.locates a new gcs structure, fills in some default
 * values, and inserts it into the connection list, then returns
 * a pointer to this newly al.located structure. */
gcs * gnut_connection_new()
{
  gcs *gc;

  dqi(0x00f8);
  
  gd_s(3,"gnut_conn.ection.new entering\n");

  gc = (gcs *) ycaloc(sizeof(gcs), 1, 489);
  
  gc->gc_sock = -1;

#ifndef PTHREADS_DRAFT4
  pthread_mutex_init(&gc->out_q_mutex, 0);
#else
  pthread_mutex_init(&gc->out_q_mutex,pthread_mutexattr_default);
#endif

  dqi(0x00f9);

  gc->out_queue = gnut_queue_new(515);

  gc->sent_packets = 0;
  gc->received_packets = 0;
  gc->dropped_packets = 0;
  gc->bad_packets = 0;

  gc->snt_bytes = 0;
  gc->rcv_bytes = 0;

  gc->rate_snt_bytes = 0;
  gc->rate_rcv_bytes = 0;
  gc->rate_start_time = time(0);

  gc->routing_errors = 0;

  gc->seen_pong = 0;

  gc->ctype = TYPE_UNSPEC;
  gc->cstate = STATE_UNCON;
  
  gc->tid = pthread_self();

  pthread_mutex_lock(&gc_list_mutex);

  gc_list = gnut_list_append(gc_list, gc, 367);

  pthread_mutex_unlock(&gc_list_mutex);
  
  gd_s(3, "gnut_connection_new returning success\n");

  dqi(0x00fa);
  
  return gc;
}

/* int gnut_connection.delete(gcs *gc)
 *
 * takes the gcs structure, fr.ee's up any memory
 * associated with it, and removes it from the connection list
 * returns 0 on success */
int gnut_connection_delete(gcs *gc)
{
  Gnut_List *gltmp;
  gnutella_packet *gpa;
  Gnut_List *unlinked;

  gd_s(1, "gnut_connection_delete entering gc="); gd_p(1, gc); gd_s(1, "\n");

  pthread_mutex_lock(&gc_list_mutex);

  /* this didn't work, what we need to do is just unlink the list
   * then wait the second to let all references to it past,
   * then it will be safe to delete... */

  gc_list = gnut_list_unlink(gc_list, gc, &unlinked);

  pthread_mutex_unlock(&gc_list_mutex);

  gd_s(1, "gnut_connection_delete unlinked\n");

  /* it's possible, although HIGHLY unlikely, that some other
   * function managed to get a hold on our pointer during the remove
   * operation... since I don't want to always lock the connection list
   * we need some other way of making sure that this particular pointer
   * isn't in use.  Perhaps a reference count of some sort....  sleeping
   * for a fraction of a second is probably an easier solution to
   * implement however */
  route_guid_clear(gc);

  sleep(2);

  if (unlinked) { /* 0.4.28.c13 */
    fre_gl(&unlinked, 241);
  }

  /* now that it's out of the connection list, we can
     go about fr.eeing it up */
  for (gltmp = gc->out_queue->head; gltmp; gltmp = gltmp->next) {
    gpa = gltmp->data;
    if (gpa->data) {
      fre_v(&(gpa->data), 242);
    }
  }

  gnut_queue_fre(gc->out_queue);
  fre_gq(&(gc->out_queue), 485);

  pthread_mutex_destroy(&(gc->out_q_mutex));

  fre_gc(&gc, 243);

  gd_s(1, "gnut_connection_delete returning success\n");

  return 0;
}

/* 0.4.27.c21 */
int eqhd_flageval(uint8 validbyte, uint8 flagbyte, uint8 mask)
{
  int rv;

  validbyte &= mask;
  flagbyte &= mask;
  rv = (validbyte ? 2 : 0) + (flagbyte ? 1 : 0);

  return rv;
}

/* 0.4.27.c21 
   Parse the extended query hits descriptor, which is described here:
     http://www.clip2.com/GnutellaProtocol04.pdf
     http://www.clip2.com/qhd_extension.html  */
void parse_eqhd(uint8 *einfo, int32 infolen, /* The descriptor */
		uint8 *ip, uint32 speed,      /* Other info from this qreply */
		int16 *type, int16 *rating)   /* Return values */
{
  int16 ty;
  int16 i;
  uint16 odsize;
  int16 st_meas_speed;
  int16 st_proven;
  int16 st_busy;
  int16 st_must_push;
  float64 adj_speed;
  int16 score;
  int16 max_possible;
  float64 stars;

  /* Default all the return values */
  *type = 0;
  *rating = 0;

  /* Length must be at least 5 to be of any use */
  if (infolen < 5) {
    return;
  }

  /* Try to skip Gnotella data. If there is *no* Gnotella data, then the
     two preceding bytes will be nulls. If there *is* Gnotella data, only
     one of the preceding bytes is a null, and there is another null somewhere
     within our EQHD. To skip it, we just have to find the null, if any, and
     skip to the end of it. If not found, either the EQHD is invalid or the
     last query result is invalid, in either case we just return. */
  if ((einfo[-1] == 0) && (einfo[-2] == 0)) {
    /* It's okay, we don't have to skip a null */
  } else {
    for(i=0; einfo[i] && (i<infolen); i++) {
    }
    /* Now einfo[i] is the found null, or we didn't find one */
    i++;
    /* Subtract from infolen */
    infolen -= i;
    einfo += i;
    /* Error-check length again */
    if (infolen < 5) {
      return;
    }
  }

  /* Determine host server type */
  ty = 0;
  for(i=1; (ty == 0) && (evendors[i].vcode); i++) {
    if (strncmp(einfo, evendors[i].vcode, 4) == 0) {
      ty = i;
    }
  }
  *type = ty;
  /* printf("%3d.%3d.%3d.%3d ", ip[0], ip[1], ip[2], ip[3]); */
     /* printf("%3d %11s ", infolen, evendors[ty].appname); */

  /* printf("%4u ", speed); */

  /* Get opendata length */
  odsize = (uint16) (einfo[4]);

  /* Errorcheck length again */
  if (5 + odsize > infolen) {
    /* printf("invalid odsize\n"); */
    return;
  }

  /* If no open data, there is nothing else to parse */
  if (odsize == 0) {
    /* printf("no opendata\n"); */
    return;
  }

  /* If only one opendata byte, treat it as an error */
  if (odsize < 2) {
    /* printf("not enough opendata\n"); */
    return;
  }

  /* If more than two opendata bytes, treat it as an error */
  if (odsize > 2) {
    /* printf("too much opendata\n"); */
    return;
  }

  /* Evaluate the flags. This would have been a lot easier if the flag
     bytes were arranged logically (what were those guys thinking?) */
  st_meas_speed = eqhd_flageval(einfo[5], einfo[6], 0x10);
  st_proven     = eqhd_flageval(einfo[5], einfo[6], 0x08);
  st_busy       = eqhd_flageval(einfo[5], einfo[6], 0x04);
  st_must_push  = eqhd_flageval(einfo[6], einfo[5], 0x01);

  /* Errorcheck the flags. None should be 1, because that would mean that
     the flag is set but the valid flag is clear */
  if ((st_meas_speed == 1) || (st_proven == 1)
      || (st_busy == 1) || (st_must_push == 1)) {
    /* printf("invalid flags\n"); */
    return;
  }

  /* Normalize speed of the served file relative to our own, produce a
     value from 0 to 100. 100 means we can download the file at 100% of our
     own local ISP speed. */
  adj_speed = 0;
  switch(st_meas_speed) {
  case 3:
    /* Measured speed */
    if (speed > gc_speed) {
      /* Note this includes the case of gc_speed == 0 */
      adj_speed = 1.0;
    } else {
      adj_speed = ((float64) speed) / ((float64) gc_speed);
    }
    break;
  case 2:
  default:
    /* Non-measured speed. Here we have to estimate how much speed we might
       get, the estimate is 1/4 of the advertized speed or our own speed,
      whichever is greater. */
    if (gc_speed == 0) {
      adj_speed = 1.0;
    } else {
      adj_speed = ((float64) speed) / ((float64) gc_speed);
      adj_speed /= 4.0;
      if (adj_speed > 1.0) {
	adj_speed = 1.0;
      }
    }
    break;
  }
  adj_speed *= 100.0;

  /* Compute max possible score */
  max_possible = 100 + gc_score_available + gc_score_did_upload + gc_score_did_receive;

  score = adj_speed;

  /* Add scores for the flags. For each flag the possible values are 0 2 or 3.
     One of the values 2 or 3 is "good", the other is "bad", and a 0
     is "unknown". "bad" counts for 0 points, and "unknown" counts for half
     as many points as "good". */
  /* Most important attribute is shown first. */
  switch(st_busy) {
  case 2: score += gc_score_available; break;
  case 0: score += (gc_score_available / 2); break;
  }
  switch(st_must_push) {
  case 2: score += gc_score_did_receive; break;
  case 0: score += (gc_score_did_receive / 2); break;
  }
  switch(st_proven) {
  case 3: score += gc_score_did_upload; break;
  case 0: score += (gc_score_did_upload / 2); break;
  }

  /* Convert this to stars. There are 9 possible ratings from 0 to 8
     -- for example, 2 1/2 stars is 5, 0 stars is 0, 4 stars is 8. We
     multiply by 15.9999 to add 4 "fraction bits" to the integer result. */
  stars = 15.9999 * 9.0 * ((float64) score) / ((float64) max_possible);
  *rating = (int16) stars;

  /*  printf("%s", ratings[(*rating) >> 4]); */

  /*  printf("  mspeed %d proven %d busy %d mustpu %d\n", st_meas_speed, st_proven, st_busy, st_must_push); */

  if (0) {
    int x, i;
    uint8 *d;

    d = einfo;
    x = infolen;

    if ((d[0] == ':') && (isdigit(d[1])) && (isdigit(d[2]))) {
      /* Ignore these */
    } else if (isprint(d[0]) && isprint(d[1]) && isprint(d[2]) && isprint(d[3])) { /* Enable this to see the extra data BearShare servers return */
      printf("%3d qreply extra: ", x);
      if (x > 18) {
	x = 18;
      }
      for(i=0; i<4; i++) {
	if (isprint(d[i])) {
	  printf("%c", d[i]);
	} else {
	  printf(" ");
	}
      }
      for(i=4; i<x; i++) {
	if (i) {
	  printf(".");
	}
	printf("%02x", d[i]);
      }
      printf(" ");
      for(i=4; i<x; i++) {
	if (isprint(d[i])) {
	  printf("%c", d[i]);
	}
      }
      printf("\n");
    }
  }
}

/* Sync packet formats
 *
 * Self-sync packet: Sent out a single port to sync the return stream
 * on that port. It is simply a PING with a TTL of 1 (which should
 * generate only one PONG, and get immediately discarded)
 *
 * 00 55 AA FF 53 4E 43 48 22 22 22 22 22 22 xx xx 00 01 00 00 00 00 00
 *              S  N  C  H
 *
 * If you receive more than one PONG from this PING, the servent at
 * the other end has a TTL bug, and you should immediately close the
 * connection.
 *
 * Network sync packet: gnut servents may also participate in the
 * initiation of network-wide sync packets, which are QUERY packets that
 * appear periodically in all connection streams.
 *
 * 00 55 AA FF 53 4E 43 48 33 33 33 33 33 33 xx xx
 *              S  N  C  H
 * 80 tt hh 10 00 00 00
 *
 * 00 00 33 33 33 33 33 33 53 79 4E 4C 48 67 6E 00
 *        3  3  3  3  3  3  S  y  N  C  H  g  n
 *
 * A new packet is generated only if *all* incoming streams have been
 * fr.ee of sync packets for at least 2048 bytes.
 */

/* Two lru_hash tables are used to implement the mreply and mpush
 * commands, as well as the instant-search feature.
 *
 * The netfiles hash table has the following key and data:
 *
 *   key = [crc]
 *   data = [ip, port, speed, length, filename]
 *
 * The query-reply hashtable maps equivalent GUID/refnum pairs
 * onto each other. It has this key and data:
 *
 *   key = [guid, refnum]
 *   data = [crc]
 *
 * When query-replies are routed, each data item is added to the netfiles
 * hashtable if it is new, and each GUID/refnum pair is also added to
 * the query-reply hashtable. If the mreply command is active, any new
 * netfiles entries are displayed as they are added.
 *
 * When push requests are routed, the GUID and refnum in the push request
 * is indexed through the query-replies hashtable to get a CRC which is used
 * to get the netfiles entry, for display by the mpush command.
 *
 * To implement instant search, a complete scan of the netfiles
 * hashtable is executed, copying the data into the search results
 * list if matching data is found. */

/* name_tok_cb is used to implement strict_search when multi_enable is set:
 * it is a callback to match a query reply against all one of the searches
 * in the current active search list */
char name_tok_cb(GnutSearch *s, void *data)
{
  return keyword_match(s->query, data, 1, 1);
}

/* do whatever processing is necessary */
int connection_handle_packet(gcs *gc, gnutella_packet *gpa)
{
  unsigned char *dataraw;
  gcs *mgc;
  gnutella_packet *mgpa;
  gnutella_servent *gm;
  gnutella_results *grh;
  gnutella_results_name *grn;
  gnutella_results_suffix *grs;
  gnutella_push *gp;
  uint32 num;
  float64 sh_size;
  int isnew;
  Gnut_List *results;
  int n_reslts;
  char *buf;
  uint32 speed;
  uint32 dlen;
  int i;
  int accept;
  uint32 pref;
  uint32 psiz1; /* 0.4.27.c11 (I also renamed this variable to make it
		   easier to distinguish from other psize's declared
                   elsewhere) */
  route_entry *the_re;
  int no_drop;
  int multi_enable;
  GnutSearch *msearch = 0;
  struct in_addr ipt;
  int16 htype; /* 0.4.27.c21 */
  int16 rating; /* 0.4.27.c21 */

  dqi(0x0129);

  YPROBE(gpa->data, 156)

  no_drop = 0;
  isnew = 0;
  
  gd_s(3, "conn_hdl_packet entering gc="); gd_p(3, gc); gd_s(3, ", gpa=");
  gd_p(3, gpa); gd_s(3, "\n");

  memcpy(&dlen, gpa->gh.dlen, 4);
  dlen = GUINT32_FROM_LE(dlen);

#ifdef MOREPACKETSTATS
  ++gc->packet_count[gpa->gh.func];
  if (dlen > gc->max_packet_size[gpa->gh.func]) {
    gc->max_packet_size[gpa->gh.func] = dlen;
  }
  gc->packet_byte_count[gpa->gh.func] += dlen;
  ++gc->ttl_count[gpa->gh.ttl];
#endif

  if (dlen>67075) {
    gd_s(0, "conn_hdl_packet WOAH! packet of size: "); gd_i(0, dlen);
    gd_s(0, " made it through!\n");
    return -1;
  }

  if (gnut_lib_debug>=2) {
    gd_s(2, "conn_hdl_packet received following packet\n");
    gp_print(gpa);
  }

  /* If the ttl is greater than 0200, it's a packet that was forwarded by
   * a client with the "TTL wraparound bug", probably Windows Gnutella
   * prior to 0.56. These should be dropped. */
  if (gpa->gh.ttl>200 || gpa->gh.hops>200) {
    gd_s(2, "conn_hdl_packet ttl too high: ttl="); gd_i(2, gpa->gh.ttl);
    gd_s(2, "\n");
    return 0;    
  }

  /* Any other packets with a TTL bigger than our TTL should have their
   * TTL diminished, and if their hopcount is greater than our TTL we
   * should drop 'em */
  if (gpa->gh.ttl > gc_ttl) {
    gpa->gh.ttl = gc_ttl;
  }
  if (gpa->gh.hops > gc_ttl) {
    gd_s(2, "conn_hdl_packet hops too high: ");
    gd_i(2, gpa->gh.hops);
    gd_s(2, "\n");
    if (0) {
      /* %%%RPM diagnostics: bad TTLs implies strange data in packets,
       * perhaps ASCII data like query replies or chatroom messages */
      int j; char c; char *t;
      printf("%i'", gpa->gh.hops);
      t = (char *) (&(gpa->gh.guid));
      for(j=0; j<22; j++) {
	c = *t++;
	if (c>31 && c<127)
	  printf("%c", c);
      }
      printf("'");
    }
    return 0;
  }

  /* Check for GUID's we've seen before, and locate the connection
   * that GUID originated from. If it's a known GUID we might drop the
   * packet, or we might route it to that particular connection, depending
   * on the type of packet. */
  mgc = route_guid_find(gpa->gh.guid, &the_re);
  if (mgc) {
    isnew=0;
  } else {
    isnew=1;
  }
  
  gd_s(3, "conn_hdl_packet entering packet switch\n");
  
  switch(gpa->gh.func) {
  case 0x00: /* ------------------------------------------------------ PING */
    /* a PING */

    dqi(0x0001);

    if (isnew == 0) {
      /* gc->dropped_packets++; */
      return 0;

      /* Where can I apply the blacklist rules
       * for pings? */
      /* } else if (!gnut_blacklist.allow(BLACKLIST_TYPE_PING, 0, */
      /*  IP-goes-here , 0)) { */
      /* return 0; */
    } else if ((gpa->gh.hops > 2) && (gpa->gh.hops > (gc_ttl / 2))) {
      /* We discourage PINGs that have travelled a long way before
       * reaching us, by silently dropping them. */
      if (gc_debug_opts & 32) {
        printf("drop1\n");
      }
      /* gc->dropped_packets++; */
      return 0;
    } else if (!conf_get_int("hidden")) {
      gd_s(3,"conn_hdl_packet PING packet\n");
      if (dlen>0) {
	gd_s(1, "conn_hdl_packet dlen=");
	gd_i(1, dlen);
	gd_s(1, "\n");
	return -2;
      }
      gd_s(2, "conn_hdl_packet handling PING\n");

      /* this is a ping packet, we should reply
       * to it with a servent packet... */
      share_totals(&num, &sh_size);

      /* When sending back our shared size, we have to divide by 1024
	 because that's what the GnutellaNet protocol specifies */
      mgpa = gp_pong_make(gpa->gh.guid, num,
			  ((uint32) sh_size / 1024.0), 
		  	  net_local_ip() , conf_get_int("local_port") , 
			  gc_ttl);

      send_to_one(gc, mgpa);

      /* We're done with the PONG packet; fr.ee it. */
      if (mgpa->data) {
	fre_v(&(mgpa->data), 244);
      }
      fre_gpa(&mgpa, 245);
    }
    break;

  case 0x01: /* ------------------------------------------------------ PONG */
    /* A PONG (ping reply) */

    dqi(0x0002);

    if (dlen != sizeof(gnutella_servent)) {
      gd_s(1, "conn_hdl_packet bad PONG packet dlen=");
      gd_i(1, dlen);
      gd_s(1, " should be=");
      gd_i(1, sizeof(gnutella_servent));
      gd_s(1, "\n");
      return -3;
    }
    gm = gpa->data;

    dataraw = gpa->data;
#if 0
    printf("PONG IP %u.%u.%u.%u:%u.%u\n", (unsigned int)(dataraw[2]),
	   (unsigned int)(dataraw[3]), (unsigned int)(dataraw[4]),
           (unsigned int)(dataraw[5]), (unsigned int)(dataraw[0]),
           (unsigned int)(dataraw[1]));
#endif

    if (gc->seen_pong == 0) {
      if (gc_debug_opts & 32) {
        printf("First PONG on %u.%u.%u.%u is %u.%u.%u.%u\n",
	       (unsigned int)(gc->ip.b[0]), (unsigned int)(gc->ip.b[1]),
               (unsigned int)(gc->ip.b[2]), (unsigned int)(gc->ip.b[3]),
	       (unsigned int)(dataraw[2]), (unsigned int)(dataraw[3]),
               (unsigned int)(dataraw[4]), (unsigned int)(dataraw[5]));
      }
      gc->seen_pong = 1;
    }

    dqi(0x00fb);

    /* Apply the blacklist rules. */
    memcpy(&ipt, &(gm->ip), sizeof(ipt));
    if (!gnut_blacklist_allow(BLACKLIST_TYPE_PING, 0, &ipt, 0, 0)) {
      return 0;
    }
	
    dqi(0x0136);
    gd_s(2, "conn_hdl_packet handling PONG\n");

    if (host_find(gm->ip, gm->port) == -1) {
      dqi(0x0137);
      if (host_ok_for_get(gm->ip)) {
	uint32 host_size;
        host_entry *he;

        dqi(0x00fc);

	gd_s(2, "conn_hdl_packet adding new servent\n");
	/* the host wasn't found in the database, so we'll add it */
	memcpy(&num, gm->files, 4);
	num = GUINT32_FROM_LE(num);

	memcpy(&host_size, gm->kbytes, 4);
	host_size = GUINT32_FROM_LE(host_size);

	he = (host_entry *) ymaloc(sizeof(host_entry),
				   330); /* There are two 0330's on purpose! tagok */
	memcpy(he->ip, gm->ip, 4);
	memcpy(he->port, gm->port, 2);
	he->files = num;
	he->kbytes = host_size;
        host_put_lru(he);
	he = host_add(he);
        if (he) {
          fre_hen(&he, 552);
        }
      }
    }
    dqi(0x0138);
    if (memcmp(conf_get_str("update_guid"), gpa->gh.guid, 16)==0) {
      dqi(0x0176);
      return 0;
    }
    dqi(0x0177);
    break;

  case 0x40: /* ------------------------------------------------------ PUSH */
    /* A PUSH request */

    dqi(0x0003);

    if (dlen!=sizeof(gnutella_push)) {
      gd_s(1, "conn_hdl_packet bad push request! dlen="); gd_i(1, dlen);
      gd_s(1, " should be="); gd_i(1, sizeof(gnutella_push)); gd_s(1, "\n");
      return -4;
    }

    gp=gpa->data;
    gd_s(3, "conn_hdl_packet push request\n");
    if (memcmp(gp->guid, conf_get_str("guid"), 16)==0) {
      gd_s(2, "conn_hdl_packet my own push request\n");
      gnut_push_new(gp);
      /* return because we don't want to route this sucker */
      return 0;
    }

    /* now we're going to see if this is in our routing table, if
     * it is, then we send to just that one and quit, otherwise we will
     * let it drop through into the regular routing code which will end
     * up broadcasting it */

    /* screw that, only do pushes that we saw the original query for
     * that should help congestion a bit */
    mgc = route_guid_find(gp->guid, &the_re);
    if (mgc) {
      memcpy(&pref, gp->ref, 4); pref = GUINT32_FROM_LE(pref);

      if (monitor_push) {
	guid_ref key;
	uint32 len;
	void * data;

	/* construct the key */
	memcpy(key.guid, gp->guid, 16);
	memcpy(key.ref, gp->ref, 4);

	/* look it up */
	pthread_mutex_lock(&qrepl_mutex);
	data = lru_copy(&qreplies, &key, sizeof(key), &len);
	pthread_mutex_unlock(&qrepl_mutex);
	if (data) {
	  /* printf("Found PUSH "); print_ascii((char *) data); printf("\n"); */
	  monitor_search_add(data, 0);
	  fre_v(&data, 246);
	} else {
	  /* printf("Didn't find PUSH\n"); */
	}
      }

      gd_s(2, "conn_hdl_packet routing PUSH\n");
      /* here we do the routing */
      gpa->gh.hops++;
      if ((gpa->gh.ttl != 0) && (--gpa->gh.ttl>0)) {
	send_to_one(mgc,gpa);
      } else {
	/* gc->dropped_packets++; */
      }
    }

    /* Always return -- if we didn't just send this packet, we drop it */
    return 0;

    break;
	
  case 0x80: /* ----------------------------------------------------- QUERY */
    /* QUERY */

    dqi(0x0004);

    buf = gpa->data;

    /* First make sure strlen will work */
    if (dlen > 0) {
      buf[dlen-1] = 0;
    }

    dqi(0x0017);

    /* Filter as per client recommendation */
    if (dlen > 257) { /* tagok */
      gd_s(1, "conn_hdl_packet Query packet too long dlen="); gd_i(1, dlen);
      gd_s(1, "\n");
      dqi(0x0018);
      return -4;
    }

    /* Test for query strings of 3 characters or less, e.g. "jpg".
       Such queries produce more query replies than the requestor
       can even store, much less display. So, we drop the packet in
       order to discourage such queries, and more importantly, avoid
       the flood of return query replies that would be routed through
       us.

       We also filter really long queries, which have appeared as spam
       lately. 

       Update, Aug 2001: We're still getting spam; most of it looks like
       full filenames copied from qreply packets. In other words, someone
       is taking every filename that arrives in a qreply and sending it out as
       a query string! */
    if (dlen > 80) {
      dqi(0x0019);
      /* spam packet. Suppress it by truncating its data */
      if (0) {
	printf("truncate long query '"); print_ascii(&buf[2]); printf("'\n");
      }
      /* buf[dlen-1] = 0; printf("truncate spam dlen %d to 0\n%s\n", dlen, &buf[2]); */
      /* Set data length to 3: 2 bytes for minspeed and 1 byte for null-string */
      dlen = 3; /* 0.4.27.c30 */
      dlen = GUINT32_TO_LE(dlen); /* 0.4.28.c10 */
      memcpy(gpa->gh.dlen, &dlen, 4);
      dlen = GUINT32_FROM_LE(dlen); /* 0.4.28.c10 */
      buf[2] = 0; /* null string */
      no_drop = 1;
    } else if (dlen <= (2 + 3 + 1)) {
      dqi(0x001a);
      return -5;
    } else if (strlen(&buf[2]) > dlen - 3) {
      dqi(0x001b);
      /* not a C string */
      /* printf("drop non-C-string query\n"); */
      return -5;
    } else if (strlen_an(&buf[2]) <= 3) {
      dqi(0x001c);
      /* Less than 3 "real" characters in query */
      /* printf ("drop dlen %i query '%s'\n", dlen-3, &buf[2]); */
      return -5;
    }
    gd_s(3, "conn_hdl_packet query request\n");

    dqi(0x001d);

    if (isnew) {
      dqi(0x001e);
      /* Run the query against our shared files */
      con_num_queries++;
      gd_s(3, "conn_hdl_packet new query request\n");
      buf[dlen-1] = 0;

      /* Drop any packets which are on our blacklist. */
      if (!gnut_blacklist_allow(BLACKLIST_TYPE_SEARCH, &buf[2], 0, 0, 0)) {
        dqi(0x001f);
	return 0;
      }

      dqi(0x0020);
#ifdef USE_DRR_SEARCH
      results = share_srch_hash(&buf[2], conf_get_int("max_results"),
				&n_reslts);
#else
      results = share_search(&buf[2], conf_get_int("max_results"),
			     &n_reslts);
#endif
      dqi(0x0021);

      /* printf("kept dlen %i query '%s'\n", dlen-3, &buf[2]); */
      
      if (monitor_query) {
	monitor_search_add(&buf[2], !!results);
      }

      dqi(0x0022);

      if (gc_search_log) {
	if (gc_search_log_all || n_reslts) {
	  fprintf(gc_search_log, "%d\t%d\t%s\n", (int) (time(0)), n_reslts,
		  &buf[2]);
	}
      }

      dqi(0x0023);

      if (results) {
        /* 0.4.27.c28 num_responses++ moved into gp_reply.make */

	gd_s(3, "conn_hdl_packet responding to query: "); gd_s(3, &buf[2]);
	gd_s(3, "\n");

	/* 0.4.27.c27 best_urate needs to be converted from bytes per second
	   (the units we calculated it in) to kbits per second (the units
	   it needs to have for the packet). We add 64 before dividing by
	   0128 so it'll round to nearest rather than rounding down */
	mgpa = gp_reply_make(gpa->gh.guid, results,
			     net_local_ip(), conf_get_int("local_port"),
			     ((best_urate + 64) >> 7), /* 0.4.27.c27 */
			     conf_get_str("guid"), conf_get_int("ttl"));

	send_to_one(gc, mgpa);
        
#ifdef USE_DRR_SEARCH
	share_hash_clear_list(results); 
#else
	share_clear_list(results);
#endif
        
	fre_v(&(mgpa->data), 247);
	fre_gpa(&mgpa, 248);
      }
    } else if (isnew == 0) {
      dqi(0x0024);
      /* This is the normal route for detecting small loops -- duplicate
       * packets get dropped. */
      gc->dropped_packets++;
    }
    dqi(0x0025);

    YPROBE(gpa->data, 162)
    break;

  case 0x81: /* ---------------------------------------------------- QREPLY */
    dqi(0x0005);

    /* HITS (also called Query Reply, QREPLY and QHD) */
    if (dlen<sizeof(gnutella_results)) {
      gd_s(1, "conn_hdl_packet QREPLY dlen="); gd_i(1, dlen);
      gd_s(1, " should be > "); gd_i(1, sizeof(gnutella_results));
      gd_s(1, "\n");
      return -7;
    }
    gd_s(3, "conn_hdl_packet handling QREPLY\n");
	
    grh = gpa->data;

    grs = (gnutella_results_suffix *) ((char *) gpa->data + dlen - 16);
    grn = (gnutella_results_name *)
      ((char *) gpa->data+sizeof(gnutella_results));

    /* 0.4.27.c22 */
    memcpy(&speed, grh->speed, 4); speed = GUINT32_FROM_LE(speed);

    /* we stick this into the routing table, so that it is possible to
     * route push request packets back to the place that originally sent
     * us the query */
    route_guid_add(grs->guid, gc);

    dqi(0x0119);

    /* Add all query replies to the qreply lru_hash. This is to support the
     * mpush command and the (future) instant-search feature */
    for (i=0; i<grh->num; i++)  {
      guid_ref key;
      int l;

      /* 0.4.27.c22 */
      /* Get current item's refnum and size */
      memcpy(&pref, grn->ref, 4); pref = GUINT32_FROM_LE(pref);
      memcpy(&psiz1, grn->size, 4); psiz1 = GUINT32_FROM_LE(psiz1);

      if (((char *)&grn->name - (char *)gpa->data) > dlen) {
	gd_s(1, "conn_hdl_packet bad response packet\n");
	return -8;
      }

      /* construct the key */
      memcpy(key.guid, grs->guid, 16);
      memcpy(key.ref, grn->ref, 4);

      /* Right now we don't construct the data, but in the future, we'll
       * be storing filename, IP, port, refnum and size in a structure. */

      l = strlen_an(grn->name);
      if ((l > 5) && (l < 80)) {
	int new;

        dqi(0x011a);

	/* Store it. */
	pthread_mutex_lock(&qrepl_mutex);
	new = lru_storecopy(&qreplies, &key, sizeof(key), grn->name,
			    strlen(grn->name)+1, 370);
	pthread_mutex_unlock(&qrepl_mutex);
        dqi(0x011b);
	if (new) {
	  if (monitor_reply) {
	    int accept;

	    accept = 0;
	    if (monitor_filter) {
	      if (keyword_match(monitor_filter, grn->name, 1, 0)) {
		accept = 1;
	      }
	    } else {
	      accept = 1;
	    }

	    if (accept) {
	      monitor_search_add(grn->name, 0);
	    }
	  }
	}
      }

      dqi(0x011c);
      /* Skip to the next one. Note the extra "+ 1", this is to skip the
         second null byte at the end of the name. */
      grn = (gnutella_results_name *)
	((char *) grn + sizeof(gnutella_results_name) + strlen(grn->name) + 1);
    }

    { /* 0.4.27.c21 Parse the extended info, if any */
      int32 x;
      uint8 *d;

      d = (uint8 *) grn;
      x = ((uint8 *) grs) - d;
      if (x > 0) {
	parse_eqhd(d, x, grh->ip, speed, &htype, &rating);
      }
    }

    dqi(0x011d);

    /* reset name pointer back to beginning */
    grn = (gnutella_results_name *)
      ((char *) gpa->data+sizeof(gnutella_results));

    /* Get the first item's refnum and size */
    memcpy(&pref, grn->ref, 4); pref = GUINT32_FROM_LE(pref); /* 0.4.27.c22 */
    memcpy(&psiz1, grn->size, 4); psiz1 = GUINT32_FROM_LE(psiz1);

    dqi(0x011e);

#ifdef GNUT_HTTP_SEARCH
    {
      int ret;

      memcpy(&ipt, &(grh->ip), sizeof(ipt));

      /* Drop any packets which are on our blacklist. */
      if (!gnut_blacklist_allow(BLACKLIST_TYPE_SEARCH | BLACKLIST_TYPE_HTTP,
				0, &ipt, 0, 0)) {
	return 0;
      }

      /* If this was requested through an HTTP request, append it to the
       * appropriate cache, and not to our interactive query list. */
      ret = gnut_http_result_append(grh->ip, grh->port, pref, speed,
				    psiz1, grn->name, gpa->gh.guid
				    , htype, rating /* 0.4.27.c21 */
				    );
      if (ret) {
	return 0;
      }
    }
#endif

    /* Drop any packets which are on our blacklist. */
    memcpy(&ipt, &(grh->ip), sizeof(ipt));
    if (!gnut_blacklist_allow(BLACKLIST_TYPE_SEARCH, 0, &ipt, 0, 0)) {
      return 0;
    }

    dqi(0x011f);

    /* If we have a query, look for matches. If strich_search is on, we
     * can match any query reply packet, getting more and quicker results */
    multi_enable = conf_get_int("multi_enable");
    if ((multi_enable && current_searches)
	|| (!multi_enable && current_query_packet)) {
      int match_guid;
      int acpt2;  /* 0.4.27.c21 */

      match_guid = 0;
      acpt2 = 1;  /* 0.4.27.c21 */

      dqi(0x0120);
      /* We now only compare the first 14 bytes of the query GUID, because
       * the last two are used for subsequent invocations of the same query
       * over new GnutellaNet connections. */
      match_guid = 0;
      if (multi_enable) {
	if ((msearch = gnut_search_find_by_guid(gpa->gh.guid))) {
	  match_guid = 1;
	}
      } else {
	match_guid = (memcmp(conf_get_str("query_guid"), gpa->gh.guid, 14)==0);
      }

      if (rating < gc_search_min_rating) {  /* 0.4.27.c21 */
        acpt2 = 0;
      }

      /* The logic here looks odd: match_guid means the GUID was just matched;
         gc_strict_search means the user wants us to double-check search
         results to see if they match the query they sent out. If
         strict_search is set we ignore the GUID, because we're going to check
         the search results against the search string ourselves, and if
         the search results contain a match, what does it matter if the GUID
         doesn't? We actually get a few more matches this way than we otherwise
         would. */
      if (acpt2 &&  /* 0.4.27.c21 */
	  (match_guid || gc_strict_search)) {
	char *exts;
	char *ourquery;
		
        dqi(0x0121);
	gd_s(3, "conn_hdl_packet its our query reply\n");
		
	ourquery = (current_query_packet->data)+2;
	exts = conf_get_str("search_extensions");
	for (i=0; i<grh->num; i++)  {
	  /* Make sure we haven't gone off the end of the packet */
	  if (((char *)&grn->name - (char *)gpa->data) > dlen) {
	    gd_s(1, "conn_hdl_packet bad response packet\n");
	    return -8;
	  }

	  /* get size */
	  memcpy(&psiz1, grn->size, 4); psiz1 = GUINT32_FROM_LE(psiz1);
		  
	  /* Match query reply against current query to see if it's really a
	   * match. %%% We want to make this code usable by the HTTP query too,
	   * but that will require using the GUID to find out what the query
	   * string was; also gnut_http_result_append() needs to be called
	   * on each query result (not just the first) */
	  accept = 0;
	  if (gc_strict_search) {
	    /* Make sure this filename matches what we were searching for */
	    if (multi_enable) {
	      /* gnut_search_find checks it against all the pending searches */
	      if ((msearch = gnut_search_find(name_tok_cb, grn->name))) {
		accept=1;
	      }
	    } else {
	      accept = keyword_match(ourquery, grn->name, 1, 1); 
	    }
	  } else if (match_guid) {
	    /* Strict search is off, but the packet matches our GUID so we
	     * accept all results */
	    accept = 1;
	  }

	  /* Ignore anything that cannot be transferred by either 'get'
	   * or 'put' */
	  if (host_ok_for_neither(grh->ip)) {
	    accept = 0;
	  }

	  /* Match query filename extension */
	  if (accept && exts) {
	    /* search_extensions must be a non-null string, and not starting
	     * with "*" which is the special value that causes all files to
	     * match */
	    if ((*exts) && (*exts != '*')) {
	      char *ext;

	      /* Now see if the file has an extension -- if not we'll skip
	       * the test */
	      ext = strrchr(grn->name, '.');
	      if (ext) {
		ext++;
				
		if (ext) {
		  /* Whew. We are actually ready to check for a match */
		  if (strstr(exts, ext)) {
		    /* Found, leave it alone */
		  } else {
		    /* Not found, ignore this query response entry */
		    accept = 0;
		  }
		}
	      }
	    }
	  }

	  /* Filter against minimum size */
	  if (accept && gc_search_min_size) {
            if (psiz1 < gc_search_min_size) {
	      accept = 0;
	    }
          }
		  
	  if (accept) {
	    memcpy(&pref, grn->ref, 4); pref = GUINT32_FROM_LE(pref);
			
	    query_add(grh->ip, grh->port, pref, speed, psiz1, grn->name,
		      grs->guid, htype, rating);

	    if (multi_enable) {
	      if (msearch) {
		msearch->responses++;
	      }
	    }
	  }

	  grn = (gnutella_results_name *)
	    ((char *) grn+sizeof(gnutella_results_name) + strlen(grn->name)+1);
	}

	if (match_guid) {
	  return 0;		
	}
      }
    }

    dqi(0x0122);
    /* If the cache is enabled and it's time to start another
     * download, then cache the first file in this query reply. */
    if (cache_time_for_update()) {
      uint32 pref2; /* 0.4.27.c11 (As above. I renamed to disambiguate) */
      uint32 psz2; /* 0.4.27.c11  (ditto) */

      gd_s(1, "conn_hdl_packet caching from query reply\n");
      grh = gpa->data;

      /* no loop... just cache the first in the replies (FIXME - Ray) */
      grn = (gnutella_results_name *) ((char *) gpa->data+sizeof(gnutella_results));
      if (((char *)&grn->name - (char *)gpa->data) > dlen) {
	gd_s(1, "conn_hdl_packet bad response packet\n");
	return -8;
      }
	  
      memcpy(&pref2, grn->ref, 4); pref2 = GUINT32_FROM_LE(pref2);
      memcpy(&psz2, grn->size, 4); psz2 = GUINT32_FROM_LE(psz2);

      dqi(0x0123);
      query_add_cache(grh->ip, grh->port, pref2, speed, psz2,
		      grn->name, grs->guid, htype, rating);

      return 0;
    }

    YPROBE(gpa->data, 169)

    break;

  default:
    dqi(0x0026);
    gd_s(1, "conn_hdl_packet packet type: "); gd_i(1, gpa->gh.func);
    gd_s(1, "\n");

    /* "spam" packet. (Actually, I've found that most of these are the
     * result of misaligned data being forwarded as a packet by a dumb
     * routine algorithm in another client). We suppress the packet
     * by forwarding it with no data -- thus its GUID will be cached,
     * but it won't take up a lot of bandwidth. */
    if(0) {
      printf("truncate spam packet type %i len %i data '", gpa->gh.func, dlen);
      if (dlen > 0) {
	print_asc_n(gpa->data, dlen);
      }
      printf("'\n");
    }
    gpa->gh.func = 0x80;
    dlen = 0;
    memcpy(gpa->gh.dlen, &dlen, 4);
    no_drop = 1;

    /* return here so we don't pass along bad packets... */
    /* return -9; */

    YPROBE(gpa->data, 185)
  }

  dqi(0x0027);

  gd_s(2, "conn_hdl_packet running routing logic\n");

  if (gpa->gh.func & 1) {
    dqi(0x0028);
    if (mgc == 0) {
      dqi(0x015d);
      gd_s(2, "conn_hdl_packet Routing error!\n");
      gc->routing_errors++;
      return 0;
    } else {
      dqi(0x015c);
      gd_s(2, "conn_hdl_packet Routing packet...\n");
      gpa->gh.hops++;
      if (gpa->gh.ttl && --gpa->gh.ttl>0) {
	send_to_one(mgc, gpa);
      } else {
	/* gc->dropped_packets++; */
      }
    }
  } else {
    dqi(0x0029);
    /* broadcast packets: ping or query (push request doesn't reach
     * here because case 0x40 returns rather than dropping through) */
    if (!mgc) {
      /* not in routing table, means not seen yet */

      gd_s(2, "conn_hdl_packet Spreading packet!\n");

      /* Add a broadcast packet (ping or query) to the routing table */
      dqi(0x0168);
      route_guid_add(gpa->gh.guid, gc);

      gpa->gh.hops++;
      if (gpa->gh.ttl && (--gpa->gh.ttl)>0) {
        dqi(0x0169);
        send_to_all_except(gc, gpa, no_drop);
      } else {
        /* gc->dropped_packets++; */
      }
    }
  }

  dqi(0x002a);

  gd_s(3, "conn_hdl_packet returning success\n");
  return 0;
}

/* loops through any packets available on the out stack,
 * and sends them in turn... */
int connection_send_packets(gcs *gc)
{
  gnutella_packet *gpa;
  int ret;
  fd_set fsw;
  int i=0;
  struct timeval tv;
  int sendit;

  gd_s(4, "connection_send_packets entering\n");

  while (gc->out_queue->size > 0) {
    gd_s(3, "connection_send_packets there are ");
    gd_i(3, gc->out_queue->size); gd_s(3, " packets left to send\n");

    FD_ZERO(&fsw);
    FD_SET(gc->gc_sock, &fsw);

    sendit = 1;

    /* Wait 1.5 seconds for socket to be writable. What this really means
       is we're waiting for the current I/O to complete. It takes longer
       if the connection at the other end is slow */
    tv.tv_sec = 1;
    tv.tv_usec = 500000;
    ret = select(gc->gc_sock+1, 0, &fsw, 0, &tv);

    if (ret == 0) {
      dqi(0x003c);
      sendit = 0;
    } else {
      dqi(0x003d);
    }

    /* we don't want anyone trying to change the out stack
     * while we're playing around with it... */
    pthread_mutex_lock(&gc->out_q_mutex);
    gpa = gnut_queue_remove(gc->out_queue);
    pthread_mutex_unlock(&gc->out_q_mutex);

    if (sendit) {
      dqi(0x003e);
      ret = send_packet(gc->gc_sock, gpa);
      dqi(0x0165);

      if (ret >= 0) {
	gd_s(4, "connection_send_packets wrote packet, size: ");
	gd_i(4, ret); gd_s(4, "\n");
	gc->snt_bytes += ((float64) ret);
	gc->rate_snt_bytes += ((float64) ret);
	gc->sent_packets++;
	g_snt_bytes += ((float64) ret);
	con_num_sent++;
      }
    
#ifdef MOREPACKETSTATS
      ++gc->out_packet_count[gpa->gh.func];
      gc->out_packet_byte_count[gpa->gh.func] += (ret - 23);
      ++gc->out_ttl_count[gpa->gh.ttl];
      if ((ret - 23) > gc->out_max_packet_size[gpa->gh.func]) {
	gc->out_max_packet_size[gpa->gh.func] = (ret - 23);
      }
#endif
    } else {
      dqi(0x003f);

      /* We're gonna just drop the packet. Better than dropping the
       * whole connection! */
      if (gc_debug_opts & 32) {
        printf("drop2\n");
      }
      gc->dropped_packets++;
      ret = 0;
    }

    if (gpa->data) {
      fre_v(&(gpa->data), 249);
    }
    fre_gpa(&gpa, 250);

    dqi(0x0040);

    if  (ret<0) {
      gd_s(1, "connection_send_packets send_packet error ");
      gd_s(1, strerror(errno));
      dqi(0x0041);
      return ret;
    }
    
    /* Limit to sending 5 packets at a time. This is so we can still receive stuff over the link... */
    if ((++i)>5) {
      dqi(0x0042);
      return 0;
    }
  }

  gd_s(4, "connection_send_packets returning success\n");

  dqi(0x0043);

  return 0;
}

/* this needs to read a packet if available, and it NEEDS to timeout
 * after 1/4 second or less
 * returns 0 on success or timeout
 * returns <0 on error */
int connection_receive_packet(gcs *gc)
{
  fd_set fsr;
  struct timeval tv;
  int ret;
  gnutella_packet * gpa;

  dqi(0x0047);
  
  gd_s(4, "conn_rcv_packet entering\n");
  
  FD_ZERO(&fsr);
  FD_SET(gc->gc_sock, &fsr);
  
  dqi(0x0048);
  ret = (int) (gc->gc_sock);
  dqi(0x0127);

  tv.tv_sec = 0;
  tv.tv_usec = 250000;
  ret = select(gc->gc_sock+1, &fsr, 0, 0, &tv);

  if (ret<=0) {
    dqi(0x0049);
    /*    perror("conn_rcv_packet, select"); */
    return ret;
  } else {
    dqi(0x004a);
  }
  
  gd_s(3,"conn_rcv_packet trying to read packet\n");
  gpa = (gnutella_packet *) ycaloc(sizeof(gnutella_packet), 1, 488);

  dqi(0x004b);
  
  gnut_mprobe(gpa);
  ret = read_packet(gc->gc_sock, gpa);

  dqi(0x004c);

  gnut_mprobe(gpa);

  dqi(0x0124);

  if ((ret<0) && (errno != EINTR)) {
    gd_s(1, "conn_rcv_packet read_packet returned:"); gd_i(1, ret);
    gd_s(1, "\n");
    if (gpa->data) {
      fre_v(&(gpa->data), 251);
    }
    fre_gpa(&gpa, 252);
    return ret;
  }
  
  if (ret>0) {
    /* read.packet succeeded and got ret bytes
       (where ret = 23 + packet payload length) */
    g_rcv_bytes += ((float64) ret);
    gc->rcv_bytes += ((float64) ret);
    gc->rate_rcv_bytes += ((float64) ret);
    gc->received_packets++;
    con_num_messages++;
  
    /* now we can do any actual processing on the packet! */

    dqi(0x0125);
    ret = connection_handle_packet(gc, gpa);
    dqi(0x002b);

    YPROBE(gpa->data, 159)
  
    gnut_mprobe(gc);
    gnut_mprobe(gpa);
    if (gpa->data) {
      gnut_mprobe(gpa->data);
    }

    dqi(0x002c);

    if (ret<0) {
      dqi(0x002d);
      gd_s(1, "conn_rcv_packet bad packet on tid: ");
#if defined(PTHREADS_DRAFT4) && defined(hpux)
      gd_li(1, ((uint32) ((HPUX_hack)gc->tid).longpart)); /* 0.4.28.c12 */
#else
      gd_li(1, ((uint32) gc->tid));
#endif
      gd_s(1, " with ret="); gd_i(1, ret); gd_s(1, "\n");
      gc->bad_packets++;
      if (gc->bad_packets>BAD_PACKET_MAX) {
        dqi(0x002e);
        gd_s(1,"conn_rcv_packet max bad packets reached on this connection\n");
        return -1;
      }
      gd_s(2, "conn_rcv_packet ret="); gd_i(2, ret); gd_s(2, "\n");
    }
  
    gd_s(3, "conn_rcv_packet trying to fr""ee gpa->data\n");
  
    if (gpa->data) {
      dqi(0x002f);
      gnut_mprobe(gpa->data);
      fre_v(&(gpa->data), 253); /* CANARY error (past end, alloc 0486) */
    }
  }
  gd_s(3,"conn_rcv_packet trying to fr""ee gpa\n");
  
  dqi(0x0030);
  gnut_mprobe(gpa);
  fre_gpa(&gpa, 254);
  
  gd_s(3, "conn_rcv_packet returning success\n");

  dqi(0x0031);
  
  return 0;
}


/* int connection_loop(gcs *gc)
 *
 * this is the main loop which handles sending and receiving
 * all packets over a connection.  It should be called after
 * all initial negotiation is complete.  When it returns, it should
 * be time to remove the connection from the connection
 * list.
 */
int connection_loop(gcs *gc)
{
  int ret=1;
  
  gd_s(3, "conn_loop entering\n");
  
  gc->cstate=STATE_CONNECTED;
  
  /* If there is a current query, we send it out the new connection.
   * The effect of this is that, if you issue a 'find' command right
   * after starting gnut, you'll get just as many results as you would
   * have gotten if you waited for all 4 (or however many) connections
   * to be up before doing the 'find'. This is good, because it means
   * you don't have to wait every time you launch gnut, or (even
   * worse :-) repeat the search command 'just to make sure'. */
  if (query_num() < conf_get_int("max_responses")) {
    pthread_mutex_lock(&query_packet_mutex);
    if (current_query_packet) {
      /* There is a query (find) in progress, and we have just established
       * a new connection into the network.
       *
       * Because this new connection gives us access to hosts that were
       * hitherto unreachable within the TTL, we want to send out the
       * query packet so those new hosts can get the query and respond to it.
       *
       * (For maximum search results, we would also have to change the GUID
       * in the query. If we don't, and if the host we've just connected to
       * has already seen the query (which is very likely, since we got the
       * host's IP address from a packet and therefore the host was within
       * TTL of a previous connection) it would just throw the packet away
       * and the packet would not reach our new hosts. Also, if a match
       * occurred, the results would get routed through the old path, but
       * we want them to come back on the new path. */
      if (gc_debug_opts & 2) {
	/* Changing the GUID isn't that great, because some people think it's
	 * spam */
	current_query_packet->gh.guid[14]++;
	if (current_query_packet->gh.guid[14] == 0) {
	  current_query_packet->gh.guid[15]++;
	}
      }
      send_to_one(gc, current_query_packet);
    }
    pthread_mutex_unlock(&query_packet_mutex);
  }

  /* If for some reason our host list is empty or nearly empty, we send a
   * PING packet. This will happen most often with new users trying to
   * start up without installing a .gnut_hosts file. */
  {
    uint h_num;
    int do_ping;

    do_ping = 0;

    host_totals(&h_num, 0, 0);
    if (h_num < 10) {
      do_ping = conf_get_int("ttl");
      /* Limit TTL to 4 so we only bother about 4 x 3^3 hosts */
      if (do_ping > 4) {
        do_ping = 4;
      }
    }

    /* Always send a PING with at least a TTL 1. This is required for
     * some of the newer, noncompliant servents like Limewire and
     * BearShare. */
    if (do_ping < 1) {
      do_ping = 1;
    }

    if (do_ping) {
      gnutella_packet *gpa;
  
      if (0) {
        gpa = gp_ping_make(conf_get_str("mac"), conf_get_int("ttl"));
      } else {
        gpa = gp_ping_make(conf_get_str("mac"), do_ping);
      }

      conf_set_str_len("update_guid", gpa->gh.guid, 16);
	
      send_to_one(gc, gpa);
	
      if (gpa->data) {
	fre_v(&(gpa->data), 255);
      }
      fre_gpa(&gpa, 256);
    }
  }
  
  while((gc->cstate==STATE_CONNECTED) && (ret >= 0)) {
    dqi(0x0044);
    ret = connection_receive_packet(gc);
    dqi(0x0034);

    gd_s(3, "conn_rcv_packet returned: ");
    gd_i(3, ret);
    gd_s(3, "\n");
    if (ret<0) {
      dqi(0x0035);
      /* the line is closed, we must be done! */
      gd_s(1, "conn_rcv_packet returned: ");
      gd_i(1, ret);
      gd_s(1, "\n");
    }
	
    if (ret >= 0) {
      dqi(0x0036);
      /* send all queued up packets */
      gd_s(3, "conn_rcv_packet about to call csp with gc=");
      gd_p(3, gc);
      gd_s(3, "\n");
      ret = connection_send_packets(gc);
      gd_s(3, "conn_rcv_packet csp returned: ");
      gd_i(3, ret);
      gd_s(3, "\n");
      if (ret < 0) {
        dqi(0x0037);
      }
      gd_s(3, "conn_rcv_packet gc->cstate is readable\n");
    }
  }
  
  if (ret >= 0) {
    dqi(0x0038);
    /* We exited on cstate */
  } else {
    dqi(0x0039);
  }

  gd_s(3, "conn_loop returning success\n");
  return 0;
}

/* calls function a for every connection in the list */
int gnut_connection_enumerate( int (*a)(gcs *))
{
  Gnut_List *gl;
  
  pthread_mutex_lock(&gc_list_mutex);
  
  for (gl=gc_list;gl;gl=gl->next) {
    if ((*a)(gl->data)==-1) {
      break;
    }
  }
  
  pthread_mutex_unlock(&gc_list_mutex);
  return 0;
}

/* kill the oldest incoming connection */
int gnut_connection_kill_oldest()
{
  Gnut_List *gl;
  gcs *gc=0;
  gcs *gtc;
  int going;
  
  pthread_mutex_lock(&gc_list_mutex);
  
  going = 1;
  for (gl=gc_list; gl && going; gl=gl->next) {
    gtc=gl->data;
    if ((gtc->ctype==TYPE_INCOMING) && (gtc->cstate==STATE_CONNECTED)) {
      gc=gtc;
      going = 0;
    }
  }

  if (gc) {
    /* The instance of connection_loop() handling this connection
     * will notice this and actually close the connection */
    gc->cstate = STATE_DEAD;
  }

  pthread_mutex_unlock(&gc_list_mutex);

  return 0;
}

/* droprate returns ratio of two longs, used to compute
   ratio of dropped to total received packets. Result is an
   8-bit fractional number: 0256 means 1.0 or above; 0 means 0.0;
   in-between is a fraction, for example 0192 means 0.75 */
long droprate(long drop, long recv)
{
  long rv;

  if ((recv == 0) || (drop == 0)) {
    rv = 0;
  } else if (drop > recv) {
    rv = 256;
  } else if (drop > 65536L) {
    rv = drop / (recv >> 8L);
  } else {
    rv = (drop << 8L) / recv;
  }

  /* convert to a percentage */
  rv = ((rv * 100) + 128) / 256; /* tagok */

  return rv;
}

/* kill the worst (by dropped packet percentage) connection */
int gnut_connection_kill_worst(int interact)
{
  Gnut_List *gl;
  gcs *worst_gc=0;
  gcs *gtc;
  long this_dr;
  long worst_dr;
  int killed_ip[4];

  pthread_mutex_lock(&gc_list_mutex);

  worst_dr = conf_get_int("autokill_thres");
  for (gl=gc_list; gl; gl=gl->next) {
    gtc=gl->data;
    if (((gtc->ctype==TYPE_OUTGOING) || (gtc->ctype==TYPE_INCOMING))
         && (gtc->cstate==STATE_CONNECTED)
         && ((gtc->sent_packets + gtc->received_packets + gtc->dropped_packets) > 1000))
    /* %%%RPM also want to test for really low transfer/sec stats. It
     * needs to measured relative to other connections, to auto-compensate
     * for our ISP speed, which may vary with time (cable modems) */
    {
      this_dr = droprate(gtc->dropped_packets, gtc->received_packets);

    /* 20000611 change > to >=; break ties by last found rather than first; (this is newest connection after reversing order of connection list insertion) */
      if (this_dr >= worst_dr) {
        worst_dr = this_dr;
        worst_gc=gtc;
      }
    }
  }
  if (worst_gc) {
    /* record the IP of the connection we killed (no printf inside mutex) */
    killed_ip[0] = worst_gc->ip.b[0];
    killed_ip[1] = worst_gc->ip.b[1];
    killed_ip[2] = worst_gc->ip.b[2];
    killed_ip[3] = worst_gc->ip.b[3];

    /* set flag to kill connection:
     * the instance of connection_loop() handling this connection
     * will notice this and actually close the connection */
    worst_gc->cstate = STATE_DEAD;
  }

  pthread_mutex_unlock(&gc_list_mutex);

  if (interact) {
    /* indicate which , if any, we killed */
    if (worst_gc) {
      printf(UI_CN_KILLING, killed_ip[0],
			 killed_ip[1], killed_ip[2], killed_ip[3]);
    } else {
      printf(UI_CN_NO_WORST);
    }
  }

  return 0;
}
