/* -*-C-*-

   Josh Pieper, (c) 2000
   Robert Munafo, (c) 2001

   This file is distributed under the GPL, see file COPYING for details

 cli: command-line interface

 The main loop for inputting commands is command_loop() in this file;
 the actual main() is at the end of this file.

 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#ifndef WIN32
# include <unistd.h>
#endif
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <pthread.h>

#ifdef HAVE_READLINE
# include <readline/readline.h>
# include <readline/history.h>
#endif

#ifdef HAVE_REGEX_H
# include <regex.h>
#endif

#ifdef HAVE_TERMIO_H
# include <termio.h>
#endif

#ifdef HAVE_TERMCAP_H
# include <termcap.h>
#else
# ifdef HAVE_TERM_H
#   include <curses.h>
#   include <term.h>
# endif
#endif

#include "cli_readline.h"
#include "conf.h"
#include "connection.h"
#include "hash.h"
#include "lib.h"
#include "list.h"
#include "threads.h"
#include "ui.h"
#include "gnut.h"
#include "host.h"
#include "route.h"
#include "share.h"
#include "cache.h"
#include "qry.h"
#include "monitor.h"
#include "transfer.h"
#include "net.h"
#include "cli_output.h"
#include "cli.h"
#include "blacklist.h"
#include "player.h"
#include "prompt.h"

#if GNUT_HTTP_FILE_LIST || GNUT_HTTP_SEARCH
# include "http.h"
#endif

int com_blacklist(char *),
  com_clear(char *), com_cls(char *),
  com_debug(char *),
  com_eval(char *),
  com_find(char *), com_forget(char *),
  com_get(char *),
  com_help(char *), com_hosts(char *),
  com_info(char *),
  com_kill(char *),
  com_last(char *), com_lclear(char *), com_limit(char *),
    com_list(char *), com_load(char *), com_log(char *),
    com_lshare(char *),
  com_monitor(char *), com_mpush(char *), com_mreply(char *),
  com_open(char  *),
  com_play(char *), com_player(char *), com_push(char *),
  com_quit(char *),
  com_response(char *),
  com_save(char *), com_scan(char *), com_set(char *),
    com_share(char *), com_shell(char *), com_sleep(char *),
    com_stop(char *),
  com_update(char *), com_version(char *);

int recu_level;
int recu_disable;
int recu_count;
int readline_firsttime;
int readline_crashed;

char * ratings[] = { /* 0.4.27.c21 */
  "    ", ".   ", "*   ", "*.  ", "**  ", "**. ", "*** ", "***.", "****"
};

COMMAND commands[] = {
  {"blacklist", com_blacklist, "Blacklist IP addresses or patterns.", "
blacklist *args* - Add one or more hosts, port numbers, subnets,
or search reply strings to the blacklist. All Gnutella nodes on the
blacklist, and all query replies that match the blacklist will be
ignored. Used properly, this can eliminate lots of spam. (See also
the *search_extensions* and *strict_search* variables)

Syntax of the arguments is:

 blacklist [ host *hostname* [subnet *bits*]] [search *string*]
           [ all | ping | search | incoming | outgoing | http ] 

(all should be on one line)

Examples:

  blacklist host 1.2.3.4 all        Blacklists one host: All queries, replies,
                                    connection attempts, etc. from that host
                                    will be ignored.

  blacklist host 1.2.3.4 incoming   Prevents incoming connections from
                                    one host, but other activity (searches,
                                    etc) involving that host is still allowed.

  blacklist host 1.2.3.1 subnet 8   Blacklists everyone from 1.2.3.1 to
                                    1.2.3.255 (the '8' refers to the last 8
                                    bits of the IP address).

  blacklist search hitler search    Stops any search that includes 'hitler'
                                    from passing through this server.
"
},
  {"clear",     com_clear,  "Clear finished transfer(s).", "
clear *range* - Removes finished transfers from the transfer list. If
no range is given, all completed transfers will be removed. You
can also set the 'auto_clear' variable to make this automatic.
Range numbers work the same way as for the 'stop' command.
"},
  {"cls",       com_cls,	"Clears the screen.", 0},
  {"debug",     com_debug,  "Do whatever I feel like today.", "
debug - A command used by Josh to try out and/or debug new parts of
gnut. Since version 0.3.29 it has shown a list of shared files
that match a query for 'mpg'.
"},
  {"eval",      com_eval,   "Run a command through a shell, then execute its output as gnut commands.", "
eval *cmd* *args* - Evaluate shell escape: runs a shell and sends a
command to it, then executes its output as gnut commands.
(UNIX users only) Here is an example (IMPORTANT: This should all be
typed as a single line):

   eval perl -e \"$bl=\\\"gnut_settings\\\";
          system \\\"wget -q http://10.1.2.3:8080/$bl\\\";
          open FILE, \\\"$bl\\\" or die \\\"Couldn't open $bl\\\";
          print \\\"load $bl\\\";\"

This example uses 'wget' (type '! man wget' to see if you have this
tool) to try to get a file called 'gnut_settings' from an HTTP
server called '10.2.3.4:8080', and if the file is successfully
loaded, runs the gnut command 'load gnut_settings'. This method can
be used inside an organization like a school or company to load a
centralized 'gnut' settings file, or perhaps an up-to-date host
list.

An 'eval' command can output multiple lines; all of which will
be interpreted as gnut commands.
"},
  {"exit",      com_quit,   "Synonym for 'quit'.", 0},
  {"find",      com_find,   "Search the Gnutella network.", "
find *string* - Search the Gnutella network for *string*. The string
you supply is is sent as-is to the network. Nearly every Gnutella node
(servent) specifies this as a boolean AND search with all
non-alphanumeric characters ignored and ignoring upper/lower case. If
you want to force *all* searches to be boolean AND matches, set the
'strict_search' configuration variable. If you want to force
non-alphanumeric characters to be ignored, simply leave all
non-alphanumeric characters out of your search string.

'find' with no arguments displays search responses. If you wait a while
and repeat, you usually have more responses.
"},
  {"forget",    com_forget, "Remove item(s) from search responses list.", "
forget *range* - Remove search responses referenced by *range* from the
search responses list. You won't see the specified items again until you
issue a new 'search' command, but if multi-searching is enabled (by settimg
the \"multi_enable\" configuration flag) then the 'forget' items stay forgotten
until you use 'lclear' to clear the whole response list.

NOTE: The number you give 'forget' is the number from the latest 'response'
command, (or 'find' or 'search' with no argument). Examples:

    forget 1                - Forget first file
    forget 3-6              - Forget 3,4,5,6
    forget 2,10-12          - Forget 2,10,11,12
"},
  {"get",       com_get,    "Start download(s).", "
get *range* - Start downloading files referenced by *range*. If more
downloads are requested than allowed by max_uploads, then the
downloads are queued.

NOTE: The number you give 'get' is the number from the latest 'response'
command, (or 'find' or 'search' with no argument). Examples:

    get 1                   - Get first file
    get 3-6                 - Get 3,4,5,6
    get 2,10-12             - Get 2,10,11,12
"},
  {"help",      com_help,   "Display help.", "
help *command* - Describe a command and its options (sometimes with
examples)

help - Show a one-line summary for all commands.
"},
  {"hosts",     com_hosts,  "Display host catcher.", "
hosts - Displays all known hosts. NOTE: This can result in a lot
of information!

hosts *file* - Reads a file containing host addresses, which should
be in 'gnutella.txt' format.
"},
  {"info",      com_info,   "Display various information.", "
info [hqnsct] - Display various information. Any combination of the
following may be given:
     h - host totals
     q - queries received and replies sent
     n - network traffic totals
     s - shared files on this machine
     c - connections (GnutellaNet only)
     t - transfers in progress
     u - upload transfers in progress
     d - download transfers in progress
"},
  {"kill",      com_kill,   "Terminate connection(s).", "

kill *range* - Terminate the gnutella connections referenced by
*range*. The numbers you give the 'kill' command come from the 'info'
or 'info c' command. There is also the special parameter 'w' for
killing the worst connection. Examples:

    kill 1                  - Kill connection #1
    kill w                  - Kill the connection with the worst
                              dropped-duplicate packets statistic.
    kill 3-6                - Kill connections 3,4,5,6
    kill 2,10-12            - Kill connections 2,10,11,12
"},
  {"last",    com_last, "Show last turd.", "
last - Shows message from last failed download
"},
  {"lclear",    com_lclear, "Clear search list.", "
lclear - Clear all active searches and responses.

lclear GUID
lclear \"string\" - Clear an active search (the GUID is as shown in
the 'list' command)

lclear only works if multi-searching is enabled (by settimg the
\"multi_enable\" configuration flag)

Examples:

    lclear 4124          # removes the search with GUID 4124
    lclear 'grateful'    # removes the search for \"grateful\"
    lclear \"she's\"       # removes the search for \"she's\"
    lclear               # removes all searches and responses
"},
  {"limit",     com_limit,  "Impose bandwidth limits on transfers.", "
limit *num* *range* - Limit the transfers referenced by *range* to
*num* bytes per second.  The numbers you give the 'limit' command
come from the 'info' or 'info t' command. Examples:

    limit 1500 1            - Limit transfer #1 to 1500 bytes per second
    limit 100               - Limit *all* transfers to 100 bytes per second
    limit 2000 2,10-12      - Limit transfers 2,10,11,12 to 2000 bytes
                              per second.
"},
  {"list",      com_list,   "List all current searches.", "
list - List the active searches (if multi-searching is enabled by
setting the \"multi_enable\" configuration flag).
"},
  {"load",      com_load,   "Load a configuration file.", "
load *file* - load a file containing gnut commands. This is a useful
way to automate certain complex actions. For example, I have a file
that contains the following:

    # close down the GnutellaNet
    set max_uploads 0
    set min_connections 0
    set max_incoming 0
    kill
    # Prevent new uploads
    set max_uploads 0
    # Unthrottle all current uploads
    set default_upload_cap 0    
    limit 0
    # see what's loading right now
    info t

This file is called 'shutdown'. When I type 'load shutdown', it runs
all these commands. It closes all Gnutelllanet connections, prevents
any new uploads from starting and removes any transfer speed limits
(if any) on current uploads. I do this just before I'm about to quit
gnut, to allow uploads to finish as soon as possible without new
uploads starting; this enables me to quit without cutting them off.
"}, 
  {"log",       com_log,    "Set logging level.", "
log *num* - Set the current log level to *num*. This prints debugging
info. Bigger numbers produce more; anything above 2 is asking for
trouble!
"},
  {"lshare",    com_lshare, "List shared files.", "
lshare - list pathnames and refnums of all shared files.
"},
  {"monitor",   com_monitor,"Monitor search requests.", "
monitor *keywords* - Monitor incoming search queries. If arguments
are supplied, they are used for a boolean AND match (ignoring case)
before displaying search queries. Example:

    monitor                  - Display all searches (lots of output)
    monitor fred jpg         - Only display searches that contain both
                               'fred' and 'jpg'.
"},
  {"ls",    com_list,   "Synonym for 'list'.", "
ls - synonym for 'list' (type 'help list')
"},
  {"mpush",     com_mpush,  "Monitor push requests routed to others.", "
mpush *keywords* - Monitor push requests issued by others. While
*mreply* lets you see what files are available on the network,
*mpush* lets you find out which of those files are actually in demand.

NOTE: Push requests are kind of rare, and gnut cannot always identify the
file being requested. So, you won't see much when you use 'mpush'. Try it
sometime when you can let it run for an hour or two.
"},
  {"mreply",    com_mreply, "Monitor search replies routed to others.", "
mreply *keywords* - Monitor replies to search queries that others
have issued. While *monitor* lets you see what people are searching for,
*mreply* lets you find out what's actually available on the network.
As with *monitor*, arguments can be given to specify a
boolean AND match (ignoring case)  before displaying filenames.
"},
  {"open",      com_open,   "Open a new connection.", "
open *host*:*port* - Open outgoing connection to *host*. Examples:

    open 127.0.0.1:5346      - Create a connection to yourself (useful
                               for searching your own files)
    open 10.23.45.67         - Default port number is 6346, the normal
                               Gnutella port number.
    open nonexist.site:1234  - You can give host name instead of IP
"},
  {"play",      com_play,   "Start play_prog on selected queries.", 0},
  {"player",    com_player, "Player for downloads", 0},
  {"push",      com_push,   "Request a push connection.", "
push *range* - Same as 'get', however only a push connection is
attempted.

NOTE: If the IP address of the requested file is not push-able, or if
it is accessible via a normal 'get', gnut will complain. You can turn on
the 'no_rfc_1597' variable to stop this complaint.
"},
  {"quit",      com_quit,   "Quit.", 0},
  {"response",  com_response,"Show current query responses.", "
response [regexp] - Show the current query responses which match
the (optional) regular expression. The responses are sorted according
to the 'sort_order' configuration variable. Examples:

    response                 - Show all responses
    response fred            - Show only responses containing 'fred'
"},
  {"save",      com_save,   "Save configuration variables.", "
save - Saves all your current variable settings to your .gnutrc file,
your host list to .gnut_hosts, and your blacklist settings to
.gnut_blacklist. WARNING: This overwrites .gnutrc, if you have any
commands in .gnutrc other than 'set' commands, they will be lost.
"},
  {"scan",      com_scan,   "Scan share paths to make index of shared files.", "
scan - Scan the path or paths set in the share_paths variable for
files to share. The follow_symlinks variable tells whether to follow
symbolic links. #scan# tries to the 'find' command, or 'perl' if 'find'
does not exist. Any '.gnut' files (partial downloads) are ignored.
"},
  {"search",    com_find,   "Synonym for 'find'.", "
search - synonym for 'find' (type 'help find')
"},
  {"set",       com_set,    "Change configuration values.", "
set *name* *val* - Sets a named configuration variable to a value.

set *name* - Display value of a named configuration variable.

set - Display values of all configuration variables.
"},
  {"share",     com_share,  "Set share paths.", "
share *paths* - Takes a ':' delimited list of paths to share. (';'
on Win32). After changing this, you must use the 'scan' command for
the files to be indexed and appear to other users' searches.
"},
  {"?",         com_help,   "Synonym for 'help'.", "
? - synonym for 'help' (type 'help help')
"},
  {"!",         com_shell,  "Executes a command or drops to shell.", "
! *cmd* *args* - Shell escape: runs a shell and sends a command to it
(UNIX users only) Examples:

    ! ls                    - List your current directory
    ! cd dir1               - This will *not* have any lasting effect
    ! ls dir1               - List files in directory 'dir1'
    ! cd dir1 ; ls          - multiple shell commands work
    ! mpg123 gdead.mp3 &    - play an MP3 file in the background
"},
  {"shell",     com_shell,  "Synonym for '!'.", "
shell - synonym for ! (type 'help !')
"},
  {"sleep",     com_sleep,  "Sleep for n seconds.", "
sleep *n* - Do nothing (sleep) for n seconds.
"},
  {"stop",      com_stop,   "Stop transfer(s).", "
stop [*type*] 0|*range* - Stop the file transfers referenced by
*range*. The numbers you give the 'stop' command come from the 'info'
command. *type* g, p, d, u limits the command to stop only gets,
pushes, all downloads or all uploads respectively. At most one letter
can be used and it must be first. If '0' appears alone or at the end
of a range, all zero-byte transfers are stopped. '0' elsewhere in a
range makes the following numbered range(s) apply only to zero-byte
transfers. Examples:

    stop                    - Stop all file transfers
    stop d                  - Stop all downloads
    stop 0                  - Stop all transfers that have transferred no
                              data
    stop p 0                - Stop any push downloads that have transferred
                              no data
    stop 1                  - Stop first transfer
    stop u 3-6              - Stop any uploads with numbers from 3 to 6
    stop 2,0,10-12          - Stop number 2 and any in the range 10-12 that
                              have transferred no data
"},
  {"update",    com_update, "Update the current host list.", "
update - Send out ping packets to all connected hosts. If
configuration variable 'update_clear' is set, then the current host
list is discarded as well.
"},
  {"version",   com_version,"Display version information.", 0},
  {0, 0, 0, 0}
};

/* 0.4.27.c14 */
void gpl_top(void)
{
  cli_out_lines("                    GNU GENERAL PUBLIC LICENSE",
                "                       Version 2, June 1991",
                " ",
                " Copyright (C) 1989, 1991 Free Software Foundation, Inc.",
                "                          675 Mass Ave, Cambridge, MA 02139, USA",
                " Everyone is permitted to copy and distribute verbatim copies");
  cli_out_lines(" of this license document, but changing it is not allowed.",
                " ",
                "                            Preamble",
                " ",
                "  The licenses for most software are designed to take away your",
                "freedom to share and change it.  By contrast, the GNU General Public");
  cli_out_lines("License is intended to guarantee your freedom to share and change free",
                "software--to make sure the software is free for all its users.  This",
                "General Public License applies to most of the Free Software",
                "Foundation's software and to any other program whose authors commit to",
                "using it.  (Some other Free Software Foundation software is covered by",
                "the GNU Library General Public License instead.)  You can apply it to");
  cli_out_lines("your programs, too.",
                " ",
                "  When we speak of free software, we are referring to freedom, not",
                "price.  Our General Public Licenses are designed to make sure that you",
                "have the freedom to distribute copies of free software (and charge for",
                "this service if you wish), that you receive source code or can get it");
  cli_out_lines("if you want it, that you can change the software or use pieces of it",
                "in new free programs; and that you know you can do these things.",
                " ",
                "  To protect your rights, we need to make restrictions that forbid",
                "anyone to deny you these rights or to ask you to surrender the rights.",
		"These restrictions translate to certain responsibilities for you if you");
  cli_out_lines("distribute copies of the software, or if you modify it.",
                " ",
                "  For example, if you distribute copies of such a program, whether",
                "gratis or for a fee, you must give the recipients all the rights that",
                "you have.  You must make sure that they, too, receive or can get the",
                "source code.  And you must show them these terms so they know their");
  cli_out_lines("rights.",
                " ",
                "  We protect your rights with two steps: (1) copyright the software, and",
                "(2) offer you this license which gives you legal permission to copy,",
                "distribute and/or modify the software.",
                " ");
  cli_out_lines("  Also, for each author's protection and ours, we want to make certain",
                "that everyone understands that there is no warranty for this free",
                "software.  If the software is modified by someone else and passed on, we",
                "want its recipients to know that what they have is not the original, so",
                "that any problems introduced by others will not reflect on the original",
                "authors' reputations.");
  cli_out_lines(" ",
                "  Finally, any free program is threatened constantly by software",
                "patents.  We wish to avoid the danger that redistributors of a free",
                "program will individually obtain patent licenses, in effect making the",
                "program proprietary.  To prevent this, we have made it clear that any",
                "patent must be licensed for everyone's free use or not licensed at all.");
  cli_out_lines(" ",
                "  The precise terms and conditions for copying, distribution and",
                "modification follow.",
                " ", 0, 0);
}

/* 0.4.27.c14 */
void gpl_copying(void)
{
  cli_out_lines("                    GNU GENERAL PUBLIC LICENSE",
                "   TERMS AND CONDITIONS FOR COPYING, DISTRIBUTION AND MODIFICATION",
                " ",
                "  0. This License applies to any program or other work which contains",
                "a notice placed by the copyright holder saying it may be distributed",
                "under the terms of this General Public License.  The \"Program\", below,");
  cli_out_lines("refers to any such program or work, and a \"work based on the Program\"",
		"means either the Program or any derivative work under copyright law:",
		"that is to say, a work containing the Program or a portion of it,",
		"either verbatim or with modifications and/or translated into another",
		"language.  (Hereinafter, translation is included without limitation in",
		"the term \"modification\".)  Each licensee is addressed as \"you\".");
  cli_out_lines(" ",
		"Activities other than copying, distribution and modification are not",
		"covered by this License; they are outside its scope.  The act of",
		"running the Program is not restricted, and the output from the Program",
		"is covered only if its contents constitute a work based on the",
		"Program (independent of having been made by running the Program).");
  cli_out_lines("Whether that is true depends on what the Program does.",
		" ",
		"  1. You may copy and distribute verbatim copies of the Program's",
		"source code as you receive it, in any medium, provided that you",
		"conspicuously and appropriately publish on each copy an appropriate",
		"copyright notice and disclaimer of warranty; keep intact all the");
  cli_out_lines("notices that refer to this License and to the absence of any warranty;",
		"and give any other recipients of the Program a copy of this License",
		"along with the Program.",
		" ",
		"You may charge a fee for the physical act of transferring a copy, and",
		"you may at your option offer warranty protection in exchange for a fee.");
  cli_out_lines(" ",
		"  2. You may modify your copy or copies of the Program or any portion",
		"of it, thus forming a work based on the Program, and copy and ",
		"distribute such modifications or work under the terms of Section 1",
		"above, provided that you also meet all of these conditions:",
		" ");
  cli_out_lines("    a) You must cause the modified files to carry prominent notices",
		"    stating that you changed the files and the date of any change.",
		"",
		"    b) You must cause any work that you distribute or publish, that in",
		"    whole or in part contains or is derived from the Program or any",
		"    part thereof, to be licensed as a whole at no charge to all third");
  cli_out_lines("    parties under the terms of this License.",
		" ",
		"    c) If the modified program normally reads commands interactively",
		"    when run, you must cause it, when started running for such",
		"    interactive use in the most ordinary way, to print or display an",
		"    announcement including an appropriate copyright notice and a");
  cli_out_lines("    notice that there is no warranty (or else, saying that you provide",
		"    a warranty) and that users may redistribute the program under",
		"    these conditions, and telling the user how to view a copy of this",
		"    License.  (Exception: if the Program itself is interactive but",
		"    does not normally print such an announcement, your work based on",
		"    the Program is not required to print an announcement.)");
  cli_out_lines(" ",
		"These requirements apply to the modified work as a whole.  If ",
		"identifiable sections of that work are not derived from the Program,",
		"and can be reasonably considered independent and separate works in",
		"themselves, then this License, and its terms, do not apply to those",
		"sections when you distribute them as separate works.  But when you");
  cli_out_lines("distribute the same sections as part of a whole which is a work based",
		"on the Program, the distribution of the whole must be on the terms of",
		"this License, whose permissions for other licensees extend to the",
		"entire whole, and thus to each and every part regardless of who wrote it.",
		" ",
		"Thus, it is not the intent of this section to claim rights or contest");
  cli_out_lines("your rights to work written entirely by you; rather, the intent is to",
		"exercise the right to control the distribution of derivative or",
		"collective works based on the Program.",
		" ",
		"In addition, mere aggregation of another work not based on the Program",
		"with the Program (or with a work based on the Program) on a volume of");
  cli_out_lines("a storage or distribution medium does not bring the other work under",
		"the scope of this License.",
		" ",
		"  3. You may copy and distribute the Program (or a work based on it,",
		"under Section 2) in object code or executable form under the terms of",
		"Sections 1 and 2 above provided that you also do one of the following:");
  cli_out_lines(" ",
		"    a) Accompany it with the complete corresponding machine-readable",
		"    source code, which must be distributed under the terms of Sections",
		"    1 and 2 above on a medium customarily used for software interchange; or",
		" ",
		"    b) Accompany it with a written offer, valid for at least three");
  cli_out_lines("    years, to give any third party, for a charge no more than your",
		"    cost of physically performing source distribution, a complete",
		"    machine-readable copy of the corresponding source code, to be",
		"    distributed under the terms of Sections 1 and 2 above on a medium",
		"    customarily used for software interchange; or,",
		" ");
  cli_out_lines("    c) Accompany it with the information you received as to the offer",
		"    to distribute corresponding source code.  (This alternative is",
		"    allowed only for noncommercial distribution and only if you",
		"    received the program in object code or executable form with such",
		"    an offer, in accord with Subsection b above.)",
		" ");
  cli_out_lines("The source code for a work means the preferred form of the work for",
		"making modifications to it.  For an executable work, complete source",
		"code means all the source code for all modules it contains, plus any",
		"associated interface definition files, plus the scripts used to",
		"control compilation and installation of the executable.  However, as a",
		"special exception, the source code distributed need not include");
  cli_out_lines("anything that is normally distributed (in either source or binary",
		"form) with the major components (compiler, kernel, and so on) of the",
		"operating system on which the executable runs, unless that component",
		"itself accompanies the executable.",
		" ",
		"If distribution of executable or object code is made by offering");
  cli_out_lines("access to copy from a designated place, then offering equivalent",
		"access to copy the source code from the same place counts as  ",
		"distribution of the source code, even though third parties are not",
		"compelled to copy the source along with the object code.",
		" ",
		"  4. You may not copy, modify, sublicense, or distribute the Program");
  cli_out_lines("except as expressly provided under this License.  Any attempt ",
		"otherwise to copy, modify, sublicense or distribute the Program is",
		"void, and will automatically terminate your rights under this License.",
		"However, parties who have received copies, or rights, from you under",
		"this License will not have their licenses terminated so long as such",
		"parties remain in full compliance.");
  cli_out_lines(" ",
		"  5. You are not required to accept this License, since you have not",
		"signed it.  However, nothing else grants you permission to modify or",
		"distribute the Program or its derivative works.  These actions are",
		"prohibited by law if you do not accept this License.  Therefore, by",
		"modifying or distributing the Program (or any work based on the");
  cli_out_lines("Program), you indicate your acceptance of this License to do so, and",
		"all its terms and conditions for copying, distributing or modifying",
		"the Program or works based on it.",
		" ",
		"  6. Each time you redistribute the Program (or any work based on the",
		"Program), the recipient automatically receives a license from the");
  cli_out_lines("original licensor to copy, distribute or modify the Program subject to",
		"these terms and conditions.  You may not impose any further   ",
		"restrictions on the recipients' exercise of the rights granted herein.",
		"You are not responsible for enforcing compliance by third parties to",
		"this License.",
		" ");
  cli_out_lines("  7. If, as a consequence of a court judgment or allegation of patent",
		"infringement or for any other reason (not limited to patent issues),",
		"conditions are imposed on you (whether by court order, agreement or",
		"otherwise) that contradict the conditions of this License, they do not",
		"excuse you from the conditions of this License.  If you cannot",
		"distribute so as to satisfy simultaneously your obligations under this");
  cli_out_lines("License and any other pertinent obligations, then as a consequence you",
		"may not distribute the Program at all.  For example, if a patent",
		"license would not permit royalty-free redistribution of the Program by",
		"all those who receive copies directly or indirectly through you, then",
		"the only way you could satisfy both it and this License would be to",
		"refrain entirely from distribution of the Program.");
  cli_out_lines(" ",
		"If any portion of this section is held invalid or unenforceable under",
		"any particular circumstance, the balance of the section is intended to",
		"apply and the section as a whole is intended to apply in other",
		"circumstances.",
		" ");
  cli_out_lines("It is not the purpose of this section to induce you to infringe any",
		"patents or other property right claims or to contest validity of any",
		"such claims; this section has the sole purpose of protecting the",
		"integrity of the free software distribution system, which is  ",
		"implemented by public license practices.  Many people have made",
		"generous contributions to the wide range of software distributed");
  cli_out_lines("through that system in reliance on consistent application of that",
		"system; it is up to the author/donor to decide if he or she is willing",
		"to distribute software through any other system and a licensee cannot",
		"impose that choice.",
		" ",
		"This section is intended to make thoroughly clear what is believed to");
  cli_out_lines("be a consequence of the rest of this License.",
		" ",
		"  8. If the distribution and/or use of the Program is restricted in",
		"certain countries either by patents or by copyrighted interfaces, the",
		"original copyright holder who places the Program under this License",
		"may add an explicit geographical distribution limitation excluding");
  cli_out_lines("those countries, so that distribution is permitted only in or among",
		"countries not thus excluded.  In such case, this License incorporates",
		"the limitation as if written in the body of this License.",
		" ",
		"  9. The Free Software Foundation may publish revised and/or new versions",
		"of the General Public License from time to time.  Such new versions will");
  cli_out_lines("be similar in spirit to the present version, but may differ in detail to",
		"address new problems or concerns.",
		" ",
		"Each version is given a distinguishing version number.  If the Program",
		"specifies a version number of this License which applies to it and \"any",
		"later version\", you have the option of following the terms and conditions");
  cli_out_lines("either of that version or of any later version published by the Free",
		"Software Foundation.  If the Program does not specify a version number of",
		"this License, you may choose any version ever published by the Free Software",
		"Foundation.",
		" ",
		"  10. If you wish to incorporate parts of the Program into other free");
  cli_out_lines("programs whose distribution conditions are different, write to the author",
		"to ask for permission.  For software which is copyrighted by the Free",
		"Software Foundation, write to the Free Software Foundation; we sometimes",
		"make exceptions for this.  Our decision will be guided by the two goals",
		"of preserving the free status of all derivatives of our free software and",
		"of promoting the sharing and reuse of software generally.");
  cli_out_lines(" ", 0, 0, 0, 0, 0);
}

/* 0.4.27.c14 */
void gpl_no_warranty(void)
{
  cli_out_lines("                            NO WARRANTY",
		" ",
		"  11. BECAUSE THE PROGRAM IS LICENSED FREE OF CHARGE, THERE IS NO WARRANTY",
		"FOR THE PROGRAM, TO THE EXTENT PERMITTED BY APPLICABLE LAW.  EXCEPT WHEN",
		"OTHERWISE STATED IN WRITING THE COPYRIGHT HOLDERS AND/OR OTHER PARTIES",
		"PROVIDE THE PROGRAM \"AS IS\" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED");
  cli_out_lines("OR IMPLIED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF",
		"MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.  THE ENTIRE RISK AS",
		"TO THE QUALITY AND PERFORMANCE OF THE PROGRAM IS WITH YOU.  SHOULD THE",
		"PROGRAM PROVE DEFECTIVE, YOU ASSUME THE COST OF ALL NECESSARY SERVICING,",
		"REPAIR OR CORRECTION.",
		" ");
  cli_out_lines("  12. IN NO EVENT UNLESS REQUIRED BY APPLICABLE LAW OR AGREED TO IN WRITING",
		"WILL ANY COPYRIGHT HOLDER, OR ANY OTHER PARTY WHO MAY MODIFY AND/OR",
		"REDISTRIBUTE THE PROGRAM AS PERMITTED ABOVE, BE LIABLE TO YOU FOR DAMAGES,",
		"INCLUDING ANY GENERAL, SPECIAL, INCIDENTAL OR CONSEQUENTIAL DAMAGES ARISING",
		"OUT OF THE USE OR INABILITY TO USE THE PROGRAM (INCLUDING BUT NOT LIMITED",
		"TO LOSS OF DATA OR DATA BEING RENDERED INACCURATE OR LOSSES SUSTAINED BY");
  cli_out_lines("YOU OR THIRD PARTIES OR A FAILURE OF THE PROGRAM TO OPERATE WITH ANY OTHER",
		"PROGRAMS), EVEN IF SUCH HOLDER OR OTHER PARTY HAS BEEN ADVISED OF THE",
		"POSSIBILITY OF SUCH DAMAGES.",
		0,
		0,
		0);
}

/* 0.4.27.c14 */
void gpl_bottom(void)
{
  cli_out_lines(" ",
                "                     END OF TERMS AND CONDITIONS",
                " ",
                "        Appendix: How to Apply These Terms to Your New Programs",
                " ",
                "  If you develop a new program, and you want it to be of the greatest");
  cli_out_lines("possible use to the public, the best way to achieve this is to make it",
                "free software which everyone can redistribute and change under these terms.",
                " ",
                "  To do so, attach the following notices to the program.  It is safest",
                "to attach them to the start of each source file to most effectively",
                "convey the exclusion of warranty; and each file should have at least");
  cli_out_lines("the \"copyright\" line and a pointer to where the full notice is found.",
                " ",
                "    <one line to give the program's name and a brief idea of what it does.>",
                "    Copyright (C) 19yy  <name of author>",
                " ",
                "    This program is free software; you can redistribute it and/or modify");
  cli_out_lines("    it under the terms of the GNU General Public License as published by",
                "    the Free Software Foundation; either version 2 of the License, or",
                "    (at your option) any later version.",
                " ",
                "    This program is distributed in the hope that it will be useful,",
                "    but WITHOUT ANY WARRANTY; without even the implied warranty of");
  cli_out_lines("    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the",
                "    GNU General Public License for more details.",
                " ",
                "    You should have received a copy of the GNU General Public License",
                "    along with this program; if not, write to the Free Software",
                "    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.");
  cli_out_lines(" ",
                "Also add information on how to contact you by electronic and paper mail.",
                " ",
                "If the program is interactive, make it output a short notice like this",
                "when it starts in an interactive mode:",
                " ");
  cli_out_lines("    Gnomovision version 69, Copyright (C) 19yy name of author",
                "    Gnomovision comes with ABSOLUTELY NO WARRANTY; for details type `show w'.",
                "    This is free software, and you are welcome to redistribute it",
                "    under certain conditions; type `show c' for details.",
                " ",
                "The hypothetical commands `show w' and `show c' should show the appropriate");
  cli_out_lines("parts of the General Public License.  Of course, the commands you use may",
                "be called something other than `show w' and `show c'; they could even be",
                "mouse-clicks or menu items -- whatever suits your program.",
                " ",
                "You should also get your employer (if you work as a programmer) or your",
                "school, if any, to sign a \"copyright disclaimer\" for the program, if");
  cli_out_lines("necessary.  Here is a sample; alter the names:",
                " ",
                "  Yoyodyne, Inc., hereby disclaims all copyright interest in the program",
                "  `Gnomovision' (which makes passes at compilers) written by James Hacker.",
                " ",
                "  <signature of Ty Coon>, 1 April 1989");
  cli_out_lines("  Ty Coon, President of Vice",
                " ",
                "This General Public License does not permit incorporating your program into",
                "proprietary programs.  If your program is a subroutine library, you may",  
                "consider it more useful to permit linking proprietary applications with the",
                "library.  If this is what you want to do, use the GNU Library General");
  cli_out_lines("Public License instead of this License.",
                0, 0, 0, 0, 0);
}

/* 0.4.27.c14 */
void gpl_full(void)
{
  gpl_top();
  gpl_copying();
  gpl_no_warranty();
  gpl_bottom();
}

/* 0.4.27.c14 */
void gpl_small()
{
  printf("\n"
         "gnut comes with ABSOLUTELY NO WARRANTY; for details type `help warranty'.\n"
         "This is free software, and you are welcome to redistribute it under certain\n"
         "conditions; type `help gpl' for details.\n\n");
}

int recu_up(void)
{
  recu_level++;
  if (recu_level > 10) {
    printf("Error: recursion too deep.\n");
    recu_disable = 1;
    return 0;
  }
  return 1;
}

void recu_down(void)
{
  recu_level--;
  if (recu_level <= 0) {
    recu_level = 0;
    recu_disable = 0;
    recu_count = 0;
  }
}

int recu_click(int cl)
{
  if (recu_level <= 0) {
    return 1;
  }
  recu_count+=cl;
  if (recu_count <= 10) {
    return 1;
  }
  return 0;
}

int get_command_index(char *command)
{
  int i;
  int flag;
  
  flag = -1;
  for (i=0; commands[i].name; i++) {
    if ( (strlen(command) == strlen(commands[i].name))
	 && (strncasecmp(command, commands[i].name, strlen(commands[i].name))==0)
	 ) {
      /* exact match */
      return i;
    } else if (strncasecmp(command, commands[i].name, strlen(command))==0) {
      if (flag!=-1) {
	/* duplicate */
	flag = -2;
      }
      flag=i;
    }
  }
  return flag;
}

int parse_command(char *a)
{
  char *command;
  char *arg;
  char buf1[1024] = "";
  char buf2[1024] = "";
  char buf3[1024] = "";
  char *bqs; /* backquote start */
  char *bqe; /* backquote end */
  char found_bt=0;
  FILE *p = 0;
  int cnum;
  int i;
  int len;

  bqe = 0;

  /* Skip any leading spaces and/or tabs 0.4.27.c06 */
  while(g_isblank(*a)) {
    a++;
  }

  /* Ignore comments in .gnutrc (or if typed by the user :-) */
  if (a[0] == '#') {
    return 0;
  }

  /* Allow our IRC fanatics to pretend they're in IRC 0.4.27.c06 */
  if (*a == '/') {
    a++;
  }

  len = strlen(a);

  /* Do back-quote interpolation */
  strncpy(buf1, a, sizeof(buf1));
  do {
    found_bt = 0;
    for (bqs=buf1; *bqs; bqs++) {
      if (*bqs=='`' && (bqs-buf1==0 || *(bqs-1) != '\\')) {
	for (bqe=bqs+1; *bqe; bqe++) {
	  if (*bqe=='`' && *(bqe-1) != '\\') {
	    found_bt = 1;
	    break;
	  }
	}
	if (!found_bt) {
	  printf("Error: unterminated backtick at \"%s\".\n", bqs);
	  return 0;
	} 
		
	/* fprintf(stderr, "DEBUG: execing \"%s\".\n", bqs+1); */
	*bqs = 0; *bqe = 0;
	p = popen(bqs+1, "r");
	if (!p) {
	  printf("Error: could not open pipe \"%s\".\n", bqs+1);
	  return 0;
	}
	/* get the last line output by the cmd. this is arguably the
	 * wrong behavior -- we might want to be using only
	 * the first line, or all of them.. etc etc */
	while (fgets(buf3, sizeof(buf3), p)) { 
	}

	if (strlen(buf3) && buf3[strlen(buf3)-1]=='\n') {
	  buf3[strlen(buf3)-1]=0;
	}
	pclose(p);

	/* fprintf(stderr,"DEBUG: replacing \"%s\" with \"%s\".\n", bqs+1,
	   buf3); */
	/* The following three lines are equivalent to snprintf(buf2,
	   sizeof(buf2), "%s%s%s", buf1, buf3, bqe+1); */
        strcpy(buf2, buf1);
        strcatlim(buf2, buf3, sizeof(buf2));
        strcatlim(buf2, bqe+1, sizeof(buf2));

	/* fprintf(stderr,"DEBUG: final string = \"%s\"\n", buf2); */
	printf("%s\n", buf2);
	strncpy(buf1, buf2, sizeof(buf1));
	break;
      }
    }
  } while (found_bt);
  
  /* Start by assuming there's nothing */
  len = strlen(buf1);
  command = 0;
  arg = 0;

  /* skip initial space */
  for (i=0; i<len && isspace(buf1[i]); i++) {
  }
  if (i==len) {
    /* It's all space -- no command or arg */
  } else {
    /* got a command, at least */
    command=&buf1[i];
    /* Scan past command */
    for (;i<len && !isspace(buf1[i]);i++) {
    }
    if (i==len) {
      /* nothing after command */
      arg=0;
    } else {
      /* space after command, but what about an arg? */
      buf1[i++]=0;
      /* skip space */
      for (;i<len && isspace(buf1[i]);i++) {
      }
      if (i==len) {
        /* nope, no arg */
	arg=0;
      } else {
        /* yes, there's an argument */
	arg=&buf1[i];
      }
    }
  }
  
  if (arg) {
    /* delete trailing space after arg */
    for (i=len; i>0 && isspace(buf1[i]); i--) {
    }
    if (i!=len) {
      buf1[i+1]=0;
    }
  }
  
  if (command) {
    /* Look up the command */
    cnum=get_command_index(command);
    if (cnum==-1) {
      /* not found at all */
      printf(UI_IN_NOSUCH, command);
      return 0;
    }
    if (cnum==-2) {
      /* we found two matches, e.g. "s" for "set" and "sleep" */
      printf(UI_IN_AMBIG, command);
      return 0;
    }

    /* dispatch to command's function and pass the arg */
    (*commands[cnum].func)(arg);
  }
  return 0;
}

int echo_comset = 1;

int run_config_file(char *f, int doall, int echo, int comset)
{
  FILE *fp;
  char buf[256];

  echo_comset = comset;

  fp = fopen(f, "r");
  if (fp == 0) {
    return -1;
  }
  
  while (fgets(buf, sizeof(buf), fp)) {
    if (buf[strlen(buf)-1] == '\n') {
      buf[strlen(buf)-1] = 0;
    }
    if (buf[strlen(buf)-1] == '\r') {
      buf[strlen(buf)-1] = 0;
    }
    if (strncmp(buf, "set ", 4) == 0) {
      if (echo) {
	printf("%s\n", buf);
      }
      parse_command(buf);
    } else if (doall) {
      if (buf[0] == '#') {
      } else if (echo) {
	printf("%s\n", buf);
      }
      parse_command(buf);
    }
  }
  fclose(fp);

  echo_comset = 1;
  
  return 0;
}

/* This routine parses a range such as "2-3" or "4,6-8", and calls
   a function once on each number that results */
int input_parse_range(char *range, int (*a)(int32, void *), void *userdata)
{
  int i;
  int r1,r2;
  char *ptr;
  char *ptr2;
  
  if (range==0) return 0;
  
  ptr=range;
  while ((*ptr)!=0) {
    gd_s(5, "input_parse.range here\n");
    /* get past any initial whitespace */
    for (;!IS_DIGIT(*ptr) && *ptr!=0;ptr++)
      { }
    if (*ptr==0) {
      return -1;
    }
    r1 = strtol(ptr,&ptr2,10);
    gd_s(5, "input_parse.range here2\n");
    ptr = ptr2;
    
    /* now that we have one number, we need to look for either
     * a control character [,-] or end of string, otherwise it's
     * an error */
    
    for (;isspace(*ptr) && *ptr!=0 ;ptr++)
      { }
    gd_s(5, "input_parse.range here3\n");
    if (*ptr == 0) {
      /* this was the last number */
      (*a)(r1, userdata);
      return 0;
    } else if (*ptr == '-') {
      gd_s(5, "input_parse.range here4\n");
      /* this is a range, so we need to find the second number
       * if there is one, then do the whole range */
      for (ptr++; !IS_DIGIT(*ptr) && *ptr!=0; ptr++)
	{ }
      if (*ptr == 0) {
	return -1; /* no second number! */
      }
      gd_s(5, "input_parse.range here5\n");

      r2 = strtol(ptr,&ptr2,10);
      ptr = ptr2;
      /* k, we have a second number, so lets do the range!!! */
      
      for (i=r1;i<=r2;i++) {
	(*a)(i,userdata);
      }
      
      /* now we need to see if the string is ended, or if there is a
       * comma yet... */
      for (; *ptr!=',' && *ptr!=0; ptr++) { }
      if (*ptr == 0) {
	return 0;
      }
      ptr++;
    } else if (*ptr == ',') {
      (*a)(r1,userdata);
      ptr++;
    }
  }    

  return 0;
}

char *info_str_type(int a)
{
  switch(a) {
  case 4:
    return "RCV PUSH";
  case 3:
    return "SND PUSH";
  case 2:
    return "IN";
  case 1:
    return "OUT";
  }
  return 0;
}

char *info_str_state(int a, int s)
{
  switch(a) {
  case STATE_CONNECTING:
    if (gc_debug_opts == 0) {
      return "CONN ";
    }
    if (s == 1) { /* Parse substate field */
      return "CONN1";
    } else if (s == 2) {
      return "CONN2";
    } else if (s == 3) {
      return "CONN3";
    } else if (s == 4) {
      return "CONN4";
    } else if (s == 5) {
      return "CONN5";
    } else {
      return "CONN";
    }
  case STATE_NEGOTIATING:
    return "NEGOT";
  case STATE_CONNECTED:
    return "UP   ";
  case STATE_CLEANABLE:
    return "CLEAN";
  case STATE_DEAD:
    return "DONE ";
  case STATE_ERROR:
    return "ERR  ";
  case STATE_QUEUED:
    return "QUEUE";
  case STATE_RETRY_WAIT:
    return "RETRY";
  }
  return "Unkno";
}

int32 transfer_num = 0; /* 0.4.27.c11 */
int connection_num = 0;
float inb=0;
float outb=0;
long total_sent;
int xfer_print_up;
int xfer_print_dn;

int info_transfer_callback(gnut_transfer *gt)
{
  uint32 rateb; /* 0.4.27.c11 */
  float c;
  char *t1, *t2, *t3, *eta;
  char buf[256];
  char ipstr[32];
  float eta_in_seconds;
  int doprint;
  int err;

  /* Detect unused entries and skip */
  if (gt->fname == 0) {
    gt->gt_uindex = 0;
    return 0;
  }

  transfer_num++;
  gt->gt_uindex = transfer_num;
  rateb = gt->rate_bytes >> 4L;
  t1 = format_si(rateb, 0);
  if (gt->total) {
    c = 100.0f * ((float) gt->gt_bytes / (float) gt->total);
  } else {
    c = 0;
  }

  if (gt->state == STATE_CONNECTED) {
    if ((gt->type & 1) == 0) {
      inb += rateb;
    } else {
      outb += rateb;
    }
  }

  if ((gt->type & 1) == 0) {
    doprint = xfer_print_dn;
  } else {
    doprint = xfer_print_up;
  }

  if (rateb > 0) {
    eta_in_seconds = ((gt->total - gt->gt_bytes) / rateb);
  } else {
    eta_in_seconds = 0;
  }
  eta = munge_time(eta_in_seconds);

  sprintf(ipstr,"%i.%i.%i.%i:%i",gt->ip[0],gt->ip[1],gt->ip[2],gt->ip[3],
	  gt->port);

  if (gc_munge) {
    t2 = format_si(gt->gt_bytes, 0);
    t3 = format_si(gt->total, 0);
  } else {
    t2 = ymaloc(10, 345);
    t3 = ymaloc(10, 346);
    sprintf(t2,"%u", gt->gt_bytes); /* 0.4.27.c11 */
    sprintf(t3,"%u", gt->total); /* 0.4.27.c11 */
  }

  sprintf(buf, "%u)%-19s %5.1f%%  %9s/%-9s ", /* 0.4.27.c11 */
	  gt->gt_uindex, ipstr, c, t2, t3);
  fre_str(&t2, 186);
  fre_str(&t3, 187);
  err = 0;
  if (doprint) {
    err = cli_output_line(1,buf);
  }
  if(err)
    return -1;

  sprintf(buf, UI_INFO_RATE, t1);
  fre_str(&t1, 188);
  if (doprint) {
    err = cli_output_line(1,buf);
  }
  if (err) {
    return -1;
  }

  sprintf(buf, UI_INFO_ETA, eta);
  fre_str(&eta, 189);

  if (doprint) {
    err = cli_output_line(1,buf);
  }
  if (err) {
    return -1;
  }
  
  sprintf(buf, UI_INFO_TID, 
#if defined(hpux) && defined(PTHREADS_DRAFT4)
	  ((HPUX_hack)gt->tid).longpart
#else
	  (long) gt->tid
#endif
	  );
  if (doprint) {
    err = cli_output_line(1, buf);
  }
  if (err) {
    return -1;
  }
  
  sprintf(buf, UI_INFO_TYPE_STATE,
	  info_str_type(gt->type), info_str_state(gt->state, gt->substate));
  if (doprint) {
    err = cli_output_line(1, buf);
  }
  if (err) {
    return -1;
  }
  
  sprintf(buf, UI_INFO_NAME);
  if (doprint) {
    err = cli_output_line(1, buf);
  }
  if (err) {
    return -1;
  }

  printf("%s", output_bold_on);
  t1 = (gt->fname) ? ystdup(gt->fname, 472) : ystdup(UI_INFO_UNKNOWN, 473);
  t2 = strstr(t1, ".gnut");
  if (t2) {
    *t2 = 0;
  }
  sprintf(buf, "%s", t1);
  fre_str(&t1, 190);
  if (doprint) {
    err = cli_output_line(1, buf);
  }
  if (err) {
    return -1;
  }
  printf("%s", output_special_off);
  if (doprint) {
    err = cli_output_line(1, "\n");
  }
  if (err) {
    return -1;
  }

  return 0;
}

int info_transfer(int up, int dn)
{
  char buf[256];
  char *t1,*t2;

  xfer_print_up = up;
  xfer_print_dn = dn;

  if (gnut_xfer_num() == 0) {
    return 0;
  }
  if (cli_output_line(3, UI_INFO_XFER_STATS)) {
    return 0;
  }
  cli_output_line(2,"---------------\n");
  transfer_num = 0;
  inb = 0;
  outb = 0;

  gnut_xfer_enumerate(info_transfer_callback);

  t1 = format_si(outb, 0);
  t2 = format_si(inb, 0);
  sprintf(buf, UI_INFO_TOTALS, t1, t2);
  fre_str(&t1, 191);
  fre_str(&t2, 192);
  if (cli_output_line(0, buf)) {
    return -1;
  }
  return 0;
}

#define FMT_TERSE 1

int info_connection_callback(gcs *gc)
{
  float64 s, r;
  time_t i;
  char *t1,*t2;
  char buf[4096];
  char buf2[100];
  char ipstr[32];
  int fmt;

  fmt = conf_get_int("stats_format");
  
  connection_num++;
  gc->uindex = connection_num;
  
  sprintf(ipstr, "%i.%i.%i.%i:%i", gc->ip.b[0], gc->ip.b[1],
	  gc->ip.b[2], gc->ip.b[3], gc->port);
  
  if (gc_munge) {
    t1 = format_si(gc->snt_bytes, 0);
    t2 = format_si(gc->rcv_bytes, 0);
  } else {
    t1 = ymaloc(20, 347);
    sprintf(t1, "%7g", gc->snt_bytes);
    t2 = ymaloc(20, 348);
    sprintf(t2, "%-7g", gc->rcv_bytes);
  }
  
  /* First line of stats */
  if (fmt & FMT_TERSE) {
    sprintf(buf,"%2i) %-20s",
	    connection_num, ipstr);
    sprintf(buf2, "%i,%i,%i (%2li) B:%i",
	    gc->sent_packets, gc->received_packets,
	    gc->dropped_packets,
	    droprate(gc->dropped_packets, gc->received_packets),
	    gc->bad_packets );
  } else {
    sprintf(buf, UI_INFO_CONN1,
	    connection_num, ipstr, gc->sent_packets,
	    gc->received_packets, gc->dropped_packets, gc->bad_packets,
	    t1,t2);
  }
  fre_str(&t1, 193);
  fre_str(&t2, 194);
  if (cli_output_line(1, buf))
    return -1;

  total_sent += gc->sent_packets;
  
  i = time(0) - gc->rate_start_time;
  if (i) {
    s = gc->rate_snt_bytes / ((float64) i);
    t1 = format_si(s, 0);

    r = gc->rate_rcv_bytes / ((float64) i);
    t2 = format_si(r, 0);
  } else {
    t1 = ymaloc(10, 349);
    t2 = ymaloc(10, 350);
    strcpy(t1, "0");
    strcpy(t2, "0");
    s = 0;
    r = 0;
  }
  if (gc->cstate == STATE_CONNECTED) {
    inb += r;
    outb += s;
  }

  /* Second line of stats */
  if (fmt & FMT_TERSE) { 
    if (gc->cstate == STATE_CONNECTED) {
      sprintf(buf," %s OK: %s\n", info_str_type(gc->ctype), buf2);
    } else {
      sprintf(buf," %s %s\n", info_str_type(gc->ctype),
	      info_str_state(gc->cstate, 4));
    }
  } else { 
    sprintf(buf, UI_INFO_CONN2,
#if defined(hpux) && defined(PTHREADS_DRAFT4)
	    ((HPUX_hack) gc->tid).longpart,
#else
	    (long) gc->tid,
#endif
	    info_str_type(gc->ctype), info_str_state(gc->cstate, 4),
	    t1, t2);
  }
  
  fre_str(&t1, 195);
  fre_str(&t2, 196);
  if (cli_output_line(0,buf)) {
    return -1;
  }
  
#ifdef MOREPACKETSTATS
  if (conf_get_int("packet_stats")) {
    int n,n2;

    for (n = 0; n < 256; n++) { 
      if (gc->packet_count[n]) {
        sprintf(buf, UI_INFO_CONN3,
		n, gc->packet_count[n],
		gc->packet_byte_count[n]/gc->packet_count[n],
		gc->max_packet_size[n]);
        if (cli_output_line(0,buf)) {
	  return -1;
	}
      }
    } 
	
    if (gc->packet_count[0]) {
      n2 = sprintf(buf, UI_INFO_CONN4);
      for (n = 0; n < 256; n++) {
        if (gc->ttl_count[n]) {
          n2 += sprintf(&buf[n2], "%d(%d) ", n, gc->ttl_count[n]);
        }
      }
      sprintf(&buf[n2],"\n");
      if (cli_output_line(0,buf)) {
	return -1;
      }
    }
	
    for (n = 0; n < 256; n++) {
      if (gc->out_packet_count[n]) {
        sprintf(buf, UI_INFO_CONN5, n, gc->out_packet_count[n],
		gc->out_packet_byte_count[n]/gc->out_packet_count[n],
		gc->out_max_packet_size[n]);
        if (cli_output_line(0,buf)) {
	  return -1;
	}
      }
    }

    if (gc->out_packet_count[0]) { 
      n2 = sprintf(buf, UI_INFO_CONN6);
      for (n = 0; n < 256; n++) {
        if (gc->out_ttl_count[n]) {
          n2 += sprintf(&buf[n2],"%d(%d) ",
			n, gc->out_ttl_count[n]);
        }
      }
      sprintf(&buf[n2],"\n");
      if (cli_output_line(0,buf)) {
	return -1;
      }
    }
  }
#endif
    
  return 0;
}

int info_connection()
{
  char buf[256];
  char *t1,*t2;
  
  if (gnut_connection_num()==0) {
    return 0;
  }
  
  if (cli_output_line(3,"CONNECTION STATS:\n")) {
    return -1;
  }

  cli_output_line(2,"-----------------\n");
  connection_num=0;
  inb = 0;
  outb = 0;
  total_sent = 0;

  gnut_connection_enumerate(info_connection_callback);
  
  t1 = format_si((unsigned long) outb, 0);
  t2 = format_si((unsigned long) inb, 0);
  sprintf(buf, UI_INFO_CONN_TOTALS, t1, t2, total_sent);
  fre_str(&t1, 197);
  fre_str(&t2, 198);
  if (cli_output_line(0,buf)) {
    return -1;
  }
  return 0;
}

int info_shares()
{
  uint32 num; /* 0.4.27.c */
  float64 size;
  char *t1;
  char buf[256];

  share_totals(&num, &size);
  if (gc_munge) {
    t1 = format_si(size, 0);
  } else {
    t1 = ymaloc(20, 351);
    sprintf(t1, "%g k bytes", size / 1000.0);
  }
  sprintf(buf, UI_INFO_SHARE, num, t1);
  fre_str(&t1, 199);
  if (cli_output_line(0, buf)) {
    return -1;
  }

  sprintf(buf, urate_clipped ? UI_INFO_SHAR2 : UI_INFO_SHAR3,
	  (best_urate + 64) >> 7,
	  (num_uploads < gc_max_uploads) ? 1 : 0,
	  gh_did_upload, gh_did_receive);
  if (cli_output_line(0, buf)) {
    return -1;
  }

  return 0;
}

int info_net()
{
  int messages, sent;
  char *t1,*t2;
  float64 n_rcv_bytes, n_snt_bytes;
  float64 n_blacklisted;
  char buf[256];
  
  gnut_connection_net_totals(&messages, &sent, &n_rcv_bytes, &n_snt_bytes,
			     &n_blacklisted);
  if (gc_munge) {
    t1 = format_si((float64) messages, 0);
    t2 = format_si((float64) sent, 0);
  } else {
    t1 = ymaloc(20, 352);
    t2 = ymaloc(20, 353);
    sprintf(t1, "%i", messages);
    sprintf(t2, "%i", sent);
  }
  sprintf(buf, UI_INFO_NET, t1, t2);
  fre_str(&t1, 200);
  fre_str(&t2, 201);
  if (cli_output_line(1,buf)) {
    return -1;
  }
  
  sprintf(buf, "             Unique GUIDs in memory: %i\n", route_queue->size);
  if (cli_output_line(0,buf)) {
    return -1;
  }

  if (gc_munge) {
    t1 = format_si(n_rcv_bytes, 0);
    t2 = format_si(n_snt_bytes, 0);
  } else {
    t1 = ymaloc(20, 354);
    t2 = ymaloc(20, 355);
    sprintf(t1, "%g", n_rcv_bytes);
    sprintf(t2, "%g", n_snt_bytes);
  }
  sprintf(buf, "             Bytes Rcvd: %-7s       Bytes Sent: %-7s\n",
	  t1, t2);
  fre_str(&t1, 202);
  fre_str(&t2, 203);
  if (cli_output_line(0,buf)) {
    return -1;
  }

  return 0;
}

int info_query()
{
  char buf[256];
  
  sprintf(buf,"QUERY STATS: Queries: %-6u   Responses Sent: %-6u\n",
	  con_num_queries, con_num_responses);
  if (cli_output_line(0,buf)) {
    return -1;
  }

#ifdef USE_DRR_SEARCH
  if (t_counter) {
    r = (((long)t_hits) * 1000L) / ((long) t_counter);
  } else {
    r = 0;
  }
  sprintf(buf,"             QREPLY Cache hit rate: %2i.%1i%%\n",
	  r/10, r%10);
  if (cli_output_line(0, buf)) {
    return -1;
  }
#endif

  return 0;
}

int info_hosts()
{
  uint h_num;
  float64 bytes, files;
  char buf[256];
  char *t1,*t2;

  host_totals(&h_num, &files, &bytes);

  if (gc_munge) {
    t1 = format_si(files, 0);
    t2 = format_si(bytes, 0);
  } else {
    t1 = ymaloc(20, 356);
    t2 = ymaloc(20, 357);
    sprintf(t1, "%g", files);
    sprintf(t2, "%g M", bytes / 1000000.0);
  }
  sprintf(buf, "HOST STATS:  Hosts: %-6i Files: %-7s Size: %-7s\n",
	  h_num, t1, t2);
  fre_str(&t1, 204);
  fre_str(&t2, 205);
  if (cli_output_line(0,buf)) {
    return -1;
  }

  sprintf(buf, "             LRU Cache: %4u\n", hlru_num);
  if (cli_output_line(0,buf)) {
    return -1;
  }
  return 0;
}  

int info_all()
{
  cli_output_reset();
  if (info_hosts())
    return 0;
  if (info_net())
    return 0;
  if (info_query())
    return 0;
  if (info_shares())
    return 0;
  if (info_connection())
    return 0;
  if (info_transfer(1, 1))
    return 0;
  return 0;
}

int com_info(char *arg)
{
  char *s;

  if (!arg) {
    info_all();
  } else {
    for(s=arg; *s; s++) {
      switch(tolower(*s)) {
      case 'h':
	cli_output_reset();
	info_hosts();
	break;
      case 'q':
	cli_output_reset();
	info_query();
	break;
      case 'n':
	cli_output_reset();
	info_net();
	break;
      case 's':
	cli_output_reset();
	info_shares();
	break;
      case 'c':
	cli_output_reset();
	info_connection();
	break;
      case 't':
	cli_output_reset();
	info_transfer(1, 1);
	break;
      case 'u':
	cli_output_reset();
	info_transfer(1, 0);
	break;
      case 'd':
	cli_output_reset();
	info_transfer(0, 1);
	break;
      }
    }
  }
  return 0;
}

int com_open(char *arg)
{
  if (recu_click(1)) {
    if (arg) {
      gnut_outgoing_new(arg);
    }
  }
  return 0;
}

void show_responses_count()
{
  static int a;
  int b;

  b = query_num();

  if (b == 0) {
    a = 0;
  }
  if (b && (a == 0)) {
    if (conf_get_int("beep_on_first")) {
      putchar(7);
    }
  } else if (b) {
    if (conf_get_int("beep_on_all")) {
      putchar(7);
    }
  }

  if (b != a) {
    printf("%i responses received.", b);
    fflush(stdout);
    printf("\r");
  }

  a = b;
}

int com_find(char *arg)
{
  gnutella_packet *gpa;
  int             maxs;

  if (recu_click(5)) {
    if (arg) {
      /* Get pointer to any existing query packet, zero it and deal.locate */
      pthread_mutex_lock(&query_packet_mutex);
      gpa = current_query_packet;
      current_query_packet = 0;
      pthread_mutex_unlock(&query_packet_mutex);
      if (gpa) {
	fre_v(&(gpa->data), 206);
	fre_gpa(&gpa, 207);
	gpa = 0;
      }

      /* Create a new query packet, including a new random GUID */
      gpa = gp_request_make(conf_get_str("mac"), arg, gc_ttl);
      conf_set_str_len("query_guid", gpa->gh.guid, 16);

      if (conf_get_int("multi_enable")) {
	maxs = conf_get_int("max_searches");
	if (maxs && ((gnut_search_count()+1) >= maxs)) {
	  printf("The limit (of %i) searches reached.", maxs);
	  printf("Removing oldest search string \"%s\".\n",
		 current_searches->query);
	  gnut_search_remove(current_searches);
	}
	gnut_search_add(gpa->gh.guid, arg);
      } else {
	query_clear();
      }
	  
      /* Save a pointer to the query packet data */
      pthread_mutex_lock(&query_packet_mutex);
      current_query_packet = gpa;
      pthread_mutex_unlock(&query_packet_mutex);
	  
      send_to_all(gpa); /* %%%RPM on new connection, do a send_to_one if a query is currently pending */
	  
      if (!conf_get_int("wait_after_find")) {
	printf(UI_FIND_WHEN_YOU);
	return 0;
      }
      printf("Searching the gnutella network for: %s \n", arg);
      printf("Press any key to continue\n");
	  
      printf("0 responses received.\r");
      wait_key_callback(show_responses_count);
      printf("\n");
      if (conf_get_int("auto_response")) {
	com_response(0);
      }
    } else {
      /* #find# with no argument displays current query responses */
      com_response(0);
    }
  }
  return 0;
}

void clear_search_cb(GnutSearch *s, void *data)
{
  gnut_search_remove(s);
  return;
}

/* clear_guid_search_cb calculates a GUID hash value and returns a nonzero
 * result if it matches the value pointed to by 'data' */
char clear_guid_search_cb(GnutSearch *s, void *data)
{
  int t;
  int sum = 0;

  for (t=0; t<14; t++) {
    sum += (unsigned char) s->guid[t];
  }

  return (sum == *((int*)data));
}

int com_lclear(char *arg)
{
  GnutSearch *search;
  char       buf[1024] = "";
  int        guid_hash = 0;

  if (conf_get_int("multi_enable")) {
    if (!arg) {
      /* Clear all searches */
      gnut_search_foreach(clear_search_cb, 0);
      query_clear();
    } else {
      /* Clear a specified search */
      if ((arg[0]=='\"'&&arg[strlen(arg)-1]=='\"') ||
	  (arg[0]=='\''&&arg[strlen(arg)-1]=='\'')) {
	/* Clear by string match */
	memcpy(buf, arg+1, strlen(arg)-2);
	buf[strlen(arg)-2] = 0;
	search = gnut_search_find_by_query(buf);
      } else {
	/* Clear by numeric GUID hash match */
	guid_hash = atoi(arg);
	search = gnut_search_find(clear_guid_search_cb, &guid_hash);
      }
      if (search) {
	printf("Removing search \"%s\".\n", search->query);
	gnut_search_remove(search);
      }
    }
  } else {
    printf(UI_LIST_MULTIDIS);
  }
  return 0;
}

/* list_search_cb prints a summary of a search in the search queue */
void list_search_cb(GnutSearch *s, void *data)
{
  int t;
  int sum = 0;

  for (t=0; t<14; t++) {
    sum += (unsigned char) s->guid[t];
  }

  printf("%4d, %4d, \"%s\"\n", sum, s->responses, s->query);

  return;
}

int com_list(char *arg)
{
  if (conf_get_int("multi_enable")) {
    printf("Current searches:\n");
    printf("GUID,  Num, String\n");
    gnut_search_foreach(list_search_cb, 0);
  } else {
    printf(UI_LIST_MULTIDIS);
  }
  return 0;
}

/* Run the #update# command, which sends a PING packet causing update of
 * the host stats. This is the only place #gnut# generates a PING packet.
 * Since we get plenty of data from watching PONGs, it isn't really
 * necessary. */
int com_update(char *arg)
{
  gnutella_packet *gpa;
  
  if (recu_click(10)) {
    gpa = gp_ping_make(conf_get_str("mac"), conf_get_int("ttl"));
    if (conf_get_int("update_clear"))
      host_clear();
	
    conf_set_str_len("update_guid", gpa->gh.guid, 16);
	
    send_to_all(gpa);
	
    if (gpa->data) {
      fre_v(&(gpa->data), 208);
    }
    fre_gpa(&gpa, 209);
  }
  return 0;
}

int query_response_compare(const void *a, const void *b)
{
  query_resp **c_qrpl;
  query_resp *a_qrp1, *a_qrp2;
  char *so;
  char *e1,*e2;
  int flag = 0;
  int j;
  uint32 C1, C2; /* 0.4.27.c21 */

  c_qrpl = (query_resp **) a;
  a_qrp1 = *c_qrpl;
  c_qrpl = (query_resp **) b;
  a_qrp2 = *c_qrpl;

  so = conf_get_str("sort_order");

  while (*so) {
    switch(*so) {
    case 'i':
      flag = 0;
      for(j=0; (j<4) && (flag == 0); j++) {
	if (a_qrp1->qr_ip[j] < a_qrp2->qr_ip[j]) {
	  flag = -1;
	} else if (a_qrp1->qr_ip[j] > a_qrp2->qr_ip[j]) {
	  flag = 1;
	}
      }
      break;
    case 'r': /* 0.4.27.c21 */
      if (a_qrp1->qr_rating < a_qrp2->qr_rating) {
	flag = -1;
      } else if (a_qrp1->qr_rating > a_qrp2->qr_rating) {
	flag = 1;
      }
      break;
    case 'C': /* 0.4.27.c21 */
      C1 = ((a_qrp1->qr_size) >> 8) * ((uint32) a_qrp1->qr_rating);
      C2 = ((a_qrp2->qr_size) >> 8) * ((uint32) a_qrp2->qr_rating);
      if (C1 < C2) {
	flag = -1;
      } else if (C1 > C2) {
	flag = 1;
      }
      break;
    case 's':
      if (a_qrp1->qr_size < a_qrp2->qr_size) {
	flag = -1;
      } else if (a_qrp1->qr_size > a_qrp2->qr_size) {
	flag = 1;
      }
      break;
    case 'p':
      if (a_qrp1->qr_speed < a_qrp2->qr_speed) {
	flag = -1;
      } else if (a_qrp1->qr_speed > a_qrp2->qr_speed) {
	flag = 1;
      }
      break;
    case 'e':
      e1 = strrchr(a_qrp1->qr_ndata, '.');
      if (!e1) {
	break;
      }
      e2 = strrchr(a_qrp2->qr_ndata, '.');
      if (!e2) {
	break;
      }
      flag = strcasecmp(e1,e2);
      break;
    case 'n':
      flag = strcasecmp(a_qrp1->qr_ndata, a_qrp2->qr_ndata);
      break;
    }
    if (flag) {
      break;
    }
    if (!(*(++so))) {
      break;
    }
    so++;
  }
  if (*so) {
    so++;
    if ((*so)=='-') {
      flag = flag * -1;
    }
  }
  
  return flag;
}

/* This routine was added for 0.4.27.c24. It is almost completely different
from the corresponding code in 0.4.26 */
int cr_display(query_resp * a_qrp, int32 uindex)
{
  char buf[256];
  char bf2[32];
  uint16 port;
  char *t1;
  char *fmt;

  /* Skip any items marked by 'forget' command. */
  if (a_qrp->qr_flag == 'f') {
    return 0;
  }

  fmt = conf_get_str("response_format");
  if (fmt) {
    int in_braces;
    char c;

    in_braces = 0;
    while (*fmt) {
      c = *fmt;
      if ((c == '{') && (in_braces == 0)) {
	in_braces = 1;
      } else if ((c == '}') && (in_braces == 1)) {
	in_braces = 0;
      } else if (in_braces) {
	switch (c) {
	case '#': /* item number */
	  a_qrp->qr_uindex = uindex + 1;
	  sprintf(buf, "%u", a_qrp->qr_uindex); /* 0.4.27.c11 */
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case 'f' : /* flag */
	  if (a_qrp->qr_flag != ' ') {
	    sprintf(buf, "%c ", a_qrp->qr_flag);
	    if (cli_output_line(1, buf)) {
	      return 1;
	    }
	  }
	  break;
	case 'I': /* IP Address */
	  sprintf(buf, "%3i.%3i.%3i.%3i", a_qrp->qr_ip[0], a_qrp->qr_ip[1],
		  a_qrp->qr_ip[2], a_qrp->qr_ip[3]);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case 'N': /* name */
	  printf("%s", output_bold_on);
	  sprintf(buf, "%s", a_qrp->qr_ndata);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  printf("%s", output_special_off);
	  break;
	case 'n': /* refnum */
	  sprintf(buf, "%4u", a_qrp->qr_ref);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case 'P': /* Port */
	  memcpy(&port, a_qrp->qr_port, 2);
	  port = GUINT16_FROM_LE(port);
	  sprintf(bf2, "%i", port);
	  sprintf(buf, "%-5s", bf2);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case 'R': /* Rating 0.4.27.c21 */
	  if (cli_output_line(1, ratings[(a_qrp->qr_rating) >> 4])) {
	    return 1;
	  }
	  break;
	case 'r': /* Score (rating on a scale from 0 to 0144) 0.4.27.c21 */
	  sprintf(buf, "%3d", a_qrp->qr_rating);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case 'S': /* Size */
	  if (gc_munge) {
	    t1 = format_si(a_qrp->qr_size, 0);
	  } else {
	    t1 = ymaloc(15, 360);
	    sprintf(t1, "%u", a_qrp->qr_size); /* 0.4.27.c11 */
	  }
	  sprintf(buf, "%-9s", t1);
	  fre_str(&t1, 214);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case 's': /* speed */
	  sprintf(buf, "%4u", a_qrp->qr_speed);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	  break;
	case '/': /* Newline */
	  if (cli_output_line(0, "\n")) {
	    return 1;
	  }
	  break;
	default:
	  sprintf(buf, "%c", c);
	  if (cli_output_line(1, buf)) {
	    return 1;
	  }
	}
      } else {
	sprintf(buf, "%c", c);
	if (cli_output_line(1, buf)) {
	  return 1;
	}
      }
      fmt++;
    }
    if (cli_output_line(0, "\n")) {
      return 1;
    }
  }

  return 0;
}

/* I've got some new logic for the reponses.
 * First a count of all eligible responses is made, this is done by 
 * running the current expression through the entire list, incrementing
 * 1 each time.  This is limited by "max_responses" in the config.
 * After a count is made, an array of pointers to query_resp strucs
 * is al.located, and the responses are copied into it.  Then qsort is used
 * to sort this array.  Finally it is displayed, and the array is fr.ee'ed
 * once displaying has ended. */
int com_response(char *arg)
{
  Gnut_List *tmp;
  query_resp *a_qrp;
  int32 uindex = 1;
  int cre = 0;       /* use re */
  int ig_case = 1;   /* ignore case in re */
  int inv_re = 0;    /* invert RE matches */
  char *re = 0;
  char buf[256];
  query_resp **disp_qrplist = 0;
  int response_count = 0;
  int multi_enable = 0;

  /* regcomp is much more full featured than re_comp, so prefer it if both
   * are available. */
#ifdef HAVE_REGCOMP
  regex_t creg;
  int re_opt;
#else
#ifdef HAVE_RE_COMP
  char * cre_err = 0;
#endif
#endif
  cli_output_reset();

  pthread_mutex_lock(&query_packet_mutex);
  multi_enable = conf_get_int("multi_enable");
  if (!multi_enable && current_query_packet) {
    printf("Current query is '%s'\n",
	   (((char *) (current_query_packet->data))+2));
  } else if (multi_enable&&current_searches) {
    com_list(0);
  }
  pthread_mutex_unlock(&query_packet_mutex);
  
  if (arg && (*arg)) {
    if (('/' == arg[0]) && arg[1]) {
      int in,out;
      re = ymaloc(strlen(arg), 358);
	  
      for(in=1,out=0; arg[in] && (arg[in] != '/'); in++) {
        if(('\\' == arg[in]) && arg[in+1]) {
          switch (arg[in+1]) {
	  case '\\':
	    re[out] = '\\';
	    out ++;
	    in  ++;
	    break;
	  case '/':
	    re[out] = '/';
	    out ++;
	    in  ++;
	    break;
          }
        } else {
          re[out] = arg[in];
          out++;
        }
      }
	  
      re[out] = '\0';
	  
      if (arg[in] && arg[in+1]) {
        in++;
        while(arg[in]) {
          switch (arg[in]) {
	  case 'i':
	    break;
	  case 'I':
	    ig_case=0;
	    break;
	  case 'v':
	    inv_re=1;
	    break;
          }
          in++;
        }
      }
    } else {
      re = arg;
    }

#ifdef HAVE_REGCOMP
    if (ig_case) {
      re_opt = REG_EXTENDED|REG_ICASE|REG_NOSUB;
    } else {
      re_opt = REG_EXTENDED|REG_NOSUB;
    }
    cre = !regcomp(&creg,re,re_opt);
#else
# ifdef HAVE_RECOMP
    if (ig_case) {
      fprintf(stderr, "re_comp does cannot ignore case\n");
    }
    cre_err = (char *) re_comp(re);
    cre = (cre_err == 0);
# else
    cre=1;
# endif
#endif
    if (cre) {
      sprintf(buf, "Responses matching /%s/:\n", arg);
    } else {
      sprintf(buf, "All responses (%s not valid RE):\n", arg);
    }
  } else {
    /* No argument prints all responses */
    sprintf(buf, "All responses:\n");
  }
  cli_output_line(3,buf);

  query_lock();

  /* here we count the number of responses that meet the criteria */
  for (tmp=query_retrieve(); tmp; tmp=tmp->next) {
    int result = 1;

    a_qrp = tmp->data;
    a_qrp->qr_uindex = -1;
    if (cre) {
#ifdef HAVE_REGCOMP
      result = !regexec(&creg, a_qrp->qr_ndata, 0, 0, 0);
#else
# ifdef HAVE_RE_COMP
      result = re_exec(a_qrp->qr_ndata);
# else
      {
	char *t1;
	char *t2;

	t1 = ystdup(a_qrp->qr_ndata, 474);
	t2 = ystdup(re, 475);
	result = keyword_match(t2, t1, 1, 0);
	fre_str(&t1, 210);
	fre_str(&t2, 211);
      }
# endif
#endif
      if (inv_re) {
        result = !result;
      }
    }

    if (result) {
      response_count++;
    }
  }    
  
  if (response_count > conf_get_int("max_responses")) {
    response_count = conf_get_int("max_responses");
  }
  
  /* now that we have the list counted, we can al.locate the array */
  disp_qrplist=(query_resp **)
    ymaloc(response_count * sizeof(query_resp *), 359);
  
  /* now we have to go back through the list and fill in the array with
   * pointers.... */
  uindex = 0;
  for (tmp = query_retrieve();
       tmp && (uindex < response_count) ; tmp=tmp->next) {
    int result = 1;

    a_qrp = tmp->data;
    if (cre) {
#ifdef HAVE_REGCOMP
      result = !regexec(&creg, a_qrp->qr_ndata, 0, 0, 0);
#else
#ifdef HAVE_RE_COMP
      result = re_exec(a_qrp->qr_ndata);
#else
      {
	char *t1;
	char *t2;

	t1 = ystdup(a_qrp->qr_ndata, 476);
	t2 = ystdup(re, 477);
	result = keyword_match(t2, t1, 1, 0);
	fre_str(&t1, 212);
	fre_str(&t2, 213);
      }
#endif
#endif
      if (inv_re) {
        result = !result;
      }
    }
	
    if (result) {
      disp_qrplist[uindex] = a_qrp;
      uindex++;
    }
  }    
  
  /* now that we have the list of reponses in the array, we can unlock
   * the query list */
  query_unlock();  

  response_count = uindex; /* 0.4.28.c24 */

  /* Sort the query results */
  qsort(disp_qrplist, response_count, sizeof(query_resp *),
	query_response_compare);
  
  /* Display responses */
  for (uindex = 0; uindex < response_count; uindex++) {
    if (cr_display(disp_qrplist[uindex], uindex)) { /* 0.4.27.c24 */
      break;
    }
  }

  fre_qrpl(&disp_qrplist, 215);

#ifdef HAVE_REGCOMP
  if (arg && (*arg)) {
    regfree(&creg);
  }
#endif
  if (arg && (*arg) && (re != arg)) {
    fre_str(&re, 216);
  }
  return 0;
}

int push_warn;

int com_push_callback(int32 a, void *b)
{
  query_resp *a_qrp;
  gnutella_packet *gpa;
  char flag; /* 0.4.27.c01 */

  if (recu_click(10)) {
    flag = gc_auto_forget ? 'f' : 'p'; /* 0.4.27.c01 */

    /* Get a copy of the specified qreply */
    a_qrp = query_index(a, flag); /* qr-new */ /* 0.4.27.c01 */

    if (a_qrp == 0) {
      /* Index out of range */
      return 0;
    }

    /* If no warning yet, we check for errors */
    if (gc_no_rfc_1597) {
      /* No error checking */
    } else if (host_ok_for_both(a_qrp->qr_ip)) {
      if (push_warn == 0) {
	/* Warn about a bogus push */
	printf(UI_PUSH_VPN_ERR3, a_qrp->qr_ip[0], 
	       a_qrp->qr_ip[1], a_qrp->qr_ip[2], a_qrp->qr_ip[3]);
	push_warn = 1;
      }
      return 0;
    } else if (host_ok_for_get(a_qrp->qr_ip)) {
      if (push_warn == 0) {
	/* 'get' will work but not 'push' */
	printf(UI_PUSH_VPN_ERR1, net_local_ip()[0], net_local_ip()[1],
	       net_local_ip()[2], net_local_ip()[3], a_qrp->qr_ip[0], 
	       a_qrp->qr_ip[1], a_qrp->qr_ip[2], a_qrp->qr_ip[3]);
	push_warn = 1;
      }
      return 0;
    } else if (host_ok_for_push(a_qrp->qr_ip)) {
      /* No problemo */
    } else {
      if (push_warn == 0) {
	/* Totally ungettable in either direction */
	printf(UI_PUSH_VPN_ERR2, net_local_ip()[0], net_local_ip()[1],
	       net_local_ip()[2], net_local_ip()[3], a_qrp->qr_ip[0], 
	       a_qrp->qr_ip[1], a_qrp->qr_ip[2], a_qrp->qr_ip[3]);
	push_warn = 1;
      }
      return 0;
    }

    /* This is sort of lame. Right now it sends its push request to all
     * connections, rather than the one connection from which the
     * query-reply actually arrived. Of course, the others will just filter
     * out the PUSH in their routing code, but it would be better if
     * we were smarter and filtered it ourselves. */
    gpa = gp_push_make(conf_get_str("mac"), conf_get_int("ttl"),
		     a_qrp->qr_guid, a_qrp->qr_ref, net_local_ip(),
		     conf_get_int("local_port"));

    query_push_add(a_qrp); /* qr-del */

    send_to_all(gpa);

    fre_v(&(gpa->data), 217);
    fre_gpa(&gpa, 218);
  }
  return 0;
}

int com_push(char *a)
{
  if (recu_click(5)) {
    push_warn = 0;
    input_parse_range(a, com_push_callback, 0);
  }
  return 0;
}

int com_forget_callback(int32 a, void *b)
{
  query_resp *a_qrp;
  
  /* Get the Nth item in the query list, unless it's gone now.
   * If found, query.index will al.locate and return a COPY */
  a_qrp = query_index(a, 'f'); /* qr-new */

  if (a_qrp == 0) {
    /* Usually this happens because the item did not recieve an
     * item number by the last invocation of the 'response' command.
     * And that happens because the item was forgotten by a previous
     * 'forget' command. So, we don't print any message, just return. */
    return 0;
  }

  /* We don't actually do anything with the copy, we just needed
   * query.index to mark it with the 'f' flag. So, now we delete the
   * copy that was made for us. */
  query_kill(&a_qrp, 267); /* 0.4.27.c13 */
  
  return 0;
}

char *cf_tokens[MAX_TOKENS]; /* 0.4.27.c04 */
int cf_nt; /* 0.4.27.c04 */
char cf_buf[256]; /* 0.4.27.c04 */
int cf_isexact; /* 0.4.27.c04 */
uint32 cf_tsize; /* 0.4.27.c04 */

/* This routine is for 0.4.27.c04 */
int com_forget_alpha_filter(query_resp *a_qrp)
{
  int rv;

  strncpy(cf_buf, a_qrp->qr_ndata, 255); /* tagok */
  cf_buf[255] = 0;
  make_lc(cf_buf);
  rv = match_tok_2(cf_tokens, cf_nt, cf_buf, cf_isexact);
  return rv;
}

/* This routine is for 0.4.27.c04 */
int com_forget_size_filter(query_resp *a_qrp)
{
  return (a_qrp->qr_size == cf_tsize);
}

int com_forget(char *arg)
{
  char c;

  if (arg == 0) { /* 0.4.28.c09 */
    return 0;
  }

  while(*arg == ' ') { /* 0.4.27.c04 */
    arg++;
  }
  c = *arg; /* 0.4.27.c04 */
  if (isdigit(c)) { /* 0.4.27.c04 */
    input_parse_range(arg, com_forget_callback, 0);
  } else if (c == '=') { /* 0.4.27.c04 */
    /* Forget queries equal to specified query, boolean AND of keywords
       with exact match option (which the callback passes to match_tok_2) */
    arg++;
    while (*arg == ' ') {
      arg++;
    }
    c = *arg;
    if (isdigit(c)) {
      query_resp *a_qrp;
      int32 n;

      n = strtol(arg, 0, 10);
      a_qrp = query_index(n, 0);
      if (a_qrp) {
	make_lc(a_qrp->qr_ndata);
	cf_nt = sh_tokenize(a_qrp->qr_ndata, cf_tokens, MAX_TOKENS);
	cf_isexact = 1;
	query_tag_filter('f', com_forget_alpha_filter);
	query_kill(&a_qrp, 110);
      }
    }
  } else if (c == 's') { /* 0.4.27.c04 */
    /* Forget queries with same size as specified query */
    arg++;
    while (*arg == ' ') {
      arg++;
    }
    c = *arg;
    if (isdigit(c)) {
      query_resp *a_qrp;
      int32 n;

      n = strtol(arg, 0, 10);
      a_qrp = query_index(n, 0);
      if (a_qrp) {
        cf_tsize = a_qrp->qr_size;
	query_tag_filter('f', com_forget_size_filter);
	query_kill(&a_qrp, 268);
      }
    }
  } else if (c) { /* 0.4.27.c04 */
    /* Filter out by boolean-AND keywords */
    make_lc(arg);
    cf_nt = sh_tokenize(arg, cf_tokens, MAX_TOKENS);
    cf_isexact = 0;
    query_tag_filter('f', com_forget_alpha_filter);
  }

  return 0;
}

int com_get_callback(int32 a, void *b)
{
  char flag; /* 0.4.27.c01 */
  query_resp *a_qrp;

  flag = gc_auto_forget ? 'f' : 'g'; /* 0.4.27.c01 */

  /* Get the Nth item in the query list, unless it's gone now.
   * If found, query.index will al.locate and return a COPY */
  a_qrp = query_index(a, flag); /* 0.4.27.c01 */ /* qr-new */

  if (a_qrp == 0) {
    printf("No such ref num: %i\n", a);
    return 0;
  }

  a_qrp->qr_retry = 1;

  gnut_xfer_start(a_qrp); /* in gnut.c   qr-del */

  return 0;
}

int com_get(char *arg)
{
  if (recu_click(1)) {
    input_parse_range(arg, com_get_callback, 0);
  }
  return 0;
}

#define CSM_NORMAL 0
#define CSM_ZERO   1

int com_stop_mode = CSM_NORMAL;
int32 com_stop_index = 0;
char com_stop_type;
int com_stop_full_zero_pending;

int com_stop_cb2(gnut_transfer *gt)
{
  int dostop;
  int stateok;
  int typeok;
  int indexok;

  dostop = 0;
  stateok = 0;
  typeok = 0;
  indexok = 0;

  /* We check the state partly to avoid printing "stopping transfer 2"
     twice if they type "stop 2,1-3" */
  if ( (gt->state == STATE_CONNECTING)
       || (gt->state == STATE_CONNECTED)
       || (gt->state == STATE_QUEUED)
       || (gt->state == STATE_RETRY_WAIT)
       ) {
    stateok = 1;
  }

  if ((com_stop_type == 'a') || (com_stop_type == 't')) {
    typeok = 1;
  } else if ((com_stop_type == 'g') && (gt->type == TYPE_DOWNLOAD)) {
    typeok = 1;
  } else if ((com_stop_type == 'p') && (gt->type == TYPE_RECV_PUSH)) {
    typeok = 1;
  } else if ((com_stop_type == 'd') && ((gt->type & 1) == 0)) {
    typeok = 1;
  } else if ((com_stop_type == 'c') && (gt->gt_dest_cache)) {
    typeok = 1;
  } else if ((com_stop_type == 'u') && (gt->type & 1)) {
    typeok = 1;
  }

  if (((gt->gt_uindex == com_stop_index)
       || (com_stop_index == -1))) {
    indexok = 1;
  }

  if(com_stop_mode == CSM_NORMAL) {
    dostop = 1;
  } else if (com_stop_mode == CSM_ZERO) {
    if (gt->gt_bytes == 0) {
      dostop = 1;
    }
  }

  if (dostop && stateok && typeok && indexok) {
    char *c;

    printf("Stopping transfer%5u ", gt->gt_uindex); /* 0.4.27.c11 */
    c = gt->fname;
    if (c) {
      if (strlen(c) > 55) {
        c = c + (strlen(c) - 52);
        printf("...");
      }
      printf("%s\n", c);
    } else {
      printf("\n");
    }
    gt->state = STATE_DEAD;
  }
  return 0;
}

int com_stop_cb1(int32 a, void *b)
{
  com_stop_index = a;

  /* Recognize '0' if present */
  if (a == 0) {
    /* From now on until the end of this stop command we're in 0 mode */
    com_stop_mode = CSM_ZERO;

    /* We don't want to do a full scan in zero mode unless the 0 is
       the last number in the parse_range, so we'll remember that with this
       flag */
    com_stop_full_zero_pending = 1;
  } else {
    com_stop_full_zero_pending = 0;
    gnut_xfer_enumerate(com_stop_cb2);
  }
  return 0;
}

int com_stop(char *arg)
{
  char c1;
  char *a2;

  /* Set defaults */
  com_stop_mode = CSM_NORMAL;
  com_stop_type = 'a';
  com_stop_full_zero_pending = 0;

  /* Recognize and strip letter argument if present */
  a2 = arg;
  if (a2) {
    c1 = *a2;
    if ((c1=='d') || (c1=='g') || (c1=='p') || (c1=='u') || (c1=='c')
	|| (c1=='t')) {
      com_stop_type = c1;
      a2 ++;
      while(*a2 == ' ') {
	a2++;
      }
    }
  }

  if (a2 && (*a2)) {
    input_parse_range(a2, com_stop_cb1, 0);
    if (com_stop_full_zero_pending) {
      com_stop_index = -1;
      gnut_xfer_enumerate(com_stop_cb2);
    }
  } else {
    /* #stop# with no argument or just a letter option stops all transfers */
    com_stop_index = -1;
    gnut_xfer_enumerate(com_stop_cb2);
  }

  return 0;
}

int32 com_clear_index = 0;

int com_clear_callback2(gnut_transfer *gt)
{
  if (((gt->gt_uindex == com_clear_index) || (com_clear_index == -1))
      && (gt->state==STATE_DEAD || gt->state==STATE_ERROR)) {
    printf("Clearing transfer: %u\n", gt->gt_uindex); /* 0.4.27.c11 */
    gt->state = STATE_CLEANABLE;
  }
  return 0;
}

int com_clear_callback(int32 a, void *b)
{
  com_clear_index = a;
  gnut_xfer_enumerate(com_clear_callback2);
  return 0;
}

int com_clear(char *arg)
{
  if (arg) {
    input_parse_range(arg, com_clear_callback, 0);
  } else {
    com_clear_index = -1;
    gnut_xfer_enumerate(com_clear_callback2);
  }
  return 0;
}

int32 com_kill_index = 0;

int com_kill_callback2(gcs *gc)
{
  if (gc->uindex == com_kill_index || com_kill_index == -1) {
    printf("Killing %i\n", gc->uindex);
    gc->cstate = STATE_DEAD;
  }
  return 0;
}

int com_kill_callback(int32 a, void *b)
{
  com_kill_index = a;
  gnut_connection_enumerate(com_kill_callback2);
  return 0;
}

int com_kill(char *arg)
{
  if (arg) {
    if (strcmp(arg, "w") == 0) {
      /* kill w */
      gnut_connection_kill_worst(1);
    } else {
      /* kill a number */
      input_parse_range(arg, com_kill_callback, 0);
    }
  } else {
    /* #kill# with no argument kills all */
    com_kill_index = -1;
    gnut_connection_enumerate(com_kill_callback2);
  }
  return 0;
}

int host_disp(void *a, void *b)
{
  host_entry *he;
  uint16 port;
  char ipstr[32];
  char buf[256];
  float64 bytes;
  char mbb[50];

  he = a;
  memcpy(&port, he->port, 2);
  port = GUINT16_FROM_LE(port);

  sprintf(ipstr, "%i.%i.%i.%i:%i", he->ip[0], he->ip[1], he->ip[2],
	  he->ip[3], port);
  bytes = he->kbytes;
  bytes = 1024.0 * bytes;
  if (gc_munge) {
    format_si(bytes, mbb);
    sprintf(buf, " %20s %5d %s\n", ipstr, he->files, mbb);
  } else {
    sprintf(buf, " %20s %5d %-16.15g\n", ipstr, he->files, bytes);
  }

  return cli_output_line(0,buf);
}

int com_hosts(char *arg)
{
  Gnut_Hash *gl;

  if (arg) {
    return host_restore(arg);
  }

  host_lock();
  /* locking for a long time, but oh well, it's the user's fault..! ;) */

  gl = host_retrieve();
  gnut_hash_foreach(gl, host_disp, 0);

  host_unlock();
  return 0;
}

#define MAX_LAST 1024

int com_last(char *a)
{
  int ci;
  char c;
  int in_tag;
  FILE * turd;
  int crflag;
  int i;

  turd = fopen(".gnut_last_turd", "r");
  if (turd == 0) {
    return -1;
  }
  in_tag = 0;
  crflag = 1;
  i = 0;
  while ((i < MAX_LAST) && ((ci = fgetc(turd)) != EOF)) {
    c = ci;
    /* Filter out HTML tags, which are common inside Gnutella server
     * error-turds */
    if (c == '<') {
      in_tag = 1;
    } else if (in_tag) {
      if (c == '>') {
        in_tag = 0;
      }
    } else {
      /* For cross-platform compatibility, we translate both ^J and ^M into
       * the compiler's "\n" */
      if ((c == 10) || (c == 13)) {
        if (crflag) {
        } else {
          putchar('\n');
          i++;
        }
        crflag = 1;
      } else if ((c < 32) || (c > 126)) { /* tagok */
        /* Don't print control chars and non-ASCII */
      } else {
        putchar(c);
        i++;
        crflag = 0;
      }
    }
  }
  fclose(turd);
  if (crflag == 0) {
    putchar('\n');
  }

  /* GRR did it this way, but I don't like to add new OS-dependent code to
   * gnut if I can help it. Also, I wanted to implement filtering of HTML
   * tags and non-printable characters, etc.
   system("/usr/bin/tr '\r' '\n' < .gnut_last_turd"); */

  return 0;
}

int32 com_limit_amount = 0;
int32 com_limit_index = 0;

int com_limit_callback2(gnut_transfer *gt)
{
  if (gt->gt_uindex == com_limit_index || com_limit_index == -1) {
    printf("Limiting %i to %i bytes/s %i\n", gt->gt_uindex, com_limit_amount,
	   gt->gt_bytes);
    gt->rate_limit = com_limit_amount;
  }
  return 0;
}

int com_limit_callback(int a, void *b)
{
  com_limit_index=a;
  gnut_xfer_enumerate(com_limit_callback2);
  return 0;
}

int com_limit(char *arg)
{
  char *ptr,*ptr2;

  if (!arg)
	return 0;
  
  ptr=arg;
  /* Skip to first digit */
  for (; !IS_DIGIT(*ptr) && *ptr!=0; ptr++)
	{ }
  if (ptr==0)
	return -1;
  com_limit_amount=strtol(ptr, &ptr2, 10);

  /* Find range param, or detect lack of same */
  for (ptr=ptr2; !IS_DIGIT(*ptr) && *ptr!=0; ptr++)
	{ }
  if (*ptr!=0) {
	/* A range was specified */
    input_parse_range(ptr2, com_limit_callback, 0);
  } else {
    com_limit_index=-1;
	/* No range, limit all */
    gnut_xfer_enumerate(com_limit_callback2);
  }

  return 0;
}

int com_load(char *arg)
{
  char *buf;

  if (recu_up()) {
    buf = 0;
    if (arg) {
      /* Ignore first space and anything following. The practical benefit of
       * this is for users who use readline completion (hit TAB to complete
       * filename) */
      buf = arg;
      while(*buf) {
        if (*buf == ' ') {
          *buf = 0;
        } else {
	  buf++;
	}
      }
      buf = expand_path(arg);
      printf("Loading config file \"%s\".\n", buf);
      run_config_file(buf, 1, 1, 0);
      fre_str(&buf, 219);
    }
  }
  recu_down();
  return 0;
}

int com_log(char *arg)
{
  if (arg) {
    printf("New log level is: %i\n", gnut_lib_debug=atoi(arg));
  } else {
    printf("Current log level is: %i\n", gnut_lib_debug);
  }
  return 0;
}

int com_share(char *arg)
{
  if (arg) {
    conf_set_str("share_paths",arg);
  } else {
    conf_set_str("share_paths","");
  }
  return 0;
}

int com_scan(char *arg)
{
  char *a;
  char *b;
  char *c;
  uint32 num;
  float64 sh_size;
  char *path;

#if GNUT_HTTP_FILE_LIST || GNUT_HTTP_SEARCH
  gnut_http_template_scan();  
#endif

  gd_s(5, "com_scan calling share.clear\n");
  share_clear();
  gd_s(5, "com_scan done calling share.clear\n");

  b = ystdup(conf_get_str("share_paths"), 478);
  if (b) {
    gd_s(5, "com_scan working through paths\n");
#ifdef HAVE_STRTOK_R
    a = strtok_r(b,path_separator,&c);
#else
    a = strtok(b,path_separator);
#endif
    while (a) {
      path = expand_path(a);
      gd_s(1, "scanning directory: ");
      gd_s(1, path);
      gd_s(1, "\n");
      share_scan_dir(path, 1);
#ifdef HAVE_STRTOK_R
      a = strtok_r(0, path_separator, &c);
#else
      a = strtok(0, path_separator);
#endif
      fre_str(&path, 220);
    }
  
    fre_str(&b, 221);
  }

  share_totals(&num, &sh_size);

  cache_snapshot();

  b = conf_get_str("cache_path");
  if ((b) && (strlen(b)>1)) {
    c = expand_path(b);
    gd_s(2, "scanning cache: ");
    gd_s(2, c);
    gd_s(2, "\n");
    share_scan_dir(c, 1);
    fre_str(&c, 222);
  }

  share_hash_init();

  printf("Scanned %u files and %g bytes.\n", num, sh_size); /* 0.4.27.c11 */

  return 0;
}

/* 0.4.27.c02 */
int cdconf_n;
int cdconf_alloc;
char * cdconflist[100];

int cli_disp_conf(void *a, void *b)
{
  conf_key_pair *ckp;
  char buf[256];
  int i, l; /* 0.4.27.c02 */

  /* Don't display the "hidden" (internal) variables */
  ckp = a;
  if (!strcmp(ckp->ckp_key, "update_guid")) {
    return 0;
  } else if (!strcmp(ckp->ckp_key, "query_guid")) {
    return 0;
  } else if (!strcmp(ckp->ckp_key, "mac")) {
    return 0;
  } else if (!strcmp(ckp->ckp_key, "guid")) {
    return 0;
  } else if (!strncmp(ckp->ckp_key, "GDJ_", 4)) {
    return 0;
  }

  /* 0.4.27.c02 right-justify the variable name without relying on printf */
  l = strlen(ckp->ckp_key);
  for(i=0; i+l < 30; i++) {
    buf[i] = ' ';
  }
  buf[i] = 0;
  strcatlim(buf, ckp->ckp_key, sizeof(buf));
  strcatlim(buf, " = ", sizeof(buf));
  strcatlim(buf, ckp->ckp_val, sizeof(buf));
  strcatlim(buf, "\n", sizeof(buf));

  if(cdconf_n < cdconf_alloc) { /* 0.4.27.c02 */
    cdconflist[cdconf_n] = ystdup(buf, 555);
    cdconf_n++;
  }
  return 0;
}

/* 0.4.27.c02 Compare function for qsort */
int comset_cmpfunc(const void * a, const void * b)
{
  char * c;
  char * d;
  int rv;

  c = (char *) *((char **)a); d = (char *) *((char **)b);
  while (*c && (*c == ' ') ) {
    c++;
  }
  while (*d && (*d == ' ') ) {
    d++;
  }
  rv = strcmp(c,d);
  return rv;
}

int com_set(char *arg)
{
  char *arg2;
  char *b;
  int i;

  if (!arg) {
    /* display a list of all config values */
    /* 0.4.27.c02 */
    cdconf_n = 0;
    cdconf_alloc = 100;
    gnut_hash_foreach(conf_hash, cli_disp_conf, 0);
    qsort(cdconflist, cdconf_n, sizeof(char *), comset_cmpfunc);
    cli_output_reset();
    for(i=0; i<cdconf_n; i++) {
      cli_output_line(0, cdconflist[i]);
      fre_str(&(cdconflist[i]), 556);
    }
    return 0;
  }

  /* scan through first argument (variable to set), also changing any '-'
   * characters to '_' */
  for (arg2=arg; *arg2 && !isspace(*arg2); arg2++) {
    if (*arg2 == '-') {
      *arg2 = '_';
    }
  }

  if (*arg2==0) {
    arg2=0;
  } else {
    *arg2=0;
    arg2++;
    for (; (*arg2) && isspace(*arg2); arg2++) {
    }

    if ((*arg2)==0) {
      arg2 = 0;
    }
  }

  if (arg2==0) {
    /* we'll display the current value... */
    b = conf_get_str(arg);
    if (echo_comset) {
      if (b == 0) {
	printf("No such configuration option: %s\n", arg);
      } else {
	printf("Config item: %s = %s\n", arg, b);
      }
    }
  } else {
    i = conf_set_str(arg, arg2);
    if (echo_comset) {
      if (i<0) {
	printf("No such configuration option: %s\n", arg);
      } else {
	printf("Config item: %s = %s\n", arg, arg2);
      }
    }
  }
  return 0;
}

int com_help(char *arg)
{
  int cnum;
  char *s;

  if (arg) {
    /* 0.4.27.c14 */
    if(gl_strncmp(arg, "copying", 7, 1) == 0) {
      cli_output_reset(); gpl_copying();
    } else if(gl_strncmp(arg, "gpl", 3, 1) == 0) {
      cli_output_reset(); gpl_full();
    } else if(gl_strncmp(arg, "warranty", 8, 1) == 0) {
      cli_output_reset(); gpl_no_warranty();
    } else {
      cnum = get_command_index(arg);
      if (cnum >=0 ) {
	s = commands[cnum].longdoc;
	if (s) {
	  /* command has long help */
	  printf("%s", s);
	} else {
	  /* just print the normal help */
	  printf("\n  %s: %s\n", commands[cnum].name, commands[cnum].doc);
	}
      }
    }
  } else {
    printf("\n");
    for (cnum=0; commands[cnum].name; cnum++) {
      printf("  %s: %s\n", commands[cnum].name, commands[cnum].doc);
    }
    printf("\nType 'help command' for more extensive help on 'command'.\n");
    gpl_small();
  }
  printf("\n");
  return 0;
}

int com_quit(char *arg)
{
  char *d;

#ifdef DMALLOC
  dmalloc_shutdown();
#endif

  gd_s(3, "com_quit calling expand_path\n");
  d = expand_path("~/.gnut_hosts");
  gd_s(3, "com_quit calling host_save\n");
  host_save(d);
  gd_s(3, "com_quit fr""ee'ing mem\n");
  fre_str(&d, 223); /* Why do we even bother? (-; */

  exit(0);
  return 0;
}

int com_version(char *arg)
{
  printf("gnut version %s (c) 2000-2001 Josh Pieper and Robert Munafo\n", VERSION);
  return 0;
}

int monitor_query = 0;
int monitor_reply = 0;
int monitor_push = 0;
char * monitor_filter = 0;

void monitor_callback()
{
  monitored_search *ms;
  int accept;

  while((ms = monitor_search_get())) {
    make_lc(ms->search);
    if (monitor_filter) {
      accept = keyword_match(monitor_filter, ms->search, 1, 0);
    } else {
      accept = 1;
    }

    if (accept) {
      if (monitor_query) {
	printf("Search: '");
      } else if (monitor_reply) {
	printf("Search reply: '");
      } else {
	printf("PUSH request: '");
      }
      /* Print out only the ASCII characters in the search string */
      print_ascii(ms->search);

      printf("' %s\n", ms->matched ? " (matched)" : "");
    }

    fre_str(&(ms->search), 224);
    fre_ms(&ms, 225);
    fflush(stdout);
  }
}

int com_monitor(char *arg)
{
  monitored_search *ms;

  /* tell the packet-handling thread to watch the searches */
  monitor_query = 1;

  printf("Monitoring search requests.\nPress any key to continue\n");

  if (arg) {
    make_lc(arg);
  }
  monitor_filter = arg;
  wait_key_callback(monitor_callback);

  printf("\n");

  /* tell the other threads to stop adding the searches to the list */
  monitor_query = 0;

  /* delete the rest of the list */
  while((ms = monitor_search_get())) {
    fre_str(&(ms->search), 226);
    fre_ms(&ms, 227);
  }

  return 0;
}

int com_mreply(char *arg)
{
  monitored_search *ms;

  /* tell the packet-handling thread to watch the packets */
  monitor_reply = 1;

  printf("Monitoring search replies.\nPress any key to continue\n");

  if (arg) {
    make_lc(arg);
  }
  monitor_filter = arg;
  wait_key_callback(monitor_callback);

  printf("\n");

  /* tell the other threads to stop adding the replies to the list */
  monitor_reply = 0;

  /* delete the rest of the list */
  while((ms = monitor_search_get())) {
    fre_str(&(ms->search), 228);
    fre_ms(&ms, 229);
  }

  return 0;
}

int com_mpush(char *arg)
{
  monitored_search *ms;

  /* tell the packet-handling thread to watch the packets */
  monitor_push = 1;

  printf("Monitoring PUSH requests.\nPress any key to continue\n");

  if (arg) {
    make_lc(arg);
  }
  monitor_filter = arg;
  wait_key_callback(monitor_callback);

  printf("\n");

  /* tell the other threads to stop adding the PUSHs to the list */
  monitor_push = 0;

  /* delete the rest of the list */
  while((ms = monitor_search_get())) {
    fre_str(&(ms->search), 230);
    fre_ms(&ms, 231);
  }

  return 0;
}

static char **com_play_scratch = 0;
static int com_play_scratch_size = 0, com_play_scratch_index;

void com_play_append_string(char *str)
{
  /* make sure we have enough room */
  if ((com_play_scratch_index + 1) >= com_play_scratch_size) {
    char ** cps_new;
    int cpss;

    cpss = com_play_scratch_size * 1.5;
    cps_new = (char **) ycaloc(cpss * sizeof(char *), 1, 492);
    memcpy(cps_new, com_play_scratch, com_play_scratch_size * sizeof(char *));
    com_play_scratch_size = cpss;
    fre_strl(&com_play_scratch, 493);
    com_play_scratch = cps_new;
  }

  /* append a copy of this string, make sure the array is terminated
     (in case of an early bailout in com_play) */
  com_play_scratch[com_play_scratch_index++] = ystdup(str, 479);
  com_play_scratch[com_play_scratch_index] = 0;
}

int com_play_callback(int32 a, void *arg)
{
  char *tmp;
  query_resp *a_qrp;
  uint16 port;
  int len;

  a_qrp = query_index(a, 0); /* qr-new */
  if (a_qrp == 0) {
    printf("No such ref num: %i\n", a);
    return 0;
  }

  memcpy(&port, a_qrp->qr_port, 2);
  port = GUINT16_FROM_LE(port);

  len = a_qrp->qr_nlen + 60; /* 0.4.27.c13 */
  tmp = ymaloc(len, 361);
  sprintf(tmp, "http://%i.%i.%i.%i:%i/get/%u/%s", /* 0.4.27.c11 */
    a_qrp->qr_ip[0], a_qrp->qr_ip[1], a_qrp->qr_ip[2], a_qrp->qr_ip[3],
    port, a_qrp->qr_ref, a_qrp->qr_ndata);
  com_play_append_string(tmp);
  fre_str(&tmp, 232);

  query_kill(&a_qrp, 546); /* qr-del */ /* 0.4.27.c13 */

  return 0;
}

int com_play(char *arg)
{
#ifdef WIN32
  return 0;
#else
  int i;
  char *temp, *a, *b;

  /* al.locate the play command argument storage if necessary */
  if (!com_play_scratch) {
    com_play_scratch_size = 100;
    com_play_scratch = (char **)
      ymaloc(com_play_scratch_size * sizeof(char *), 422);
  } else {
    /* clear any previous commands */
    i = 0;
    while (com_play_scratch[i]) {
      fre_str(&(com_play_scratch[i++]), 233);
    }
  }

  com_play_scratch[0] = 0;

  com_play_scratch_index = 0;

  temp = ystdup(conf_get_str("play_prog"), 480);
#ifdef HAVE_STRTOK_R
  a = strtok_r(temp," \t",&b);
#else
  a = strtok(temp," \t");
#endif
  if (!a) {
    printf("Empty play_prog configured!\n");
    return 0;
  }

  while (a) {
    com_play_append_string(a);
#ifdef HAVE_STRTOK_R
    a = strtok_r(0, " \t", &b);
#else
    a = strtok(0, " \t");
#endif
  }

  fre_str(&temp, 234);

  /* now parse the arguments to "play" parsing out ranges for
     any argument starting with a digit */

  /* Ray, that str.dup wasn't necessary. ;) */
  temp = arg;

#ifdef HAVE_STRTOK_R
  a = strtok_r(temp," \t",&b);
#else
  a = strtok(temp," \t");
#endif

  while (a) {
    if (IS_DIGIT(a[0])) {
      input_parse_range(a, com_play_callback, 0);
    } else {
      com_play_append_string(a);
    }
#ifdef HAVE_STRTOK_R
    a = strtok_r(0, " \t", &b);
#else
    a = strtok(0, " \t");
#endif
  }
  printf("The final command was:");
  i = 0;
  while (com_play_scratch[i]) {
    printf(" %s", com_play_scratch[i++]);
  }
  printf("\n");

  if (!fork()) {
    execvp(com_play_scratch[0], com_play_scratch);
    printf("Failure executing %s\n", com_play_scratch[0]);
    exit(0);
  }

  return 0;
#endif
}

int com_debug(char *a)
{
#if 0
  Gnut_List *gl;
  Gnut_List *tmp;
  share_item *si;

  gl = share_cache_run_search("mpg",64);

  for (tmp = gl; tmp; tmp=tmp->next) {
    si = tmp->data;
    printf("ref: %i  name: %s\n", si->ref, si->fpath);
  }

  share_clear_list(gl);
#else
  tell_leaks();
#endif

  /* print_share(); */

  /* %% include "ss_bench.c" */

  /* printf("4294960000  %u\n", gl_stou32("4294960000", 0));
  printf("0  %u\n", gl_stou32("0", 0));
  printf(" +23-45  %u\n", gl_stou32(" +23-45", 0)); */

  return 0;
}

int com_lshare(char *a)
{
  Gnut_List *gl;
  Gnut_List *tmp;
  share_item *si;

  gl = share_copy_list(share_root);

  for (tmp=gl; tmp; tmp=tmp->next) {
    si = tmp->data;
    printf("%3u  %s\n", si->ref, si->fpath); /* 0.4.27.c11 */
  }

  share_clear_list(gl);
  return 0;
}

int com_sleep(char *a)
{
  if (a) {
    sleep(atoi(a));
  }
  return 0;
}

static char *get_arg_string(char *args, char *key)
{
  char *arg_string;
  char *s;
  char *new_arg_string;
  int len = 0;

  arg_string = strstr(args, key);
  if (!arg_string) {
    return 0;
  }

  arg_string += strlen(key)+1;
  if (!*arg_string) {
    return 0;
  }

  s = arg_string;

  while ((*s) && (*s != ' ')) {
    s++;
    len++;
  }

  new_arg_string = (char *)ymaloc(len+1, 362);

  strncpy(new_arg_string, arg_string, len);
  *(new_arg_string+len) = '\0';

  return new_arg_string;
}

static int blacklist_printer(blacklist_t *bl, void *data)
{
  printf("-------------------------------------------\n");

  printf("Type: %c%c%c%c%c\n",
	 (bl->type & BLACKLIST_TYPE_INCOMING) ? 'I' : ' ',
	 (bl->type & BLACKLIST_TYPE_OUTGOING) ? 'O' : ' ',
	 (bl->type & BLACKLIST_TYPE_SEARCH) ? 'S' : ' ',
	 (bl->type & BLACKLIST_TYPE_PING) ? 'P' : ' ',
	 (bl->type & BLACKLIST_TYPE_HTTP) ? 'H' : ' ');


  printf("Filter on: %c%c%c\n",
	 (bl->which & BLACKLIST_HAVE_SEARCH) ? 'S' : ' ',
	 (bl->which & BLACKLIST_HAVE_ADDRESS) ? 'A' : ' ',
	 (bl->which & BLACKLIST_HAVE_PORT) ? 'P' : ' ');

  if (bl->which & BLACKLIST_HAVE_SEARCH) {
    printf("Search String: %s\n", bl->search_string);
  }
  if (bl->which & BLACKLIST_HAVE_ADDRESS) {
    printf("Address %i.%i.%i.%i\n",
	   (bl->addr.s_addr) & 0x0ff,
	   (bl->addr.s_addr >> 8) & 0x0ff,
	   (bl->addr.s_addr >> 16) & 0x0ff,
	   (bl->addr.s_addr >> 24) & 0x0ff);
  }
  if (bl->which & BLACKLIST_HAVE_PORT) {
    printf("Port: %i\n", bl->port);
  }
  printf("Hits %i\n", (int)bl->hits);
  return 0;
}


int com_blacklist(char *arg)
{
  char *host;
  char *port;
  char *search;
  char *subnet;
  int filter_incoming;
  int filter_outgoing;
  int filter_ping;
  int filter_search;
  int filter_http;

  if (!arg || !strlen(arg)) {
	/* no arg: just print list of all blacklisted hosts */
    gnut_blacklist_foreach(blacklist_printer, 0);
    return 0;
  }

  host = get_arg_string(arg, "host");
  port = get_arg_string(arg, "port");
  subnet = get_arg_string(arg, "subnet");
  search = get_arg_string(arg, "search");

  if (strstr(arg, "all")) {
    filter_incoming = 1;
    filter_outgoing = 1;
    filter_ping = 1;
    filter_search = 1;
    filter_http = 1;
  } else {
    filter_incoming = strstr(arg, "incoming") ? 1 : 0;
    filter_outgoing = strstr(arg, "outgoing") ? 1 : 0;
    filter_ping = strstr(arg, "ping") ? 1 : 0;
    filter_search = strstr(arg, "search") ? 1 : 0;
    filter_http = strstr(arg, "http") ? 1 : 0;
  }

  gnut_blacklist_add(search, host, subnet, port, filter_incoming,
					 filter_outgoing, filter_ping, filter_search,
					 filter_http);
  return 0;
}

int com_player(char *arg)
{
  char *mime_type;
  char *extension;
  char *player;

  mime_type = get_arg_string(arg, "mime_type");
  extension = get_arg_string(arg, "extension");
  player = get_arg_string(arg, "player");

  if (mime_type && extension && player) {
    player_add(mime_type, extension, player);
  } else {
    printf("Must provide 'mime_type', 'extension' and 'program'. See manual for details.\n");
  }

  return 0;
}

int com_cls(char *arg)
{
#ifndef WIN32
  system("clear");
#endif
  return 0;
}

int com_shell(char *arg)
{
  char *shellprog;

#ifndef WIN32
  if(arg == 0) {
    if((shellprog = getenv("SHELL")))
      return system(shellprog);
    else
      return system("/bin/sh");
  }
  /* else */
  return system(arg);
#else
  return 0;
#endif
}

int com_eval(char *arg)
{
  FILE *p = 0;
  char buf[1024] = "";

  if (recu_up()) {
#ifndef WIN32
    if (arg) {
      p = popen(arg, "r");
      if (!p) {
	return -1;
      }
      while (fgets(buf, sizeof(buf), p)) {
	if ((buf[strlen(buf)-1] == '\n')
	    || (buf[strlen(buf)-1] == '\r')
	    ) {
	  buf[strlen(buf)-1] = 0;
	}
		
        if(gc_eval_echo) {
          printf("%s\n", buf);
        }
	parse_command(buf);
      }
      pclose(p);
    }
#endif
  }
  recu_down();
  return 0;
}

int cli_save_conf(void *a, void *b)
{
  conf_key_pair *ckp = a;
  FILE *file = b;

  if ( (strcmp(ckp->ckp_key,"update_guid") == 0)
	   || (strcmp(ckp->ckp_key,"query_guid") == 0)
	   || (strcmp(ckp->ckp_key,"mac") == 0)
	   || (strcmp(ckp->ckp_key,"guid") == 0)
	   ) {
	return 0;
  }

  fprintf(file, "set %s %s\n", ckp->ckp_key, ckp->ckp_val);
  return 0;
}

int com_save(char *arg)
{
  struct stat  stats;
  FILE *file = 0;
  char *cfg = 0;
  char *tmp = 0;

  /* save config strings */
  if (!arg)	{
    tmp = conf_get_str("config_file");
    if (!tmp || !strlen(tmp))	{
      printf("Undefined configuration file (config string \"config_file\").\nPlease see the manual for more information.\n");
      return 0;
    }
    cfg = expand_path(tmp);
  } else {
    cfg = expand_path(arg);
  }

  if (stat(cfg, &stats) != -1)	{
    printf("Replacing existing file \"%s\".\n", cfg);
  } else {
    printf("Creating file \"%s\".\n", cfg);
  }
  file = fopen(cfg, "w");
  if (!file) {
    printf("Couldn't open file \"%s\" for writing.\n", cfg);
    return 0;
  }

  gnut_hash_foreach(conf_hash, cli_save_conf, file);

  /* Add a 'scan' command at the end, if they have share paths */
  tmp = conf_get_str("share_paths");
  if (tmp && strlen(tmp))	{
    fprintf(file, "scan\n");
  }
  fflush(file);
  fclose(file);
  fre_str(&cfg, 235);

  /* save hosts */
  tmp = expand_path("~/.gnut_hosts");
  host_save(tmp);
  fre_str(&tmp, 236);

  /* save blacklists */
  if (gnut_blacklist_totals()) {
    gnut_blacklist_save();
  }

  return 0;
}

int cgml_time = 0;
uint32 old_hb;

/* Check the gnut background thread to see if it died. If it goes more than
   10 seconds without updating its heartbeat, it's dead. */
void check_gnut_mainloop()
{
  if (time(0) > cgml_time) {
    if (gnut_heartbeat == old_hb) {
      printf("gnut: Restarting background loop.\n");
      trigger_bug1 = 0;
      gnut_start_main_loop();
    }
    old_hb = gnut_heartbeat;
    cgml_time = time(0) + 10;
  }
}

/* This is the main loop that parses commands. It actually runs in the
 * main (original) thread.
 * There is another "main loop", main_loop() in gnut.c, that does
 * the background monitoring of the Gnutellanet connections. */
int command_loop()
{
  char *a;
  int ret = 0;
  char buf[256];

  if (gh_error) {
    printf("\nDidn't find ~/.gnut_hosts file. You will have to manually\nopen connections with the 'open' command. Check README, INSTALL,\nTUTORIAL, manual and/or web site for more information.\n\n");
  }

  com_version(0); /* 0.4.27.c14 */
  gpl_small(); /* 0.4.27.c14 */
  printf(UI_IN_VERS_SO, VERSION);

  recu_level = 0;
  recu_disable = 0;

  {
    uint h_num;

    host_totals(&h_num, 0, 0);
    if (h_num <= 1) {
      /* 0.4.27.c26 no longer requires gc_debug_opts, and note new
       servers. Also, once it gets a few hosts it doesn't connect to
       any other host servers. */
      printf("No hosts -- trying hostcache servers...");

      /* Open connections to standard host cache servers. Each time,
       * a PING will get sent if our host cache is still under 10 hosts.
       * We sleep(1) between each one to give each one a chance to
       * send us something before we bother the next one */
      sprintf(buf, "connect%c.gnutellanet.com:6346",
	      (((int) time(0)) % 4) + '1');
      com_open(buf);
      sleep(1);
      host_totals(&h_num, 0, 0);
      if (h_num <= 1) {
	com_open("gnutella.hostscache.com:6346");
	printf(".");
	sleep(1);
      }
      host_totals(&h_num, 0, 0);
      if (h_num <= 1) {
	com_open("router.limewire.com:6346");
	printf(".");
	sleep(1);
      }
      host_totals(&h_num, 0, 0);
      if (h_num <= 1) {
	com_open("gnutellahosts.com:6346");
	printf(".");
	sleep(1);
      }
      host_totals(&h_num, 0, 0);
      if (h_num <= 1) {
	com_open("gnet1.ath.cx:6346");
      }
      printf("\n");
    }
  }

  while (ret >= 0) {
#ifdef HAVE_READLINE
    a = prompt_str();
    readline_crashed = 1;
    a = readline(a);
    readline_crashed = 0;
#else
    printf(prompt_str());
    fflush(stdout);
    a = fgets(buf, 256, stdin); /* tagok */
#endif
    readline_firsttime = 0;

    if (a==0) {
      return 0;
    }

#ifdef HAVE_READLINE
    if (strlen(a) >= 1) {
      add_history(a);
      ret = parse_command(a);
    }
    free(a);
#else
    buf[strlen(buf)-1]=0;
    if (strlen(buf) >= 1) {
      ret = parse_command(buf);
    }
#endif

    check_gnut_mainloop();
  }

  return 0;
}

int usage()
{
  printf("Usage gnut [OPTION] ... [HOST] ...\n");
  printf("Client for the Gnutella distributed file sharing network.\n\n");
  printf("  -l [arg]        sets the initial logging level to arg\n");
  printf("  -c [file]       specify a script to run instead of ~/.gnutrc\n");
  printf("  -i [ip]         IP address to broadcast as belonging to us\n");
  printf("  -i [interface]  which interface (eth0, eth1, etc.) to use\n");
  printf("  -p [port]       port to listen on\n");
  printf("  -d              daemon mode, never ask for input\n");
  printf("  -x              exit after having run any scripts\n");
  printf("  -v              output version information and exit\n");
  printf("  -h              display this help and exit\n\n");
  printf("Report bugs to josh@nemonet.com\n");
  return 0;
}

#ifdef MCHECK
void mcheck_callback(enum mcheck_status a)
{
  printf("mcheck called!  a=%i  PID=%i\n", a, getpid());
  while (1) {
    sleep(1);
  }
}
#endif

int main(int argc, char *argv[])
{
  int daemon=0;
  int ex=0;
  int c;
  char *d;
  char *config_file;
  char *iface;
#ifdef HAVE_GETOPT_LONG
  int option_index = 0;
#endif
  char *gnut_short_options = "xdhvc:l:p:i:";

#ifdef HAVE_GETOPT_LONG
  static struct option gnut_long_options[] =
  {
    {"exit", 0, 0, 'x'},
    {"daemon", 0, 0, 'd'},
    {"help", 0, 0, 'h'},
    {"version", 0, 0, 'v'},
    {"config", 1, 0, 'c'},
    {"loglevel", 1, 0, 'l'},
    {"port", 1, 0, 'p'},
    {"interface", 1, 0, 'i'},
    {"ip", 1, 0, 'i'},
    {0, 0, 0, 0}
  };
#endif

  config_file = 0;
  iface = 0;

  init_leaks();
  dq_init();

  readline_firsttime = 1;
  readline_crashed = 0;

#ifdef MCHECK
  if (mcheck(mcheck_callback) != 0) {
    printf("couldn't do mcheck!\n");
    exit(0);
  }
#endif

  conf_init();
  cli_output_init();

#ifdef HAVE_READLINE
  initialize_readline();
#endif

#ifdef HAVE_GETOPT_LONG
  while ((c=getopt_long(argc,argv, gnut_short_options, gnut_long_options,&option_index))!=EOF)
#else
    while ((c=getopt(argc,argv,gnut_short_options))!=EOF)
#endif
      {
	switch(c) {
	case 'p':
	  conf_set_int("local_port",atoi(optarg));
	  break;
	case 'i':
	  iface = optarg;
	  break;
	case 'x':
	  ex = 1;
	  break;
	case 'd':
	  daemon=1;
	  break;
	case 'l':
	  gnut_lib_debug = atoi(optarg);
	  break;
	case 'v':
	  printf("gnut version %s at your service\n\n",VERSION);
	  return 0;
	case 'c':
	  config_file = optarg;
	  break;
	case 'h':
	  usage();
	  return 0;
	}
      }

  /* Run the config file to get variable settings */
  if (config_file) {
    d = expand_path(config_file);
    run_config_file(d, 0, 0, 0);
    fre_str(&d, 401);
  } else {
    d = expand_path("~/.gnutrc");
    run_config_file(d, 0, 0, 0);
    fre_str(&d, 402);
  }

  dq_hi_init();
  net_init(iface);
  gnut_init();
  gnut_start_main_loop();

  if (argv[optind]) {
    gnut_outgoing_new(argv[optind]);
  }

  /* load any blacklist settings saved via a previous #save# command. */
  gnut_blacklist_init();

  /* Run the config file to execute commands */
  if (config_file) {
    d = expand_path(config_file);
    run_config_file(d, 1, 1, 0);
    fre_str(&d, 403);
  } else {
    d = expand_path("~/.gnutrc");
    run_config_file(d, 1, 1, 0);
    fre_str(&d, 404);
  }

  if (ex == 1) {
    exit(0);
  }

  /* Scan cache directory so the cached files will be shared right away */
  rescan_cache();

  if (daemon == 0) {
    command_loop();
  }

  /* daemon (no output) mode is for acting as a relay node or for HTTP-only
     access */
  {
    int32 ch = 0;
    int32 cn = 0;
    int32 cml = 0;
    int32 chi, cni, cmli;
    int32 when, what;
    int32 forever;

    printf("gnut %s Entering no output mode, have fun!\n", VERSION);

    forever = 999999999;

    chi = conf_get_int("checkpoint_hosts");
    if (chi <= 0) {
      chi = forever;
    } else if (chi <120) {
      chi = 120;
    }

    cni = conf_get_int("checkpoint_net");
    if (cni <= 0) {
      cni = forever;
    } else if (cni < 10) {
      cni = 10;
    }

    cmli = 2;

    ch = chi; cn = cni; cml = cmli;
    while(1) {
      /* find out what action needs to be done first */
      what = 1;
      when = ch;
      if (cn < when) {
	what = 2;
	when = cn;
      }
      if (cml < when) {
	what = 3;
	when = cml;
      }

      /* sleep the necessary amount */
      if (when > 0) {
        sleep(when);
        ch -= when;
        cn -= when;
        cml -= when;
      }

      /* perform the scheduled action, and reset its counter for the next
         time */
      if (what == 1) {
	ch += chi;

	gd_s(3, "checkpoint calling expand_path\n");
	d = expand_path("~/.gnut_hosts");
	gd_s(3, "checkpoint calling host_save\n");
	host_save(d);
	gd_s(3, "checkpoint fr""ee'ing mem\n");
	fre_str(&d, 539);
      } else if (what == 2) {
	cn += cni;

	gd_s(3, "checkpoint calling net_init\n");
	net_init(iface);
      } else if (what == 3) {
	cml += cmli;

	gd_s(3, "checkpoint calling check_gnut_mainloop\n");
	check_gnut_mainloop();
      }
    }
  }

  return 0;
}
