/* cache.c
  Copyright (c) 2000 Josh Pieper
  Copyright (c) 2000-2001 Robert Munafo

  This file is distributed under the GPL, see file COPYING for details */

#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>

#ifndef WIN32
# include <unistd.h>
#endif

#include <errno.h>
#include <string.h>
#include <pthread.h>
#include <ctype.h>

#include "lib.h"
#include "list.h"
#include "share.h"
#include "conf.h"
#include "threads.h"


/* Global variables */
uint32 share_cache_divider; /* 0.4.27.c11 anything > than this is in the cache */
float64 share_siz_nocache;

#ifndef PTHREADS_DRAFT4
  pthread_mutex_t ctfu_mutex = PTHREAD_MUTEX_INITIALIZER;
#else
  pthread_mutex_t ctfu_mutex;
#endif

/* Local variables */
int cache_last_update = 0;
int cache_locked = 0;

void cache_snapshot()
{
  share_cache_divider = g_share_num;
  share_siz_nocache = g_share_size;
}

/* Gnut_List *cache_clear.list(Gnut_List *l)
 *
 * clears the share list specified by l of any cache entries, and sets
 * share_num and share_size to their non-cache values */
Gnut_List *cache_clear_list(Gnut_List *list)
{
  share_item *si;

  gd_s(3, "cache_clear.list entering\n");

  pthread_mutex_lock(&share_mutex);

  /* NOTE
   * This code relies on the cache being at the head of the share list.
   * If this behavior changes, this code will break. */
  while (list) {
    si = (share_item *) list->data;
    if (si->ref > share_cache_divider) {
      share_delete(si, 0);
      list = gnut_list_remove(list, si, 1);
      fre_si(&si, 145); /* 0.4.27.c29 */
    } else {
      break;
    }
  } 

  g_share_num = share_cache_divider;
  g_share_size = share_siz_nocache;

  pthread_mutex_unlock(&share_mutex);

  gd_s(3,"cache_clear.list returning\n");
  return list;
}

/* int rescan.cache()
 *
 * clear the share list of cached items, and rescan the cache into the
 * share */
int rescan_cache()
{
  char *b;
  char *c;

  gd_s(3, "rescan_cache entering\n");

  share_root = cache_clear_list(share_root);

  b = conf_get_str("cache_path");
  if ((b) && (strlen(b)>1)) {
    c = expand_path(b);
    gd_s(5, "scanning cache: "); gd_s(5, c); gd_s(5, "\n");
    share_scan_dir(c, 0);
    fre_str(&c, 405);
  }

  gd_s(3,"rescan_cache returning\n");
  return 0;
}

int cppf;

/* callback, prints a share entry */
int cache_print(void * data, void * user_data)
{
  share_item *si;

  gd_s(3, "cache_print entering\n");

  si=data;
  if (si->ref > share_cache_divider) {
    if (cppf == 0) {
      printf("CACHE:\n");
    }
    cppf++;
    printf("%3d  %s\n", cppf, si->path);
  }

  gd_s(3,"cache_print returning\n");
  return 0;
}

void print_share()
{
  int dt;

  gd_s(3, "print_share entering\n");

  pthread_mutex_lock(&share_mutex);

  /* printf("CACHE is %s\n", cache_locked ? "LOCKED" : "UNLOCKED"); */

  if (cache_last_update) {
    dt = conf_get_int("cache_refresh");

    if (dt > 0) {
      printf("clu %d dt %d time0 %d\n", cache_last_update, dt, (int) (time(0)));
      dt = cache_last_update + dt - ((int) time(0));
      if (dt > 0) {
        printf("Next update in: %d\n", dt);
      }
    }
  }

  cppf = 0;
  gnut_list_foreach(share_root, cache_print, 0);
  if (cppf == 0) {
    printf("CACHE: empty.\n");
  }

  pthread_mutex_unlock(&share_mutex);

  gd_s(3,"print_share returning\n");
}

int cache_total_size(void * data, void * user_data)
{
  share_item *si;
  float64 *total_size;

  gd_s(3, "cache_total_size entering\n");

  si = data;
  total_size = user_data;

  if (si->ref > share_cache_divider) {
    (*total_size) += ((float64) si->size);
  }

  gd_s(3,"cache_total_size returning\n");
  return 0;
}

int cache_find_oldest(void * data, void * user_data)
{
  share_item *si;
  share_item **ret_si;

  gd_s(3, "cache_find_oldest entering\n");

  si = data;
  ret_si = user_data;

  if ((si->ref > share_cache_divider) &&
      ((*ret_si == 0) ||
       ((*ret_si)->mtime > si->mtime))) {
    *ret_si = si;
  }

  gd_s(3,"cache_find_oldest returning\n");
  return 0;
}

int cache_shrink(float64 desired_fre)
{
  float64 max_cache_sz, cache_sz;
  share_item *si;

  gd_s(3, "cache_shrink entering\n");

  max_cache_sz = ((double) conf_get_int("cache_size")) * 1000.0;
  if ((max_cache_sz - desired_fre) < 0) {
    fprintf(stderr, "Trying to fr""ee more space in cache than cache size!\n");
    gd_s(3, "cache_shrink a returning -1\n");
    return -1;
  }

  pthread_mutex_lock(&share_mutex);
  cache_sz = 0;
  gnut_list_foreach(share_root, cache_total_size, &cache_sz);
  pthread_mutex_unlock(&share_mutex);

  while ((max_cache_sz - cache_sz) < desired_fre) {
    si = 0;
    pthread_mutex_lock(&share_mutex);
    gnut_list_foreach(share_root, cache_find_oldest, &si);
    pthread_mutex_unlock(&share_mutex);

    if (si == 0) {
      fprintf(stderr, "Failed to find an oldest member of the cache!\n");
      gd_s(3, "cache_shrink b returning -1\n");
      return -1;
    }

    /* remove the oldest file in the cache, and adjust the cache size */
    if (unlink(si->fpath) != 0) {
      fprintf(stderr, "Failed to remove %s from cache!\n", si->fpath);
      gd_s(3, "cache_shrink c returning -1\n");
      return -1;
    }

    pthread_mutex_lock(&share_mutex);
    cache_sz -= ((float64) si->size);
    g_share_size -= ((float64) si->size);
    g_share_num--;

    share_delete(si, 0);
    share_root = gnut_list_remove(share_root, si, 2);
    fre_si(&si, 155); /* 0.4.27.c29 */
    pthread_mutex_unlock(&share_mutex);
  }

  gd_s(3, "cache_shrink returning 0\n");

  return 0;
}

int ctfu_active = 0;

int cache_time_for_update()
{
  int dt;
  char *c;
  int rv;

  rv = 0;

  pthread_mutex_lock(&ctfu_mutex);

  /* This is a partial test for whether the cache is enabled, we also
   * test cache_refresh below */
  c = conf_get_str("cache_path");
  if (c) {
    /* It's set */
    if (strlen(c) > 1) {
      /* Yes, the cache is probably enabled */

      /* is there already a cache download going on? */
      if (ctfu_active >= gc_cache_parallel) {
	rv = 0;
      } else {
	dt = conf_get_int("cache_refresh");
	if (dt > 0) {
	  if (time(0) > (cache_last_update + dt)) {
	    rv = 1;
	    cache_last_update = time(0);
	  }
	}
      }
    }
  }
  pthread_mutex_unlock(&ctfu_mutex);

  /* no cache path, or no update time set */
  return rv;
}

int cache_lock()
{
  int rv;

  pthread_mutex_lock(&ctfu_mutex); {
    if (ctfu_active >= gc_cache_parallel) {
      /* we're already busy enough! */
      rv = 0;
    } else {
      ctfu_active++;
      rv = 1;
    }
  } pthread_mutex_unlock(&ctfu_mutex);

  return rv;
}

void cache_unlock()
{
  pthread_mutex_lock(&ctfu_mutex); {
    if (ctfu_active) {
      ctfu_active--;
    }
  } pthread_mutex_unlock(&ctfu_mutex);
}

void cache_timestamp()
{
  cache_last_update = time(0);
}

/* Due to a bug in the transfer thread code or the query packet routines,
   (nobody knows where) transfer threads sometimes finish abnormally, and
   don't clean up after themselves. However they leave the filename field
   blank, which gives us an easy way to detect when it has happened.
   cache_fixlock detects any dead threads and resets ctfu_active
   appropriately. */

int cfl_ccount;
int cfl_time = 0;

int cfl_callback(gnut_transfer *gt)
{
  if (gt->fname) {
    if (gt->gt_dest_cache) {
      cfl_ccount++;
    }
  }
  return 0;
}

void cache_fixlock()
{
  if (time(0) > cfl_time) {
    cfl_ccount = 0;
    gnut_xfer_enumerate(cfl_callback);
    ctfu_active = cfl_ccount;
    cfl_time = time(0) + gc_cache_refresh;
  }
}
