/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *\
 * This is GNU GO, a Go program. Contact gnugo@gnu.org, or see   *
 * http://www.gnu.org/software/gnugo/ for more information.      *
 *                                                               *
 * Copyright 1999 by the Free Software Foundation.               *
 *                                                               *
 * This program is free software; you can redistribute it and/or *
 * modify it under the terms of the GNU General Public License   *
 * as published by the Free Software Foundation - version 2.     *
 *                                                               *
 * This program is distributed in the hope that it will be       *
 * useful, but WITHOUT ANY WARRANTY; without even the implied    *
 * warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR       *
 * PURPOSE.  See the GNU General Public License in file COPYING  *
 * for more details.                                             *
 *                                                               *
 * You should have received a copy of the GNU General Public     *
 * License along with this program; if not, write to the Free    *
 * Software Foundation, Inc., 59 Temple Place - Suite 330,       *
 * Boston, MA 02111, USA                                         *
\* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */




/*
 * Module to evaluate moyos.
 *
 * It uses a dilation/erosion algorithm , currently 5 dilate/21 erode
 * to get the image of potential territory. This values seem to have these
 * properties :
 * - to have accurate evaluation, dead stones must be "removed" 
 * - 5 dilations to be sure to have influence on very large territory
 * - 21 erosions induce that "isolated" stones don't make territory, 
 *   they just limit the expansion of other color influence.
 * - in the end of the game, the evaluated territory match the actual score.
 *
 * see docs/MOYO for further details about this algorithm
 */

/* -m [level]
 * use bits for printing these debug reports :
 * 0x001 = ascii printing of territorial evaluation (5/21)
 * 0x002 = table of delta_terri values
 * 0x004 = ascii printing of moyo evaluation (5/10)
 * 0x008 = table of delta_moyo values
 * 0x010 = ascii printing of area (weak groups?) (4/0)
 * 0x020 = list of area characteristics
 * 0x040 = table of meta_connect values
 *
 * (example: -m 0x9)
 */


#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <ctype.h>
#include <string.h>		/* for memset */


/* the actual values of 5/21, 5/10 are defined as MAX_DILAT,... in liberty.h */
#include "liberty.h"
#include "interface.h"
#include "ttsgf.h"
#include "sgfana.h"

/* enabling color, the same as in showbord.c  */

#ifdef CURSES

#ifdef _AIX
#define _TPARM_COMPAT
#endif
#include <curses.h>
#include <term.h>
#endif

#ifdef CURSES
/* terminfo attributes */
static char *setaf;		/* terminfo string to set color */
static int max_color;		/* terminfo max colour */

static void init_curses(void)
{
  static int init = 0;

  /* compiler is set to make string literals  const char *
   * But system header files dont prototype things correctly.
   * These are equivalent to a non-const string literals
   */
  static char setaf_literal[]="setaf";
  static char colors_literal[]="colors";
  static char empty_literal[]="";
  if (init)
    return;
  
  init = 1;

  setupterm(NULL, 2, NULL);
  setaf = tigetstr(setaf_literal);
  if (!setaf)
    setaf = empty_literal;
  max_color = tigetnum(colors_literal) - 1;
  if (max_color < 1)
    max_color = 1;
  else if (max_color > 30)
    max_color = 30;
}


#define DRAW_COLOR_CHAR(c,x) do {  \
  fprintf(stderr, " %s%c", tparm(setaf,(c)), (x)); \
  fputs(tparm(setaf,max_color), stderr); \
} while(0)

#elif defined(ANSI_COLOR)

#define max_color 7

#define DRAW_COLOR_CHAR(c,x)  fprintf(stderr, " \033[%dm%c\033[0m", 30+(c), (x))

#else

#define DRAW_COLOR_CHAR(c,x)  fprintf(stderr, " %c", (x))

#endif


/* linux console :
 *  0=black
 *  1=red
 *  2=green
 *  3=yellow/brown
 *  4=blue
 *  5=magenta
 *  6=cyan
 *  7=white
 */


/* Shadow is the state of the goban after a dilation, there is a
 * Shadow for black and one for white. shadow are made of : 
 *   left :            image of stones one intersection to the left.
 *   right, up, down : idem to the other directions 
 *   tot :             is the image of the current color influence
 *                     after dilation in all directions. in the stack,
 *   mostack[WHITE-1][0].tot is the current position of white stones
 *                     really on the goban (before any dilation).  
 *   zone :            is the result of a crude dilation of one intersection 
 *                     in the four directions. Useful to get the limitation 
 *                     of influence when dilating the other color. 
 *                     It's different from "tot", "tot" doesn't dilate
 *                     against opposite color, "zone" does.
 */


/*
 * typedef struct {
 *      binmap_t left, right, up, down, tot, zone;
 * } Shadow;
 */

/* Structure for evaluating area ownership (aka looking for weak groups).
 *
 * Currently, this is used as a sorted stack, and area_grid[19][19] 
 * contains the relevant reference of the stack
 *
 * FIXME : - using a [200] stack could cause an overflow ? I think
 * that the max group number is 190 on a goban.
 * - using a [361] stack use memory without real need
 * - using malloc to construct a real size stack is slow 
 */
#define GROUP_STACK_MAX 190

/* size of the stack for delta_moyo_color, delta_area_color, 
 * and delta_terri_color (this is intentionally left low to
 * minimise memory usage. When the stack is full, the older
 * values are supressed.
 */

#define COLOR_STACK_SIZE 80

static goban_t boardstack[COLOR_STACK_SIZE];
static int boardstack_level;
static int where_board[COLOR_STACK_SIZE][2];

/* number of dilation for area ownership, must be <= MAX_DILAT */
#define OWNER_DILAT 4

/* number of bonus point for each group connected and opponent group cut */
#define GR_BONUS_CONNECT 15
#define GR_BONUS_CUT 10

typedef struct {
  int m, n;			/* origin point of the area */
  int color;			/* color of owner : WHITE or BLACK or EMPTY */
  int stone;			/* nb of stones of estimated group */
  int space;			/* nb of EMPTY around group */
  int tag;			/* for future usage : how the group is tagged ("weak", ...) */
  int d1m, d1n, d2m, d2n;       /* some dragons of the area */
  int d3m, d3n, d4m, d4n, d5m, d5n; 
} area_t;


static area_t area_stack[GROUP_STACK_MAX];
static int areas_level;
static goban_t area_grid;
static int nb_group_col[2];	/* number of groups of each colors */
static int nb_weak[2];		/* number of weak groups of each color */

static area_t test_area_s[GROUP_STACK_MAX];
static int te_areas_level;
static goban_t te_area_grid;

/* for caching already calculated values of delta_area, in meta_connect() */
static goban_t area_cached_move[2];
static goban_t delta_area_cache[2];
static goban_t board_area_move[2];	/* cache board for delta_area_color */


static unsigned long mask, bord;	/* mask=011..10  bord=100..01 */
static Shadow empty_shadow;	/* to quickly initialize Shadow values */
static binmap_t mask_binmap;	/* full of stone, for some binary operators */

/* the stack of successive iterations of the dilation, it is useful just
 * before erosion for computing territorial weight of each intersection 
 */
static Shadow mostack[2][MAX_DILAT + 1]; 

/* for caching already calculated values of delta_moyo */
static goban_t moyo_cached_move[2];
static goban_t board_moyo_move[2];	/* cache board for delta_moyo_color */
static goban_t delta_moyo_cache[2];

/* for caching already calculated values of delta_terri */
static goban_t terri_cached_move[2];
static goban_t board_terri_move[2];	/* cache board for delta_terri_color */
static goban_t delta_terri_cache[2];

/* static int ikomi;               komi */

static goban_t empty_goban;
static goban_t moyo_goban;
static goban_t terri_goban;
static goban_t d_moyo_goban;
static goban_t d_terri_goban;;

int terri_eval[3];
int terri_test[3];
int moyo_eval[3];
int moyo_test[3];
int very_big_move[3];


/**********************************/
/* functions declared in liberty.h :
 * void init_moyo(void);
 * int make_moyo(int color);
 * int delta_moyo(int ti, int tj, int color);
 * int delta_moyo_simple(int ti, int tj, int color);
 * int delta_terri(int ti, int tj, int color);
 * int diff_terri(int ti, int tj, int color);
 * int diff_moyo(int ti, int tj, int color);
 * void print_moyo(int color);
 * int area_stone(int m,int n);
 * int area_space(int m,int n);
 * int area_color(int m,int n);
 * int area_tag(int m,int n);
 * void set_area_tag(int m,int n,int tag);
 * int meta_connect(int ti, int tj,int color);
 * int terri_color(int m,int n);
 * int moyo_color(int m,int n);
 * int delta_terri_color(int ti,int tj, int color, int m, int n);
 * int delta_moyo_color(int ti,int tj, int color, int m, int n);
 * int delta_area_color(int ti,int tj, int color, int m, int n);
 * void search_big_move(int ti, int tj,int color,int val);
 * int number_weak(int color);
 */

/* static int influence(binmap_t * s); */
static void clear_moyo(int i);
static void sum_shadow(Shadow * s);
static void stack2goban(int num, goban_t * gob);
static void load_dragon_color_tot(Shadow * s, board_t what);
static void or_binmap(binmap_t * v1, binmap_t * v2);
static void inv_binmap(binmap_t * v1);
static void zone_dilat(int ns);
static void dilation(int ns);
static void erosion(goban_t * gob);
static void count_goban(goban_t * gob, int *score, int b);
static int compute_delta_moyo(int x, int y, int color, int b);
static int compute_delta_terri(int x, int y, int color);
static void compute_ownership(goban_t * gob, goban_t * grid,
			      area_t * area, int *alevel);
static void diffuse_area(goban_t * gob, goban_t * grid, 
			 area_t * area, int colval, int gridval, int i, int j);
static void clean_grid(int gridtest, goban_t * grid, area_t * area,
		       int gridval, int i, int j);
static void diffuse2_area(int *cpt, int gridtest, goban_t * grid,
			  int gridval, int i, int j);
static int compute_d_area(int x, int y, int color);
static void count_groups(area_t * area, int alevel, int *gcol, int color);
static void test_weak(goban_t * grid, area_t * area, int tw);
static int free_board(int color);

/***************************************/
static void print_ascii_moyo(goban_t * gob);
static void print_ascii_area(void);
static void print_txt_area(void);
static void print_delta_moyo(int color);
static void print_delta_terri(int color);
static void print_txt_connect(int color);
/* static void print_delta_terri_color(int ti,int tj,int color); */



/*
 * Initialize the moyo module.
 */

void
init_moyo(void)
{
  int i, j;
  binmap_t empty;

  for (i = 0; i < MAX_BOARD + 2; i++)
    empty[i] = 0;

  memcpy(empty_shadow.left, empty, sizeof(binmap_t));
  memcpy(empty_shadow.right, empty, sizeof(binmap_t));
  memcpy(empty_shadow.up, empty, sizeof(binmap_t));
  memcpy(empty_shadow.down, empty, sizeof(binmap_t));
  memcpy(empty_shadow.tot, empty, sizeof(binmap_t));
  memcpy(empty_shadow.zone, empty, sizeof(binmap_t));

  for (i = 0; i < board_size; i++)
    for (j = 0; j < board_size; j++)
      empty_goban[i][j] = 0;

  memset(area_stack, 0, sizeof(area_stack));

  /*
   *    mask = 011..10  bord = 100..01
   */
  mask = 0;
  for (i = 1; i <= board_size; i++)
    mask += (1 << i);
  bord = mask + 1 + 2;
  for (i = 1; i <= board_size; i++)
    mask_binmap[i] = mask;


  for (i = 0; i < MAX_BOARD ; i++)
    for (j = 0; j < MAX_BOARD ; j++) {
      board_moyo_move[0][i][j] = -1;
      board_moyo_move[1][i][j] = -1;
      board_terri_move[0][i][j] = -1;
      board_terri_move[1][i][j] = -1;
      board_area_move[0][i][j] = -1;
      board_area_move[1][i][j] = -1;
      area_cached_move[0][i][j] = 0;
      area_cached_move[1][i][j] = 0;
      delta_area_cache[0][i][j] = 0;
      delta_area_cache[1][i][j] = 0;
      moyo_cached_move[0][i][j] = 0;
      moyo_cached_move[1][i][j] = 0;
      delta_moyo_cache[0][i][j] = 0;	
      delta_moyo_cache[1][i][j] = 0;	
      terri_cached_move[0][i][j] = 0;
      terri_cached_move[1][i][j] = 0;
      delta_terri_cache[0][i][j] = 0;	
      delta_terri_cache[1][i][j] = 0;	
    }

  for (i = 0; i < COLOR_STACK_SIZE; i++)
    where_board[i][0] = where_board[i][1] = 0;

  boardstack_level = 0;
}



/*
 * Main moyo function. Return the white-black territorial balance
 * and put values in :
 *
 * 5/21 : after 21 erode + prisonners => territory approxim.
 * terri_eval[WHITE] : white territory
 * terri_eval[BLACK] : black territory
 * terri_eval[0] : difference in territory (color - other_color)
 *
 * these 3 are initialized to zero by make_moyo() :
 * terri_test[WHITE] : territory evaluation from delta_terri()
 * terri_test[BLACK] : ...
 * terri_test[0] :     Return of delta_terri(), difference in territorial
 *                     evaluation between terri_test and first call
 *                     to make_moyo the evaluation is for(color - other_color)
 *
 * 5/10 or 4/8 ...
 * after 10 erode, no prisoners, but with count of DEAD stones 
 * => for moyo approximation & delta moyo
 * moyo_eval[WHITE] : first moyo evaluation from make_moyo()
 * moyo_eval[BLACK] : ...
 * moyo_eval[0] : difference in moyo (color - other_color)
 *
 * These 3 ones are initialized to zero by make_moyo() :
 * moyo_test[WHITE] : moyo evaluation from delta_moyo()
 * moyo_test[BLACK] : ...
 * moyo_test[0] :    return of delta_moyo(), difference in moyo between 
 *                   test moyo and first moyo evaluation (color - other_color)
 *
 * 5/0 : used for group analysis, see area_grid & area_stack
 */

/*
 * make_moyo() returns terri_eval[0] and computes terri_eval & moyo_eval.
 *
 * It also computes the area_grid for area ownership & weak group search 
 */

int
make_moyo(int color)
{
  int i;

  /* FIXME: We don't have a good value for board_size in init_moyo() */
  mask = 0;
  for (i = 1; i <= board_size; i++)
    mask += (1 << i);
  bord = mask + 1 + 2;

  very_big_move[0]=0; 

  for (i = 0; i <= MAX_DILAT; i++)
    clear_moyo(i);

  /*
   * if (boardstack_level>69) gprintf("bs %d\n",boardstack_level);
   */
  boardstack_level = 0;

  terri_eval[WHITE] = terri_eval[BLACK] = 0;
  terri_test[WHITE] = terri_test[BLACK] = terri_test[0] = 0;
  moyo_eval[WHITE] = moyo_eval[BLACK] = 0;
  moyo_test[WHITE] = moyo_test[BLACK] = moyo_test[0] = 0;

  load_dragon_color_tot(&(mostack[WHITE - 1][0]), WHITE);
  load_dragon_color_tot(&(mostack[BLACK - 1][0]), BLACK);

  /* If MOY_DILAT != MAX_DILAT, we need to compute a little more (at least
   * until some modifications are made in dilat(), and change the function.
   * Since it's #define, gcc should optimize function size.
   */
  if (MOY_DILAT == MAX_DILAT) {
    if (OWNER_DILAT < MAX_DILAT) {
      dilation(OWNER_DILAT);
      memcpy(moyo_goban, empty_goban, sizeof(goban_t));
      stack2goban(OWNER_DILAT, &moyo_goban);

      /* Compute area_grid and the area_stack. */
      compute_ownership(&moyo_goban, &area_grid, area_stack, &areas_level);
      count_groups(area_stack, areas_level, nb_group_col, color);
      for (i = 1; i <= MAX_DILAT; i++)
	clear_moyo(i);
    }

    dilation(MAX_DILAT);
    memcpy(moyo_goban, empty_goban, sizeof(goban_t));
    stack2goban(MAX_DILAT, &moyo_goban);

    /* Compute area_grid and the area_stack. */
    if (OWNER_DILAT == MAX_DILAT) {
      compute_ownership(&moyo_goban, &area_grid, area_stack, &areas_level);
      count_groups(area_stack, areas_level, nb_group_col, color);
    }
    for (i = 1; i <= MOY_ERODE; i++)
      erosion(&moyo_goban);

    /* Compute moyo. */
    count_goban(&moyo_goban, moyo_eval, -1);

    /* Make last part of erode. */
    memcpy(terri_goban, moyo_goban, sizeof(goban_t));
    for (i = MOY_ERODE; i < MAX_ERODE; i++)
      erosion(&terri_goban);

    /* Compute territory. */
    count_goban(&terri_goban, terri_eval, -1);
  } else {
    /* MOY_DILAT != MAX_DILAT */

    if (OWNER_DILAT != MAX_DILAT && OWNER_DILAT != MOY_DILAT) {

      /* Compute area_grid and the area_stack */
      dilation(OWNER_DILAT);
      memcpy(moyo_goban, empty_goban, sizeof(goban_t));
      stack2goban(OWNER_DILAT, &moyo_goban);
      compute_ownership(&moyo_goban, &area_grid, area_stack, &areas_level);
      count_groups(area_stack, areas_level, nb_group_col, color);
      for (i = 1; i <= MAX_DILAT; i++)
	clear_moyo(i);
    }

    dilation(MAX_DILAT);
    memcpy(terri_goban, empty_goban, sizeof(goban_t));
    stack2goban(MAX_DILAT, &terri_goban);

    /* Compute area_grid and the area_stack */
    if (OWNER_DILAT == MAX_DILAT) {
      compute_ownership(&terri_goban, &area_grid, area_stack, &areas_level);
      count_groups(area_stack, areas_level, nb_group_col, color);
    }
    for (i = 1; i <= MAX_ERODE; i++)
      erosion(&terri_goban);

    /* Compute territory. */
    count_goban(&terri_goban, terri_eval, -1);


    /* Now redo it for moyo_eval. */
    for (i = 1; i <= MOY_DILAT; i++)
      clear_moyo(i);

    dilation(MOY_DILAT);
    memcpy(moyo_goban, empty_goban, sizeof(goban_t));
    stack2goban(MOY_DILAT, &moyo_goban);

    /* Compute area_grid and the area_stack. */
    if (OWNER_DILAT == MOY_DILAT) {
      compute_ownership(&moyo_goban, &area_grid, area_stack, &areas_level);
      count_groups(area_stack, areas_level, nb_group_col, color);
    }
    for (i = 1; i <= MOY_ERODE; i++)
      erosion(&moyo_goban);

    /* Compute moyo. */
    count_goban(&moyo_goban, moyo_eval, -1);
  }

  /* Look for weak groups. */
  nb_weak[0] = nb_weak[1] = 0;
  for (i = 1; i <= areas_level; i++) {
    test_weak(&area_grid, area_stack, i);
    if (area_stack[i].tag >= 3)
      nb_weak[area_stack[i].color - 1]++;
  }

  terri_eval[WHITE] += black_captured;	/* no komi added */
  terri_eval[BLACK] += white_captured;

  moyo_eval[0] = moyo_eval[color] - moyo_eval[OTHER_COLOR(color)];
  terri_eval[0] = terri_eval[color] - terri_eval[OTHER_COLOR(color)];

  return terri_eval[0];
}



int
delta_moyo(int ti, int tj, int color)
{
  return delta_moyo_simple(ti, tj, color) + meta_connect(ti, tj, color);
}



/* 
 * delta_moyo_simple is the old basic delta_moyo, without the meta
 * connect add.
 */

int
delta_moyo_simple(int ti, int tj, int color)
{
  if (moyo_cached_move[color - 1][ti][tj] != movenum) {
    delta_moyo_cache[color-1][ti][tj] = compute_delta_moyo(tj, ti, color, -1);
    moyo_cached_move[color-1][ti][tj] = movenum;
  }

  return delta_moyo_cache[color - 1][ti][tj];
}


/*
 * Look for territorial oriented moves, try to evaluate the value of
 * a move at (ti,tj) by this rule :
 *((terri. gain if color plays move
 *  + terri. gain if other_color plays move) * a + b + val*c)
 *
 * The a,b,c values are empiric, and can be modified for tuning, see below
 * for actual values.
 */

void
search_big_move(int ti, int tj, int color, int val)
{
  int dt;

  if (terri_color(ti, tj) == EMPTY ) {
    if (terri_eval[0] > -20)
      dt = diff_terri(ti, tj, color);
    else
      dt = diff_terri(ti, tj, color) / 2 + delta_moyo(ti, tj, color) / 2 ;

    if (dt >= 4) {
      dt = dt * 0.9 + 15 + val * 0.7;
      if (dt > very_big_move[0]) {
	very_big_move[0] = dt;
	very_big_move[1] = ti;
	very_big_move[2] = tj;
      }
    }
  }
}



int
delta_terri(int ti, int tj, int color)
{
  /* delta_terri automatically computes the color board if necessary. */
  if (terri_cached_move[color-1][ti][tj] != movenum 
      || board_terri_move[color-1][ti][tj] == -1) 
  {
    delta_terri_cache[color-1][ti][tj] = compute_delta_terri(tj, ti, color);
    terri_cached_move[color-1][ti][tj] = movenum;
  }

  return delta_terri_cache[color-1][ti][tj];
}


/*
 * wrapper for delta_terri(..,color) + delta_terri(..,other_color)
 */

int
diff_terri(int ti, int tj, int color)
{
  return delta_terri(ti, tj, color) + delta_terri(ti, tj, OTHER_COLOR(color));
}


/*
 * wrapper for delta_moyo(..,color) + delta_moyo(..,other_color) 
 */

int
diff_moyo(int ti, int tj, int color)
{
  return delta_moyo(ti, tj, color) + delta_moyo(ti, tj, OTHER_COLOR(color));
}


int
meta_connect(int ti, int tj, int color)
{
  if (area_cached_move[color - 1][ti][tj] != movenum) {

    /* We need the delta_moyo value for computing cut_bonus in
     * compute_d_area().
     */
    if (moyo_cached_move[color - 1][ti][tj] != movenum) {
      delta_moyo_cache[color - 1][ti][tj] 
	= compute_delta_moyo(tj, ti, color, -1);
      moyo_cached_move[color - 1][ti][tj] = movenum;
    }
    delta_area_cache[color - 1][ti][tj] = compute_d_area(tj, ti, color);
    area_cached_move[color - 1][ti][tj] = movenum;
  }

  return delta_area_cache[color - 1][ti][tj];
}


/*
 * delta_terri computes the board in the stack if not already cached
 */

int
delta_terri_color(int ti, int tj, int color, int m, int n)
{

  delta_terri(ti, tj, color);

  return boardstack[board_terri_move[color-1][ti][tj]][m][n];
}



int
delta_moyo_color(int ti, int tj, int color, int m, int n)
{
  int b = board_moyo_move[color - 1][ti][tj];

  if (moyo_cached_move[color - 1][ti][tj] != movenum || b == -1) {

    /* Find a free board in the stack & compute again delta moyo. */
    b = free_board(color);
    board_moyo_move[color - 1][ti][tj] = b;
    where_board[b][0] = ti;
    where_board[b][1] = tj;

    delta_moyo_cache[color - 1][ti][tj] = compute_delta_moyo(tj, ti, color, b);
    moyo_cached_move[color - 1][ti][tj] = movenum;
  }

  return boardstack[b][m][n];
}


int
delta_area_color(int ti, int tj, int color, int m, int n)
{
  int b;
  int dummy3[3];

  b = board_area_move[color - 1][ti][tj];
  if (area_cached_move[color - 1][ti][tj] != movenum || b == -1) {

    /* find a free board in the stack & compute again delta moyo */
    b = free_board(color);
    board_area_move[color - 1][ti][tj] = b;
    where_board[b][0] = ti;
    where_board[b][1] = tj;
    area_cached_move[color - 1][ti][tj] = -1;
    meta_connect(ti, tj, color);
    count_goban(&d_moyo_goban, dummy3, b);

  }

  return boardstack[b][m][n];
}


static int
free_board(int color)
{
  int b;

  b = boardstack_level;
  if (board_terri_move[color-1][where_board[b][0]][where_board[b][1]] == b)
    board_terri_move[color-1][where_board[b][0]][where_board[b][1]] = -1;

  if (board_moyo_move[color - 1][where_board[b][0]][where_board[b][1]] == b)
    board_moyo_move[color - 1][where_board[b][0]][where_board[b][1]] = -1;

  if (board_area_move[color - 1][where_board[b][0]][where_board[b][1]] == b)
    board_area_move[color - 1][where_board[b][0]][where_board[b][1]] = -1;

  boardstack_level++;

  /* cache overflow : we reuse the beginning of the stack */
  if (boardstack_level == COLOR_STACK_SIZE)
    boardstack_level = 0;

  return b;
}


static int
compute_delta_moyo(int x, int y, int color, int b)
{
  int i;

  moyo_test[WHITE] = 0;
  moyo_test[BLACK] = 0;

  /* Beware : we don't verify anything about value of x and y. 
   * Let's hope the pattern matcher only sends us correct values
   * (eg on the goban) 
   */
  ASSERT((x >= 0 && x < board_size && y >= 0 && y < board_size), y, x);

  for (i = 1; i <= MOY_DILAT; i++)
    clear_moyo(i);

  /* Add the stone to the binmap. */
  mostack[color - 1][0].tot[y + 1] |= (2 << x);

  /* Dilate and erode. */
  dilation(MOY_DILAT);
  memcpy(d_moyo_goban, empty_goban, sizeof(goban_t));
  stack2goban(MOY_DILAT, &d_moyo_goban);
  for (i = 1; i <= MOY_ERODE; i++)
    erosion(&d_moyo_goban);

  /* Compute delta moyo. */
  count_goban(&d_moyo_goban, moyo_test, b);

  /* Remember the added stone is NOT in p[j][i] */
  if (p[y][x] == EMPTY) {
    if (d_moyo_goban[x][y] < 0)
      moyo_test[WHITE]--;
    else if (d_moyo_goban[x][y] > 0)
      moyo_test[BLACK]--;
  }

  /* Remove the added stone. */
  mostack[color - 1][0].tot[y + 1] &= (~(2 << x) & mask);

  moyo_test[0] = moyo_test[color] - moyo_test[OTHER_COLOR(color)] -
      (moyo_eval[color] - moyo_eval[OTHER_COLOR(color)]);

  return moyo_test[0];
}


static int
compute_delta_terri(int x, int y, int color)
{
  /* This function uses MAX_DILAT dilation and MAX_ERODE erode. */
  int i, b;

  terri_test[WHITE] = terri_test[BLACK] = 0;

  /* Find a free board in the stack. */
  b = free_board(color);
  board_terri_move[color-1][y][x] = b;
  where_board[b][0] = y;
  where_board[b][1] = x;

  /* Beware : We don't verify anything about value of x and y.
   * Let's hope the pattern matcher only sends us correct values
   * (eg on the goban).
   */
  ASSERT((x >= 0 && x < board_size && y >= 0 && y < board_size), y, x);

  for (i = 1; i <= MAX_DILAT; i++)
    clear_moyo(i);

  /* Add the stone to the binmap. */
  mostack[color - 1][0].tot[y + 1] |= (2 << x);

  /* Dilate and erode. */
  dilation(MAX_DILAT);
  memcpy(d_terri_goban, empty_goban, sizeof(goban_t));
  stack2goban(MAX_DILAT, &d_terri_goban);
  for (i = 1; i <= MAX_ERODE; i++)
    erosion(&d_terri_goban);

  /* Compute delta terri. */
  count_goban(&d_terri_goban, terri_test, b);

  /* Remember the added stone is NOT in p[j][i]. */
  if (p[y][x] == EMPTY) {
    if (d_terri_goban[x][y] < 0)
      terri_test[WHITE]--;
    else if (d_terri_goban[x][y] > 0)
      terri_test[BLACK]--;
  }

  /* Remove the added stone. */
  mostack[color - 1][0].tot[y + 1] &= (~(2 << x) & mask);

  terri_test[WHITE] += black_captured;	/* no komi added */
  terri_test[BLACK] += white_captured;

  terri_test[0] = terri_test[color] - terri_test[OTHER_COLOR(color)] -
      (terri_eval[color] - terri_eval[OTHER_COLOR(color)]);

  return terri_test[0];
}


static int
compute_d_area(int x, int y, int color)
{
 /* Beware: delta_moyo() need to be computed before to have a correct
  * cut_bonus.   See meta_connect() for this.
  */

  int i, gcol[2], cut_bonus;

  for (i = 1; i <= OWNER_DILAT; i++)
    clear_moyo(i);

  /* Add the stone to the binmap. */
  mostack[color - 1][0].tot[y + 1] |= (2 << x);

  dilation(OWNER_DILAT);
  memcpy(d_moyo_goban, empty_goban, sizeof(goban_t));
  stack2goban(OWNER_DILAT, &d_moyo_goban);

  compute_ownership(&d_moyo_goban, &te_area_grid, 
		    test_area_s, &te_areas_level);
  count_groups(test_area_s, te_areas_level, gcol, color);

  /* Remove the added stone. */
  mostack[color - 1][0].tot[y + 1] &= (~(2 << x) & mask);

  /* Compute the cut_bonus of the move :
   * If the tested move creates a new isolated group with :
   *   number of stone <= 2 and  number of space around <= 10
   * then 
   *   return ( - delta_moyo_simple) to suppress the delta_moyo bonus, 
   * else
   *   if we don't create an isolated stone, 
   *   then
   *     return the difference of number of groups * bonus value.     
   */
  if (test_area_s[te_area_grid[x][y]].stone <= 2 
      && test_area_s[te_area_grid[x][y]].space <= 10) 
    {
      cut_bonus = -delta_moyo_cache[color - 1][y][x];
    }
  else if (test_area_s[te_area_grid[x][y]].stone > 1) {
    cut_bonus = 0;
#if 0
    if (terri_eval[0] > -15) {
#endif
      cut_bonus += GR_BONUS_CONNECT * (nb_group_col[color - 1] 
				       - gcol[color - 1]);
      cut_bonus += GR_BONUS_CUT * (gcol[OTHER_COLOR(color) - 1]
				   - nb_group_col[OTHER_COLOR(color) - 1]);
#if 0
    } else if (terri_eval[0] > -30) {
      cut_bonus += GR_BONUS_CONNECT * (nb_group_col[color - 1]
				       - gcol[color - 1]);
      cut_bonus += 1.5 * GR_BONUS_CUT * (gcol[OTHER_COLOR(color) - 1] 
					 -nb_group_col[OTHER_COLOR(color)-1]);
    } else {
      cut_bonus += 1.5 * GR_BONUS_CONNECT * (nb_group_col[color - 1] 
					     - gcol[color - 1]);
      cut_bonus += 2 * GR_BONUS_CUT * (gcol[OTHER_COLOR(color) - 1]
				       - nb_group_col[OTHER_COLOR(color) - 1]);
    }
#endif
  } else
    cut_bonus = 0;

  return cut_bonus;
}


int
terri_color(int m, int n)
{
  if (m >= 0 && m < board_size && n >= 0 && n < board_size) {
    if (terri_goban[n][m] > 0)
      return BLACK;
    else if (terri_goban[n][m] < 0)
      return WHITE;
    else
      return EMPTY;
  } else
    return -1;
}


int
moyo_color(int m, int n)
{
  if (m >= 0 && m < board_size && n >= 0 && n < board_size) {
    if (moyo_goban[n][m] > 0)
      return BLACK;
    else if (moyo_goban[n][m] < 0)
      return WHITE;
    else
      return EMPTY;
  } else
    return -1;
}


int
area_stone(int m, int n)
{
  ASSERT((m >= 0 && m < board_size && n >= 0 && n < board_size), m, n);

  return area_stack[area_grid[n][m]].stone;
}


int
area_space(int m, int n)
{
  ASSERT((m >= 0 && m < board_size && n >= 0 && n < board_size), m, n);

  return area_stack[area_grid[n][m]].space;
}


int
area_color(int m, int n)
{
  ASSERT((m >= 0 && m < board_size && n >= 0 && n < board_size), m, n);

  return area_stack[area_grid[n][m]].color;
}


int
area_tag(int m, int n)
{
  ASSERT((m >= 0 && m < board_size && n >= 0 && n < board_size), m, n);

  return area_stack[area_grid[n][m]].tag;
}


void
set_area_tag(int m, int n, int tag)
{
  ASSERT((m >= 0 && m < board_size && n >= 0 && n < board_size), m, n);
  area_stack[area_grid[n][m]].tag = tag;
}


int
number_weak(int color)
{
  if (color >= 1 && color <= 2)
    return nb_weak[color - 1];
  else
    return 0;
}


static void
count_groups(area_t * area, int alevel, int *gcol, int color)
{
  int i;

  gcol[0] = gcol[1] = 0;

  for (i = 1; i <= alevel; i++) {
    if (area[i].color == color)
      gcol[color - 1]++;
    else if (area[i].color == OTHER_COLOR(color))
      gcol[OTHER_COLOR(color) - 1]++;
  }
}


static void
compute_ownership(goban_t * gob, goban_t * grid, area_t * area, int *alevel)
{
  int i, j;

  /* Some level of the area may be useless (color=-1). */
  *alevel = 0;
  memcpy(*grid, empty_goban, sizeof(goban_t));

  for (i = 0; i < board_size; i++)
    for (j = 0; j < board_size; j++) {
      if ((*grid)[i][j] == 0) {
	(*alevel)++;

	/* The empty space without clear ownership belong to EMPTY. */
	if ((*gob)[i][j] < 0)
	  area[(*alevel)].color = WHITE;
	else if ((*gob)[i][j] > 0)
	  area[(*alevel)].color = BLACK;
	else {
	  /* Look for first stage kosumi connection, not seen without this. */
	  if (j > 0 && (*gob)[i][j - 1] != 0) {
	    if (((mostack[WHITE - 1][0].tot[j] & (2 << i))
		 && ((i - 1) < board_size)
		 && (mostack[WHITE - 1][0].tot[j + 1] & (4 << i))
		 && ((*gob)[i + 1][j - 1] == 0)
		 && (*grid)[i + 1][j] != (*grid)[i][j - 1])
		|| ((mostack[BLACK - 1][0].tot[j] & (2 << i))
		    && ((i - 1) < board_size)
		    && (mostack[BLACK - 1][0].tot[j + 1] & (4 << i))
		    && ((*gob)[i + 1][j - 1] == 0)
		    && (*grid)[i + 1][j] != (*grid)[i][j - 1]))
	    {
	      /* level to free */
	      if ((*grid)[i + 1][j] != 0) {
		clean_grid((*grid)[i + 1][j], grid, area,
			   (*grid)[i][j - 1], i + 1, j);
	      } else {
		diffuse_area(gob, grid, area, 
			     area[(*grid)[i][j - 1]].color,
			     (*grid)[i][j - 1], i + 1, j);
	      }
	    }

	    if (((mostack[WHITE - 1][0].tot[j] & (2 << i))
		 && (i > 0)
		 && (mostack[WHITE - 1][0].tot[j + 1] & (1 << i))
		 && ((*gob)[i - 1][j - 1] == 0)
		 && (*grid)[i - 1][j] != (*grid)[i][j - 1])
		|| ((mostack[BLACK - 1][0].tot[j] & (2 << i))
		    && (i > 0)
		    && (mostack[BLACK - 1][0].tot[j + 1] & (1 << i))
		    && ((*gob)[i - 1][j - 1] == 0)
		    && (*grid)[i - 1][j] != (*grid)[i][j - 1]))
	    {
	      /* level to free */
	      clean_grid((*grid)[i - 1][j], grid, area,
			 (*grid)[i][j - 1], i - 1, j);
	    }
	  }
	  area[(*alevel)].color = EMPTY;
	}

	area[(*alevel)].stone = 0;
	area[(*alevel)].space = 0;
	area[(*alevel)].tag = 0;
	area[(*alevel)].m = j;
	area[(*alevel)].n = i;
	area[(*alevel)].d1m = area[(*alevel)].d1n = -1;
	area[(*alevel)].d2m = area[(*alevel)].d2n = -1;
	area[(*alevel)].d3m = area[(*alevel)].d3n = -1;
	area[(*alevel)].d4m = area[(*alevel)].d4n = -1;
	area[(*alevel)].d5m = area[(*alevel)].d5n = -1;

	diffuse_area(gob, grid, area, area[(*alevel)].color, (*alevel), i, j);
      }
    }

  /* compute bonus */
  /*
   *   for (i = 1; i <= (*alevel); i++) {
   *     if (area[i].color == EMPTY) {
   *       if (area[i].space < 20)
   *      area[i].tag = 10;
   *     } else {
   *       if (area[i].color >= 0 && area[i].stone < 10)
   *      area[i].tag += area[i].stone;
   *       if (area[i].color >= 0 && area[i].stone < 15
   *           && (2 * area[i].stone > area[i].space))
   *      area[i].tag += 10;
   *     }
   *   }
   */
}


/*
 *
 * A string is defined to be WEAK if it has size 2 or more and its
 * space is between 0 and 20 points.
 */

static void
test_weak(goban_t * grid, area_t * area, int tw)
{
  int i, j, m, n;
  int gen = 0;
  int nbdrag = 0;
  char cs;

  if (area[tw].color >= 1 && area[tw].space < 21) {
    area[tw].tag = 2;
    if (printmoyo & 16)
      gprintf("weak area %d  %m: color %c,  %d stone  %d spaces\n",
	      tw, area[tw].m, area_stack[tw].n,
	      area[tw].color == WHITE ? 'W' : 'B',
	      area[tw].stone, area[tw].space);

    /* Find area. */
    for (i = 0; i < board_size; i++)
      for (j = 0; j < board_size; j++) {
	if ((*grid)[i][j] == tw && p[j][i] && dragon[j][i].weak != DEAD) {
	  if (nbdrag < 6
	      && area[tw].d1m != dragon[j][i].origini
	      && area[tw].d1n != dragon[j][i].originj
	      && area[tw].d2m != dragon[j][i].origini
	      && area[tw].d2n != dragon[j][i].originj
	      && area[tw].d3m != dragon[j][i].origini
	      && area[tw].d3n != dragon[j][i].originj
	      && area[tw].d4m != dragon[j][i].origini
	      && area[tw].d4n != dragon[j][i].originj
	      && area[tw].d5m != dragon[j][i].origini
	      && area[tw].d5n != dragon[j][i].originj) 
	  {
	    nbdrag++;
	    gen += dragon[j][i].genus;

	    if (printmoyo & 16) {
	      switch (dragon[j][i].weak) {
	      case ALIVE:
		cs = 'A';
		break;
	      case DEAD:
		cs = 'D';
		break;
	      case UNKNOWN:
		cs = 'U';
		break;
	      case CRITICAL:
		cs = 'C';
		break;
	      default:
		cs = '?';
	      }
	      gprintf("drag %m  gen %d  weakness: %c ",
		      j, i, dragon[j][i].genus, cs);
	    }

	    if (dragon[j][i].weak == UNKNOWN) {
	      dragon[dragon[j][i].origini]
		    [dragon[j][i].originj].weak = CRITICAL;
	      for (m = 0; m < board_size; m++)
		for (n = 0; n < board_size; n++) {
		  struct dragon_data *d = &(dragon[m][n]);

		  dragon[m][n] = dragon[d->origini][d->originj];
		}

              area[tw].tag ++;
	      if (printmoyo & 16)
		gprintf("=> critical\n");
	    } else {
	      if (printmoyo & 16)
		gprintf("\n");
	    }

	    if (area[tw].d1m == -1) {
	      area[tw].d1m = dragon[j][i].origini;
	      area[tw].d1n = dragon[j][i].originj;
	    } else if (area[tw].d2m == -1) {
	      area[tw].d2m = dragon[j][i].origini;
	      area[tw].d2n = dragon[j][i].originj;
	    } else if (area[tw].d3m == -1) {
	      area[tw].d3m = dragon[j][i].origini;
	      area[tw].d3n = dragon[j][i].originj;
	    } else if (area[tw].d4m == -1) {
	      area[tw].d4m = dragon[j][i].origini;
	      area[tw].d4n = dragon[j][i].originj;
	    } else if (area[tw].d5m == -1) {
	      area[tw].d5m = dragon[j][i].origini;
	      area[tw].d5n = dragon[j][i].originj;
	    }
	  }
	}
      }
  }
}


static void
diffuse_area(goban_t * gob, goban_t * grid, area_t * area, 
	     int colval, int gridval, int i, int j)
{
  (*grid)[i][j] = gridval;

  /* If a neighbour is of the same color, apply recursively. */
  if (colval == WHITE) {
    if (mostack[WHITE - 1][0].tot[j + 1] & (2 << i))
      area[gridval].stone++;
    else
      area[gridval].space++;

    if (i > 0 && (*grid)[i - 1][j] == 0 && (*gob)[i - 1][j] < 0)
      diffuse_area(gob, grid, area, colval, gridval, i - 1, j);

    if (j > 0 && (*grid)[i][j - 1] == 0 && (*gob)[i][j - 1] < 0)
      diffuse_area(gob, grid, area, colval, gridval, i, j - 1);

    if ((i + 1) < board_size && (*grid)[i + 1][j] == 0 && (*gob)[i + 1][j] < 0)
      diffuse_area(gob, grid, area, colval, gridval, i + 1, j);

    if ((j + 1) < board_size && (*grid)[i][j + 1] == 0 && (*gob)[i][j + 1] < 0)
      diffuse_area(gob, grid, area, colval, gridval, i, j + 1);
  } else if (colval == BLACK) {
    if (mostack[BLACK - 1][0].tot[j + 1] & (2 << i))
      area[gridval].stone++;
    else
      area[gridval].space++;

    if (i > 0 && (*grid)[i - 1][j] == 0 && (*gob)[i - 1][j] > 0)
      diffuse_area(gob, grid, area, colval, gridval, i - 1, j);

    if (j > 0 && (*grid)[i][j - 1] == 0 && (*gob)[i][j - 1] > 0)
      diffuse_area(gob, grid, area, colval, gridval, i, j - 1);

    if ((i + 1) < board_size && (*grid)[i + 1][j] == 0 && (*gob)[i + 1][j] > 0)
      diffuse_area(gob, grid, area, colval, gridval, i + 1, j);

    if ((j + 1) < board_size && (*grid)[i][j + 1] == 0 && (*gob)[i][j + 1] > 0)
      diffuse_area(gob, grid, area, colval, gridval, i, j + 1);

  } else if (colval == EMPTY) {
    area[gridval].space++;

    if (i > 0
	&& (*grid)[i - 1][j] == 0
	&& (*gob)[i - 1][j] == 0)
      diffuse_area(gob, grid, area, colval, gridval, i - 1, j);

    if (j > 0
	&& (*grid)[i][j - 1] == 0
	&& (*gob)[i][j - 1] == 0)
      diffuse_area(gob, grid, area, colval, gridval, i, j - 1);

    if ((i + 1) < board_size
	&& (*grid)[i + 1][j] == 0
	&& (*gob)[i + 1][j] == 0)
      diffuse_area(gob, grid, area, colval, gridval, i + 1, j);

    if ((j + 1) < board_size
	&& (*grid)[i][j + 1] == 0
	&& (*gob)[i][j + 1] == 0)
      diffuse_area(gob, grid, area, colval, gridval, i, j + 1);
  }
}


/* 
 * clean_grid adds the area "gridtest" at (i,j) to the area "gridval"
 * using the recursive function diffuse2_area.
 */

static void
clean_grid(int gridtest, goban_t * grid, area_t * area, 
	   int gridval, int i, int j)
{
  int cpt, ii, jj;

  area[gridval].stone += area[gridtest].stone;
  area[gridval].space += area[gridtest].space;
  area[gridtest].color = -1;

  /* Nb of vertices to find. */
  cpt = area[gridtest].stone + area[gridtest].space;

  diffuse2_area(&cpt, gridtest, grid, gridval, i, j);

  if (cpt > 0) {
    /* Bad luck. There are second stage kosumi connection, 
     * and we need to apply a heavy method.
     */
    for (ii = 0; ii < board_size; ii++)
      for (jj = 0; jj < board_size; jj++) {
	if ((*grid)[ii][jj] == gridtest) {
	  (*grid)[ii][jj] = gridval;
	  cpt--;
	  if (cpt == 0)
	    return;
	}
      }

    ASSERT(cpt == 0, j, i);
  }

  return;
}


static void
diffuse2_area(int *cpt, int gridtest, goban_t * grid, 
	      int gridval, int i, int j)
{
  (*grid)[i][j] = gridval;
  (*cpt)--;

  /* If a neighbour is of the same gridtest, apply recursively. */
  if (i > 0 && (*grid)[i - 1][j] == gridtest)
    diffuse2_area(cpt, gridtest, grid, gridval, i - 1, j);

  if (j > 0 && (*grid)[i][j - 1] == gridtest)
    diffuse2_area(cpt, gridtest, grid, gridval, i, j - 1);

  if ((i + 1) < board_size && (*grid)[i + 1][j] == gridtest)
    diffuse2_area(cpt, gridtest, grid, gridval, i + 1, j);

  if ((j + 1) < board_size && (*grid)[i][j + 1] == gridtest)
    diffuse2_area(cpt, gridtest, grid, gridval, i, j + 1);
}


/* ================================================================ */
/*                    low level functions                           */
/* ================================================================ */


static void
count_goban(goban_t * gob, int *score, int b)
{
  int i, j;

  if (b == -1) {

    for (i = 0; i < board_size; i++)
      for (j = 0; j < board_size; j++) {
	if (p[j][i] == EMPTY) {
	  if ((*gob)[i][j] < 0)
	    score[WHITE]++;
	  else if ((*gob)[i][j] > 0)
	    score[BLACK]++;
	} else {
	  if (dragon[j][i].weak == DEAD) {
	    if ((*gob)[i][j] < 0)
	      score[WHITE] += 2;
	    else if ((*gob)[i][j] > 0)
	      score[BLACK] += 2;
	  }
	}
      }
  } else {
    for (i = 0; i < board_size; i++)
      for (j = 0; j < board_size; j++) {
	if (p[j][i] == EMPTY) {
	  if ((*gob)[i][j] < 0) {
	    score[WHITE]++;
	    boardstack[b][j][i] = WHITE;
	  } else if ((*gob)[i][j] > 0) {
	    score[BLACK]++;
	    boardstack[b][j][i] = BLACK;
	  } else {
	    boardstack[b][j][i] = EMPTY;
	  }
	} else {
	  if (dragon[j][i].weak == DEAD) {
	    if ((*gob)[i][j] < 0) {
	      score[WHITE] += 2;
	      boardstack[b][j][i] = WHITE;
	    } else if ((*gob)[i][j] > 0) {
	      score[BLACK] += 2;
	      boardstack[b][j][i] = BLACK;
	    } else {
	      boardstack[b][j][i] = EMPTY;
	    }
	  } else {
	    boardstack[b][j][i] = p[j][i];
	  }
	}
      }
  }
}


/*
 * sum of the bits of a binmap, aka influence
 */

#if 0
static int
influence(binmap_t * s)
{
  int i, j, k;
 
  k = 0;
  for (i = 1; i <= board_size; i++)
    for (j = 0; j < board_size; j++)
      if ((*s)[i] & (1 << j))
	k++;
  return k;
}
#endif


/*
 * The stack need to be cleared at every cycle.
 */

static void
clear_moyo(int i)
{
  mostack[WHITE - 1][i] = empty_shadow;
  mostack[BLACK - 1][i] = empty_shadow;
}


/* 
 * Compute the "tot" of a shadow by adding the 4 directions
 * influences.
 */

static void
sum_shadow(Shadow * s)
{
  memcpy(s->tot, s->left, sizeof(binmap_t));
  or_binmap(&(s->tot), &(s->right));
  or_binmap(&(s->tot), &(s->up));
  or_binmap(&(s->tot), &(s->down));
}

/*
 * Sum of the levels of dilations from the moyo stack in a numeric
 * goban for later use of this goban in erode phase.
 * add from 1 to "int num" level of the stack.
 */

static void
stack2goban(int num, goban_t * gob)
{
  int i, j, k;

  for (i = 1; i <= board_size; i++)
    for (j = 1; j <= board_size; j++) {
      if ((mostack[WHITE - 1][0].tot[j]) & (1 << i))
	(*gob)[i - 1][j - 1] -= 128;
      if ((mostack[BLACK - 1][0].tot[j]) & (1 << i))
	(*gob)[i - 1][j - 1] += 128;
      for (k = 1; k <= num; k++) {
	(*gob)[i - 1][j - 1] -= (mostack[WHITE - 1][k].left[j] >> i) & 1;
	(*gob)[i - 1][j - 1] -= (mostack[WHITE - 1][k].right[j] >> i) & 1;
	(*gob)[i - 1][j - 1] -= (mostack[WHITE - 1][k].up[j] >> i) & 1;
	(*gob)[i - 1][j - 1] -= (mostack[WHITE - 1][k].down[j] >> i) & 1;
	(*gob)[i - 1][j - 1] += (mostack[BLACK - 1][k].left[j] >> i) & 1;
	(*gob)[i - 1][j - 1] += (mostack[BLACK - 1][k].right[j] >> i) & 1;
	(*gob)[i - 1][j - 1] += (mostack[BLACK - 1][k].up[j] >> i) & 1;
	(*gob)[i - 1][j - 1] += (mostack[BLACK - 1][k].down[j] >> i) & 1;
      }
    }
}


/* 
 * Load the current live stones of one color from the global dragon.
 */

static void
load_dragon_color_tot(Shadow * s, board_t what)
{
  int i, j;

  for (i = 0; i < board_size; i++)
    for (j = 0; j < board_size; j++) {
      if (p[j][i] == what && dragon[j][i].weak != DEAD)
	s->tot[j + 1] |= (2 << i);
    }
}


#if 0
static void
add_stone_tot(Shadow * s, int x, int y)
{
  x++;
  y++;
  s->tot[y] |= (1 << x);
}


static void
del_stone_tot(Shadow * s, int x, int y)
{
  x++;
  y++;
  s->tot[y] &= (~(1 << x) & mask);
}
#endif


/*
 * Make a bit to bit OR between 2 binmap_t
 */

static void
or_binmap(binmap_t * v1, binmap_t * v2)
{
  int i;

  /* No OR on 0 and 21 line */
  for (i = 1; i <= board_size; i++)
    (*v1)[i] |= (*v2)[i];
}

/* 
 * Invert the binary binmap. 
 * 0=>1 and 1=>0, except the border.
 */

static void
inv_binmap(binmap_t * v1)
{
  int i;

  for (i = 1; i <= board_size; i++)
    (*v1)[i] = ~(*v1)[i] & mask;
}


#if 0
static void
and_binmap(binmap_t * v1, binmap_t * v2)
{
  int i;

  for (i = 1; i <= board_size; i++)
    (*v1)[i] &= (*v2)[i];
}


static void
xor_binmap(binmap_t * v1, binmap_t * v2)
{
  int i;

  for (i = 1; i <= board_size; i++)
    (*v1)[i] ^= (*v2)[i];
}

/*
 * Subtract v1 = v1 - v2
 */

static void
sub_binmap(binmap_t * v1, binmap_t * v2)
{                            
  int i;
 
  for (i = 1; i <= board_size; i++)
    (*v1)[i] &= (~(*v2)[i]) & mask;
}
#endif


/*
 * Make the crude dilatation in the shadow.zone binmap for one
 * moyo of the stack.
 */

static void
zone_dilat(int ns)
{
  int i;

  memcpy(mostack[WHITE - 1][ns].zone, 
	 mostack[WHITE - 1][ns].tot, sizeof(binmap_t));

  for (i = 1; i <= board_size; i++) {
    mostack[WHITE - 1][ns].zone[i] 
      |= ((mostack[WHITE - 1][ns].tot[i] << 1) 
	  | (mostack[WHITE - 1][ns].tot[i] >> 1)) & mask;
    mostack[WHITE - 1][ns].zone[i] |= mostack[WHITE - 1][ns].tot[i - 1];
    mostack[WHITE - 1][ns].zone[i] |= mostack[WHITE - 1][ns].tot[i + 1];
  }

  memcpy(mostack[BLACK - 1][ns].zone,
	 mostack[BLACK - 1][ns].tot, sizeof(binmap_t));
  for (i = 1; i <= board_size; i++) {
    mostack[BLACK - 1][ns].zone[i] 
      |= ((mostack[BLACK - 1][ns].tot[i] << 1) 
	  | (mostack[BLACK - 1][ns].tot[i] >> 1)) & mask;
    mostack[BLACK - 1][ns].zone[i] |= mostack[BLACK - 1][ns].tot[i - 1];
    mostack[BLACK - 1][ns].zone[i] |= mostack[BLACK - 1][ns].tot[i + 1];
  }
}


/*
 * Compute one dilation. Make the four unitary dilations and the sum in
 * shadow.tot, for both colors of a moyo of the stack.
 */

static void
dilation(int ns)
{
  int i, j;
  binmap_t zone;

  for (i = 1; i <= ns; i++) {
    zone_dilat(i - 1);

    memcpy(zone, mostack[BLACK - 1][i - 1].zone, sizeof(binmap_t));
    inv_binmap(&zone);
    for (j = 1; j <= board_size; j++) {
      mostack[WHITE - 1][i].left[j] 
	= (mostack[WHITE - 1][i - 1].tot[j] << 1) & mask & zone[j];
      mostack[WHITE - 1][i].right[j] 
	= (mostack[WHITE - 1][i - 1].tot[j] >> 1) & mask & zone[j];
      mostack[WHITE - 1][i].up[j]
	= mostack[WHITE - 1][i - 1].tot[j + 1] & zone[j];
      mostack[WHITE - 1][i].down[j]
	= mostack[WHITE - 1][i - 1].tot[j - 1] & zone[j];
    }

    sum_shadow(&(mostack[WHITE - 1][i]));
    or_binmap(&(mostack[WHITE - 1][i].tot), &(mostack[WHITE - 1][i - 1].tot));

    memcpy(zone, mostack[WHITE - 1][i - 1].zone, sizeof(binmap_t));
    inv_binmap(&zone);
    for (j = 1; j <= board_size; j++) {
      mostack[BLACK - 1][i].left[j] 
	= (mostack[BLACK - 1][i - 1].tot[j] << 1) & mask & zone[j];
      mostack[BLACK - 1][i].right[j] 
	= (mostack[BLACK - 1][i - 1].tot[j] >> 1) & mask & zone[j];
      mostack[BLACK - 1][i].up[j] 
	= mostack[BLACK - 1][i - 1].tot[j + 1] & zone[j];
      mostack[BLACK - 1][i].down[j] 
	= mostack[BLACK - 1][i - 1].tot[j - 1] & zone[j];
    }

    sum_shadow(&(mostack[BLACK - 1][i]));
    or_binmap(&(mostack[BLACK - 1][i].tot), &(mostack[BLACK - 1][i - 1].tot));
  }
}



/*
 * The erode function : don't use binary operators.
 */

static void
erosion(goban_t * gob)
{
  int i, j, m, mb;
  goban_t outgob;

  /*** different color or null => erode 1 point ***/
  for (j = 0; j < board_size; j++) {
    for (i = 0; i < board_size; i++) {
      if ((mb = (*gob)[i][j]) != 0) {
	m = 0;

	if (mb > 0) {
	  i++;
	  if (i < board_size)
	    if ((*gob)[i][j] * mb <= 0)
	      m--;
	  i--;
	  j++;

	  if (j < board_size)
	    if ((*gob)[i][j] * mb <= 0)
	      m--;
	  i--;
	  j--;

	  if (i >= 0)
	    if ((*gob)[i][j] * mb <= 0)
	      m--;
	  i++;
	  j--;

	  if (j >= 0)
	    if ((*gob)[i][j] * mb <= 0)
	      m--;
	  j++;

	  if (-m >= mb)
	    mb = 0;
	  else
	    mb += m;
	} else {
	  i++;
	  if (i < board_size)
	    if ((*gob)[i][j] * mb <= 0)
	      m++;
	  i--;
	  j++;

	  if (j < board_size)
	    if ((*gob)[i][j] * mb <= 0)
	      m++;
	  i--;
	  j--;

	  if (i >= 0)
	    if ((*gob)[i][j] * mb <= 0)
	      m++;
	  i++;
	  j--;

	  if (j >= 0)
	    if ((*gob)[i][j] * mb <= 0)
	      m++;
	  j++;

	  if (m >= -mb)
	    mb = 0;
	  else
	    mb += m;
	}
      }
      outgob[i][j] = mb;
    }
  }

  memcpy(*gob, outgob, sizeof(goban_t));

  return;
}


/* ================================================================ */
/*                test & printing functions                         */
/* ================================================================ */


void
who_wins(int color, float fkomi, FILE * stdwhat)
{
  float white_score;
  float black_score;
  float result;
  int winner;

  if (color != BLACK && color != WHITE)
    color = BLACK;

  make_moyo(color);
  white_score = terri_eval[WHITE] + fkomi;
  black_score = terri_eval[BLACK];
  if (white_score > black_score) {
    winner = WHITE;
    result = white_score - black_score;
  } else {
    winner = BLACK;
    result = black_score - white_score;
  }

  fprintf(stdwhat, "Result: %c+%.1f   ",
	  (winner == WHITE) ? 'W' : 'B', result);
  if (color == winner)
    fprintf(stdwhat, "%c says \"I win!\"\n", (color == WHITE) ? 'W' : 'B');
  else
    fprintf(stdwhat, "%c says \"I lost!\"\n", (color == WHITE) ? 'W' : 'B');
}



void
print_moyo(int color)
{
  if (printmoyo & 1)
    print_ascii_moyo(&terri_goban);

  if (printmoyo & 2)
    print_delta_terri(color);

  if (printmoyo & 4)
    print_ascii_moyo(&moyo_goban);

  if (printmoyo & 8)
    print_delta_moyo(color);

  if (printmoyo & 16)
    print_ascii_area();

  if (printmoyo & 32)
    print_txt_area();

  if (printmoyo & 64)
    print_txt_connect(color);
}


static void
print_delta_moyo(int color)
{
  int i, j;

#ifdef CURSES
  init_curses();
#endif

  fprintf(stderr, "delta_moyo :\n");
  for (i = 0; i < board_size; i++) {
    for (j = 0; j < board_size; j++) {
      switch (p[i][j]) {
      case EMPTY:
	fprintf(stderr, "%3d", delta_moyo(i, j, color));
	break;
      case BLACK:
	fprintf(stderr, " ");
	DRAW_COLOR_CHAR(5, 'X');
	break;
      case WHITE:
	fprintf(stderr, " ");
	DRAW_COLOR_CHAR(2, 'O');
	break;
      }
    }

    fprintf(stderr, "\n");
  }

  fprintf(stderr, "\n");
}


static void
print_delta_terri(int color)
{
  int i, j;

#ifdef CURSES
  init_curses();
#endif

  fprintf(stderr, "delta_terri :\n");
  for (i = 0; i < board_size; i++) {
    for (j = 0; j < board_size; j++) {
      switch (p[i][j]) {
      case EMPTY:
	fprintf(stderr, "%3d", delta_terri(i, j, color));
	break;
      case BLACK:
	fprintf(stderr, " ");
	DRAW_COLOR_CHAR(5, 'X');
	break;
      case WHITE:
	fprintf(stderr, " ");
	DRAW_COLOR_CHAR(2, 'O');
	break;
      }
    }

    fprintf(stderr, "\n");
  }

  fprintf(stderr, "\n");
}


#if 0
static void
print_delta_terri_color(int ti,int tj,int color)
{
  int i, j;

  gprintf("delta_terri_color %m :\n",ti,tj);
  for (i = 0; i < board_size; i++) {
    for (j = 0; j < board_size; j++) {
      fprintf(stderr, "%3d", delta_moyo_color(ti, tj, color,i,j));
    }

    fprintf(stderr, "\n");
  }

  fprintf(stderr, "\n");
}
#endif


static void
print_txt_connect(int color)
{
  int i, j, k;

#ifdef CURSES
  init_curses();
#endif

  fprintf(stderr, "meta_connect :\n");
  for (i = 0; i < board_size; i++) {
    for (j = 0; j < board_size; j++) {
      switch (p[i][j]) {
      case EMPTY:
	k = meta_connect(i, j, color);
	if (k == 0)
	  fprintf(stderr, "  .");
	else
	  fprintf(stderr, "%3d", meta_connect(i, j, color));
	break;
      case BLACK:
	fprintf(stderr, " ");
	DRAW_COLOR_CHAR(5, 'X');
	break;
      case WHITE:
	fprintf(stderr, " ");
	DRAW_COLOR_CHAR(2, 'O');
	break;
      }
    }

    fprintf(stderr, "\n");
  }

  fprintf(stderr, "\n");
}


static void
print_ascii_moyo(goban_t * gob)
{
  int i, j, k;

#ifdef CURSES
  init_curses();
#endif

  if (board_size == 9)
    fprintf(stderr, "   A B C D E F G H J\n");
  else if (board_size == 13)
    fprintf(stderr, "   A B C D E F G H J K L M N\n");
  else
    fprintf(stderr, "   A B C D E F G H J K L M N O P Q R S T\n");

  for (j = 0; j < board_size; j++) {
    fprintf(stderr, "%2d", board_size - j);
    for (i = 0; i < board_size; i++) {
      if (p[j][i] && dragon[j][i].weak != DEAD) {
	k = p[j][i];
      } else {
	k = EMPTY;
      }
      switch (k) {
      case EMPTY:
	if ((*gob)[i][j] > 0)
	  DRAW_COLOR_CHAR(5, 'b');
	else if ((*gob)[i][j] < 0)
	  DRAW_COLOR_CHAR(2, 'w');
	else
	  fprintf(stderr, " .");
	break;
      case BLACK:
	fprintf(stderr, " X");
	break;
      case WHITE:
	fprintf(stderr, " O");
	break;
      }
    }
    fprintf(stderr, " %d", board_size - j);

    if (j == 5)
      fprintf(stderr, "     White territory %d", terri_eval[WHITE]);

    if (j == 7)
      fprintf(stderr, "     Black territory %d", terri_eval[BLACK]);

    if (j == 9)
      fprintf(stderr, "   W/B moyo %d/%d : %d",
	      moyo_eval[WHITE], moyo_eval[BLACK], moyo_eval[0]);

    fprintf(stderr, "\n");
  }

  if (board_size == 9)
    fprintf(stderr, "   A B C D E F G H J\n");
  else if (board_size == 13)
    fprintf(stderr, "   A B C D E F G H J K L M N\n");
  else
    fprintf(stderr, "   A B C D E F G H J K L M N O P Q R S T\n");
}


/* ================================================================ */
/*                         some test functions                      */
/* ================================================================ */


static void
print_ascii_area(void)
{
  int i, j, k;

#ifdef CURSES
  init_curses();
#endif

  if (board_size == 9)
    fprintf(stderr, "   A B C D E F G H J\n");
  else if (board_size == 13)
    fprintf(stderr, "   A B C D E F G H J K L M N\n");
  else
    fprintf(stderr, "   A B C D E F G H J K L M N O P Q R S T\n");

  for (j = 0; j < board_size; j++) {
    fprintf(stderr, "%2d", board_size - j);
    for (i = 0; i < board_size; i++) {
      if (p[j][i] && dragon[j][i].status != DEAD) {
	k = p[j][i];
      } else {
	k = EMPTY;
      }
      switch (k) {
      case EMPTY:
	if (area_color(j, i) == BLACK)
	  DRAW_COLOR_CHAR(5, 'b');
	else if (area_color(j, i) == WHITE)
	  DRAW_COLOR_CHAR(2, 'w');
	else
	  fprintf(stderr, " .");
	break;
      case BLACK:
	DRAW_COLOR_CHAR(5, 'X');
	break;
      case WHITE:
	DRAW_COLOR_CHAR(2, 'O');
	break;
      }
    }
    fprintf(stderr, " %d", board_size - j);

    if (j == 5)
      fprintf(stderr, "     White territory %d", terri_eval[WHITE]);

    if (j == 7)
      fprintf(stderr, "     Black territory %d", terri_eval[BLACK]);

    if (j == 9)
      fprintf(stderr, "   W/B moyo %d/%d : %d",
	      moyo_eval[WHITE], moyo_eval[BLACK], moyo_eval[0]);

    fprintf(stderr, "\n");
  }

  if (board_size == 9)
    fprintf(stderr, "   A B C D E F G H J\n");
  else if (board_size == 13)
    fprintf(stderr, "   A B C D E F G H J K L M N\n");
  else
    fprintf(stderr, "   A B C D E F G H J K L M N O P Q R S T\n");
}


void
print_txt_area(void)
{
  int i;

  for (i = 1; i <= areas_level; i++)
    if (area_stack[i].color > EMPTY)
      gprintf("area %d  %m: color %c,  %d stone  %d spaces\n",
	      i, area_stack[i].m, area_stack[i].n,
	      area_stack[i].color == WHITE ? 'W' : 'B',
	      area_stack[i].stone, area_stack[i].space);
}


void
sgfShowMoyo(int mode, int as_variant)
{
  int i,j;
  SGFNodeP startnode=0;
  SGFNodeP pr=startnode;

  if (as_variant) {
      startnode=sgfStartVariant(0);
      pr=sgfAddComment(startnode,"Moyo");
  }

  for (j=0; j < board_size; j++)
    for (i=0; i < board_size; i++) {
      switch(mode) {
      case SGF_SHOW_VALUE:
	sgfBoardNumber(pr,i,j,moyo_goban[j][i]);
	break;
      case SGF_SHOW_COLOR:
	if ((p[i][j]==EMPTY) ||(dragon[i][j].status!=ALIVE)) {
	  if (moyo_goban[j][i] > 0)
	    sgfTerritory(pr,i,j,BLACK);
	  else if (moyo_goban[j][i] < 0)
	    sgfTerritory(pr,i,j,WHITE);
	}
	break;
      }
    }
}


void
sgfShowTerri(int mode, int as_variant)
{
  int i,j;
  SGFNodeP startnode=0;
  SGFNodeP pr=startnode;

  if (as_variant) {
    startnode=sgfStartVariant(0);
    pr=sgfAddComment(startnode,"Terri");
  }

  for (j=0; j < board_size; j++)
    for (i=0; i < board_size; i++) {
      switch (mode) {
      case SGF_SHOW_VALUE:
	sgfBoardNumber(pr,i,j,terri_goban[j][i]);
	break;
      case SGF_SHOW_COLOR:
	if ((p[i][j]==EMPTY) ||(dragon[i][j].status!=ALIVE)) {
	  if (terri_goban[j][i] > 0)
	    /*FIXME better moyo_goban?*/
	    sgfTerritory(pr,i,j,BLACK);
	  else if (terri_goban[j][i] < 0)
	    sgfTerritory(pr,i,j,WHITE);
	}
	break;
      }
    }
}


void
sgfShowArea(int mode, int as_variant)
{
  int i,j,col;
  SGFNodeP startnode=0;
  SGFNodeP pr=startnode;

  if (as_variant) {
    startnode=sgfStartVariant(0);
    pr=sgfAddComment(startnode,"Area");
  }

  for (j=0; j < board_size; j++)
    for (i=0; i < board_size; i++) {
      switch (mode) {
      case SGF_SHOW_VALUE:
	sgfBoardNumber(pr,i,j,area_grid[j][i]); /*FIXME*/
	break;
      case SGF_SHOW_COLOR:
	if ((p[i][j]==EMPTY) ||(dragon[i][j].status!=ALIVE)) {
	  col=area_color(i,j);
	  if((col==BLACK)||(col==WHITE))
	    sgfTerritory(pr,i,j,col);
	}
	break;
      }
    }
}



/*
 * Local Variables:
 * tab-width: 8
 * c-basic-offset: 2
 * End:
 */
