/* gMUDix -- MUDix for X windows
 * Copyright (c) 2002 Marko Boomstra (m.boomstra@chello.nl)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include "mudix.h"
#ifdef IAC_DEBUG
  #define TELCMDS
  #define TELOPTS
#endif
#include <arpa/telnet.h>    /* has to be after IAC_DEBUG */

#define TELOPT_COMPRESS        (85)
#define TELOPT_COMPRESS2       (86)

#define PUTSHORT(cp, x) { if ((*cp++ = ((x)>>8)&0xFF) == IAC) *cp++ = IAC; \
                          if ((*cp++ = ((x))   &0xFF) == IAC) *cp++ = IAC; }


static void send_telnet(USER *user, unsigned char command, unsigned char option)
{
    unsigned char iac_buf[3];

#ifdef IAC_DEBUG
    printf("SEND: %6s - %02X\n", TELCMD(command), option);
#endif

    iac_buf[0] = IAC;
    iac_buf[1] = command;
    iac_buf[2] = option;

    /* lock the mutex */
    g_mutex_lock(user_network_mutex);

    write(user->net.sock, iac_buf, 3); /* bypassing normal send */

    /* unlock the mutex */
    g_mutex_unlock(user_network_mutex);
}


static void send_data_telnet(USER *user, unsigned char *data, int len)
{
#ifdef IAC_DEBUG
    printf("SEND: %d - ", len);
    {
	unsigned char *pData = data;
		 int   len2 = len;

	while (len2--) 
        {
	    printf("%02X ", *pData);
	    pData++;
	}
	printf("\n");
    }
#endif

    /* lock the mutex */
    g_mutex_lock(user_network_mutex);

    write(user->net.sock, data, len); /* bypassing normal send */

    /* unlock the mutex */
    g_mutex_unlock(user_network_mutex);
}


void send_naws(USER *user)
{
    unsigned char  temp[128];
    unsigned char *cp = temp;
             gint  x, y;
           
    if (!(user->flags & FLG_NAWS_UPDATES))
    {
        /* did not receive DO NAWS yet */
        return;
    }

    /* grab the x and y */
    gui_user_get_xy(user, &x, &y);

    *cp++ = IAC;
    *cp++ = SB;
    *cp++ = TELOPT_NAWS;
    PUTSHORT(cp, x);
    PUTSHORT(cp, y);
    *cp++ = IAC;
    *cp++ = SE;
    send_data_telnet(user, temp, cp-temp);
}


int check_iac(USER *user, unsigned char *pIac, unsigned char *pEnd)
{
    unsigned char *pStart = pIac;
    unsigned char  command = pIac[1];
    unsigned char  option = 0, temp[128];

    pIac++;
    /* an IAC length of 1 is way too small ;) */
    if (pEnd == pIac)
    {
        return IAC_BUFFER_END;
    }

    /* two IAC's in a row, escaped IAC */
    if (command == IAC)
    {
        return IAC_ESCAPED;
    }

    /* just advance the pointer */
    pIac++;

    /* only SB, WILL, WONT, DO, DONT have an option */
    if (command >= SB && command <= DONT)
    {
        /* grab the option */
        option = *pIac++;
    }

    if (command == SB)
    {
        /* read till end of SB command */
        while (pIac < pEnd)
        {
            if (*pIac == IAC && *(pIac-1) == IAC)
            {
                /* escaped IAC, skip it */
                pIac++;
            }
            else if (*pIac == SE && *(pIac-1) == IAC)
            {
                /* end of SB */
                break;
            }
            pIac++;
        }
        pIac++;
    }

#ifdef IAC_DEBUG
    printf("RECV: %6s - %02X\n", TELCMD(command), option);
#endif

    if (pIac > pEnd)
    {
        return IAC_BUFFER_END;      /* pointer crossed the boundary */
    }

    /* process IAC and produce a response */
    switch (command) 
    {
	case DONT:                  /* you are not to use option */
            switch (option) 
            {
		default:
        	    send_telnet(user, WONT, option);
		    break;
		case TELOPT_ECHO:
                    if (user->net.port != TELNET_PORT)
                    {
                        gui_input_visible(user, FALSE);
                    }
        	    send_telnet(user, WONT, option);
		    break;
	    }
	    break;
	case DO:                    /* please, you use option */
            switch (option) 
            {
		default:
	    	    send_telnet(user, WONT, option);
		    break;
		case TELOPT_NAWS:
                    /* first send WILL */
	    	    send_telnet(user, WILL, option);
                    user->flags |= FLG_NAWS_UPDATES;
                    /* then send a NAWS update */
                    send_naws(user);
                    break;
		case TELOPT_TTYPE:
	    	    send_telnet(user, WILL, option);
		    break;
		case TELOPT_ECHO:
                    gui_input_visible(user, TRUE);
	    	    send_telnet(user, WILL, option);
		    break;
	    }
	    break;
	case WONT:                  /* I won't use option */
            switch (option) 
            {
		default:
		    break;
		case TELOPT_ECHO:
                    gui_input_visible(user, TRUE);
		    break;
	    }
	    break;
	case WILL:                  /* I will use option */
            switch (option) 
            {
		default:
                    break;
                case TELOPT_COMPRESS:
                    send_telnet(user, DONT, option);
                    break;
                case TELOPT_COMPRESS2:
                    send_telnet(user, DO, option);
                    break;
		case TELOPT_ECHO:
                    if (user->net.port != TELNET_PORT)
                    {
                        gui_input_visible(user, FALSE);
                    }
		    break;
	    }
	    break;
	case SB:	            /* interpret as subnegotiation */
            switch (option) 
            {
		default:
		    break;
                case TELOPT_COMPRESS2:
                    if (MCCP_USER(user))
                    {
                        /* stream already allocated!? */
                        break;
                    }

                    /* setup MCCP */
                    mccp_open(user);

                    return IAC_MCCP_START;
		case TELOPT_TTYPE:
		    sprintf((char *)temp, "%c%c%c%c%s%c%c", 
			IAC, SB, option, TELQUAL_IS, "vt100", IAC, SE);
		    send_data_telnet(user, temp, strlen((char *)temp+4)+4);
		    break;
	    }

            break;
	case GA:                /* you may reverse the line */
	case EL:                /* erase the current line */
	case EC:                /* erase the current character */
	case AYT:               /* are you there */
	case AO:                /* abort output--but let prog finish */
	case IP:                /* interrupt process--permanently */
	case BREAK:             /* break */
	case DM:                /* data mark--for connect. cleaning */
	case NOP:               /* nop */
	case SE:                /* end sub negotiation */
	case EOR:               /* end of record (transparent mode) */
	case ABORT:             /* Abort process */
	case SUSP:              /* Suspend process */
	case xEOF:              /* End of file: EOF is already used... */
        default:
	    break;
    }

    return pIac - pStart;       /* return the length of the IAC */
}

