/* gmpc-lyricwiki (GMPC plugin)
 * Copyright (C) 2008-2009 Qball Cow <qball@sarine.nl>
 * Project homepage: http://gmpcwiki.sarine.nl/
 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
*/

#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libmpd/debug_printf.h>
#include <gmpc/gmpc_easy_download.h>
#include <gmpc/plugin.h>
#include <glib.h>
#include <config.h>

/**
 * Get/Set enabled 
 */
static int lyricwiki_get_enabled()
{
	return cfg_get_single_value_as_int_with_default(config, "lyricwiki-plugin", "enable", TRUE);
}
static void lyricwiki_set_enabled(int enabled)
{
	cfg_set_single_value_as_int(config, "lyricwiki-plugin", "enable", enabled);
}

/* Get priority */
static int lyricwiki_fetch_cover_priority(void){
	return cfg_get_single_value_as_int_with_default(config, "lyricwiki-plugin", "priority", 80);
}
static void lyricwiki_fetch_cover_priority_set(int priority){
	cfg_set_single_value_as_int(config, "lyricwiki-plugin", "priority", priority);
}
static int lyricwiki_fetch_get_lyric(mpd_Song *song,MetaDataType type, char **path)
{
	int result = 0;
    gchar *artist, *title;
    int retv = META_DATA_UNAVAILABLE;

    /* Check if enabled */
	if(lyricwiki_get_enabled() == FALSE)
	{
		return META_DATA_UNAVAILABLE;
	}
    /* Can I fetch this type? */
    if(type != META_SONG_TXT)
    {
        return META_DATA_UNAVAILABLE;
    }
    /* Check available fields */
    if(song->artist == NULL || song->title == NULL)
    {
        return META_DATA_UNAVAILABLE;
    }
       /* escape, for html query */
    artist = gmpc_easy_download_uri_escape(song->artist);
    title =  gmpc_easy_download_uri_escape(song->title);

    if(artist && title)
    {
        gmpc_easy_download_struct dld = {NULL, 0, -1, NULL, NULL};
        /* create the uri */
        gchar *uri_path = g_strdup_printf("http://lyricwiki.org/api.php?artist=%s&song=%s&fmt=xml",
                artist, title);
        if(gmpc_easy_download(uri_path, &dld))
        {
            xmlDocPtr doc = xmlParseMemory(dld.data,dld.size);
            if(doc)
            {
                xmlNodePtr root = xmlDocGetRootElement(doc);
                xmlNodePtr cur = root->xmlChildrenNode;
                for(;cur;cur = cur->next){
                    if(xmlStrEqual(cur->name, (xmlChar *)"lyrics")){
                        xmlChar *lyric = xmlNodeGetContent(cur);
                        if(lyric && strcmp(lyric,"Not found")!= 0)
                        {
                            /* now we want to store it somewhere */
                            gchar *full_path =  gmpc_get_metadata_filename(META_SONG_TXT, song, NULL);
                            /* don't need fancy writers/loaders, just dump the file. */
                            if(g_file_set_contents(full_path, lyric, -1, NULL))
                            {
                                *path = full_path;
                                retv = META_DATA_AVAILABLE;
                            }
                            else
                                g_free(full_path);
                        }
                        xmlFree(lyric);

                    }
                }
                xmlFreeDoc(doc);
            }

        }
        gmpc_easy_download_clean(&dld);
        g_free(uri_path);
        
    }
    if(artist) g_free(artist);
    if(title) g_free(title);
    return retv; 
}	

/**
 * Preferences
 */
static void preferences_enable_cb_callback(GtkToggleButton *button, gpointer data)
{
	lyricwiki_set_enabled(gtk_toggle_button_get_active(button));
}
static void preferences_construct(GtkWidget *container)
{
	GtkWidget *vbox = gtk_vbox_new(FALSE, 6);
	GtkWidget *checkbox = gtk_check_button_new_with_mnemonic("_Enable lyricwiki.org as lyric source.");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(checkbox), lyricwiki_get_enabled());

	g_signal_connect(G_OBJECT(checkbox), "toggled", G_CALLBACK(preferences_enable_cb_callback), NULL);
	gtk_box_pack_start(GTK_BOX(vbox), checkbox, FALSE, FALSE,0);

	gtk_widget_show_all(vbox);
	gtk_container_add(GTK_CONTAINER(container), vbox);
}

/**
 * Generic de-construct function
 */
static void preferences_destroy(GtkWidget *container)
{
	GtkWidget *widget = gtk_bin_get_child(GTK_BIN(container));
	if(widget)
	{
		gtk_container_remove(GTK_CONTAINER(container),widget);
	}
}

gmpcMetaDataPlugin lw_cover = {
	.get_priority   = lyricwiki_fetch_cover_priority,
    .set_priority   =  lyricwiki_fetch_cover_priority_set,
    .get_image      = lyricwiki_fetch_get_lyric
};

gmpcPrefPlugin lw_pref = {
	.construct = preferences_construct,
	.destroy = preferences_destroy
};

int plugin_api_version = PLUGIN_API_VERSION;

gmpcPlugin plugin = {
	.name           = "LyricWiki.org lyric source",
	.version        = {PLUGIN_MAJOR_VERSION,PLUGIN_MINOR_VERSION,PLUGIN_MICRO_VERSION},
	.plugin_type    = GMPC_PLUGIN_META_DATA,
	.metadata       = &lw_cover,
	.get_enabled    = lyricwiki_get_enabled,
	.set_enabled    = lyricwiki_set_enabled,
	.pref 					= &lw_pref
};
