#include <stdio.h>
#include <string.h>
#include <gtk/gtk.h>
#include <libxml/parser.h>
#include <gmpc/plugin.h>
#include <gmpc/gmpc_easy_download.h>
#include "magnatune.h"

static void magnatune_cleanup_xml();
xmlDocPtr magnatune_xmldoc = NULL;
GMutex *mt_db_lock = NULL;


void magnatune_db_destroy(void)
{
    g_mutex_lock(mt_db_lock);
    g_mutex_unlock(mt_db_lock);
    g_mutex_free(mt_db_lock);
    if(magnatune_xmldoc)
    {
        xmlFreeDoc(magnatune_xmldoc);
        xmlCleanupParser();
        magnatune_xmldoc = NULL;
    }
}

/* run this before using the other fucntions */
void magnatune_db_init()
{
    mt_db_lock = g_mutex_new();
}

void magnatune_db_open()
{
    gchar *path = g_strdup_printf("%s/.gmpc/magnatune.xml", g_get_home_dir());

    g_mutex_lock(mt_db_lock);	
    /** 
     * if the db doesn't exists exit
     */
    if(!g_file_test(path, G_FILE_TEST_EXISTS))
    {
        g_free(path);
        g_mutex_unlock(mt_db_lock);
        return;
    }
    /**
     * if open close it
     */
    if(magnatune_xmldoc)
    {
        xmlFreeDoc(magnatune_xmldoc);
        xmlCleanupParser();
        magnatune_xmldoc = NULL;
    }
    magnatune_xmldoc = xmlParseFile(path);

    g_mutex_unlock(mt_db_lock);
    g_free(path);
}

gboolean magnatune_db_has_data()
{
    gboolean data = FALSE;
    g_mutex_lock(mt_db_lock);
    data = (magnatune_xmldoc != NULL)? TRUE:FALSE;
    g_mutex_unlock(mt_db_lock);
    return data;
}

GList * magnatune_db_get_genre_list()
{
    GList *list = NULL;
    xmlNodePtr root;
    xmlNodePtr cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);
    if(magnatune_xmldoc == NULL)
    {
        g_mutex_unlock(mt_db_lock);
        return NULL;
    }

    root = xmlDocGetRootElement(magnatune_xmldoc);
    cur = root->xmlChildrenNode;
    while(cur != NULL)
    {
        xmlNodePtr cur2;
        if(xmlStrEqual(cur->name, (xmlChar *)"Album"))
        {
            xmlChar *temp = NULL;
            char *genre = NULL;
            cur2 = cur->xmlChildrenNode;
            while(cur2 != NULL)
            {
                if(xmlStrEqual(cur2->name, (xmlChar *)"magnatunegenres"))
                {
                    gchar **tokens = NULL;
                    genre = xmlNodeGetContent(cur2);
                    if(genre)
                    {
                        int i =0;
                        tokens = g_strsplit(genre,",",0);
                        for(i=0; tokens[i];i++)
                        {
                            GList *node = g_list_find_custom(list, tokens[i], (GCompareFunc)strcmp);
                            if(node == NULL) {
                                list = g_list_append(list, g_strdup(tokens[i]));
                            } 
                        }
                        g_strfreev (tokens);
                        xmlFree(genre);
                    }
                }
                cur2 = cur2->next;
            }
        }

        cur = cur->next;
    }
    g_mutex_unlock(mt_db_lock);
    return g_list_sort(list, (GCompareFunc)strcasecmp);
}

gchar *magnatune_db_get_value(const char *artist, const char *album, int type)
{
    gchar *retval = NULL;
    xmlNodePtr root;
    xmlNodePtr cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);
    if(magnatune_xmldoc == NULL || !artist )
    {
        g_mutex_unlock(mt_db_lock);
        return NULL;
    }

    root = xmlDocGetRootElement(magnatune_xmldoc);
    cur = root->xmlChildrenNode;
    while(cur != NULL && !retval)
    {
        xmlNodePtr cur2;
        if(xmlStrEqual(cur->name, (xmlChar *)"Album"))
        {
            xmlChar *temp = NULL;
            char *gartist = NULL;
            char *galbum = NULL;
            char *gvalue = NULL;
            cur2 = cur->xmlChildrenNode;
            while(cur2 != NULL && !retval)
            {
                if(!gartist && xmlStrEqual(cur2->name, (xmlChar *)"artist"))
                {
                    gartist = xmlNodeGetContent(cur2);
                }
                else if(!galbum && xmlStrEqual(cur2->name, (xmlChar *)"albumname"))
                {
                    galbum= xmlNodeGetContent(cur2);
                }
                else if(!gvalue && xmlStrEqual(cur2->name, (xmlChar *)((type== META_ARTIST_ART)?"artistphoto":"cover_small")))
                {
                    gvalue= xmlNodeGetContent(cur2);
                }
                cur2 = cur2->next;
            }
            if(gvalue && artist && !strncmp(gartist, artist,strlen(artist)))
            {
                if(type == META_ARTIST_ART)
                {
                    retval = g_strdup(gvalue);
                }
                else if(galbum && !strcmp(galbum, album)){
                    retval = g_strdup(gvalue);			
                }
            }
            if(gartist)
                xmlFree(gartist);
            if(galbum)
                xmlFree(galbum);
            if(gvalue)
                xmlFree(gvalue);

        }
        cur = cur->next;
    }

    g_mutex_unlock(mt_db_lock);
    return retval;
}
/**
 * This removes duplicate/unused fields from every Track entry, 
 * This remove around 1/2 the file size
 * NOT THREAD SAFE, USE INTERNALLY
 */
static void magnatune_cleanup_xml()
{
    xmlNodePtr root;
    xmlNodePtr cur;	
    if(!magnatune_xmldoc)
        return;
    root = xmlDocGetRootElement(magnatune_xmldoc);
    cur = root->xmlChildrenNode;
    while(cur != NULL) {
        xmlNodePtr cur2;
        if(xmlStrEqual(cur->name, (xmlChar *)"Album")) {
            cur2 = cur->xmlChildrenNode;	
            while(cur2 != NULL) {
                if(xmlStrEqual(cur2->name, (xmlChar *)"Track"))	{
                    xmlNodePtr cur3 = cur2->xmlChildrenNode;
                    while(cur3) {
                        if(xmlStrEqual(cur3->name, (xmlChar *)"albumname") || 
                                xmlStrEqual(cur3->name, (xmlChar *)"artist") ||
                                xmlStrEqual(cur3->name, (xmlChar *)"artistdesc") ||
                                xmlStrEqual(cur3->name, (xmlChar *)"home") ||
                                xmlStrEqual (cur3->name, (xmlChar *)"artistphoto") ||
                                xmlStrEqual (cur3->name, (xmlChar *)"mp3lofi") ||
                                xmlStrEqual (cur3->name, (xmlChar *)"albumsku") ||
                                xmlStrEqual (cur3->name, (xmlChar *)"mp3genre") ||
                                xmlStrEqual (cur3->name, (xmlChar *)"magnatunegenres") ||
                                xmlStrEqual (cur3->name, (xmlChar *)"buy"))
                        {
                            xmlUnlinkNode(cur3);
                            xmlFreeNode(cur3);
                            cur3  = cur2->xmlChildrenNode;
                        } else {
                            cur3 = cur3->next;
                        }
                    }
                }	
                cur2 = cur2->next;
            }	
        }
        cur = cur->next;
    }
}

void magnatune_db_download_xml(ProgressCallback cb, gpointer data )
{

    gmpc_easy_download_struct dld = {NULL, 0, -1, cb, data};
    g_mutex_lock(mt_db_lock);

    if(magnatune_xmldoc)
    {
        xmlFreeDoc(magnatune_xmldoc);
        xmlCleanupParser();
        magnatune_xmldoc = NULL;
    }

    if(gmpc_easy_download("http://www.magnatune.com/info/album_info.xml", &dld))
    {
        gchar *path = NULL;
        magnatune_xmldoc = xmlParseMemory(dld.data, dld.size);
        magnatune_cleanup_xml();
        path = g_strdup_printf("%s/.gmpc/magnatune.xml", g_get_home_dir());
        xmlSaveFile(path,magnatune_xmldoc); 
        g_free(path);
    }
    else 
    {
        /* update */
        gchar *path = g_strdup_printf("%s/.gmpc/magnatune.xml", g_get_home_dir());
        if(g_file_test(path, G_FILE_TEST_EXISTS))
        {
            magnatune_xmldoc = xmlParseFile(path);
        }
        g_free(path);
    }
    /**
     * cleanup
     */
    gmpc_easy_download_clean(&dld);
    g_mutex_unlock(mt_db_lock);
}

GList * magnatune_db_get_artist_list(char *wanted_genre)
{
    GList *list = NULL;
    xmlNodePtr root;
    xmlNodePtr cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);
    if(magnatune_xmldoc == NULL || wanted_genre == NULL)
    {
        g_mutex_unlock(mt_db_lock);
        return NULL;
    }

    root = xmlDocGetRootElement(magnatune_xmldoc);
    cur = root->xmlChildrenNode;
    while(cur != NULL)
    {
        xmlNodePtr cur2;
        if(xmlStrEqual(cur->name, (xmlChar *)"Album"))
        {
            xmlChar *temp = NULL;
            char *genre = NULL;
            char *artist = NULL;
            cur2 = cur->xmlChildrenNode;
            while(cur2 != NULL)
            {
                if(xmlStrEqual(cur2->name, (xmlChar *)"magnatunegenres"))
                {
                    genre = xmlNodeGetContent(cur2);
                }
                else if(xmlStrEqual(cur2->name, (xmlChar *)"artist"))
                {
                    artist = xmlNodeGetContent(cur2);
                }
                cur2 = cur2->next;
            }
            if(genre && artist && strstr(genre, wanted_genre))
            {
                GList *node = g_list_find_custom(list, artist, (GCompareFunc)strcmp);
                if(node == NULL) {
                    list = g_list_append(list, g_strdup(artist));
                } 

            }
            if(genre)
                xmlFree(genre);
            if(artist)
                xmlFree(artist);
        }
        cur = cur->next;
    }
    g_mutex_unlock(mt_db_lock);
    return g_list_sort(list, (GCompareFunc)strcasecmp);
}
GList * magnatune_db_get_album_list(char *wanted_genre,char *wanted_artist)
{
    GList *list = NULL;
    xmlNodePtr root;
    xmlNodePtr cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);
    if(magnatune_xmldoc == NULL || wanted_genre == NULL || wanted_artist == NULL)
    {
        g_mutex_unlock(mt_db_lock);
        return NULL;
    }

    root = xmlDocGetRootElement(magnatune_xmldoc);
    cur = root->xmlChildrenNode;
    while(cur != NULL)
    {
        xmlNodePtr cur2;
        if(xmlStrEqual(cur->name, (xmlChar *)"Album"))
        {
            xmlChar *temp = NULL;
            char *genre = NULL;
            char *album = NULL;
            char *artist = NULL;
            cur2 = cur->xmlChildrenNode;
            while(cur2 != NULL)
            {
                if(xmlStrEqual(cur2->name, (xmlChar *)"magnatunegenres"))
                {
                    genre = xmlNodeGetContent(cur2);
                }
                else if(xmlStrEqual(cur2->name, (xmlChar *)"artist"))
                {
                    artist = xmlNodeGetContent(cur2);
                }
                else if(xmlStrEqual(cur2->name, (xmlChar *)"albumname"))
                {
                    album = xmlNodeGetContent(cur2);
                }
                cur2 = cur2->next;
            }
            if(genre && artist&&album && strstr(genre, wanted_genre) && !strcmp(artist, wanted_artist))
            {
                GList *node = g_list_find_custom(list, album, (GCompareFunc)strcmp);
                if(node == NULL) {
                    list = g_list_append(list, g_strdup(album));
                } 

            }
            if(genre)
                xmlFree(genre);
            if(artist)
                xmlFree(artist);
            if(album)
                xmlFree(album);
        }
        cur = cur->next;
    }
    g_mutex_unlock(mt_db_lock);
    return g_list_sort(list, (GCompareFunc)strcasecmp);
}


GList * magnatune_db_get_url_list(const char *wanted_genre,const char *wanted_artist, const char *wanted_album)
{
    GList *list = NULL;
    xmlNodePtr root;
    xmlNodePtr cur;
    /** check if there is data */
    g_mutex_lock(mt_db_lock);
    if(magnatune_xmldoc == NULL || wanted_genre == NULL) 
    {
        g_mutex_unlock(mt_db_lock);
        return NULL;
    }

    root = xmlDocGetRootElement(magnatune_xmldoc);
    cur = root->xmlChildrenNode;
    while(cur != NULL)
    {
        xmlNodePtr cur2;
        if(xmlStrEqual(cur->name, (xmlChar *)"Album"))
        {
            xmlChar *temp = NULL;
            char *genre = NULL;
            char *album = NULL;
            char *artist = NULL;
            gboolean add_urls = FALSE;
            xmlNodePtr tracks;
            cur2 = cur->xmlChildrenNode;
            while(cur2 != NULL)
            {
                if(xmlStrEqual(cur2->name, (xmlChar *)"magnatunegenres"))
                {
                    genre = xmlNodeGetContent(cur2);
                }
                else if(xmlStrEqual(cur2->name, (xmlChar *)"artist"))
                {
                    artist = xmlNodeGetContent(cur2);
                }
                else if(xmlStrEqual(cur2->name, (xmlChar *)"albumname"))
                {
                    album = xmlNodeGetContent(cur2);
                }
                cur2 = cur2->next;
            }
            if(genre && strstr(genre, wanted_genre))
            {
                if(wanted_artist && wanted_album) {
                    if(!strcmp(wanted_artist,artist) && !strcmp(wanted_album, album)) {
                        add_urls = TRUE;
                    }
                }else if(wanted_artist) {
                    if(!strcmp(wanted_artist, artist)) {
                        add_urls = TRUE;
                    }
                } else {
                    add_urls = TRUE;
                }
            }
            if(genre)
                xmlFree(genre);
            if(artist)
                xmlFree(artist);
            if(album)
                xmlFree(album);

            if(add_urls)
            {
                cur2 = cur->xmlChildrenNode;
                while(cur2)
                {
                    if (xmlStrEqual(cur2->name, (xmlChar *)"Track"))
                    {
                        tracks = cur2->xmlChildrenNode;
                        while(tracks)
                        {
                            if (xmlStrEqual(tracks->name, (xmlChar *)"url"))
                            {
                                xmlChar *url = xmlNodeGetContent(tracks);
                                list = g_list_append(list, g_strdup(url));
                                xmlFree(url);
                            }
                            tracks = tracks->next;
                        }
                    }
                    cur2 = cur2->next;
                }
            }
        }
        cur = cur->next;
    }
    g_mutex_unlock(mt_db_lock);
    return list;
}
