/* GKrellMSS - GKrellM Sound Scope
|  Copyright (C) 2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellmss.h"

#include "../pixmaps/krell_vu.xpm"
#include "../pixmaps/krell_vu_peak.xpm"

#include "../pixmaps/button_sweep_dn.xpm"
#include "../pixmaps/button_sweep_up.xpm"

#include "../pixmaps/krell_sensitivity.xpm"

/* ======================================================================== */
#define DEFAULT_CHART_HEIGHT 30

SoundMonitor		*sound;

static Monitor		*mon_sound;
Chart				*chart;
static ChartConfig	*chart_config;

static gint			style_id;

static GdkPixmap	*button_sweep_up_pixmap,
					*button_sweep_dn_pixmap;
static GdkBitmap	*button_sweep_up_mask,
					*button_sweep_dn_mask;

static Decal		*decal_sweep_up,
					*decal_sweep_dn;

static Decal		*decal_mode_button;

Decal				*decal_label0,	/* Chart labels */
					*decal_label1;

static Krell		*krell_in_motion;

static gint			vu_meter_width;

static void
draw_mode_button(gboolean pressed)
	{
	gint	frame, x;

	x = gkrellm_chart_width() - decal_mode_button->w + 1;
	frame = pressed ? D_MISC_BUTTON_IN : D_MISC_BUTTON_OUT;
	gkrellm_draw_decal_pixmap(NULL, decal_mode_button, frame);
	gkrellm_draw_decal_on_chart(chart, decal_mode_button, x, 0);
	}

void
draw_sound_chart(gboolean force_reset, gboolean draw_grid)
	{
	if (sound->mode == SOUND_MODE_OSCOPE)
		draw_oscope(force_reset, draw_grid);
	else if (sound->mode == SOUND_MODE_SPECTRUM)
		draw_spectrum(force_reset, draw_grid);
	if (sound->mouse_in_chart)
		draw_mode_button(sound->mode_button_pressed);
	gkrellm_draw_chart_to_screen(chart);
	sound->streaming = FALSE;
	}

static gint
cb_chart_release(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	sound->mode_button_pressed = FALSE;
	return TRUE;
	}

static gint
cb_chart_press(GtkWidget *widget, GdkEventButton *ev, gpointer data)
	{
	if (gkrellm_in_decal(decal_mode_button, ev))
		{
		sound->mode += 1;
		if (sound->mode >= N_SOUND_MODES)
			sound->mode = 0;
		sound->mode_button_pressed = TRUE;
		gkrellm_config_modified();
		draw_sound_chart(FORCE_RESET, DRAW_GRID);
		}
	else if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
		{
		if (sound->mode == SOUND_MODE_SPECTRUM && !spectrum->reset)
			{
			if (spectrum->x_highlight > 0)
				spectrum->x_highlight = 0;
			else
				spectrum->x_highlight = sound->x_mouse;
			}
		}
	else if (ev->button == 2)
		{
		if (sound->fd < 0)
			{
			sound_open_stream(sound->host);
			}
		else
			{
			sound->extra_info = !sound->extra_info;
			gkrellm_config_modified();
			draw_sound_chart(FORCE_RESET, 0);
			}
		}
	else if (   ev->button == 3
			 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
			)
		gkrellm_chartconfig_window_create(chart);
	return TRUE;
	}

static void
update_sound(void)
	{
	gint		d, left, right, left_peak, right_peak;

	if ((left = sound->left_value) > oscope->vert_max)
		left = oscope->vert_max;
	if ((right = sound->right_value) > oscope->vert_max)
		right = oscope->vert_max;
	left_peak = sound->left_peak_value - oscope->vert_max / 30;
	if ((d = sound->left_peak_value - left) > 0)
		left_peak -= d / 30;
	right_peak = sound->right_peak_value - oscope->vert_max / 30;
	if ((d = sound->right_peak_value - right) > 0)
		right_peak -= d / 30;

	if (left_peak < left)
		left_peak = left;
	if (right_peak < right)
		right_peak = right;

	gkrellm_update_krell(chart->panel, sound->krell_left, left);
	gkrellm_update_krell(chart->panel, sound->krell_left_peak, left_peak);
	gkrellm_update_krell(chart->panel, sound->krell_right, right);
	gkrellm_update_krell(chart->panel, sound->krell_right_peak, right_peak);

	sound->left_peak_value = left_peak;
	sound->right_peak_value = right_peak;
	sound->left_value = 0;
	sound->right_value = 0;

	d = sound->krell_sensitivity_y_target - sound->krell_sensitivity_y;
	if (d > 0)
		{
		sound->krell_sensitivity_y += 1 + d / 4;
		gkrellm_move_krell_yoff(chart->panel, sound->krell_sensitivity,
					sound->krell_sensitivity_y);
		}
	else if (d < 0)
		{
		sound->krell_sensitivity_y -= 1 - d / 4;
		gkrellm_move_krell_yoff(chart->panel, sound->krell_sensitivity,
					sound->krell_sensitivity_y);
		}

	gkrellm_draw_panel_layers(chart->panel);

	draw_sound_chart(0, 0);
	sound->streaming = FALSE;
	}


static void
sound_vertical_scaling(void)
	{
	gfloat	audio_taper;

	audio_taper = log(1 + sound->vert_sensitivity) / log(2.0);

	oscope->vert_max = (gint) (audio_taper * 32767.0);
	spectrum->vert_max = (gint) (audio_taper * 24);
	gkrellm_set_krell_full_scale(sound->krell_left, oscope->vert_max, 1);
	gkrellm_set_krell_full_scale(sound->krell_right, oscope->vert_max, 1);
	gkrellm_set_krell_full_scale(sound->krell_left_peak, oscope->vert_max, 1);
	gkrellm_set_krell_full_scale(sound->krell_right_peak, oscope->vert_max, 1);
	}

static void
update_slider_position(Krell *k, gint x_ev)
	{
	gint	x,
			w = gkrellm_chart_width();

	if (x_ev < sound->vu_x0)
		x_ev = sound->vu_x0;
	if (x_ev >= w)
		x_ev = w - 1;
	sound->x_sensitivity_raw = x_ev;
	x = x_ev - sound->vu_x0;
	x = k->full_scale * x / (vu_meter_width - 1);
	if (x < 0)
		x = 0;

	sound->vert_sensitivity = (100.0 - (gfloat) x) / 100.0;
	if (sound->vert_sensitivity < 0.05)
		sound->vert_sensitivity = 0.05;
	if (sound->vert_sensitivity > 1.0)
		sound->vert_sensitivity = 1.0;
	sound_vertical_scaling();
	gkrellm_config_modified();
	gkrellm_update_krell(chart->panel, k, (gulong) x);
	gkrellm_draw_panel_layers(chart->panel);
	}


static gint
expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GdkPixmap	*pixmap	= NULL;

	if (widget == chart->panel->drawing_area)
		pixmap = chart->panel->pixmap;
	else if (widget == chart->drawing_area)
		pixmap = chart->pixmap;
	if (pixmap)
		gdk_draw_pixmap(widget->window, gkrellm_draw_GC(1), pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
	return FALSE;
	}


static gboolean
cb_chart_enter(GtkWidget *w, GdkEventButton *ev, gpointer data)
	{
	sound->mouse_in_chart = TRUE;
	draw_mode_button(FALSE);
	gkrellm_draw_chart_to_screen(chart);
	return TRUE;
	}

static gint
cb_chart_leave(GtkWidget *w, GdkEventButton *ev, gpointer data)
	{
	sound->mouse_in_chart = FALSE;
	sound->mode_button_pressed = FALSE;
	draw_sound_chart(FORCE_RESET, 0);
	return TRUE;
	}

static gint
cb_chart_motion(GtkWidget *widget, GdkEventButton *ev)
	{
	sound->x_mouse = ev->x;
	return TRUE;
	}

static gint
cb_panel_enter(GtkWidget *widget, GdkEventButton *ev)
	{
	sound->krell_sensitivity_y_target = sound->krell_sensitivity_y_dn;
	return TRUE;
	}

static gint
cb_panel_leave(GtkWidget *widget, GdkEventButton *ev)
	{
	sound->krell_sensitivity_y_target = sound->krell_sensitivity_y_up;
	return TRUE;
	}

static gint
cb_panel_release(GtkWidget *widget, GdkEventButton *ev)
	{
	gint	dx = 0;
	gint	delta;

	delta = vu_meter_width / 30;
	if (delta == 0)
		delta = 1;
	if (ev->button == 1)
		krell_in_motion = NULL;
	else if (ev->button == 4)
		dx = delta;
	else if (ev->button == 5)
		dx = -delta;
	if (dx)
		update_slider_position(sound->krell_sensitivity,
					sound->x_sensitivity_raw + dx);
	return TRUE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
    {
	Krell	*k = sound->krell_sensitivity;

	if (ev->button == 3)
		{
		gkrellm_open_config_window(mon_sound);
		return TRUE;
		}
	if (   ev->button == 1 && ev->x > sound->vu_x0
		&& ev->y >= k->y0 && ev->y <= k->y0 + k->h_frame)
		{
		krell_in_motion = k;
		update_slider_position(krell_in_motion, ev->x);
		}
	return TRUE;
	}

static gint
cb_panel_motion(GtkWidget *widget, GdkEventButton *ev)
	{
	GdkModifierType	state;
	Krell			*k;

	if (krell_in_motion)
		{
		/* Check if button is still pressed, in case missed button_release
		*/
		state = ev->state;
		if (!(state & GDK_BUTTON1_MASK))
			{
			krell_in_motion = NULL;
			return TRUE;
			}
		k = krell_in_motion;
		update_slider_position(krell_in_motion, ev->x);
		}
	return TRUE;
	}


static void
height_changed(gpointer data)
	{
	if (sound->mode == SOUND_MODE_SPECTRUM)
		{
		load_spectrum_images();
		}
	draw_sound_chart(FORCE_RESET, DRAW_GRID);
	}

static void
create_chart(GtkWidget *vbox, gint first_create)
	{
	Chart		*cp = chart;

	/* Create some decals that will be drawn on the chart.  They won't live
	|  in a panel where they can be automatically destroyed, so I must
    |  destroy them at create events and then recreate them with NULL panel
	|  and style pointers.
	*/
	gkrellm_destroy_decal(decal_label0);
	gkrellm_destroy_decal(decal_label1);
	decal_label0 = gkrellm_create_decal_text(NULL, "888 msec",
			gkrellm_chart_alt_textstyle(DEFAULT_STYLE_ID), NULL, 2, 0, 0);
	decal_label1 = gkrellm_create_decal_text(NULL, "8.8K",
			gkrellm_chart_alt_textstyle(DEFAULT_STYLE_ID), NULL, 2, 0, 0);

	gkrellm_destroy_decal(decal_mode_button);
	decal_mode_button = gkrellm_create_decal_pixmap(NULL,
			gkrellm_decal_misc_pixmap(), gkrellm_decal_misc_mask(),
			N_MISC_DECALS, NULL, 0, 0);

	gkrellm_set_chart_height_default(cp, DEFAULT_CHART_HEIGHT);
	gkrellm_chart_create(vbox, mon_sound, cp, &chart_config);

	gkrellm_set_chartconfig_flags(chart_config, NO_CONFIG_FIXED_GRIDS);

	/* The only config is height and I don't call gkrellm draw chart routines,
	|  so set the chart draw function to just reset the chart.  It will only
	|  be called when the chart height is changed.
	*/
	gkrellm_set_draw_chart_function(cp, height_changed, NULL);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
				(GtkSignalFunc) expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_chart_press, cp);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"button_release_event", (GtkSignalFunc) cb_chart_release, cp);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "enter_notify_event",
				(GtkSignalFunc) cb_chart_enter, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "leave_notify_event",
				(GtkSignalFunc) cb_chart_leave, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),
				"motion_notify_event", (GtkSignalFunc) cb_chart_motion, NULL);
		gtk_widget_show(vbox);
		}
	oscope_horizontal_scaling();
	load_spectrum_images();
	draw_sound_chart(FORCE_RESET, DRAW_GRID);
	}

static void
cb_sweep_pressed(DecalButton *button)
	{
	gint	dir;

	dir = GPOINTER_TO_INT(button->data);
	dir = (dir == 0) ? 1 : -1;
	if (sound->mode == SOUND_MODE_OSCOPE)
		{
		oscope->usec_per_div = gkrellm_125_sequence(oscope->usec_per_div + dir,
				TRUE, LOW_SWEEP, HIGH_SWEEP, FALSE, FALSE);
		oscope_horizontal_scaling();
		}
	else if (sound->mode == SOUND_MODE_SPECTRUM)
		change_spectrum_scale(dir);
	draw_sound_chart(FORCE_RESET, 0);
	}

static void
create_panel_buttons(Panel *p)
	{
	GdkImlibImage	*im	= NULL;
	Style			*style;
	Margin			*margin;
	gint			x, y;

	style = gkrellm_meter_style(style_id);
	margin = gkrellm_get_style_margins(style);

	y = 3;
	gkrellm_get_gkrellmrc_integer("sound_button_sweep_yoff", &y);

	if (!gkrellm_load_image("buttom_sweep_dn", NULL, &im, STYLE_NAME))
		gkrellm_load_image("button_sweep_dn", button_sweep_dn_xpm,
					&im, STYLE_NAME);
	gkrellm_render_to_pixmap(im, &button_sweep_dn_pixmap,
				&button_sweep_dn_mask, 0, 0);
	decal_sweep_dn = gkrellm_create_decal_pixmap(p, button_sweep_dn_pixmap,
				button_sweep_dn_mask, 2, style, margin->left, y);

	x = decal_sweep_dn->x;
	y = decal_sweep_dn->y;

	if (!gkrellm_load_image("buttom_sweep_up", NULL, &im, STYLE_NAME))
		gkrellm_load_image("button_sweep_up", button_sweep_up_xpm,
					&im, STYLE_NAME);
	gkrellm_render_to_pixmap(im, &button_sweep_up_pixmap,
				&button_sweep_up_mask, 0, 0);
	decal_sweep_up = gkrellm_create_decal_pixmap(p, button_sweep_up_pixmap,
				button_sweep_up_mask, 2, style,
				x + decal_sweep_dn->w, y);


	gkrellm_decal_on_top_layer(decal_sweep_dn, TRUE);
	gkrellm_decal_on_top_layer(decal_sweep_up, TRUE);

	sound->vu_x0 = decal_sweep_up->x + decal_sweep_up->w;
	vu_meter_width = gkrellm_chart_width() - sound->vu_x0;
	}

static Krell *
default_or_themed_krell(Panel *p, gchar *name, GdkImlibImage *im, gint yoff,
	gint depth, gint x_hot, gint expand, gint left_margin, gint right_margin)
	{
	Krell	*k;
	Style	*style;

	/* Work with a copy since we may modify the style.
	*/
	style = gkrellm_copy_style(gkrellm_meter_style_by_name(name));
	if (!gkrellm_style_is_themed(style))
		gkrellm_set_style_krell_values(style, yoff, depth, x_hot,
				expand, left_margin, right_margin);
	else
		{
		if (style->krell_left_margin == 0)
			style->krell_left_margin = left_margin;
		}
	k = gkrellm_create_krell(p, im, style);
	gkrellm_monotonic_krell_values(k, FALSE);

	/* Unlike the style pointer passed to gkrellm_panel_configure(), the krells
	|  don't need the style to persist.
	*/
	g_free(style);
	return k;
	}

static void
create_panel(GtkWidget *vbox, gint first_create)
	{
	GdkImlibImage	*im = NULL;
	Panel			*p;
	gint			x;

	p = chart->panel;
	create_panel_buttons(p);
	x = sound->vu_x0;

	gkrellm_load_image("krell_vu", krell_vu_xpm, &im, STYLE_NAME);
	sound->krell_left = default_or_themed_krell(p, "sound.vu_left",
				im, 3, 1, 59, KRELL_EXPAND_LEFT, x, 0);
	sound->krell_right = default_or_themed_krell(p, "sound.vu_right",
				im, 9, 1, 59, KRELL_EXPAND_LEFT, x, 0);

	gkrellm_load_image("krell_vu_peak", krell_vu_peak_xpm, &im, STYLE_NAME);
	sound->krell_left_peak = default_or_themed_krell(p, "sound.vu_left_peak",
				im, 2, 5, -1, KRELL_EXPAND_NONE, x, 0);
	sound->krell_right_peak = default_or_themed_krell(p, "sound.vu_right_peak",
				im, 8, 5, -1, KRELL_EXPAND_NONE, x, 0);

	sound_vertical_scaling();

	gkrellm_load_image("krell_sensitivity", krell_sensitivity_xpm,
				&im, STYLE_NAME);
	sound->krell_sensitivity = default_or_themed_krell(p, "sound.sensitivity",
				im, 0, 1, -1, KRELL_EXPAND_NONE, x, 0);
	gkrellm_set_krell_full_scale(sound->krell_sensitivity, 100, 1);

	sound->krell_sensitivity_y_up = -10;
	sound->krell_sensitivity_y_dn = 0;
	gkrellm_get_gkrellmrc_integer("sound_krell_sensitivity_y_up",
			&sound->krell_sensitivity_y_up);
	gkrellm_get_gkrellmrc_integer("sound_krell_sensitivity_y_dn",
			&sound->krell_sensitivity_y_dn);
	sound->krell_sensitivity_y_target = sound->krell_sensitivity_y_up;

	if (im)
		gdk_imlib_kill_image(im);

	gkrellm_panel_configure(p, NULL, gkrellm_meter_style(style_id));
	gkrellm_panel_create(vbox, mon_sound, p);

	gkrellm_make_decal_button(p, decal_sweep_dn, cb_sweep_pressed,
				GINT_TO_POINTER(0), 1, 0);
	gkrellm_make_decal_button(p, decal_sweep_up, cb_sweep_pressed,
				GINT_TO_POINTER(1), 1, 0);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"expose_event", (GtkSignalFunc) expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_release_event", (GtkSignalFunc) cb_panel_release,NULL);
		gtk_signal_connect(GTK_OBJECT (p->drawing_area),
				"motion_notify_event", (GtkSignalFunc) cb_panel_motion, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"enter_notify_event", (GtkSignalFunc) cb_panel_enter, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"leave_notify_event", (GtkSignalFunc) cb_panel_leave, NULL);
		}
	gkrellm_update_krell(chart->panel, sound->krell_sensitivity,
				(gulong) (100.0 * (1.0 - sound->vert_sensitivity)));
	sound->x_sensitivity_raw = (gint) ((1.0 - sound->vert_sensitivity)
				* (gfloat) vu_meter_width) + sound->vu_x0;

	/* To get the negative y_up, must move the krell since the create
	|  interprets negative offsets as special placement codes.
	*/
	gkrellm_move_krell_yoff(p, sound->krell_sensitivity,
				sound->krell_sensitivity_y_up);
	}

static void
create_sound(GtkWidget *vbox, gint first_create)
	{
	if (first_create)
		{
		chart = gkrellm_chart_new0();
		chart->panel = gkrellm_panel_new0();
		sound_open_stream(sound->host);
		}
	sound->gc = gkrellm_draw_GC(1);
	create_chart(vbox, first_create);
	create_panel(vbox, first_create);
	}


static void
save_config(FILE *f)
	{
	fprintf(f, "%s mode %d\n", CONFIG_KEYWORD, sound->mode);
	fprintf(f, "%s sensitivity %f\n", CONFIG_KEYWORD, sound->vert_sensitivity);
	fprintf(f, "%s extra_info %d\n", CONFIG_KEYWORD, sound->extra_info);
	fprintf(f, "%s usec_per_div %d\n", CONFIG_KEYWORD, oscope->usec_per_div);
	fprintf(f, "%s spectrum_scale %d\n", CONFIG_KEYWORD,spectrum->scale_index);
	gkrellm_save_chartconfig(f, chart_config, CONFIG_KEYWORD, NULL);
	}

static void
load_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n != 2)
		return;
	if (!strcmp(config, "mode"))
		sscanf(item, "%d", &sound->mode);
	else if (!strcmp(config, "sensitivity"))
		{
		sscanf(item, "%f", &sound->vert_sensitivity);
		if (sound->vert_sensitivity < 0.05)
			sound->vert_sensitivity = 0.05;
		if (sound->vert_sensitivity > 1.0)
			sound->vert_sensitivity = 1.0;
		}
	else if (!strcmp(config, "extra_info"))
		sscanf(item, "%d", &sound->extra_info);
	else if (!strcmp(config, "usec_per_div"))
		sscanf(item, "%d", &oscope->usec_per_div);
	else if (!strcmp(config, "spectrum_scale"))
		sscanf(item, "%d", &spectrum->scale_index);
	else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
		gkrellm_load_chartconfig(&chart_config, item, 0);
	}

/* --------------------------------------------------------------------- */
static void
apply_config(void)
	{
	}


static gchar	*info_text[] =
{
N_("<b>GKrellMSS - GKrellM Sound Scope\n"),
"\n",
N_("<b>VU Meter\n"),
N_("Two fast response krells show left and right audio signal levels\n"
	"averaged over about 1/20 sec.  There are also two slow decay peak\n"
	"response krells.\n"),
"\n",

N_("<b>Chart\n"),
N_("The chart has oscilloscope and spectrum analyzer modes toggled by\n"
	"pressing the mode button on the chart.\n"),
"\n",

N_("<b>Panel Buttons\n"),
N_("In oscilloscope mode, the two buttons to the left of the VU Meter select\n"
	"an oscope horizontal sweep speed ranging from 100 microseconds (usec)\n"
	"per division to 50 miliseconds (msec) per division.  There are 5\n"
	"horizontal divisions, so a trace sweep time can range from 500 usec\n"
	"(1/2000 sec) to 250 msec (1/4 sec).\n"),
"\n",
N_("In spectrum analyzer mode, the two buttons select a frequency resolution\n"
	"and range.  As the '>' button is pressed, there is better lower\n"
	"frequency resolution at the expense of slower update times.\n"),
"\n",
N_("<b>\nMouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("click and drag in the VU Meter krell region to adjust VU Meter.\n"
	"\t\tand chart sensitivity.\n"),
N_("<b>\tLeft "),
N_("on the chart in spectrum analyzer mode toggles a frequency\n"
	"\t\tto draw highlighted.\n"),

N_("<b>\tWheel "),
N_("button anywhere in the panel also adjusts sensitivity.\n"),
N_("<b>\tMiddle "),
N_("on the chart toggles display of chart labels.\n"),
N_("<b>\tRight "),
N_("on the panel opens this configuration window.\n")

};

static void
create_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs;
	GtkWidget	*vbox;
	GtkWidget	*text, *label;
	gchar		*buf;
	gint		i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

#if 0
/* --Options Tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));
#endif


/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(info_text[i]));

/* --About tab */
	vbox = gkrellm_create_framed_tab(tabs, _("About"));
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

	buf = g_strdup_printf(_("GKrellMSS %d.%d\nGKrellM Sound Scope\n\n"
				"Copyright (c) 2002 by Bill Wilson\n"
				"bill@gkrellm.net\n"
				"http://gkrellm.net\n\n"
				"Released under the GNU Public License"),
				GKRELLMSS_VERSION_MAJOR, GKRELLMSS_VERSION_MINOR);
	label = gtk_label_new(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	g_free(buf);

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
	}


static Monitor	monitor_sound =
	{
	N_(CONFIG_NAME),	/* Name, for config tab.	*/
	0,					/* Id,  0 if a plugin		*/
	create_sound,		/* The create function		*/
	update_sound,		/* The update function		*/
	create_tab,			/* The config tab create function	*/
	apply_config,		/* Apply the config function		*/

	save_config,		/* Save user conifg			*/
	load_config,		/* Load user config			*/
	CONFIG_KEYWORD,		/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	MON_APM,			/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_plugin(void)
	{
	monitor_sound.name = _(monitor_sound.name);
	mon_sound = &monitor_sound;

	sound = g_new0(SoundMonitor, 1);
	sound->x_sensitivity_raw = gkrellm_chart_width() / 2;
	sound->extra_info = TRUE;
	sound->vert_sensitivity = 0.5;

	style_id = gkrellm_add_meter_style(mon_sound, STYLE_NAME);

	init_oscope();
	init_spectrum();

	return &monitor_sound;
	}

/* check sensitivity yoff
   left_margin == 0 => use vu_y0
*/
