/* GKrellM Volume plugin
 |  Copyright (C) 1999-2000 Sjoerd Simons
 |
 |  Author:  Sjoerd Simons  sjoerd@luon.net
 |
 |  This program is free software which I release under the GNU General Public
 |  License. You may redistribute and/or modify this program under the terms
 |  of that license as published by the Free Software Foundation; either
 |  version 2 of the License, or (at your option) any later version.
 |
 |  This program is distributed in the hope that it will be useful,
 |  but WITHOUT ANY WARRANTY; without even the implied warranty of
 |  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 |  GNU General Public License for more details.
 |
 |  To get a copy of the GNU General Puplic License,  write to the
 |  Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>
#include <glob.h>

#include <sys/param.h>
#if defined(__FreeBSD__) && __FreeBSD_version < 500000
  #include <machine/soundcard.h>
#else
  #include <sys/soundcard.h>
#endif

#include "mixer.h"
#include "common_mixer.h"

/* tries to open a mixer device, returns NULL on error or otherwise an mixer_t
 * struct */
mixer_t *mixer_open(char *id) {
  mixer_t *result;
  int fd,devices,nr,i;
#ifdef SOUND_MIXER_INFO
  mixer_info minfo;
#endif
  char *sound_labels[] = SOUND_DEVICE_LABELS;

  if ((fd = open(id,O_RDWR)) == -1) return NULL;
  if ( (ioctl(fd,SOUND_MIXER_READ_DEVMASK,&devices) < 0)) {
    close(fd);
    return NULL;
  }

#ifdef SOUND_MIXER_INFO
  if ((ioctl(fd,SOUND_MIXER_INFO,&minfo) < 0)) {
    close(fd);
    return NULL;
  }
#endif

  result = malloc(sizeof(mixer_t));
  result->fd = fd;
#ifdef SOUND_MIXER_INFO
  result->name = strdup(minfo.name);
#else
  result->name = strdup(id);
#endif

  /* get the nr of devices */
  nr = 0;
  for (i = 0 ; i < SOUND_MIXER_NRDEVICES; i++) { if (devices & (1<<i)) nr++; }

  result->nrdevices = nr;

  result->table     = malloc(nr * sizeof(int));
  result->dev_rnames = malloc(nr * sizeof(char *));

  result->dev_names = malloc(nr * sizeof(char*));
  memset(result->dev_names,0,nr * sizeof(char *));


  nr = 0;
  for (i = 0 ; i < SOUND_MIXER_NRDEVICES; i++)
    if (devices & (1<<i)) {
      result->table[nr] = i;
      result->dev_rnames[nr] = strdup(sound_labels[i]);
      nr++;
    }
  return result;
}

void mixer_close(mixer_t *mixer) {
  int i;
  close(mixer->fd);
  for (i=0;i < mixer->nrdevices ; i++) {
    free(mixer->dev_names[i]);
    free(mixer->dev_rnames[i]);
  }
  free(mixer->dev_names);
  free(mixer->dev_rnames);
  free(mixer->table);
  free(mixer);
}

/* Returns a pointer to the name of the mixer */
/* Shouldn't be freed */
char *mixer_get_name(mixer_t *mixer) {
  return mixer->name;
}
/* Returns the number of devices a mixer has */
int   mixer_get_nr_devices(mixer_t *mixer) {
  return mixer->nrdevices;
}

/* devid is the number of a device in a mixer */
/* 0 <= devid < mixer_get_nr_devices(mixer) */
/* get the real name of a device */
char *mixer_get_device_real_name(mixer_t *mixer,int devid) {
  return mixer->dev_rnames[devid];
}
/* get and set the user specified name of a device */
char *mixer_get_device_name(mixer_t *mixer,int devid) {
  return mixer->dev_names[devid] != NULL ? 
                            mixer->dev_names[devid] : mixer->dev_rnames[devid];
}

void  mixer_set_device_name(mixer_t *mixer,int devid,char *name) {
  free(mixer->dev_names[devid]);
  mixer->dev_names[devid] = strdup(name);
}

/* get the full scale of a device and get/set the volume */
long   mixer_get_device_fullscale(mixer_t *mixer,int devid) {
  return 100;
}

void mixer_get_device_volume(mixer_t *mixer, int devid,int *left,int *right) {
  long amount;
  ioctl(mixer->fd,MIXER_READ(mixer->table[devid]),&amount);
  *left = amount & 0xff;
  *right = amount >> 8;
}

void  mixer_set_device_volume(mixer_t *mixer, int devid,int left,int right) {
  long amount = (right << 8) + (left & 0xff);
  ioctl(mixer->fd,MIXER_WRITE(mixer->table[devid]),&amount);
}

mixer_idz_t *mixer_get_id_list(void) {
  mixer_idz_t *result = NULL;
  char *device[] = { "/dev/mixer*","/dev/sound/mixer*"};
  glob_t pglob;
  int i,n;

  for (n = 0; n < (sizeof(device)/sizeof(char *)); n++) {
    if (glob(device[n],0,NULL,&pglob) == 0) {
      for (i = 0; i < pglob.gl_pathc; i++) { 
        char *rc,buffer[PATH_MAX];
        rc = realpath(pglob.gl_pathv[i],buffer);

        if (rc == NULL) continue;
        result = mixer_id_list_add(rc,result); 
      }
      globfree(&pglob);
    }
  }
  return result;
}
