/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

  /* For grid images of height 2 pixels, make room at bottom of chartdata
  |  window so both pixel lines will show.
  */
#define GRID_HEIGHT_Y_OFFSET_ADJUST(cp) \
		(((cp)->bg_grid_image->rgb_height < 2) ? 0 : 1)

  /* Map internal y values with origin at lower left to X screen coordinates
  |  which have origin at upper left.
  */
#define Y_SCREEN_MAP(cp,y)	((cp)->h - (y) - 1)


#define	MIN_CHARTHEIGHT	5
#define	MAX_CHARTHEIGHT	200

static GList	*chart_list;

// #define DEBUG1
// #define DEBUG2
// #define DEBUG3

/* ----------------------------------------------------------------------- */

static gint
computed_index(Chart *cp, gint i)
	{
	gint	x;
	
	x = (cp->position + i + 1) % cp->w;
	return x;
	}

GList *
gkrellm_get_chart_list()
	{
	return chart_list;
	}

gint
gkrellm_get_chart_scalemax(Chart *cp)
	{
	return cp->scale_max;
	}

gint
gkrellm_get_current_chartdata(ChartData *cd)
	{
	return cd->data[cd->chart->position];
	}

gint
gkrellm_get_chartdata_data(ChartData *cd, gint index)
	{
	gint	x;

	if (!cd)
		return 0;
	x = computed_index(cd->chart, index);
	return cd->data[x];
	}

void
gkrellm_clear_chart(Chart *cp)
	{
	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_src_pixmap,
						0, 0,	0, 0,   cp->w, cp->h);
	if (cp->drawing_area->window)
		gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC, cp->pixmap,
						0, 0,	0, 0,   cp->w, cp->h);
	}

void
gkrellm_clear_chart_pixmap(Chart *cp)
	{
	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_src_pixmap,
						0, 0,	0, 0,   cp->w, cp->h);
	}

void
gkrellm_clean_bg_src_pixmap(Chart *cp)
	{
	if (!cp)
		return;
	if (!draw_rootpixmap_onto_transparent_chart(cp))
		gdk_draw_pixmap(cp->bg_src_pixmap, GK.draw1_GC,
				cp->bg_clean_pixmap, 0, 0, 0, 0, cp->w, cp->h);
	}

void
gkrellm_draw_chart_grid_line(Chart *cp, GdkPixmap *pixmap, gint y)
	{
	if (!cp)
		return;
	gdk_draw_pixmap(pixmap, GK.draw1_GC,
				cp->bg_grid_pixmap, 0, 0, cp->x, y, cp->w,
				((GdkWindowPrivate *) cp->bg_grid_pixmap)->height);
	}

void
gkrellm_draw_chart_to_screen(Chart *cp)
	{
	/* Draw the expose pixmap onto the screen.
	*/
	if (cp && cp->drawing_area->window)
		gdk_draw_pixmap(cp->drawing_area->window, GK.draw1_GC, cp->pixmap,
						0, 0,	0, 0,   cp->w, cp->h);
	}

static void
default_draw_chart_function(Chart *cp)
	{
	gkrellm_draw_chartdata(cp);
	gkrellm_draw_chart_to_screen(cp);
	}

void
gkrellm_set_draw_chart_function(Chart *cp, void (*func)(), gpointer data)
	{
	cp->draw_chart = func;
	cp->draw_chart_data = data;
	}

void
gkrellm_scale_chartdata(ChartData *cd, gint multiplier, gint divisor)
	{
	gint	i;

	if (!cd || !cd->data || divisor < 1)
		return;
	for (i = 0; i < cd->chart->w; ++i)
		cd->data[i] = cd->data[i] * multiplier / divisor;
	cd->previous = cd->previous * multiplier / divisor;
	}

void
gkrellm_reset_chart(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;
	gint		i;

	cp->scale_max = 0;
	cp->maxval = 0;
	cp->redraw_all = TRUE;
	cp->position = cp->w - 1;
	cp->primed = FALSE;

	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		cd->maxval = 0;
		cd->previous = 0;
		if (cd->data)
			for (i = 0; i < cp->w; ++i)
				cd->data[i] = 0;
		}
	}

void
gkrellm_reset_and_draw_chart(Chart *cp)
	{
	if (!cp)
		return;
	gkrellm_reset_chart(cp);
	if (cp->draw_chart)
		(*(cp->draw_chart))(cp->draw_chart_data);
	}

void
gkrellm_monotonic_chartdata(ChartData *cd, gboolean value)
	{
	if (cd)
		cd->monotonic = value;
	}

void
gkrellm_set_chartdata_draw_style(ChartData *cd, gint dstyle)
	{
	if (cd)
		cd->draw_style = dstyle;
	}

void
gkrellm_set_chartdata_draw_style_default(ChartData *cd, gint dstyle)
	{
	if (cd && cd->chart->config && !cd->chart->config->config_loaded)
		cd->draw_style = dstyle;
	}

void
gkrellm_set_chartdata_flags(ChartData *cd, gint flags)
	{
	if (cd)
		cd->flags = flags;
	}

static gint
chartdata_ycoord(Chart *cp, ChartData *cd, gint yd)
	{
	glong	y;
	
	if (cp->scale_max <= 0)
		cp->scale_max = 1;
	y = ((glong) yd * cd->h / cp->scale_max);
	if (y < 0)
		y = 0;
	if (y >= cd->h)
		y = cd->h - 1;
	if (cd->inverted)
		y = cd->h - y - 1;
	y += cd->y;
	return Y_SCREEN_MAP(cp, y);
	}

static void
draw_layer_grid_lines(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;
	gint		y, y0, grid_res, lines;
	gint		active_split, current_split;
	gboolean	do_next_split, done_once_per_split, tmp;

	gdk_draw_pixmap(cp->bg_pixmap, GK.draw1_GC,
				cp->bg_src_pixmap, 0, 0,  0, 0,  cp->w, cp->h);
	do_next_split = TRUE;
	for (active_split = 0; do_next_split; ++active_split)
		{
		do_next_split = FALSE;
		done_once_per_split = FALSE;
		current_split = 0;
		for (list = cp->cd_list; list; list = list->next)
			{
			cd = (ChartData *) list->data;
			if (cd->hide)
				continue;
			current_split += cd->split_chart;
			if (active_split != current_split)
				{
				if (current_split > active_split)
					do_next_split = TRUE;
				continue;
				}
			gdk_draw_pixmap(cd->layer.pixmap, GK.draw1_GC,
					*(cd->layer.src_pixmap), 0, 0,  0, 0,  cp->w, cp->h);

			grid_res = cp->config->grid_resolution;
			lines = cp->scale_max / grid_res;
			if (lines && cd->h / lines > 2)	/* No grids if h is too small */
				{
				for (y = 0; y <= cp->scale_max; y += grid_res)
					{
					if (   GK.bg_grid_mode == GRID_MODE_RESTRAINED
						&& ((y == 0 && cp->y == 0) || y == cp->scale_max)
					   )
						continue;
					tmp = cd->inverted;	  /* Draw grid lines in one direction*/
					cd->inverted = FALSE; /* else, may not line up by 1 pixel*/
					y0 = chartdata_ycoord(cp, cd, y);
					cd->inverted = tmp;
					gdk_draw_pixmap(cd->layer.pixmap, GK.draw1_GC,
						cd->layer.grid_pixmap, 0, 0, cp->x, y0, cp->w,
						((GdkWindowPrivate *) cd->layer.grid_pixmap)->height);

					if (!done_once_per_split)
						gdk_draw_pixmap(cp->bg_pixmap, GK.draw1_GC,
							cp->bg_grid_pixmap, 0, 0, cp->x, y0, cp->w,
							((GdkWindowPrivate *) cp->bg_grid_pixmap)->height);
					}
				}
			if (current_split > 0 && !done_once_per_split)
				{
				y = cd->y - 1;		/* Get separator y value */
				y -= GRID_HEIGHT_Y_OFFSET_ADJUST(cp);
				if (y >= 0)
					{
					gdk_draw_pixmap(cp->bg_pixmap, GK.draw1_GC,
							GK.bg_separator_pixmap,
							0, 0, cp->x, Y_SCREEN_MAP(cp, y),
							cp->w, GK.bg_separator_height);
					}
				}
			done_once_per_split = TRUE;
			}
		}
	}

  /* Return TRUE as long as there is a next split with impulse data needing
  |  to be drawn.
  */
static gboolean
draw_chartdata_impulses(Chart *cp, GList *cd_list,
			gint i0, gint active_split)
	{
	GList		*list;
	ChartData	*cd;
	gint		n, x, y, y0, y1, yN, yI;
	gint		current_split;
	gboolean	need_next_split = FALSE;

	if (!cd_list)
		return FALSE;
	for (n = i0; n < cp->w; ++n)
		{
		x = computed_index(cp, n);
		y0 = y1 = -1;
		yN = yI = 0;
		current_split = 0;
		for (list = cp->cd_list; list; list= list->next)
			{
			cd = (ChartData *) list->data;
			if (cd->hide)
				continue;
			current_split += cd->split_chart;
			if (   cd->draw_style != CHARTDATA_IMPULSE
				|| current_split != active_split
			   )
				{
				if (   current_split > active_split
					&& cd->draw_style == CHARTDATA_IMPULSE
				   )
					need_next_split = TRUE;
				continue;
				}
			if (cd->inverted)
				{
				if (y1 < 0)
					y1 = chartdata_ycoord(cp, cd, 0);
				yI += cd->data[x];
				y = chartdata_ycoord(cp, cd, yI);
				if (cd->data[x] > 0)
					gdk_draw_line(cd->data_bitmap, GK.bit1_GC, n, y1, n, y);
				y1 = y;
				}
			else
				{
				if (y0 < 0)
					y0 = chartdata_ycoord(cp, cd, 0);
				yN += cd->data[x];
				y = chartdata_ycoord(cp, cd, yN);
				if (cd->data[x] > 0)
					gdk_draw_line(cd->data_bitmap, GK.bit1_GC, n, y0, n, y);
				y0 = y;
				}
			}
		}
	/* Push the grided pixmaps through the data bitmaps onto the expose pixmap
	*/
	current_split = 0;
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->hide)
			continue;
		current_split += cd->split_chart;
		if (cd->draw_style == CHARTDATA_LINE || current_split != active_split)
			continue;
		if (cd->maxval > 0)
			{
			y0 = chartdata_ycoord(cp, cd, 0);
			y1 = chartdata_ycoord(cp, cd, cd->maxval);
			gdk_gc_set_clip_mask(GK.draw1_GC, cd->data_bitmap);
			if (cd->inverted)
				gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cd->layer.pixmap,
						0, y0,  0, y0,  cp->w, y1 - y0 + 1);
			else
				gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cd->layer.pixmap,
						0, y1,  0, y1,  cp->w, y0 - y1 + 1);
			}
		}
	return need_next_split;
	}

static gint
fix_y(gint yn, gint yp, gint ypp)
	{
	gint	y;

	if (yp > ypp && yn < yp)
		{
		y = (yp + yn) / 2;
		if (y < ypp)
			y = ypp;
		}
	else if (yp < ypp && yn > yp)
		{
		y = (yp + yn) / 2;
		if (y > ypp)
			y = ypp;
		}
	else
		y = yp;
	return y;
	}

static void
draw_chartdata_lines(Chart *cp, ChartData *cd, gint i0)
	{
	gint		n, x, y, y0, y1;

	y0 = chartdata_ycoord(cp, cd, 0);
	for (n = i0; n < cp->w; ++n)
		{
		x = computed_index(cp, n);
		y1 = chartdata_ycoord(cp, cd, cd->data[x]);
		if (n == 0)
			cd->y_p = y1;
		y = fix_y(y1, cd->y_p, cd->y_pp);
		cd->y_pp = y;
		cd->y_p = y1;
		if (cd->data[x] > 0 || (cd->inverted ? (y > y0) : (y < y0)))
			{
			if (y == y1)
				gdk_draw_point(cd->data_bitmap, GK.bit1_GC, cp->x + n, y1);
			else
				gdk_draw_line(cd->data_bitmap, GK.bit1_GC,
						cp->x + n, y, cp->x + n, y1);
			}
		}
	/* Push the grided pixmap through the data bitmap onto the expose pixmap
	*/
	if (cd->maxval > 0)
		{
		y0 = chartdata_ycoord(cp, cd, 0);
		y1 = chartdata_ycoord(cp, cd, cd->maxval);
		gdk_gc_set_clip_mask(GK.draw1_GC, cd->data_bitmap);
		if (cd->inverted)
			gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cd->layer.pixmap,
					0, y0,  0, y0,  cp->w, y1 - y0 + 1);
		else
			gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cd->layer.pixmap,
					0, y1,  0, y1,  cp->w, y0 - y1 + 1);
		}
	}

  /* Keep a maxval_auto_base that is 1/5 (1/FULL_SCALE_GRIDS) the peak maxval.
  |  * If using a fixed grid, auto resolution means set the resolution/grid to
  |    the smallest value in base to peak range (rounded up in resolution
  |    table) that draws data without clipping.
  |  * If using auto grids, auto resolution means set resolution to
  |    maxval_auto_base (rounded down in resolution table) so drawing peak
  |    valued data requires at least 5 (FULL_SCALE_GRIDS) grids.  Rounding
  |    down gives some tolerance to spurious data spikes which can set the
  |    base resolution too high.
  */
static void
set_auto_grid_resolution(Chart *cp, gint maxval)
	{
	static void	set_grid_resolution_spin_button(Chart *, gint);
	ChartConfig	*cf = cp->config;
	gint		grids, grid_res, maxval_base;

	if (maxval <= cp->maxval_auto_base)
		maxval = cp->maxval_auto_base;
	else
		{
		if (maxval > cp->maxval_peak)
			cp->maxval_peak = maxval;
		maxval_base = maxval / FULL_SCALE_GRIDS;
		if (maxval_base > cp->maxval_auto_base)
			cp->maxval_auto_base = maxval_base;
		}
	cp->maxval_auto = maxval;

	grids = cf->fixed_grids;
	if (!grids)		/* Auto grids mode */
		grid_res = gkrellm_125_sequence(cp->maxval_auto_base, cf->sequence_125,
					cf->low, cf->high, TRUE, FALSE);
	else
		{
		if (cf->auto_resolution_stick)
			maxval = cp->maxval_peak;
		grid_res = gkrellm_125_sequence(maxval / grids, cf->sequence_125,
					cf->low, cf->high, TRUE, TRUE);
		}
	if (grid_res != cf->grid_resolution)
		{
		cf->grid_resolution = grid_res;
		set_grid_resolution_spin_button(cp, grid_res);
		if (cf->cb_grid_resolution)
			(*cf->cb_grid_resolution)(cf, cf->cb_grid_resolution_data);
		cp->redraw_all = TRUE;
		}
	}

static gint
setup_chart_scalemax(Chart *cp)
	{
	ChartConfig	*cf = cp->config;
	glong		scalemax;
	gint		grid_res, i0;

	/* maxval may change at any gkrellm_store_chartdata(), so at each chart
	|  draw compute a scalemax and compare to last cp->scale_max.
	|  Redraw grided background if different.
	*/
	if (   cf->auto_grid_resolution
		&& cp->maxval != cp->maxval_auto
		&& (   cp->maxval > cp->maxval_auto
			|| cp->maxval_auto != cp->maxval_auto_base
		   )
	   )
		set_auto_grid_resolution(cp, cp->maxval);
	grid_res = cf->grid_resolution;
	if (cf->fixed_grids)
		scalemax = grid_res * cf->fixed_grids;
	else	/* Auto scale to cp->maxval */
		{
		if (cp->maxval == 0)
			scalemax = grid_res;
		else
			scalemax = ((cp->maxval - 1) / grid_res + 1) * grid_res;
		}
	if (scalemax != cp->scale_max || cp->redraw_all)
		{
		cp->redraw_all = FALSE;
		i0 = 0;				/* Will draw all data on chart */
		cp->scale_max = scalemax;
		draw_layer_grid_lines(cp);
		}
	else
		i0 = cp->w - 1;		/* Will draw the last data point only */
	return i0;
	}

void
gkrellm_draw_chartdata(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;
	gint		i0, active_split, current_split;
	gboolean	have_impulse_splits = FALSE;

	if (!cp)
		return;
	i0 = setup_chart_scalemax(cp);
	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_pixmap,
						0, 0,	0, 0,   cp->w, cp->h);

	current_split = active_split = 0;
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->hide)
			continue;
		current_split += cd->split_chart;
		if (!have_impulse_splits && cd->draw_style == CHARTDATA_IMPULSE)
			{
			have_impulse_splits = TRUE;
			active_split = current_split;
			}
		/* Clear the area of the data bitmaps that data will be drawn into
		*/
		gdk_draw_rectangle(cd->data_bitmap, GK.bit0_GC, TRUE,
					i0, 0, cd->w - i0, cp->h);
		}

	for (  ; have_impulse_splits; ++active_split)
		have_impulse_splits = draw_chartdata_impulses(cp, cp->cd_list,
					i0, active_split);

	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->draw_style == CHARTDATA_LINE && !cd->hide)
			draw_chartdata_lines(cp, cd, i0);
		}
	gdk_gc_set_clip_mask(GK.draw1_GC, NULL);
	}

void
gkrellm_alloc_chartdata(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;
	size_t		w;

	if (!cp)
		return;
	w = (size_t) cp->w;
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->w == w && cd->data)
			continue;
		cd->w = w;
		if (cd->data)
			g_free(cd->data);
		cd->data = (gint *) g_new0(gint, w);
		cd->maxval = 0;
		cp->position = cp->w - 1;
		cp->tail = cp->position;
		}
	cp->alloc_width = w;
	cp->maxval = 0;
	cp->scale_max = 0;
	cp->redraw_all = TRUE;
	}

static void
scroll_chartdata_bitmaps(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;

	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->hide)
			continue;
		gdk_draw_pixmap(cd->data_bitmap, GK.bit1_GC, cd->data_bitmap,
						1, 0, 0, 0, cp->w - 1, cp->h);
		}
	}

static gboolean
scan_for_impulse_maxval(Chart *cp, gint active_split)
	{
	GList		*list;
	ChartData	*cd;
	gint		N, I;
	gint		i, current_split;
	gboolean	need_next_split = FALSE;

	for (i = 0; i < cp->w; ++i)
		{
		/* N is normal and I inverted cumulative impulse data/split
		*/
		N = I = 0;
		current_split = 0;
		for (list = cp->cd_list; list; list = list->next)
			{
			cd = (ChartData *) list->data;
			if (cd->hide)
				continue;
			current_split += cd->split_chart;
			if (   cd->draw_style != CHARTDATA_IMPULSE
				|| current_split != active_split
			   )
				{
				if (   current_split > active_split
					&& cd->draw_style == CHARTDATA_IMPULSE
				   )
					need_next_split = TRUE;
				continue;
				}
			if (cd->inverted)
				{
				I += cd->data[i];
				if (I > cd->maxval)
					cd->maxval = I;
				}
			else
				{
				N += cd->data[i];
				if (N > cd->maxval)
					cd->maxval = N;
				}
			if (N + I > cp->maxval)
				cp->maxval = N + I;
			}
		}
	return need_next_split;
	}

static void
scan_for_maxval(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;
	gint		i, current_split, active_split;
	gboolean	have_impulse_splits = FALSE;

	cp->maxval = 0;
	current_split = 0;
	active_split = 0;
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->hide)
			continue;
		cd->maxval = 0;
		current_split += cd->split_chart;
		if (cd->draw_style == CHARTDATA_LINE)
			for (i = 0; i < cp->w; ++i)
				{
				if (cd->data[i] > cd->maxval)
				    cd->maxval = cd->data[i];
				if (cd->maxval > cp->maxval)
					cp->maxval = cd->maxval;
				}
		if (!have_impulse_splits && cd->draw_style == CHARTDATA_IMPULSE)
			{
			have_impulse_splits = TRUE;
			active_split = current_split;
			}
		}
	for ( ; have_impulse_splits; ++active_split)
		have_impulse_splits = scan_for_impulse_maxval(cp, active_split);
	}

void
gkrellm_store_chartdata(Chart *cp, gulong total, ...)
	{
	va_list		args;
	GList		*list;
	ChartData	*cd;
	gulong		range, total_diff;
	gint		n, N_discard, I_discard, N, I;
	gint		active_split, current_split;
	gboolean	need_scan = FALSE;

	if (!cp)
		return;
	va_start(args, total);
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		cd->current = va_arg(args, gulong);
		if (!cd->monotonic)
			{
			cd->previous = 0;
			cp->previous_total = 0;		/* All or none if total is used */
			}
		/* Prime the pump.  Also handle data wrap around or reset to zero.
		*/
		if (cd->current < cd->previous || !cp->primed)
			cd->previous = cd->current;
		}
	va_end(args);
	if (total < cp->previous_total || !cp->primed)
		cp->previous_total = total;	  /* Wrap around, this store won't scale */
	total_diff = total - cp->previous_total;
	cp->previous_total = total;
		
	/* Increment position in circular buffer and remember the data
	|  value to be thrown out.
	*/
	cp->position = (cp->position + 1) % cp->w;
	cp->tail = (cp->tail + 1) % cp->w;
	n = cp->position;
	active_split = current_split = 0;
	N_discard = I_discard = 0;

	/* N is normal and I inverted cumulative impulse data/split
	*/
	N = I = 0;
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		cd->discard = cd->data[cp->tail];
		cd->data[n] = (gint)(cd->current - cd->previous);
		cd->previous = cd->current;

		/* If using totals, scale the stored data to range between 0 and the
		|  max chart value.  Max chart value is number of grids * grid res.
		|  No. of grids is 5 in auto grid mode.  For plotting data as a %.
		*/
		if (total_diff > 0)
			{
			range = (cp->config->fixed_grids ? cp->config->fixed_grids
							: FULL_SCALE_GRIDS) * cp->config->grid_resolution;
			if (range != total_diff)
				cd->data[n] = cd->data[n] * range / total_diff;
			}
		if (cd->hide || need_scan)
			continue;

		/* Compare discarded data to new data (accounting for stacked impulse
		|  data) and decide if a new maxval must be set (new data > maxval)
		|  or if a complete rescan of the data is needed to find a new
		|  maxval (a discard > a new).
		*/
		current_split += cd->split_chart;
		if (cd->draw_style == CHARTDATA_IMPULSE)
			{
			if (current_split != active_split)
				{
				active_split = current_split;
				N_discard = I_discard = 0;
				N = I = 0;
				}
			if (cd->inverted)
				{
				I_discard += cd->discard;
				I += cd->data[n];
				if (I_discard && I_discard >= cd->maxval)
					need_scan = TRUE;
				else if (I > cd->maxval)
					cd->maxval = I;
				}
			else
				{
				N_discard += cd->discard;
				N += cd->data[n];
				if (N_discard && N_discard >= cd->maxval)
					need_scan = TRUE;
				else if (N > cd->maxval)
					cd->maxval = N;
				}
			if (N + I > cp->maxval)
				cp->maxval = N + I;
			}
		else if (cd->draw_style == CHARTDATA_LINE)
			{
			if (cd->discard && cd->discard >= cd->maxval)
				need_scan = TRUE;
			else
				{
				if (cd->data[n] > cd->maxval)
					cd->maxval = cd->data[n];
				if (cd->maxval > cp->maxval)
					cp->maxval = cd->maxval;
				}
			}
		}
	cp->primed = TRUE;
	if (need_scan || cp->redraw_all)
		scan_for_maxval(cp);
	scroll_chartdata_bitmaps(cp);
	}



/* =================================================================== */

static gint
chartdata_text_y(Chart *cp, char key, gint ascent)
	{
	GList		*list;
	ChartData	*cd;
	gint		n, y;

	n = key - '0';
	y = -1;
	if (n >= 0)
		{
		list = g_list_nth(cp->cd_list, n / 2);
		if (list)
			{
			cd = (ChartData *) list->data;
			if (!cd->hide)
				{
				if (n & 1)
					y = cd->y + cd->h - 3 - ascent;
				else
					y = cd->y + 1;
				y = Y_SCREEN_MAP(cp, y);
				}
			}
		}
	return y;
	}

void
gkrellm_draw_chart_text(Chart *cp, gint style_id, gchar *str)
	{
	TextStyle	*ts, *ts_default, *ts_alt;
	gchar		c, *s, *t;
	gint		text_length, ascent, descent, field_width, fw;
	gint		xx, x, y, w, l, r, a, d, center;
	gboolean	right, set_fw, fw_right;

	if (!cp || !str)
		return;
	ts_default = gkrellm_chart_textstyle(style_id);
	ts_alt = gkrellm_chart_alt_textstyle(style_id);

	w = l = r = a = d = 0;
	x = xx = 2;
	gdk_text_extents(ts_default->font, _("Ay8"), 3, &l, &r, &w,
					&ascent, &descent);
	y = 2 + ascent;
	fw = 0;
	for (s = str; *s; s += text_length)
		{
		c = '\0';
		center = 0;
		right = FALSE;
		set_fw = FALSE;
		fw_right = FALSE;
		ts = ts_default;
		field_width = 0;
		while (*s == '\\')
			{
			if ((c = *(++s)) != '\0')
				++s;
			if (c == 'n')
				{
				y += ascent + descent;
				x = xx;
				}
			else if (c == 'c')
				center = 1;
			else if (c == 'C')
				center = 2;
			else if (c == 'N')
				{
				x = xx;		/* A conditional newline.  If previous string */
				if (a > 2)	/* was spaces, no nl.  A space has nonzero a  */
					y += ascent + descent;
				}
			else if (c == 'b')
				{
				y = cp->h - 3;
				x = xx;
				}
			else if (c == 't')
				{
				y = 2 + ascent;
				x = xx;
				}
			else if (c == 'r')
				right = TRUE;
			else if (c == 'p')
				{
				y -= ascent + descent;
				x = xx;
				}
			else if (c == 'w')
				set_fw = TRUE;
			else if (c == 'a')
				field_width = fw;
			else if (c == 'e')
				{
				field_width = fw;
				fw_right = TRUE;
				}
			else if (c == 'f')
				ts = ts_alt;
			else if (c == 'x' && isdigit(*s))
				xx = *s++ - '0';
			else if (c == 'y' && isdigit(*s))
				y = *s++ - '0';
			else if (c == 'D')
				{
				y = chartdata_text_y(cp, *s++, ascent);
				x = xx;
				}
			}
		t = strchr(s, (gint) '\\');
		if (t)
			text_length = t - s;
		else
			text_length = strlen(s);

		if (y < 0)	/* asked for a chartdata that is hidden */
			continue;

		gdk_text_extents(ts->font, s, text_length, &l, &r, &w, &a, &d);

		if (set_fw)
			{
			fw = r + ts->effect;
			continue;
			}
		if (center == 1)
			x = (cp->w - r) / 2;
		else if (center == 2)
			x = cp->w / 2;
		else if (fw_right)
			x = x + fw - r - ts->effect;
		else if (right)
			x = cp->w - r - 2 - ts->effect;
		if (text_length > 1 || (text_length == 1 && *s != ' '))
			{
			gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_pixmap,
					x - l, y - a - 1, x - l, y - a - 1,
					l + r + ts->effect + 1, a + d + ts->effect + 1);
    		gkrellm_draw_text(cp->pixmap, ts, x, y, s, text_length);
			}
		if (field_width && !fw_right)
			x += (field_width > r + l) ? field_width : r + l;
		else
			x += r + l + ts->effect;
		}
	}

gint
gkrellm_draw_chart_label(Chart *cp, TextStyle *ts, gint x, gint y, gchar *s)
	{
	gint		w, l, r, a, d;

	if (!cp || !ts || !s)
		return x;
	gdk_string_extents(ts->font, s, &l, &r, &w, &a, &d);

	gdk_draw_pixmap(cp->pixmap, GK.draw1_GC, cp->bg_pixmap,
		x - l, y - a, x - l, y - a, l + r + ts->effect, a + d + ts->effect);
    gkrellm_draw_string(cp->pixmap, ts, x, y, s);

	return x + r;
	}

void
gkrellm_destroy_chartdata_list(GList **cd_list_head)
	{
	GList		*list;
	ChartData	*cd;

	if (!cd_list_head)
		return;
	for (list = *cd_list_head; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->label)
			g_free(cd->label);
		if (cd->data)
			g_free(cd->data);
		if (cd->data_bitmap)
			gdk_pixmap_unref(cd->data_bitmap);
		if (cd->layer.pixmap)
			gdk_pixmap_unref(cd->layer.pixmap);
		/* cd->layer.src_pixmap & cd->layer.grid_pixmap must be handled by
		|  creating monitor.
		*/
		}
	free_glist_and_data(cd_list_head);
	}


static void
free_chart_pixmaps(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;

	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		gdk_pixmap_unref(cd->data_bitmap);
		gdk_pixmap_unref(cd->layer.pixmap);
		cd->data_bitmap = NULL;
		cd->layer.pixmap = NULL;
		/* cd->layer.src_pixmap & cd->layer.grid_pixmap must be handled by
		|  creating monitor.
		*/
		}
	/* If new theme or size, the cd_list will not be destroyed so I can
	|  reuse the data arrays.   Pixmaps will be recreated.
	*/
	cp->cd_list_index = 0;

	gdk_imlib_free_pixmap(cp->bg_pixmap);
	gdk_imlib_free_pixmap(cp->bg_src_pixmap);
	gdk_imlib_free_pixmap(cp->bg_grid_pixmap);

	gdk_imlib_free_pixmap(cp->bg_clean_pixmap);
	/* Imlib says masks are automatically destroyed */

	gdk_imlib_free_pixmap(cp->pixmap);
	}

static void
destroy_chart_data(Chart *cp)
	{
	GList		*list;
	ChartData	*cd;

	free_chart_pixmaps(cp);
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->label)
			g_free(cd->label);
		if (cd->data)
			g_free(cd->data);
		cd->label = NULL;
		cd->data = NULL;
		}
	/* Don't free the cd_list.  It is in the ChartConfig struct.
	*/
	}

  /* Destroy everything inside a chart except leave cp->config alone since
  |  the config is managed by each monitor.
  */
void
gkrellm_chart_destroy(Chart *cp)
	{
	if (!cp)
		return;
	gkrellm_freeze_side_frame_packing();
	if (cp->panel)
		gkrellm_panel_destroy(cp->panel);
	destroy_chart_data(cp);
	gtk_signal_handlers_destroy(GTK_OBJECT(cp->drawing_area));
	gtk_widget_destroy(cp->hbox);
	chart_list = g_list_remove(chart_list, cp);
	gkrellm_chartconfig_window_destroy(cp);
	if (cp->shown)
		gkrellm_monitor_height_adjust(- cp->h);
	g_free(cp);
	gkrellm_thaw_side_frame_packing();
	}

void
gkrellm_chartconfig_destroy(ChartConfig **cf)
	{
	if (!cf || !*cf)
		return;
	gkrellm_destroy_chartdata_list((*cf)->chart_cd_list);
	g_free(*cf);
	*cf = NULL;
	}

void
gkrellm_chart_bg_image_override(Chart *cp, GdkImlibImage *bg_image,
			GdkImlibImage *bg_grid_image)
	{
	if (!cp || !bg_image || !bg_grid_image)
		return;
	cp->bg_image = bg_image;
	cp->bg_grid_image = bg_grid_image;
	cp->bg_image_override = TRUE;
	}

static void
set_chartdata_split_heights(Chart *cp)
	{
	GList		*list, *list1;
	ChartData	*cd, *cd1;
	gint		splits;
	gint		i, y0, h_avail, h_free, y_offset;

	for (i = 0, splits = 0, list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->hide)
			continue;
		if (++i > 1 && cd->split_chart)	/* Can't split before the first one */
			++splits;
		cd->split_share = 1.0;
		for (list1 = list->next; list1; list1 = list1->next)
			{
			cd1 = (ChartData *) list1->data;
			if (!cd1->split_chart || cd1->hide)
				continue;
			cd->split_share = cd1->split_fraction;
			break;
			}
		}
	y_offset = GRID_HEIGHT_Y_OFFSET_ADJUST(cp);
	y0 = cp->y + y_offset;
	h_avail = cp->h - cp->y - ((splits + 1) * y_offset)
				- splits * GK.bg_separator_height;
	h_free = h_avail;
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if (cd->hide)
			continue;
		cd->y = y0;
		for (list1 = list->next; list1; list1 = list1->next)
			if (!((ChartData *) list1->data)->hide)
				break;
		if (!list1)
			cd->h = h_free;
		else
			cd->h = (gint) (cd->split_share * (gfloat) h_free);
		if (cd->h < 1)
			cd->h = 1;
		if (list1 && ((ChartData *) list1->data)->split_chart)
			{
			y0 += cd->h + GK.bg_separator_height + y_offset;
			h_free -= cd->h;
			}
		}
	}

ChartData *
gkrellm_add_default_chartdata(Chart *cp, gchar *label)
	{
	GdkPixmap	**src_pixmap, *grid_pixmap;

	if (!cp)
		return NULL;
	if (cp->cd_list_index & 1)
		{
		src_pixmap = &GK.data_in_pixmap;
		grid_pixmap = GK.data_in_grid_pixmap;
		}
	else
		{
		src_pixmap = &GK.data_out_pixmap;
		grid_pixmap = GK.data_out_grid_pixmap;
		}
	return gkrellm_add_chartdata(cp, src_pixmap, grid_pixmap, label);
	}

  /* Need a GdkPixmap ** because the src pixmap can change (re-rendered at
  |  size or theme changes).
  */
ChartData *
gkrellm_add_chartdata(Chart *cp, GdkPixmap **src_pixmap,
			GdkPixmap *grid_pixmap, gchar *label)
	{
	GtkWidget	*top_win	= gkrellm_get_top_window();
	GList		*list;
	ChartData	*cd;

	if (!cp || !src_pixmap || !grid_pixmap || !label)
		return NULL;

	/* To handle theme and vert size changes without loosing data, reuse the
	|  ChartData structs in the cd_list.
	*/
	list = g_list_nth(cp->cd_list, cp->cd_list_index++);
	if (!list)
		{
		cd = g_new0(ChartData, 1);
		cp->cd_list = g_list_append(cp->cd_list, cd);
		cp->config->cd_list = cp->cd_list;
		cd->split_fraction = 0.5;
		}
	else
		cd = (ChartData *) list->data;
	cd->chart = cp;
	gkrellm_dup_string(&cd->label, label);
	cd->monotonic = TRUE;
	cd->data_bitmap = gdk_pixmap_new(top_win->window, cp->w, cp->h, 1);
	cd->layer.pixmap = gdk_pixmap_new(top_win->window, cp->w, cp->h, -1);
	cd->layer.src_pixmap = src_pixmap;
	cd->layer.grid_pixmap = grid_pixmap;

	set_chartdata_split_heights(cp);
	return cd;
	}

void
gkrellm_render_data_grid_pixmap(GdkImlibImage *im, GdkPixmap **pixmap,
			GdkColor *color)
	{
	GtkWidget   *top_win = gkrellm_get_top_window();
	gint		w, h = 0;

	w = gkrellm_chart_width();
	if (!*pixmap || w != ((GdkWindowPrivate *) (*pixmap))->width)
		{
		if (im)
			{
			if ((h = im->rgb_height) > 2)
				h = 2;
			gkrellm_render_to_pixmap(im, pixmap, NULL, w, h);
			}
		else
			{
			if (*pixmap)
				gdk_imlib_free_pixmap(*pixmap);
			*pixmap = gdk_pixmap_new(top_win->window, w, 1, -1);
			if (color)
				gdk_gc_set_foreground(GK.draw1_GC, color);
			else
				gdk_gc_set_foreground(GK.draw1_GC, &GK.in_color_grid);
			gdk_draw_rectangle(*pixmap, GK.draw1_GC, TRUE, 0, 0, w, 1);
			}
		}
	}

void
gkrellm_render_data_pixmap(GdkImlibImage *im, GdkPixmap **pixmap,
			GdkColor *color, gint h)
	{
	GtkWidget   *top_win = gkrellm_get_top_window();
	gint		w;

	w = gkrellm_chart_width();
	if (   !*pixmap
		|| w != ((GdkWindowPrivate *) (*pixmap))->width
		|| h != ((GdkWindowPrivate *) (*pixmap))->height
	   )
		{
		if (!gkrellm_render_to_pixmap(im, pixmap, NULL, w, h))
			{
			*pixmap = gdk_pixmap_new(top_win->window, w, h, -1);
			if (color)
				gdk_gc_set_foreground(GK.draw1_GC, color);
			else
				gdk_gc_set_foreground(GK.draw1_GC, &GK.in_color_grid);
			gdk_draw_rectangle(*pixmap, GK.draw1_GC, TRUE, 0,0,w,h);
			}
		}
	}

static void
render_default_data_pixmaps(Chart *cp)
	{
	GList		*list;
	GdkPixmap	*pixmap;
	gint		w, h;

	gkrellm_render_data_grid_pixmap(GK.data_in_grid_image,
				&GK.data_in_grid_pixmap, &GK.in_color_grid);
	gkrellm_render_data_grid_pixmap(GK.data_out_grid_image,
				&GK.data_out_grid_pixmap, &GK.out_color_grid);

	w = gkrellm_chart_width();
	pixmap = GK.bg_separator_pixmap;
	if (   !pixmap
		|| ((GdkWindowPrivate *) pixmap)->width != w
	   )
		{
		if ((h = GK.bg_separator_height) < 1 || h > 5)
			h = 2;
		if (GK.bg_separator_image)
			gkrellm_render_to_pixmap(GK.bg_separator_image,
						&GK.bg_separator_pixmap, NULL, w, h);
		else
			{
			GdkImlibImage	*im;

			im = gkrellm_bg_panel_image(DEFAULT_STYLE_ID);
			gkrellm_render_to_pixmap(im, &GK.bg_separator_pixmap, NULL, w, h);
			}
		}

	h = 2;
	for (list = chart_list; list; list = list->next)
		{
		cp = (Chart *) list->data;
		if (cp->h > h)
			h = cp->h;
		}
	gkrellm_render_data_pixmap(GK.data_in_image,
				&GK.data_in_pixmap, &GK.in_color, h);
	gkrellm_render_data_pixmap(GK.data_out_image,
				&GK.data_out_pixmap, &GK.out_color, h);
	}

void
setup_charts(void)
	{
	gkrellm_free_pixmap(&GK.data_in_pixmap);
	gkrellm_free_pixmap(&GK.data_in_grid_pixmap);
	gkrellm_free_pixmap(&GK.data_out_pixmap);
	gkrellm_free_pixmap(&GK.data_out_grid_pixmap);
	gkrellm_free_pixmap(&GK.bg_separator_pixmap);
	}

#if 0
static gint
compare_chartlist(gconstpointer a, gconstpointer b)
    {
	Chart	*cp_a	= (Chart *) a;
	Chart	*cp_b	= (Chart *) b;
	gint	result;

	if (cp_a->style_id < cp_b->style_id)
		result = -1;
	else (if cp_a->style_id > cp_b->style_id)
		result = 1;
	else
		result = 0;
	return result;
    }
#endif

static void
insert_in_chartlist(Chart *cp)
	{
	GList	*list;
	Chart	*cp_x;
	gint	position = 0;

	for (list = chart_list; list; list = list->next, ++position)
		{
		cp_x = (Chart *) list->data;
		if (cp_x->style_id > cp->style_id)
			{
			chart_list = g_list_insert(chart_list, cp, position);
			return;
			}
		}
	chart_list = g_list_append(chart_list, cp);
	}

void
gkrellm_chart_hide(Chart *cp, gboolean hide_panel)
	{
	if (!cp || !cp->shown)
		return;
	gtk_widget_hide(cp->hbox);
	gkrellm_freeze_side_frame_packing();
	if (hide_panel)
		gkrellm_panel_hide(cp->panel);
	gkrellm_monitor_height_adjust(- cp->h);
	cp->shown = FALSE;
	gkrellm_thaw_side_frame_packing();
	}

void
gkrellm_chart_show(Chart *cp, gboolean show_panel)
	{
	if (!cp || cp->shown)
		return;
	gtk_widget_show(cp->hbox);
	gkrellm_freeze_side_frame_packing();
	if (show_panel)
		gkrellm_panel_show(cp->panel);
	cp->shown = TRUE;
	gkrellm_monitor_height_adjust(cp->h);
	gkrellm_thaw_side_frame_packing();
	}

gboolean
gkrellm_is_chart_visible(Chart *cp)
	{
	if (!cp)
		return FALSE;
	return cp->shown;
	}

gboolean
gkrellm_chart_enable_visibility(Chart *cp, gboolean new_vis,
					gboolean *current_vis)
	{
	gboolean	changed = FALSE;

	if (new_vis  && ! *current_vis)
		{
		gkrellm_chart_show(cp, TRUE);
		*current_vis  = TRUE;
		changed = TRUE;
		}
	if (!new_vis  && *current_vis)
		{
		gkrellm_chart_hide(cp, TRUE);
		*current_vis  = FALSE;
		changed = TRUE;
		}
	return changed;
	}

void
gkrellm_set_chart_height_default(Chart *cp, gint h)
	{
	if (!cp || (cp->config && cp->config->config_loaded))
		return;

	if (h < MIN_CHARTHEIGHT)
		h = MIN_CHARTHEIGHT;
	if (h > MAX_CHARTHEIGHT)
		h = MAX_CHARTHEIGHT;
	cp->h = h;
	}

void
gkrellm_chart_create(GtkWidget *vbox, Monitor *mon, Chart *cp,
			ChartConfig **config)
	{
	GtkWidget		*hbox;
	ChartConfig		*cf;
	Style			*style;
	gint			h, style_id;

	if (!vbox || !mon || !cp || !config)
		return;
	if (mon->privat->style_type == CHART_PANEL_TYPE)
		style_id = mon->privat->style_id;
	else
		style_id = DEFAULT_STYLE_ID;
	cp->monitor = (gpointer) mon;
	if (!cp->bg_image_override)
		{
		cp->bg_image = gkrellm_bg_chart_image(style_id);
		cp->bg_grid_image = gkrellm_bg_grid_image(style_id);
		}
	cp->bg_image_override = FALSE;
	cp->x = 0;
/*	cp->y = 0; */
	cp->w = GK.chart_width;
	if (!*config)
		{
		*config = gkrellm_chartconfig_new0();
		(*config)->auto_grid_resolution = TRUE;		/* the default */
		(*config)->h = cp->h;					/* In case default */
		}
	cf = cp->config = *config;
	if (cf->h < 5)
		cf->h = 40;
	cp->h = cf->h;
	if (cf->grid_resolution < 1)
		cf->grid_resolution = 1;
	cp->cd_list = cp->config->cd_list;
	cp->config->chart_cd_list = &cp->cd_list;

	if (!cp->hbox)
		{
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_add (GTK_CONTAINER(vbox), hbox);
		cp->hbox = hbox;
		cp->drawing_area = gtk_drawing_area_new();
		gtk_widget_set_events(cp->drawing_area, GDK_EXPOSURE_MASK
				| GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK
				| GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK
				| GDK_POINTER_MOTION_MASK);
		gtk_box_pack_start(GTK_BOX(hbox), cp->drawing_area, FALSE, FALSE, 0);
		gtk_widget_show(cp->drawing_area);
		gtk_widget_show(hbox);
		cp->shown = TRUE;
		gtk_widget_realize(hbox);
		gtk_widget_realize(cp->drawing_area);
		cp->style_id = style_id;
		insert_in_chartlist(cp);
		}
	else
		free_chart_pixmaps(cp);

	gtk_drawing_area_size(GTK_DRAWING_AREA (cp->drawing_area), cp->w, cp->h);

	gdk_imlib_render(cp->bg_image, cp->w, cp->h);
	cp->bg_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_src_pixmap = gdk_imlib_copy_image(cp->bg_image);
	if ((h = cp->bg_grid_image->rgb_height) > 2)
		h = 2;
	gdk_imlib_render(cp->bg_grid_image, cp->w, h);
	cp->bg_grid_pixmap = gdk_imlib_copy_image(cp->bg_grid_image);

	/* In case the background layer has transparency (which must be
	|  handled when gkrellm moves), I must have an extra "clean"
	|  background pixmap (and mask).
	*/
	cp->bg_clean_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_mask = gdk_imlib_copy_mask(cp->bg_image);

	/* And finally the expose pixmap
	*/
	cp->pixmap = gdk_imlib_copy_image(cp->bg_image);

	style = gkrellm_chart_style(style_id);
	cp->transparency = style->transparency;
	GK.any_transparency |= cp->transparency;

	gkrellm_set_draw_chart_function(cp, default_draw_chart_function, cp);
	cp->redraw_all = TRUE;
	render_default_data_pixmaps(cp);
	if (cp->shown)
		gkrellm_monitor_height_adjust(cp->h);
	}

void
gkrellm_refresh_chart(Chart *cp)
	{
	if (!cp)
		return;
	cp->redraw_all = TRUE;
	cp->maxval_auto = -1;
	if (cp->draw_chart)
		(*(cp->draw_chart))(cp->draw_chart_data);
	}

void
gkrellm_rescale_chart(Chart *cp)
	{
	if (!cp)
		return;
	scan_for_maxval(cp);
	gkrellm_refresh_chart(cp);
	}

/* =================================================================== */


static gint		map_125_table[] =
	{
	1, 2, 5,
	10, 20, 50,
	100, 200, 500,
	1000, 2000, 5000,
	10000, 20000, 50000,
	100000, 200000, 500000,
	1000000, 2000000, 5000000,
	10000000, 20000000, 50000000,
	100000000, 200000000, 500000000
	};

static gint		map_12357_table[] =
	{
	1, 2, 3, 5, 7,
	10, 15, 20, 30, 50, 70,
	100, 150, 200, 300, 500, 700,
	1000, 1500, 2000, 3000, 5000, 7000,
	10000, 15000, 20000, 30000, 50000, 70000,
	100000, 150000, 200000, 300000, 500000, 700000,
	1000000, 1500000, 2000000, 3000000, 5000000, 7000000,
	10000000, 15000000, 20000000, 30000000, 50000000, 70000000,
	100000000, 150000000, 200000000, 300000000, 500000000, 700000000
	};

gint
gkrellm_125_sequence(gint value, gboolean use125,
			gint low, gint high,
			gboolean snap_to_table, gboolean roundup)
	{
	gint    i, table_size;
	gint	*table;

	if (use125)
		{
		table = map_125_table;
		table_size = sizeof(map_125_table) / sizeof(gint);
		}
	else
		{
		table = map_12357_table;
		table_size = sizeof(map_12357_table) / sizeof(gint);
		}
	if (value < low)
		value = low;
	if (value > high)
		value = high;
	if (value < table[0])
		return table[0];
	if (value > table[table_size - 1])
		return table[table_size - 1];
	for (i = 0; i < table_size; ++i)
		{
/*printf("  mapping[%d] value=%d table=%d\n", i, value, table[i]); */
		if (value == table[i])
			return table[i];
		else if (value == table[i] - 1)
			return table[i - 1];
		else if (value == table[i] + 1)
			return table[i + 1];
		}
	if (snap_to_table && !roundup)
		{
		for (i = table_size - 1; i >= 0; --i)
			{
			if (value > table[i])
				{
				value = table[i];
				break;
				}
			}
		}
	if (snap_to_table && roundup)
		{
		for (i = 0; i < table_size; ++i)
			{
			if (value < table[i])
				{
				value = table[i];
				break;
				}
			}
		}
	return value;
	}

static void
set_grid_resolution_spin_button(Chart *cp, gint res)
	{
	GtkSpinButton	*spin;

	if (!cp || !cp->config_window || !cp->config->grid_resolution_spin_button)
		return;
	spin = GTK_SPIN_BUTTON(cp->config->grid_resolution_spin_button);
	gtk_spin_button_set_value(spin, (gfloat) res);	
	}

static void
cb_chart_grid_resolution(GtkWidget *adjustment, Chart *cp)
	{
	GtkSpinButton	*spin;
	ChartConfig		*cf;
	gint			res;
	gfloat			fres;

	cf = cp->config;
	spin = GTK_SPIN_BUTTON(cf->grid_resolution_spin_button);
	if (cf->map_sequence)
		{
		res = gtk_spin_button_get_value_as_int(spin);
		if (res != cf->grid_resolution)
			{
			res = gkrellm_125_sequence(res, cf->sequence_125,
						cf->low, cf->high, FALSE, FALSE);
			cf->grid_resolution = res;
			gtk_spin_button_set_value(spin, (gfloat) res);
			if (cf->cb_grid_resolution)
				(*cf->cb_grid_resolution)(cf, cf->cb_grid_resolution_data);
			gkrellm_refresh_chart(cp);
			}
		}
	else
		{
		fres = gtk_spin_button_get_value_as_float(spin);
		if (cf->spin_factor > 0.0)
			fres *= cf->spin_factor;
		cf->grid_resolution = (gint) fres;
		if (cf->grid_resolution < 1)
			cf->grid_resolution = 1;
		if (cf->cb_grid_resolution)
			(*cf->cb_grid_resolution)(cf, cf->cb_grid_resolution_data);
		gkrellm_refresh_chart(cp);
		}
	}


/* ---- ChartConfig functions ---- */
void
gkrellm_set_chartconfig_grid_resolution(ChartConfig *cf, gint res)
	{
	if (!cf || res <= 0)
		return;
	cf->grid_resolution = res;
	}

gint
gkrellm_get_chartconfig_grid_resolution(ChartConfig *cf)
	{
	if (!cf)
		return 0;
	return cf->grid_resolution;
	}

void
gkrellm_chartconfig_grid_resolution_connect(ChartConfig *cf,
			void (*func)(gpointer), gpointer data)
	{
	if (!cf)
		return;
	cf->cb_grid_resolution = func;
	cf->cb_grid_resolution_data = data;
	}

void
gkrellm_set_chartconfig_flags(ChartConfig *cf, gint flags)
	{
	if (!cf)
		return;
	cf->flags = flags;
	}

void
gkrellm_chartconfig_grid_resolution_adjustment(ChartConfig *cf,
			gboolean map_sequence, gfloat spin_factor,
			gfloat low, gfloat high, gfloat step0, gfloat step1, gint digits,
			gint width)
	{
	if (!cf)
		return;
	cf->map_sequence = map_sequence;
	cf->width = width;
	if (map_sequence)
		{
		cf->low = 1;
		cf->low = (gfloat) gkrellm_125_sequence((gint) low, cf->sequence_125,
							low, high, TRUE, FALSE);
		cf->high = (gfloat) gkrellm_125_sequence((gint) high,
							cf->sequence_125, low, high, TRUE, TRUE);
		cf->step0 = 1.0;
		cf->step1 = 1.0;
		cf->digits = 0;
		}
	else
		{
		cf->low = low;
		cf->high = high;
		cf->step0 = step0;
		cf->step1 = step1;
		cf->digits = digits;
		cf->spin_factor = spin_factor;
		}
	if (cf->spin_factor < 1.0)
		cf->spin_factor = 1.0;
	cf->adjustment_is_set = TRUE;
	}

void
gkrellm_chartconfig_grid_resolution_label(ChartConfig *cf, gchar *label)
	{
	if (!cf)
		return;
	gkrellm_dup_string(&cf->grid_resolution_label, label);
	}

void
gkrellm_set_chartconfig_auto_grid_resolution(ChartConfig *cf, gboolean ato)
	{
	if (cf)
		cf->auto_grid_resolution = ato;
	}

void
gkrellm_set_chartconfig_auto_resolution_stick(ChartConfig *cf, gboolean stick)
	{
	if (cf)
		cf->auto_resolution_stick = stick;
	}

void
gkrellm_set_chartconfig_sequence_125(ChartConfig *cf, gboolean seq)
	{
	if (cf)
		cf->sequence_125 = seq;
	}

void
gkrellm_set_chartconfig_fixed_grids(ChartConfig *cf, gint fixed_grids)
	{
	if (!cf || fixed_grids < 0 || fixed_grids > 5)
		return;
	cf->fixed_grids = fixed_grids;
	}

gint
gkrellm_get_chartconfig_fixed_grids(ChartConfig *cf)
	{
	if (!cf)
		return 0;
	return cf->fixed_grids;
	}

void
gkrellm_chartconfig_fixed_grids_connect(ChartConfig *cf,
			void (*func)(gpointer), gpointer data)
	{
	if (!cf)
		return;
	cf->cb_fixed_grids = func;
	cf->cb_fixed_grids_data = data;
	}

gint
gkrellm_get_chartconfig_height(ChartConfig *cf)
	{
	if (!cf)
		return 0;
	return cf->h;
	}

void
gkrellm_chartconfig_height_connect(ChartConfig *cf,
			void (*func)(gpointer), gpointer data)
	{
	if (!cf)
		return;
	cf->cb_height = func;
	cf->cb_height_data = data;
	}

void
gkrellm_set_chart_height(Chart *cp, gint h)
	{
	GtkWidget   *top_win = gkrellm_get_top_window();
	GList		*list;
	ChartData	*cd;
	ChartConfig	*cf;
	gint		dy;

	if (!cp || cp->h == h)
		return;
	dy = h - cp->h;
	cp->h = h;
	cp->config->h = h;
	render_default_data_pixmaps(cp);
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		gdk_pixmap_unref(cd->data_bitmap);
		gdk_pixmap_unref(cd->layer.pixmap);
		cd->data_bitmap = gdk_pixmap_new(top_win->window, cp->w, cp->h, 1);
		cd->layer.pixmap = gdk_pixmap_new(top_win->window, cp->w, cp->h, -1);
		}
    gdk_imlib_free_pixmap(cp->bg_pixmap);
    gdk_imlib_free_pixmap(cp->bg_src_pixmap);
    gdk_imlib_free_pixmap(cp->bg_clean_pixmap);
    gdk_imlib_free_pixmap(cp->pixmap);

	gdk_imlib_render(cp->bg_image, cp->w, cp->h);
	cp->bg_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_src_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_clean_pixmap = gdk_imlib_copy_image(cp->bg_image);
	cp->bg_mask = gdk_imlib_copy_mask(cp->bg_image);
	cp->pixmap = gdk_imlib_copy_image(cp->bg_image);

	cf = cp->config;
	if (cf->cb_height)
		(*cf->cb_height)(cf, cf->cb_height_data);
	gtk_drawing_area_size(GTK_DRAWING_AREA(cp->drawing_area), cp->w, cp->h);
	set_chartdata_split_heights(cp);
	if (cp->shown)
		{
		gkrellm_monitor_height_adjust(dy);
		gkrellm_pack_side_frames();
		gkrellm_refresh_chart(cp);
		}
	}

gboolean
gkrellm_get_chartdata_hide(ChartData *cd)
	{
	if (cd && cd->hide)
		return TRUE;
	return FALSE;
	}


/* =================================================================== */

#define CONTROL_BRANCH		"/Control"
#define CONTROL_ITEM		"/Control/-"
#define RECALIBRATE_ITEM	"/Control/Auto mode recalibrate"
#define STICK_ITEM			"/Control/Auto mode sticks at peak value"
#define SEQUENCE_125_ITEM	"/Control/Sequence.../1 2 5"
#define SEQUENCE_12357_ITEM	"/Control/Sequence.../1 1.5 2 3 5 7"


static void
chart_config_window_close(GtkWidget *widget, Chart *cp)
	{
	if (cp->config_window)
		gtk_widget_destroy(cp->config_window);
	cp->config_window = NULL;
	}

void
gkrellm_chartconfig_window_destroy(Chart *cp)
	{
	chart_config_window_close(NULL, cp);
	}

static gint
chart_config_window_delete_event(GtkWidget *widget, GdkEvent *ev,
			gpointer data)
	{
	chart_config_window_close(widget, data);
	return FALSE;
	}

static void
set_resolution_menubar_items_sensitivity(ChartConfig *cf)
	{
	GtkWidget	*w;

	if (!cf->auto_resolution_item_factory)
		return;
	w = gtk_item_factory_get_widget(cf->auto_resolution_item_factory,
				_(STICK_ITEM));
	if (cf->fixed_grids && cf->auto_grid_resolution)
		{
		GTK_CHECK_MENU_ITEM(w)->active = cf->auto_resolution_stick;
		gtk_widget_set_sensitive(w, TRUE);
		}
	else	/* Auto number of grids, auto res always sticks at base value */
		{
		GTK_CHECK_MENU_ITEM(w)->active = cf->fixed_grids ?
				cf->auto_resolution_stick : FALSE;
		gtk_widget_set_sensitive(w, FALSE);
		}

	w = gtk_item_factory_get_widget(cf->auto_resolution_item_factory,
				_(RECALIBRATE_ITEM));
	if (cf->auto_grid_resolution)
		gtk_widget_set_sensitive(w, TRUE);
	else
		gtk_widget_set_sensitive(w, FALSE);
	}

static void
cb_chart_height(GtkWidget *adjustment, Chart *cp)
	{
	GtkSpinButton	*spin;
	gint			h;

	GK.config_modified = TRUE;
	spin = GTK_SPIN_BUTTON(cp->config->height_spin_button);
	h = gtk_spin_button_get_value_as_int(spin);
	gkrellm_set_chart_height(cp, h);
	}

static void
cb_chart_fixed_grids(GtkWidget *adjustment, Chart *cp)
	{
	GtkSpinButton	*spin;
	ChartConfig		*cf = cp->config;

	GK.config_modified = TRUE;
	spin = GTK_SPIN_BUTTON(cf->fixed_grids_spin_button);
	cf->fixed_grids = gtk_spin_button_get_value_as_int(spin);
	if (cf->auto_grid_resolution)
		set_auto_grid_resolution(cp, cp->maxval_auto);
	if (cf->cb_fixed_grids)
		(*cf->cb_fixed_grids)(cf, cf->cb_fixed_grids_data);

	set_resolution_menubar_items_sensitivity(cf);

	gkrellm_refresh_chart(cp);
	}

static void
cb_line_draw_style(GtkWidget *widget, ChartData *cd)
	{
	GK.config_modified = TRUE;
	cd->draw_style = GTK_TOGGLE_BUTTON(widget)->active;
	gkrellm_rescale_chart(cd->chart);
	}

static void
cb_auto_resolution(GtkWidget *widget, Chart *cp)
	{
	GtkWidget	*button;
	ChartConfig	*cf = cp->config;

	GK.config_modified = TRUE;
	cf->auto_grid_resolution = GTK_TOGGLE_BUTTON(widget)->active;

	set_resolution_menubar_items_sensitivity(cf);
	button = cf->grid_resolution_spin_button;
	if (cf->auto_grid_resolution)
		gtk_widget_set_sensitive(button, FALSE);
	else
		gtk_widget_set_sensitive(button, TRUE);
	gkrellm_rescale_chart(cp);
	}

static void
cb_inverted_draw_mode(GtkWidget *widget, ChartData *cd)
	{
	GK.config_modified = TRUE;
	cd->inverted = GTK_TOGGLE_BUTTON(widget)->active;
	gkrellm_rescale_chart(cd->chart);
	}

static void
cb_hide(GtkWidget *widget, ChartData *cd)
	{
	GK.config_modified = TRUE;
	cd->hide = GTK_TOGGLE_BUTTON(widget)->active;
	set_chartdata_split_heights(cd->chart);
	gkrellm_rescale_chart(cd->chart);
	}

static void
cb_split_mode(GtkWidget *widget, ChartData *cd)
	{
	GK.config_modified = TRUE;
	cd->split_chart = GTK_TOGGLE_BUTTON(widget)->active;
	gtk_widget_set_sensitive(cd->split_fraction_spin_button, cd->split_chart);
	set_chartdata_split_heights(cd->chart);
	gkrellm_rescale_chart(cd->chart);
	}

static void
cb_split_fraction(GtkWidget *adjustment, ChartData *cd)
	{
	GtkSpinButton	*spin;

	GK.config_modified = TRUE;
	spin = GTK_SPIN_BUTTON(cd->split_fraction_spin_button);
	cd->split_fraction = gtk_spin_button_get_value_as_float(spin);
	set_chartdata_split_heights(cd->chart);
	gkrellm_rescale_chart(cd->chart);
	}


/* =================================================================== */

static void
cb_auto_res_control(Chart *cp, guint option, GtkWidget* widget)
    {
    ChartConfig	*cf;
	gint		grid_res;
	gboolean	active;

	cf = cp->config;
    switch (option)
        {
        case 0:
			cp->maxval_auto_base = 0;
			cp->maxval_peak = 0;
            break;
        case 1:
            active = GTK_CHECK_MENU_ITEM(widget)->active;
			cf->auto_resolution_stick = active;
            break;
        case 2:
            active = GTK_CHECK_MENU_ITEM(widget)->active;
			if (cf->sequence_125 && active)
				return;
			cf->sequence_125 = active;
			break;
        case 3:
            active = GTK_CHECK_MENU_ITEM(widget)->active;
			if (!cf->sequence_125 && active)
				return;
			cf->sequence_125 = !active;

			grid_res = gkrellm_125_sequence(cf->grid_resolution,
						cf->sequence_125, cf->low, cf->high, TRUE, FALSE);
			cf->grid_resolution = grid_res;
			set_grid_resolution_spin_button(cp, grid_res);
            break;
        }
	gkrellm_refresh_chart(cp);
    }


static GtkItemFactoryEntry	auto_res_control_items[] =
    {
{N_(CONTROL_BRANCH),	NULL,	NULL,				 0,	"<LastBranch>" },
{N_(CONTROL_ITEM),		NULL,	NULL,				 0,	"<Separator>"},
{N_(RECALIBRATE_ITEM),	NULL,	cb_auto_res_control, 0,	"<Item>"},
{N_(STICK_ITEM),		NULL,	cb_auto_res_control, 1,	"<ToggleItem>"},
{N_(CONTROL_ITEM),		NULL,    NULL,				 0,	"<Separator>"},
{N_(SEQUENCE_125_ITEM),	NULL,	cb_auto_res_control, 2,	"<RadioItem>"},
{N_(SEQUENCE_12357_ITEM), NULL,	cb_auto_res_control, 3,	N_(SEQUENCE_125_ITEM)},
{N_(CONTROL_ITEM),		 NULL,	NULL,				 0,	"<Separator>"},
    };

static void
auto_resolution_control_menubar(GtkWidget **menubar, Chart *cp)
	{
	GtkItemFactory	*item_factory;
	ChartConfig		*cf = cp->config;
	gint			n;

	n = sizeof(auto_res_control_items) / sizeof(GtkItemFactoryEntry);
	item_factory = gtk_item_factory_new(GTK_TYPE_MENU_BAR, "<main>", NULL);
	gtk_item_factory_create_items(item_factory, n, auto_res_control_items, cp);
	cf->auto_resolution_item_factory = item_factory;
	set_resolution_menubar_items_sensitivity(cf);

	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(item_factory,
			_(SEQUENCE_125_ITEM)))->active = cf->sequence_125;
	GTK_CHECK_MENU_ITEM(gtk_item_factory_get_widget(item_factory,
			_(SEQUENCE_12357_ITEM)))->active = !cf->sequence_125;

	if (menubar)
		*menubar = gtk_item_factory_get_widget(item_factory, "<main>");
	}

void
gkrellm_chartconfig_window_create(Chart *cp)
	{
	GtkWidget	*main_vbox, *vbox, *vbox1, *vbox2, *hbox;
	GtkWidget	*button;
	GList		*list;
	ChartConfig	*cf;
	ChartData	*cd;
	Panel		*p;
	gchar		*s;

	if (!cp)
		return;
	if (cp->config_window)
		{
		gdk_window_raise(cp->config_window->window);
		return;
		}
	cp->config_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(cp->config_window), "delete_event",
			(GtkSignalFunc) chart_config_window_delete_event, cp);
	gtk_window_set_policy(GTK_WINDOW(cp->config_window), FALSE, FALSE, TRUE);

	p = cp->panel;
	cf = cp->config;
	if (p && p->label)
		s = p->label->string;
	else
		s = NULL;
	gtk_window_set_title(GTK_WINDOW(cp->config_window),
			_("GKrellM Chart Config"));

	main_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(cp->config_window), main_vbox);
	vbox = gkrellm_framed_vbox(main_vbox, s, 4, FALSE, 4, 3);

	hbox = gtk_hbox_new(TRUE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	for (list = cp->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		if ((cd->flags & CHARTDATA_NO_CONFIG) == CHARTDATA_NO_CONFIG)
			continue;
		vbox1 = gkrellm_framed_vbox(hbox, cd->label, 2, TRUE, 2, 2);

		if (!(cd->flags & CHARTDATA_NO_CONFIG_DRAW_STYLE))
			{
			gkrellm_check_button(vbox1, &button, cd->draw_style, FALSE, 0,
					_("Line style"));
			gtk_signal_connect(GTK_OBJECT(button), "toggled",
					GTK_SIGNAL_FUNC(cb_line_draw_style), cd);
			}
		if (!(cd->flags & CHARTDATA_NO_CONFIG_INVERTED))
			{
			gkrellm_check_button(vbox1, &button, cd->inverted, FALSE, 0,
					_("Inverted"));
			gtk_signal_connect(GTK_OBJECT(button), "toggled",
					GTK_SIGNAL_FUNC(cb_inverted_draw_mode), cd);
			}
		if (list != cp->cd_list && !(cd->flags & CHARTDATA_NO_CONFIG_SPLIT))
			{
			vbox2 = gkrellm_framed_vbox(vbox1, NULL, 2, FALSE, 2, 2);
			gkrellm_check_button(vbox2, &button, cd->split_chart, FALSE, 0,
					_("Split view"));
			gtk_signal_connect(GTK_OBJECT(button), "toggled",
					GTK_SIGNAL_FUNC(cb_split_mode), cd);
			gkrellm_spin_button(vbox2, &button, cd->split_fraction,
					0.05, 0.95, 0.01, 0.05, 2, 55,
					cb_split_fraction, cd, FALSE, "");
			gtk_widget_set_sensitive(button, cd->split_chart);
			cd->split_fraction_spin_button = button;
			}
		if (cd->flags & CHARTDATA_ALLOW_HIDE)
			{
			gkrellm_check_button(vbox1, &button, cd->hide, FALSE, 0,
					_("Hide"));
			gtk_signal_connect(GTK_OBJECT(button), "toggled",
					GTK_SIGNAL_FUNC(cb_hide), cd);
			}
		}

	cf->auto_resolution_control_menubar = NULL;
	cf->auto_resolution_item_factory = NULL;
	cf->grid_resolution_spin_button = NULL;
	cf->fixed_grids_spin_button = NULL;

	if (cf->adjustment_is_set)
		{
		gfloat	value;

		vbox1 = gkrellm_framed_vbox(vbox, _("Resolution per Grid"), 2, FALSE,
					2, 2);
		if (cf->map_sequence)
			value = (gfloat) cf->grid_resolution;
		else
			value = cf->grid_resolution / cf->spin_factor;
		gkrellm_spin_button(vbox1, &button, value,
			cf->low, cf->high, cf->step0, cf->step1, cf->digits, cf->width,
			cb_chart_grid_resolution, cp, FALSE, cf->grid_resolution_label);
		cf->grid_resolution_spin_button = button;
		if (cp->config->auto_grid_resolution)
			gtk_widget_set_sensitive(button, FALSE);

		if (!(cp->config->flags & NO_CONFIG_AUTO_GRID_RESOLUTION))
			{
			hbox = gtk_hbox_new (FALSE, 0);
			gtk_container_border_width(GTK_CONTAINER(hbox), 2);
			gtk_container_add(GTK_CONTAINER(vbox1), hbox);
			gkrellm_check_button(hbox, &button,
					cp->config->auto_grid_resolution, TRUE, 0,
					_("Auto"));
			gtk_signal_connect(GTK_OBJECT(button), "toggled",
					GTK_SIGNAL_FUNC(cb_auto_resolution), cp);

			auto_resolution_control_menubar(
					&cf->auto_resolution_control_menubar, cp);
			gtk_box_pack_start(GTK_BOX(hbox),
				cf->auto_resolution_control_menubar, FALSE, TRUE, 10);
			}
		}
	if (!(cp->config->flags & NO_CONFIG_FIXED_GRIDS))
		{
		vbox1 = gkrellm_framed_vbox(vbox, _("Number of Grids"), 2, FALSE,
				2, 2);
		gkrellm_spin_button(vbox1, &button, (gfloat) cf->fixed_grids,
				0, 5, 1.0, 1.0, 0, 50,
				cb_chart_fixed_grids, cp, FALSE,
				_("0: Auto    1-5: Constant"));
		cf->fixed_grids_spin_button = button;
		}

	vbox1 = gkrellm_framed_vbox(vbox, NULL, 2, FALSE, 2, 2);
	gkrellm_spin_button(vbox1, &button, (gfloat) cp->h,
			(gfloat) GK.chart_height_min, (gfloat) GK.chart_height_max,
			5.0, 10.0, 0, 50,
			cb_chart_height, cp, FALSE,
			_("Chart height"));
	cf->height_spin_button = button;

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);

	button = gtk_button_new_with_label(_("OK"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) chart_config_window_close, cp);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 15);
	gtk_widget_grab_default(button);

	button = gtk_button_new_with_label(_("Save Config"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) save_user_config, NULL);
    gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 15);
	if (GK.demo)
		gtk_widget_set_sensitive(button, FALSE);

	gtk_widget_show_all(cp->config_window);
	}


/* Example:
cpu chart_config cpu0 h grid_res grids : type inv split : type inv split
*/

void
gkrellm_save_chartconfig(FILE *f, ChartConfig *cf, gchar *mon_keyword,
			gchar *name)
	{
	GList		*list;
	ChartData	*cd;

	if (!f || !cf || !mon_keyword)
		return;
	if (name)
		fprintf(f, "%s %s %s ", mon_keyword, GKRELLM_CHARTCONFIG_KEYWORD,name);
	else
		fprintf(f, "%s %s ", mon_keyword, GKRELLM_CHARTCONFIG_KEYWORD);
	fprintf(f, "%d %d %d %d %d %d", cf->h, cf->grid_resolution,
				cf->fixed_grids, cf->auto_grid_resolution,
				cf->auto_resolution_stick, cf->sequence_125);
	for (list = cf->cd_list; list; list = list->next)
		{
		cd = (ChartData *) list->data;
		fprintf(f, " : %d %d %d %d %.2f",
				cd->draw_style, cd->inverted, cd->hide,
				cd->split_chart, cd->split_fraction);
		}
	fprintf(f, "\n");
	}

void
gkrellm_load_chartconfig(ChartConfig **config, gchar *string, gint max_cd)
	{
	GList		*list;
	ChartData	*cd;
	ChartConfig	*cf;
	gchar		*s;
	gint		index = 0;

	if (!config || !string)
		return;
	if (!*config)
		{
		*config = gkrellm_chartconfig_new0();
		(*config)->auto_grid_resolution = TRUE;		/* the default */
		}
	cf = *config;
	sscanf(string, "%d %d %d %d %d %d", &cf->h, &cf->grid_resolution,
				&cf->fixed_grids, &cf->auto_grid_resolution,
				&cf->auto_resolution_stick, &cf->sequence_125);
	for (s = strchr(string, (int) ':'); s ; s = strchr(s, (int) ':'))
		{
		++s;
		list = g_list_nth(cf->cd_list, index++);
		if (!list)
			{
			cd = g_new0(ChartData, 1);
			cd->split_fraction = 0.5;
			cf->cd_list = g_list_append(cf->cd_list, cd);
			}
		else
			cd = (ChartData *) list->data;
		sscanf(s, "%d %d %d %d %f",
				&cd->draw_style, &cd->inverted, &cd->hide,
				&cd->split_chart, &cd->split_fraction);
		if (cd->split_fraction <= 0.01 || cd->split_fraction >= 0.99)
			cd->split_fraction = 0.5;

		cf->config_loaded = TRUE;
		if (max_cd && index >= max_cd)
			break;
		}
	}

void
debug_dump_chart_list()
	{
	GList		*list, *cdlist;
	Chart		*cp;
	Panel		*p;
	ChartData	*cd;

	printf("\n");
	for (list = gkrellm_get_chart_list(); list; list = list->next)
		{
		cp = (Chart *) list->data;
		p = cp->panel;
		if (p && p->label && p->label->string)
			printf("%s [%d]: ", p->label->string, cp->style_id);
		else
			printf("(null) [%d]: ", cp->style_id);
		for (cdlist = cp->cd_list; cdlist; cdlist = cdlist->next)
			{
			cd = (ChartData *) cdlist->data;
			printf("%s %p->data ", cd->label, cd->data);
			}
		printf("\n");
		}
	}
