/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"

#include <pwd.h>
#include <errno.h>
#include <sys/wait.h>


/* ===============  string utility functions ================= */

gint
strcmp_basename(gchar *str1, gchar *str2)
	{
	gchar	*s1, *s2;

	if ((s1 = strrchr(str1, '/')) == NULL)
		s1 = str1;
	else
		++s1;
	if ((s2 = strrchr(str2, '/')) == NULL)
		s2 = str2;
	else
		++s2;
	return strcmp(s1, s2);
	}

  /* Obsolete */
gboolean
dup_string(gchar **dst, gchar *src)
	{
	return gkrellm_dup_string(dst, src);
	}

gboolean
gkrellm_dup_string(gchar **dst, gchar *src)
	{
	if (!*dst && !src)
		return FALSE;
	if (*dst)
		{
		if (src && !strcmp(*dst, src))
			return FALSE;
		g_free(*dst);
		}
	*dst = g_strdup(src);
	return TRUE;
	}

void
gkrellm_free_pixmap(GdkPixmap **pixmap)
	{
	if (!pixmap)
		return;
	if (*pixmap)
		gdk_imlib_free_pixmap(*pixmap);
	*pixmap = NULL;
	}

  /* Cut out an optionally quoted string.  This is destructive to the src.
  */
gchar *
cut_quoted_string(gchar *src, gchar **endptr)
	{
	gchar	*s;

	while (*src == ' ' || *src == '\t')
		++src;
	if (*src == '"')
		{
		s = strchr(++src, '"');
		if (s == NULL)
			{
			if (endptr)
				*endptr = src;
			printf(_("Unterminated quote\n"));
			return NULL;
			}
		*s = '\0';
		if (endptr)
			*endptr = s + 1;
		}
	else
		{
		for (s = src; *s != '\0' && *s != ' ' && *s != '\t'; ++s)
			;
		if (endptr)
			*endptr = *s ? s + 1 : s;
		*s = '\0';
		}
	return src;
	}

  /* If there is a line in the gstring ('\n' delimited) copy it to the
  |  line buffer including the newline and erase it from the gstring.
  */
gboolean
getline_from_gstring(GString **gstring, gchar *line, gint size)
	{
	GString	*gstr	= *gstring;
	gchar   *s;
	gint    len, n;

	if (gstr && gstr->str && (s = strchr(gstr->str, '\n')) != NULL)
		{
		n = len = s - gstr->str + 1;
		if (n >= size)
			n = size - 1;				/* Truncate the line to fit */
		strncpy(line, gstr->str, n);
		line[n] = '\0';
		*gstring = g_string_erase(gstr, 0, len);
		return TRUE;
		}
	return FALSE;
	}

/* ===============  list utility functions ================= */

void
free_glist_and_data(GList **list_head)
	{
	GList	*list;

	if (*list_head == NULL)
		return;

	/* could use g_list_foreach(*list_head, (G_FUNC)g_free, NULL);
	*/
	for (list = *list_head; list; list = list->next)
		if (list->data)
			g_free(list->data);
	g_list_free(*list_head);
	*list_head = NULL;
	}

GList *
basename_in_list(GList *list, gchar *name)
	{
	gchar	*s, *slash;

	for ( ; list; list = list->next)
		{
		s = (gchar *) list->data;
		if (!s)
			continue;
		if (*s == '.')
			++s;
		if ((slash = strrchr(s, '/')) != NULL)
			s = slash + 1;
		if (! strcmp(s, name))
			return list;
		}
	return NULL;
	}

GList *
string_in_list(GList *list, gchar *s)
	{
	for ( ; list; list = list->next)
		{
		if (strcmp(list->data, s) == 0)
			return list;
		}
	return NULL;
	}

#if 0
gint
name_in_clist(GtkWidget *clist, gchar *name)
	{
	gchar	*s;
	gint	row;

	for (row = 0; row < GTK_CLIST(clist)->rows; ++row)
		{
		gtk_clist_get_text(GTK_CLIST(clist), row, 0, &s);
		if (!strcmp(s, name))
			return TRUE;
		}
	return FALSE;
	}
#endif



/* ===============  file utility functions ================= */
  /* obsolete */
gchar *
homedir(void)
	{
	return gkrellm_homedir();
	}

gchar *
gkrellm_homedir(void)
	{
	static gchar	*homedir;
	struct passwd	*pw;

	if (   homedir == NULL && (homedir = getenv("HOME")) == NULL)
		{
		if ((pw = getpwuid(getuid())))
			homedir = pw->pw_dir;
		}
	return homedir;
	}

gboolean
isdir(gchar *dir, gchar *subdir)
	{
	struct stat st;
	gchar		*s;
	gint		result	= FALSE;

	s = g_strconcat(dir, "/", subdir, NULL);
	if (stat(s, &st) == 0 && S_ISDIR(st.st_mode))
		result = TRUE;
	g_free(s);
	return result;
	}

gboolean
isfile(gchar *s)
	{
	struct stat st;

	if (stat(s, &st) == 0 && S_ISREG(st.st_mode))
		return TRUE;
	return FALSE;
	}


gboolean
make_home_subdir(gchar *subdir)
	{
	gchar	*dir;
	gint	result	= FALSE;

	dir = g_strdup_printf("%s/%s", gkrellm_homedir(), subdir);
	if (!isdir(dir, NULL))
		{
		if (mkdir(dir, 0755) < 0)
			printf(_("Cannot create directory: %s\n"), dir);
		else
			result = TRUE;
		}
	g_free(dir);
	return result;
	}

void
dup_home_pathname(gchar **path, gchar *subdir, gchar *fname)
	{
	if (*path)
		g_free(*path);
	if (subdir)
		*path = g_strdup_printf("%s/%s/%s", gkrellm_homedir(), subdir, fname);
	else
		*path = g_strdup_printf("%s/%s", gkrellm_homedir(), fname);
	}


/* ===============  GtkWidget utility functions ================= */

  /* obsolete */
gchar *
entry_get_alpha_text(GtkWidget **entry)
	{
	return gkrellm_entry_get_text(entry);
	}

gchar *
gkrellm_entry_get_text(GtkWidget **entry)
	{
	gchar	*s = "";

	if (*entry)
		s = gtk_entry_get_text(GTK_ENTRY(*entry));
	while (*s == ' ' || *s == '\t')
		++s;
	return s;
	}


/* ===============  Miscellaneous utility functions ================= */

  /* Print a size, abbreviating it to kilo, mega, or giga units depending
  |  on its magnitude.
  |  An aside:  Memory capacities are traditionally reported in binary
  |  units (Kib, Mib, etc) while just about everything else should be
  |  reported in decimal units (KB, MB, etc).  This includes transfer
  |  rates, and disk capacities, contrary to what many people think.
  |  Take a look at http://www.pcguide.com/intro/fun/bindec.htm
  */
gint
format_size_abbrev(gchar *buf, size_t buflen, gfloat size,
		size_abbrev_table *tbl, size_t tbl_size)
	{
	gint	i;

	for (i = 0; i < tbl_size - 1; ++i)
		if (size < tbl[i].limit)
			break;
	return snprintf(buf, buflen, tbl[i].format, size / tbl[i].divisor);
	}

  /* Linux /proc always reports floats with decimal points, but sscanf() for
  |  some locales needs commas in place of decimal points.  So, if sscanf()
  |  fails, I call this routine (temperatures, voltages, load average).
  */
gfloat
locale_float_fix(gchar *s)
	{
	gchar	ibuf[16], dbuf[16];
	gint	place;
	gfloat	v	= 0;

	ibuf[0] = dbuf[0] = '\0';
	sscanf(s, "%[^.].%s", ibuf, dbuf);
	for (s = dbuf, place = 10; *s; ++s, place *= 10)
		v += (gfloat)(*s - '0') / place;
	if (*ibuf == '-')
		v = -(v + atof(ibuf + 1));
	else
		v += atof(ibuf);
	return v;
	}


  /* Need a system() call that closes all open files (except stdin, stdout,
  |  and stderr) in the child process.
  */
gint
gkrellm_system(gchar *cmd)
	{
	pid_t	pid, child_pid;
	gint	fd, max_open, status = -1;

	if (!cmd)
		return status;
	pid = fork();
	if (pid == (pid_t) 0)
		{	/* Child */
#if defined(_SC_OPEN_MAX)
		max_open = sysconf(_SC_OPEN_MAX);
#elif defined (_NFILE)
		max_open = _NFILE;
#else
		max_open = 40;
#endif
		for (fd = 3; fd < max_open; ++fd)
			close(fd);
		system(cmd);
		_exit(0);
		}
	else if (pid > (pid_t) 0)
		{	/* Parent */
#if defined(WNOHANG) && defined(WUNTRACED)
		do
			child_pid = waitpid(pid, &status, 0);
		while (child_pid == (pid_t) -1 && errno == EINTR);

		if (child_pid != pid)
			status = -1;
#else
		do
			{
			child_pid = wait(&status);
			if (child_pid <= -1 && errno != EINTR)
				{
				status = -1;
				break;
				}
			}
		while (child_pid != pid);
#endif
		}
	return status;
	}
