/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* This file contains the general sensor interface as well as a voltage
|  monitor.  The general sensor interface provides data and configuration
|  for temperature, fan, and voltage sensors.  The monitoring of this data
|  is handled here (voltage monitor) and in the CPU and Proc monitors (fan
|  and temperature monitoring).  So the organization here is:
|       sensor interface (temperature, fan, voltage)
|       voltage monitor
|       unified config for sensor interface and voltage monitor
*/

/*
|  10/29/2001  NetBSD code contributed by
|                Anthony Mallet anthony.mallet@useless-ficus.net
|   2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <dirent.h>

  /* chips in a class use a common VoltDefault struct */
typedef enum
	{
	lm78,		/* includes lm78 lm79 w83781d sis5595 */
	w83xxx,		/* includes w83782d w83783s w83627hf */
	lm80,
	gl518,
	gl520,
	via686,
	adm1025,
	mtp008,
	it87
	}
	ChipClass;

typedef struct
	{
	gchar	*name;
	gfloat	factor;
	gfloat	offset;
	gchar	*vref;
	}
	VoltDefault;

typedef struct _sensor
	{
	gchar		*name;			/* cpuX, mb, Vx name mapped to this sensor */
	gchar		*path;			/* Pathname to sensor data or device file */
	gint		type;
#if defined(__FreeBSD__) || defined(__NetBSD__)
	gint		iodev;
	gint		interface;
#endif
	gint		id;
	gint		enabled;		/* voltage enable */
	gfloat		factor;			/* Scale sensor reading		*/
	gfloat		offset;			/* Add to sensor reading	*/
	struct _sensor
				*vref;			/* A neg volt may be function of a ref volt */

	ChipClass	chip_class;
	VoltDefault	*voltdefault;
	gint		voltdefaultsize;
	gint		volt_order;
	gboolean	has_config;

	GtkWidget	*select_button;	/* Voltages tab */
	GtkWidget	*entry;
	GtkWidget	*factor_spin_button;
	GtkWidget	*offset_spin_button;
	GtkWidget	*volt_order_spin_button;
	GtkWidget	*alert_button;
	GtkWidget	*vref_entry;

	gfloat		value;
	gboolean	value_valid;

	Alert		*alert;
	void		(*cb_alert)();
	gpointer	cb_alert_data;
	}
	Sensor;


static GList	*sensor_list	= NULL;

static gint		n_volt_sensors;
static gboolean	need_vref_config;

static gboolean	(*get_temperature)(Sensor *);
static gboolean	(*get_fan)(Sensor *);
static gboolean	(*get_voltage)(Sensor *);


  /* Tables of voltage correction factors and offsets derived from the
  |  compute lines in sensors.conf.  See the README file.
  */
	/* "lm78-*" "lm78-j-*" "lm79-*" "w83781d-*" "sis5595-*" "as99127f-*" */
	/* Values from LM78/LM79 data sheets	*/
static VoltDefault	voltdefault0[] =
	{
	{ "Vcor1",	1.0,    0, NULL },
	{ "Vcor2",	1.0,    0, NULL },
	{ "+3.3V",	1.0,    0, NULL },
	{ "+5V",	1.68,   0, NULL },		/* in3 ((6.8/10)+1)*@	*/
	{ "+12V",	4.0,    0, NULL },		/* in4 ((30/10)+1)*@	*/
	{ "-12V",	-4.0,   0, NULL },		/* in5 -(240/60)*@		*/
	{ "-5V",	-1.667, 0, NULL }		/* in6 -(100/60)*@		*/
	};

#if defined(__linux__)
	/* "w83782d-*" "w83783s-*" "w83627hf-*" */
static VoltDefault	voltdefault1[] =
	{
	{ "Vcor1",	1.0,   0, NULL },
	{ "Vcor2",	1.0,   0, NULL },
	{ "+3.3V",	1.0,   0, NULL },
	{ "+5V",	1.68,  0, NULL },		/* in3 ((6.8/10)+1)*@		*/
	{ "+12V",	3.80,  0, NULL },		/* in4 ((28/10)+1)*@		*/
	{ "-12V",	5.14 , -14.91, NULL },	/* in5 (5.14 * @) - 14.91	*/
	{ "-5V",	3.14 , -7.71,  NULL },	/* in6 (3.14 * @) -  7.71	*/
	{ "V5SB",	1.68,  0, NULL },		/* in7 ((6.8/10)+1)*@		*/
	{ "VBat",	1.0,   0, NULL }
	};

	/* lm80-*	*/
static VoltDefault	voltdefault2[] =
	{
	{ "+5V",	2.633, 0, NULL },		/* in0 (24/14.7 + 1) * @	*/
	{ "VTT",	1.0,   0, NULL },
	{ "+3.3V",	1.737, 0, NULL },		/* in2 (22.1/30 + 1) * @	*/
	{ "Vcore",	1.474, 0, NULL },		/* in3 (2.8/1.9) * @		*/
	{ "+12V",	6.316, 0, NULL },		/* in4 (160/30.1 + 1) * @	*/
	{ "-12V",	5.482, -4.482, "in0" },	/* in5 (160/35.7)*(@ - in0) + @	*/
	{ "-5V",	3.222, -2.222, "in0" }	/* in6 (36/16.2)*(@ - in0) + @	*/
	};

	/* gl518sm-*	*/
static VoltDefault	voltdefault3[] =
	{
	{ "+5V",	1.0,   0, NULL },		/* vdd */
	{ "+3.3V",	1.0,   0, NULL },		/* vin1 */
	{ "+12V",	4.192, 0, NULL },		/* vin2 (197/47)*@	*/
	{ "Vcore",	1.0,   0, NULL }		/* vin3 */
	};

	/* gl520sm-*	*/
static VoltDefault	voltdefault4[] =
	{
	{ "+5V",	1.0,   0,    NULL },	/* vdd */
	{ "+3.3V",	1.0,   0,    NULL },	/* vin1 */
	{ "+12V",	4.192, 0,    NULL },	/* vin2 (197/47)*@	*/
	{ "Vcore",	1.0,   0,    NULL },	/* vin3 */
	{ "-12V",	5.0,   -4.0, "vdd" }	/* vin4 (5*@)-(4*vdd)	*/
	};

	/* via686a-*	*/
static VoltDefault	voltdefault5[] =
	{
	{ "Vcor",	1.02,  0, NULL },		/* in0 */
	{ "+2.5V",	1.0,   0, NULL },		/* in1 */
	{ "I/O",	1.02,  0, NULL },		/* in2 */
	{ "+5V",	1.009, 0, NULL },		/* in3 */
	{ "+12V",	1.04,  0, NULL }		/* in4 */
	};

	/* mtp008-*	*/
static VoltDefault	voltdefault6[] =
	{
	{ "Vcor1",	1.0,   0,       NULL },		/* in0 */
	{ "+3.3V",	1.0,   0,       NULL },		/* in1 */
	{ "+12V",	3.8,   0,       NULL },		/* in2 @ * 38 / 10*/
	{ "Vcor2",	1.0,   0,       NULL },		/* in3 */
	{ "?",		1.0,   0,       NULL },		/* in4 */
	{ "-12V",	5.143, -16.944, NULL },		/* in5 (@ * 36 - 118.61) / 7*/
	{ "Vtt",	1.0,   0,       NULL }		/* in6 */
	};

	/* adm1025-*	*/
static VoltDefault	voltdefault7[] =
	{
	{ "+2.5V",	1.0,   0,       NULL },		/* in0 */
	{ "VCCP",	1.0,   0,       NULL },		/* in1 */
	{ "+3.3V",	1.0,   0,       NULL },		/* in2 */
	{ "+5V",	1.0,   0,       NULL },		/* in3 */
	{ "+12V",	1.0,   0,       NULL },		/* in4 */
	{ "VCC",	1.0,   0,       NULL }		/* in5 */
	};

	/* it878-*	*/
static VoltDefault	voltdefault8[] =
	{
	{ "Vcor1",	1.0,    0, NULL },
	{ "Vcor2",	1.0,    0, NULL },
	{ "+3.3V",	2.0,    0, NULL },		/* in2 (1 + 1)*@		*/
	{ "+5V",	1.68,   0, NULL },		/* in3 ((6.8/10)+1)*@	*/
	{ "+12V",	4.0,    0, NULL },		/* in4 ((30/10)+1)*@	*/
	{ "-12V",	-7.2,   0, NULL },		/* in5 -(72/10)*@		*/
	{ "-5V",	-5.6,   0, NULL },		/* in6 -(56/10)*@		*/
	{ "Stby",	1.68,   0, NULL }		/* in7 ((6.8/10)+1)*@	*/
	};
#endif	/* __linux__ */

/* ====== System dependent interface ====================================== */

/* ------- FreeBSD ------------------------------------------------------- */
#if defined(__FreeBSD__)
#include <osreldate.h>
#include <machine/cpufunc.h>
#if __FreeBSD_version >= 300000
#include <machine/smb.h>
#endif

/* Interface types */
#define INTERFACE_IO		0
#define INTERFACE_SMB		1

/* Addresses to use for /dev/io */
#define WBIO1			0x295
#define WBIO2			0x296

/* LM78/79 addresses */
#define LM78_VOLT(val)		(0x20 + (val))
#define LM78_TEMP		0x27
#define LM78_FAN(val)		(0x28 + (val))
#define LM78_FANDIV		0x47

#define	SENSORS_DIR		"/dev"


static gint
get_data(int iodev, u_char command, int interface, u_char *ret)
	{
	u_char byte = 0;

	if (interface == INTERFACE_IO)
		{
		outb(WBIO1, command);
		byte = inb(WBIO2);
		}
#if __FreeBSD_version >= 300000
	else if (interface == INTERFACE_SMB)
		{
		struct smbcmd cmd;

		bzero(&cmd, sizeof(cmd));
		cmd.data.byte_ptr = &byte;
		cmd.slave         = 0x5a;
		cmd.cmd           = command;
		if (ioctl(iodev, SMB_READB, (caddr_t)&cmd) == -1)
			{
			close(iodev);
			return FALSE;
			}
		}
#endif
	else
		{
		return FALSE;
		}
	if (byte == 0xff)
		return FALSE;
	*ret = byte;
	return TRUE;
	}

static gint
get_freebsd_sensor(Sensor *sensor)
	{
	gint iodev = sensor->iodev;
	gint interface = sensor->interface;
	gint id = sensor->id;
	u_char byte;

	if (sensor->type == SENSOR_TEMPERATURE)
		{
		if (get_data(iodev, LM78_TEMP, interface, &byte))
			{
			sensor->value = (gfloat) byte;
			return TRUE;
			}
		}
	else if (sensor->type == SENSOR_FAN)
		{
		if (get_data(iodev, LM78_FAN(id), interface, &byte))
			{
			if (byte == 0)
				return FALSE;
			sensor->value = 1.35E6 / (gfloat) byte;
			return TRUE;
			}
		}
	else if (sensor->type == SENSOR_VOLTAGE)
		{
		if (get_data(iodev, LM78_VOLT(id), interface, &byte))
			{
			sensor->value = (gfloat) byte / 64.0;
			return TRUE;
			}
		}
	return FALSE;
	}

void
scan_for_sensors(void)
	{
	gchar		*chip_dir = SENSORS_DIR;
#if __FreeBSD_version >= 300000
	DIR		*dir;
	struct dirent	*dentry;
#endif
	Sensor		*sensor = NULL;
	gchar		temp_file[256];
	gint		iodev, interface, id;
	gint		fandiv[3];
	u_char		byte;

#if __FreeBSD_version >= 300000
	if ((dir = opendir(chip_dir)) != NULL)
		{
		while ((dentry = readdir(dir)) != NULL)
			{
			if (dentry->d_name[0] == '.' || dentry->d_ino <= 0)
				continue;
			if (strncmp(dentry->d_name, "smb", 3) != 0)
				continue;
			snprintf(temp_file, sizeof(temp_file), "%s/%s",
						chip_dir, dentry->d_name);
			if ((iodev = open(temp_file, O_RDWR)) == -1)
				continue;
			sensor = g_new0(Sensor, 1);
			sensor->iodev = iodev;
			sensor->interface = INTERFACE_SMB;
			sensor->type = SENSOR_TEMPERATURE;
			if (!get_freebsd_sensor(sensor))
				{
				g_free(sensor);
				close(iodev);
				continue;
				}
			sensor->path = g_strdup_printf("%s/%s%s",
						       SENSORS_DIR, "temp",
						       dentry->d_name + 3);
			sensor->name = g_strdup("");
			sensor->factor = 1.0;
			sensor->offset = 0;
			sensor_list = g_list_append(sensor_list, sensor);
			}
		closedir(dir);
		}
#endif
	if (!sensor_list)
		{
		snprintf(temp_file, sizeof(temp_file), "%s/%s",
			 chip_dir, "io");
		if ((iodev = open(temp_file, O_RDWR)) == -1)
			return;
		sensor = g_new0(Sensor, 1);
		sensor->iodev = iodev;
		sensor->interface = INTERFACE_IO;
		sensor->type = SENSOR_TEMPERATURE;
		if (!get_freebsd_sensor(sensor))
			{
			g_free(sensor);
			close(iodev);
			return;
			}
		sensor->path = g_strdup_printf("%s/%s", SENSORS_DIR, "temp0");
		sensor->name = g_strdup("");
		sensor->factor = 1.0;
		sensor->offset = 0;
		sensor_list = g_list_append(sensor_list, sensor);
		}
	if (sensor_list)
		{
		iodev = sensor->iodev;
		interface = sensor->interface;
		if (get_data(iodev, LM78_FANDIV, interface, &byte))
			{
			fandiv[0] = 1 << ((byte & 0x30) >> 4);
			fandiv[1] = 1 << ((byte & 0xc0) >> 6);
			fandiv[2] = 2;
			for (id = 0; id < 3; ++id)
				{
				sensor = g_new0(Sensor, 1);
				sensor->iodev = iodev;
				sensor->interface = interface;
				sensor->type = SENSOR_FAN;
				sensor->id = id;
				if (!get_freebsd_sensor(sensor))
					{
					g_free(sensor);
					continue;
					}
				sensor->path = g_strdup_printf("%s/%s%d",
							       SENSORS_DIR,
							       "fan", id);
				sensor->name = g_strdup("");
				sensor->factor = 1.0 / (gfloat) fandiv[id];
				sensor->offset = 0;
				sensor_list = g_list_append(sensor_list,
							    sensor);
				}
			}
		for (id = 0; id < 7; ++id)
			{
			sensor = g_new0(Sensor, 1);
			sensor->iodev = iodev;
			sensor->interface = interface;
			sensor->type = SENSOR_VOLTAGE;
			sensor->id = id;
			if (!get_freebsd_sensor(sensor))
				{
				g_free(sensor);
				continue;
				}
			sensor->path = g_strdup_printf("%s/%s%d", SENSORS_DIR,
						       "in", id);
			sensor->chip_class = lm78;
			sensor->volt_order = n_volt_sensors++;
			sensor->voltdefault = &voltdefault0[0];
 			sensor->voltdefaultsize =
						sizeof (voltdefault0) / sizeof (VoltDefault);
			sensor->offset = 0;
			gkrellm_dup_string(&sensor->name, voltdefault0[id].name);
			sensor->factor = voltdefault0[id].factor;
			sensor_list = g_list_append(sensor_list, sensor);
			}
		}
	}

#endif	/* __FreeBSD__ */


/* ------- Linux ------------------------------------------------------- */
#if defined(__linux__)

#define	SENSORS_DIR	"/proc/sys/dev/sensors"

static gboolean
get_linux_sensor(Sensor *sensor)
	{
	FILE	*f;
	gchar	buf[64], sbuf[3][16];
	gint	n;
	gfloat	t[3];

	if ((f = fopen(sensor->path, "r")) == NULL)
		return FALSE;
	fgets(buf, sizeof(buf), f);
	fclose(f);
	t[0] = t[1] = t[2] = 0;
	if (sensor->type == SENSOR_TEMPERATURE)
		{
		if ((n = sscanf(buf, "%*f %*f %f %f %f", &t[0], &t[1], &t[2])) < 1)
			{	/* Could fail because of decimal "comma" locales */
			if ((n = sscanf(buf, "%*d.%*d %*d.%*d %16s %16s %16s",
						sbuf[0], sbuf[1], sbuf[2])) < 1)
				return FALSE;
			sensor->value = locale_float_fix(sbuf[n - 1]);
			}
		else
			sensor->value = t[n - 1];
		}
	else if (sensor->type == SENSOR_FAN)
		{
		if (sscanf(buf, "%*f %f", &sensor->value) != 1)
			return FALSE;
		}
	else if (sensor->type == SENSOR_VOLTAGE)
		{
		if (sscanf(buf, "%*f %*f %f", &sensor->value) != 1)
			{
			if (sscanf(buf, "%*d.%*d %*d.%*d %16s", sbuf[0]) != 1)
				return FALSE;
			sensor->value = locale_float_fix(sbuf[0]);
			}
		}
	else
		return FALSE;

	return TRUE;
	}

static void
get_volt_default(gchar *chip_name, VoltDefault **vdf, gint *vdfsize,
		ChipClass *chip)
	{
	if (!strncmp(chip_name, "it87", 4))
		{
		*chip = it87;
		*vdf = &voltdefault8[0];
		*vdfsize = sizeof (voltdefault8) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "adm1025", 7))
		{
		*chip = adm1025;
		*vdf = &voltdefault7[0];
		*vdfsize = sizeof (voltdefault7) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "mtp008", 6))
		{
		*chip = mtp008;
		*vdf = &voltdefault6[0];
		*vdfsize = sizeof (voltdefault6) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "via686", 6))
		{
		*chip = via686;
		*vdf = &voltdefault5[0];
		*vdfsize = sizeof (voltdefault5) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "gl520", 5))
		{
		*chip = gl520;
		need_vref_config = TRUE;
		*vdf = &voltdefault4[0];
		*vdfsize = sizeof (voltdefault4) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "gl518", 5))
		{
		*chip = gl518;
		*vdf = &voltdefault3[0];
		*vdfsize = sizeof (voltdefault3) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "lm80", 4))
		{
		*chip = lm80;
		need_vref_config = TRUE;
		*vdf = &voltdefault2[0];
		*vdfsize = sizeof (voltdefault2) / sizeof (VoltDefault);
		}
	else if (!strncmp(chip_name, "w83", 3) && strncmp(chip_name, "w83781", 6))
		{
		*chip = w83xxx;
		*vdf = &voltdefault1[0];
		*vdfsize = sizeof (voltdefault1) / sizeof (VoltDefault);
		}
	else
		{
		*chip = lm78;
		*vdf = &voltdefault0[0];
		*vdfsize = sizeof (voltdefault0) / sizeof (VoltDefault);
		}
	}


static void
scan_for_sensors(void)
	{
	DIR				*dir, *chip_dir;
	struct dirent	*dentry;
	FILE			*f, *f1;
	Sensor			*sensor;
	VoltDefault		*voltdefault;
	ChipClass		chip;
	gint			id = 0;
	gint			type, divisor, voltdefaultsize, order_base;
	gchar			*name;
	gchar			path[256], fan_div[256];

	if ((dir = opendir(SENSORS_DIR)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (dentry->d_name[0] == '.' || dentry->d_ino <= 0)
			continue;
		snprintf(path, sizeof(path), "%s/%s", SENSORS_DIR, dentry->d_name);
		if ((chip_dir = opendir(path)) == NULL)
			continue;
		order_base = n_volt_sensors;
		get_volt_default(dentry->d_name, &voltdefault, &voltdefaultsize,&chip);
		snprintf(fan_div, sizeof(fan_div), "%s/%s", path, "fan_div");
		f = fopen(fan_div, "r");
		while ((dentry = readdir(chip_dir)) != NULL)
			{
			name = dentry->d_name;
			if (!strncmp(name, "temp", 4))
				type = SENSOR_TEMPERATURE;
			else if (   !strncmp(name, "fan", 3)
					 && (!name[3] || (isdigit(name[3]) && !name[4]))
					)
				type = SENSOR_FAN;
			else if (!strncmp(name, "in", 2) && isdigit(name[2]))
				{
				type = SENSOR_VOLTAGE;
				id = name[2] - '0';
				}
			else if (!strncmp(name, "vi", 2) && isdigit(name[3]))
				{
				type = SENSOR_VOLTAGE;
				id = name[3] - '0';
				}
			else if (!strcmp(name, "vdd"))
				{
				type = SENSOR_VOLTAGE;
				id = 0;
				}
			else
				continue;

			sensor = g_new0(Sensor, 1);
			sensor->path = g_strdup_printf("%s/%s", path, name);
			sensor->name = g_strdup("");
			sensor->factor = 1.0;
			sensor->offset = 0;
			sensor->type = type;
			sensor->chip_class = chip;
			sensor_list = g_list_append(sensor_list, sensor);

			if (type == SENSOR_FAN)
				{
				divisor = 0;
				if (!f || fscanf(f, "%d", &divisor) != 1)
					{
					snprintf(fan_div, sizeof(fan_div), "%s_div", sensor->path);
					if ((f1 = fopen(fan_div, "r")) != NULL)
						{
						fscanf(f1, "%d", &divisor);
						fclose(f1);
						}
					}
				if (divisor > 0)
					sensor->factor /= (gfloat) divisor;
				}
			if (type == SENSOR_VOLTAGE)
				{
				++n_volt_sensors;
				sensor->voltdefault = voltdefault;
				sensor->voltdefaultsize = voltdefaultsize;
				sensor->id = id;
				sensor->volt_order = order_base + id;
				if (id < voltdefaultsize)
					{
					gkrellm_dup_string(&sensor->name, voltdefault[id].name);
					sensor->factor = voltdefault[id].factor;
					sensor->offset = voltdefault[id].offset;
					}
				else
					gkrellm_dup_string(&sensor->name, name);
				}
			if (GK.debug_level & DEBUG_SENSORS)
				printf("scan_for_sensors: %s %d\n",
						sensor->path, sensor->type);
			}
		closedir(chip_dir);
		if (f)
			fclose(f);
		}
	closedir(dir);
	}

#endif  /* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__)

/* The SENSORS_DIR is not defined as a directory, but directly points on
 * the "sysmon" device, which implements envsys(4) API. This #define is
 * not really useful for NetBSD since every sensor will use that device,
 * but it still provides the location of the device inside the GUI. */
#define SENSORS_DIR	"/dev/sysmon"

#include <sys/envsys.h>

/*
 * scan_for_sensors -----------------------------------------------------
 *
 * At the moment, only two chips are supported: lm78 and alike (see
 * lm(4)), and VT82C68A South Bridge for VIA chipsets (see viaenv(4)).
 * Both support the envsys(4) API.
 *
 * XXX /dev/sysmon is opened but never closed. This is a problem since
 * the driver wants an exclusive lock (e.g. envstat won't work when
 * GKrellM will be running). But, at this time, I don't want to
 * open/close sysmon each time a reading is needed. See README for
 * details.
 */

static void
scan_for_sensors(void)
{
   envsys_basic_info_t info;	/* sensor misc. info */
   Sensor *sensor;		/* GKrellM sensor struct */
   int fd;			/* file desc. for /dev/sysmon */
   int id = 0;			/* incremented for each sensor */
   char *s;

   /* check if some sensor is configured */
   fd = open(SENSORS_DIR, O_RDONLY); /* never closed :( */ 
   if (fd < 0) return;

   /* iterate through available sensors, until the first invalid */
   for(info.sensor=0; ; info.sensor++) {

      /* stop if we can't ioctl() */
      if (ioctl(fd, ENVSYS_GTREINFO, &info) < 0) break;
      /* stop if that sensor is not valid */
      if (!(info.validflags & ENVSYS_FVALID)) break;

      sensor = g_new0(Sensor, 1);
      switch(info.units) {
	 case ENVSYS_STEMP:
	    sensor->type = SENSOR_TEMPERATURE;	break;
	 case ENVSYS_SFANRPM:
	    sensor->type = SENSOR_FAN;		break;
	 case ENVSYS_SVOLTS_DC:
	    sensor->type = SENSOR_VOLTAGE;		break;
	 default:
	    /* unwanted sensor type: continue */
	    g_free(sensor);
	    continue;
      }

      /* ok, we've got one working sensor */
      sensor->iodev = fd;
      sensor->interface = info.sensor;
      sensor->id = id++;	

      sensor->offset = 0.0;
      sensor->factor = 1.0;

      sensor->vref = NULL;
      sensor->enabled = 1;
      sensor->chip_class = lm78; /* there's no way to compute this */
      sensor->volt_order = n_volt_sensors++;
      sensor->voltdefault = &voltdefault0[0];
      sensor->voltdefaultsize =
	 sizeof (voltdefault0) / sizeof (VoltDefault);

      sensor->path = g_strdup_printf("%s/%s", SENSORS_DIR, info.desc);
      gkrellm_dup_string(&sensor->name, info.desc);
      /* must map spaces into something else (for config file items) */
      for(s=strchr(sensor->path, ' '); s != NULL; s=strchr(s, ' '))
	 *s++ = '_';
      for(s=strchr(sensor->name, ' '); s != NULL; s=strchr(s, ' '))
	 *s++ = '_';

      sensor_list = g_list_append(sensor_list, sensor);
   }
}


/*
 * get_netbsd_sensor ----------------------------------------------------
 *
 * Perform sensor reading
 */

static gboolean
get_netbsd_sensor(Sensor *sensor)
{
   envsys_tre_data_t data;	/* sensor data */

   data.sensor = sensor->interface;
   if (ioctl(sensor->iodev, ENVSYS_GTREDATA, &data) < 0) return FALSE;
   if (!(data.validflags & (ENVSYS_FVALID|ENVSYS_FCURVALID))) return FALSE;

   switch(data.units) {
      case ENVSYS_STEMP:	/* values in uK */
	 sensor->value = (data.cur.data_us / 1.0e6) - 273.15/*0K*/ ;
	 break;

      case ENVSYS_SFANRPM:	/* values in RPM */
	 sensor->value = data.cur.data_us;
	 break;

      default:			/* values in uV */
	 sensor->value = data.cur.data_s / 1.0e6;
	 break;
   }

   return TRUE;
}

#endif /* __NetBSD__ */


/* ----- Other Systems -------------------------------------------------- */
#if !defined(__FreeBSD__) && !defined(__linux__) && !defined(__NetBSD__)

#define	SENSORS_DIR		""

#endif


/* ----- Pick a system interface ----------------------------------------- */

static gboolean
setup_sensor_interface(void)
	{
#if defined(__FreeBSD__)
	get_temperature = get_freebsd_sensor;
	get_fan = get_freebsd_sensor;
	get_voltage = get_freebsd_sensor;
#endif

#if defined(__linux__)
	scan_for_sensors();
	get_temperature = get_linux_sensor;
	get_fan = get_linux_sensor;
	get_voltage = get_linux_sensor;
#endif

#if defined(__NetBSD__)
	scan_for_sensors();
	get_temperature = get_netbsd_sensor;
	get_fan = get_netbsd_sensor;
	get_voltage = get_netbsd_sensor;
#endif

	return sensor_list ? TRUE : FALSE;
	}

/* ======================================================================== */

static gint		units_fahrenheit;


  /* Sort so that sensors are ordered: temp, fan, voltage.
  */
static gint
strcmp_sensor_path(Sensor *s1, Sensor *s2)
	{
	if (s1->type == SENSOR_TEMPERATURE && s2->type != SENSOR_TEMPERATURE)
		return -1;
	if (s1->type != SENSOR_TEMPERATURE && s2->type == SENSOR_TEMPERATURE)
		return 1;

	if (s1->type == SENSOR_FAN && s2->type != SENSOR_FAN)
		return -1;
	if (s1->type != SENSOR_FAN && s2->type == SENSOR_FAN)
		return 1;

	return strcmp(s1->path, s2->path);
//	return strcmp_basename(s1->path, s2->path);
	}

  /* Return pointer to the sensor basename part of a sensor pathname.
  |  On Linux, this basename includes the parent chip directory, so
  |  "lm78/temp" will be a basename, and not just "temp".  This is to
  |  allow unique identification in case of multiple "temp" files.
  |  ie, more than 1 chip directory, each with a "temp" file.
  */
static gchar *
sensor_basename(gchar *path)
	{
	gchar	*s;

	s = path + strlen(SENSORS_DIR);
	if (*s == '/')
		return s + 1;
	return "";
	}

  /* A name (mb, cpu, cpu0, cpu1, ...) must be associated with a real sensor
  |  name (w83782d-i2c-0-2d/tempX, etc.).  The CPU and Proc monitors
  |  request sensor readings for mb/cpu names which the user must have
  |  mapped to real sensor file names in the config.
  */
static Sensor *
map_sensor_name(gchar *name, gchar *sensor_name)
	{
	GList	*list;
	Sensor	*sensor;
	gchar	*s;

	for (list = sensor_list; list; list = list->next)
		{
		sensor = (Sensor *) list->data;
		if (sensor->path)
			{
			s = sensor_basename(sensor->path);
			if (*s && strcmp(s, sensor_name) == 0)
				{
				if (GK.debug_level & DEBUG_SENSORS)
					printf("\tmap_sensor_name: <%s> to <%s>\n", s, name);
				gkrellm_dup_string(&sensor->name, name);
				return sensor;
				}
			}
		}
	return NULL;
	}

gboolean
gkrellm_sensors_available(void)
	{
	return (sensor_list || GK.demo) ? TRUE : FALSE;
	}

  /* The cpu and proc monitors both need a couple of sensor decals
  |  created on their panels.  The left one will only display fan speeds
  |  while the right one will display both fan and temps depending on modes.
  */
void
sensor_create_decals(Panel *p, gint style_id, Decal **dsensor, Decal **dfan)
	{
	Style		*style;
	TextStyle	*ts;
	Decal		*ds	= NULL,
				*df	= NULL;
	gint		w, w_avail, margin;

	if (sensor_list || GK.demo)
		{
		style = gkrellm_panel_style(style_id);
		ts = gkrellm_panel_alt_textstyle(style_id);
		margin = style->margin;
		w_avail = gkrellm_chart_width() - 2 * margin;

		/* Fan decal goes at left margin
		*/
		df = gkrellm_create_decal_text(p, "8888", ts, style, -1, -1, 0);

		/* Sensor decal (fan and/or temp) carves out space remaining to right.
		|  Try to get enough for .1 deg resolution, otherwise what is left.
		*/
		w = gdk_string_width(ts->font, "188.8C");
		if (w >= w_avail - df->w)
			w = gdk_string_width(ts->font, "88.8C");
		if (w >= w_avail - df->w - 8)
			{
			/* violate margins to try a squeeze fit
			*/
			margin = 0;
			w_avail = gkrellm_chart_width() - 2 * margin;
			if (w > w_avail - df->w - 4)
				w = w_avail - df->w - 4;	/* units pushed off the right */
			}
		ds = gkrellm_create_decal_text(p, "8.C", ts, style, -1, -1, w);
		ds->x = w_avail + margin - w;
		df->x = margin;
		}
	*dsensor = ds;
	*dfan = df;
	}

void
gkrellm_sensor_draw_fan_decal(Panel *p, Decal *d, gfloat f)
	{
	gchar	buf[8];
	gint	w;

	if (!p || !d)
		return;
	snprintf(buf, sizeof(buf), "%.0f", f);
	w = gdk_string_measure(d->text_style.font, buf) + d->text_style.effect;
	d->x_off = d->w - w;
	if (d->x_off < 0)
		d->x_off = 0;
	gkrellm_draw_decal_text(p, d, buf, atoi(buf) /* draw when changes */);
	}

void
gkrellm_sensor_draw_temperature_decal(Panel *p, Decal *d, gfloat t,
							gchar units)
	{
	gchar	*s, buf[8];
	gint	w, value = 0;

	if (!p || !d)
		return;
	snprintf(buf, sizeof(buf), "%.1f%c", t, units);
	if ((s = strchr(buf, '.')) == NULL)
		s = strchr(buf, ',');			/* Locale may use commas */
	w = gdk_string_measure(d->text_style.font, buf) + d->text_style.effect;
	if (w > d->w)
		{
		snprintf(buf, sizeof(buf), "%.0f%c", t, units);
		w = gdk_string_measure(d->text_style.font, buf) + d->text_style.effect;
		value = atoi(buf);
		}
	else
		value = 10 * atoi(buf) + (s ? atoi(s+1) : 0);
	
	d->x_off = d->w - w;
	if (d->x_off < 0)
		d->x_off = 0;
	gkrellm_draw_decal_text(p, d, buf, value /* draw when changes */);
	}

gpointer
get_mapped_sensor(gchar *name, gint sensor_type)
	{
	GList	*list;
	Sensor	*sensor;

	if (!name || !*name)
		return NULL;
	for (list = sensor_list; list; list = list->next)
		{
		sensor = (Sensor *) list->data;
		if (sensor->type != sensor_type)
			continue;
		if (sensor->name && !strcmp(sensor->name, name))
			return sensor;
		}
	return NULL;
	}

Sensor *
lookup_sensor_from_basename(gchar *basename)
	{
	GList	*list;
	Sensor	*s;

	if (!basename)
		return NULL;
	for (list = sensor_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (!strcmp(sensor_basename(s->path), basename))
			return s;
		}
	return NULL;
	}

  /* Given a in0, in1, ... name as a reference to use for a sensor,
  |  find the sensor with that name for the same chip as sr.
  */
static Sensor *
lookup_vref(Sensor *sr, gchar *name)
	{
	GList	*list;
	Sensor	*sv;
	gchar	*s, buf[128];

	snprintf(buf, 96, "%s", sensor_basename(sr->path));
	s = strrchr(buf, '/');
	if (s)
		++s;
	else
		s = buf;
	snprintf(s, 31, "%s", name);
	for (list = sensor_list; list; list = list->next)
		{
		sv = (Sensor *) list->data;
		if (   sv->type == SENSOR_VOLTAGE
			&& !strcmp(sensor_basename(sv->path), buf)
		   )
			return sv;
		}
	return NULL;
	}

Alert *
gkrellm_mapped_sensor_alert(gchar *name, gint sensor_type)
	{
	Sensor	*sensor;

	sensor = (Sensor *) get_mapped_sensor(name, sensor_type);
	if (sensor)
		return sensor->alert;
	return NULL;
	}

void
gkrellm_mapped_sensor_alert_connect(gchar *name, gint sensor_type,
		void (*cb_func)(), gpointer data)
	{
	Sensor	*sensor;

	sensor = (Sensor *) get_mapped_sensor(name, sensor_type);
	if (!sensor)
		return;
	sensor->cb_alert = cb_func;
	sensor->cb_alert_data = data;
	gkrellm_alert_trigger_connect(sensor->alert, cb_func, data);
	}

  /* Temperature names are mapped to "cpuX" or "mb".  The CPU and Proc monitors
  |  will ask for such names and the user must have made the map in the config
  */
gboolean
gkrellm_sensor_read_temperature(gchar *name, gfloat *temp, gchar *units)
	{
	Sensor	*sensor;
	gfloat	t = 0;
	gint	found_temp = FALSE;

	if (   (sensor = get_mapped_sensor(name, SENSOR_TEMPERATURE)) != NULL
		&& get_temperature && (*get_temperature)(sensor)
	   )
		{
		t = sensor->value * sensor->factor + sensor->offset;
		if (units_fahrenheit)
			t = 1.8 * t + 32.0;
		sensor->value = t;
		found_temp = TRUE;
		}
	if (! found_temp && GK.demo)
		{
		t = 90.0 + (gfloat)(random() & 0xf);
		found_temp = TRUE;
		}
	if (temp)
		*temp = t;
	if (units)
		*units = units_fahrenheit ? 'F':'C';
	if (GK.debug_level & DEBUG_SENSORS)
		printf("read_temperature: %s(%d) t=%f\n", name, found_temp, t);
	if (found_temp && sensor)
		gkrellm_check_alert(sensor->alert, t);
	return found_temp;
	}

  /* Fan names are mapped to "cpuX" or "mb".  The CPU and Proc monitors
  |  will ask for such names and the user must have made the map in the config
  */
gboolean
gkrellm_sensor_read_fan(gchar *name, gfloat *fan)
	{
	Sensor	*sensor;
	gfloat	f	= 0;
	gint	found_fan = FALSE;

	if (   (sensor = get_mapped_sensor(name, SENSOR_FAN)) != NULL
		&& get_fan && (*get_fan)(sensor)
	   )
		{
		sensor->value *= sensor->factor;
		f = sensor->value;
		found_fan = TRUE;
		}
	if (! found_fan && GK.demo)
		{
		f = 4980 + (gfloat)(random() & 0x3f);
		found_fan = TRUE;
		}
	if (fan)
		*fan = f;
	if (GK.debug_level & DEBUG_SENSORS)
		printf("read_fan: %s(%d) rpm=%f\n", name, found_fan, f);
	if (found_fan && sensor)
		gkrellm_check_alert(sensor->alert, f);
	return found_fan;
	}

static gboolean
sensor_read_voltage(Sensor *sensor, gfloat *voltage)
	{
	gfloat	v = 0;
	gfloat	offset;
	gint	found_voltage = FALSE;

	if (sensor && get_voltage && (*get_voltage)(sensor))
		{
		offset = sensor->offset;
		if (sensor->vref)	/* A negative voltage is level shifted by vref */
			offset *= sensor->vref->value;
		sensor->value = sensor->value * sensor->factor + offset;
		v = sensor->value;
		found_voltage = TRUE;
		}
	if (! found_voltage && GK.demo)
		{
		v = 2.9 + (gfloat)(random() & 0x7) * 0.1;
		found_voltage = TRUE;
		}
	if (voltage)
		*voltage = v;
	if (GK.debug_level & DEBUG_SENSORS)
		printf("read_voltage: %s(%d) v=%f\n", sensor->name, found_voltage, v);
	if (found_voltage && sensor)
		gkrellm_check_alert(sensor->alert, v);
	return found_voltage;
	}

gboolean
gkrellm_sensor_read_voltage(gchar *name, gfloat *voltage)
	{
	Sensor	*sensor;

	sensor = get_mapped_sensor(name, SENSOR_VOLTAGE);
	return sensor_read_voltage(sensor, voltage);
	}

/* =================================================================== */
/* The voltage monitor */

#define	SENSOR_STYLE_NAME	"sensors"


typedef struct
	{
	Decal	*name_decal,
			*volt_decal;
	Sensor	*sensor;
	}
	VoltMon;

GList	*volt_list;

GtkWidget		*voltage_vbox;

static Panel	*pVolt;
static gint		style_id;
static gint		volt_mon_width,
				volt_mon_height,
				volt_name_width,
				bezel_width;

  /* Display modes */
#define	DIGITAL_WITH_LABELS	0
#define	DIGITAL_NO_LABELS	1
#define	N_DISPLAY_MODES		2

#define	MONITOR_PAD		6
#define	NAME_PAD		4

Monitor			*mon_volt;
Monitor			*mon_sensors;

static GdkImlibImage	*bezel_image;

static Style	*bezel_style;		/* Just for the bezel image border */

static gint		display_mode,
				voltages_modified,
				correction_modified,
				have_negative_volts;

static gint		minus_width;		/* If will be drawing neg voltages */

  /* If drawing '-' sign, grub a pixel or two to tighten the layout */
static gint		pixel_grub;

static Sensor	*demo0_sensor, *demo1_sensor;

static void
add_volt_mon(Sensor *s)
	{
	VoltMon		*volt;
	VoltDefault	*vdf;

	if (s->type == SENSOR_VOLTAGE)
		{
		vdf = &s->voltdefault[s->id];
		if (!s->has_config && vdf->vref)
			s->vref = lookup_vref(s, vdf->vref);
		}
	if (s->type != SENSOR_VOLTAGE || !s->enabled)
		return;
	volt = g_new0(VoltMon, 1);
	volt->sensor = s;
	volt_list = g_list_append(volt_list, volt);
	}

  /* Verify config of volt orders has not been munged.
  */
static void
check_volt_order(void)
	{
	GList	*list;
	Sensor	*sr;
	gint	i;

	for (i = 0; i < n_volt_sensors; ++i)
		{
		for (list = sensor_list; list; list = list->next)
			if (((Sensor *) list->data)->volt_order == i)
				break;
		if (!list)	/* Did not find an order number */
			{
			i = 0;
			for (list = sensor_list; list; list = list->next)
				{
				sr = (Sensor *) list->data;
				if (sr->type == SENSOR_VOLTAGE)
					sr->volt_order = i++;
				}
			break;
			}
		}
	}

static void
get_volt_list(void)
	{
	GList	*list;

	free_glist_and_data(&volt_list);
	check_volt_order();
	for (list = sensor_list; list; list = list->next)
		add_volt_mon((Sensor *) list->data);

	/* Make sure the themers can see voltage monitor.
	*/
	if (!volt_list && GK.demo)
		{
		if (!demo0_sensor)
			{
			demo0_sensor = g_new0(Sensor, 1);
			demo0_sensor->name = g_strdup(voltdefault0[0].name);
			demo0_sensor->type = SENSOR_VOLTAGE;
			demo0_sensor->enabled = TRUE;
			demo0_sensor->chip_class = lm78;
			demo0_sensor->volt_order = demo0_sensor->id = 0;
			demo0_sensor->voltdefault = &voltdefault0[0];
			demo0_sensor->voltdefaultsize =
					sizeof (voltdefault0) / sizeof (VoltDefault);

			demo1_sensor = g_new0(Sensor, 1);
			*demo1_sensor = *demo0_sensor;
			demo1_sensor->name = g_strdup(voltdefault0[2].name);
			demo1_sensor->volt_order = demo1_sensor->id = 1;
			n_volt_sensors = 2;
			}
		add_volt_mon(demo0_sensor);
		add_volt_mon(demo1_sensor);
		}
	}

#include "pixmaps/sensors/bg_volt.xpm"

static void
draw_volt_bezels(void)
	{
	GList			*list;
	GdkImlibBorder	*b		= &bezel_style->border;
	VoltMon			*volt;
	Decal			*dv;
	gint			x;

	if (!bezel_image)
		return;
	for (list = volt_list; list; list = list->next)
		{
		volt = (VoltMon *) list->data;
		dv = volt->volt_decal;
		x = dv->x + pixel_grub;
		gdk_imlib_paste_image(bezel_image, pVolt->bg_pixmap,
				x - b->left, dv->y - b->top, bezel_width, volt_mon_height);
		gdk_imlib_paste_image(bezel_image, pVolt->pixmap,
				x - b->left, dv->y - b->top, bezel_width, volt_mon_height);
		}
	}

static gboolean
any_negative_volts(void)
	{
	GList		*list;
	Sensor		*s;
	VoltMon		*volt;
	gfloat		v;

	/* This routine can be called before any volt decals exist, but reading
	|  voltages can trigger alerts which expect to find decals.  Hence freeze.
	*/
	for (list = volt_list; list; list = list->next)
		{
		volt = (VoltMon *) list->data;
		gkrellm_freeze_alert(volt->sensor->alert);
		s = volt->sensor->vref;
		if (s && s->value == 0)
			sensor_read_voltage(s, &v);
		sensor_read_voltage(volt->sensor, &v);
		gkrellm_thaw_alert(volt->sensor->alert);
		if (v < 0.0)
			return TRUE;
		}
	return FALSE;
	}

static VoltMon *
voltmon_first(void)
	{
	GList	*list;
	VoltMon	*volt = NULL;
	VoltMon	*vtest;

	for (list = volt_list; list; list = list->next)
		{
		vtest = (VoltMon *) list->data;
		if (!volt)
			volt = vtest;
		else if (volt->sensor->volt_order > vtest->sensor->volt_order)
			volt = vtest;
		}
	return volt;
	}

static VoltMon *
voltmon_next(VoltMon *vcur)
	{
	GList	*list;
	VoltMon	*volt = NULL;
	VoltMon	*vtest;

	for (list = volt_list; list; list = list->next)
		{
		vtest = (VoltMon *) list->data;
		if (vtest->sensor->volt_order > vcur->sensor->volt_order)
			{
			if (!volt)
				volt = vtest;
			else if (vtest->sensor->volt_order < volt->sensor->volt_order)
				volt = vtest;
			}
		}
	return volt;
	}

static void
make_volt_decals(Panel *p, Style *style)
	{
	GdkImlibBorder	*b		= &bezel_style->border;
	VoltMon			*volt;
	Decal			*dv, *dn;
	TextStyle		*ts_volt, *ts_name;
	gchar			*fmt;
	gint			w_volt;

	ts_name = gkrellm_meter_alt_textstyle(style_id);
	ts_volt = gkrellm_meter_textstyle(style_id);

	volt_mon_width = 0;
	volt_mon_height = 0;
	volt_name_width = 0;
	w_volt = 0;

	minus_width = 0;
	have_negative_volts = FALSE;
	fmt = "8.88";
	if (any_negative_volts())
		{
		have_negative_volts = TRUE;
		minus_width = 1;
		fmt = "-8.88";
		}

	for (volt = voltmon_first(); volt; volt = voltmon_next(volt))
		{
		if (display_mode == DIGITAL_WITH_LABELS)
			{
			volt->name_decal = dn = gkrellm_create_decal_text(p,
						volt->sensor->name, ts_name, style, 0, 0, 0);
			if (dn->w > volt_name_width)
				volt_name_width = dn->w;
			}
		dv = gkrellm_create_decal_text(p, fmt, ts_volt, style, 0, 0, 0);
		volt->volt_decal = dv;
		if (minus_width == 1)
			minus_width = gdk_string_width(dv->text_style.font, "-");
		w_volt = dv->w;			/* Same for all volt decals */
		if (dv->h > volt_mon_height)
			volt_mon_height = dv->h;
		}
	if (minus_width)
		pixel_grub = (b->left > 1) ? 2 : 1;
	else
		pixel_grub = 0;
	bezel_width = b->left + w_volt + b->right - pixel_grub;
	volt_mon_height += b->top + b->bottom;

	/* If name decal I let bezel left border encroach into NAME_PAD space
	*/
	if (volt_name_width)
		volt_mon_width = volt_name_width + NAME_PAD + w_volt + b->right;
	else
		volt_mon_width = w_volt;	/* borders encroach into MONITOR_PAD */
	}

static void
layout_volt_decals(Panel *p, Style *style)
	{
	VoltMon	*volt;
	Decal	*dv, *dn;
	gint	x, y, w, c, n, cols;

	w = gkrellm_chart_width() - 2 * style->margin;
	cols = (w + MONITOR_PAD) / (volt_mon_width + MONITOR_PAD);
	if (cols < 1)
		cols = 1;
	n = g_list_length(volt_list);
	if (cols > n)
		cols = n;;
	volt_mon_width = w / cols;		/* spread them out */
	x = (w - cols * volt_mon_width) / 2 + style->margin;
		
	gkrellm_get_top_bottom_margins(style, &y, NULL);
	c = 0;
	for (volt = voltmon_first(); volt; volt = voltmon_next(volt))
		{
		dn = volt->name_decal;
		dv = volt->volt_decal;
		/* Right justify the volt decal in each volt_mon field
		*/
		dv->x = x + (c+1) * volt_mon_width - dv->w - bezel_style->border.right;
		if (cols > 1 && !dn)
			dv->x -= (volt_mon_width - bezel_width) / 2;
		dv->y = y + bezel_style->border.top;
		if (dn)
			{
			if (cols == 1)
				dn->x = style->margin;
			else
				dn->x = dv->x - volt_name_width - NAME_PAD;
			dn->y = y + bezel_style->border.top;
			if (dn->h < dv->h)
				dn->y += (dv->h - dn->h + 1) / 2;
			}
		if (++c >= cols)
			{
			c = 0;
			y += volt_mon_height;
			}
		}
	}

static void
draw_voltages(gint do_names)
	{
	GList		*list;
	VoltMon		*volt;
	Sensor		*sensor;
	Decal		*d;
	gchar		*fmt, buf[32];
	gfloat		v;

	for (list = volt_list; list; list = list->next)
		{
		volt = (VoltMon *) list->data;
		sensor = volt->sensor;
		sensor->value_valid = FALSE;
		if (do_names && volt->name_decal)
			gkrellm_draw_decal_text(pVolt, volt->name_decal, sensor->name, -1);
		if ((d = volt->volt_decal) != NULL)
			{
			if (sensor->vref && !sensor->vref->value_valid)
				sensor_read_voltage(sensor->vref, NULL);
			sensor_read_voltage(sensor, &v);
			sensor->value_valid = TRUE;
			if ((v < 10.0 && v > 0.0) || (v > -10.0 && v < 0.0))
				fmt = "%.2f";
			else
				fmt = "%.1f";
			snprintf(buf, sizeof(buf), fmt, v);
			d->x_off = (v < 0.0) ? 0 : minus_width;
			gkrellm_draw_decal_text(pVolt, d, buf, -1);
			}
		}
	gkrellm_draw_panel_layers(pVolt);
	}

static void
update_voltage(void)
	{
	if (GK.five_second_tick)
		{
		draw_voltages(FALSE);
		}
	}

static gint
expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	if (widget == pVolt->drawing_area)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pVolt->pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
	return FALSE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_sensors);
	return TRUE;
	}

static void
cb_volt_alert_trigger(Alert *alert, Sensor *sensor)
	{
	GList		*list;
	VoltMon		*volt;
	AlertDecal	*ad;
	Decal		*d;

	ad = &alert->ad;
	alert->panel = pVolt;
	for (list = volt_list; list; list = list->next)
		{
		volt = (VoltMon *) list->data;

		/* Make the AlertDecal show up under the right volt decal.
		*/
		if (volt->sensor != sensor)
			continue;
		d = volt->volt_decal;
		if (d)
			{
			ad->x = d->x - 2;
			ad->y = d->y - 2;
			ad->w = d->w + 3;
			ad->h = d->h + 4;
			gkrellm_render_default_alert_decal(alert);
			}
		break;
		}
	}

static void
destroy_voltage_monitor(void)
	{
	if (!pVolt)
		return;
	gkrellm_panel_destroy(pVolt);
	pVolt = NULL;
	}

static GdkImlibBorder	default_bezel_border = {2,1,1,1};

static void
create_voltage_monitor(GtkWidget *vbox, gint first_create)
	{
	Style	*style;
	gchar	**xpm;

	get_volt_list();
	if (!volt_list)
		return;
	if (first_create)
		{
		pVolt = gkrellm_panel_new0();
		bezel_style = gkrellm_style_new0();
		}
	style = gkrellm_meter_style(style_id);

	/* Here is where I define the volt panel theme image extensions.  I ask
	|  for a theme extension image:
	|      THEME_DIR/sensors/bg_volt.png
	|  and for a border for it from the gkrellmrc in the format:
	|      set_image_border sensors_bg_volt l,r,t,b
	| There is no default for bg_volt image, ie it may end up being NULL. 
	*/
	xpm = gkrellm_using_default_theme() ? bg_volt_xpm : NULL;
	if (bezel_image)
		gdk_imlib_kill_image(bezel_image);
	bezel_image = NULL;
	gkrellm_load_image("bg_volt", xpm, &bezel_image, SENSOR_STYLE_NAME);
	if (!gkrellm_set_image_border("sensors_bg_volt", bezel_image, bezel_style))
		bezel_style->border = default_bezel_border;

	make_volt_decals(pVolt, style);
	layout_volt_decals(pVolt, style);
	
	gkrellm_panel_configure(pVolt, NULL, style);

	/* Make the bottom margin reference against the bottom volt decals
	|  bezel image.  The volt decal height does not include the bezel so
	|  gkrellm_panel_configure() did not account for the bezel.
	*/
	pVolt->label->h_panel += bezel_style->border.bottom;
	gkrellm_panel_create(vbox, mon_volt, pVolt);

	draw_volt_bezels();

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(pVolt->drawing_area), "expose_event",
				(GtkSignalFunc) expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(pVolt->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		}
	draw_voltages(TRUE);
	}

static void
create_voltage(GtkWidget *vbox, gint first_create)
	{
	voltage_vbox = vbox;
	create_voltage_monitor(vbox, first_create);
	}

static Monitor	monitor_voltages =
	{
	N_("Voltage"),		/* Voltage config handled in Sensors tab */
	MON_VOLTAGE,		/* Id,  0 if a plugin		*/
	create_voltage,		/* The create function		*/
	update_voltage,		/* The update function		*/
	NULL,				/* The config tab create function	*/
	NULL,				/* Voltage apply handled in sensors apply */

	NULL,				/* Voltage save config is in sensors save */
	NULL,				/* Voltage load config is in sensors load */
	NULL,				/* config keyword - use sensors */

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_voltage_monitor(void)
	{
	style_id = gkrellm_add_meter_style(&monitor_voltages, SENSOR_STYLE_NAME);
	mon_volt = &monitor_voltages;
	return &monitor_voltages;
	}

/* End of voltage monitor */
/* =================================================================== */
/* Unified config for sensor interface and voltage monitor	*/

#define	SENSOR_CONFIG_KEYWORD	"sensor"

static void
create_sensor_alert(Sensor *s)
	{
	if (s->type == SENSOR_VOLTAGE)
		{
		s->alert = gkrellm_alert_create(NULL, s->name,
				_("Sensor Volt Limits"),
				TRUE, TRUE, TRUE, 20, -20, 0.01, 0.5, 2);
		gkrellm_alert_trigger_connect(s->alert, cb_volt_alert_trigger, s);
		}
	else if (s->type == SENSOR_TEMPERATURE)
		{
		s->alert = gkrellm_alert_create(NULL, s->name,
				_("Sensor Temperature Limits (in displayed degree units)"),
				TRUE, FALSE, TRUE, 300, 0, 1.0, 5.0, 1);
		gkrellm_alert_trigger_connect(s->alert, s->cb_alert, s->cb_alert_data);
		}
	else if (s->type == SENSOR_FAN)
		{
		s->alert = gkrellm_alert_create(NULL, s->name,
				_("Sensor Fan RPM Limits"),
				FALSE, TRUE, TRUE, 20000, 0, 100, 1000, 0);
		gkrellm_alert_trigger_connect(s->alert, s->cb_alert, s->cb_alert_data);
		}
	else
		return;
	}


static void
save_sensors_config(FILE *f)
	{
	GList	*list;
	Sensor	*s;

	for (list = sensor_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (s->name && *(s->name))
			{
			fprintf(f, "%s %s %s %.4f %.4f %d %d\n", SENSOR_CONFIG_KEYWORD,
				s->name, sensor_basename(s->path),
				s->factor, s->offset, s->enabled, s->volt_order);
			if (s->alert)
				gkrellm_save_alertconfig(f, s->alert,
						SENSOR_CONFIG_KEYWORD, sensor_basename(s->path));
			}
		}
	for (list = sensor_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (s->vref)
			fprintf(f, "%s vref %s %s\n", SENSOR_CONFIG_KEYWORD,
					sensor_basename(s->path), sensor_basename(s->vref->path));
		}
	fprintf(f, "%s units_fahrenheit %d\n", SENSOR_CONFIG_KEYWORD,
				units_fahrenheit);
	fprintf(f, "%s volt_display_mode %d\n", SENSOR_CONFIG_KEYWORD,
				display_mode);
	}

static void
load_sensors_config(gchar *arg)
	{
	Sensor	*s;
	gchar	config[32], item[CFG_BUFSIZE], item1[CFG_BUFSIZE];
	gchar	name[64], sensor_name[CFG_BUFSIZE];
	gint	n;
	gfloat	f	= 1.0,
			o	= 0.0;
	gint	e	= 0,
			order= -1;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n != 2)
		return;
	if (GK.debug_level & DEBUG_SENSORS)
		printf("load_sensors_config: <%s> <%s>\n", config, item);
	if (!strcmp(config, "units_fahrenheit"))
		sscanf(item, "%d", &units_fahrenheit);
	else if (!strcmp(config, "volt_display_mode"))
		sscanf(item, "%d", &display_mode);
	else if (!strcmp(config, "vref"))
		{
		sscanf(item, "%63s %63s", name, item1);
		s = lookup_sensor_from_basename(name);
		s->vref = lookup_sensor_from_basename(item1);
		}
	else if (!strcmp(config, GKRELLM_ALERTCONFIG_KEYWORD))
		{
		n = sscanf(item, "%63s %[^\n]", name, item1);
		s = lookup_sensor_from_basename(name);
		if (s && n == 2)
			{
			if (!s->alert)
				create_sensor_alert(s);
			gkrellm_load_alertconfig(&s->alert, item1);
			}
		}
	else if (   sscanf(arg, "%63s %s %f %f %d %d",
						name, sensor_name, &f, &o, &e, &order) > 1
			 && (s = map_sensor_name(name, sensor_name)) != NULL
			)
		{
		s->factor = f;
		s->offset = o;
		s->enabled = e;
		if (s->type == SENSOR_VOLTAGE && order >= 0)
			s->volt_order = order;
		s->has_config = TRUE;
		}
	if (display_mode < 0 || display_mode >= N_DISPLAY_MODES)
		display_mode = N_DISPLAY_MODES - 1;
	}


/* ------------------------------------------------------------------- */
static GtkWidget	*fahrenheit_button;

static GtkWidget	*display_mode_button[2];

static void
set_alert_button_state(Sensor *s)
	{
	gboolean	activated;

	if (   (s->type == SENSOR_VOLTAGE && !s->enabled)
		|| (s->type != SENSOR_VOLTAGE && !*(s->name))
	   )
		gtk_widget_set_sensitive(s->alert_button, FALSE);
	else
		gtk_widget_set_sensitive(s->alert_button, TRUE);

	activated = (!s->alert || !s->alert->activated) ? FALSE : TRUE;
	gtk_label_set_text(GTK_LABEL(GTK_BIN(s->alert_button)->child),
			activated ? "*" : "-");
	}

static void
fix_temp_alert(Sensor *s)
	{
	Alert	*a = s->alert;

	if (!a)
		return;
	if (units_fahrenheit)
		{
		if (a->high.warn_limit > 0)
			a->high.warn_limit = a->high.warn_limit * 9.0 / 5.0 + 32.0;
		if (a->high.alarm_limit > 0)
			a->high.alarm_limit = a->high.alarm_limit * 9.0 / 5.0 + 32.0;
		}
	else
		{
		if (a->high.warn_limit > 0)
			a->high.warn_limit = (a->high.warn_limit - 32.0) * 5.0 / 9.0;
		if (a->high.alarm_limit > 0)
			a->high.alarm_limit = (a->high.alarm_limit - 32.0) * 5.0 / 9.0;
		}
	gkrellm_alert_window_destroy(&s->alert);
	}

static void
apply_sensors_config(void)
	{
	GList	*list;
	Sensor	*s;
	gchar	*name;
	gint	i, previous;

	previous = units_fahrenheit;
	units_fahrenheit = GTK_TOGGLE_BUTTON(fahrenheit_button)->active;
	for (list = sensor_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (s->entry)
			{
			name = gkrellm_entry_get_text(&s->entry);
			if (   !*name && s->type == SENSOR_VOLTAGE
				&& s->id < s->voltdefaultsize
			   )
				{
				name = s->voltdefault[s->id].name;
				gtk_entry_set_text(GTK_ENTRY(s->entry), name);
				gtk_label_set_text(
						GTK_LABEL(GTK_BIN(s->select_button)->child), name);
				}
			if (gkrellm_dup_string(&s->name, name) && s->alert)
				{
				if (s->type == SENSOR_VOLTAGE)
					{
					g_free(s->alert->name);
					s->alert->name = g_strdup_printf("%s %s", s->name,
							_("Voltage"));
					}
				else
					{
					gkrellm_alert_destroy(&s->alert);
					set_alert_button_state(s);
					}
				}
			}
		if (s->vref_entry)
			{
			name = gkrellm_entry_get_text(&s->vref_entry);
			s->vref = lookup_vref(s, name);
			gtk_entry_set_text(GTK_ENTRY(s->vref_entry),
						s->vref ? g_basename(s->vref->path) : "");
			}
		if (s->factor_spin_button)
			s->factor = gtk_spin_button_get_value_as_float(
						GTK_SPIN_BUTTON(s->factor_spin_button));
		if (s->offset_spin_button)
			s->offset = gtk_spin_button_get_value_as_float(
						GTK_SPIN_BUTTON(s->offset_spin_button));
		set_alert_button_state(s);
		if (   s->type == SENSOR_TEMPERATURE
			&& previous != units_fahrenheit
			&& s->alert
		   )
			fix_temp_alert(s);
		}

	/* Temp and fan labels must be unique */
	for (list = sensor_list; list; list = list->next)
		{
		s = (Sensor *) list->data;
		if (s->type != SENSOR_TEMPERATURE && s->type != SENSOR_FAN)
			continue;
		if (*(s->name) && s != (Sensor *) get_mapped_sensor(s->name, s->type))
			{
			gkrellm_dup_string(&s->name, "");
			if (s->entry)
				gtk_entry_set_text(GTK_ENTRY(s->entry), s->name);
			if (s->alert)
				{
				gkrellm_alert_destroy(&s->alert);
				set_alert_button_state(s);
				}
			gkrellm_message_window(_("GKrellM Config Error"),
				_("Duplicate label for either a temperature or fan"), NULL);
			}
		}

	previous = display_mode;
	for (i = 0; i < N_DISPLAY_MODES; ++i)
		if (GTK_TOGGLE_BUTTON(display_mode_button[i])->active)
			display_mode = i;

	if (   correction_modified && ! voltages_modified
		&& have_negative_volts != any_negative_volts()
	   )
		voltages_modified = TRUE;
	if (display_mode != previous || voltages_modified)
		{
		destroy_voltage_monitor();
		create_voltage_monitor(voltage_vbox, TRUE);
		}
	else if (correction_modified)
		draw_voltages(FALSE);

	voltages_modified = FALSE;
	correction_modified = FALSE;
	}

static void
cb_correction_modified(void)
	{
	correction_modified = TRUE;
	}

static void
cb_volt_enabled(GtkWidget *button, Sensor *voltage)
	{
	voltage->enabled = !voltage->enabled;
	voltages_modified = TRUE;
	}

static void
cb_volt_entry(GtkWidget *entry, Sensor *voltage)
	{
	gchar	*name;

	voltages_modified = TRUE;
	name = gkrellm_entry_get_text(&entry);
	if (voltage->select_button)
		gtk_label_set_text(
				GTK_LABEL(GTK_BIN(voltage->select_button)->child), name);
	}

static void
cb_vref_entry(GtkWidget *entry, gpointer data)
	{
	voltages_modified = TRUE;
	}

static void
cb_volt_order(GtkWidget *button, Sensor *sv)
	{
	GtkSpinButton	*spin;
	GList			*list;
	Sensor			*sr;
	gint			order, replaced_order;

	voltages_modified = TRUE;
	spin = GTK_SPIN_BUTTON(sv->volt_order_spin_button);
	order = gtk_spin_button_get_value_as_int(spin);
	if (order == sv->volt_order)
		return;
	replaced_order = sv->volt_order;
	sv->volt_order = order;

	for (list = sensor_list; list; list = list->next)
		{
		sr = (Sensor *) list->data;
		if (sr->type != SENSOR_VOLTAGE || sr == sv)
			continue;
		if (sr->volt_order == sv->volt_order)
			{
			sr->volt_order = replaced_order;
			spin = GTK_SPIN_BUTTON(sr->volt_order_spin_button);
			gtk_spin_button_set_value(spin, (gfloat) sr->volt_order);
			break;
			}
		}
	}

static void
cb_sensor_alert_config(Alert *alert, Sensor *sensor)
	{
	if (gkrellm_config_window_shown())
		set_alert_button_state(sensor);
	}

static void
cb_sensor_set_alert(GtkWidget *entry, Sensor *s)
	{
	if (!s->alert)
		create_sensor_alert(s);
	gkrellm_alert_config_connect(s->alert, cb_sensor_alert_config, s);
	gkrellm_alert_config_window(&s->alert);
	}

static gchar	*sensor_info_text0[]	= 
	{
N_("Depending on the hardware sensor implementation of your motherboard,\n"
"there may be temperature, fan, and voltage sensors available to monitor.\n"
"If there are temperature or fan sensors, the data for these may be\n"
"displayed on the CPU and Proc panels.  If there are voltage sensors, the\n"
"voltage data may be displayed on a separate panel below the Proc panel.\n"),

N_("<b>\nSetup\n"),

N_("On the Setup page you can enter a Sensor Label next to a temperature or\n"
"fan Sensor File to make the data from that sensor appear on a CPU or Proc\n"
"panel.  The label must be one of these:\n")
	};

static gchar	*sensor_info_text1[]	= 
	{
N_("\nwhere the \"mb\" label is for linking to the Proc panel.\n\n"),
N_("Voltages are displayed in their own panel so do not need a special label\n"
"to make them appear.  They have a default label already assigned (which may\n"
"be overridden), and are enabled separately in the Voltages tab.\n\n"),

N_("The second function of the Setup page is to assign data scaling factors\n"
"and offsets to the various sensors.  Some fan and voltage sensors will have\n"
"default values, but you should review them for correctness - be sure to \n"
"read the GKrellM README and lm_sensor documetation for more info.  Note\n"
"that wherever documentation refers to possible fan divisors of 2, the\n"
"correct entry here would be a fan factor of 0.5.  Also, any temperature\n"
"Sensor Correction Offset values must be in centigrade units.\n"),
"\n",
N_("There will be a Vref entry for negative voltages if you have a sensor\n"
"chip that uses an external voltage reference for level shifting negative\n"
"input voltages (such as the lm80).  See the README or do a man gkrellm.\n")
	};

static gchar *sensor_name_set[4] =
	{
	"<b>\t\t\"cpu\" or \"mb\"",
	"<b>\t\t\"cpu0\" or \"mb\"",
	"<b>\t\t\"cpu0\", \"cpu1\", or \"mb\"",
	"<b>\t\t\"cpu0\", \"cpu1\"..., or \"mb\"",
	};

static void
create_sensors_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs;
	GtkWidget	*table;
	GtkWidget	*button;
	GtkWidget	*text;
	GtkWidget	*vbox, *vbox1, *hbox, *hbox1, *box;
	GtkWidget	*label;
	GtkWidget	*entry;
	GList		*list;
	Sensor		*sr;
	gchar		*s, *dirname, *chipname;
	gint		i, n;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	table = gtk_table_new(2, 6, FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 0);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 3);

	label = gtk_label_new(_("Sensor Label"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 1, 2);

#if !defined(__FreeBSD__)
	label = gtk_label_new(SENSORS_DIR);
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 0, 1);
#endif
	label = gtk_label_new(_("Sensor Files"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 1, 2);

	label = gtk_label_new(_("Sensor Correction"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 4, 0, 1);
	label = gtk_label_new(_("Factor"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 1, 2);
	label = gtk_label_new(_("Offset"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 3, 4, 1, 2);
	if (need_vref_config)
		{
		label = gtk_label_new(_("Vref"));
		gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 5, 1, 2);
		}
	label = gtk_label_new(_("Alerts"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 5, 6, 1, 2);

	box = gkrellm_scrolled_vbox(vbox, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	table = gtk_table_new(g_list_length(sensor_list), 5, FALSE);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(box), table, FALSE, FALSE, 2);

	for (i = 0, list = sensor_list; list; list = list->next, ++i)
		{
		sr = (Sensor *) list->data;
		entry = gtk_entry_new_with_max_length(8);
		sr->entry = entry;
		gtk_entry_set_text(GTK_ENTRY(entry), sr->name);
		gtk_widget_set_usize(entry, 36, 0);
		gtk_table_attach_defaults(GTK_TABLE(table), entry, 0, 1, i, i + 1);
		if (sr->type == SENSOR_VOLTAGE)
			gtk_signal_connect(GTK_OBJECT(entry), "changed",
						(GtkSignalFunc) cb_volt_entry, sr);

		label = gtk_label_new(sensor_basename(sr->path));
		gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, i, i + 1);

		box = gtk_hbox_new(TRUE, 0);
		gtk_table_attach_defaults(GTK_TABLE(table), box, 2, 3, i, i+1);
		gkrellm_spin_button(box, &sr->factor_spin_button, sr->factor, -1000.0,
					1000.0, 0.01, 1.0, 4, 60, NULL, NULL, FALSE, NULL);
		gtk_signal_connect(GTK_OBJECT(sr->factor_spin_button), "changed",
					(GtkSignalFunc) cb_correction_modified, NULL);
		if (sr->type != SENSOR_FAN)
			{
			n = (sr->type == SENSOR_VOLTAGE) ? 3 : 2;
			box = gtk_hbox_new(TRUE, 0);
			gtk_table_attach_defaults(GTK_TABLE(table), box, 3, 4, i, i+1);
			gkrellm_spin_button(box, &sr->offset_spin_button, sr->offset,
				-10000.0, 10000.0, 1.0, 5.0, n, 60, NULL, NULL, FALSE, NULL);
			gtk_signal_connect(GTK_OBJECT(sr->offset_spin_button), "changed",
					(GtkSignalFunc) cb_correction_modified, NULL);
			}
		if (   sr->type == SENSOR_VOLTAGE
			&& (   (sr->chip_class == gl520 && sr->id == 4)
				|| (sr->chip_class == lm80 && (sr->id == 5 || sr->id == 6))
			   )
		   )
			{	/* These chips use a Vref for their negative inputs.	*/
			entry = gtk_entry_new_with_max_length(8);
			sr->vref_entry = entry;
			gtk_entry_set_text(GTK_ENTRY(entry),
						sr->vref ? g_basename(sr->vref->path) : "");
			gtk_widget_set_usize(entry, 36, 0);
			gtk_table_attach_defaults(GTK_TABLE(table), entry, 4, 5, i, i + 1);
			gtk_signal_connect(GTK_OBJECT(entry), "changed",
						(GtkSignalFunc) cb_vref_entry, NULL);
			}
		box = gtk_hbox_new(FALSE, 0);
		gtk_table_attach_defaults(GTK_TABLE(table), box, 5, 6, i, i+1);
		gkrellm_button_connected(box, &sr->alert_button, TRUE, TRUE, 10,
				cb_sensor_set_alert, sr, "");
		set_alert_button_state(sr);
		}
	gkrellm_check_button(vbox, &fahrenheit_button, units_fahrenheit, FALSE, 2,
				_("Display fahrenheit"));

/* --Voltages tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Voltages"));
	hbox = gtk_hbox_new(TRUE, 0);
	gtk_container_add(GTK_CONTAINER(vbox), hbox);

	vbox1 = gkrellm_framed_vbox(hbox, _("Select"), 4, TRUE, 0, 2);
	vbox1 = gkrellm_scrolled_vbox(vbox1, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	box = vbox1;
	dirname = chipname = NULL;
	for (list = sensor_list; list; list = list->next)
		{
		sr = (Sensor *) list->data;
		if (sr->type != SENSOR_VOLTAGE)
			continue;
		s = sensor_basename(sr->path);
		dirname = g_dirname(s);
		if (*dirname != '.' && (!chipname || strcmp(dirname, chipname)))
			{
			chipname = dirname;
			dirname = NULL;
			box = gkrellm_framed_vbox(vbox1, chipname, 4, TRUE, 0, 2);
			}
		hbox1 = gtk_hbox_new(TRUE, 0);
		gtk_box_pack_start(GTK_BOX(box), hbox1, FALSE, FALSE, 0);
		gkrellm_check_button_connected(hbox1, &sr->select_button, sr->enabled,
				TRUE, TRUE, 0, cb_volt_enabled, sr, sr->name);
		gkrellm_spin_button(hbox1, &sr->volt_order_spin_button, sr->volt_order,
				0, n_volt_sensors - 1, 1.0, 1.0, 0, 40,
				cb_volt_order, sr, FALSE, NULL);
		}
	if (chipname)
		g_free(chipname);
	if (dirname)
		g_free(dirname);
	box = gkrellm_framed_vbox(hbox, _("Display Mode"), 4, FALSE, 0, 2);

	button = gtk_radio_button_new_with_label(NULL,
				_("Normal with labels"));
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, TRUE, 0);
	display_mode_button[DIGITAL_WITH_LABELS] = button;

	button = gtk_radio_button_new_with_label(
				gtk_radio_button_group(GTK_RADIO_BUTTON (button)),
				_("Compact with no labels"));
	gtk_box_pack_start(GTK_BOX(box), button, FALSE, TRUE, 0);
	display_mode_button[DIGITAL_NO_LABELS] = button;

	button = display_mode_button[display_mode];
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
	insert_expanded_filler(box);

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	for (i = 0; i < sizeof(sensor_info_text0) / sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(sensor_info_text0[i]));
	if ((i = n_smp_cpus) > 3)
		i = 3;
	gkrellm_add_info_text_string(text, sensor_name_set[i]);
	for (i = 0; i < sizeof(sensor_info_text1) / sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(sensor_info_text1[i]));
	}


static Monitor	monitor_sensors =
	{
	N_("Sensors"),				/* Name, for config tab.	*/
	-1,					/* Id,  0 if a plugin		*/
	NULL,				/* The create function		*/
	NULL,				/* The update function		*/
	create_sensors_tab,	/* The config tab create function	*/
	apply_sensors_config, /* Apply the config function		*/

	save_sensors_config, /* Save user conifg			*/
	load_sensors_config, /* Load user config			*/
	SENSOR_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_sensors_monitor(void)
	{
	if (!setup_sensor_interface() && !GK.demo)
		return NULL;
	sensor_list = g_list_sort(sensor_list, (GCompareFunc) strcmp_sensor_path);
    monitor_sensors.name = _(monitor_sensors.name);
	mon_sensors = &monitor_sensors;
	return &monitor_sensors;
	}

