/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
|  1/23/2001  Hajimu UMEMOTO added init_freebsd_proc(), improved
|               read_freebsd_proc()
| 10/12/2000  NetBSD code contributed by Anthony Mallet
|		<anthony.mallet@useless-ficus.net> 
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <math.h>


typedef struct
	{
	GtkWidget	*vbox;
	Chart		*chart;
	ChartData	*forks_cd;
	ChartConfig	*chart_config;
	gint		enabled;
	gboolean	extra_info;
	gint		save_label_position;
	Decal		*sensor_decal,
				*fan_decal;

	/* (*read_system_proc_info)() needs to fill in this data	*/
	gint		n_users;
	gint		n_processes;
	gint		n_running;
	gulong		n_forks;
	gfloat		fload;
	}
	ProcMon;

ProcMon	proc;

void	(*read_system_proc_info)();


/* ====== System dependent interface ====================================== */

/* ------- FreeBSD ----------------------------------------------------- */
#if defined(__FreeBSD__)
#include <osreldate.h>
#include <sys/sysctl.h>
#if __FreeBSD_version >= 400000
#include <sys/user.h>
#endif

/*
 * This is ugly, but we need PID_MAX, in anyway.  Since 5.0-RELEASE
 * will have vm.stats.vm.v_forks, this will be obsolete in the future.
 */
#if __FreeBSD_version >= 400000
#define	PID_MAX		99999
#else
#include <sys/signalvar.h>
#define KERNEL
#include <sys/proc.h>
#undef KERNEL
#endif

#include <kvm.h>
#include <limits.h>
#include <paths.h>
#include <utmp.h>

extern	kvm_t	*kvmd;

static struct nlist nl[] = {
#define N_NEXTPID	0
	{ "_nextpid" },
	{ "" }
};

static int	oid_v_forks[CTL_MAXNAME + 2];
static int	oid_v_vforks[CTL_MAXNAME + 2];
static int	oid_v_rforks[CTL_MAXNAME + 2];
static size_t	oid_v_forks_len = sizeof(oid_v_forks);
static size_t	oid_v_vforks_len = sizeof(oid_v_vforks);
static size_t	oid_v_rforks_len = sizeof(oid_v_rforks);
static int	have_v_forks = 0;

static void
init_freebsd_proc()
	{
	static int	oid_name2oid[2] = { 0, 3 };
	static char	*name = "vm.stats.vm.v_forks";
	static char	*vname = "vm.stats.vm.v_vforks";
	static char	*rname = "vm.stats.vm.v_rforks";

	/* check if vm.stats.vm.v_forks is available */
	if (sysctl(oid_name2oid, 2, oid_v_forks, &oid_v_forks_len,
		   (void *)name, strlen(name)) < 0)
		return;
	if (sysctl(oid_name2oid, 2, oid_v_vforks, &oid_v_vforks_len,
		   (void *)vname, strlen(vname)) < 0)
		return;
	if (sysctl(oid_name2oid, 2, oid_v_rforks, &oid_v_rforks_len,
		   (void *)rname, strlen(rname)) < 0)
		return;
	oid_v_forks_len /= sizeof(int);
	oid_v_vforks_len /= sizeof(int);
	oid_v_rforks_len /= sizeof(int);
	++have_v_forks;
	}

static void
read_freebsd_proc()
	{
#if __FreeBSD_version >= 400000
	static int	oid_proc[] = { CTL_KERN, KERN_PROC, KERN_PROC_ALL };
#endif
	double		avenrun;
	static u_int	n_forks = 0, curpid = -1;
	u_int		n_vforks, n_rforks;
	gint		r_forks, r_vforks, r_rforks;
	gint		len;
	gint		nextpid, nforked;
	FILE		*ut;
	struct utmp	utmp;
	struct stat	sb, s;
	static time_t	utmp_mtime;
	gchar		ttybuf[MAXPATHLEN];

	if (getloadavg(&avenrun, 1) > 0)
		proc.fload = avenrun;

	if (have_v_forks)
		{
		/* We don't want to just use sysctlbyname().  Because,
                 * we call it so often. */
		len = sizeof(n_forks);
		r_forks = sysctl(oid_v_forks, oid_v_forks_len,
				 &n_forks, &len, NULL, 0);
		len = sizeof(n_vforks);
		r_vforks = sysctl(oid_v_vforks, oid_v_vforks_len,
				  &n_vforks, &len, NULL, 0);
		len = sizeof(n_rforks);
		r_rforks = sysctl(oid_v_rforks, oid_v_rforks_len,
				  &n_rforks, &len, NULL, 0);
		if (r_forks >= 0 && r_vforks >= 0 && r_rforks >= 0)
			proc.n_forks = n_forks + n_vforks + n_rforks;
		}
	else
		{
		/* workaround: Can I get total number of processes? */
		if (kvmd != NULL)
			{
			if (nl[0].n_type == 0)
				kvm_nlist(kvmd, nl);
			if (nl[0].n_type != 0 &&
			    kvm_read(kvmd, nl[N_NEXTPID].n_value,
				     (char *)&nextpid,
				     sizeof(nextpid)) == sizeof(nextpid))
				{
				if (curpid < 0)
					curpid = nextpid;
				if ((nforked = nextpid - curpid) < 0)
					n_forks += PID_MAX - 100;
				n_forks += nforked;
				curpid = nextpid;
				proc.n_forks = n_forks;
				}
			}
		}

#if __FreeBSD_version >= 400000
	if (sysctl(oid_proc, 3, NULL, &len, NULL, 0) >= 0)
		proc.n_processes = len / sizeof(struct kinfo_proc);
#else
	if (kvmd != NULL)
		kvm_getprocs(kvmd, KERN_PROC_ALL, 0, &proc.n_processes);
#endif

	if (!GK.five_second_tick || !proc.extra_info)
		return;
	if (stat(_PATH_UTMP, &s) != 0 || s.st_mtime == utmp_mtime)
		return;
	if ((ut = fopen(_PATH_UTMP, "r")) != NULL)
		{
		proc.n_users = 0;
		while (fread(&utmp, sizeof(utmp), 1, ut))
			{
			if (utmp.ut_name[0] == '\0')
				continue;
			(void)snprintf(ttybuf, sizeof(ttybuf), "%s/%s",
				       _PATH_DEV, utmp.ut_line);
			/* corrupted record */
			if (stat(ttybuf, &sb))
				continue;
			++proc.n_users;
			}
		(void)fclose(ut);
		}
	utmp_mtime = s.st_mtime;
	}
#endif


/* ------- Linux ----------------------------------------------------- */
#if defined(__linux__)
#include <utmp.h>
#include <paths.h>

#define	PROC_LOADAVG_FILE	"/proc/loadavg"

static void
read_linux_proc(void)
	{
	FILE			*f;
	gchar			buf[160], loadbuf[16];
	struct utmp		*ut;
	struct stat		s;
	static time_t	utmp_mtime;
	gint			n;

	if ((f = fopen(PROC_LOADAVG_FILE, "r")) != NULL)
		{
		/* sscanf(buf, "%f") might fail to convert because for some locales
		|  commas are used for decimal points.
		*/
		fgets(buf, sizeof(buf), f);
		n = sscanf(buf,"%f %*f %*f %d/%d %lu", &proc.fload,
					&proc.n_running, &proc.n_processes, &proc.n_forks);
		if (n != 4)
			{
			if (sscanf(buf,"%16s %*d.%*d %*d.%*d %d/%d %lu", loadbuf,
					&proc.n_running, &proc.n_processes, &proc.n_forks) == 4)
				proc.fload = locale_float_fix(loadbuf);
			}
		fclose(f);
		}
	if (GK.five_second_tick && proc.extra_info)
		{
		if (stat(_PATH_UTMP, &s) == 0 && s.st_mtime != utmp_mtime)
			{
			proc.n_users = 0;
			setutent();
			while ((ut = getutent()) != NULL)
				if (ut->ut_type == USER_PROCESS && ut->ut_name[0] != '\0')
					++proc.n_users;
			endutent();
			utmp_mtime = s.st_mtime;
			}
		}
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__)

#include <sys/proc.h>
#include <sys/sysctl.h>
#include <uvm/uvm_extern.h>
#include <kvm.h>

#include <utmp.h>

static struct nlist nl[] = {
#define X_UVM_EXP    0
   { "_uvmexp" },
   { NULL }
};

extern	kvm_t	*kvmd;

void
read_netbsd_proc()
{
   static int mib[] = { CTL_KERN, KERN_PROC, KERN_PROC_ALL };
   double avenrun;
   static time_t utmp_mtime;
   struct utmp utmp;
   struct stat s;
   struct uvmexp *uvmexp;
   int len, i;
   FILE *ut;

   if (sysctl(mib, 3, NULL, &len, NULL, 0) >= 0) {
      proc.n_processes = len / sizeof(struct kinfo_proc);
   }

   /* get name list if it is not done yet */
   if (kvmd == NULL) return;
   if (nl[0].n_type == 0) kvm_nlist(kvmd, nl);

   if (nl[0].n_type != 0) {
      uvmexp = (struct uvmexp *)nl[X_UVM_EXP].n_value;
      if (kvm_read(kvmd, (u_long)&uvmexp->forks, &i, sizeof(i)) == sizeof(i))
	 proc.n_forks = i;
   }

   if (!GK.second_tick)	/* Only one read per second */
      return;

   if (getloadavg(&avenrun, 1) > 0) proc.fload = avenrun;

   if (proc.extra_info) {
      if (stat(_PATH_UTMP, &s) == 0 && s.st_mtime != utmp_mtime) {
	 if ((ut = fopen(_PATH_UTMP, "r")) != NULL) {
	    proc.n_users = 0;
	    while (fread(&utmp, sizeof(utmp), 1, ut)) {
	       if (utmp.ut_name[0] == '\0') continue;
	       ++proc.n_users;
	    }
	    (void)fclose(ut);
	 }
	 utmp_mtime = s.st_mtime;
      }
   }
}

#endif /* __NetBSD__ || __OpenBSD__ */

/* ----- Solaris ----------------------------------------------------- */
#if defined(__solaris__) 

#include <utmp.h>
#include <sys/types.h>
#include <dirent.h>
#include <sys/loadavg.h>
#include <kstat.h>
#include <fcntl.h>
#include <kvm.h>
#include <sys/sysinfo.h>

static void
read_solaris_proc() 
{

    static struct utmp *utmpp;
    double avenrun[LOADAVG_NSTATS];

    int last_pid;
    extern kstat_ctl_t *kc;
    kstat_t *ksp;
    kstat_named_t *knp;

    extern kvm_t *kd;
    extern struct nlist nl[];

    if (!GK.second_tick) /* Only one read per second */
        return;

    if (getloadavg(avenrun, LOADAVG_NSTATS) > 0)
        proc.fload = avenrun[LOADAVG_1MIN];

    if (kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return;
    }
    ksp = kstat_lookup(kc, "unix", -1, "system_misc");
    if (ksp && kstat_read(kc, ksp, NULL) >= 0) {
        knp = (kstat_named_t *)kstat_data_lookup(ksp, "nproc");
        if (knp) { 
            proc.n_processes = knp->value.ui32;
        }
    }

    if (kd) {
        if (kvm_kread(kd, nl[0].n_value, (char *)&last_pid, sizeof(int)) != -1)
            proc.n_forks = last_pid;
    } else {
        proc.n_forks = 0;
    }

    if (!GK.five_second_tick || !proc.extra_info) 
        return;

    proc.n_users = 0;
    setutent();
    while ((utmpp = getutent()) != NULL) {
        if (utmpp->ut_type == USER_PROCESS && utmpp->ut_name[0] != '\0')
            proc.n_users++;
    }    

    /* NOTE: code to get 'n_running' is not implemented (stays untouched).
     * but it wouldn't do any harm since nobody seems to refer to it.
     */
}

#endif /* __solaris__ */

/* ----- Others ------------------------------------------------------ */
#if defined(USE_LIBGTOP)

#include <glibtop/loadavg.h>

static void
read_glibtop_proc(void)
	{
	glibtop_loadavg		glt_loadavg;

	/* If this is expensive, may want to second_tick it
	*/
	glibtop_get_loadavg (&glt_loadavg);
	proc.fload = (float) glt_loadavg.loadavg[0];

	/* Not all systems fill in the rest of these
	*/
	proc.n_running = (gint) glt_loadavg.nr_running;
	proc.n_processes = (gint) glt_loadavg.nr_tasks;
	proc.n_forks = (gint) glt_loadavg.last_pid;

	proc.n_users = -1;
	}
#endif


/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_proc_interface(void)
	{
#if defined(__FreeBSD__)
	init_freebsd_proc();
	read_system_proc_info = read_freebsd_proc;
#elif defined(__linux__)
	read_system_proc_info = read_linux_proc;
#elif defined(__NetBSD__) || defined(__OpenBSD__)
	read_system_proc_info = read_netbsd_proc;
#elif defined(__solaris__)
        read_system_proc_info = read_solaris_proc;
#else
	read_system_proc_info = read_glibtop_proc;
#endif
	return TRUE;
	}

/* ======================================================================== */

  /* Charts are integer only and load is a small real, so scale all load
  |  reading by 100.
  */
#define	LOAD_SCALING	100.0

static Monitor	*mon_proc;

static Launcher	proc_launch;

static gint		style_id;
static gint		mapped_sensors;
static gint		sensor_separate_mode;
static gint		fork_scaling = 1;
static gchar	*text_format;

static void
format_chart_text(ProcMon *p, gchar *buf, gint size)
	{
	Chart	*cp;
	gchar	c, *s;
	gint	len, value;

	--size;
	*buf = '\0';
	cp = p->chart;
	for (s = text_format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			value = -1;
			if ((c = *(s + 1)) == 'L')
				len = snprintf(buf, size, "%.1f",
					(gfloat) gkrellm_get_chart_scalemax(cp) / LOAD_SCALING);
			else if (c == 'F' && fork_scaling > 0)
				len = snprintf(buf, size, "%d",
						gkrellm_get_chart_scalemax(cp) / fork_scaling);
			else if (c == 'l')
				len = snprintf(buf, size, "%.1f", p->fload);
			else if (c == 'p')
				value = p->n_processes;
			else if (c == 'u')
				value = p->n_users;
			else if (c == 'f')
				value = gkrellm_get_current_chartdata(p->forks_cd)
						/ fork_scaling;
			if (value >= 0)
				len = snprintf(buf, size, "%d", value);
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';	
	}

static void
draw_proc_extra(void)
	{
	gchar		buf[128];

	if (!proc.chart || !proc.extra_info)
		return;
	format_chart_text(&proc, buf, sizeof(buf));
	gkrellm_draw_chart_text(proc.chart, style_id, buf);
	}

static void
refresh_proc_chart(Chart *cp)
	{
	if (proc.enabled)
		{
		gkrellm_draw_chartdata(cp);
		if (proc.extra_info)
			draw_proc_extra();
		gkrellm_draw_chart_to_screen(cp);
		}
	}

static void
draw_sensor_decals(void)
	{
	Panel	*p		= proc.chart->panel;
	gchar	*name	= "mb";
	gchar	units;
	gfloat	t, f;
	gint	toggle;

	if (   sensor_separate_mode
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& (mapped_sensors & SENSOR_FAN)
	   )
		{
		gkrellm_sensor_read_temperature(name, &t, &units);
		gkrellm_sensor_draw_temperature_decal(p, proc.sensor_decal, t, units);
		gkrellm_sensor_read_fan(name, &f);
		gkrellm_sensor_draw_fan_decal(p, proc.fan_decal, f);
		}
	else
		{
		toggle = time_now & 2;
		if (   (mapped_sensors & SENSOR_FAN)
			&& (toggle || !(mapped_sensors & SENSOR_TEMPERATURE))
		   )
			{
			gkrellm_sensor_read_fan(name, &f);
			gkrellm_sensor_draw_fan_decal(p, proc.sensor_decal, f);
			}
		else if (   (mapped_sensors & SENSOR_TEMPERATURE)
				 && (!toggle || !(mapped_sensors & SENSOR_FAN))
				)
			{
			gkrellm_sensor_read_temperature(name, &t, &units);
			gkrellm_sensor_draw_temperature_decal(p, proc.sensor_decal, t,
					units);
			}
		}
	}

static void
update_proc(void)
	{
	Chart	*cp;
	Panel	*p;
	gint	load;

	if (!proc.enabled)
		return;
	(*read_system_proc_info)();

	cp = proc.chart;
	p = cp->panel;
	gkrellm_update_krell(p, KRELL(p), proc.n_forks);
	gkrellm_draw_panel_layers(p);

	if (GK.second_tick)
		{
		/* Scale load since it is a small real and charts are integer only.
		|  Scale the forks number by fork_scaling.  See setup_proc_scaling().
		*/
		load = (int) (LOAD_SCALING * proc.fload);
		gkrellm_store_chartdata(cp, 0, load, fork_scaling * proc.n_forks);
		refresh_proc_chart(cp);
		}
	if (   (GK.two_second_tick && !sensor_separate_mode)
		|| (GK.five_second_tick && sensor_separate_mode)
	   )
		draw_sensor_decals();
	}


static gint
proc_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	Chart		*cp		= proc.chart;
	GdkPixmap	*pixmap	= NULL;

	if (cp)
		{
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		}
	if (pixmap)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
	return FALSE;
	}


static gint
cb_proc_extra(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
		{
		proc.extra_info = !proc.extra_info;
		gkrellm_config_modified();
		refresh_proc_chart(proc.chart);
		}
	else if (   ev->button == 3
			 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
			)
		gkrellm_chartconfig_window_create(proc.chart);
	return TRUE;
	}

static void
setup_proc_scaling(void)
	{
	Chart		*cp		= proc.chart;
	gint		grids, res, new_fork_scaling;

	if (!cp)
		return;

	grids = gkrellm_get_chartconfig_fixed_grids(cp->config);
	if (!grids)
		grids = FULL_SCALE_GRIDS;

	res = gkrellm_get_chartconfig_grid_resolution(cp->config);

	/* Since grid_resolution is set for load, set krell_full_scale explicitely
	|  to get what I want, which is 10 forks full scale.
	|  When res or number of grids is changed, scale all fork data to keep a
	|  fixed 50 forks/sec max on the chart.
	*/
	KRELL(cp->panel)->full_scale = 10;

	new_fork_scaling = grids * res / 50;
	if (new_fork_scaling < 1)		/* shouldn't happen... */
		new_fork_scaling = 1;

	/* If load grid_resolution changed, scale all fork data to keep the
	|  constant 50 forks/sec.
	*/
	if (fork_scaling != new_fork_scaling)
		{
		/* When called as a callback a chart refresh will follow, but I'll
		|  need a rescale here.
		*/
		gkrellm_scale_chartdata(proc.forks_cd, new_fork_scaling, fork_scaling);
		gkrellm_rescale_chart(proc.chart);
		}
	fork_scaling = new_fork_scaling;
	}

static void
destroy_proc_monitor(void)
	{
	Chart	*cp		= proc.chart;

	if (proc_launch.button)
		gkrellm_destroy_button(proc_launch.button);
	proc_launch.button = NULL;
    proc_launch.tooltip = NULL;
	gkrellm_chart_destroy(cp);
	proc.chart = NULL;
	proc.enabled = FALSE;
	}

static void
cb_mb_temp_alert_trigger(Alert *alert, ProcMon *proc)
	{
	AlertDecal	*ad;
	Decal		*d;

	if (alert && proc && proc->chart)
		{
		ad = &alert->ad;
		d = proc->sensor_decal;
		if (d)
			{
			ad->x = d->x - 1;
			ad->y = d->y - 1;
			ad->w = d->w + 2;
			ad->h = d->h + 2;
			gkrellm_render_default_alert_decal(alert);
			}
		alert->panel = proc->chart->panel;
		}
	}

static void
cb_mb_fan_alert_trigger(Alert *alert, ProcMon *proc)
	{
	AlertDecal	*ad;
	Decal		*d;

	if (alert && proc && proc->chart)
		{
		ad = &alert->ad;
		if (sensor_separate_mode)
			d = proc->fan_decal;
		else
			d = proc->sensor_decal;
		if (d)
			{
			ad->x = d->x - 1;
			ad->y = d->y - 1;
			ad->w = d->w + 2;
			ad->h = d->h + 2;
			gkrellm_render_default_alert_decal(alert);
			}
		alert->panel = proc->chart->panel;
		}
	}

  /* Next routine is same as in cpu.c - perhaps should make into one?*/
  /* How to decide when to make the sensor_decal and fan_decal visible.
  |  The sensor_decal can show temp values, fan values, or alternating
  |  temp/fan values.  The fan_decal only shows fan values when non
  |  alternating mode is selected. The sensors are mapped in sensors.c
  |
  |   Sensor and fan decal display truth table:
  |   |-----decals visible----||--sensors mapped--|   separate  |
  |	  |sensor_decal  fan_decal||   temp     fan   |    mode     |
  |   |-----------------------||--------------------------------|
  |   |     0           0     ||     0       0          0       |
  |   |     1           0     ||     1       0          0       |
  |   |     1           0     ||     0       1          0       |
  |   |     1           0     ||     1       1          0       |
  |   |     0           0     ||     0       0          1       |
  |   |     1           0     ||     1       0          1       |
  |   |     1           0     ||     0       1          1       |
  |   |     1           1     ||     1       1          1       |
  |   |----------------------------------------------------------
  */
static gint
adjust_sensors_display(gint force)
	{
	Panel	*p;
	Decal	*ds, *df;
	Alert	*alert;
	gint	position	= 0;

	ds = proc.sensor_decal;
	df = proc.fan_decal;
	if (!ds || !df)
		return FALSE;
	/* The test for state change is decal state vs success at reading
	|  a temperature.
	*/
	p = proc.chart->panel;
	mapped_sensors = 0;
	if (!GK.demo)
		{
		gkrellm_mapped_sensor_alert_connect("mb",
				SENSOR_TEMPERATURE, cb_mb_temp_alert_trigger, &proc);
		gkrellm_mapped_sensor_alert_connect("mb",
				SENSOR_FAN, cb_mb_fan_alert_trigger, &proc);
		}

	/* If a fan alert is triggered, turn it off in case fan decal being used
	|  is changed.  The alert will just retrigger at next fan update.
	*/
	alert = gkrellm_mapped_sensor_alert("mb", SENSOR_FAN);
	gkrellm_reset_alert(alert);

	if (get_mapped_sensor("mb", SENSOR_TEMPERATURE) || GK.demo)
		mapped_sensors |= SENSOR_TEMPERATURE;
	if (get_mapped_sensor("mb", SENSOR_FAN) || GK.demo)
		mapped_sensors |= SENSOR_FAN;

	if (mapped_sensors & (SENSOR_TEMPERATURE | SENSOR_FAN))
		{
		if (! gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_visible(p, ds);
		position = 0;
		}
	else
		{
		if (gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_invisible(p, ds);
		position = proc.save_label_position;
		}
	if (   (mapped_sensors & SENSOR_FAN)
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& sensor_separate_mode
	   )
		{
		if (! gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_visible(p, df);
		position = -1;
		}
	else
		{
		if (gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_invisible(p, df);
		}
	if (position != p->label->position || force)
		{
		if (proc.save_label_position >= 0)	/* Reassign position only if the */
			p->label->position = position;	/* original label was visible.   */
		gkrellm_draw_panel_label(p, gkrellm_bg_panel_image(style_id));
		draw_sensor_decals();
		gkrellm_draw_panel_layers(p);
		return TRUE;
		}
	return FALSE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_proc);
	return TRUE;
	}

static void
create_proc_monitor(GtkWidget *vbox, gint first_create)
	{
	Chart		*cp;
	ChartData	*cd;
	ChartConfig	*cf;
	Panel		*p;
	Style		*style;

	if (first_create)
		{
		proc.chart = gkrellm_chart_new0();
		proc.chart->panel = gkrellm_panel_new0();
		}
	cp = proc.chart;
	p = cp->panel;

	style = gkrellm_panel_style(style_id);
	gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

	gkrellm_chart_create(proc.vbox, mon_proc, cp, &proc.chart_config);
	gkrellm_set_draw_chart_function(cp, refresh_proc_chart, proc.chart);
	cd = gkrellm_add_default_chartdata(cp, _("Load"));
	gkrellm_monotonic_chartdata(cd, FALSE);
	gkrellm_set_chartdata_draw_style_default(cd, CHARTDATA_LINE);
	proc.forks_cd = gkrellm_add_default_chartdata(cp, _("Forks"));
	gkrellm_set_chartdata_flags(proc.forks_cd, CHARTDATA_ALLOW_HIDE);

	cf = cp->config;
	gkrellm_chartconfig_fixed_grids_connect(cf, setup_proc_scaling, NULL);
	gkrellm_chartconfig_grid_resolution_connect(cf, setup_proc_scaling, NULL);
	gkrellm_chartconfig_grid_resolution_adjustment(cf, FALSE,
			LOAD_SCALING, 0.5, 5.0, 0.5, 0.5, 1, 50);
	gkrellm_chartconfig_grid_resolution_label(cf,
					_("Average process load per minute"));

	gkrellm_alloc_chartdata(cp);
	setup_proc_scaling();

	/* I put motherboard temp on Proc panel (if temperature sensors found)
	*/
	sensor_create_decals(p, style_id,
					&proc.sensor_decal, &proc.fan_decal);

	gkrellm_panel_configure(p, _("Proc"), style);

	gkrellm_panel_create(proc.vbox, mon_proc, p);

	proc.save_label_position = p->label->position;
	if (proc.sensor_decal)
		adjust_sensors_display(TRUE);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
				(GtkSignalFunc) proc_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area), "expose_event",
				(GtkSignalFunc) proc_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),"button_press_event",
				(GtkSignalFunc) cb_proc_extra, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		}
	else
		refresh_proc_chart(cp);
	gkrellm_setup_launcher(p, &proc_launch, CHART_PANEL_TYPE, 4);
	}

static void
create_proc(GtkWidget *vbox, gint first_create)
	{
	proc.vbox = vbox;
	if (proc.enabled)
		create_proc_monitor(proc.vbox, first_create);
	}



#define	PROC_CONFIG_KEYWORD	"proc"

static void
save_proc_config(FILE *f)
	{
	fprintf(f, "%s enable %d %d\n", PROC_CONFIG_KEYWORD,
				proc.enabled, proc.extra_info);
	fprintf(f, "%s launch %s\n", PROC_CONFIG_KEYWORD, proc_launch.command);
	fprintf(f, "%s tooltip_comment %s\n", PROC_CONFIG_KEYWORD,
				proc_launch.tooltip_comment);
	fprintf(f, "%s sensor_mode %d\n", PROC_CONFIG_KEYWORD,
				sensor_separate_mode);
	fprintf(f, "%s text_format %s\n", PROC_CONFIG_KEYWORD, text_format);
	gkrellm_save_chartconfig(f, proc.chart_config,
				PROC_CONFIG_KEYWORD, NULL);
	}

static void
load_proc_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (!strcmp(config, "enable"))
			sscanf(item, "%d %d\n", &proc.enabled, &proc.extra_info);
		else if (!strcmp(config, "launch"))
			proc_launch.command = g_strdup(item);
		else if (!strcmp(config, "tooltip_comment"))
			proc_launch.tooltip_comment = g_strdup(item);
		else if (!strcmp(config, "sensor_mode"))
			sscanf(item, "%d\n", &sensor_separate_mode);
		else if (!strcmp(config, "text_format"))
			gkrellm_dup_string(&text_format, item);
		else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
			gkrellm_load_chartconfig(&proc.chart_config, item, 2);
		}
	}

/* ---------------------------------------------------------------------- */
#define DEFAULT_TEXT_FORMAT	"\\w88\\a$p\\f procs\\n\\e$u\\f users"

static GtkWidget	*proc_enable_button,
					*sensor_separate_button;
static GtkWidget	*proc_launch_entry,
					*proc_tooltip_entry;

static GtkWidget	*text_format_combo;

static void
apply_proc_config(void)
	{
	gchar		*s;
	gint		pe;

	if (sensor_separate_button)
		sensor_separate_mode =
				GTK_TOGGLE_BUTTON(sensor_separate_button)->active;

	pe = GTK_TOGGLE_BUTTON(proc_enable_button)->active;
	if (pe && ! proc.enabled)
		create_proc_monitor(proc.vbox, TRUE);
	else if (! pe && proc.enabled)
		destroy_proc_monitor();
	proc.enabled = pe;

	if (proc.enabled)
		{
		gkrellm_apply_launcher(&proc_launch_entry, &proc_tooltip_entry,
					proc.chart->panel, &proc_launch, gkrellm_launch_button_cb);
		if (adjust_sensors_display(FALSE) && proc_launch.button)
			{
			gkrellm_destroy_button(proc_launch.button);
			proc_launch.button = 
				gkrellm_put_label_in_panel_button(proc.chart->panel,
					gkrellm_launch_button_cb, &proc_launch, proc_launch.pad);
			}
		}
	s = gkrellm_entry_get_text(&(GTK_COMBO(text_format_combo)->entry));
	if (gkrellm_dup_string(&text_format, s))
		gkrellm_refresh_chart(proc.chart);
	}


static gchar	*proc_info_text[] =
{
N_("<b>Proc Chart"),
"\n",
N_("The krell shows process forks with a full scale value of 10 forks.\n"),
N_("While both load and fork data are drawn on the chart, the grid\n"
"resolution can be set for load only.  The resolution per grid for forks is\n"
"fixed at 10 when using the auto number of grids mode, and at 50 divided by\n"
"the number of grids when using a fixed number of grids mode.\n"),
"\n",
N_("<b>Chart Labels\n"),
N_("Substitution variables for the format string for chart labels:\n"),
N_("\t$L    maximum chart value (load)\n"),
N_("\t$F    maximum chart value (forks)\n"),
N_("\t$l    load\n"),
N_("\t$f    forks\n"),
N_("\t$p    processes\n"),
N_("\t$u    users\n")
};

static void
create_proc_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs, *table, *vbox, *vbox1, *hbox, *text;
	GList		*list;
	gint		i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* ---Options tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Options"), 4, FALSE, 0, 2);
	gkrellm_check_button(vbox1, &proc_enable_button, proc.enabled, FALSE, 4,
			_("Enable Proc chart"));

	if (gkrellm_sensors_available())
		gkrellm_check_button(vbox1, &sensor_separate_button,
					sensor_separate_mode, FALSE, 0,
		_("Draw fan and temperature values separately (not alternating)."));

	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Chart Labels"),
				4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	text_format_combo = gtk_combo_new();
	gtk_widget_set_usize (GTK_WIDGET(text_format_combo), 350, 0);
	gtk_box_pack_start(GTK_BOX(hbox), text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, _(DEFAULT_TEXT_FORMAT));
	list = g_list_append(list,
			_("\\f$L\\r\\f$F \\w88\\b\\p\\a$p\\f procs\\n\\e$u\\f users"));
	gtk_combo_set_popdown_strings(GTK_COMBO(text_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(text_format_combo)->entry),
			text_format);

	vbox1 = gkrellm_framed_vbox_end(vbox, _("Launch Commands"), 4, FALSE, 0, 2);
	table = gkrellm_launcher_table_new(vbox1, 1);
	gkrellm_config_launcher(table, 0,  &proc_launch_entry, &proc_tooltip_entry,
			_("Proc"), &proc_launch);

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(proc_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(proc_info_text[i]));
	}

static Monitor	monitor_proc =
	{
	N_("Proc"),				/* Name, for config tab.	*/
	MON_PROC,			/* Id,  0 if a plugin		*/
	create_proc,		/* The create function		*/
	update_proc,		/* The update function		*/
	create_proc_tab,	/* The config tab create function	*/
	apply_proc_config,	/* Apply the config function		*/

	save_proc_config,	/* Save user conifg			*/
	load_proc_config,	/* Load user config			*/
	PROC_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_proc_monitor(void)
	{
	ChartConfig	*cf;

	monitor_proc.name = _(monitor_proc.name);
	proc.enabled = TRUE;
	proc.extra_info = TRUE;
	style_id = gkrellm_add_chart_style(&monitor_proc, PROC_STYLE_NAME);
	text_format = g_strdup(_(DEFAULT_TEXT_FORMAT));
	mon_proc = &monitor_proc;
	if (setup_proc_interface())
		{
		/* Set chart config defaults.  Turn off auto grid resolution and
		|  don't let user config it back on.
		*/
		cf = proc.chart_config = gkrellm_chartconfig_new0();
		gkrellm_set_chartconfig_grid_resolution(cf, 100);
		gkrellm_set_chartconfig_auto_grid_resolution(cf, FALSE);
		gkrellm_set_chartconfig_flags(cf, NO_CONFIG_AUTO_GRID_RESOLUTION);

		return &monitor_proc;
		}
	return NULL;
	}

