/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
| 10/29/2001  Tomas Ogren <stric@cs.umu.se> Solaris patch avoids adding not
|               found disks to the disk list.
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
| 10/12/2000  NetBSD code contributed by Anthony Mallet
|               <anthony.mallet@useless-ficus.net>
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/


#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "cpu_disk.h"

#define	MIN_GRID_RES		10
#define	MAX_GRID_RES		500000
#define	DEFAULT_GRID_RES	2000

void	(*read_system_disk_info)();

static DiskMon	*lookup_disk_by_name(gchar *);

gint		n_disks;
gboolean	using_DiskN_names;
GList		*disk_mon_list;



/* ====== System dependent interface ====================================== */

/* ----- FreeBSD ----------------------------------------------------- */

#if defined(__FreeBSD__)
#include <osreldate.h>
#include <sys/dkstat.h>
#if __FreeBSD_version >= 300000
#include <devstat.h>
static struct statinfo	statinfo_cur;
#else
#include <kvm.h>
static struct nlist nl[] = {
#define N_DK_NDRIVE	0
	{ "_dk_ndrive" },
#define N_DK_XFER	1
	{ "_dk_xfer" },
	{ "" }
};

extern	kvm_t	*kvmd;
#endif

static DiskMon *
add_disk_device(gint major, gint minor)
	{
	return NULL;
	}

#if __FreeBSD_version < 300000
static void
read_freebsd_disk()
	{
	static gint	data_read_tick = -1;
	int		ndevs;
	long		*cur_dk_xfer;
	int		dn;
	GList		*list;
	DiskMon		*disk;
   
	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (kvmd == NULL)
		return;
	if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
		return;
	(void) kvm_read(kvmd, nl[N_DK_NDRIVE].n_value,
			(char *)&ndevs, sizeof(ndevs));
	if (ndevs <= 0)
		return;
	if ((cur_dk_xfer = calloc(ndevs, sizeof(long))) == NULL)
		return;
	if (kvm_read(kvmd, nl[N_DK_XFER].n_value, (char *)cur_dk_xfer,
		     ndevs * sizeof(long)) == ndevs * sizeof(long))
		{
		composite_disk->rblk = 0;
		composite_disk->wblk = 0;
		dn = 0;
		for (list = disk_mon_list->next; list; list = list->next)
			{
			if (dn >= ndevs)
				break;
			disk = (DiskMon *) list->data;
			disk->rblk = 0;
			disk->wblk = cur_dk_xfer[dn];
			composite_disk->wblk += disk->wblk;
			++dn;
			}
		}
	free(cur_dk_xfer);
	}

static void
register_disks(gboolean re_checking)
	{
	gint	i;
	DiskMon	*disk;

	if (re_checking)
		return;
	if (kvmd == NULL)
		return;
	if (kvm_nlist(kvmd, nl) >= 0 && nl[0].n_type != 0)
		(void) kvm_read(kvmd, nl[N_DK_NDRIVE].n_value,
				(char *)&n_disks, sizeof(n_disks));
	for (i = 0; i < n_disks; ++i)
		{
		disk = g_new0(DiskMon, 1);
		disk_mon_list = g_list_append(disk_mon_list, disk);
		disk->name = g_strdup_printf("%s%c", _("Disk"), 'A' + i);
		}
	using_DiskN_names = TRUE;
	}
#else
static void
read_freebsd_disk()
	{
	static gint		data_read_tick = -1;
	int			ndevs;
	int			num_selected;
	int			num_selections;
	int			maxshowdevs = 10;
	struct device_selection	*dev_select = NULL;
	long			select_generation;
	int			dn;
	DiskMon			*disk;

	if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
		return;
	data_read_tick = GK.timer_ticks;

	if (getdevs(&statinfo_cur) < 0)
		return;
	ndevs = statinfo_cur.dinfo->numdevs;
	if (selectdevs(&dev_select, &num_selected, &num_selections,
		       &select_generation, statinfo_cur.dinfo->generation,
		       statinfo_cur.dinfo->devices, ndevs,
		       NULL, 0, NULL, 0,
		       DS_SELECT_ONLY, maxshowdevs, 1) >= 0)
		{
		composite_disk->rblk = 0;
		composite_disk->wblk = 0;
		for (dn = 0; dn < ndevs; ++dn)
			{
			int		di;
			struct devstat	*dev;
			int		block_size;
			int		blocks_read, blocks_written;

			di = dev_select[dn].position;
			dev = &statinfo_cur.dinfo->devices[di];
			block_size = (dev->block_size > 0)
				   ? dev->block_size : 512;
			blocks_read = dev->bytes_read / block_size;
			blocks_written = dev->bytes_written / block_size;

			if ((disk = lookup_disk_by_device(dev->device_number,
							  0)) != NULL)
				{
				disk->rblk = blocks_read;
				disk->wblk = blocks_written;
				}
			composite_disk->rblk += blocks_read;
			composite_disk->wblk += blocks_written;
			}
		free(dev_select);
		}
	}

static void
register_disks(gboolean re_checking)
	{
	DiskMon			*disk;
	GList			*list;
	static gint		initted = FALSE;
	int			num_selected;
	int			num_selections;
	int			maxshowdevs = 10;
	struct device_selection	*dev_select = NULL;
	long			select_generation;
	gint			dn, di, i;
	struct devstat		*dev;

	if (!initted)
		{
		n_disks = getnumdevs();
		bzero(&statinfo_cur, sizeof(statinfo_cur));
		statinfo_cur.dinfo = (struct devinfo *)malloc(sizeof(struct devinfo));
		bzero(statinfo_cur.dinfo, sizeof(struct devinfo));
		initted = TRUE;
		}
	if (getdevs(&statinfo_cur) < 0)
		return;
	n_disks = statinfo_cur.dinfo->numdevs;
	if (selectdevs(&dev_select, &num_selected, &num_selections,
		       &select_generation,
		       statinfo_cur.dinfo->generation,
		       statinfo_cur.dinfo->devices, n_disks,
		       NULL, 0, NULL, 0,
		       DS_SELECT_ONLY, maxshowdevs, 1) < 0)
		return;
	for (dn = 0; dn < n_disks; ++dn)
		{
		di = dev_select[dn].position;
		dev = &statinfo_cur.dinfo->devices[di];
		if (lookup_disk_by_device(dev->device_number, 0) != NULL)
			continue;
		disk = g_new0(DiskMon, 1);
		disk->name = g_strdup_printf("%s%d", dev->device_name,
					     dev->unit_number);
		disk->major = dev->device_number;
		disk->minor = 0;
		disk->order = dev->device_type + 1;	/* Skip the composite disk */
		i = 1;
		for (list = disk_mon_list->next; list; list = list->next, ++i)
			if (disk->order < ((DiskMon *) list->data)->order)
				break;
		disk_mon_list = g_list_insert(disk_mon_list, disk, i);
		}
	free(dev_select);
	using_DiskN_names = TRUE;
	}
#endif
#endif


/* ----- Linux ----------------------------------------------------- */

/* See array.c (2.2.x) or proc_misc.c (2.3.99+) in /usr/src/linux/fs/proc
|  to see how info for disks is limited to 4 devices.
|  See drive_stat_acct() in in /usr/src/linux/drivers/block/ll_rw_blk.c
|  to see how device numbers for ide and scsi are mapped into one of the
|  four disk devices reported in /proc.
|  The devices currently reported are DAC960_MAJOR (55), SCSI_DISK0_MAJOR (8),
|  IDE0_MAJOR (3), and IDE1_MAJOR (22)
*/

#if defined(__linux__)
#define	PROC_STAT_FILE	"/proc/stat"

#include <linux/major.h>
#if ! defined (SCSI_DISK0_MAJOR)
#define SCSI_DISK0_MAJOR	8
#endif
#if ! defined (MD_MAJOR)
#define MD_MAJOR	9
#endif

/* The read_proc_stat() routine is in cpu.c		*/

struct _disk_name_map
	{
	gchar	*name;
	gint	major;
	gchar	suffix_base;
	};

  /* Disk charts will appear in GKrellm in the same order as this table.
  */
struct _disk_name_map	disk_name_map[] =
	{
	{"hd",	IDE0_MAJOR,			'a' },		/* 3:  hda, hdb */
	{"hd",	IDE1_MAJOR,			'c' },		/* 22: hdc, hdd */
	{"hd",	IDE2_MAJOR,			'e' },		/* 33: hde, hdf */
	{"hd",	IDE3_MAJOR,			'g' },		/* 34: hdg, hdh */
	{"sd",	SCSI_DISK0_MAJOR,	'a' },		/* 8:  sda-sdh */
	{"sg",	SCSI_GENERIC_MAJOR,	'0' },		/* 21: sg0-sg16 */
	{"scd",	SCSI_CDROM_MAJOR,	'0' },		/* 11: scd0-scd16 */
	{"md",	MD_MAJOR,			'0' },		/* 9:  md0-md3 */
	{"fd",	FLOPPY_MAJOR,		'0' },		/* 2:  fd0-fd3  */
	};

static DiskMon *
add_disk_device(gint major, gint i_disk)
	{
	struct _disk_name_map	*dm	= NULL;
	DiskMon	*disk;
	GList	*list;
	gint	i;
	gchar	buf[32], suffix;

	for (i = 0; i < sizeof(disk_name_map) / sizeof(struct _disk_name_map); ++i)
		{
		if (major == disk_name_map[i].major)
			{
			dm = &disk_name_map[i];
			break;
			}
		}
	if (dm)
		{
		suffix = dm->suffix_base + i_disk;
		sprintf(buf, "%s%c", dm->name, suffix);
		if ((disk = lookup_disk_by_name(buf)) == NULL)
			{
			disk = g_new0(DiskMon, 1);
			disk->name = g_strdup(buf);
			disk->major = major;
			disk->minor = i_disk;
			disk->order = i + 1;	/* Skip the composite disk */
			i = 1;
			for (list = disk_mon_list->next; list; list = list->next, ++i)
				if (disk->order < ((DiskMon *) list->data)->order)
					break;
			disk_mon_list = g_list_insert(disk_mon_list, disk, i);
			++n_disks;
			}
		return disk;
		}
	return NULL;
	}

  /* Linux kernel 2.2 has 4 disk fields or 8 if my patch has been applied.
  |  Kernel 2.4 has variable number of fields.  Fields are added as drives
  |  are mounted and generate I/O.  re_checking is FALSE if this is the first
  |  call to register_disks().
  */
static void
register_disks(gboolean re_checking)
	{
	FILE	*f;
	DiskMon	*disk;
	gchar	buf[1024], *item, *s;
	gchar	c	= 'A';
	gint	major, minor;

	if ((f = fopen(PROC_STAT_FILE, "r")) == NULL)
		return;
	while (fgets(buf, sizeof(buf), f))
		{
		item = strtok(buf, " \t\n");
		if (!strncmp(item, "disk_io:", 8))	/* Kernel 2.4 */
			{
			while ((s = strtok(NULL, " \t\n")) != NULL)
				if (sscanf(s, "(%d,%d)", &major, &minor) == 2)
					add_disk_device(major, minor);
			break;
			}
		if (re_checking)	/* Can't have added fields for kernel 2.2 */
			continue;
		if (!strncmp(item, "disk_rblk", 9))	/* Kernel 2.2, count fields */
			{
			using_DiskN_names = TRUE;
			while ((s = strtok(NULL, " \t\n")) != NULL)
				{
				disk = g_new0(DiskMon, 1);
				disk_mon_list = g_list_append(disk_mon_list, disk);
				disk->name = g_strdup_printf("%s%c", _("Disk"), c);
				++c;
				++n_disks;
				}
			break;
			}
		}
	fclose(f);
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__)

#include <sys/dkstat.h>
#include <sys/disk.h>
#include <kvm.h>

static struct nlist nl[] = {
#define X_DISK_COUNT    0
   { "_disk_count" },      /* number of disks */
#define X_DISKLIST      1
   { "_disklist" },        /* TAILQ of disks */
   { NULL },
};

static struct disk *dkdisks;	/* kernel disk list head */

extern	kvm_t	*kvmd;

DiskMon *
add_disk_device(gint major, gint minor)
{
   /* should something be done there ? */
   return NULL;
}

void
read_netbsd_disk()
{
   static gint data_read_tick = -1;
   GList *list;
   DiskMon *disk;
   struct disk d, *p;
   
   if (data_read_tick == GK.timer_ticks)	/* Only one read per tick */
      return;
   data_read_tick = GK.timer_ticks;

   if (kvmd == NULL) return;
   if (n_disks <= 0) return;		/* computed by register_disks() */
   if (nl[0].n_type == 0) return;	/* ditto. */

   composite_disk->rblk = 0;
   composite_disk->wblk = 0;

   p = dkdisks;
   for(list = disk_mon_list->next; list; list = list->next) {
      if (kvm_read(kvmd, (u_long)p, &d, sizeof(d)) == sizeof(d)) {
	 
	 /* It seems impossible to get the read and write transfers
	  * separately. It's just a matter of choice to put the total in
	  * the rblk member but I like the blue color so much :).
	  *
	  * If someone as an idea for the wblk value... It though of the
	  * d.xfer value (number of transfers) but there a scaling problem:
	  * d.xfer is usually much more smaller thant d.dk_bytes... */

	 disk = (DiskMon *) list->data;
	 disk->rblk = d.dk_bytes / 512;	/* block size hardcoded :
					 * d.dk_byteshift is always 0 ?? */
	 disk->wblk = 0;

	 composite_disk->rblk += disk->rblk;
	 composite_disk->wblk += disk->wblk; /* useless, isn't it ? :-D */
      }
      p = d.dk_link.tqe_next;
   }
}

void
register_disks(gboolean re_checking)
{
   gint	i;
   DiskMon *disk;
   struct disklist_head head;
   struct disk d, *p;
   char buf[20];

   if (re_checking) return;
   if (kvmd == NULL) return;

   /* get disk count */
   if (kvm_nlist(kvmd, nl) >= 0 && nl[0].n_type != 0)
      if (kvm_read(kvmd, nl[X_DISK_COUNT].n_value,
		   (char *)&n_disks, sizeof(n_disks)) != sizeof(n_disks))
	 n_disks = 0;

   /* get first disk */
   if (n_disks > 0) {
      if (kvm_read(kvmd, nl[X_DISKLIST].n_value, 
		   &head, sizeof(head)) != sizeof(head))
	 n_disks = 0;

      dkdisks = head.tqh_first;
   }

   /* get disk names */
   for (i = 0, p = dkdisks; i < n_disks; i++) {
      disk = g_new0(DiskMon, 1);
      disk_mon_list = g_list_append(disk_mon_list, disk);

      if (kvm_read(kvmd, (u_long)p, &d, sizeof(d)) != sizeof(d) ||
	  kvm_read(kvmd, (u_long)d.dk_name, buf, sizeof(buf)) != sizeof(buf))
	 /* fallback to default name if kvm_read failed */
	 disk->name = g_strdup_printf("%s%c", _("Disk"), 'A' + i);
      else
	 disk->name = strdup(buf);

      p = d.dk_link.tqe_next;
   }

   using_DiskN_names = TRUE;
}

#endif /* __NetBSD__ || __OpenBSD__ */

/* ----- Solaris --------------------------------------------------------- */

#if defined(__solaris__)

#include <sys/types.h>
#include <sys/sysinfo.h>
#include <unistd.h>
#include <kstat.h>
#include <libdevinfo.h>
#include <errno.h>
#include <sys/dkio.h>

#define UNIT_SHIFT 3

#define NAME2MAJOR 0
#define MAJOR2NAME 1

typedef struct {
    gint major;
    gchar name[32];
} name_to_major_t;

static gint check_media_type(kstat_t *);
static gint isharddisk(kstat_t *);
void solaris_list_harddisks(void);                   /* called from main.c */

static gint lookup_name_to_major(name_to_major_t *, int);
static gint get_major(gchar *);
static gint get_devname(gint, gchar *);
static gint get_minor(gint);
static gint get_instance(gint); 

typedef struct {
    char name[8];
} probed_harddisk;

GList *hard_disk_list;

static DiskMon *
add_disk_device(gint major, gint minor) {
    DiskMon *disk;
    gchar *devid = NULL;
    gchar devname[32];
    gint instance;

    if ((disk = lookup_disk_by_device(major, minor)) == NULL) {
        get_devname(major, devname);
        instance = get_instance(minor);
        if (!(devname && instance))
            return NULL;
        devid = g_strdup_printf("%s%d", devname, instance);

        disk = g_new0(DiskMon, 1);
        disk->name = g_strdup(devid);
        disk->major = major;
        disk->minor = minor;
        disk_mon_list = g_list_append(disk_mon_list, disk);
        n_disks++;

        g_free(devid);
    }  
    return disk;
}

static void
read_solaris_disk() 
{
    DiskMon *disk;
    GList *list;

    extern kstat_ctl_t *kc;
    kstat_t *ksp;
    kstat_io_t kios;

#if 1
    gint block_size = 512;
#else
    gint block_size = 8192; /* default ufs logical blocksize for sun4u arch */
#endif
    guint blocks_read, blocks_written;

    if (!GK.second_tick)
        return;

    composite_disk->rblk = composite_disk->wblk = 0;

    if (kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return;
    }
    for (ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
        for (list = disk_mon_list; list; list = list->next) {
            disk = (DiskMon *) list->data;

            if(strcmp(disk->name, ksp->ks_name))
                continue;

            blocks_read = blocks_written = 0;

            memset((void *)&kios, 0, sizeof(kstat_io_t));
            kstat_read(kc, ksp, &kios);

            if (kios.nread)   
                blocks_read = kios.nread / block_size;
            if (kios.nwritten)
                blocks_written = kios.nwritten / block_size;

            disk->rblk = blocks_read;
            disk->wblk = blocks_written;

            composite_disk->rblk += blocks_read;
            composite_disk->wblk += blocks_written;
        }
    }
}

static void
register_disks(gboolean re_checking)
{
    gint i;
    DiskMon *disk;
    GList *list;
    gint major, minor;

    extern kstat_ctl_t *kc;
    kstat_t *ksp;

    probed_harddisk *drive;

    if (re_checking)
        return; 

    n_disks = 0;

    /* Solaris port would not use 'DiskN' type of disk naming. 
     * Instead, use sd0, sd1, ... just like you would see on iostat,
     * in the hope that this could be more scalable.
     */
    using_DiskN_names = FALSE; 

    if (kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return;
    }
    for (ksp = kc->kc_chain, i = 0; ksp; ksp = ksp->ks_next) {
        for (list = hard_disk_list; list; list = list->next) {
            drive = (probed_harddisk *)list->data;

            if(strcmp(drive->name, ksp->ks_name)) 
                continue;
            disk = g_new0(DiskMon, 1);
            if (using_DiskN_names) 
                disk->name = g_strdup_printf("%s%c", _("Disk"), 'A' + i);  
	    else 
                disk->name = g_strdup(ksp->ks_name);

            major = get_major(ksp->ks_module);
            minor = get_minor(ksp->ks_instance);

            disk->major = major; 
            disk->minor = minor;
            disk_mon_list = g_list_append(disk_mon_list, disk);
            n_disks++;
            i++;
        }
    } 
}

static gint
lookup_name_to_major(name_to_major_t *name_to_major, gint type) {
    FILE *fp;
    char line[80];
    char *name, *maj;
    gint name2major, major2name;
    gint majnum;

    name2major = major2name = 0;
    switch (type) {
        case NAME2MAJOR:
            name2major = 1;
            break;
        case MAJOR2NAME:
            major2name = 1;
            break;
        default:
            break;
    }

    if ((fp = fopen("/etc/name_to_major", "r")) == NULL) {
        perror("fopen");
        return -1;
    }

    while (fgets(line, sizeof(line), fp) != NULL) {
        name = strtok(line, " \t");
        if (name == NULL)
            continue;

        maj = strtok(NULL, "\n");
        if (maj == NULL)
            continue;
        majnum = (gint) atol(maj); 

        if (name2major) {
            if (strcmp(name_to_major->name, name) == 0) {
                name_to_major->major = majnum;
                fclose(fp);
                return 0;
            }
        } else if (major2name) {
            if (name_to_major->major == majnum) {
                strcpy(name_to_major->name, name);
                fclose(fp);
                return 0; 
            }
        }
    }
    fclose(fp);
    return -1;
}

static gint
get_major(gchar *devname) {
    /* xlation from device name to major (e.g. sd -> 32) */
    name_to_major_t name_to_major;

    strcpy(name_to_major.name, devname);
    if (lookup_name_to_major(&name_to_major, NAME2MAJOR) < 0)
        return -1;
    return name_to_major.major;
}

static gint
get_devname(gint major, gchar *devname) {
    /* xlation from major to device name (e.g. 118 -> ssd) */
    name_to_major_t name_to_major;

    name_to_major.major = major;
    if (lookup_name_to_major(&name_to_major, MAJOR2NAME) < 0)
        return -1;
    strcpy(devname, name_to_major.name);
    return 0;
}

static gint
get_minor(gint instance) {
    return instance << UNIT_SHIFT;
}

static gint
get_instance(gint minor) {
    return minor >> UNIT_SHIFT;
}

/* 
 * An sd instance could be a cdrom or a harddrive. You can't simply tell,
 * from contents of kstat, which type of device an sd device is 
 * (well, maybe you could, but at least i can't.)
 * It, however, doesn't make much sense to count cdrom read/write as 
 * "Disk" activity. So I'd like to exclude removable media's from 
 * monitoring target. In order to do this, I try to open a physical 
 * device of a corresponding sd instance. If it's succeeded, I assume
 * it's a hard drive. If I get ENXIO or EBUSY, I'll guess it's CDROM.
 * If you come up with a better (simpler or safer) way to tell it's 
 * a removable media or a hard drive, please drop me an e-mail at 
 * Daisuke Yabuki <dxy@acm.org>. 
 * I don't know any other driver which handle both hard drive and 
 * removable media, by the way. I hope it wouldn't do any harm on
 * other type of devices, i.e. ssd, or IDE drivers. 
 */ 
static gint
check_media_type(kstat_t *ksp) {
    gint fd;
    char *phys_path, devices_path[256]; /* or OBP_MAXPATHLEN? */
    di_node_t node;
    static di_node_t root_node = NULL;
#if 0
    /* Not supported on Solaris 7 */
    struct dk_minfo dk;
#else
    int dkRemovable;
#endif

    if (root_node == NULL) {
        if ((root_node = di_init("/", DINFOCPYALL)) == DI_NODE_NIL) {
                perror("di_init");
                return -1;
        }
    }

    node = di_drv_first_node(ksp->ks_module, root_node);
    while (node != DI_NODE_NIL) {
        if (di_instance(node) != ksp->ks_instance) {
            node = di_drv_next_node(node);
            continue;
        }
        if ((phys_path = di_devfs_path(node)) == NULL) {
            perror("di_devfs_path");
            return -1;
        }
        if (sprintf(devices_path, "/devices%s:c,raw", phys_path) <= 0) {
            di_devfs_path_free(phys_path);
            return -1;
        }
        if ((fd = open(devices_path, O_RDONLY)) == -1) {
            if (errno == ENXIO || errno == EBUSY) {
                close(fd);
                di_devfs_path_free(phys_path);
                return 0; /* guess it's removable media */
            } else {
#ifdef DEBUG
                printf("opening %s\n", devices_path);
                printf("unexpected errno: %d\n", errno);
                printf("disabled auto-detection/exclusion of removable media\n");
#endif
                close(fd);
                di_devfs_path_free(phys_path);
                return -1; /* EACCESS (unless setgid sys) or suchlike */
            }
        }
#if 0
	/* Not supported on Solaris 7 */
        if (ioctl(fd, DKIOCGMEDIAINFO, &dk) < 0)
#else
	if (ioctl(fd, DKIOCREMOVABLE, &dkRemovable) < 0)
#endif
	{
            close(fd);
            di_devfs_path_free(phys_path);
            return -1;
        }
#if 0
        if (dk.dki_media_type == DK_FIXED_DISK)
#else
        if (!dkRemovable)
#endif
	{
	   close(fd);
	   di_devfs_path_free(phys_path);
	   return 1;
        }
    }
    return -1; /* shouldn't be reached */
} 

static gint
isharddisk(kstat_t *ksp) {
    if (ksp->ks_type != KSTAT_TYPE_IO) 
        return 0;
    if (strncmp(ksp->ks_class, "disk", 4)) 
        return 0; /* excluding nfs etc. */
    if (!strcmp(ksp->ks_module, "fd"))
        return 0; /* excluding fd */
    if (check_media_type(ksp) == 0)
        return 0; /* guess it's removable media (e.g. CD-ROM, CD-R/W etc) */
    return 1;
}

/* 
 * creating a preliminary list of drives, which should be a complete
 * list of drives available on the system. the list is not supposed to
 * contain nfs, fd, cdrom, cdrw etc.
 */
void
solaris_list_harddisks(void) {
    extern kstat_ctl_t *kc;
    kstat_t *ksp;
    probed_harddisk *drive;

    if (kstat_chain_update(kc) == -1) {
        perror("kstat_chain_update");
        return;
    }
    
    for (ksp = kc->kc_chain; ksp; ksp = ksp->ks_next) {
        if(isharddisk(ksp)) {
             drive = g_new0(probed_harddisk, 1);
             hard_disk_list = g_list_append(hard_disk_list, drive);
             strcpy(drive->name, ksp->ks_name);
        }
    }
}

#endif /* __solaris__ */

/* ----- Other Systems -------------- */

#if defined(USE_LIBGTOP)

/* libgtop can't read disk stats */

static DiskMon *
add_disk_device(gint major, gint minor)
	{
	return NULL;
	}

static void
register_disks(gint x)
	{
	}
#endif

/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_disk_interface(void)
	{
	gint	available = FALSE;

#if defined(__FreeBSD__)
	register_disks(FALSE);
	read_system_disk_info = read_freebsd_disk;	/* above */
	available = TRUE;
#endif

#if defined(__linux__)
	register_disks(FALSE);
	read_system_disk_info = read_proc_stat;		/* In cpu.c */
	available = TRUE;
#endif

#if defined(__NetBSD__) || defined(__OpenBSD__)
	register_disks(FALSE);
	read_system_disk_info = read_netbsd_disk;	/* above */
	available = TRUE;
#endif

#if defined(__solaris__)
	register_disks(FALSE);
	read_system_disk_info = read_solaris_disk;      /* above */
	available = TRUE; 
#endif

	return available;
	}

/* ======================================================================== */
static Monitor	*mon_disk;

DiskMon			*composite_disk;
static gint		ascent;
static gint		style_id;

DiskMon *
lookup_disk_by_device(gint major, gint minor)
	{
	DiskMon	*disk;
	GList	*list;

	for (list = disk_mon_list->next; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (disk->major == major && disk->minor == minor)
			return disk;
		}
	return NULL;
	}

static DiskMon *
lookup_disk_by_name(gchar *name)
	{
	DiskMon	*disk;
	GList	*list;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon * ) list->data;
		if (!strcmp(name, disk->name))
			return disk;
		}
	return NULL;
	}


static size_abbrev_table	disk_blocks_abbrev[]	=
	{
	{ KB_SIZE(1),		1,				"%.0f" },
	{ KB_SIZE(20),		KB_SIZE(1),		"%.1fK" },
	{ MB_SIZE(1),		KB_SIZE(1),		"%.0fK" },
	{ MB_SIZE(20),		MB_SIZE(1),		"%.1fM" }
	};


static gchar    *text_format;

static void
format_chart_text(DiskMon *disk, gchar *buf, gint size)
	{
	Chart	*cp;
	gchar	c, *s;
	size_t	tbl_size;
	gint	len, r_blocks, w_blocks, blocks;

	--size;
	*buf = '\0';
	cp = disk->chart;
	r_blocks = gkrellm_get_current_chartdata(disk->rb_cd);
	w_blocks = gkrellm_get_current_chartdata(disk->wb_cd);
	tbl_size = sizeof(disk_blocks_abbrev) / sizeof(size_abbrev_table);
	for (s = text_format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			blocks = -1;
			if ((c = *(s + 1)) == 'T')
				blocks = r_blocks + w_blocks;
			else if (c == 'M')
				blocks = gkrellm_get_chart_scalemax(cp);
			else if (c == 'r')
				blocks = r_blocks;
			else if (c == 'w')
				blocks = w_blocks;
			else
				{
				*buf = *s;
				if (size > 1)
					{
					*(buf + 1) = *(s + 1);
					++len;
					}
				}
			if (blocks >= 0)
				len = format_size_abbrev(buf, size, (gfloat) blocks,
						&disk_blocks_abbrev[0], tbl_size);
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';	
	}

static void
draw_disk_extra(DiskMon *disk)
	{
	gchar		buf[128];

	if (!disk->extra_info)
		return;
	format_chart_text(disk, buf, sizeof(buf));
	gkrellm_draw_chart_text(disk->chart, style_id, buf);
	}

static void
draw_disk_chart(DiskMon *disk)
	{
	gkrellm_draw_chartdata(disk->chart);
	draw_disk_extra(disk);
	gkrellm_draw_chart_to_screen(disk->chart);
	}

static void
update_disk(void)
	{
	GList		*list;
	DiskMon		*disk;
	Chart		*cp;

	if (n_disks == 0)
		return;
	(*read_system_disk_info)();

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if ((cp = disk->chart) == NULL)		/* or disk->enabled FALSE */
			continue;
		if (GK.second_tick)
			{
			gkrellm_store_chartdata(cp, 0, disk->wblk, disk->rblk);
			draw_disk_chart(disk);
			}
		gkrellm_update_krell(cp->panel, KRELL(cp->panel),
										disk->wblk + disk->rblk);
		gkrellm_draw_panel_layers(cp->panel);
		}
	}


static gint
disk_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GList		*list;
	Chart		*cp;
	GdkPixmap	*pixmap	= NULL;

	for (list = disk_mon_list; list; list = list->next)
		{
		if ((cp = ((DiskMon *) list->data)->chart) == NULL)
			continue;
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		if (pixmap)
			{
			gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}

static gint
cb_disk_extra(GtkWidget *widget, GdkEventButton *ev)
	{
	GList	*list;
	DiskMon	*disk;

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (!disk->enabled || widget != disk->chart->drawing_area)
			continue;
		if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
			{
			disk->extra_info = !disk->extra_info;
			draw_disk_chart(disk);
			gkrellm_config_modified();
			}
		else if (   ev->button == 3
				 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
				)
			gkrellm_chartconfig_window_create(disk->chart);
		break;
		}
	return TRUE;
	}

static void
setup_disk_scaling(ChartConfig *cf, Chart *cp)
	{
	gint	grids, res;

	grids = gkrellm_get_chartconfig_fixed_grids(cf);
	if (!grids)
		grids = FULL_SCALE_GRIDS;
	res = gkrellm_get_chartconfig_grid_resolution(cf);

	KRELL(cp->panel)->full_scale = res * grids / gkrellm_update_HZ();
	}

  /* Destroy everything in a DiskMon structure except for the vbox which
  |  is preserved so disk ordering will be maintained.  Compare this to
  |  destroying an InetMon where really everything is destroyed including
  |  the InetMon structure.  Here the DiskMon structure is not destroyed.
  */
static void
destroy_disk_monitor(DiskMon *disk)
	{
	if (disk->launch.button)
		gkrellm_destroy_button(disk->launch.button);
	disk->launch.button = NULL;
	disk->launch.tooltip = NULL;
	gkrellm_dup_string(&disk->launch.command, "");
	gkrellm_dup_string(&disk->launch.tooltip_comment, "");
	gkrellm_chart_destroy(disk->chart);
	disk->chart = NULL;
	disk->enabled = FALSE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_disk);
	return TRUE;
	}

static void
create_disk_monitor(DiskMon *disk, gint first_create)
	{
	Panel		*p;
	Style		*style;
	Chart		*cp;

	if (first_create)
		{
		disk->chart = gkrellm_chart_new0();
		disk->chart->panel = gkrellm_panel_new0();
		}
	cp = disk->chart;
	p = cp->panel;

	style = gkrellm_panel_style(style_id);
	gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

	gkrellm_chart_create(disk->vbox, mon_disk, cp, &disk->chart_config);
	disk->wb_cd = gkrellm_add_default_chartdata(cp, _("Write Blocks"));
	disk->rb_cd = gkrellm_add_default_chartdata(cp, _("Read Blocks"));
	gkrellm_set_draw_chart_function(cp, draw_disk_chart, disk);

	gkrellm_chartconfig_fixed_grids_connect(cp->config,
				setup_disk_scaling, cp);
	gkrellm_chartconfig_grid_resolution_connect(cp->config,
				setup_disk_scaling, cp);
	gkrellm_chartconfig_grid_resolution_adjustment(cp->config, TRUE,
				0, (gfloat) MIN_GRID_RES, (gfloat) MAX_GRID_RES, 0, 0, 0, 70);
	gkrellm_chartconfig_grid_resolution_label(cp->config,
				_("Disk I/O blocks per sec"));
	if (gkrellm_get_chartconfig_grid_resolution(cp->config) < MIN_GRID_RES)
		gkrellm_set_chartconfig_grid_resolution(cp->config, DEFAULT_GRID_RES);
	gkrellm_alloc_chartdata(cp);

	setup_disk_scaling(cp->config, cp);

	gkrellm_panel_configure(p, disk->name, style);
	gkrellm_panel_create(disk->vbox, mon_disk, p);
	disk->enabled = TRUE;

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
				(GtkSignalFunc) disk_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT (p->drawing_area), "expose_event",
				(GtkSignalFunc) disk_expose_event, NULL);

		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "button_press_event",
				(GtkSignalFunc) cb_disk_extra, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);
		}
	else
		draw_disk_chart(disk);

	gkrellm_configure_tooltip(p, &disk->launch);
	if (*(disk->launch.command) != '\0')
		disk->launch.button = gkrellm_put_label_in_panel_button(p,
				gkrellm_launch_button_cb, &disk->launch, disk->launch.pad);
	}


static GtkWidget	*disk_vbox;

static void
create_disk(GtkWidget *vbox, gint first_create)
	{
	GList	*list;
	DiskMon	*disk;

	ascent = 0;
	disk_vbox = vbox;
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (first_create)
			{
			disk->vbox = gtk_vbox_new(FALSE, 0);
			gtk_box_pack_start(GTK_BOX(vbox), disk->vbox, FALSE, FALSE, 0);
			gtk_widget_show(disk->vbox);
			}
		gkrellm_setup_launcher(NULL, &disk->launch, CHART_PANEL_TYPE, 4);
		if (disk->enabled)
			create_disk_monitor(disk, first_create);
		}
	}

  /* Kernel 2.4 will not show a disk until it has I/O, and some systems
  |  may dynamically add a drive.
  */
static void
check_for_new_disks(void)
	{
	GList	*list;
	DiskMon	*disk;

	register_disks(TRUE);

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (disk->vbox == NULL)
			{
			disk->vbox = gtk_vbox_new(FALSE, 0);
			gtk_box_pack_start(GTK_BOX(disk_vbox), disk->vbox, FALSE, FALSE,0);
			gtk_widget_show(disk->vbox);
			gkrellm_setup_launcher(NULL, &disk->launch, CHART_PANEL_TYPE, 4);
			}
		}
	}

/* --------------------------------------------------------------------	*/
#define	DISK_CONFIG_KEYWORD	"disk"

static GtkWidget	*launch_vbox,
					*text_format_combo;

static void
save_disk_config(FILE *f)
	{
	GList	*list;
	DiskMon	*disk;

	if (n_disks == 0)
		return;
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		if (!using_DiskN_names && disk != composite_disk)
			fprintf(f, "%s enable (%d,%d) %d %d\n", DISK_CONFIG_KEYWORD,
					disk->major, disk->minor, disk->enabled,
					disk->extra_info);
		else
			fprintf(f, "%s enable %s %d %d\n", DISK_CONFIG_KEYWORD,
					disk->name, disk->enabled,
					disk->extra_info);
		if (*(disk->launch.command) != '\0')
			fprintf(f, "%s launch %s %s\n", DISK_CONFIG_KEYWORD,
						disk->name, disk->launch.command);
		if (*(disk->launch.tooltip_comment) != '\0')
			fprintf(f, "%s tooltip_comment %s %s\n", DISK_CONFIG_KEYWORD,
						disk->name, disk->launch.tooltip_comment);
		gkrellm_save_chartconfig(f, disk->chart_config,
					DISK_CONFIG_KEYWORD, disk->name);
		}
	fprintf(f, "%s text_format %s\n", DISK_CONFIG_KEYWORD, text_format);
	}

static void
load_disk_config(gchar *arg)
	{
	DiskMon	*disk;
	gchar	config[32], item[CFG_BUFSIZE],
			disk_name[32], command[CFG_BUFSIZE];
	gint	major, minor, n, extra;

	if (n_disks == 0)
		return;
	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (!strcmp(config, "text_format"))
			gkrellm_dup_string(&text_format, item);
		else if (strcmp(config, "enable") == 0)
			{
			n = FALSE;
			extra = 0;
			disk = NULL;
			if (item[0] == '(')
				{	/* Avoid user_config pathology (eg kernel 2.4 -> 2.2) */
				if (   !using_DiskN_names
					&& sscanf(item, "(%d,%d) %d %d",
						&major, &minor, &n, &extra) > 1
				   )
					disk = add_disk_device(major, minor);
				}
			else
				{
				sscanf(item, "%31s %d %d", disk_name, &n, &extra);
				disk = lookup_disk_by_name(disk_name);
				}
			if (disk)
				{
				disk->enabled = n;
				disk->extra_info = extra;
				}
			}
		else if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
			{
			sscanf(item, "%31s %[^\n]", disk_name, command);
			if ((disk = lookup_disk_by_name(disk_name)) != NULL)
				gkrellm_load_chartconfig(&disk->chart_config, command, 2);
			}
		else if (strcmp(config, "launch") == 0)
			{
			sscanf(item, "%31s %[^\n]", disk_name, command);
			if ((disk = lookup_disk_by_name(disk_name)) != NULL)
				disk->launch.command = g_strdup(command);
			}
		else if (strcmp(config, "tooltip_comment") == 0)
			{
			sscanf(item, "%31s %[^\n]", disk_name, command);
			if ((disk = lookup_disk_by_name(disk_name)) != NULL)
				disk->launch.tooltip_comment = g_strdup(command);
			}
		}
	}

static void
add_launch_entry(GtkWidget *vbox, DiskMon *disk)
	{
	GtkWidget   *table;

	table = gkrellm_launcher_table_new(vbox, 1);
	gkrellm_config_launcher(table, 0,  &disk->launch_entry,
				&disk->tooltip_entry, disk->name,
				&disk->launch);
	disk->launch_table = table;
	gtk_widget_show_all(table);
    }

static void
apply_disk_config(void)
	{
	GList		*list;
	DiskMon		*disk;
	gchar		*s;

	if (n_disks == 0)
		return;

	s = gkrellm_entry_get_text(&(GTK_COMBO(text_format_combo)->entry));
	gkrellm_dup_string(&text_format, s);

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		
		if (disk->launch_table)		/* Not NULL only if enabled */
			{
			gkrellm_apply_launcher(&disk->launch_entry, &disk->tooltip_entry,
				disk->chart->panel, &disk->launch, gkrellm_launch_button_cb);
			gtk_widget_destroy(disk->launch_table);
			}
		disk->launch_table = NULL;
		}
	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;
		disk->new_disk = GTK_TOGGLE_BUTTON(disk->enable_button)->active;
		if (disk->new_disk)
			add_launch_entry(launch_vbox, disk);
		if (disk->new_disk && ! disk->enabled)
			create_disk_monitor(disk, TRUE);
		else if (! disk->new_disk && disk->enabled)
			destroy_disk_monitor(disk);
		disk->enabled = disk->new_disk;
		}
	}


#define	DEFAULT_TEXT_FORMAT	"$T"

static gchar	*disk_info_text[] =
{
N_("<b>Chart Labels\n"),
N_("Substitution variables for the format string for chart labels:\n"),
N_("\t$M    maximum chart value\n"),
N_("\t$T    total read blocks + write blocks\n"),
N_("\t$r    read blocks\n"),
N_("\t$w    write blocks\n"),
};

static void
create_disk_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1, *hbox;
	GtkWidget		*text;
	GList			*list;
	DiskMon			*disk;
	gchar			*s;
	gint			i;

	check_for_new_disks();
	if (n_disks == 0)
		return;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* -- Options tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));
	vbox1 = gkrellm_scrolled_vbox(vbox, NULL,
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	for (list = disk_mon_list; list; list = list->next)
		{
		disk = (DiskMon *) list->data;

		if (list == disk_mon_list)
			s = g_strdup_printf("%s - %s", disk->name,
					_("Composite chart combines data for all disks"));
		else
			s = g_strdup(disk->name);
		gkrellm_check_button(vbox1, &disk->enable_button, disk->enabled,
				FALSE, 0, s);
		g_free(s);
		}

/* -- Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));
	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Chart Labels"),
			 4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	text_format_combo = gtk_combo_new();
	gtk_widget_set_usize (GTK_WIDGET(text_format_combo), 350, 0);
	gtk_box_pack_start(GTK_BOX(hbox), text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_TEXT_FORMAT);
	list = g_list_append(list, "$T\\C\\f$M");
	list = g_list_append(list, "\\c\\f$M\\b$T");
	list = g_list_append(list,
					_("\\f\\ww\\C\\f$M\\D2\\f\\ar\\. $r\\D1\\f\\aw\\. $w"));
	gtk_combo_set_popdown_strings(GTK_COMBO(text_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(text_format_combo)->entry),
			text_format);

	vbox = gkrellm_framed_vbox_end(vbox, _("Launch Commands"), 4, TRUE, 0, 2);
	launch_vbox = gkrellm_scrolled_vbox(vbox, NULL,
						GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	for (i = 0, list = disk_mon_list; list; list = list->next, ++i)
		{
		disk = (DiskMon *) list->data;
		if (disk->enabled)
			add_launch_entry(launch_vbox, disk);
		}

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(disk_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(disk_info_text[i]));
	}

static Monitor	monitor_disk =
	{
	N_("Disk"),				/* Name, for config tab.	*/
	MON_DISK,			/* Id,  0 if a plugin		*/
	create_disk,		/* The create function		*/
	update_disk,				/* The update function		*/
	create_disk_tab,	/* The config tab create function	*/
	apply_disk_config,	/* Apply the config function		*/

	save_disk_config,	/* Save user conifg			*/
	load_disk_config,	/* Load user config			*/
	DISK_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_disk_monitor(void)
	{
	DiskMon	*disk;
	GList	*list;

	monitor_disk.name = _(monitor_disk.name);

	disk = g_new0(DiskMon, 1);
	disk_mon_list = g_list_append(disk_mon_list, disk);
	disk->name = g_strdup(_("Disk"));
	composite_disk = disk;
	text_format = g_strdup(DEFAULT_TEXT_FORMAT);
	mon_disk = &monitor_disk;

	if (setup_disk_interface())
		{
		for (list = disk_mon_list->next; list; list = list->next)
			{
			disk = (DiskMon *) list->data;
			disk->enabled = FALSE;
			disk->extra_info = TRUE;
			}
		if (n_disks > 0)
			composite_disk->enabled = TRUE;
		style_id = gkrellm_add_chart_style(&monitor_disk, DISK_STYLE_NAME);
		return &monitor_disk;
		}
	return NULL;
	}
