/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/* 3/13/2000  Patch from Tom Gilbert <gilbertt@btinternet.com> allows window
|			  managers to differentiate options dialog from GKrellM itself.
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include <dirent.h>



typedef struct
	{
	GtkText		*text;
	gchar		*string;
	gint		translate;
	}
	InfoText;

static GList	*info_text_list;


#define		N_FONT_OVERRIDES	3

GtkWidget		*config_window;

static GdkFont	*bold_info_font,
				*italic_info_font;

static gint		rebuild_flag;

/* void gtk_adjustment_set_value( GtkAdjustment *adjustment, gfloat value); */

  /* Popup a message close to a widget (should be one with a window, eg
  |  drawing_area).
  */
static void
message_window(gchar *title, gchar *message, GtkWidget *widget, gint mainwin)
	{
	GtkWidget	*top_win;
	GtkWidget	*window;
	GtkWidget	*scrolled;
	GtkWidget	*vbox, *vbox1, *hbox;
	GtkWidget	*label;
	GtkWidget	*separator;
	GtkWidget	*button;
	gchar		*s;
	gint		x, y, w, h, w_main, nlines, dummy;

	if (!message)
		return;
	top_win = gkrellm_get_top_window();
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	if (widget == NULL)
		widget = top_win;

	gtk_window_set_policy(GTK_WINDOW(window), TRUE, TRUE, FALSE);
	gtk_window_set_title(GTK_WINDOW(window), title);
	gtk_container_border_width(GTK_CONTAINER(window), 8);

	vbox = gtk_vbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(window),vbox);

	for (nlines = 0, s = message; *s; ++s)
		if (*s == '\n')
			++nlines;
	if (nlines > 20)
		{
		vbox1 = gkrellm_scrolled_vbox(vbox, &scrolled,
					GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);
		gtk_widget_set_usize(scrolled, 0, 300);	/* A guess */
		}
	else
		vbox1 = vbox;
	label = gtk_label_new(message);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_box_pack_start(GTK_BOX(vbox1), label, FALSE, FALSE,0);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 4);

	hbox = gtk_hbox_new(TRUE, 2);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);
    button = gtk_button_new_with_label(_("  OK  "));
    gtk_box_pack_start(GTK_BOX(hbox), button, FALSE, FALSE, 0);

	gtk_signal_connect_object(GTK_OBJECT(button), "clicked",
		(GtkSignalFunc) gtk_widget_destroy, GTK_OBJECT(window));
	gtk_widget_show_all(window);
	gdk_window_get_origin(widget->window, &x, &y);
	gdk_window_get_size(window->window, &w, &h);
	if (mainwin)
		{
		gdk_window_get_origin(top_win->window, &x, &dummy);
		gdk_window_get_size(top_win->window, &w_main, &dummy);
		x = (x < w_display / 2) ? x + w_main : x - w - 4;
		}
	else
		x -= w/2;

	if (y + h > h_display)
		y = h_display - h;
	if (y < 25)
		y = 25;
	gtk_widget_set_uposition(window, x, y);
	}

void
gkrellm_message_window(gchar *title, gchar *message, GtkWidget *widget)
	{
	message_window(title, message, widget, 1);
	}

void
gkrellm_config_message_window(gchar *title, gchar *message, GtkWidget *widget)
	{
	message_window(title, message, widget, 0);
	}

static void
add_info_text(InfoText *info)
	{
	GdkFont	*font	= NULL;
	gchar	*s, *s_tab = "\t\t";

#if defined(ENABLE_NLS)
	s = info->translate ? gettext(info->string) : info->string;
#else
	s = info->string;
#endif

	if (strncmp(s, "<b>", 3) == 0)
		{
		font = bold_info_font;
		s += 3;
		}
	else if (strncmp(s, "<i>", 3) == 0)
		{
		font = italic_info_font;
		s += 3;
		}
	else
		font = NULL;
	if (font && *s == '\t')		/* Print tabs at default style width */
		{
		++s;
		if (*(s+1) != '\t')
			++s_tab;
		else
			++s;
		gtk_text_insert(GTK_TEXT(info->text), NULL, NULL, NULL, s_tab, -1);
		}
	gtk_text_insert(GTK_TEXT(info->text), font, NULL, NULL, s, -1);
	}

void
gkrellm_add_info_text_string(GtkWidget *text, gchar *string)
	{
	InfoText	*info;

	info = g_new0(InfoText, 1);
	info->text = GTK_TEXT(text);
	info->string = g_strdup(string);
	info->translate = FALSE;
	info_text_list = g_list_append(info_text_list, info);
	add_info_text(info);
	}

void
gkrellm_add_info_text(GtkWidget *text, gchar **string, gint n_strings)
	{
	InfoText	*info;
	gint		i;

	for (i = 0; i < n_strings; ++i)
		{
		info = g_new0(InfoText, 1);
		info->text = GTK_TEXT(text);
		info->string = g_strdup(string[i]);
		info->translate = TRUE;
		info_text_list = g_list_append(info_text_list, info);
		add_info_text(info);
		}
	}

void
gkrellm_launch_button_cb(DecalButton *button)
	{
	Launcher    *launch;
	gchar		*command;

	launch = (Launcher *) button->data;
	command = g_strconcat(launch->command, " &", NULL);
	gkrellm_system(command);
	g_free(command);
	}

void
gkrellm_configure_tooltip(Panel *p, Launcher *launch)
	{
	if (launch->tooltip == NULL)
		{
		if (*launch->tooltip_comment != '\0' && *launch->command != '\0')
			{
			launch->tooltip = gtk_tooltips_new();
			gtk_tooltips_set_tip(launch->tooltip, p->drawing_area,
					launch->tooltip_comment, NULL);
			gtk_tooltips_set_delay(launch->tooltip, 750);
			}
		return;
		}
	if (*launch->tooltip_comment != '\0' && *launch->command != '\0')
		{
		gtk_tooltips_set_tip(launch->tooltip, p->drawing_area,
					launch->tooltip_comment, NULL);
		gtk_tooltips_enable(launch->tooltip);
		}
	else
		gtk_tooltips_disable(launch->tooltip);
	}

void
gkrellm_apply_launcher(GtkWidget **launch_entry, GtkWidget **tooltip_entry,
			Panel *p, Launcher *launch, void (*func)())
	{
	gchar	*command, *tip_comment;

	if (!launch_entry || !launch || !p)
		return;
	command = gkrellm_entry_get_text(launch_entry);
	tip_comment = tooltip_entry ? gkrellm_entry_get_text(tooltip_entry) : "";
	if (   !strcmp(command, launch->command)
		&& !strcmp(tip_comment, launch->tooltip_comment)
	   )
		return;
	if (*command != '\0')		
		{
		if (launch->button == NULL)
			{
			if (launch->type == METER_PANEL_TYPE)
				launch->button = gkrellm_put_label_in_meter_button(p, func,
										launch, launch->pad);
			else
				launch->button = gkrellm_put_label_in_panel_button(p, func,
										launch, launch->pad);
			}
		gkrellm_dup_string(&launch->command, command);
		}
	else
		{
		if (launch->button)
			gkrellm_destroy_button(launch->button);
		launch->button = NULL;
		launch->pipe = NULL;		/* Close?? */
		gkrellm_dup_string(&launch->command, "");
		}
	gkrellm_dup_string(&launch->tooltip_comment, tip_comment);
	gkrellm_configure_tooltip(p, launch);
	}

GtkWidget *
gkrellm_launcher_table_new(GtkWidget *vbox, gint n_launchers)
	{
	GtkWidget	*table;

	table = gtk_table_new(2 * n_launchers, 3, TRUE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 2);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 2);
	return table;
	}

void
gkrellm_config_launcher(GtkWidget *table, gint n, GtkWidget **launch_entry,
			GtkWidget **tooltip_entry, gchar *desc, Launcher *launch)
	{
	gchar		buf[64];
	GtkWidget	*label, *hbox;
	gint		row;

	if (!table || !launch_entry)
		return;
	row = (tooltip_entry ? 2 : 1) * n;
	hbox = gtk_hbox_new(FALSE, 0);
    /* Attach left right top bottom */
    gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 1, row, row + 1);

	snprintf(buf, sizeof(buf), _("%s command:"), desc);
    label = gtk_label_new(buf);
	gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
	gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);

    *launch_entry = gtk_entry_new_with_max_length(255);
    gtk_table_attach_defaults(GTK_TABLE(table), *launch_entry,
                1, 3, row, row + 1);
	gtk_entry_set_text(GTK_ENTRY(*launch_entry),
					(launch && launch->command) ? launch->command : "");

	if (tooltip_entry)
		{
		hbox = gtk_hbox_new(FALSE, 0);
	    gtk_table_attach_defaults(GTK_TABLE(table), hbox,
					0, 1, row + 1, row + 2);
		snprintf(buf, sizeof(buf), _("%s comment:"), desc);
	    label = gtk_label_new(buf);
		gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
		gtk_box_pack_start(GTK_BOX(hbox), label, TRUE, TRUE, 4);
	    *tooltip_entry = gtk_entry_new_with_max_length(255);
	    gtk_table_attach_defaults(GTK_TABLE(table), *tooltip_entry,
	                1, 2, row + 1, row + 2);
		gtk_entry_set_text(GTK_ENTRY(*tooltip_entry),
					(launch && launch->tooltip_comment)
					? launch->tooltip_comment : "");
		}
	}

  /* FIXME: this guy is called on panels at create events
  |  when this situation has occured:  the GKrellM rebuild has destroyed
  |  the decal button list.  But existing launchers have not had their
  |  button pointer set to NULL!  This could cause a problem with
  |  code that tries to check for button pointers in the create routines.
  */
void
gkrellm_setup_launcher(Panel *p, Launcher *launch, gint type, gint pad)
	{
	if (!launch)
		return;
	if (launch->command == NULL)
		launch->command = g_strdup("");
	if (launch->tooltip_comment == NULL)
		launch->tooltip_comment = g_strdup("");
	launch->type = type;
	launch->pad = pad;
	if (p)
		{
		gkrellm_configure_tooltip(p, launch);
		if (*(launch->command) != '\0')
			launch->button = gkrellm_put_label_in_meter_button(p,
					gkrellm_launch_button_cb, launch, launch->pad);
		else
			launch->button = NULL;	/* In case dangling pointer, see above */
		}
	}

void
insert_expanded_filler(GtkWidget *box)
	{
	GtkWidget	*label;

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 0);
	}

void
gkrellm_check_button(GtkWidget *box, GtkWidget **button, gboolean active,
			gboolean expand, gint pad, gchar *string)
	{
	GtkWidget	*b;

	if (!string || !box)
		return;
	b = gtk_check_button_new_with_label(string);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b), active);
	gtk_box_pack_start(GTK_BOX(box), b, expand, FALSE, pad);
	if (button)
		*button = b;
	}

void
gkrellm_check_button_connected(GtkWidget *box, GtkWidget **button,
			gboolean active, gboolean expand, gboolean fill, gint pad,
			void (*cb_func)(), gpointer data, gchar *string)
	{
	GtkWidget	*b;

	if (!string || !box)
		return;
	b = gtk_check_button_new_with_label(string);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(b), active);
	gtk_box_pack_start(GTK_BOX(box), b, expand, fill, pad);
	if (cb_func)
		gtk_signal_connect(GTK_OBJECT(b), "clicked",
				GTK_SIGNAL_FUNC(cb_func), data);
	if (button)
		*button = b;
	}

void
gkrellm_button_connected(GtkWidget *box, GtkWidget **button,
			gboolean expand, gboolean fill, gint pad,
			void (*cb_func)(), gpointer data, gchar *string)
	{
	GtkWidget	*b;

	if (!string || !box)
		return;
	b = gtk_button_new_with_label(string);
	gtk_box_pack_start(GTK_BOX(box), b, expand, fill, pad);
	if (cb_func)
		gtk_signal_connect(GTK_OBJECT(b), "clicked",
				GTK_SIGNAL_FUNC(cb_func), data);
	if (button)
		*button = b;
	}

void
gkrellm_spin_button(GtkWidget *box, GtkWidget **spin_button, gfloat value,
		gfloat low, gfloat high, gfloat step0, gfloat step1,
		gint digits, gint width,
		void (*cb_func)(), gpointer data, gboolean right_align, gchar *string)
	{
	GtkWidget		*hbox	= NULL,
					*label;
	GtkSpinButton	*spin;
	GtkAdjustment	*adj;

	if (string && box)
		{
	    hbox = gtk_hbox_new (FALSE, 0);
    	gtk_box_pack_start(GTK_BOX(box), hbox, FALSE, FALSE, 2);
		box = hbox;
		}
    adj = (GtkAdjustment *) gtk_adjustment_new (value,
								low, high, step0, step1, 0.0);
    *spin_button = gtk_spin_button_new(adj, 0.5, digits);
	if (width > 0)
		gtk_widget_set_usize(*spin_button, width, 0);
    spin = GTK_SPIN_BUTTON(*spin_button);
    gtk_spin_button_set_numeric(spin, TRUE);
	if (data == NULL)
		data = (gpointer) spin;
	if (cb_func)
		gtk_signal_connect(GTK_OBJECT(adj), "value_changed",
				GTK_SIGNAL_FUNC(cb_func), data);
	if (box)
		{
		if (right_align && string)
			{
			label = gtk_label_new(string);
			gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
			gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 2);
			}
		gtk_box_pack_start(GTK_BOX(box), *spin_button, FALSE, FALSE, 2);
		if (!right_align && string)
			{
			label = gtk_label_new(string);
			gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
			gtk_box_pack_start(GTK_BOX(box), label, TRUE, TRUE, 2);
			}
		}
	}

GtkWidget *
gkrellm_scrolled_vbox(GtkWidget *box, GtkWidget **scr,
		GtkPolicyType h_policy, GtkPolicyType v_policy)
	{
	GtkWidget	*scrolled,
				*vbox;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			h_policy, v_policy);
	gtk_box_pack_start(GTK_BOX(box), scrolled, TRUE, TRUE, 0);
	vbox = gtk_vbox_new(FALSE, 2);
	gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(scrolled), vbox);
	if (scr)
		*scr = scrolled;
	return vbox;
	}

GtkWidget *
gkrellm_scrolled_text(GtkWidget *box, GtkWidget **scr,
		GtkPolicyType h_policy, GtkPolicyType v_policy)
	{
	GtkWidget	*scrolled,
				*text;

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			h_policy, v_policy);
	gtk_box_pack_start(GTK_BOX(box), scrolled, TRUE, TRUE, 0);
	text = gtk_text_new(NULL, NULL);
	gtk_text_set_editable(GTK_TEXT(text), FALSE);
	gtk_container_add(GTK_CONTAINER(scrolled), text);

	if (scr)
		*scr = scrolled;
	return text;
	}

  /* frame_border_width - border around outside of frame.
  |  vbox_pad - pad between widgets to be packed in returned vbox.
  |  vbox_border_width - border between returned vbox and frame.
  */
GtkWidget *
gkrellm_framed_vbox(GtkWidget *box, gchar *label, gint frame_border_width,
		gboolean frame_expand, gint vbox_pad, gint vbox_border_width)
	{
	GtkWidget	*frame;
	GtkWidget	*vbox;

	frame = gtk_frame_new(label);
	gtk_container_set_border_width(GTK_CONTAINER(frame), frame_border_width);
    gtk_box_pack_start(GTK_BOX(box), frame, frame_expand, frame_expand, 0);
    vbox = gtk_vbox_new(FALSE, vbox_pad);
    gtk_container_border_width(GTK_CONTAINER(vbox), vbox_border_width);
    gtk_container_add(GTK_CONTAINER(frame), vbox);
	return vbox;
	}

GtkWidget *
gkrellm_framed_vbox_end(GtkWidget *box, gchar *label, gint frame_border_width,
		gboolean frame_expand, gint vbox_pad, gint vbox_border_width)
	{
	GtkWidget	*frame;
	GtkWidget	*vbox;

	frame = gtk_frame_new(label);
	gtk_container_set_border_width(GTK_CONTAINER(frame), frame_border_width);
    gtk_box_pack_end(GTK_BOX(box), frame, frame_expand, frame_expand, 0);
    vbox = gtk_vbox_new(FALSE, vbox_pad);
    gtk_container_border_width(GTK_CONTAINER(vbox), vbox_border_width);
    gtk_container_add(GTK_CONTAINER(frame), vbox);
	return vbox;
	}

static void
create_about_tab(GtkWidget *vbox)
	{
	GtkWidget	*label;
	gchar		*buf;

	vbox = gkrellm_framed_vbox(vbox, NULL, 2, TRUE, 0, 0);
	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

	buf = g_strdup_printf(_("GKrellM %d.%d.%d\nGNU Krell Monitors\n\n"
				"Copyright (c) 1999-2001 by Bill Wilson\n"
				"bill@gkrellm.net\n"
				"http://gkrellm.net\n\n"
				"Released under the GNU Public License"),
				GKRELLM_VERSION_MAJOR, GKRELLM_VERSION_MINOR,
				GKRELLM_VERSION_REV);
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);

	label = gtk_label_new("");
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);

#if defined(__FreeBSD__)
	buf = g_strdup_printf(_("FreeBSD code was contributed by:\n"
							"Hajimu UMEMOTO <ume@mahoroba.org>"));
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
#endif
#if defined(__NetBSD__)
	buf = g_strdup_printf(_("NetBSD code was contributed by:\n"
							"Anthony Mallet <metall@ficus.yi.org>"));
	label = gtk_label_new(buf);
	g_free(buf);
	gtk_box_pack_start(GTK_BOX(vbox), label, TRUE, TRUE, 0);
#endif
	}


/* ------------------General Settings---------------------------------*/
static GtkWidget
			*enable_hst_button,
			*hostname_short_button,
			*enable_sysname_button,
			*save_position_button,
			*update_HZ_spin_button,
			*width_spin_button;

#if defined(TRACK_GTK_THEMES)
GtkWidget	*track_gtk_button;
#endif

static gchar	*info_fonts[3][2] =
{
  {
  "-adobe-helvetica-bold-r-normal-*-*-80-*-*-*-*-iso8859-*,-*-*-bold-r-*-*-8-*-*-*-*-*-iso8859-*,-*-*-medium-r-*-*-10-*-*-*-*-*-iso8859-*",
  "-adobe-helvetica-bold-o-normal-*-*-80-*-*-*-iso8859-*,-*-*-bold-o-*-*-8-*-*-*-*-*-iso8859-*,-*-*-medium-r-*-*-10-*-*-*-*-*-iso8859-*"
  },
  {
  "-adobe-helvetica-bold-r-normal-*-*-100-*-*-*-*-iso8859-*,-*-*-bold-r-*-*-10-*-*-*-*-*-iso8859-*,-*-*-medium-r-*-*-10-*-*-*-*-*-iso8859-*",
  "-adobe-helvetica-bold-o-normal-*-*-100-*-*-*-iso8859-*,-*-*-bold-o-*-*-10-*-*-*-*-*-iso8859-*,-*-*-medium-r-*-*-10-*-*-*-*-*-iso8859-*"
  },
  {
  "-adobe-helvetica-bold-r-normal-*-*-120-*-*-*-*-iso8859-*,-*-*-bold-r-*-*-12-*-*-*-*-*-iso8859-*,-*-*-medium-r-*-*-12-*-*-*-*-*-iso8859-*",
  "-adobe-helvetica-bold-o-normal-*-*-120-*-*-*-iso8859-*,-*-*-bold-o-*-*-12-*-*-*-*-*-iso8859-*,-*-*-medium-r-*-*-12-*-*-*-*-*-iso8859-*"
  }
};

static void
free_info_font(void)
	{
	if (bold_info_font)
		gdk_font_unref(bold_info_font);
	if (italic_info_font)
		gdk_font_unref(italic_info_font);
	bold_info_font = NULL;
	italic_info_font = NULL;	
	}

#define	INFO_TEST_STRING	"Width"

static void
init_info_font(GtkWidget *widget)
	{
	GdkFont		*font;
	gint		w_style, w;
	static gint	i_font = -1;

	free_info_font();
	font = NULL;
	if (i_font < 0 && widget && widget->style)
		{
		w_style = gdk_string_width(widget->style->font, INFO_TEST_STRING);
		for (i_font = 2; i_font > 0; --i_font)
			{
			font = gdk_fontset_load(info_fonts[i_font][0]);
			if (font)
				{
				w = gdk_string_width(font, INFO_TEST_STRING);
				if (w < w_style + 7)
					break;
				gdk_font_unref(font);
				font = NULL;
				}
			}
		}
	if (!font && i_font >= 0)
		font = gdk_fontset_load(info_fonts[i_font][0]);
	bold_info_font = font;
	italic_info_font = gdk_fontset_load(info_fonts[i_font][1]);
	}

static void
cb_width_changed(GtkWidget *widget, GtkSpinButton *spin)
	{
	rebuild_flag = TRUE;
	}

static void
apply_general_config(void)
	{
	gint	n;

#if defined(TRACK_GTK_THEMES)
	GK.track_gtk_rcfiles = GTK_TOGGLE_BUTTON(track_gtk_button)->active;
#endif
	GK.enable_hostname = GTK_TOGGLE_BUTTON(enable_hst_button)->active;
	GK.hostname_short = GTK_TOGGLE_BUTTON(hostname_short_button)->active;
	GK.enable_system_name = GTK_TOGGLE_BUTTON(enable_sysname_button)->active;

	GK.save_position = GTK_TOGGLE_BUTTON(save_position_button)->active;
	n = GK.update_HZ;
	GK.update_HZ = gtk_spin_button_get_value_as_int(
					GTK_SPIN_BUTTON(update_HZ_spin_button));
	if (n != GK.update_HZ)
		start_timer(GK.update_HZ);
	GK.chart_width =
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(width_spin_button));
	}

static gchar	*general_info_text[]	=
{
N_("<b>Krells\n"),
N_("Krells are the horizontally moving indicators below each chart and\n"
"on meter style monitors.  Depending on the monitor, they show fast\n"
"response data rates, a percentage of some capacity, or something else.\n"),
"\n",
N_("<b>Charts\n"),
N_("The default for most charts is to automatically adjust the number of\n"
	"grid lines drawn and the resolution per grid so drawn data will be\n"
	"nicely visible.  You may change this to fixed grids of 1-5 and/or\n"
	"fixed grid resolutions in the chart config windows.  However,\n"
	"some combination of the auto scaling modes may give best results.\n"),
"\n",
N_("<i>Auto number of grids:\n"),
N_("\tThis means to scale the chart to have as many grids as is necessary\n"
	"\tto show all the data on the chart without clipping.\n"),
N_("\tOnce a grid density saturates a chart, grids will not be visible.\n"),
"\n",
N_("<i>Auto grid resolution"),
N_(" has two meanings:\n"),
N_("\t1) If using a fixed number of grids, it means to set the\n"
	"\tresolution per grid to the smallest value in a base to peak range\n"
	"\tthat draws data without clipping.  The peak is the maximum data\n"
	"\tvalue encountered and the base is 1/5 the peak.\n"),
N_("\t2) If using auto number of grids, it means to set the resolution\n"
	"\tper grid to a value such that drawing the peak value encountered\n"
	"\trequires at least 5 grids to be drawn.\n"),
"\n",
N_("<b>Chart Labels\n"),
N_("Chart label format strings place text on charts using position codes:\n"),
N_("\t\\t    top left\n"),
N_("\t\\b    bottom left\n"),
N_("\t\\n    next line\n"),
N_("\t\\N    next line only if last string had visible characters\n"),
N_("\t\\p    previous line\n"),
N_("\t\\c    center the text\n"),
N_("\t\\C    begin drawing text at the center\n"),
N_("\t\\r    right justify\n"),
N_("\t\\f    use alternate font for the next string\n"),
N_("\t\\w    use the following string to define a field width\n"),
N_("\t\\a    draw left justified in the defined field width\n"),
N_("\t\\e    draw right justified in the defined field width\n"),
N_("\t\\.     no-op.  Used to break a string into two strings.\n"),
N_("\t\\D0   bottom of charts first data view (D2 for second data view ...)\n"),
N_("\t\\D1   top of charts first data view (D3 for second data view ...)\n"),

N_("<b>\nCommands\n"),
N_("\tMany monitors can be configured to launch commands.  Just enter the\n"
   "\tcommand where you see a \"command\" entry and also a comment if you\n"
   "\twant a tooltip to appear for the command.  After a command is entered,\n"
   "\tfor a monitor, a button for launching it will become visible when you\n"
   "\tmove the mouse into the panel area of the monitor.\n\n"),

N_("<b>\nMouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("clicking on charts will toggle a display of some extra info.\n"),
N_("<b>\tRight "),
N_("clicking on charts brings up a chart configuration window.\n"),
N_("<b>\tRight "),
N_("clicking on many panels opens its monitor configuration window.\n")
};

static void
create_general_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1;
	GtkWidget		*hbox;
	GtkWidget		*text;
	gint			i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* --Options tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Options"));
	vbox1 = gtk_vbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), vbox1, FALSE, FALSE, 0);
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, FALSE, 0);

	gkrellm_check_button(hbox, &enable_hst_button, GK.enable_hostname,
			FALSE, 0, _("Hostname display"));

	gkrellm_check_button(hbox, &hostname_short_button, GK.hostname_short,
			FALSE, 10, _("Short hostname"));

	gkrellm_check_button(vbox, &enable_sysname_button, GK.enable_system_name,
			FALSE, 0, _("System name display"));

	gkrellm_check_button(vbox, &save_position_button, GK.save_position, FALSE,
		10,
		_("Remember screen location at exit and move to it at next startup"));

#if defined(TRACK_GTK_THEMES)
	gkrellm_check_button(vbox, &track_gtk_button, GK.track_gtk_rcfiles, FALSE,
		0, _("Track Gtk theme changes for similarly named themes"));
#endif

	vbox = gkrellm_framed_vbox_end(vbox, NULL, 4, FALSE, 0, 2);
	gkrellm_spin_button(vbox, &update_HZ_spin_button, (gfloat) GK.update_HZ,
			2.0, 10.0, 1.0, 1.0, 0, 55, NULL, NULL, FALSE,
			_("Krell and LED updates per second."));

	gkrellm_spin_button(vbox, &width_spin_button, (gfloat) GK.chart_width,
			(gfloat) CHART_WIDTH_MIN, (gfloat) CHART_WIDTH_MAX,
			1.0, 5.0, 0, 55, cb_width_changed, NULL, FALSE,
			_("GKrellM width"));

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	for (i = 0; i < sizeof(general_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(general_info_text[i]));
	}


/* ------------------Themes Tab----------------------------------*/

#define	THEME_RELOAD	0
#define	THEME_NEXT		1
#define	THEME_PREV		2
#define	THEME_ALT_NEXT	3
#define	THEME_ALT_PREV	4

static GList		*themes_list;
static GList		*theme_position_in_list;

static GtkWidget	*theme_clist;
static GtkWidget	*theme_spin_button;
static GtkWidget	*theme_alt_label;
static GtkWidget	*default_theme_entry;
static GtkWidget	*font_button,
					*fontset_button,
					*font_preview_button;
static gint			theme_width, author_width;
static gboolean		theme_modified,
					font_modified,
					font_preview,
					need_font_restore,
					save_override,
					save_fontset;

typedef struct
	{
	GtkFontSelectionDialog	*fontseldlg;
	GtkWidget				*entry;
	GtkWidget				*browse_button;
	gchar					*name;
	gchar					*string;
	gchar					*save_string;
	GdkFont					*font;
	}
	AltFontSelect;

static AltFontSelect
			large_font,
			normal_font,
			small_font;

gchar *
get_large_font_override(void)
	{
	return large_font.string;
	}

gchar *
get_normal_font_override(void)
	{
	return normal_font.string;
	}

gchar *
get_small_font_override(void)
	{
	return small_font.string;
	}

static gboolean
get_font_entries(void)
	{
	gchar		*s;
	gboolean	modified = FALSE;

	s = gtk_entry_get_text(GTK_ENTRY(large_font.entry));
	modified |= gkrellm_dup_string(&large_font.string, s);
	s = gtk_entry_get_text(GTK_ENTRY(normal_font.entry));
	modified |= gkrellm_dup_string(&normal_font.string, s);
	s = gtk_entry_get_text(GTK_ENTRY(small_font.entry));
	modified |= gkrellm_dup_string(&small_font.string, s);
	return modified;
	}

static void
cb_font_dialog_ok(GtkWidget *w, AltFontSelect *afs)
	{
	gchar	*fontname;

	fontname = gtk_font_selection_dialog_get_font_name(afs->fontseldlg);
	if (fontname)
		gtk_entry_set_text(GTK_ENTRY(afs->entry), fontname);
	gtk_widget_destroy(GTK_WIDGET(afs->fontseldlg));
	font_modified = TRUE;
	if (font_preview)
		{
		need_font_restore = TRUE;
		get_font_entries();
		build_gkrellm();
		}
	}

static void
cb_font_dialog(GtkWidget *widget, AltFontSelect *afs)
	{
	GtkWidget				*w;
	GtkFontSelectionDialog	*fsd;

	if (afs->fontseldlg)
		return;
	w = gtk_font_selection_dialog_new(_(afs->name));
	fsd = GTK_FONT_SELECTION_DIALOG(w);
	afs->fontseldlg = fsd;
	gtk_font_selection_dialog_set_font_name(fsd, afs->string);
	gtk_signal_connect(GTK_OBJECT(fsd->ok_button), "clicked",
			GTK_SIGNAL_FUNC(cb_font_dialog_ok), afs);
	gtk_signal_connect_object(GTK_OBJECT(fsd->cancel_button), "clicked",
			GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(fsd));
	gtk_signal_connect(GTK_OBJECT(fsd), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &afs->fontseldlg);
	gtk_widget_show(GTK_WIDGET(fsd));
	}

static gchar *
get_theme_author(gchar *path)
	{
	static gchar	buf[128];
	FILE			*f;
	gchar			*s, *q, *rcfile, line[128];

	buf[0] = '\0';
	if (!path || *path == '\0')
		return buf;
	rcfile = g_strdup_printf("%s/%s", path, GKRELLMRC);
	f = fopen(rcfile, "r");
	g_free(rcfile);
	if (f == NULL)
		return buf;
	while (fgets(line, sizeof(line), f))
		{
		if (   (s = strtok(line, " :=\t\n")) == NULL
			|| strcmp(s, "author") != 0
		   )
			continue;
		s = strtok(NULL, "\n");		/* Rest of line is Author string */
		if (s)
			{
			while (   *s == ' ' || *s == '\t' || *s == '"' || *s == '='
					|| *s == ':')
				++s;
			q = strchr(s, (int) '"');
			if (q)
				*q = '\0';
			strcpy(buf, s);
			break;
			}
		}
	fclose(f);
	return buf;
	}

static void
add_themes_to_list(GList **list, gchar *theme_dir)
	{
	DIR				*dir;
	struct dirent	*dentry;
	gchar			*path;

	if ((dir = opendir(theme_dir)) == NULL)
		return;
	while ((dentry = readdir(dir)) != NULL)
		{
		if (dentry->d_name[0] != '.' && dentry->d_ino > 0)
			{
			if (basename_in_list(*list, dentry->d_name))
				{
				if (GK.debug_level & DEBUG_MISC)
					printf("Ignoring duplicate theme %s/%s\n",
						theme_dir, dentry->d_name);
				continue;
				}
			path = g_strdup_printf("%s/%s", theme_dir, dentry->d_name);
			if (isdir(path, NULL))
				*list = g_list_append(*list, path);
			else
				g_free(path);
			}
		}
	closedir(dir);
	}

static void
find_theme_position_in_list(void)
	{
	theme_position_in_list = string_in_list(themes_list,
			(*(GK.theme_path)) ? GK.theme_path : "Default");
	if (! theme_position_in_list)	/* Shouldn't happen */
		theme_position_in_list = themes_list;
	}

void
make_themes_list(void)
	{
	gchar	*theme_dir;

	free_glist_and_data(&themes_list);
	themes_list = g_list_append(themes_list, g_strdup("Default"));
	theme_dir = g_strdup_printf("%s/%s", gkrellm_homedir(),
					GKRELLM_THEMES_DIR);
	add_themes_to_list(&themes_list, theme_dir);
	g_free(theme_dir);

	add_themes_to_list(&themes_list, LOCAL_THEMES_DIR);
	add_themes_to_list(&themes_list, SYSTEM_THEMES_DIR);
	themes_list = g_list_sort(themes_list, (GCompareFunc) strcmp_basename);

	if (GK.command_line_theme)
		themes_list = g_list_prepend(themes_list,
							g_strdup(GK.command_line_theme));

	find_theme_position_in_list();
	}

static void
cb_theme_selected(GtkWidget *clist, gint row, gint column,
					GdkEventButton *bevent, gpointer data)
	{
	gchar			*theme;

	gtk_spin_button_set_value(GTK_SPIN_BUTTON(theme_spin_button), 0.0);
	if ((theme = gtk_clist_get_row_data(GTK_CLIST(clist), row)) != NULL)
		gtk_entry_set_text(GTK_ENTRY(default_theme_entry), theme);

	gkrellm_dup_string(&GK.theme_path, theme);
	find_theme_position_in_list();
	theme_modified = TRUE;
	build_gkrellm();
	}

static void
scan_for_themes(GtkWidget *clist)
	{
	GList	*list;
	gchar	*path, *name;
	gchar	*buf[3];
	gint	row, l;

	make_themes_list();
	buf[2] = NULL;
	theme_width = 10;
	author_width = 10;

	for (list = themes_list; list; list = list->next)
		{
		path = (gchar *) list->data;
		if ((name = strrchr(path, '/')) == NULL)
			name = path;
		else
			++name;
		buf[0] = name;
		if (!strcmp(name, "Default"))
			path = "";
		buf[1] = get_theme_author(path);
		row = gtk_clist_append(GTK_CLIST(clist), buf);
		gtk_clist_set_row_data_full(GTK_CLIST(clist), row,
						g_strdup(path), (GtkDestroyNotify) g_free);
		if ((l = strlen(name)) > theme_width)
			theme_width = l;
		if ((l = strlen(buf[1])) > author_width)
			author_width = l;
		}
	}

static void
cb_theme_alternative_changed(GtkWidget *widget, GtkSpinButton *spin)
	{
	gint	i;

	i = gtk_spin_button_get_value_as_int(spin);
	if (i > GK.theme_n_alternatives)
		{
		i = GK.theme_n_alternatives;
		gtk_spin_button_set_value(spin, (gfloat) i);
		}
	if (i != GK.theme_alternative)
		{
		GK.theme_alternative = i;
		theme_modified = TRUE;
		build_gkrellm();
		}
	}

void
set_theme_alternatives_label(void)
	{
	GtkSpinButton	*spin;
	gchar			buf[64];

	if (!config_window)
		return;
	spin = GTK_SPIN_BUTTON(theme_spin_button);
	gtk_spin_button_set_value(spin, (gfloat) GK.theme_alternative);
	snprintf(buf, sizeof(buf), _("%d total theme alternatives"),
					GK.theme_n_alternatives);
	gtk_label_set_text(GTK_LABEL(theme_alt_label), buf);
	}

void
save_theme_config(void)
	{
	FILE	*f;
	gchar	*path;

	/* Assume gkrellm -t is for testing and don't save theme config changes.
	|  Similary for GK.demo.
	*/
	if (!theme_modified || GK.command_line_theme || GK.demo)
		return;

	path = gkrellm_make_config_file_name(gkrellm_homedir(),
					GKRELLM_THEME_CONFIG);

	if (GK.debug_level & DEBUG_MISC)
		printf(_("Writting new %s: <%s>\n"), path, GK.theme_path);

	if ((f = fopen(path, "w")) != NULL)
		{
		fprintf(f, "%s\n", GK.theme_path);
		fprintf(f, "%d %d %d\n",
				GK.theme_alternative, GK.font_override, GK.use_fontset);
		fprintf(f, "%s\n", large_font.string);
		fprintf(f, "%s\n", normal_font.string);
		fprintf(f, "%s\n", small_font.string);
		fclose(f);
		}
	g_free(path);
	theme_modified = FALSE;
	}

void
load_theme_config(void)
	{
	FILE	*f;
	gchar	*path, *s;
	gchar	buf[1024];
	gint	i;

	/* Need to load the theme from ~/.gkrellm/theme_config only at startup
	|  or if re-reading because of theme_event - these are only times 
	|  GK.theme_path will be NULL.  Note: GK.theme_path will not be NULL
	|  at startup if there is a command line theme so won't use alternate
	|  fonts in that case.  This will be right for theme makers testing
	|  their themes, but not using fontsets may not be right ...
	*/
	if (!GK.theme_path)
		{
		path = gkrellm_make_config_file_name(gkrellm_homedir(),
					GKRELLM_THEME_CONFIG);
		f = fopen(path, "r");
		g_free(path);
		if (f && fgets(buf, sizeof(buf), f))
			{
			s = strtok(buf, " \n\t");
			if (s && *s != '#' && *s != '\0' && strcmp(s, "Default"))
				{
				if (*s == '/')
					GK.theme_path = g_strdup(s);
			  	else
					GK.theme_path = g_strdup_printf("%s/%s/%s",
							gkrellm_homedir(), GKRELLM_THEMES_DIR, s);
				}
			for (i = 0; fgets(buf, sizeof(buf), f); ++i)
				{
				s = strchr(buf, (gint) '\n');
				if (s)
					*s = '\0';
				if (i == 0)
					sscanf(buf, "%d %d %d", &GK.theme_alternative,
							&GK.font_override, &GK.use_fontset);
				if (i == 1)
					gkrellm_dup_string(&large_font.string, buf);
				if (i == 2)
					gkrellm_dup_string(&normal_font.string, buf);
				if (i == 3)
					gkrellm_dup_string(&small_font.string, buf);
				}
			}
		if (f)
			fclose(f);
		}
	if (!GK.theme_path || !isdir(GK.theme_path, NULL))
		gkrellm_dup_string(&GK.theme_path, "");
	if (!large_font.string)
		gkrellm_dup_string(&large_font.string,
				"-b&h-lucida-bold-r-normal-*-*-140-*-*-*-*-iso8859-*");
	if (!normal_font.string)
		gkrellm_dup_string(&normal_font.string,
				"-b&h-lucida-medium-r-normal-*-*-120-*-*-*-*-iso8859-*");
	if (!small_font.string)
		gkrellm_dup_string(&small_font.string,
				"-adobe-helvetica-medium-r-*-*-*-100-*-*-*-*-iso8859-*");
	}

#if defined(TRACK_GTK_THEMES)
  /* Some Gtk themes have a suffix */
static gchar *gtk_suffix[]	=
	{
	"-gtk", "_gtk", "-GTK", "_GTK"
	};
#endif

  /* Called from cb_client_event.  Triggered by a Gtk theme switch or if
  |  an app sends _GKRELLM_READ_THEME event.
  */
void
read_theme_event(gint gkrellm_read_theme)
	{
	if (gkrellm_read_theme)
		{
		g_free(GK.theme_path);
		GK.theme_path = NULL;	/* Force reread of GKRELLM_THEME_CONFIG */
		}
#if defined(TRACK_GTK_THEMES)
	else if (GK.track_gtk_rcfiles)
		/* This code currently not accessible (GK.track_gtk_rcfiles is not
		|  setable) I've heard a master theme switcher (E, Gtk, xmms, gkrellm)
		|  may be in the works which would be a better general solution.
		*/
		{
		FILE	*f;
		gchar	*gtkrc_path, *s, *g;
		gchar	buf[512];
		gint	i;
		GList	*list;

		gtkrc_path = g_strdup_printf("%s/.gtkrc", gkrellm_homedir());
		if ((f = fopen(gtkrc_path, "r")) != NULL)
			{
				/* # -- THEME AUTO-WRITTEN DO NOT EDIT		*/
			fgets(buf, sizeof(buf), f);
				/* include "/home/bill/.themes/XX/gtk/gtkrc	*/
			fgets(buf, sizeof(buf), f);
			fclose(f);
			if ((s = strstr(buf, "/gtk/gtkrc")) != NULL)
				{
				*s = '\0';
				if ((s = strrchr(buf, (int) '/')) != NULL)
					{
					++s;
					list = basename_in_list(themes_list, s);
					if (!list)
						{
						for (i = 0; i < sizeof(gtk_suffix)/sizeof(gchar *); ++i)
							if ((g = strstr(s, gtk_suffix[i])) != NULL)
								{
								*g = '\0';
								list = basename_in_list(themes_list, s);
								break;
								}
						}
					if (list)
						{
						s = (gchar *) list->data;
						gkrellm_dup_string(&GK.theme_path, strcmp(s, "Default")
								? s : "");
						theme_modified = TRUE;
						save_theme_config();
						}
					}
				}
			}
		}
#endif
	build_gkrellm();
	}

static void
cb_load_theme(gpointer data, guint action, GtkWidget *widget)
	{
	gchar			*path;
	gint			row;

	if (GK.no_config)
		return;
	if (!themes_list)
		make_themes_list();
	if (action == THEME_ALT_NEXT || action == THEME_ALT_PREV)
		{
		GK.theme_alternative += ((action == THEME_ALT_NEXT) ? 1 : -1);
		if (GK.theme_alternative > GK.theme_n_alternatives)
			{
			GK.theme_alternative = 0;
			action = THEME_NEXT;
			}
		if (GK.theme_alternative < 0)
			{
			GK.theme_alternative = 100;
			action = THEME_PREV;
			}
		theme_modified = TRUE;
		}
	else if (action != THEME_RELOAD)
		GK.theme_alternative = 0;

	if (action == THEME_NEXT || action == THEME_PREV)
		{
		if (action == THEME_NEXT)
			{
			theme_position_in_list = theme_position_in_list->next;
			if (!theme_position_in_list)
				theme_position_in_list = themes_list;
			}
		else
			{
			theme_position_in_list = theme_position_in_list->prev;
			if (!theme_position_in_list)
				theme_position_in_list = g_list_last(themes_list);
			}
		if (config_window)
			{
			row = g_list_position(themes_list, theme_position_in_list);
			if (gtk_clist_row_is_visible(GTK_CLIST(theme_clist), row)
						!= GTK_VISIBILITY_FULL)
			gtk_clist_moveto(GTK_CLIST(theme_clist), row, 0, 0.5, 0.0);
			gtk_clist_select_row(GTK_CLIST(theme_clist), row, 0);
			return;		/* cb_theme_selected will build_gkrellm()	*/
			}
		path = (gchar *) theme_position_in_list->data;
		gkrellm_dup_string(&GK.theme_path, strcmp(path, "Default") ? path :"");
		theme_modified = TRUE;
		}
	build_gkrellm();
	}

static void
destroy_font_dialogs(void)
	{
	if (large_font.fontseldlg)
		gtk_widget_destroy(GTK_WIDGET(large_font.fontseldlg));
	if (normal_font.fontseldlg)
		gtk_widget_destroy(GTK_WIDGET(normal_font.fontseldlg));
	if (small_font.fontseldlg)
		gtk_widget_destroy(GTK_WIDGET(small_font.fontseldlg));
	}

static void
init_font_config(void)
	{
	font_modified = FALSE;
	need_font_restore = FALSE;
	save_override = GK.font_override;
	save_fontset = GK.use_fontset;
	gkrellm_dup_string(&large_font.save_string, large_font.string);
	gkrellm_dup_string(&normal_font.save_string, normal_font.string);
	gkrellm_dup_string(&small_font.save_string, small_font.string);
	}

static void
restore_fonts(void)
	{
	gkrellm_dup_string(&large_font.string, large_font.save_string);
	gkrellm_dup_string(&normal_font.string, normal_font.save_string);
	gkrellm_dup_string(&small_font.string, small_font.save_string);
	GK.font_override = save_override;
	GK.use_fontset = save_fontset;
	need_font_restore = FALSE;
	build_gkrellm();
	}

static void
close_theme_config(gint from_close)
	{
	destroy_font_dialogs();
	if (need_font_restore && from_close)
		restore_fonts();
	}

static void
apply_theme_config(void)
	{
	gboolean	active;

	active = GTK_TOGGLE_BUTTON(font_button)->active;
	if (GK.font_override != active)
		{
		theme_modified = TRUE;
		rebuild_flag = TRUE;
		}
	GK.font_override = active;

	active = GTK_TOGGLE_BUTTON(fontset_button)->active;
	if (GK.use_fontset != active)
		{
		theme_modified = TRUE;
		rebuild_flag = TRUE;
		}
	GK.use_fontset = active;

	if (font_modified)
		{
		get_font_entries();
		rebuild_flag = TRUE;
		theme_modified = TRUE;
		}
	init_font_config();
	}

static void
cb_font_preview(GtkWidget *widget, gpointer *data)
	{
	font_preview = GTK_TOGGLE_BUTTON(widget)->active;
	font_modified = TRUE;
	if (font_preview)
		{
		need_font_restore = TRUE;
		get_font_entries();
		build_gkrellm();
		}
	else if (need_font_restore)
		restore_fonts();
	}

static void
set_font_override_sensitivities(gboolean sensitive)
	{
	if (!sensitive)
		{
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(font_preview_button),
				FALSE);
		font_preview = FALSE;
		}
	gtk_widget_set_sensitive(font_preview_button, sensitive);
	gtk_widget_set_sensitive(large_font.entry, sensitive);
	gtk_widget_set_sensitive(large_font.browse_button, sensitive);
	gtk_widget_set_sensitive(normal_font.entry, sensitive);
	gtk_widget_set_sensitive(normal_font.browse_button, sensitive);
	gtk_widget_set_sensitive(small_font.entry, sensitive);
	gtk_widget_set_sensitive(small_font.browse_button, sensitive);
	}

static void
cb_font_override(GtkWidget *widget, gpointer *data)
	{
	GK.font_override = GTK_TOGGLE_BUTTON(widget)->active;
	if (font_preview)
		build_gkrellm();
	set_font_override_sensitivities(GK.font_override);
	gkrellm_config_modified();
	font_modified = TRUE;
	if (!GK.font_override)
		destroy_font_dialogs();
	}

static void
cb_use_fontset(GtkWidget *widget, gpointer *data)
	{
	if (!font_preview)
		return;
	GK.use_fontset = GTK_TOGGLE_BUTTON(widget)->active;
	gkrellm_config_modified();
	font_modified = TRUE;
	build_gkrellm();
	}


static gchar	*themes_titles[] = {N_("Theme"), N_("Author") };

static void
create_theme_tab(GtkWidget *tabs_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*vbox, *vbox1, *vbox2, *hbox;
	GtkWidget		*label;
	GtkWidget		*scrolled;
	gchar			*s;
	gint            i;

	init_font_config();

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tabs_vbox), tabs, TRUE, TRUE, 0);

/* --Theme tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Theme"));
	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);
	label = gtk_label_new(_("Theme:"));
	gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 3);

	default_theme_entry = gtk_entry_new_with_max_length(128);
	if (GK.theme_path)
		gtk_entry_set_text(GTK_ENTRY(default_theme_entry), GK.theme_path);
	gtk_box_pack_start(GTK_BOX(hbox), default_theme_entry, TRUE, TRUE,0);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
    for (i = 0; i < 2; i++)
		themes_titles[i]=_(themes_titles[i]);
	theme_clist = gtk_clist_new_with_titles(2, themes_titles);
	gtk_signal_connect (GTK_OBJECT(theme_clist), "select_row",
			(GtkSignalFunc) cb_theme_selected, NULL);

	gtk_clist_set_shadow_type(GTK_CLIST(theme_clist), GTK_SHADOW_OUT);
	gtk_clist_column_titles_passive(GTK_CLIST(theme_clist));
	gtk_container_add (GTK_CONTAINER (scrolled), theme_clist);

	scan_for_themes(theme_clist);

	gtk_clist_set_column_width(GTK_CLIST(theme_clist), 0, theme_width * 7 + 8);
	gtk_clist_set_column_width(GTK_CLIST(theme_clist), 1, author_width * 7);

	hbox = gtk_hbox_new (FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 2);

	gkrellm_spin_button(hbox, &theme_spin_button, (gfloat)GK.theme_alternative,
			0.0, 100.0, 1.0, 5.0, 0, 60,
			cb_theme_alternative_changed, NULL, FALSE, NULL);
	theme_alt_label = gtk_label_new("");
	gtk_box_pack_start (GTK_BOX (hbox), theme_alt_label, TRUE, TRUE, 4);
	gtk_misc_set_alignment(GTK_MISC(theme_alt_label), 0, 0.5);
	set_theme_alternatives_label();

/* --Fonts tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Fonts"));
	vbox1 = gkrellm_framed_vbox(vbox, NULL, 2, FALSE, 0, 2);

	vbox2 = gkrellm_framed_vbox(vbox1, _("Large font"), 4, FALSE, 0, 3);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 0);

	large_font.entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), large_font.entry, TRUE, TRUE, 0);
	gtk_entry_set_text(GTK_ENTRY(large_font.entry), large_font.string);
	gkrellm_button_connected(hbox, &large_font.browse_button, FALSE, FALSE,
			0, cb_font_dialog, &large_font, _("Browse"));

	vbox2 = gkrellm_framed_vbox(vbox1, _("Normal font"), 4, FALSE, 0, 3);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 0);
	normal_font.entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), normal_font.entry, TRUE, TRUE, 0);
	gtk_entry_set_text(GTK_ENTRY(normal_font.entry), normal_font.string);
	gkrellm_button_connected(hbox, &normal_font.browse_button, FALSE, FALSE,
			0, cb_font_dialog, &normal_font, _("Browse"));

	vbox2 = gkrellm_framed_vbox(vbox1, _("Small font"), 4, FALSE, 0, 3);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox2), hbox, FALSE, TRUE, 0);
	small_font.entry = gtk_entry_new();
	gtk_box_pack_start(GTK_BOX(hbox), small_font.entry, TRUE, TRUE, 0);
	gtk_entry_set_text(GTK_ENTRY(small_font.entry), small_font.string);
	gkrellm_button_connected(hbox, &small_font.browse_button, FALSE, FALSE,
			0, cb_font_dialog, &small_font, _("Browse"));

	gkrellm_check_button_connected(vbox1, &font_preview_button, font_preview,
			FALSE, FALSE, 2, cb_font_preview, NULL,
			_("Preview font changes"));

	gkrellm_check_button_connected(vbox, &font_button, GK.font_override,
			FALSE, FALSE, 2, cb_font_override, NULL,
			_("Use above alternate fonts instead of theme specified fonts"));
	set_font_override_sensitivities(GK.font_override);

	gkrellm_check_button_connected(vbox, &fontset_button, GK.use_fontset,
			FALSE, FALSE, 2, cb_use_fontset, NULL,
			_("Load fonts as fontsets"));

	/* --Info tab*/
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	s = g_strdup_printf(_("Untar your theme tar files in %s/%s"),
						gkrellm_homedir(), GKRELLM_THEMES_DIR);
	label = gtk_label_new(s);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	g_free(s);

	label = gtk_label_new(
			_("Download themes from the GKrellM theme site at www.muhri.net"));
	gtk_box_pack_start(GTK_BOX(vbox), label, FALSE, FALSE, 0);
	gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_LEFT);
	gtk_misc_set_alignment(GTK_MISC(label), 0, 0.5);


	}


/* -------------------------------------------------------------------*/
static GtkCTreeNode	*node_plugins,
					*node_builtins;
static GtkWidget	*notebook_config;
static GtkWidget	*config_ctree;
static gint			config_page;

static gint			expand_builtins,
					expand_plugins;


static void
close_config(gpointer data)
	{
	InfoText	*info;
	GList		*list;
	gint		from_close	= GPOINTER_TO_INT(data);

	close_plugins_config();
	expand_builtins = GTK_CTREE_ROW(node_builtins)->expanded;
	expand_plugins = GTK_CTREE_ROW(node_plugins)->expanded;
	gtk_widget_destroy(config_window);
	free_info_font();
	config_window = NULL;
	for (list = info_text_list; list; list = list->next)
		{
		info = (InfoText *) list->data;
		g_free(info->string);
		}
	free_glist_and_data(&info_text_list);
	close_theme_config(from_close);
	}


static void
apply_config(void)
	{
	GList	*list;
	Monitor	*mon;

	gkrellm_freeze_side_frame_packing();
	apply_general_config();
	apply_theme_config();
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (mon->apply_config && mon->privat->enabled)
			(*(mon->apply_config))();
		}

	save_theme_config();
	save_user_config();

	gkrellm_thaw_side_frame_packing();

	if (rebuild_flag)
		build_gkrellm();
	}

static void
OK_config(void)
	{
	apply_config();
	close_config(GINT_TO_POINTER(0));
	}

  /* Obsolete */
GtkWidget *
create_tab(GtkWidget *tabs, char *name)
	{
	return gkrellm_create_tab(tabs, name);
	}

GtkWidget *
gkrellm_create_tab(GtkWidget *tabs, char *name)
	{
	GtkWidget	*label;
	GtkWidget	*vbox;

	vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(vbox), 0);

	label = gtk_label_new(name);
	gtk_notebook_append_page(GTK_NOTEBOOK(tabs), vbox, label);

	return vbox;
	}

GtkWidget *
gkrellm_create_framed_tab(GtkWidget *tabs, char *name)
	{
	GtkWidget	*vbox;

	vbox = gkrellm_create_tab(tabs, name);
	vbox = gkrellm_framed_vbox(vbox, NULL, 2, TRUE, 0, 2);
	return vbox;
	}

static GtkWidget *
create_config_page(GtkNotebook *notebook, gchar *text, GtkCTree *ctree,
		GtkCTreeNode *node_parent, GtkCTreeNode **node_result)
	{
	GtkWidget		*vbox;
	GtkCTreeNode	*node;
	gchar			*title[1] = {text};

	vbox = gtk_vbox_new(FALSE, 0);
	node = gtk_ctree_insert_node(ctree, node_parent, NULL, title, 0,
			NULL, NULL, NULL, NULL, FALSE, FALSE);
	gtk_ctree_node_set_row_data(ctree, node, GINT_TO_POINTER(config_page++));
	gtk_notebook_append_page(notebook, vbox, NULL);
	if (node_result)
		*node_result = node;
	return vbox;
	}

void
add_plugin_config_page(Monitor *mon)
	{
	GtkWidget		*vbox;
	GtkCTreeNode	*node;

	if (config_window && mon->create_config)
		{
		vbox = create_config_page(GTK_NOTEBOOK(notebook_config), mon->name,
			GTK_CTREE(config_ctree), node_plugins, &node);
		(*(mon->create_config))(vbox);

		gtk_widget_show_all(vbox);
		mon->privat->notebook_config_widget = notebook_config;
		mon->privat->notebook_config_page =
				g_list_length(GTK_NOTEBOOK(notebook_config)->children) - 1;
		mon->privat->ctree_config_node = node;
		}
	else
		mon->privat->notebook_config_widget = NULL;
	}

void
remove_plugin_config_page(Monitor *mon)
	{
	GtkCTreeNode	*node;
	GList			*list;
	Monitor			*tmon;
	gint			page;

	if (mon->privat->notebook_config_widget)
		{
		node = mon->privat->ctree_config_node;
		while (node)
			{
			page = GPOINTER_TO_INT(gtk_ctree_node_get_row_data(
						GTK_CTREE(config_ctree), node));
			gtk_ctree_node_set_row_data(GTK_CTREE(config_ctree), node,
						GINT_TO_POINTER(page - 1));
			node = GTK_CTREE_ROW(node)->sibling;
			}
		gtk_ctree_remove_node(GTK_CTREE(config_ctree),
				mon->privat->ctree_config_node);
		gtk_notebook_remove_page(
				GTK_NOTEBOOK(mon->privat->notebook_config_widget),
				mon->privat->notebook_config_page);
		--config_page;

		/* When a config_page is removed, any greater plugin config_page must
		|  be decremented
		*/
		for (list = gkrellm_monitor_list; list; list = list->next)
			{
			tmon = (Monitor *) list->data;
			if (mon->privat->notebook_config_page
						< tmon->privat->notebook_config_page)
				tmon->privat->notebook_config_page -= 1;
			}
		}
	mon->privat->notebook_config_widget = NULL;
	}

static void
cb_select_monitor_config(GtkWidget *ctree, GtkCTreeNode *node)
	{
	gint	row;

	if (!GTK_CLIST(ctree)->selection)
		return;
	row = GPOINTER_TO_INT(gtk_ctree_node_get_row_data(GTK_CTREE(ctree), node));
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook_config), row);
	}

gboolean
gkrellm_config_window_shown(void)
	{
	return config_window ? TRUE : FALSE;
	}

void
create_config_window(void)
	{
	GtkWidget		*vbox,
					*main_vbox,
					*config_hbox,
					*hbox;
	GtkWidget		*ctree;
	GtkWidget		*scrolled;
	GtkWidget		*button;
	GtkCTreeNode	*node;
	GList			*list;
	Monitor			*mon;
	gchar			*title[1] = { _("Monitors") };
	gchar			*config_name, *window_title;
	
	if (config_window)
		{
		gdk_window_raise(config_window->window);
		return;
		}

	rebuild_flag = FALSE;
	config_page = 0;

	config_window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
	gtk_signal_connect(GTK_OBJECT(config_window), "delete_event",
			(GtkSignalFunc) close_config, GINT_TO_POINTER(1));
	gtk_window_set_policy(GTK_WINDOW(config_window), FALSE, FALSE, TRUE);

	config_name = gkrellm_make_config_file_name(NULL, "GKrellM");
	window_title = g_strdup_printf("%s %s", config_name, _("Configuration"));
	gtk_window_set_title(GTK_WINDOW(config_window), window_title);
	g_free(config_name);
	g_free(window_title);

	gtk_window_set_wmclass(GTK_WINDOW(config_window),
					"Gkrellm_conf", "Gkrellm");
	gtk_container_border_width(GTK_CONTAINER(config_window), 2);
	init_info_font(config_window);
	
	config_hbox = gtk_hbox_new(FALSE, 4);
	gtk_container_add(GTK_CONTAINER(config_window), config_hbox);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(config_hbox), scrolled, FALSE, FALSE, 0);

	main_vbox = gtk_vbox_new(FALSE, 4);
	gtk_box_pack_start(GTK_BOX(config_hbox), main_vbox, FALSE, FALSE, 0);

	ctree = gtk_ctree_new_with_titles(1, 0, title);
	config_ctree = ctree;
	gtk_ctree_set_indent(GTK_CTREE(ctree), 16);	
	gtk_clist_column_titles_passive(GTK_CLIST(ctree));
	gtk_widget_set_usize(ctree, 150, 0);
	gtk_container_add(GTK_CONTAINER(scrolled), ctree);
	gtk_signal_connect(GTK_OBJECT(ctree), "tree_select_row",
			(GtkSignalFunc) cb_select_monitor_config, NULL);

	notebook_config = gtk_notebook_new();
	gtk_notebook_set_show_tabs(GTK_NOTEBOOK(notebook_config), FALSE);
	gtk_box_pack_start(GTK_BOX(main_vbox), notebook_config, TRUE, TRUE, 0);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("General"),
			GTK_CTREE(ctree), NULL, &node);
	create_general_tab(vbox);
	gtk_ctree_select(GTK_CTREE(ctree), node);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Builtins"),
			GTK_CTREE(ctree), NULL, &node_builtins);
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (MONITOR_ID(mon) == MON_PLUGIN || ! mon->create_config)
			continue;
		vbox = create_config_page(GTK_NOTEBOOK(notebook_config), mon->name,
			GTK_CTREE(ctree), node_builtins, &node);
		mon->privat->ctree_config_node = node;
		(*(mon->create_config))(vbox);
		}
	if (expand_builtins)
		gtk_ctree_expand(GTK_CTREE(ctree), node_builtins);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Plugins"),
			GTK_CTREE(ctree), NULL, &node_plugins);
	create_plugin_config(vbox);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("Themes"),
			GTK_CTREE(ctree), NULL, NULL);
	create_theme_tab(vbox);

	vbox = create_config_page(GTK_NOTEBOOK(notebook_config), _("About"),
			GTK_CTREE(ctree), NULL, NULL);
	create_about_tab(vbox);

	/* Add plugin notebook pages last since they may need special add/remove
	|  actions if system wide plugins are enabled/disabled.
	*/
	for (list = gkrellm_monitor_list; list; list = list->next)
		{
		mon = (Monitor *) list->data;
		if (   ! mon->create_config
			|| ! mon->privat->enabled
			|| MONITOR_ID(mon) != MON_PLUGIN
		   )
			continue;
		add_plugin_config_page(mon);
		}
	if (expand_plugins)
		gtk_ctree_expand(GTK_CTREE(ctree), node_plugins);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), hbox, FALSE, FALSE, 0);

	button = gtk_button_new_with_label(_("OK"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) OK_config, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);
	gtk_widget_grab_default(button);

	button = gtk_button_new_with_label(_("Apply"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) apply_config, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

	button = gtk_button_new_with_label(_("Close"));
	GTK_WIDGET_SET_FLAGS(button, GTK_CAN_DEFAULT);
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
				(GtkSignalFunc) close_config, GINT_TO_POINTER(1));
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

	gtk_widget_show_all(config_window);
	}

void
gkrellm_open_config_window(Monitor *mon)
	{
	if (!mon || !mon->create_config)
		return;
	create_config_window();
	if (MONITOR_ID(mon) == MON_PLUGIN)
		{
		if (!GTK_CTREE_ROW(node_plugins)->expanded)
			gtk_ctree_expand(GTK_CTREE(config_ctree), node_plugins);
		}
	else if (!GTK_CTREE_ROW(node_builtins)->expanded)
		gtk_ctree_expand(GTK_CTREE(config_ctree), node_builtins);
	gtk_ctree_select(GTK_CTREE(config_ctree), mon->privat->ctree_config_node);
	}

static void
cb_debug(gpointer data, guint action, GtkWidget *widget)
	{
	gchar	*new;

	if (GK.debug_level & action)
		{
		GK.debug_level &= ~action;
		new = "OFF";
		}
	else
		{
		GK.debug_level |= action;
		new = "ON";
		}
	if (action == DEBUG_IMAGES)
		printf(_("==> Turning _%s_ theme images debugging.\n"), new);
	if (action == DEBUG_GKRELLMRC)
		printf(_("==> Turning _%s_ gkrellmrc debugging.\n"), new);
	if (action == DEBUG_STYLES)
		printf(_("==> Turning _%s_ style debugging.\n"), new);
	}


static GtkItemFactoryEntry	no_config_items[] =
	{
{"/-",				   NULL,	NULL,					0, "<Separator>"},
{N_("/Quit"),		   NULL,	gtk_main_quit,			0, "<Item>"},
{"/-",				   NULL,	NULL,					0, "<Separator>"},
	};

static GtkItemFactoryEntry	menu_items[] =
	{
{"/-",				   NULL,	NULL,					0, "<Separator>"},
{N_("/Configuration"), "F1",	create_config_window, 	0, "<Item>"},
{N_("/Theme/Prev"),		"p",	cb_load_theme,		THEME_ALT_PREV,	"<Item>"},
{N_("/Theme/Next"),		"n",	cb_load_theme,		THEME_ALT_NEXT,	"<Item>"},
{"/-",				   NULL,	NULL,					0, "<Separator>"},
{N_("/Quit"),		   NULL,	gtk_main_quit,			0, "<Item>"},
{"/-",				   NULL,	NULL,					0, "<Separator>"},
	};

static GtkItemFactoryEntry	debug_items[] =
  {
	{"/-",				    NULL,	NULL,					0, "<Separator>"},
	{N_("/Theme prev"), 	"u",	cb_load_theme,	THEME_PREV,	"<Item>"},
	{N_("/Theme next"),		"d",	cb_load_theme,	THEME_NEXT,	"<Item>"},
	{N_("/Reload Theme"),	"F5",	cb_load_theme,	THEME_RELOAD, "<Item>"},
	{N_("/Debug Images"),	"F6",	cb_debug,	DEBUG_IMAGES, 	"<Item>"},
	{N_("/Debug gkrellmrc"),"F7",	cb_debug,	DEBUG_GKRELLMRC,"<Item>"},
	{N_("/Debug styles"),	"F8",	cb_debug,	DEBUG_STYLES,	"<Item>"},
  };

static GtkItemFactory	*debug_factory;


GtkItemFactory *
create_item_factory_popup(void)
	{
	GtkWidget		*top_win;
	GtkItemFactory	*item_factory;
	GtkAccelGroup	*accel_group;
	gint			n,i;

	top_win = gkrellm_get_top_window();
	accel_group = gtk_accel_group_new ();
	gtk_window_add_accel_group(GTK_WINDOW(top_win), accel_group);

	n = sizeof(debug_items) / sizeof (GtkItemFactoryEntry);
	debug_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>", accel_group);
	gtk_item_factory_create_items(debug_factory, n, debug_items, NULL);

	if (GK.no_config)
		{
		n = sizeof(no_config_items) / sizeof (GtkItemFactoryEntry);
		item_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>",
				accel_group);
		for(i = 0; i < n; i++)
			no_config_items[i].path = _(no_config_items[i].path);
		gtk_item_factory_create_items(item_factory, n, no_config_items, NULL);
		}
	else
		{
		n = sizeof(menu_items) / sizeof (GtkItemFactoryEntry);
		item_factory = gtk_item_factory_new(GTK_TYPE_MENU, "<Main>",
				accel_group);
		for(i = 0; i < n; i++)
			menu_items[i].path = _(menu_items[i].path);
		gtk_item_factory_create_items(item_factory, n, menu_items, NULL);
		}
	return item_factory;
	}

