/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
 
/*
|  4/22/2001  Solaris code contributed by Daisuke Yabuki <dxy@acm.org>
|  1/29/2001  Hajimu UMEMOTO patched better networking code and IPv6 support
|               into the new TCP connection lookup.  Also improved
|               read_freebsd_tcp_port_data() with 1/23 patch.
| 10/12/2000  NetBSD code contributed by Anthony Mallet
|               <anthony.mallet@useless-ficus.net> 
|  2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
|               He also added IPv6 support for FreeBSD and Linux.
*/


#include "gkrellm.h"
#include "gkrellm_private_proto.h"
#include "gkrellm_threads.h"

#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>

/* Conditionals for INET6 and u_int32_t from
|  Hideaki YOSHIFUJI <yoshfuji@ecei.tohoku.ac.jp>
*/
/* Most INET6 for Linux needs libc 2.1, but sockaddr_storage is not in 2.1.1 */
#if defined(__linux__)
#if defined(__GLIBC__) && ((__GLIBC__>2) || (__GLIBC__==2 && __GLIBC_MINOR__>=2))
#define INET6
#endif
#if !defined(__GLIBC__) || __GLIBC__<2 || (__GLIBC__==2 && __GLIBC_MINOR__<1)
#define uint32_t        u_int32_t
#endif
#endif


#if defined(__FreeBSD__)
#include <osreldate.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/socketvar.h>
#include <sys/sysctl.h>
#include <net/route.h>
#if defined(__KAME__) && !defined(INET6)
#define INET6
#endif
#if defined(INET6)
#include <netinet6/ip6_var.h>
#endif
#include <netinet/in_pcb.h>
#include <netinet/ip_var.h>
#if defined(INET6)
#include <netinet/ip6.h>
#endif
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <netinet/tcp_seq.h>
#include <netinet/tcp_fsm.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

#if __FreeBSD_version < 300000
#if defined(INET6)
#include <netinet6/tcp6.h>
#include <netinet6/in6_pcb.h>
#include <netinet6/tcp6_timer.h>
#include <netinet6/tcp6_var.h>
#endif

#include <kvm.h>
#include <nlist.h>

static struct nlist nl[] = {
#define	N_TCB		0
	{ "_tcb" },
#if defined(INET6)
#define	N_TCB6		1
	{ "_tcb6" },
#endif
	{ "" },
};

extern	kvm_t	*kvmd;
extern	char	errbuf[];
#endif
#endif

#if defined(__solaris__)
/* IPv6 ? */
#include <netconfig.h>
#if defined(NC_INET6)
#define INET6
#endif
#endif /* __solaris__ */

typedef struct
	{
	GtkWidget	*vbox;
	gchar		*name;

	Chart		*chart;
	Chart		*chart_minute;
	Chart		*chart_hour;
	ChartConfig	*chart_config_minute;
	ChartConfig	*chart_config_hour;
	Panel		*panel;
	gboolean	hour_mode;
	gint		cd_length;

	gboolean	extra_info;
	Launcher	launch;
	GtkWidget	*launch_entry,
				*tooltip_entry;
	GtkWidget	*launch_table;

	gshort		*mark_data;		/* Draw marks if hits for any second */
	gint		mark_position,
				mark_prev_hits;

	Decal		*list_decal;
	DecalButton	*list_button;
	GString		*connection_string;
	GList		*tcp_save_list;
	gboolean	busy;
	gboolean	connection_string_event;

	gchar		*label0;
	gint		active0;
	gint		prev_active0;
	gulong		hits0_minute,
				hits0_hour;
	gboolean	data0_is_range;
	gulong		port0_0,
				port0_1;

	gchar		*label1;
	gint		active1;
	gint		prev_active1;
	gulong		hits1_minute,
				hits1_hour;
	gboolean	data1_is_range;
	gulong		port1_0,
				port1_1;

	gulong		krell_hits;
	}
	InetMon;


  /* Values for state.
  */
#define	TCP_DEAD	0
#define	TCP_ALIVE	1

typedef struct
	{
	gint			state;
	gint			family;
	gint			local_port;
	struct in_addr	remote_addr;
#if defined(INET6)
	struct in6_addr	remote_addr6;
#endif
	gint			remote_port;
	gint			new_hit;
	}
	ActiveTCP;

static GList		*inet_mon_list;

static GList		*active_tcp_list,
					*free_tcp_list;


/* ====== System dependent interface ====================================== */

void (*read_tcp_port_data)();

#if defined(__linux__) || defined(__FreeBSD__) || defined(__NetBSD__) \
	|| defined(__OpenBSD__) || defined(__solaris__)
static ActiveTCP *
tcp_alloc(void)
	{
	ActiveTCP	*tcp;

	if (free_tcp_list)
		{
		tcp = free_tcp_list->data;
		free_tcp_list = g_list_remove(free_tcp_list, tcp);
		}
	else
		tcp = g_new0(ActiveTCP, 1);
	return tcp;
	}

static ActiveTCP *
log_active_port(ActiveTCP *tcp)
	{
	GList		*list;
	ActiveTCP	*active_tcp, *new_tcp;
	gchar		*ap, *aap;
	gint		slen;

	for (list = active_tcp_list; list; list = list->next)
		{
		active_tcp = (ActiveTCP *) (list->data);
		if (tcp->family == AF_INET)
			{
			ap = (char *)&tcp->remote_addr;
			aap = (char *)&active_tcp->remote_addr;
			slen = sizeof(struct in_addr);
			}
#if defined(INET6)
		else if (tcp->family == AF_INET6)
			{
			ap = (char *)&tcp->remote_addr6;
			aap = (char *)&active_tcp->remote_addr6;
			slen = sizeof(struct in6_addr);
			}
#endif
		else
			return 0;
		if (   memcmp(aap, ap, slen) == 0
			&& active_tcp->remote_port == tcp->remote_port
			&& active_tcp->local_port == tcp->local_port
		   )
			{
			active_tcp->state = TCP_ALIVE;
			return active_tcp;	/* Old hit still alive, not a new hit	*/
			}
		}
	tcp->state = TCP_ALIVE;
	tcp->new_hit = 1;
	new_tcp = tcp_alloc();
	*new_tcp = *tcp;
	active_tcp_list = g_list_prepend(active_tcp_list, (gpointer) new_tcp);
	return new_tcp;		/* A new hit	*/
	}

static void
log_tcp_port_data(ActiveTCP *tcp)
	{
	GList		*list;
	InetMon		*in;
	ActiveTCP	*active_tcp = NULL;
	gint		krell_hit;

	for (list = inet_mon_list; list; list = list->next)
		{
		krell_hit = 0;
		in = (InetMon *) list->data;
		if (   (!in->data0_is_range
				&& (   in->port0_0 == tcp->local_port
					|| in->port0_1 == tcp->local_port))
			|| (in->data0_is_range
				&& tcp->local_port >= in->port0_0
				&& tcp->local_port <= in->port0_1)
		   )
			{
			++in->active0;
			active_tcp = log_active_port(tcp);
			krell_hit = active_tcp->new_hit;
			in->hits0_minute += krell_hit;
			in->hits0_hour   += krell_hit;
			}
		if (   (!in->data1_is_range
				&& (   in->port1_0 == tcp->local_port
					|| in->port1_1 == tcp->local_port))
			|| (in->data1_is_range
				&& tcp->local_port >= in->port1_0
				&& tcp->local_port <= in->port1_1)
		   )
			{
			++in->active1;
			active_tcp = log_active_port(tcp);
			krell_hit = active_tcp->new_hit;
			in->hits1_minute += krell_hit;
			in->hits1_hour   += krell_hit;
			}
		in->krell_hits += krell_hit;
		}
	/* Defer setting new hit to 0 until here so multiple inet charts can
	|  monitor the same port number.  The above active_port will be the
	|  same if found for both data0 and data1.
	*/
	if (active_tcp)
		active_tcp->new_hit = 0;
	}

#endif	/* __LINUX__ || __FreeBSD__ || __NetBSD__ || __OpenBSD__ || \
	__solaris__ */



/* ----- FreeBSD ------------------------------------------------------- */

#if defined(__FreeBSD__)
#if __FreeBSD_version < 300000
static void
read_freebsd_tcp_port_data()
	{
	ActiveTCP	tcp;
	gint		tcp_status;
	struct		inpcbhead head;
	struct		inpcb inpcb, *next;
#if defined(INET6)
	struct		in6pcb in6pcb, *prev6, *next6;
	u_long		off = nl[N_TCB6].n_value;
#endif

	if (kvmd == NULL)
		return;
	if (nl[0].n_type == 0)
		if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)
			return;
	if (kvm_read(kvmd, nl[N_TCB].n_value, (char *)&head,
		     sizeof(struct inpcbhead)) != sizeof(struct inpcbhead))
		return;

	for (next = head.lh_first; next != NULL;
	     next = inpcb.inp_list.le_next)
		{
		struct tcpcb tcpcb;

		if (kvm_read(kvmd, (u_long)next, (char *)&inpcb,
			     sizeof(inpcb)) != sizeof(inpcb))
			return;
		if (kvm_read(kvmd, (u_long)inpcb.inp_ppcb, (char *)&tcpcb,
			     sizeof(tcpcb)) != sizeof(tcpcb))
			return;
		tcp.local_port = ntohs(inpcb.inp_lport);
		tcp.remote_addr.s_addr = inpcb.inp_faddr.s_addr;
		tcp.remote_port = ntohs(inpcb.inp_fport);
		tcp_status = (tcpcb.t_state == TCPS_ESTABLISHED);
		tcp.family = AF_INET;
		if (tcp_status == TCP_ALIVE)
			log_tcp_port_data(&tcp);
		}
#if defined(INET6)
	if (kvm_read(kvmd, off, (char *)&in6pcb,
		         sizeof(struct in6pcb)) != sizeof(struct in6pcb))
		return;
	prev6 = (struct in6pcb *)off;
	if (in6pcb.in6p_next == (struct in6pcb *)off)
		return;
	while (in6pcb.in6p_next != (struct in6pcb *)off)
		{
		struct tcp6cb tcp6cb;

		next6 = in6pcb.in6p_next;
		if (kvm_read(kvmd, (u_long)next6, (char *)&in6pcb,
			     sizeof(in6pcb)) != sizeof(in6pcb))
			return;
		if (in6pcb.in6p_prev != prev6)
			break;
		if (kvm_read(kvmd, (u_long)in6pcb.in6p_ppcb, (char *)&tcp6cb,
			     sizeof(tcp6cb)) != sizeof(tcp6cb))
			return;
		tcp.local_port = ntohs(in6pcb.in6p_lport);
		memcpy(&tcp.remote_addr6, &in6pcb.in6p_faddr,
		       sizeof(struct in6_addr));
		tcp.remote_port = ntohs(in6pcb.in6p_fport);
		tcp_status = (tcp6cb.t_state == TCPS_ESTABLISHED);
		tcp.family = AF_INET6;
		if (tcp_status == TCP_ALIVE)
			log_tcp_port_data(&tcp);
		prev6 = next6;
		}
#endif
	}
#else
static void
read_freebsd_tcp_port_data()
	{
	static char	*name = "net.inet.tcp.pcblist";
	static int	oid_name2oid[2] = { 0, 3 };
	static int	oid_pcblist[CTL_MAXNAME + 2];
	static size_t	oid_pcblist_len = sizeof(oid_pcblist);
	static gint	initialized = 0;
	ActiveTCP	tcp;
	gint		tcp_status;
	struct xinpgen	*xig, *oxig;
	gchar		*buf;
	gint		len = 0;

	if (!initialized)
		{
		if (sysctl(oid_name2oid, 2, oid_pcblist, &oid_pcblist_len,
			   (void *)name, strlen(name)) < 0)
			return;
		oid_pcblist_len /= sizeof(int);
		++initialized;
		}
	if (sysctl(oid_pcblist, oid_pcblist_len, 0, &len, 0, 0) < 0)
		return;
	if ((buf = malloc(len)) == 0)
		return;
	if (sysctl(oid_pcblist, oid_pcblist_len, buf, &len, 0, 0) >= 0)
		{
		oxig = xig = (struct xinpgen *)buf;
		for (xig = (struct xinpgen *)((char *)xig + xig->xig_len);
		     xig->xig_len > sizeof(struct xinpgen);
		     xig = (struct xinpgen *)((char *)xig + xig->xig_len))
			{
			struct tcpcb *tp = &((struct xtcpcb *)xig)->xt_tp;
			struct inpcb *inp = &((struct xtcpcb *)xig)->xt_inp;
			struct xsocket *so = &((struct xtcpcb *)xig)->xt_socket;

			/* Ignore sockets for protocols other than tcp. */
			if (so->xso_protocol != IPPROTO_TCP)
				continue;

			/* Ignore PCBs which were freed during copyout. */
			if (inp->inp_gencnt > oxig->xig_gen)
				continue;

#if defined(INET6)
			if (inp->inp_vflag & INP_IPV4)
				{
				tcp.remote_addr.s_addr = inp->inp_faddr.s_addr;
				tcp.family = AF_INET;
				}
			else if (inp->inp_vflag & INP_IPV6)
				{
					memcpy(&tcp.remote_addr6,
					       &inp->in6p_faddr,
					       sizeof(struct in6_addr));
					tcp.family = AF_INET6;
				}
			else
				continue;
#else
			tcp.remote_addr.s_addr = inp->inp_faddr.s_addr;
			tcp.family = AF_INET;
#endif
			tcp.remote_port = ntohs(inp->inp_fport);
			tcp.local_port = ntohs(inp->inp_lport);
			tcp_status = (tp->t_state == TCPS_ESTABLISHED);
			if (tcp_status == TCP_ALIVE)
				log_tcp_port_data(&tcp);
			}
		}
	free(buf);
	}
#endif
#endif


/* ----- Linux -------------------------------------------------------- */
#if defined(__linux__)

#define	PROC_NET_TCP_FILE	"/proc/net/tcp"
#if defined(INET6)
#define	PROC_NET_TCP6_FILE	"/proc/net/tcp6"
#endif

static void
read_linux_tcp_port_data(void)
	{
	FILE		*f;
	ActiveTCP	tcp;
	gchar		buf[512];
	gint		tcp_status;
	gulong		addr;

	if ((f = fopen(PROC_NET_TCP_FILE, "r")) != NULL)
		{
		fgets(buf, sizeof(buf), f);		/* Waste first line */

		while (fgets(buf, sizeof(buf), f))
			{
			sscanf(buf, "%*d: %*x:%x %lx:%x %x", &tcp.local_port,
						&addr, &tcp.remote_port, &tcp_status);
			tcp.remote_addr.s_addr = (uint32_t) addr;
			tcp.family = AF_INET;
			if (tcp_status != TCP_ALIVE)
				continue;
			log_tcp_port_data(&tcp);
			}
		fclose(f);
		}
#if defined(INET6)
	if ((f = fopen(PROC_NET_TCP6_FILE, "r")) != NULL)
		{
		fgets(buf, sizeof(buf), f);		/* Waste first line */
		while (fgets(buf, sizeof(buf), f))
			{
			sscanf(buf, "%*d: %*x:%x %8x%8x%8x%8x:%x %x",
			       &tcp.local_port,
			       &tcp.remote_addr6.s6_addr32[0],
			       &tcp.remote_addr6.s6_addr32[1],
			       &tcp.remote_addr6.s6_addr32[2],
			       &tcp.remote_addr6.s6_addr32[3],
			       &tcp.remote_port, &tcp_status);
			tcp.family = AF_INET6;
			if (tcp_status != TCP_ALIVE)
				continue;
			log_tcp_port_data(&tcp);
			}
		fclose(f);
		}
#endif	/* INET6 */
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__) /* NO IPv6 SUPPORT YET */

#include <net/route.h>
#include <netinet/in_systm.h>
#include <netinet/ip.h>
#include <netinet/in_pcb.h>

#include <netinet/ip_var.h>
#include <netinet/tcp.h>
#include <netinet/tcpip.h>
#include <netinet/tcp_seq.h>
#include <netinet/tcp_fsm.h>
#include <netinet/tcp_timer.h>
#include <netinet/tcp_var.h>

#include <kvm.h>

static struct nlist nl[] = {
#define X_TCBTABLE    0
   { "_tcbtable" },
   { NULL }
};

extern	kvm_t	*kvmd;


void
read_netbsd_tcp_port_data()
{
   ActiveTCP tcp;
   gint	tcp_status;
   struct inpcbtable table;
   struct inpcb inpcb, *next;
   struct tcpcb tcpcb;

   if (kvmd == NULL) return;
   if (nl[0].n_type == 0)
      if (kvm_nlist(kvmd, nl) < 0 || nl[0].n_type == 0)	return;

   if (kvm_read(kvmd, nl[X_TCBTABLE].n_value, (char *)&table,
		sizeof(struct inpcbtable)) != sizeof(struct inpcbtable))
      return;

   next = table.inpt_queue.cqh_first;
   while(next != NULL) {
      
      if (kvm_read(kvmd, (u_long)next,
		   (char *)&inpcb, sizeof(inpcb)) == sizeof(inpcb) &&
	  kvm_read(kvmd, (u_long)inpcb.inp_ppcb,
		   (char *)&tcpcb, sizeof(tcpcb)) == sizeof(tcpcb)) {

	 tcp.local_port = ntohs(inpcb.inp_lport);
	 tcp.remote_addr.s_addr = inpcb.inp_faddr.s_addr;
	 tcp.remote_port = ntohs(inpcb.inp_fport);
	 tcp_status = (tcpcb.t_state == TCPS_ESTABLISHED);
	 tcp.family = AF_INET;
	 if (tcp_status == TCP_ALIVE)
	    log_tcp_port_data(&tcp);
      }

      next = inpcb.inp_queue.cqe_next;
   }
}

#endif /* __NetBSD__ || __OpenBSD__ */

/* ----- Solaris -------------------------------------------------------- */
#if defined(__solaris__)

#include <stropts.h>
#include <inet/mib2.h>
#include <fcntl.h>
#include <sys/tihdr.h>

void
read_solaris_tcp_port_data() {

    ActiveTCP tcp;
    gint tcp_status;

    static int tcpfd = 0;

    mib2_tcpConnEntry_t *tp;
#if defined(INET6)
    mib2_tcp6ConnEntry_t *tp6;
#endif 

    char buf[512];
    int i, flags, getcode, num_ent;
    struct strbuf ctlbuf, databuf;
    struct T_optmgmt_req *tor = (struct T_optmgmt_req *)buf;
    struct T_optmgmt_ack *toa = (struct T_optmgmt_ack *)buf;
    struct T_error_ack   *tea = (struct T_error_ack *)buf;
    struct opthdr        *mibhdr;

    if (tcpfd == 0) {
        if ((tcpfd = open("/dev/tcp", O_RDWR)) == -1) {
            perror("open");
        }
    }

    tor->PRIM_type = T_SVR4_OPTMGMT_REQ;
    tor->OPT_offset = sizeof (struct T_optmgmt_req);
    tor->OPT_length = sizeof (struct opthdr);
    tor->MGMT_flags = T_CURRENT;
    mibhdr = (struct opthdr *)&tor[1];
    mibhdr->level = MIB2_TCP;
    mibhdr->name  = 0;
    mibhdr->len   = 0;

    ctlbuf.buf = buf;
    ctlbuf.len = tor->OPT_offset + tor->OPT_length;
    flags = 0; /* request to be sent in non-priority */

    if (putmsg(tcpfd, &ctlbuf, (struct strbuf *)0, flags) == -1) {
        perror("putmsg");
    }

    mibhdr = (struct opthdr *)&toa[1];
    ctlbuf.maxlen = sizeof (buf);

    /* now receiving response from stream */

    for (;;) {
        flags = 0; /* read any messages available */
        getcode = getmsg(tcpfd, &ctlbuf, (struct strbuf *)0, &flags);

        if (getcode != MOREDATA ||
                 ctlbuf.len < sizeof (struct T_optmgmt_ack) ||
                 toa->PRIM_type != T_OPTMGMT_ACK ||
                 toa->MGMT_flags != T_SUCCESS) {
             break;
        } 

        if (ctlbuf.len >= sizeof (struct T_error_ack) &&
                 tea->PRIM_type == T_ERROR_ACK) {
             perror("ERROR_ACK");
             return;
        }

        if (getcode == 0 &&
                 ctlbuf.len >= sizeof (struct T_optmgmt_ack) &&
                 toa->PRIM_type == T_OPTMGMT_ACK &&
                 toa->MGMT_flags == T_SUCCESS) {
             return;
        } 

        /* prepare for receiving data */
        databuf.maxlen = mibhdr->len;
        databuf.len    = 0;
        databuf.buf    = (char *)malloc((int)mibhdr->len);
        if(!databuf.buf) {
            perror("malloc");
            break;
        }
        flags = 0;

        getcode = getmsg(tcpfd, (struct strbuf *)0, &databuf, &flags);

        if (mibhdr->level == MIB2_TCP && mibhdr->name == MIB2_TCP_13) {
            tp = (mib2_tcpConnEntry_t *)databuf.buf;
            num_ent = mibhdr->len / sizeof(mib2_tcpConnEntry_t);
            for (i = 0; i < num_ent; i++, tp++) {
                if (tp->tcpConnState != MIB2_TCP_established)
                    continue;
                tcp.local_port         = tp->tcpConnLocalPort;
                tcp.remote_addr.s_addr = tp->tcpConnRemAddress;
                tcp.remote_port        = tp->tcpConnRemPort;
                tcp.family             = AF_INET;
                tcp_status = (tp->tcpConnState == MIB2_TCP_established);
                if (tcp_status == TCP_ALIVE)
                    log_tcp_port_data(&tcp);
            }
        }

#if defined(INET6)
        if (mibhdr->level == MIB2_TCP6 && mibhdr->name == MIB2_TCP6_CONN) {
            tp6 = (mib2_tcp6ConnEntry_t *)databuf.buf;
            num_ent = mibhdr->len / sizeof(mib2_tcp6ConnEntry_t);
            for (i = 0; i < num_ent; i++, tp6++) {
                if (tp6->tcp6ConnState != MIB2_TCP_established)
                    continue;
                tcp.local_port          = tp6->tcp6ConnLocalPort;
                tcp.remote_port         = tp6->tcp6ConnRemPort;
                memcpy(&tcp.remote_addr6, &tp6->tcp6ConnRemAddress, 
			sizeof(struct in6_addr));
                tcp.family              = AF_INET6;
                tcp_status = (tp6->tcp6ConnState == MIB2_TCP_established);
                if (tcp_status == TCP_ALIVE)
                    log_tcp_port_data(&tcp);
            }  
        }
#endif /* INET6 */

        free(databuf.buf);
    }

}

#endif /* __solaris__ */

/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_inet_interface(void)
    {
	gint	available = FALSE;

#if defined(__FreeBSD__)
	read_tcp_port_data = read_freebsd_tcp_port_data;
	available = TRUE;
#endif

#if defined(__linux__)
	read_tcp_port_data = read_linux_tcp_port_data;
	available = TRUE;
#endif

#if defined(__NetBSD__) || defined(__OpenBSD__)
	read_tcp_port_data = read_netbsd_tcp_port_data;
	available = TRUE;
#endif

#if defined(__solaris__)
	read_tcp_port_data = read_solaris_tcp_port_data;
	available = TRUE;
#endif

    return available;
    }

/* ======================================================================== */
static Monitor		*mon_inet;

static GtkWidget	*inet_vbox;
static GdkImage		*grid;

static gchar		*inet_data_dir;
static gint			n_inet_monitors;

static gint			style_id;

static gchar		*text_format;

static void
format_chart_text(InetMon *in, gchar *buf, gint size)
	{
	Chart	*cp;
	gchar	c, *s, *s1;
	gint	len, value;

	--size;
	*buf = '\0';
	cp = in->chart;
	for (s = text_format; *s != '\0' && size > 0; ++s)
		{
		len = 1;
		if (*s == '$' && *(s + 1) != '\0')
			{
			value = -1;
			s1 = " ";
			if ((c = *(s + 1)) == 'M')
				value = gkrellm_get_chart_scalemax(cp);
			else if (c == 'a' && *in->label0)
				value = in->active0;
			else if (c == 'l' && *in->label0)
				s1 = in->label0;
			else if (c == 'A' && *in->label1)
				value = in->active1;
			else if (c == 'L' && *in->label1)
				s1 = in->label1;
			if (value >= 0)
				len = snprintf(buf, size, "%d", value);
			else
				len = snprintf(buf, size, "%s", s1);
			++s;
			}
		else
			*buf = *s;
		size -= len;
		buf += len;
		}
	*buf = '\0';	
	}

static void
draw_inet_extra(InetMon *in)
	{
	gchar	buf[128];

	if (!in->extra_info)
		return;
	format_chart_text(in, buf, sizeof(buf));
	gkrellm_draw_chart_text(in->chart, style_id, buf);
	}

  /* Use the reserved area below the main chart to draw marks if any
  |  hits in second intervals.
  */
static void
draw_inet_mark_data(InetMon *in, gint minute_mark)
	{
	Chart	*cp;
	GdkGC	*gc1, *gc2, *gc3;
	gint	hits, x, y, n;

	cp = in->chart;
	in->mark_position = (in->mark_position + 1) % cp->w;
	if (minute_mark)
		{
		in->mark_data[in->mark_position] = -1;	/* minute flag in the data */
		return;
		}
	hits = in->hits0_minute + in->hits1_minute;
	in->mark_data[in->mark_position] = hits - in->mark_prev_hits;
	in->mark_prev_hits = hits;

	gc1 = gkrellm_draw_GC(1);
	gc2 = gkrellm_draw_GC(2);
	gc3 = gkrellm_draw_GC(3);

	/* Clear out the area and redraw the marks.
	*/
	y = cp->h - cp->y;
	gdk_draw_pixmap(cp->pixmap, gc1, cp->bg_src_pixmap,
			0, y,  0, y,  cp->w, cp->y);
	gdk_gc_set_foreground(gc1, gkrellm_out_color());
	gdk_gc_set_foreground(gc2, gkrellm_in_color());
	gdk_gc_set_foreground(gc3, gkrellm_white_color());
	for (n = 0; n < cp->w; ++n)
		{
		x = (in->mark_position + n + 1) % cp->w;
		if (in->mark_data[x] > 0)
			gdk_draw_line(cp->pixmap, gc1,
						cp->x + n, cp->h - 1, cp->x + n, y);
		else if (in->mark_data[x] == -1)	/* Minute tick	*/
			gdk_draw_line(cp->pixmap, gc3,
						cp->x + n, cp->h - 1, cp->x + n, y);
		}
	gdk_draw_pixmap(cp->drawing_area->window, gc1, cp->pixmap,
			0, y,  0, y,  cp->w, cp->y);
	}

static void
draw_inet_chart(InetMon *in)
	{
	struct tm	tm;
	GdkGC		*gc3;
	Chart		*cp;
	TextStyle	*ts;
	GdkColor	tmp_color;
	gchar		buf[32];
	guint32		pixel0, pixel1;
	gint		y0, h4, h, n;

	cp = in->chart;
	gkrellm_draw_chartdata(cp);

	y0 = cp->h - cp->y;
	h4 = y0 / 4;
	h = ((GdkWindowPrivate *) in->chart->bg_grid_pixmap)->height;
	if (grid == NULL)
		grid = gdk_image_get(in->chart->bg_grid_pixmap,
						0, 0, gkrellm_chart_width(), h);
	ts = gkrellm_chart_alt_textstyle(style_id);

	tm = *gkrellm_get_current_time();
	gc3 = gkrellm_draw_GC(3);
	gdk_gc_set_foreground(gkrellm_draw_GC(3), gkrellm_white_color());
	if (in->hour_mode)
		{
		for (n = cp->w - 1; n >= 0; --n)
			{
			/* When hour ticked to 0, 23rd hour data was stored and a slot
			|  was skipped.
			*/
			if (tm.tm_hour == 0)	/* Draw day mark at midnight	*/
				{
				pixel0 = gdk_image_get_pixel(grid, cp->x + n, 0);
				tmp_color.pixel = pixel0;
				gdk_gc_set_foreground(gc3, &tmp_color);
				gdk_draw_line(cp->pixmap, gc3,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, 3);
				if (h > 1)
					{
					pixel1 = gdk_image_get_pixel(grid, cp->x + n, 1);
					tmp_color.pixel = pixel1;
					gdk_gc_set_foreground(gc3, &tmp_color);
					gdk_draw_line(cp->pixmap, gc3,
							cp->x + n, y0 - 3, cp->x + n, 3);
					}
				}
			if (in->extra_info && tm.tm_hour == 1 && n < cp->w - 5)
				{
				strftime(buf, sizeof(buf), "%a", &tm);
				buf[1] = '\0';
				gkrellm_draw_chart_label(in->chart, ts,
						cp->x + n, in->chart->h - 4, buf);
				}
			if (--tm.tm_hour < 0)
				{
				tm.tm_hour = 24;		/* Extra hour for skipped slot	*/
				if (--tm.tm_wday < 0)
					tm.tm_wday = 6;
				}
			}
		}
	else
		{
		for (n = cp->w - 1; n >= 0; --n)
			{
			/* When minute ticked to 0, 59 minute data was stored and a slot
			|  was skipped.
			*/
			if (tm.tm_min == 0)		/* Draw hour mark	*/
				{
				pixel0 = gdk_image_get_pixel(grid, cp->x + n, 0);
				pixel1 = gdk_image_get_pixel(grid, cp->x + n, 1);

				tmp_color.pixel = pixel0;
				gdk_gc_set_foreground(gc3, &tmp_color);
				gdk_draw_line(cp->pixmap, gc3,
						cp->x + n - 1, y0 - 3, cp->x + n - 1, y0 - h4);
				tmp_color.pixel = pixel1;
				gdk_gc_set_foreground(gc3, &tmp_color);
				gdk_draw_line(cp->pixmap, gc3,
						cp->x + n, y0 - 3, cp->x + n, y0 - h4);
				}
			if (--tm.tm_min < 0)
				tm.tm_min = 60;		/* extra minute for skipped slot */
			}
		}
	if (in->extra_info)
		draw_inet_extra(in);
	gkrellm_draw_chart_to_screen(cp);
	in->prev_active0 = in->active0;
	in->prev_active1 = in->active1;
	}

static void
select_hour_or_minute_chart(InetMon *in)
	{
	gkrellm_freeze_side_frame_packing();
	if (in->hour_mode && in->chart == in->chart_minute)
		{
		gkrellm_chart_hide(in->chart_minute, FALSE);
		gkrellm_chart_show(in->chart_hour, FALSE);
		in->chart = in->chart_hour;
		gkrellm_chartconfig_window_destroy(in->chart_minute);
		}
	else if (!in->hour_mode && in->chart == in->chart_hour)
		{
		gkrellm_chart_hide(in->chart_hour, FALSE);
		gkrellm_chart_show(in->chart_minute, FALSE);
		in->chart = in->chart_minute;
		gkrellm_chartconfig_window_destroy(in->chart_hour);
		}
	gkrellm_thaw_side_frame_packing();
	}

static void
update_inet(void)
	{
	InetMon		*in;
	Chart		*cp;
	ActiveTCP	*tcp;
	GList		*list;
	gchar		buf[32];
	gint		i;

	if (inet_mon_list == NULL)
		return;

	if (GK.second_tick || GK.minute_tick || GK.hour_tick)
		{
		for (list = inet_mon_list; list; list = list->next)
			{
			in = (InetMon *) list->data;
			in->active0 = 0;
			in->active1 = 0;
			}
		/* Assume all connections are dead, then read_tcp_port_data() will set
		|  still alive ones back to alive.  Then I can prune really dead ones.
		*/
		for (list = active_tcp_list; list; list = list->next)
			{
			tcp = (ActiveTCP *)(list->data);
			tcp->state = TCP_DEAD;
			}

		(*read_tcp_port_data)();

		for (list = active_tcp_list; list; )
			{
			tcp = (ActiveTCP *)(list->data);
			if (tcp->state == TCP_DEAD)
				{
				if (list == active_tcp_list)
					active_tcp_list = active_tcp_list->next;
				list = g_list_remove(list, tcp);
				free_tcp_list = g_list_prepend(free_tcp_list, tcp);
				}
			else
				list = list->next;
			}
		}

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		cp = in->chart;
		if (GK.hour_tick)
			{
			if (!*in->label0)
				in->hits0_hour = in->hits1_hour;
			gkrellm_store_chartdata(in->chart_hour, 0,
					in->hits0_hour, in->hits1_hour);
			in->hits0_hour = in->hits1_hour = 0;
			if (GK.day_tick)	/* Make room for vertical day grid */
				{
				gkrellm_store_chartdata(in->chart_hour, 0, 0, 0);
				gkrellm_store_chartdata(in->chart_hour, 0, 0, 0);
				}
			}
		if (GK.minute_tick)
			{
			if (!*in->label0)
				in->hits0_minute = in->hits1_minute;
			gkrellm_store_chartdata(in->chart_minute, 0,
					in->hits0_minute, in->hits1_minute);
			in->hits0_minute = in->hits1_minute = 0;
			if (GK.hour_tick)	/* Make room for vertical hour grid */
				{
				gkrellm_store_chartdata(in->chart_minute, 0, 0, 0);
				gkrellm_store_chartdata(in->chart_minute, 0, 0, 0);
				}
			gkrellm_refresh_chart(in->chart);
			draw_inet_mark_data(in, 1);
			}
		else if (   GK.second_tick
				&& (   in->prev_active0 != in->active0
					|| in->prev_active1 != in->active1
				   )
				)
				draw_inet_chart(in);	/* Just to update extra info draw */

		if (GK.second_tick)
			draw_inet_mark_data(in, 0);

		if (in->busy && in->list_button->cur_index == D_MISC_BUTTON_OUT)
			i = D_MISC_BUTTON_ON;
		else
			i = D_MISC_BUTTON_OUT;
		gkrellm_set_decal_button_index(in->list_button, i);

		gkrellm_update_krell(in->panel, KRELL(in->panel), in->krell_hits);
		gkrellm_draw_panel_layers(in->panel);

		if (in->connection_string_event)
			{
			snprintf(buf, sizeof(buf), _("%s Connections"), in->name);
			gkrellm_message_window(buf, in->connection_string->str, NULL);
			in->connection_string_event = FALSE;
			in->busy = FALSE;
			}
		}
	}

static gboolean
tcp_port_is_monitored(ActiveTCP *tcp, gboolean range, gulong p0, gulong p1)
	{
	if (   (!range && (p0 == tcp->local_port || p1 == tcp->local_port))
		|| ( range && tcp->local_port >= p0 && tcp->local_port <= p1)
	   )
		return TRUE;
	return FALSE;
	}

#if defined(GKRELLM_THREADS)
static void
get_connection_string_thread(void *data)
	{
	InetMon			*in	= (InetMon *) data;
	GList			*list;
	ActiveTCP		*tcp;
#if defined(INET6)
	struct sockaddr_storage	ss;
	struct sockaddr_in	*sin;
	struct sockaddr_in6	*sin6;
	gint			salen, flag = 0;
	gchar			hbuf[NI_MAXHOST];
	gchar			buf[NI_MAXHOST + 10];
#else
	struct hostent	*hostent;
	gchar			buf[64];
#endif

	if (in->connection_string)
		in->connection_string = g_string_truncate(in->connection_string, 0);
	else
		in->connection_string = g_string_new("");
	for (list = in->tcp_save_list; list; list = list->next)
		{
		tcp = (ActiveTCP *) list->data;
#if defined(INET6)
		memset(&ss, 0, sizeof(ss));
		switch (tcp->family)
			{
			case AF_INET:
				sin = (struct sockaddr_in *)&ss;
				salen = sizeof(struct sockaddr_in);
				memcpy(&sin->sin_addr,
						&tcp->remote_addr, salen);
#if defined(SIN6_LEN)
				sin->sin_len = salen;
#endif
				sin->sin_family = tcp->family;
				break;
			case AF_INET6:
				sin6 = (struct sockaddr_in6 *)&ss;
				salen = sizeof(struct sockaddr_in6);
				memcpy(&sin6->sin6_addr,
						&tcp->remote_addr6, salen);
#if defined(SIN6_LEN)
				sin6->sin6_len = salen;
#endif
				sin6->sin6_family = tcp->family;
				/* XXX: We should mention about
				|  scope, too. */
				break;
			default:
				continue;
			}
		if (getnameinfo((struct sockaddr *)&ss, salen,
					hbuf, sizeof(hbuf), NULL, 0, flag))
			continue;
		snprintf(buf, sizeof(buf), "%6d:  %s\n",
					tcp->local_port, hbuf);
#else
		hostent = gethostbyaddr((char *) &tcp->remote_addr,
					sizeof(struct in_addr), AF_INET);
		if (hostent)
			snprintf(buf, sizeof(buf),
				 "%6d:  %s\n", tcp->local_port,
				 hostent->h_name);
		else
			snprintf(buf, sizeof(buf),
				 "%6d:  %s\n", tcp->local_port,
				 inet_ntoa(tcp->remote_addr));
#endif
		g_string_append(in->connection_string, buf);
		}
	if (in->connection_string->len == 0)
		g_string_append(in->connection_string, _("No current connections."));
	in->connection_string_event = TRUE;
	free_glist_and_data(&in->tcp_save_list);
	}
#endif

static void
cb_list_button(DecalButton *button)
    {
#if defined(GKRELLM_THREADS)
	InetMon			*in	= (InetMon *) button->data;
	GList			*list;
	ActiveTCP		*tcp, *tcp_save;
	gpointer		thread_id;

	if (in->busy)
		return;
	in->busy = TRUE;

	/* Save a snapshot of active connections so I don't have to worry about
	|  the active_tcp_list changing while in the thread.
	*/
	for (list = active_tcp_list; list; list = list->next)
		{
		tcp = (ActiveTCP *) list->data;
		if (   tcp_port_is_monitored(tcp, in->data0_is_range,
						in->port0_0, in->port0_1)
			|| tcp_port_is_monitored(tcp, in->data1_is_range,
						in->port1_0, in->port1_1)
		   )
			{
			tcp_save = g_new0(ActiveTCP, 1);
			*tcp_save = *tcp;
			in->tcp_save_list = g_list_append(in->tcp_save_list, tcp_save);
			}
		}
	thread_id = new_thread(get_connection_string_thread, in);
#endif
	}

static gint
inet_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	InetMon		*in;
	GList		*list;
	GdkPixmap	*pixmap = NULL;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (widget == in->panel->drawing_area)
			pixmap = in->panel->pixmap;
		else if (widget == in->chart_minute->drawing_area)
			pixmap = in->chart_minute->pixmap;
		else if (widget == in->chart_hour->drawing_area)
			pixmap = in->chart_hour->pixmap;
		if (pixmap)
			{
			gdk_draw_pixmap(widget->window, gkrellm_draw_GC(1), pixmap,
					ev->area.x, ev->area.y, ev->area.x, ev->area.y,
					ev->area.width, ev->area.height);
			break;
			}
		}
	return FALSE;
	}

static gint
cb_inet_extra(GtkWidget *widget, GdkEventButton *ev)
	{
	InetMon		*in;
	GList		*list;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (widget != in->chart->drawing_area)
			continue;
		if (ev->button == 1 && ev->type == GDK_BUTTON_PRESS)
			{
			in->extra_info = !in->extra_info;
			gkrellm_refresh_chart(in->chart);
			gkrellm_config_modified();
			}
		else if (ev->button == 2)
			{
			in->hour_mode = !in->hour_mode;
			select_hour_or_minute_chart(in);
			gkrellm_rescale_chart(in->chart);
			}
		else if (   ev->button == 3
				 || (ev->button == 1 && ev->type == GDK_2BUTTON_PRESS)
				)
			gkrellm_chartconfig_window_create(in->chart);
		break;
		}
	return TRUE;
	}

static gint
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	if (ev->button == 3)
		gkrellm_open_config_window(mon_inet);
	return TRUE;
	}

  /* Lock the hour and minute heights together.
  */
static void
cb_inet_height(ChartConfig *cf, InetMon *in)
	{
	gint	h;

	h = gkrellm_get_chartconfig_height(cf);
	if (in->chart_minute->h != h)
		gkrellm_set_chart_height(in->chart_minute, h);
	if (in->chart_hour->h != h)
		gkrellm_set_chart_height(in->chart_hour, h);
	}

static void
destroy_inet_monitor(InetMon *in)
	{
	g_free(in->name);
	g_free(in->label0);
	g_free(in->label1);
	if (in->launch.command)
		g_free(in->launch.command);
	if (in->launch.button)
		gkrellm_destroy_button(in->launch.button);
    in->launch.tooltip = NULL;
	g_free(in->mark_data);

	/* The panel doesn't live in the chart struct, so destroy it separately
	*/
	gkrellm_panel_destroy(in->panel);

	gkrellm_chartconfig_destroy(&in->chart_config_minute);
	gkrellm_chart_destroy(in->chart_minute);

	gkrellm_chartconfig_destroy(&in->chart_config_hour);
	gkrellm_chart_destroy(in->chart_hour);

	gtk_widget_destroy(in->vbox);
	g_free(in);
	}


#define	MIN_GRID_RES		2
#define	MAX_GRID_RES		1000000
#define DEFAULT_GRID_RES	10

static void
chart_create(InetMon *in, Chart *cp, ChartConfig **cfp, gint first_create)
	{
	ChartConfig	*cf;
	ChartData	*cd;
	GdkPixmap	**src_pixmap, *grid_pixmap;

	cp->y = 3;
	gkrellm_chart_create(in->vbox, mon_inet, cp, cfp);
	cf = *cfp;

	/* I accumulate tcp hits myself, so I'm free to make the chartdata
	|  accumulate monotonically or not.  I choose not monotonic to make saving
	|  and loading data simpler.
	*/
	src_pixmap = gkrellm_data_out_pixmap();
	grid_pixmap = gkrellm_data_out_grid_pixmap();
	if (*in->label0)
		{
		cd = gkrellm_add_chartdata(cp, src_pixmap, grid_pixmap, in->label0);
		gkrellm_monotonic_chartdata(cd, FALSE);
		}
	src_pixmap = gkrellm_data_in_pixmap();
	grid_pixmap = gkrellm_data_in_grid_pixmap();
	if (*in->label1)
		{
		cd = gkrellm_add_chartdata(cp, src_pixmap, grid_pixmap, in->label1);
		gkrellm_monotonic_chartdata(cd, FALSE);
		}
	gkrellm_set_draw_chart_function(cp, draw_inet_chart, in);

	/* krell is not function of chart grids or resolution, so no interest
	|  in connecting to grid or resolution changes.
	*/
	gkrellm_chartconfig_height_connect(cf, cb_inet_height, in);
	gkrellm_chartconfig_grid_resolution_adjustment(cf, TRUE,
			0, (gfloat) MIN_GRID_RES, (gfloat) MAX_GRID_RES, 0, 0, 0, 70);
	if (gkrellm_get_chartconfig_grid_resolution(cf) < MIN_GRID_RES)
		gkrellm_set_chartconfig_grid_resolution(cf, DEFAULT_GRID_RES);

	/* Don't want to waste an hour priming the pump, and don't need to
	|  because data always starts at zero.
	*/
	cp->primed = TRUE;		/* XXX */
	gkrellm_alloc_chartdata(cp);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (cp->drawing_area), "expose_event",
				(GtkSignalFunc) inet_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),"button_press_event",
				(GtkSignalFunc) cb_inet_extra, NULL);
		}
	}

static void
create_inet_monitor(GtkWidget *vbox1, InetMon *in, gint first_create)
	{
	GtkWidget	*vbox;
	Chart		*cp;
	Panel		*p;
	Margin		*m;
	Style		*style;
	gint		x;

	if (first_create)
		{
		vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox1), vbox);
		in->vbox = vbox;
		in->chart_minute = gkrellm_chart_new0();
		in->chart_hour = gkrellm_chart_new0();
		in->panel = gkrellm_panel_new0();
		in->chart = in->chart_minute;
		in->name = g_strdup_printf(_("inet%d"), n_inet_monitors++);
		}
	else
		{
		vbox = in->vbox;
		gkrellm_destroy_decal_list(in->panel);
		gkrellm_destroy_krell_list(in->panel);
		}
	if (in->chart_config_hour && in->chart_config_minute)
		in->chart_config_hour->h = in->chart_config_minute->h;
	chart_create(in, in->chart_minute, &in->chart_config_minute, first_create);
	gkrellm_chartconfig_grid_resolution_label(in->chart_config_minute,
			_("TCP hits per minute"));
	chart_create(in, in->chart_hour, &in->chart_config_hour, first_create);
	gkrellm_chartconfig_grid_resolution_label(in->chart_config_hour,
			_("TCP hits per hour"));
	cp = in->chart;

	p = in->panel;
	style = gkrellm_panel_style(style_id);
	m = gkrellm_get_style_margins(style);
	if (style->label_position == 50 && gkrellm_chart_width() < 80)
		style->label_position = 40;		/* Not a kludge, an adjustment! */
	in->list_decal = gkrellm_create_decal_pixmap(p,
				gkrellm_decal_misc_pixmap(), gkrellm_decal_misc_mask(),
				N_MISC_DECALS, style, -1, -1);
	if (style->label_position <= 50)
		x = gkrellm_chart_width() - in->list_decal->w - m->right;
	else
		x = m->left;
	gkrellm_move_decal(p, in->list_decal, x, in->list_decal->y);

	gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

	/* Inet krells are not related to chart scale_max.  Just give a constant
	|  full scale of 5.
	*/
	KRELL(p)->full_scale = 5;
	gkrellm_panel_configure(p, in->name, style);
	gkrellm_panel_create(vbox, mon_inet, p);

	/* At first_create both charts will be visible, but this will be
	|  undone below
	*/
	in->list_button = gkrellm_make_decal_button(p, in->list_decal,
			cb_list_button, in, D_MISC_BUTTON_OUT, D_MISC_BUTTON_IN);

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),"expose_event",
				(GtkSignalFunc) inet_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area),
				"button_press_event", (GtkSignalFunc) cb_panel_press, NULL);

		gtk_widget_show(vbox);
		gkrellm_chart_hide(in->chart_hour, FALSE);
		}
	gkrellm_setup_launcher(p, &in->launch, CHART_PANEL_TYPE, 4);

	if (in->mark_data)
		g_free(in->mark_data);
	in->mark_data = g_new0(gshort, cp->w);

	if (! first_create)
		gkrellm_rescale_chart(in->chart);
	}

static void
create_inet(GtkWidget *vbox, gint first_create)
	{
	GList		*list;
	gint		new_data	= FALSE;
	gint		i;
	static gint	last_chart_width;

	inet_vbox = vbox;
	if (grid)
		{
		gdk_image_destroy(grid);
		grid = NULL;
		}
	n_inet_monitors = 0;
	if (!first_create && last_chart_width != gkrellm_chart_width())
		{  /* Will be allocating new data arrays */
		save_inet_data();
		new_data = TRUE;
		last_chart_width = gkrellm_chart_width();
		}
	for (i = 0, list = inet_mon_list; list; ++i, list = list->next)
		create_inet_monitor(inet_vbox, (InetMon *)list->data, first_create);
	if (first_create || new_data)
		load_inet_data();
	}

static InetMon   *
lookup_inet(gchar *name)
	{
	InetMon  *in;
	GList   *list;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (name && in->name && !strcmp(in->name, name))
			return in;
		}
	return NULL;
	}

/* --------------------------------------------------------------------- */
#define	INET_CONFIG_KEYWORD		"inet"

static void
save_inet_config(FILE *f)
	{
	GList	*list;
	InetMon	*in;
	gchar	buf[128];
	gchar	*l0, *l1;
	gint	i;

	for (i = 0, list = inet_mon_list; list; list = list->next, ++i)
		{
		in = (InetMon *) list->data;
		l0 = (*in->label0) ? in->label0: "NONE";
		l1 = (*in->label1) ? in->label1: "NONE";
		fprintf(f, "%s monitor %s %s %lu %lu %s %lu %lu %d %d %d\n",
				INET_CONFIG_KEYWORD, in->name,
				l0, in->port0_0, in->port0_1,
				l1, in->port1_0, in->port1_1,
				in->extra_info, in->data0_is_range, in->data1_is_range);
		snprintf(buf, sizeof(buf), "%s:minute", in->name);
		gkrellm_save_chartconfig(f, in->chart_config_minute,
				INET_CONFIG_KEYWORD, buf);
		snprintf(buf, sizeof(buf), "%s:hour", in->name);
		gkrellm_save_chartconfig(f, in->chart_config_hour,
				INET_CONFIG_KEYWORD, buf);
		fprintf(f, "%s launch %s %s\n", INET_CONFIG_KEYWORD,
					in->name, in->launch.command);
		fprintf(f, "%s tooltip %s %s\n", INET_CONFIG_KEYWORD,
					in->name, in->launch.tooltip_comment);
		}
	fprintf(f, "%s text_format all %s\n", INET_CONFIG_KEYWORD, text_format);
	}

static gint
fix_ports(InetMon *in)
	{
	gint	cd_length = 2;
	gulong	tmp;

	if (!*in->label0)
		{
		in->port0_1 = in->port0_1 = 0;
		--cd_length;
		}
	if (!*in->label1)
		{
		in->port1_1 = in->port1_1 = 0;
		--cd_length;
		}
	if (in->data0_is_range && (in->port0_1 < in->port0_0))
		{
		tmp = in->port0_1;
		in->port0_1 = in->port0_0;
		in->port0_0 = tmp;
		}
	if (in->data1_is_range && (in->port1_1 < in->port1_0))
		{
		tmp = in->port1_1;
		in->port1_1 = in->port1_0;
		in->port1_0 = tmp;
		}
	return cd_length;
	}

static void
load_inet_config(gchar *arg)
	{
	InetMon		*in;
	gchar		config[32], name[32];
	gchar		item[CFG_BUFSIZE];
	gchar		label0[16], label1[16];
	gchar		*hr_min;
	gint		n;

	if ((n = sscanf(arg, "%31s %31s %[^\n]", config, name, item)) != 3)
		return;
	hr_min = strrchr(name, (gint) ':');
	if (hr_min)
		*hr_min++ = '\0';
	if (!strcmp(config, "text_format"))
		{
		gkrellm_dup_string(&text_format, item);
		return;
		}
	else if (!strcmp(config, "monitor"))
		{
		in = g_new0(InetMon, 1);
		label0[0] = '\0';
		label1[0] = '\0';
		sscanf(item, "%15s %lu %lu %15s %lu %lu %d %d %d",
				label0, &in->port0_0, &in->port0_1,
				label1, &in->port1_0, &in->port1_1,
				&in->extra_info, &in->data0_is_range, &in->data1_is_range);
		if (!strcmp(label0, "NONE"))
			label0[0] = '\0';
		if (!strcmp(label1, "NONE"))
			label1[0] = '\0';
		in->label0 = g_strdup(label0);
		in->label1 = g_strdup(label1);
		in->cd_length = fix_ports(in);
		if (in->cd_length > 0)
			{
			in->name = g_strdup(name);
			in->chart_config_minute = gkrellm_chartconfig_new0();
			in->chart_config_hour = gkrellm_chartconfig_new0();
			inet_mon_list = g_list_append(inet_mon_list, in);
			}
		else	/* Bogus config line */
			{
			g_free(in->label0);
			g_free(in->label1);
			g_free(in);
			}
		return;
		}
	if ((in = lookup_inet(name)) == NULL)
		return;
	if (!strcmp(config, GKRELLM_CHARTCONFIG_KEYWORD))
		{
		if (hr_min && !strcmp(hr_min, "hour"))
			gkrellm_load_chartconfig(&in->chart_config_hour, item,
					in->cd_length);
		if (hr_min && !strcmp(hr_min, "minute"))
			gkrellm_load_chartconfig(&in->chart_config_minute, item,
					in->cd_length);
		}
	else if (!strcmp(config, "launch"))
		gkrellm_dup_string(&in->launch.command, item);
	else if (!strcmp(config, "tooltip"))
		gkrellm_dup_string(&in->launch.tooltip_comment, item);
	}


/* --------------------------------------------------------------------- */

  /* Read saved inet data (from a previous gkrellm process).  Return the
  |  number of missing data slots (skew).
  */
static gint
read_inet_data(Chart *cp, FILE *f, gint minute_chart,
			gint min, gint hour, gint yday, gint width)
	{
	struct tm	*tm;
	gchar		data[64];
	gint		n, in, out, cur_slot, skew, day;

	tm = gkrellm_get_current_time();
	day = tm->tm_yday - yday;

	/* Check for new years wrap around. I don't handle leap year here, will
	|  get some email, then be safe for four more years...
	*/
	if (day < 0)
		day = tm->tm_yday + ((yday < 365) ? 365 - yday : 0);

	cur_slot = day * 24 + tm->tm_hour;
	n = hour;
	if (minute_chart)
		{
		cur_slot = cur_slot * 60 + tm->tm_min;
		n = n * 60 + min;
		}
	skew = cur_slot - n;

	gkrellm_reset_chart(cp);
	for (n = 0; n < width; ++n)
		{
		if (fgets(data, sizeof(data), f) == NULL)
			break;

		if (skew >= cp->w)	/* All stored data is off the chart	*/
			continue;

		/* Use chart data storing routines to load in data so I don't care
		|  if current chart width is less or greater than stored data width.
		|  Charts will circular buff fill until data runs out.
		*/
		out = in = 0;
		sscanf(data, "%d %d", &out, &in);
		gkrellm_store_chartdata(cp, 0, out, in);
		}
	/* Need to store zero data for time slots not in read data to bring
	|  the chart up to date wrt current time.  As in update_inet() I need
	|  to skip slots for hour or minute ticks.
	|  Warning: skew can be negative if quit gkrellm, change system clock
	|  to earlier time, then restart gkrellm.
	*/
	if ((n = skew) < cp->w)		/* Do this only if some data was stored  */
		{
		while (n-- > 0)
			{
			gkrellm_store_chartdata(cp, 0, 0, 0);
			if (minute_chart && min++ == 0)
				{
				gkrellm_store_chartdata(cp, 0, 0, 0);
				gkrellm_store_chartdata(cp, 0, 0, 0);
				if (min == 60)
					min = 0;
				}
			else if (!minute_chart && hour++ == 0)
				{
				gkrellm_store_chartdata(cp, 0, 0, 0);
				gkrellm_store_chartdata(cp, 0, 0, 0);
				if (hour == 24)
					hour = 0;
				}
			}
		}
	return skew;
	}

static void
write_inet_data(Chart *cp, FILE *f)
	{
	GList		*list;
	gint		n;

	for (n = 0; n < cp->w; ++n)
		{
		for (list = cp->cd_list; list; list = list->next)
			fprintf(f, "%d ",
					gkrellm_get_chartdata_data((ChartData *) list->data, n));
		fprintf(f, "\n");
		}
	}

static gchar *
make_inet_data_fname(InetMon *in)
	{
	static gchar	idata_fname[256];
	gchar			c_sep0, c_sep1;

	c_sep0 = in->data0_is_range ? '-': '_';
	c_sep1 = in->data1_is_range ? '-': '_';
	snprintf(idata_fname, sizeof(idata_fname), "%s/%s/inet_%ld%c%ld_%ld%c%ld",
		gkrellm_homedir(), inet_data_dir,
		in->port0_0, c_sep0, in->port0_1, in->port1_0, c_sep1, in->port1_1);
	return idata_fname;
	}

void
save_inet_data(void)
	{
	FILE		*f;
	struct tm	*tm;
	GList		*list;
	InetMon		*in;
	gchar		*fname, buf[64];

	tm = gkrellm_get_current_time();
	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		fname = make_inet_data_fname(in);
		if ((f = fopen(fname, "w")) == NULL)
			continue;

		fputs("minute hour yday width\n", f);
		snprintf(buf, sizeof(buf), "%d %d %d %d\n", tm->tm_min,
				tm->tm_hour, tm->tm_yday, in->chart->w);
		fputs(buf, f);

		/* Save any accumulated hits which have not been stored into the
		|  chart data array, and then save the chart data.
		*/
		fputs("hits0_minute hits1_minute hits0_hour hits1_hour\n", f);
		fprintf(f, "%ld %ld %ld %ld\n",
			in->hits0_minute, in->hits1_minute, in->hits0_hour, in->hits1_hour);
		write_inet_data(in->chart_minute, f);
		write_inet_data(in->chart_hour, f);
		fclose(f);
		}
	}

void
load_inet_data(void)
	{
	FILE		*f;
	GList		*list;
	InetMon		*in;
	gchar		buf[96], *fname;
	gint		min, hour, yday, len, skew;

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		fname = make_inet_data_fname(in);
		if ((f = fopen(fname, "r")) == NULL)
			{
			gkrellm_reset_chart(in->chart);
			draw_inet_chart(in);
			continue;
			}
		fgets(buf, sizeof(buf), f);		/* Comment line */
		fgets(buf, sizeof(buf), f);
		sscanf(buf, "%d %d %d %d", &min, &hour, &yday, &len);
		fgets(buf, sizeof(buf), f);		/* Comment line */
		fgets(buf, sizeof(buf), f);
		sscanf(buf, "%ld %ld %ld %ld",
					&in->hits0_minute, &in->hits1_minute,
					&in->hits0_hour, &in->hits1_hour);

		skew = read_inet_data(in->chart_minute, f, 1, min, hour, yday, len);
		if (skew > 0)  /* Current minute slot is different from saved */
			in->hits0_minute = in->hits1_minute = 0;

		skew = read_inet_data(in->chart_hour, f, 0, min, hour, yday, len);
		if (skew > 0)  /* Current hour slot is different from saved */
			in->hits0_hour = in->hits1_hour = 0;

		fclose(f);
		gkrellm_rescale_chart(in->chart);
		}
	}

/* --------------------------------------------------------------------- */
#define	DEFAULT_TEXT_FORMAT	"\\t$a\\f $l\\N$A\\f $L"

static GtkWidget	*inet_clist;
static gint			selected_row,
					inet_list_modified;


static GtkWidget	*label0_entry,
					*label1_entry;
static GtkWidget	*port0_0_entry,
					*port0_1_entry,
					*port1_0_entry,
					*port1_1_entry;

static GtkWidget	*launch_vbox;

static GtkWidget	*data0_range_button,
					*data1_range_button;

static GtkWidget	*text_format_combo;

static void
reset_inet_entries(void)
	{
	gtk_entry_set_text(GTK_ENTRY(label0_entry), "");
	gtk_entry_set_text(GTK_ENTRY(port0_0_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(port0_1_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(label1_entry), "");
	gtk_entry_set_text(GTK_ENTRY(port1_0_entry), "0");
	gtk_entry_set_text(GTK_ENTRY(port1_1_entry), "0");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data0_range_button), 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data1_range_button), 0);
	}

static void
cb_inet_clist_selected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	gchar	*s;

	gtk_clist_get_text(GTK_CLIST(clist), row, 0, &s);
	gtk_entry_set_text(GTK_ENTRY(label0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 1, &s);
	gtk_entry_set_text(GTK_ENTRY(port0_0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 2, &s);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data0_range_button),
			!strcmp(s, "-") ? TRUE : FALSE);
	gtk_clist_get_text(GTK_CLIST(clist), row, 3, &s);
	gtk_entry_set_text(GTK_ENTRY(port0_1_entry), s);

	gtk_clist_get_text(GTK_CLIST(clist), row, 5, &s);
	gtk_entry_set_text(GTK_ENTRY(label1_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 6, &s);
	gtk_entry_set_text(GTK_ENTRY(port1_0_entry), s);
	gtk_clist_get_text(GTK_CLIST(clist), row, 7, &s);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(data1_range_button),
			!strcmp(s, "-") ? TRUE : FALSE);
	gtk_clist_get_text(GTK_CLIST(clist), row, 8, &s);
	gtk_entry_set_text(GTK_ENTRY(port1_1_entry), s);

	selected_row = row;
	}

static void
cb_inet_clist_unselected(GtkWidget *clist, gint row, gint column,
		GdkEventButton *bevent, gpointer data)
	{
	selected_row = -1;
	reset_inet_entries();
	}

static void
cb_enter_inet(GtkWidget *widget, gpointer data)
	{
	gchar	*buf[10];
	gint	data0_is_range, data1_is_range;

	buf[1] = buf[3] = buf[6] = buf[8] = "0";
	buf[2] = buf[7] = ",";
	data0_is_range = GTK_TOGGLE_BUTTON(data0_range_button)->active;
	data1_is_range = GTK_TOGGLE_BUTTON(data1_range_button)->active;

	buf[0] = gkrellm_entry_get_text(&label0_entry);
	if (*buf[0])
		{
		buf[1] = gkrellm_entry_get_text(&port0_0_entry);
		if (data0_is_range)
			buf[2] = "-";
		buf[3] = gkrellm_entry_get_text(&port0_1_entry);
		}
	buf[4] = "";
	buf[5] = gkrellm_entry_get_text(&label1_entry);
	if (*buf[5])
		{
		buf[6] = gkrellm_entry_get_text(&port1_0_entry);
		if (data1_is_range)
			buf[7] = "-";
		buf[8] = gkrellm_entry_get_text(&port1_1_entry);
		}
	buf[9] = NULL;

	/* Validate the values
	*/
	if (   (*buf[0] == '\0' && *buf[5] == '\0')
		|| ((*buf[0] != '\0' && atoi(buf[1]) == 0 && atoi(buf[3]) == 0))
		|| ((*buf[5] != '\0' && atoi(buf[6]) == 0 && atoi(buf[8]) == 0))
	   )
		return;

	if (selected_row >= 0)
		{
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 0, buf[0]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 1, buf[1]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 2, buf[2]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 3, buf[3]);

		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 5, buf[5]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 6, buf[6]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 7, buf[7]);
		gtk_clist_set_text(GTK_CLIST(inet_clist), selected_row, 8, buf[8]);

		gtk_clist_unselect_row(GTK_CLIST(inet_clist), selected_row, 0);
		selected_row = -1;
		}
	else
		gtk_clist_append(GTK_CLIST(inet_clist), buf);
	reset_inet_entries();
	inet_list_modified = TRUE;
	}

static void
cb_delete_inet(GtkWidget *widget, gpointer data)
	{
	reset_inet_entries();
	if (selected_row >= 0)
		{
		gtk_clist_remove(GTK_CLIST(inet_clist), selected_row);
		inet_list_modified = TRUE;
		selected_row = -1;
		}
	}

static void
add_launch_entry(GtkWidget *vbox, InetMon *in)
	{
	GtkWidget	*table;

	table = gkrellm_launcher_table_new(vbox, 1);
	gkrellm_config_launcher(table, 0,  &in->launch_entry, &in->tooltip_entry,
				in->name, &in->launch);
	in->launch_table = table;
	gtk_widget_show_all(table);
	}

static void
apply_inet_config(void)
	{
	InetMon		*in, *in_tmp;
	GList		*list, *new_inet_list;
	gchar		*s;
	gint		row;
	gboolean	new_text_format;

	s = gkrellm_entry_get_text(&(GTK_COMBO(text_format_combo)->entry));
	new_text_format = gkrellm_dup_string(&text_format, s);

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		if (in->launch_entry)
			gkrellm_apply_launcher(&in->launch_entry, &in->tooltip_entry,
					in->panel, &in->launch, gkrellm_launch_button_cb);
		}
	if (!inet_list_modified)
		{
		if (new_text_format)
			for (list = inet_mon_list; list; list = list->next)
				draw_inet_chart((InetMon *) list->data);
		return;
		}

	/* Just save all data and then later read it back in.  This avoids
	|  complicated detecting of name changes while ports the same, moving
	|  a inet down or up slots, etc.  Data is lost only if a port number
	|  for a monitor is changed.
	*/
	save_inet_data();
	new_inet_list = NULL;
	n_inet_monitors = 0;
	for (row = 0; row < GTK_CLIST(inet_clist)->rows; ++row)
		{
		in = g_new0(InetMon, 1);
		new_inet_list = g_list_append(new_inet_list, in);
		in_tmp = (InetMon *) gtk_clist_get_row_data(GTK_CLIST(inet_clist),row);
		gtk_clist_set_row_data(GTK_CLIST(inet_clist), row, in);

		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 0, &s);
		in->label0 = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 1, &s);
		in->port0_0 = atoi(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 2, &s);
		in->data0_is_range = (!strcmp(s, "-") ? TRUE : FALSE);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 3, &s);
		in->port0_1 = atoi(s);

		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 5, &s);
		in->label1 = g_strdup(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 6, &s);
		in->port1_0 = atoi(s);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 7, &s);
		in->data1_is_range = (!strcmp(s, "-") ? TRUE : FALSE);
		gtk_clist_get_text(GTK_CLIST(inet_clist), row, 8, &s);
		in->port1_1 = atoi(s);
		fix_ports(in);

		/* If an existing inet still has the same port numbers, preserve
		|  its config.  Otherwise, it is same as making a new entry.
		|  (plus the data layers could go from 2 to 1 and then there would
		|   be an extra data layer in the config - not good).
		*/
		if (   in_tmp
			&& in_tmp->port0_0 == in->port0_0
			&& in_tmp->port0_1 == in->port0_1
			&& in_tmp->port1_0 == in->port1_0
			&& in_tmp->port1_1 == in->port1_1
		   )
			{
			in->chart_config_minute = in_tmp->chart_config_minute;
			in_tmp->chart_config_minute = NULL;
			in->chart_config_hour = in_tmp->chart_config_hour;
			in_tmp->chart_config_hour = NULL;
			in->extra_info = in_tmp->extra_info;
			in->hour_mode = in_tmp->hour_mode;
			gkrellm_dup_string(&in->launch.command, in_tmp->launch.command);
			gkrellm_dup_string(&in->launch.tooltip_comment,
						in_tmp->launch.tooltip_comment);
			}
		else
			{
			in->chart_config_minute = gkrellm_chartconfig_new0();
			in->chart_config_hour = gkrellm_chartconfig_new0();
			gkrellm_set_chartconfig_auto_grid_resolution(
					in->chart_config_minute, TRUE);
			gkrellm_set_chartconfig_auto_grid_resolution(
					in->chart_config_hour, TRUE);
			in->extra_info = TRUE;
			}
		}
	while (inet_mon_list)
		{
		in = (InetMon *) inet_mon_list->data;
		if (in->launch_table)
			gtk_widget_destroy(in->launch_table);
		destroy_inet_monitor(in);
		inet_mon_list = g_list_remove(inet_mon_list, in);
		}
	inet_mon_list = new_inet_list;
	for (list = inet_mon_list; list; list = list->next)
		create_inet_monitor(inet_vbox, (InetMon *)list->data, TRUE);

	gkrellm_pack_side_frames();
	load_inet_data();

	for (list = inet_mon_list; list; list = list->next)
		{
		in = (InetMon *) list->data;
		draw_inet_chart(in);
		add_launch_entry(launch_vbox, in);
		}
	inet_list_modified = FALSE;
	}

static gchar	*inet_info_text[] =
{
N_("Inet charts show historical TCP port hits on a minute or hourly\n"
	"chart. Below the chart there is a strip where marks are drawn for\n"
	"port hits in second intervals.   The inet krell has a full scale\n"
	"value of 5 hits and samples once per second.  The extra info\n"
	"display shows current TCP port connections.\n\n"
	"For each internet monitor, you can specify two labeled data sets with\n"
	"one or two non-zero port numbers entered for each data set.  Two\n"
	"ports are allowed because some internet ports are related and you\n"
	"might want to group them.  Check /etc/services for port numbers.\n\n"
	"For example, if you created an inet monitor:\n"),

N_("<i>\thttp 80 8080   ftp 21\n"),

N_("Http hits on the standard http port 80 and www web caching service\n"
	"on port 8080 are combined and plotted in the one color.  Ftp hits\n"
	"on the single ftp port 21 are plotted in another color.\n\n"),

N_("If the range button is checked, then all port numbers between Port0 and\n"
   "Port1 are monitored and included in the plot.\n\n"),

N_("<b>Chart Labels\n"),
N_("Substitution variables for the format string for chart labels:\n"),
N_("\t$M    maximum chart value\n"),
N_("\t$a    current active connections for Data0\n"),
N_("\t$l    label for Data0\n"),
N_("\t$A    current active connections for Data1\n"),
N_("\t$L    label for Data1\n"),

"\n",
N_("<b>Mouse Button Actions:\n"),
N_("<b>\tLeft "),
N_("click on an inet chart to toggle the extra info display of\n"
	"\t\tcurrent TCP port connections.\n"),
N_("<b>\tMiddle "),
N_("click on an inet chart to toggle hour/minute charts.\n")
};


static gchar	*inet_titles[10] =
	{
	N_("Label"), N_("Port0"), "", N_("Port1"), "",
	N_("Label"), N_("Port0"), "", N_("Port1"), ""
	};

static void
create_inet_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*tabs;
	GtkWidget		*table;
	GtkWidget		*hbox, *vbox, *vbox1;
	GtkWidget		*separator;
	GtkWidget		*scrolled;
	GtkWidget		*text;
	GtkWidget		*label;
	GtkWidget		*button;
	GList			*list;
	InetMon			*in;
	gchar			*buf[10];
	gchar			p00[16], p01[16], p10[16], p11[16];
	gint			i, w_pentry, w_range;
	
	inet_list_modified = FALSE;
	selected_row = -1;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

	vbox = gkrellm_create_framed_tab(tabs, _("Ports"));

	table = gtk_table_new(6, 7, FALSE /*homogeneous*/);
	gtk_table_set_col_spacings(GTK_TABLE(table), 5);
	gtk_box_pack_start(GTK_BOX(vbox), table, FALSE, FALSE, 3);

	label = gtk_label_new(_("Data0"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 3, 0, 1);
	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 0, 3, 1, 2);
	label = gtk_label_new(_("Data1"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 7, 0, 1);
	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 4, 7, 1, 2);

	separator = gtk_vseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 3, 4, 0, 6);

	label = gtk_label_new(_("Label"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 0, 1, 2, 3);
	label = gtk_label_new(_("Port0"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 1, 2, 2, 3);
	label = gtk_label_new(_("Port1"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 2, 3, 2, 3);
	label = gtk_label_new(_("Label"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 4, 5, 2, 3);
	label = gtk_label_new(_("Port0"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 5, 6, 2, 3);
	label = gtk_label_new(_("Port1"));
	gtk_table_attach_defaults(GTK_TABLE(table), label, 6, 7, 2, 3);

	label0_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(label0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), label0_entry, 0, 1, 3, 4);
	port0_0_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(port0_0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port0_0_entry, 1, 2, 3, 4);
	port0_1_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(port0_1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port0_1_entry, 2, 3, 3, 4);

	label1_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(label1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), label1_entry, 4, 5, 3, 4);
	port1_0_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(port1_0_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port1_0_entry, 5, 6, 3, 4);
	port1_1_entry = gtk_entry_new_with_max_length(8);
	gtk_widget_set_usize(port1_1_entry, 32, 0);
	gtk_table_attach_defaults(GTK_TABLE(table), port1_1_entry, 6, 7, 3, 4);

	hbox = gtk_hbox_new(FALSE, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 0, 3, 4, 5);
	gkrellm_check_button(hbox, &data0_range_button, 0, TRUE, 0,
		_("Port0 - Port1 is a range"));

	hbox = gtk_hbox_new(FALSE, 2);
	gtk_table_attach_defaults(GTK_TABLE(table), hbox, 4, 7, 4, 5);
	gkrellm_check_button(hbox, &data1_range_button, 0, TRUE, 0,
		_("Port0 - Port1 is a range"));

	separator = gtk_hseparator_new();
	gtk_table_attach_defaults(GTK_TABLE(table), separator, 0, 7, 5, 6);

	hbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(hbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(hbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 0);

	button = gtk_button_new_with_label(_("Enter"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_enter_inet, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

	button = gtk_button_new_with_label(_("Delete"));
	gtk_signal_connect(GTK_OBJECT(button), "clicked",
			(GtkSignalFunc) cb_delete_inet, NULL);
	gtk_box_pack_start(GTK_BOX(hbox), button, TRUE, TRUE, 0);

	separator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(vbox), separator, FALSE, FALSE, 2);

	scrolled = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled),
			GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_box_pack_start(GTK_BOX(vbox), scrolled, TRUE, TRUE, 0);
	for(i = 0; i < 10; i++)
		{
		if (inet_titles[i][0] != '\0') 
			inet_titles[i] = _(inet_titles[i]);
		}
	inet_clist = gtk_clist_new_with_titles(10, inet_titles);
	gtk_clist_set_shadow_type (GTK_CLIST(inet_clist), GTK_SHADOW_OUT);
	gtk_clist_column_titles_passive(GTK_CLIST(inet_clist));
	gtk_clist_set_column_justification(GTK_CLIST(inet_clist),
			1, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(inet_clist),
			3, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(inet_clist),
			6, GTK_JUSTIFY_RIGHT);
	gtk_clist_set_column_justification(GTK_CLIST(inet_clist),
			8, GTK_JUSTIFY_RIGHT);

	w_pentry = gdk_string_width(label0_entry->style->font, "88888") + 4;
	w_range = gdk_string_width(label0_entry->style->font, "-") + 2;
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 1, w_pentry);
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 2, w_range);
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 3, w_pentry);
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 4, 40);
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 6, w_pentry);
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 7, w_range);
	gtk_clist_set_column_width(GTK_CLIST(inet_clist), 8, w_pentry);

	gtk_signal_connect (GTK_OBJECT(inet_clist), "select_row",
			(GtkSignalFunc) cb_inet_clist_selected, NULL);
	gtk_signal_connect (GTK_OBJECT(inet_clist), "unselect_row",
			(GtkSignalFunc) cb_inet_clist_unselected, NULL);
	gtk_container_add (GTK_CONTAINER (scrolled), inet_clist);
	for (i = 0, list = inet_mon_list; list; ++i, list = list->next)
		{
		in = (InetMon *) list->data;
		snprintf(p00, sizeof(p00), "%d", (int) in->port0_0);
		snprintf(p01, sizeof(p01), "%d", (int) in->port0_1);
		snprintf(p10, sizeof(p10), "%d", (int) in->port1_0);
		snprintf(p11, sizeof(p11), "%d", (int) in->port1_1);
		buf[0] = in->label0;
		buf[1] = p00;
		buf[2] = in->data0_is_range ? "-" : ",";
		buf[3] = p01;
		buf[4] = "";
		buf[5] = in->label1;
		buf[6] = p10;
		buf[7] = in->data1_is_range ? "-" : ",";
		buf[8] = p11;
		buf[9] = NULL;
		gtk_clist_append(GTK_CLIST(inet_clist), buf);
		gtk_clist_set_row_data(GTK_CLIST(inet_clist), i, in);
		}

/* --Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Format String for Chart Labels"),
				4, FALSE, 0, 2);
	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox1), hbox, FALSE, TRUE, 5);
	text_format_combo = gtk_combo_new();
	gtk_widget_set_usize (GTK_WIDGET(text_format_combo), 350, 0);
	gtk_box_pack_start(GTK_BOX(hbox), text_format_combo, FALSE, TRUE, 0);
	list = NULL;
	list = g_list_append(list, DEFAULT_TEXT_FORMAT);
	list = g_list_append(list,
				"\\C\\f $M\\t$a\\f $l\\N$A\\f $L");
	list = g_list_append(list,
				"\\C\\f $M\\D1$a\\f $l\\D2$A\\f $L");
	gtk_combo_set_popdown_strings(GTK_COMBO(text_format_combo), list);
	gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(text_format_combo)->entry),
			text_format);

	vbox1 = gkrellm_framed_vbox_end(vbox, _("Launch Commands"),
				4, TRUE, 0, 2);
	launch_vbox = gkrellm_scrolled_vbox(vbox1, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

	for (i = 0, list = inet_mon_list; list; list = list->next, ++i)
		{
		in = (InetMon *) list->data;
		add_launch_entry(launch_vbox, in);
		}

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(inet_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(inet_info_text[i]));

	reset_inet_entries();
	}



static Monitor	monitor_inet =
	{
	N_("Internet"),			/* Name, for config tab.	*/
	MON_INET,			/* Id,  0 if a plugin		*/
	create_inet,		/* The create function		*/
	update_inet,		/* The update function		*/
	create_inet_tab,	/* The config tab create function	*/
	apply_inet_config,	/* Apply the config function		*/

	save_inet_config,	/* Save user conifg			*/
	load_inet_config,	/* Load user config			*/
	"inet",				/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_inet_monitor(void)
	{
	monitor_inet.name = _(monitor_inet.name);
	style_id = gkrellm_add_chart_style(&monitor_inet, INET_STYLE_NAME);
	text_format = g_strdup(DEFAULT_TEXT_FORMAT);
	mon_inet = &monitor_inet;
	inet_data_dir = g_strdup_printf("%s/inet", GKRELLM_DATA_DIR);
	if (setup_inet_interface())
		{
		make_home_subdir(inet_data_dir);
		return &monitor_inet;
		}
	return FALSE;
	}
