/*
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
| 3/20/2002  Patch from Ben Low <ben-apm@bdlow.net> adds battery time
|              estimation feature.
| 2/4/2001   Patch from joshua stein <jcs@rt.fm> so apm will add itself
|              under Net/Open BSD
| 10/12/2000 NetBSD code contributed by Anthony Mallet
|              <anthony.mallet@useless-ficus.net>
| 3/6/2000   Patch from Kazuhisa TAKEI <takei@vinelinux.org> added a toggle
|              to show percentage time left on apm panel.
| 2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <math.h>


typedef enum
	{
	BATTERYDISPLAY_PERCENT,
	BATTERYDISPLAY_TIME,
	BATTERYDISPLAY_RATE,
	BATTERYDISPLAY_EOM	/* end of modes */
	}
	BatteryDisplayMode; 


struct
	{
	BatteryDisplayMode battery_display_mode;
	gfloat		battery_charge_rate;	/* % / min */

	/* (*(read_apm_data))() must fill in these values */
	gboolean	battery_is_available,
				ac_is_on_line,
				battery_is_charging;
	gint		battery_percentage;
	gint		battery_time_left;		/* In minutes, -1 if minutes unavail */
	}
	sys_apm;

void    (*read_apm_data)();


/* ====== System dependent interface ====================================== */


#define	L_NO_BATTERY	0x80
#define	L_ON_LINE		1
#define	L_CHARGING		3
#define L_UNKNOWN		0xFF

/* ------- FreeBSD ------------------------------------------------------- */

#if defined(__FreeBSD__) && defined(__i386__)
#include <osreldate.h>
#include <machine/apm_bios.h>
#define	APMDEV		"/dev/apm"

static void
read_freebsd_apm_data()
	{
	int		f, r;
	struct apm_info	info;

	if ((f = open(APMDEV, O_RDONLY)) == -1)
		return;
	r = ioctl(f, APMIO_GETINFO, &info);
	close(f);
	if (r == -1)
		return;
	sys_apm.battery_is_available = (info.ai_batt_stat != L_UNKNOWN);
	sys_apm.ac_is_on_line = (info.ai_acline == L_ON_LINE) ? TRUE : FALSE;
	sys_apm.battery_is_charging = (info.ai_batt_stat == L_CHARGING) ? TRUE : FALSE;
	sys_apm.battery_percentage = info.ai_batt_life;
#if defined(APM_GETCAPABILITIES)
	sys_apm.battery_time_left = info.ai_batt_time / 60;
#else
	sys_apm.battery_time_left = -1;
#endif
	}
#endif


/* ------- Linux ------------------------------------------------------- */
/* ----- see /usr/src/linux/arch/i386/kernel/apm.c ----- */
#if defined(__linux__)

#define	PROC_APM_FILE	"/proc/apm"

static void
read_linux_apm_data(void)
	{
	FILE	*f;
	gchar	buf[128];
	gint	ac_line_status,
			battery_status,
			flag,
			percentage,
			time;
	gchar	units[32];

	if ((f = fopen(PROC_APM_FILE, "r")) == NULL)
		return;
	fgets(buf, sizeof(buf), f);
	fclose(f);

	sscanf(buf, "%*s %*d.%*d %*x %x %x %x %d%% %d %s\n", &ac_line_status,
			&battery_status, &flag, &percentage, &time, units);

	if ((flag & L_NO_BATTERY) == 0 && battery_status != L_UNKNOWN)
		sys_apm.battery_is_available = TRUE;
	else
		sys_apm.battery_is_available = FALSE;

	sys_apm.ac_is_on_line = (ac_line_status == L_ON_LINE) ? TRUE : FALSE;
	sys_apm.battery_is_charging= (battery_status == L_CHARGING) ? TRUE : FALSE;
	sys_apm.battery_percentage = percentage;
	sys_apm.battery_time_left = time;
	if (!strcmp(units, "sec"))
		sys_apm.battery_time_left /= 60;
	}
#endif	/* __linux__ */


/* ------- NetBSD ------------------------------------------------------- */

#if (defined(__NetBSD__) || defined(__OpenBSD__)) && defined(__i386__)

#include <machine/apmvar.h>
#define	APMDEV		"/dev/apm"

void
read_netbsd_apm_data()
	{
	int f, r;
	struct apm_power_info info;

	if ((f = open(APMDEV, O_RDONLY)) == -1) return;
	memset(&info, 0, sizeof(info));
	r = ioctl(f, APM_IOC_GETPOWER, &info);
	close(f);
	if (r == -1) return;

	sys_apm.battery_is_available = (info.battery_state != APM_BATT_UNKNOWN);
	sys_apm.ac_is_on_line = (info.ac_state == APM_AC_ON) ? TRUE : FALSE;
	sys_apm.battery_is_charging = 
		(info.battery_state == APM_BATT_CHARGING) ? TRUE : FALSE;
	sys_apm.battery_percentage = info.battery_life;
	sys_apm.battery_time_left = info.minutes_left;
	}

#endif /* __NetBSD__ || __OpenBSD__ */


/* ------- Demo ------------------------------------------------------- */
  /* Themers need to be able to see the apm monitor.
  */
static void
read_apm_demo(void)
	{
	static gint	bump = 60;

	sys_apm.battery_is_available = TRUE;

	if (!GK.five_second_tick)
		return;

	if (bump <= 5)
		bump = 60;
	bump -= 5;
	sys_apm.ac_is_on_line = bump > 45;
	if (sys_apm.ac_is_on_line)
		{
		sys_apm.battery_is_charging = TRUE;
		sys_apm.battery_time_left = 200 + (60 - bump) * 20;
		sys_apm.battery_percentage = sys_apm.battery_time_left / 5;
		}
	else
		{
		sys_apm.battery_is_charging = FALSE;
		sys_apm.battery_time_left = bump;
		sys_apm.battery_percentage = 1 + bump;
		}
	}


/* ----- Pick a system interface ----------------------------------------- */

static gint
setup_apm_interface(void)
	{
	gint	available	= FALSE;

	if (GK.demo)
		{
		read_apm_data = read_apm_demo;
		return TRUE;
		}

#if defined(__FreeBSD__) && defined(__i386__)
	read_apm_data = read_freebsd_apm_data;
	available = TRUE;
#endif

#if defined(__linux__)
	read_apm_data = read_linux_apm_data;
	available = TRUE;
#endif

#if (defined(__NetBSD__) || defined(__OpenBSD__)) && defined(__i386__)
	read_apm_data = read_netbsd_apm_data;
	available = TRUE;
#endif

	return available;
	}



/* ======================================================================== */
#define	APM_CONFIG_KEYWORD	"apm"

Monitor			*mon_apm;

static gboolean	enable_apm, 
				enable_estimate_time;

static Panel	*apm;

static Krell	*apm_krell;
static Decal	*power_source_decal;
static Decal	*minutes_left_decal;

static Alert	*apm_alert;

static Launcher	launch;

  /* user-provided nominal battery runtimes, hrs (used to determine initial 
  |  discharge, stable, charge rate (%/min))
  */
static gfloat	estimate_time_runtime[2] = {0};
static gint		estimate_time_model[2] = {0};

static gint		style_id;

static gboolean	alert_units_are_percent,
				alert_units_need_estimate_mode;

static void
estimate_battery_time_left(void)
/* estimate (guess-timate?) battery time remaining, based on the rate of 
   discharge (and conversely the time to charge based on the rate of charge).
  - some BIOS' only provide battery levels, not any estimate of the time
    remaining
    
  Battery charge/discharge characteristics (or, why dc/dt doesn't really work)
  - the charge/discharge curves of most battery types tend to be very non-
    linear (http://www.google.com/search?q=battery+charge+discharge+curve)
  - on discharge, most battery types will initially fall somewhat rapidly
    from 100 percent, then flatten out and stay somewhat linear until
    suddenly "dropping out" when nearly depleted (approx. 10-20% capacity).
    For practical purposes we can consider this point to be the end of the
    discharge curve. This is simple enough to model via a fixed capacity
    offset to cut out just at the knee of this curve, and allows us to
    reasonably approximate the rest of the curve by a linear function
    and simple dc/dt calculation.
  - with regard to charging, however, it's not quite so easy. With a
    constant voltage charger, the battery capacity rises exponentially
    (charging current decreases as battery terminal voltage rises). The
    final stages of charging are very gradual, with a relatively long
    period at "almost but not quite 100%".

    Unfortunately a linear extrapolation at the beginning of an 
    exponential curve will be a poor approximation to the true expected
    time to charge, tending to be significantly undervalued. Using an 
    exponential model to estimate time to approx. 90-95% (2.5 * exp. time
    constant) seems to give a more reasonable fit. That said, the poor
    relative resolution at higher charge values makes estimating the
    exponential time constant difficult towards the end of the charge 
    cycle (the curve's very nearly flat). So, I've settled on a mixed 
    model - for c < ~70 I use an exponential model, and switch to linear
    above that (or if the charge rate seems to have otherwise "flatlined").

    Empirically, this method seems to give reasonable results [1] - 
    certainly  much better than seeing "0:50:00 to full" for a good half an
    hour (i.e. as happens with apmd, which uses a linear model for both
    charging + discharging). Note that a constant-current charger should
    be pretty well linear all the way along the charge curve, which means
    the linear rate extrapolation should work well in this case. The user
    can choose which model they wish to use via estimate_time_model.

    [1] I logged my Compaq Armada M300's capacity (via /proc/apm) over one
    complete discharge/charge cycle (machine was idle the whole time). The
    discharge curve was linear to approx. 14% when the BIOS alerted of 
    impending doom; upon plugging in the external power supply the capacity
    rose exponentially to 100%, with a time constant of approx. 0.8 hr (i.e. 
    approx. 2+ hrs to full charge).

  Linear rate of change calculation:
  - in an ideal, continuous world, estimated time to 0(100) would simply 
    be the remaining capacity divided by the charge rate
       ttl = c / dc(t)/dt
  - alas, the reported battery capacity is bound to integer values thus 
    c(t) is a discontinuous function. i.e. has fairly large steps. And of
    course then dc/dt is undefined at the discontinuities.
  - to avoid this issue the rate of charge is determined by the deltas from
    the start of the last state change (charge/discharge cycle) (time T)
       ttl(t) = c(t) / ((C - c(t)) / (T - t))    C = charge at time T
    Furthermore, the rate changes are windowed and filtered to mitigate 
    c(t) transients (e.g. at the start of discharge) and smooth over 
    discontinuities (and fudge for battery characteristics, ref. above).

 
*/

#define APM_SLEEP_DETECT 300		/* interval of >300s => must have slept */
#define APM_DISCHARGE_TRANSIENT 10	/* ignore first 10% of discharge cycle */
#define APM_EMPTY_CAPACITY 12		/* when is the battery "flat"? */
#define APM_RATECHANGE_WINDOW 90	/* allow rate changes after 90s */
#define APM_CHARGE_MODEL_LIMIT 60	/* linear charge model cutover point */
#define APM_RATE_SMOOTHING 0.3		/* rate smoothing weight */

/* #define APM_ESTIMATE_DEBUG */

	{
	/* ?0 = at time 0 (state change); ?1 = at last "sample" (rate change) */
	static time_t t0 = -1, t1;
	static gint p0, p1;
	static gint c0;
	static gfloat rate = 0;
	static time_t dt;
	static gint dp;

	time_t t = time(NULL);

	/* 1 charging; 0 power on and stable; -1 discharging */
	gint charging = sys_apm.battery_is_charging ? 1 : 
		(sys_apm.ac_is_on_line ? 0 : -1);

#		ifdef APM_ESTIMATE_DEBUG
		fprintf(stderr, "%ld bc?=%d ac?=%d (%+d) bp=%d\t", t, 
			sys_apm.battery_is_charging, sys_apm.ac_is_on_line, charging, 
			sys_apm.battery_percentage);
#		endif

	if (t0 < 0 || c0 != charging || (t - t1) > APM_SLEEP_DETECT)
		/* init, state change, or sleep/hibernation */
		{
		c0 = charging;

		t0 = t1 = t;
		if (charging < 0 && 
				(sys_apm.battery_percentage > 100 - APM_DISCHARGE_TRANSIENT))
			p0 = p1 = 100 - APM_DISCHARGE_TRANSIENT;
		else
			p0 = p1 = sys_apm.battery_percentage;
		dp = dt = 0;
		rate = 0.0;
		/* convert runtime (hrs) to signed rate (%/min) */
		if		(charging < 0)
			rate = -100/(estimate_time_runtime[0] * 60);
		else if (charging > 0)
			rate =  100/(estimate_time_runtime[1] * 60);

#		ifdef APM_ESTIMATE_DEBUG
		fprintf(stderr, "[r = %.3f]\t", rate);
#		endif
		}
	else
		{
		time_t dt1 = t - t1; /* delta since last rate change */
		gint dp1 = sys_apm.battery_percentage - p1;
		/* time for a rate change? */
		if (dt1 > APM_RATECHANGE_WINDOW && 
			((charging > 0 && dp1 >= 0) || (charging < 0 && dp1 <= 0)))
			{
			dt = t - t0;					/* since state change */
			dp = sys_apm.battery_percentage - p0;

			if (dp1 == 0)	/* flatlining (dp/dt = 0) */
				{
#				ifdef APM_ESTIMATE_DEBUG
				fprintf(stderr, "F");
#				endif
				rate = (1-APM_RATE_SMOOTHING/4) * rate;
				}
			else
				{
				rate = APM_RATE_SMOOTHING * ((gdouble) dp / (gdouble) (dt/60)) + 
					(1-APM_RATE_SMOOTHING) * rate;
				}

#				ifdef APM_ESTIMATE_DEBUG
				fprintf(stderr, "[dp = %+d dt = %.2f rate = %.3f]\t", dp, 
					(gdouble)dt/60, rate);
#				endif

			t1 = t;
			p1 = sys_apm.battery_percentage;
			}
		}

	if (charging && rate != 0.0)	/* (dis)charging */
		{
		gfloat eta;	/* "estimated time of arrival", minutes */

		/* estimated percentage (capacity) remaining (to full/empty) */
		gint p = charging > 0 ? 100 - sys_apm.battery_percentage : 
			sys_apm.battery_percentage - APM_EMPTY_CAPACITY;

		if (charging > 0 && estimate_time_model[1] && 
			sys_apm.battery_percentage < APM_CHARGE_MODEL_LIMIT && dp > 0)
			/* charging, use exponential: eta =~ 2.5 * time-constant (~=92%) */
			eta = -2.5 * dt/60 / (log(1 - (gdouble)dp/(gdouble)(p+dp)));
		else
			eta = abs((gdouble)p / rate);	/* use linear */

#		ifdef APM_ESTIMATE_DEBUG
		fprintf(stderr, "eta = %.2f\t", eta);
#		endif

		/* round off to nearest 5 mins */
		sys_apm.battery_time_left = (gint)((eta > 0 ? eta + 2.5: 0) / 5) * 5;
		sys_apm.battery_charge_rate = rate;

		}
	else
		{
		sys_apm.battery_time_left = INT_MAX;	/* inf */
		sys_apm.battery_charge_rate = 0.0;
		}

#		ifdef APM_ESTIMATE_DEBUG
		fprintf(stderr, "\n");
#		endif
	}

static void
draw_time_left_decal(gboolean force)
	{
	Decal	*d;
	gchar	buf[16];
	gint	x, w, t;
	static BatteryDisplayMode	last_mode = BATTERYDISPLAY_EOM;

	if (last_mode != sys_apm.battery_display_mode)
		force = TRUE;
	last_mode = sys_apm.battery_display_mode;

	switch (sys_apm.battery_display_mode)
		{
		case BATTERYDISPLAY_TIME:
			t = sys_apm.battery_time_left;
			if (t == INT_MAX || t == INT_MIN)
				snprintf(buf, sizeof(buf), "--");
			else
				snprintf(buf, sizeof(buf), "%2d:%02d", t / 60, t % 60);
			break;

		case BATTERYDISPLAY_RATE:
			/* t is used by draw_decal_text() to see if a refresh is reqd */
			t = (gint) (sys_apm.battery_charge_rate * 100.0);
			snprintf(buf, sizeof(buf), "%0.1f%%/m",
						sys_apm.battery_charge_rate);
			break;

		case BATTERYDISPLAY_PERCENT:
		default:
			t = sys_apm.battery_percentage;
			snprintf(buf, sizeof(buf), "%d%%", t);
			break;
		}

	d = minutes_left_decal;
	w = gdk_string_measure(d->text_style.font, buf);
	x = (d->w - w) / 2;
	if (x < 0)
		x = 0;
	d->x_off = x;
	gkrellm_draw_decal_text(apm, d, buf, force ? -1 : t);
	}

static void
update_apm_panel(void)
	{
	(*read_apm_data)();
	if (!sys_apm.battery_is_available)
		return;
	if (sys_apm.battery_time_left > 0 && sys_apm.battery_is_charging)
		{
		sys_apm.battery_charge_rate = (gfloat) sys_apm.battery_percentage / 
			(gfloat) sys_apm.battery_time_left;
		}
	else
		sys_apm.battery_charge_rate = 0.0;

	if (enable_estimate_time)
		estimate_battery_time_left();
	else if (sys_apm.battery_time_left == -1)
		sys_apm.battery_display_mode = BATTERYDISPLAY_PERCENT;

	gkrellm_check_alert(apm_alert, alert_units_are_percent
				? sys_apm.battery_percentage : sys_apm.battery_time_left);
	if (sys_apm.ac_is_on_line)
		{
		gkrellm_reset_alert(apm_alert);
		gkrellm_freeze_alert(apm_alert);
		gkrellm_draw_decal_pixmap(apm, power_source_decal, D_MISC_AC);
		}
	else
		{
		gkrellm_thaw_alert(apm_alert);
		gkrellm_draw_decal_pixmap(apm, power_source_decal, D_MISC_BATTERY);
		}
	draw_time_left_decal(FALSE);

	gkrellm_update_krell(apm, apm_krell, sys_apm.battery_percentage);
	gkrellm_draw_panel_layers(apm);
	}

static void
update_apm(void)
	{
	if (enable_apm && GK.five_second_tick)
		update_apm_panel();
	}


static gboolean
apm_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	if (widget == apm->drawing_area)
		{
		gdk_draw_pixmap(widget->window,
				widget->style->fg_gc[GTK_WIDGET_STATE(widget)], apm->pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
		}
	return FALSE;
	}

static gboolean
cb_panel_enter(GtkWidget *w, GdkEventButton *ev, gpointer data)
	{
	gkrellm_decal_on_top_layer(minutes_left_decal, TRUE);
	gkrellm_draw_panel_layers(apm);
	return TRUE;
	}

static gboolean
cb_panel_leave(GtkWidget *w, GdkEventButton *ev, gpointer data)
	{
	gkrellm_decal_on_top_layer(minutes_left_decal, FALSE);
	gkrellm_draw_panel_layers(apm);
	return TRUE;
	}


static gboolean
cb_panel_press(GtkWidget *widget, GdkEventButton *ev)
	{
	Decal			*d;
	static gboolean	time_unavailable_warned;

	d = launch.decal;
	if (ev->button == 3)
		gkrellm_open_config_window(mon_apm);
	else if (   ev->button == 2
			 || (ev->button == 1 && !d)
			 || (ev->button == 1 && d && ev->x < d->x)
			)
		{
		if (sys_apm.battery_time_left == -1)
			{
			if (!time_unavailable_warned)
				gkrellm_message_window(_("GKrellM APM"),
					_("Battery times are unavailable.  You\n"
					  "could select the Estimated Time option."),
					NULL);
			time_unavailable_warned = TRUE;
			sys_apm.battery_display_mode = BATTERYDISPLAY_PERCENT;
			}
		else
			{
			sys_apm.battery_display_mode++;
			if (sys_apm.battery_display_mode == BATTERYDISPLAY_EOM)
				sys_apm.battery_display_mode = 0;

			draw_time_left_decal(TRUE);
			gkrellm_draw_panel_layers(apm);
			gkrellm_config_modified();
			}
		}
	return TRUE;
	}

static void
create_apm(GtkWidget *vbox, gint first_create)
	{
	Style		*style;
	Margin		*m;
	gint		x, w;

	if (first_create)
		apm = gkrellm_panel_new0();

	style = gkrellm_meter_style(style_id);
	m = gkrellm_get_style_margins(style);
	power_source_decal = gkrellm_create_decal_pixmap(apm, GK.decal_misc_pixmap,
			GK.decal_misc_mask, N_MISC_DECALS, style, m->left, -1);

	x = power_source_decal->x + power_source_decal->w + 2;
	w = gkrellm_chart_width() - x - m->right;
	minutes_left_decal = gkrellm_create_decal_text(apm, "8/%",
						gkrellm_meter_textstyle(style_id),
						style, x, -1, w);

	apm_krell = gkrellm_create_krell(apm, gkrellm_krell_meter_image(style_id),
								style);
	gkrellm_monotonic_krell_values(apm_krell, FALSE);
	gkrellm_set_krell_full_scale(apm_krell, 100, 1);

	gkrellm_panel_configure(apm, NULL, style);
	gkrellm_panel_create(vbox, mon_apm, apm);

	/* Center the decals with respect to each other.
	*/
	if (power_source_decal->h > minutes_left_decal->h)
		minutes_left_decal->y +=
					(power_source_decal->h - minutes_left_decal->h) / 2;
	else
		power_source_decal->y +=
					(minutes_left_decal->h - power_source_decal->h) / 2;

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT (apm->drawing_area), "expose_event",
				(GtkSignalFunc) apm_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT (apm->drawing_area), "button_press_event",
				(GtkSignalFunc) cb_panel_press, NULL);
		gtk_signal_connect(GTK_OBJECT(apm->drawing_area), "enter_notify_event",
                (GtkSignalFunc) cb_panel_enter, NULL);
		gtk_signal_connect(GTK_OBJECT(apm->drawing_area), "leave_notify_event",
                (GtkSignalFunc) cb_panel_leave, NULL);
		 }
	gkrellm_setup_decal_launcher(apm, &launch, minutes_left_decal);

	/* Some laptops apparently have probs reading /proc/apm. So avoid it
	|  unless explicitely enabled.
	*/
	if (GK.demo)
		enable_apm = TRUE;
	if (enable_apm)
		{
		update_apm_panel();
		if (!sys_apm.battery_is_available)
			enable_apm = FALSE;
		}
	if (!enable_apm)
		gkrellm_panel_hide(apm);
	}

static void
cb_apm_alert_trigger(Alert *alert, gpointer data)
	{
	AlertDecal	*ad;
	Decal		*d;

	alert->panel = apm;
	ad = &alert->ad;
	d = minutes_left_decal;
	ad->x = d->x + 1;
	ad->y = d->y - 2;
	ad->w = d->w - 1;
	ad->h = d->h + 4;
	gkrellm_render_default_alert_decal(alert);
	}

  /* If the OS reports battery times, alerts will always have minutes units.
  |  If the OS does not report battery times the initial alert create will
  |  have minutes units if the estimate time option is enabled and it will
  |  have battery percent level units if estimate time option is off.  Alert
  |  creates from load config will have units in effect at last save config.
  */
static void
create_apm_alert(gboolean using_percent)
	{
	if (!apm_alert)
		{
		(*read_apm_data)();
		alert_units_are_percent = alert_units_need_estimate_mode = FALSE;
		if (   using_percent
			|| (sys_apm.battery_time_left == -1 && !enable_estimate_time)
		   )
			{
			apm_alert = gkrellm_alert_create(NULL, _("APM"),
					_("Battery Percent Limits"),
					FALSE, TRUE, TRUE, 99, 0, 1, 10, 0);
			alert_units_are_percent = TRUE;
			}
		else
			{
			apm_alert = gkrellm_alert_create(NULL, _("APM"),
					_("Battery Minutes Remaining Limits"),
					FALSE, TRUE, TRUE, 500, 0, 1, 10, 0);
			if (sys_apm.battery_time_left == -1)
				alert_units_need_estimate_mode = TRUE;
			}
		}
	gkrellm_alert_trigger_connect(apm_alert, cb_apm_alert_trigger, NULL);
	}


static void
save_apm_config(FILE *f)
	{
	fprintf(f, "%s enable %d\n", APM_CONFIG_KEYWORD, enable_apm);
	fprintf(f, "%s estimate_time %d\n", APM_CONFIG_KEYWORD,
				enable_estimate_time);
	fprintf(f, "%s estimate_time_discharge %.3f\n", APM_CONFIG_KEYWORD,
				estimate_time_runtime[0]);
	fprintf(f, "%s estimate_time_charge %.3f\n", APM_CONFIG_KEYWORD,
				estimate_time_runtime[1]);
	/* don't make config keywords > 31 chars */
	/* fprintf(f, "%s estimate_time_discharge_model %d\n", APM_CONFIG_KEYWORD,
				estimate_time_model[0]); */
	fprintf(f, "%s estimate_time_charge_model %d\n", APM_CONFIG_KEYWORD,
				estimate_time_model[1]);
	fprintf(f, "%s launch1 %s\n", APM_CONFIG_KEYWORD, launch.command);
	fprintf(f, "%s tooltip_comment %s\n",
				APM_CONFIG_KEYWORD, launch.tooltip_comment);
	fprintf(f, "%s display_mode %d\n", APM_CONFIG_KEYWORD,
				sys_apm.battery_display_mode);
	fprintf(f, "%s alert_units_percent %d\n", APM_CONFIG_KEYWORD,
				alert_units_are_percent);
	gkrellm_save_alertconfig(f, apm_alert, APM_CONFIG_KEYWORD, NULL);
	}

static void
load_apm_config(gchar *arg)
	{
	gchar			config[32], item[CFG_BUFSIZE];
	static gboolean	using_percent;

	if (sscanf(arg, "%31s %[^\n]", config, item) == 2)
		{
		if (!strcmp(config, "enable"))
			sscanf(item, "%d", &enable_apm);
		else if (!strcmp(config, "estimate_time"))
			sscanf(item, "%d", &enable_estimate_time);
		else if (!strcmp(config, "estimate_time_discharge"))
			sscanf(item, "%f", &estimate_time_runtime[0]);
		else if (!strcmp(config, "estimate_time_charge"))
			sscanf(item, "%f", &estimate_time_runtime[1]);
		/* getting close to 32 char limit ... */
		/* else if (!strcmp(config, "estimate_time_discharge_model"))
			sscanf(item, "%d", &estimate_time_model[0]); */
		else if (!strcmp(config, "estimate_time_charge_model"))
			sscanf(item, "%d", &estimate_time_model[1]);
		else if (!strcmp(config, "launch1"))
			launch.command = g_strdup(item);
		else if (!strcmp(config, "tooltip_comment"))
			launch.tooltip_comment = g_strdup(item);
		else if (!strcmp(config, "display_mode"))
			sscanf(item, "%d", (gint *)&sys_apm.battery_display_mode);
		else if (!strcmp(config, "alert_units_percent"))
			sscanf(item, "%d", &using_percent);
		else if (!strcmp(config, GKRELLM_ALERTCONFIG_KEYWORD))
			{
			create_apm_alert(using_percent);
			gkrellm_load_alertconfig(&apm_alert, item);
			}
		}
	}

static GtkWidget	*enable_apm_button,
					*enable_estimate_time_button;
static GtkWidget	*launch_entry,
					*tooltip_entry;
static GtkWidget	*alert_button;

static GtkWidget	*estimate_time_runtime_spin_button[2],
					*estimate_time_runtime_model_option[2];

static void
cb_apm_set_alert(GtkWidget *entry, gpointer data)
	{
	create_apm_alert(FALSE);
	gkrellm_alert_config_window(&apm_alert);
	}

static void
cb_enable_estimate_time_button(GtkWidget *widget, GtkWidget	*box)
	{
	gtk_widget_set_sensitive(box, 
		GTK_TOGGLE_BUTTON(enable_estimate_time_button)->active);
	}

static void 
apply_apm_config(void)
	{
	gboolean	new_enable;
	gboolean	update = FALSE;

	new_enable = GTK_TOGGLE_BUTTON(enable_apm_button)->active;
	if (!enable_apm && new_enable)
		{
		(*read_apm_data)();
		if (! sys_apm.battery_is_available)
			{
			gkrellm_config_message_window(_("APM Config Error"),
				_("No battery available."), NULL);
 			gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(enable_apm_button),
					FALSE);
			return;
			}
		update = TRUE;
		}
	enable_apm = new_enable;

	new_enable = GTK_TOGGLE_BUTTON(enable_estimate_time_button)->active;
	if (enable_estimate_time != new_enable)
		{
		/* If alert units need estimated time mode and estimation switches off,
		|  destroy the alert because the alert units can now only be percent.
		*/
		if (apm_alert && (!new_enable && alert_units_need_estimate_mode))
			{
			gkrellm_alert_destroy(&apm_alert);
			gkrellm_config_message_window(_("GKrellM APM"),
				_("The APM alert units are changed\n"
				  "and the alert must be reconfigured."), NULL);
			}
		update = TRUE;
		}
	enable_estimate_time = new_enable;

	estimate_time_runtime[0] = gtk_spin_button_get_value_as_float(
			GTK_SPIN_BUTTON(estimate_time_runtime_spin_button[0]));
	estimate_time_runtime[1] = gtk_spin_button_get_value_as_float(
			GTK_SPIN_BUTTON(estimate_time_runtime_spin_button[1]));
	estimate_time_model[1] = 
			GTK_TOGGLE_BUTTON(estimate_time_runtime_model_option[1])->active;

	if (enable_apm)
		{
		gkrellm_panel_show(apm);
		if (update)
			update_apm_panel();
		}
	else
		gkrellm_panel_hide(apm);

	gkrellm_apply_launcher(&launch_entry, &tooltip_entry, apm,
				&launch, gkrellm_launch_button_cb);
	}

static gchar	*apm_info_text[] =
{
N_("<b>Display Estimated Time\n"),
N_("If battery times are not reported by the BIOS or if the reported times\n"
"are inaccurate, select this option to display a battery time remaining or\n"
"time to charge which is calculated based on the current battery percentage\n"
"level, user supplied total battery times, and a linear extrapolation model.\n"),
"\n",
N_("<b>Total Battery Times\n"),
N_("Enter the typical total run time and total charge time in hours for your\n"
"battery.\n"),
"\n",
N_("<b>Exponential Charge Model\n"),		/* xgettext:no-c-format */
N_("For some charging systems battery capacity rises exponentially, which\n"
"means the simple linear model will grossly underestimate the time to 100%.\n"
"Select the exponential model for more accuracy in this case.\n"),
"\n",
N_("<b>Mouse Button Actions:\n"),
N_("<b>\tLeft "),
N_(" click on the charging state decal to toggle the display mode\n"
"\t\tbetween a minutes, percentage, or charging rate display.\n"),
N_("<b>\tMiddle "),
N_(" clicking anywhere on the APM panel also toggles the display mode.\n")
};

static void
create_apm_tab(GtkWidget *tab_vbox)
	{
	GtkWidget	*tabs, *table, *vbox, *vbox1, *hbox, *hbox2, *vbox2, *text;
	gint		i;

	tabs = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(tabs), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(tab_vbox), tabs, TRUE, TRUE, 0);

/* -- Setup tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Setup"));

	vbox1 = gkrellm_framed_vbox(vbox, _("Options"),
			4, FALSE, 0, 2);
	gkrellm_check_button(vbox1, &enable_apm_button, enable_apm, FALSE, 10,
			_("Enable APM"));

	vbox2 = gtk_vbox_new(FALSE, 0);
	gkrellm_check_button_connected(vbox1, &enable_estimate_time_button, 
			enable_estimate_time, FALSE, FALSE, 0, 
			(GtkSignalFunc) cb_enable_estimate_time_button, vbox2, 
			_("Display estimated time remaining and time to charge"));
	gtk_widget_set_sensitive(vbox2, enable_estimate_time ? TRUE : FALSE);
	gtk_box_pack_start(GTK_BOX(vbox1), vbox2, FALSE, FALSE, 0);

	hbox2 = gtk_hbox_new(FALSE, 10);
	gkrellm_spin_button(hbox2, &estimate_time_runtime_spin_button[0], 
			estimate_time_runtime[0], 0.1, 24, 0.1, 1.0, 1, 55, NULL, NULL, 
			FALSE, _("Total battery run time in hours"));
	gtk_box_pack_start(GTK_BOX(vbox2), hbox2, FALSE, FALSE, 0);

	hbox2 = gtk_hbox_new(FALSE, 10);
	gkrellm_spin_button(hbox2, &estimate_time_runtime_spin_button[1], 
			estimate_time_runtime[1], 0.1, 24, 0.1, 1.0, 1, 55, NULL, NULL, 
			FALSE, _("Total battery charge time in hours"));
	/* to become an option button if/when other models are implemented? */
	gkrellm_check_button(hbox2, &estimate_time_runtime_model_option[1], 
            estimate_time_model[1], FALSE, 0,
			_("Exponential charge model"));
	gtk_box_pack_start(GTK_BOX(vbox2), hbox2, FALSE, FALSE, 0);


	hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
	gkrellm_button_connected(hbox, &alert_button, FALSE, FALSE, 0,
			cb_apm_set_alert, NULL, _("Alerts"));

	vbox1 = gkrellm_framed_vbox_end(vbox, _("Launch Commands"),
			4, FALSE, 0, 2);
	table = gkrellm_launcher_table_new(vbox1, 1);
	gkrellm_config_launcher(table, 0, &launch_entry, &tooltip_entry, 
					_("APM"), &launch);

/* --Info tab */
	vbox = gkrellm_create_framed_tab(tabs, _("Info"));
	text = gkrellm_scrolled_text(vbox, NULL,
				GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	for (i = 0; i < sizeof(apm_info_text)/sizeof(gchar *); ++i)
		gkrellm_add_info_text_string(text, _(apm_info_text[i]));
	}

static Monitor	monitor_apm =
	{
	N_("APM"),			/* Name, for config tab.	*/
	MON_APM,			/* Id, 0 if a plugin		*/
	create_apm,			/* The create function		*/
	update_apm,			/* The update function		*/
	create_apm_tab,		/* The config tab create function	*/
	apply_apm_config,	/* Apply the config function		*/

	save_apm_config,	/* Save user conifg			*/
	load_apm_config,	/* Load user config			*/
	APM_CONFIG_KEYWORD,	/* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_apm_monitor(void)
	{
	enable_apm = FALSE;			/* Some people have probs reading /proc/apm */
	enable_estimate_time = FALSE;	/* use the BIOS value */
	estimate_time_runtime[0] =  1.5;	/* 1.5 hour battery */
	estimate_time_runtime[1] =	3.0;	/* 3 hour recharge */
	monitor_apm.name=_(monitor_apm.name);
	style_id = gkrellm_add_meter_style(&monitor_apm, APM_STYLE_NAME);
	mon_apm = &monitor_apm;
	if (setup_apm_interface())
		return &monitor_apm;
	return NULL;
	}

