/* GKrellM
|  Copyright (C) 1999-2002 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.  Version 2 is in the
|  COPYRIGHT file in the top level directory of this distribution.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

static GList	*panel_list;


GList *
gkrellm_get_panel_list(void)
	{
	return panel_list;
	}

void
gkrellm_panel_configure_add_height(Panel *p, gint h)
	{
	if (!p)
		return;
	p->label->h_panel += h;
	}

/* Deprecated function - name change */
void
gkrellm_configure_panel(Panel *p, gchar *string, Style *style)
	{
	gkrellm_panel_configure(p, string, style);
	}

static void
setup_panel_style(Panel *p, Style *style)
	{
	Label	*lbl	= p->label;

	p->style = style;
	if (!style)
		return;
	lbl->position = style->label_position;
	p->transparency = style->transparency;
	GK.any_transparency |= p->transparency;
	if (!p->textstyle)
		{
		p->textstyle = gkrellm_textstyle_new0();
		p->textstyle->internal = TRUE;
		}
	if (p->textstyle->internal)
		{
		*(p->textstyle) = style->label_tsA;
		p->textstyle->internal = TRUE;
		p->textstyle->font = *(style->label_tsA.font_seed);
		}
	}

  /* Calculate panel height required for given border, label height,
  |  krell heights, and decal heights.  Also calculate label extents
  |  and load all this info into a Label structure.
  |  After panel height is calculated, calling routine may want to adjust
  |  krell or decal y offsets for centering.  
  */
void
gkrellm_panel_configure(Panel *p, gchar *string, Style *style)
	{
	GList			*list;
	Krell			*k;
	Decal			*d;
	Label			*lbl;
	TextStyle		*ts = NULL;
	gint			y, h_panel, h_decal;
	gint			top_margin = 0, bottom_margin = 0;

	if (!p)
		return;
	p->style = style;
	lbl = p->label;
	lbl->position = GKRELLM_LABEL_CENTER;
	setup_panel_style(p, style);
	gkrellm_get_top_bottom_margins(style, &top_margin, &bottom_margin);

	ts = p->textstyle;
	gkrellm_dup_string(&lbl->string, string);
	if (lbl->string && ts && lbl->position >= 0)
		gdk_string_extents(ts->font, string, &lbl->lbearing,
				&lbl->rbearing, &lbl->width, &lbl->ascent, &lbl->descent);
	else
		{
		lbl->lbearing = 0;
		lbl->rbearing = 0;
		lbl->width = 0;
		lbl->ascent = 0;
		lbl->descent = 0;
		}
	h_panel = top_margin + bottom_margin;
	h_panel += lbl->ascent + lbl->descent + (ts ? ts->effect : 0);

	/* If krell_yoff is -1, then in gkrellm_create_krell() k->y0 was put
	|  at the top margin.  If krell_yoff is < -1, then here I will bottom
	|  justify the krell (so must go through krell list twice).
	|  Decals are assumed to fit inside of margins.  Caller must subtract
	|  off bottom_margin if this is not so.
	*/
	for (list = p->decal; list; list = list->next)
		{
		d = (Decal *) list->data;
		h_decal = d->y + d->h + bottom_margin;
		if (h_decal > h_panel)
			h_panel = h_decal;
		}
	for (list = p->krell; list; list = list->next)
		{
		k = (Krell *) list->data;
		y = (k->y0 < 0) ? 0 : k->y0;
		if (y + k->h_frame > h_panel)
			h_panel = y + k->h_frame;
		}
	for (list = p->krell; list; list = list->next)
		{
		k = (Krell *) list->data;
		if (k->y0 < 0)
			k->y0 = h_panel - k->h_frame;		/* Bottom justify */
		}

	if (h_panel <= 0)
		h_panel = 1;

	lbl->y_baseline = top_margin + lbl->ascent;
	lbl->h_panel = h_panel;
	}

static void
draw_panel(Panel *p, gint to_screen)
	{
	Label		*lbl	= p->label;
	TextStyle	*ts		= p->textstyle;
	Margin		*m;
	gchar		*s;
	gint		xdst;

	if (!draw_rootpixmap_onto_transparent_panel(p))
		gdk_draw_pixmap(p->bg_pixmap, GK.draw1_GC, p->bg_clean_pixmap,
					0, 0, 0, 0, p->w, p->h);

	gdk_draw_pixmap(p->pixmap, GK.draw1_GC, p->bg_pixmap,
					0, 0, 0, 0, p->w, p->h);
	if (   lbl && ts
		&& ((s = lbl->string) != NULL)
		&& lbl->position >= 0
	   )
		{
		m = gkrellm_get_style_margins(p->style);
		lbl->width = gdk_string_width(ts->font, s);
		xdst = x_label_position(lbl->position, p->w, lbl->width, m->left);
		lbl->x_panel = xdst;
		gkrellm_draw_string(p->pixmap, ts, xdst, lbl->y_baseline, s);
		gkrellm_draw_string(p->bg_pixmap, ts, xdst, lbl->y_baseline, s);
		}
	if (p->drawing_area->window && to_screen)
		{
		gdk_draw_pixmap(p->drawing_area->window, GK.draw1_GC, p->pixmap,
					0, 0, 0, 0, p->w, p->h);
		gkrellm_draw_panel_layers_force(p);
		}
	}

void
gkrellm_draw_panel_label(Panel *p, GdkImlibImage *image /* obsolete */)
	{
	draw_panel(p, TRUE);
	}

void
gkrellm_panel_destroy(Panel *p)
	{
	if (!p)
		return;
	gkrellm_reset_panel_alerts(p);
	gkrellm_destroy_krell_list(p);
	gkrellm_destroy_decal_list(p);

	if (p->label)
		{
		if (p->label->string)
			g_free(p->label->string);
		g_free(p->label);
		}
	if (p->textstyle && p->textstyle->internal)
		g_free(p->textstyle);
	if (p->pixmap)
		gdk_pixmap_unref(p->pixmap);
	if (p->bg_pixmap)
		gdk_pixmap_unref(p->bg_pixmap);
	if (p->bg_clean_pixmap)
		gdk_imlib_free_pixmap(p->bg_clean_pixmap);

	if (p->drawing_area)
		gtk_signal_handlers_destroy(GTK_OBJECT(p->drawing_area));
	if (p->hbox)
		gtk_widget_destroy(p->hbox);
	panel_list = g_list_remove(panel_list, p);
	if (p->shown)
		gkrellm_monitor_height_adjust(- p->h);
	g_free(p);
	gkrellm_pack_side_frames();
	}

/* Deprecated function - name change */
void
gkrellm_destroy_panel(Panel *p)
	{
	if (!p)
		return;
	gkrellm_reset_panel_alerts(p);
	gkrellm_destroy_krell_list(p);
	gkrellm_destroy_decal_list(p);

	if (p->label)
		{
		if (p->label->string)
			g_free(p->label->string);
		g_free(p->label);
		}
	if (p->pixmap)
		gdk_pixmap_unref(p->pixmap);
	if (p->bg_pixmap)
		gdk_pixmap_unref(p->bg_pixmap);
	if (p->bg_clean_pixmap)
		gdk_imlib_free_pixmap(p->bg_clean_pixmap);

	if (p->drawing_area)
		gtk_signal_handlers_destroy(GTK_OBJECT(p->drawing_area));
	if (p->hbox)
		gtk_widget_destroy(p->hbox);
	panel_list = g_list_remove(panel_list, p);
	memset(p, 0, sizeof(Panel));
	}

void
gkrellm_panel_bg_image_override(Panel *p, GdkImlibImage *bg_image)
	{
	if (!p || !bg_image)
		return;
	p->bg_image = bg_image;
	p->bg_image_override = TRUE;
	}

void
gkrellm_panel_keep_lists(Panel *p, gboolean keep)
	{
	if (p)
		p->keep_lists = keep;
	}

void
gkrellm_panel_create(GtkWidget *vbox, Monitor *mon, Panel *p)
	{
	GtkWidget	*hbox;
	GtkWidget	*fixed;
	GtkWidget   *top_win = gkrellm_get_top_window();

	if (!vbox || !mon || !p)
		return;
	p->monitor = (gpointer) mon;
	if (!p->style)	/* gkrellm_panel_configure() may not have been called. */
		setup_panel_style(p, mon->privat->panel_style);

	if (!p->bg_image_override)
		{
		if (mon->privat->style_type == CHART_PANEL_TYPE)
			p->bg_image = gkrellm_bg_panel_image(mon->privat->style_id);
		else
			p->bg_image = gkrellm_bg_meter_image(mon->privat->style_id);
		}
	p->bg_image_override = FALSE;

	/* If not being called from rebuild or after a panel destroy, then panel
	|  still has a height that must be accounted for.
	*/
	if (p->h && p->shown)
		gkrellm_monitor_height_adjust(- p->h);
	p->h = p->label->h_panel;
	p->w = GK.chart_width;

	if (p->hbox == NULL)
		{
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_add (GTK_CONTAINER(vbox), hbox);
		p->hbox = hbox;

		/* And create the drawing area that the middle panel lives in and put
		|  it into a fixed widget.
		*/
		fixed = gtk_fixed_new();
		gtk_box_pack_start(GTK_BOX(hbox), fixed, TRUE, TRUE, 0);
		gtk_widget_show(fixed);
		p->fixed = fixed;
		p->drawing_area = gtk_drawing_area_new();
		gtk_widget_set_events (p->drawing_area, GDK_EXPOSURE_MASK
				| GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK
				| GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK
				| GDK_POINTER_MOTION_MASK);
		gtk_fixed_put(GTK_FIXED(fixed), p->drawing_area, 0, 0);
		gtk_widget_show(p->drawing_area);
		gtk_widget_show(hbox);
		p->shown = TRUE;
		gtk_widget_realize(hbox);
		gtk_widget_realize(p->fixed);
		gtk_widget_realize(p->drawing_area);
		panel_list = g_list_append(panel_list, p);
		}
	gtk_drawing_area_size(GTK_DRAWING_AREA (p->drawing_area), p->w, p->h);
	gkrellm_render_to_pixmap(p->bg_image, &p->bg_clean_pixmap,
			&p->bg_mask, p->w, p->h);
	if (p->bg_pixmap)
		gdk_pixmap_unref(p->bg_pixmap);
	p->bg_pixmap = gdk_pixmap_new(top_win->window, p->w, p->h, -1);
	if (p->pixmap)
		gdk_pixmap_unref(p->pixmap);
	p->pixmap = gdk_pixmap_new(top_win->window, p->w, p->h, -1);

	if (p->shown)
		{
		gkrellm_monitor_height_adjust(p->h);
		gkrellm_pack_side_frames();
		}
	draw_panel(p, FALSE);
	}

/* Deprecated function - prototype and name change */
void
gkrellm_create_panel(GtkWidget *vbox, Panel *p, GdkImlibImage *bg_image)
	{
	GtkWidget	*hbox;
	GtkWidget	*fixed;
	GtkWidget   *top_win = gkrellm_get_top_window();

	p->h = p->label->h_panel;
	p->w = GK.chart_width;

	if (p->hbox == NULL)
		{
		hbox = gtk_hbox_new(FALSE, 0);
		gtk_container_add (GTK_CONTAINER(vbox), hbox);
		p->hbox = hbox;

		/* And create the drawing area that the middle panel lives in and put
		|  it into a fixed widget.
		*/
		fixed = gtk_fixed_new();
		gtk_box_pack_start(GTK_BOX(hbox), fixed, TRUE, TRUE, 0);
		gtk_widget_show(fixed);
		p->fixed = fixed;
		p->drawing_area = gtk_drawing_area_new();
		gtk_widget_set_events (p->drawing_area, GDK_EXPOSURE_MASK
				| GDK_LEAVE_NOTIFY_MASK | GDK_ENTER_NOTIFY_MASK
				| GDK_BUTTON_RELEASE_MASK | GDK_BUTTON_PRESS_MASK
				| GDK_POINTER_MOTION_MASK);
		gtk_fixed_put(GTK_FIXED(fixed), p->drawing_area, 0, 0);
		gtk_widget_show(p->drawing_area);
		gtk_widget_show(hbox);
		gtk_widget_realize(hbox);
		gtk_widget_realize(p->fixed);
		gtk_widget_realize(p->drawing_area);
		panel_list = g_list_append(panel_list, p);
		}
	gtk_drawing_area_size(GTK_DRAWING_AREA (p->drawing_area), p->w, p->h);
	gkrellm_render_to_pixmap(bg_image, &p->bg_clean_pixmap,
			&p->bg_mask, p->w, p->h);
	if (p->bg_pixmap)
		gdk_pixmap_unref(p->bg_pixmap);
	p->bg_pixmap = gdk_pixmap_new(top_win->window, p->w, p->h, -1);
	if (p->pixmap)
		gdk_pixmap_unref(p->pixmap);
	p->pixmap = gdk_pixmap_new(top_win->window, p->w, p->h, -1);
	p->keep_lists = TRUE;
	draw_panel(p, FALSE);
	}

void
gkrellm_panel_hide(Panel *p)
	{
	if (!p || !p->shown)
		return;
	gtk_widget_hide(p->hbox);
	p->shown = FALSE;
	gkrellm_monitor_height_adjust(- p->h);
	gkrellm_pack_side_frames();
	}

void
gkrellm_panel_show(Panel *p)
	{
	if (!p || p->shown)
		return;
	gtk_widget_show(p->hbox);
	p->shown = TRUE;
	gkrellm_monitor_height_adjust(p->h);
	gkrellm_pack_side_frames();
	}

gboolean
gkrellm_is_panel_visible(Panel *p)
    {
    if (!p)
        return FALSE;
    return p->shown;
    }

gboolean
gkrellm_panel_enable_visibility(Panel *p, gboolean new_vis,
					gboolean *current_vis)
	{
	gboolean	changed = FALSE;

	if (new_vis  && ! *current_vis)
		{
		gkrellm_panel_show(p);
		*current_vis  = TRUE;
		changed = TRUE;
		}
	if (!new_vis  && *current_vis)
		{
		gkrellm_panel_hide(p);
		*current_vis  = FALSE;
		changed = TRUE;
		}
	return changed;
	}

  /* Called from rebuild.  All panels must be cleaned out of things
  |  that will be recreated in the create() routines.  GKrellM <= 1.0.x
  |  left it up to plugins to destroy decal/krell lists.  Now this is
  |  enforced, but grandfather out the plugins have not upgraded to using
  |  the new gkrellm_panel_create() functions.
  */
void
clean_out_panels(void)
	{
	GList	*list, *list1;
	Panel	*p;

	destroy_decal_button_list();
	for (list = panel_list; list; list = list->next)
		{
		p = (Panel *) list->data;
		if (!p->keep_lists)
			{
			gkrellm_destroy_krell_list(p);
			gkrellm_destroy_decal_list(p);
			}
		else
			{
			for (list1 = p->decal; list1; list1 = list1->next)
				((Decal *) list1->data)->value = -1;
			for (list1 = p->krell; list1; list1 = list1->next)
				((Krell *) list1->data)->x_position = -1;
			}
		p->h = 0;
		p->label->h_panel = 0;
		}
	}
