/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

time_t		time_now;

static Panel	*pclock,
				*pcal;

static Decal	*d_wday,
				*d_mday,
				*d_month,
				*d_clock,
				*d_seconds;


static Launcher	clock_launch,
				cal_launch;

typedef struct
	{
	TextStyle	ts;
	gint		lbearing,
				rbearing,
				width,
				ascent,
				descent;
	}
	Extents;

static gint			clock_enable,
					clock_12hr_seconds,
					clock_24hr_format;

static gint			clock_visible,
					cal_visible;
static gint			clock_needs_refresh;

static Extents		wday_extents,
					mday_extents,
					month_extents,
					time_extents,
					sec_extents,
					ampm_extents;

static struct tm	*pTm;
struct tm			current_tm;

static gint			clock_style_id,
					cal_style_id;


struct tm *
gkrellm_get_current_time(void)
	{
	return &current_tm;
	}

static void
string_extents(gchar *string, Extents *ext)
	{
	gdk_string_extents(ext->ts.font, string, &ext->lbearing, &ext->rbearing,
			&ext->width, &ext->ascent, &ext->descent);
	}


static void
draw_clock(struct tm *t)
	{
	gchar	buf[32], sec_ampm[32];
	gint	w;

	if (t == NULL || clock_visible == FALSE)
		return;

	if (clock_needs_refresh)
		d_clock->value = -1;
	if (t->tm_min != current_tm.tm_min || clock_needs_refresh)
		{
		strftime(buf, sizeof(buf), "%a", t);  /* Abbreviated weekday name */
		w = gdk_string_width(d_wday->text_style.font, buf);
		d_wday->x_off = (w < d_wday->w) ? (d_wday->w - w) / 2 : 0;
		gkrellm_draw_decal_text(pcal, d_wday, buf, t->tm_min);

		/*strftime(buf, sizeof(buf), "%e", t);*/  /* Day in month 1-31 */
		snprintf(buf, sizeof(buf), "%d", t->tm_mday);
		w = gdk_string_width(d_mday->text_style.font, buf);
		d_mday->x_off = (w < d_mday->w) ? (d_mday->w - w) / 2 : 0;
		gkrellm_draw_decal_text(pcal, d_mday, buf, t->tm_min);

		strftime(buf, sizeof(buf), "%b", t); /* Abbreviated month name */
		gkrellm_draw_decal_text(pcal, d_month, buf, t->tm_min);
		gkrellm_draw_layers(pcal);

		if (clock_24hr_format)
			strftime(buf, sizeof(buf), "%k:%M", t);   /* 24 hour:minute */
		else
			strftime(buf, sizeof(buf), "%l:%M", t);   /* 12 hour:minute */
		w = gdk_string_width(d_clock->text_style.font, buf);
		d_clock->x_off = (w < d_clock->w) ? (d_clock->w - w) / 2 : 0;
		gkrellm_draw_decal_text(pclock, d_clock, buf, t->tm_min);
		}
	if (clock_12hr_seconds || clock_24hr_format)
		{
		strftime(sec_ampm, sizeof(sec_ampm), "%S", t);   /* seconds 00-59 */
		gkrellm_draw_decal_text(pclock, d_seconds, sec_ampm, t->tm_sec);
		}
	else
		{
		strftime(sec_ampm, sizeof(sec_ampm), "%p", t);   /* AM/PM */
		gkrellm_draw_decal_text(pclock, d_seconds, sec_ampm, t->tm_min);
		}
	gkrellm_draw_layers(pclock);
	clock_needs_refresh = FALSE;
	}

static void
update_clock(void)
	{
	GK.two_second_tick = FALSE;
	GK.five_second_tick = FALSE;
	GK.ten_second_tick = FALSE;
	GK.minute_tick = FALSE;
	GK.hour_tick = FALSE;
	GK.day_tick = FALSE;

	if (GK.second_tick)
		{
		pTm = localtime(&time_now);
		draw_clock(pTm);
		GK.two_second_tick = ((pTm->tm_sec % 2) == 0) ? TRUE : FALSE;
		GK.five_second_tick = ((pTm->tm_sec % 5) == 0) ? TRUE : FALSE;
		GK.ten_second_tick = ((pTm->tm_sec % 10) == 0) ? TRUE : FALSE;
		GK.minute_tick = (pTm->tm_min  != current_tm.tm_min)  ? TRUE : FALSE;
		GK.hour_tick   = (pTm->tm_hour != current_tm.tm_hour) ? TRUE : FALSE;
		GK.day_tick   = (pTm->tm_mday != current_tm.tm_mday) ? TRUE : FALSE;
		current_tm = *pTm;
		}
	}

static gint
expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	GdkPixmap	*pixmap = NULL;

	if (widget == pcal->drawing_area)
		pixmap = pcal->pixmap;
	else if (widget == pclock->drawing_area)
		pixmap = pclock->pixmap;
	if (pixmap)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				ev->area.width, ev->area.height);
	return FALSE;
	}

static void
create_clock(GtkWidget *vbox, gint first_create)
	{
	Style	*style;
	gint	w;

	if (first_create)
		{
		pclock = gkrellm_panel_new0();
		pcal = gkrellm_panel_new0();
		}
	else
		{
        gkrellm_destroy_decal_list(pclock);
        gkrellm_destroy_krell_list(pclock);
        gkrellm_destroy_decal_list(pcal);
        gkrellm_destroy_krell_list(pcal);
		}
	style = gkrellm_meter_style(cal_style_id);

	wday_extents.ts = *gkrellm_meter_textstyle(cal_style_id);
	string_extents("Wed", &wday_extents);
	mday_extents.ts = *gkrellm_meter_alt_textstyle(cal_style_id);
	string_extents("28", &mday_extents);
	month_extents.ts = *gkrellm_meter_textstyle(cal_style_id);
	string_extents("Aug", &month_extents);

	d_wday = gkrellm_create_decal_text(pcal, "88", &wday_extents.ts,
				style, 0, -1 /*top margin*/, wday_extents.width + 2);
	d_mday = gkrellm_create_decal_text(pcal, "88", &mday_extents.ts,
				style, 0, -1 /*top margin*/, mday_extents.width + 2);
	d_month = gkrellm_create_decal_text(pcal, "88", &month_extents.ts,
				style, 0, -1 /*top margin*/, month_extents.width + 2);
	w = d_wday->w + d_mday->w + d_month->w;
	d_wday->x = (UC.chart_width - w) / 2;
	d_mday->x = d_wday->x + d_wday->w;
	d_month->x = d_mday->x + d_mday->w;
	if (d_wday->h < d_mday->h)
		{
		d_wday->y = d_mday->y + d_mday->h - d_wday->h - 1;
		d_month->y = d_wday->y;
		}
	else if (d_mday->h < d_wday->h)
		d_mday->y = d_wday->y + d_wday->h - d_mday->h - 1;
	gkrellm_configure_panel(pcal, NULL, style);
	gkrellm_create_panel(vbox, pcal, gkrellm_bg_meter_image(cal_style_id));


	style = gkrellm_meter_style(clock_style_id);

	time_extents.ts = *gkrellm_meter_textstyle(clock_style_id);
	string_extents("00:00", &time_extents);
	sec_extents.ts = *gkrellm_meter_alt_textstyle(clock_style_id);
	string_extents("8M", &sec_extents);
	ampm_extents.ts = *gkrellm_meter_alt_textstyle(clock_style_id);
	string_extents("8M", &ampm_extents);

	d_clock = gkrellm_create_decal_text(pclock, "88", &time_extents.ts,
				style, 0, -1 /*top margin*/, time_extents.width + 2);
	d_seconds = gkrellm_create_decal_text(pclock, "88", &sec_extents.ts,
				style, 0, -1, sec_extents.width + 2);

	d_clock->x = (UC.chart_width - d_clock->w - d_seconds->w) / 2;
	d_seconds->x = d_clock->x + d_clock->w + 2;
	d_seconds->y = d_clock->y + d_clock->h - d_seconds->h;

	gkrellm_configure_panel(pclock, NULL, style);
	gkrellm_create_panel(vbox, pclock, gkrellm_bg_meter_image(clock_style_id));

	if (first_create)
		{
		/* Help the motion out hack. If starting a move in host panel and mouse
		|  jerks into the pclock/pcal drawing areas, we stop moving unless this:
		*/
		extern void gkrellm_motion(GtkWidget *, GdkEventMotion *, gpointer);

		gtk_signal_connect(GTK_OBJECT(pcal->drawing_area),
				"motion_notify_event", (GtkSignalFunc) gkrellm_motion, NULL);
		gtk_signal_connect(GTK_OBJECT (pcal->drawing_area), "expose_event",
				(GtkSignalFunc) expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT (pclock->drawing_area), "expose_event",
				(GtkSignalFunc) expose_event, NULL);
		}
	gkrellm_setup_launcher(pcal, &cal_launch, METER_PANEL_TYPE, 0);
	gkrellm_setup_launcher(pclock, &clock_launch, METER_PANEL_TYPE, 0);

	if (clock_enable)
		{
		GK.monitor_height += pcal->h + pclock->h;
		clock_visible = TRUE;
		cal_visible = TRUE;
		}
	else
		{
		gtk_widget_hide(pcal->hbox);
		gtk_widget_hide(pclock->hbox);
		}

	time(&time_now);
	pTm = localtime(&time_now);
	clock_needs_refresh = TRUE;
	draw_clock(pTm);
	}

#define	CLOCK_CONFIG_KEYWORD	"clock_cal"

static void
save_clock_config(FILE *f)
	{
	fprintf(f, "%s clock_launch %s\n", CLOCK_CONFIG_KEYWORD,
				clock_launch.command);
	fprintf(f, "%s clock_tooltip %s\n", CLOCK_CONFIG_KEYWORD,
				clock_launch.tooltip_comment);
	fprintf(f, "%s cal_launch %s\n", CLOCK_CONFIG_KEYWORD,
				cal_launch.command);
	fprintf(f, "%s cal_tooltip %s\n", CLOCK_CONFIG_KEYWORD,
				cal_launch.tooltip_comment);
	fprintf(f, "%s clock_options %d %d %d\n", CLOCK_CONFIG_KEYWORD,
				clock_enable, clock_12hr_seconds, clock_24hr_format);
	}

static void
load_clock_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "clock_launch") == 0)
			clock_launch.command = g_strdup(item);
		else if (strcmp(config, "clock_tooltip") == 0)
			clock_launch.tooltip_comment = g_strdup(item);
		else if (strcmp(config, "cal_launch") == 0)
			cal_launch.command = g_strdup(item);
		else if (strcmp(config, "cal_tooltip") == 0)
			cal_launch.tooltip_comment = g_strdup(item);
		else if (strcmp(config, "clock_options") == 0)
			sscanf(item, "%d %d %d", &clock_enable, &clock_12hr_seconds,
					&clock_24hr_format);
		}
	}

/* --------------------------------------------------------------------- */
static GtkWidget	*cal_launch_entry,
					*cal_tooltip_entry,
					*clock_launch_entry,
					*clock_tooltip_entry,
					*enable_clk_button,
					*clk_24hr_button,
					*clk_12hr_sec_button;


static void
apply_clock_config()
	{
	clock_enable = GTK_TOGGLE_BUTTON (enable_clk_button)->active;
	clock_24hr_format = GTK_TOGGLE_BUTTON (clk_24hr_button)->active;
	clock_12hr_seconds = GTK_TOGGLE_BUTTON (clk_12hr_sec_button)->active;

	gkrellm_apply_launcher(&cal_launch_entry, &cal_tooltip_entry, pcal,
					&cal_launch, gkrellm_launch_button_cb);
	gkrellm_apply_launcher(&clock_launch_entry, &clock_tooltip_entry, pclock,
					&clock_launch, gkrellm_launch_button_cb);

	gkrellm_enable_visibility(clock_enable, &clock_visible, pclock->hbox,
						pclock->h);
	gkrellm_enable_visibility(clock_enable, &cal_visible, pcal->hbox,
						pcal->h);
	clock_needs_refresh = TRUE;
	draw_clock(pTm);
	}


static void
create_clock_tab(GtkWidget *vbox)
	{
	GtkWidget	*table;

	gkrellm_check_button(vbox, &enable_clk_button, clock_enable, TRUE, 0,
				_("Enable Calendar/Clock"));

	gkrellm_check_button(vbox, &clk_12hr_sec_button, clock_12hr_seconds, FALSE,
			4, _("Show seconds instead of am/pm in 12 hour time mode"));

	gkrellm_check_button(vbox, &clk_24hr_button, clock_24hr_format, FALSE, 4,
					_("Display 24 hour instead of 12 hour time"));
	insert_expanded_filler(vbox);

	table = gkrellm_launcher_table_new(vbox, 2);

	gkrellm_config_launcher(table, 0,  &cal_launch_entry,
				&cal_tooltip_entry, _("Calendar"), &cal_launch);
	gkrellm_config_launcher(table, 1,  &clock_launch_entry,
				&clock_tooltip_entry, _("Clock"), &clock_launch);
	}


static Monitor	monitor_clock =
	{
	N_("Calendar Clock"), /* Name, for config tab.	*/
	MON_CLOCK,			/* Id,  0 if a plugin		*/
	create_clock,		/* The create function		*/
	update_clock,		/* The update function		*/
	create_clock_tab,	/* The config tab create function	*/
	apply_clock_config,	/* Apply the config function		*/

	save_clock_config,	/* Save user conifg			*/
	load_clock_config,	/* Load user config			*/
	CLOCK_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_clock_monitor(void)
	{
	clock_enable = TRUE;
    monitor_clock.name = _(monitor_clock.name);
	clock_12hr_seconds = TRUE;
	clock_style_id = gkrellm_add_meter_style(&monitor_clock, CLOCK_STYLE_NAME);
	return &monitor_clock;
	}

  /* Note: since clock and calendar are both created by create_clock, a
  |  plugin cannot actually insert itself between cal and clock.  FIXME
  */
static Monitor	monitor_cal =
	{
	N_("Calendar"),	/* Name, for config tab.	*/
	MON_CAL,		/* Id,  0 if a plugin		*/
	NULL,			/* The create function		*/
	NULL,			/* The update function		*/
	NULL,			/* The config tab create function	*/
	NULL,			/* Apply the config function		*/

	NULL,			/* Save user conifg			*/
	NULL,			/* Load user config			*/
	NULL,			/* config keyword			*/

	NULL,			/* Undef 2	*/
	NULL,			/* Undef 1	*/
	NULL,			/* Undef 0	*/

	0,				/* insert_before_id - place plugin before this mon */

	NULL,			/* Handle if a plugin, filled in by GKrellM		*/
	NULL			/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_cal_monitor(void)
	{
	cal_style_id = gkrellm_add_meter_style(&monitor_cal, CAL_STYLE_NAME);
	return &monitor_cal;
	}

