/* GKrellM
|  Copyright (C) 1999-2001 Bill Wilson
|
|  Author:  Bill Wilson    bill@gkrellm.net
|  Latest versions might be found at:  http://gkrellm.net
|
|  This program is free software which I release under the GNU General Public
|  License. You may redistribute and/or modify this program under the terms
|  of that license as published by the Free Software Foundation; either
|  version 2 of the License, or (at your option) any later version.
|
|  This program is distributed in the hope that it will be useful,
|  but WITHOUT ANY WARRANTY; without even the implied warranty of
|  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
|  GNU General Public License for more details.
| 
|  To get a copy of the GNU General Puplic License, write to the Free Software
|  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

/*
|  1/23/2001 Hajimu UMEMOTO added init_freebsd_proc(), improved
|				read_freebsd_proc()
| 10/12/2000 NetBSD code contributed by Anthony Mallet <metall@ficus.yi.org>
| 2/25/2000  FreeBSD code contributed by Hajimu UMEMOTO ume@mahoroba.org
*/

#include "gkrellm.h"
#include "gkrellm_private_proto.h"

#include <math.h>


typedef struct
	{
	GtkWidget	*vbox;
	Chart		*chart;
	gint		enabled;
	gint		height;
	gint		extra_info;
	gint		save_label_position;
	Decal		*sensor_decal,
				*fan_decal;

	/* (*read_system_proc_info)() needs to fill in this data	*/
	gint		n_users;
	gint		n_processes;
	gint		n_running;
	gulong		n_forks;
	gfloat		fload;
	}
	ProcMon;

ProcMon	proc;

void	(*read_system_proc_info)();


/* ====== System dependent interface ====================================== */

/* ------- FreeBSD ----------------------------------------------------- */
#if defined(__FreeBSD__)
#include <osreldate.h>
#include <sys/sysctl.h>
#if __FreeBSD_version >= 400000
#include <sys/user.h>
#endif

/*
 * This is ugly, but we need PID_MAX, in anyway.  Since 5.0-RELEASE
 * will have vm.stats.vm.v_forks, this will be obsolete in the future.
 */
#if __FreeBSD_version >= 400000
#define	PID_MAX		99999
#else
#include <sys/signalvar.h>
#define KERNEL
#include <sys/proc.h>
#undef KERNEL
#endif

#include <kvm.h>
#include <limits.h>
#include <paths.h>
#include <utmp.h>

extern	kvm_t	*kvmd;

static struct nlist nl[] = {
#define N_NEXTPID	0
	{ "_nextpid" },
	{ "" }
};

static int	oid_v_forks[CTL_MAXNAME + 2];
static int	oid_v_vforks[CTL_MAXNAME + 2];
static int	oid_v_rforks[CTL_MAXNAME + 2];
static size_t	oid_v_forks_len = sizeof(oid_v_forks);
static size_t	oid_v_vforks_len = sizeof(oid_v_vforks);
static size_t	oid_v_rforks_len = sizeof(oid_v_rforks);
static int	have_v_forks = 0;

static void
init_freebsd_proc()
	{
	static int	oid_name2oid[2] = { 0, 3 };
	static char	*name = "vm.stats.vm.v_forks";
	static char	*vname = "vm.stats.vm.v_vforks";
	static char	*rname = "vm.stats.vm.v_rforks";

	/* check if vm.stats.vm.v_forks is available */
	if (sysctl(oid_name2oid, 2, oid_v_forks, &oid_v_forks_len,
		   (void *)name, strlen(name)) < 0)
		return;
	if (sysctl(oid_name2oid, 2, oid_v_vforks, &oid_v_vforks_len,
		   (void *)vname, strlen(vname)) < 0)
		return;
	if (sysctl(oid_name2oid, 2, oid_v_rforks, &oid_v_rforks_len,
		   (void *)rname, strlen(rname)) < 0)
		return;
	oid_v_forks_len /= sizeof(int);
	oid_v_vforks_len /= sizeof(int);
	oid_v_rforks_len /= sizeof(int);
	++have_v_forks;
	}

static void
read_freebsd_proc()
	{
#if __FreeBSD_version >= 400000
	static int	oid_proc[] = { CTL_KERN, KERN_PROC, KERN_PROC_ALL };
#endif
	double		avenrun;
	static u_int	n_forks = 0, curpid = -1;
	u_int		n_vforks, n_rforks;
	gint		r_forks, r_vforks, r_rforks;
	gint		len;
	gint		nextpid, nforked;
	FILE		*ut;
	struct utmp	utmp;
	struct stat	sb, s;
	static time_t	utmp_mtime;
	gchar		ttybuf[MAXPATHLEN];

	if (getloadavg(&avenrun, 1) > 0)
		proc.fload = avenrun;

	if (have_v_forks)
		{
		/* We don't want to just use sysctlbyname().  Because,
                 * we call it so often. */
		len = sizeof(n_forks);
		r_forks = sysctl(oid_v_forks, oid_v_forks_len,
				 &n_forks, &len, NULL, 0);
		len = sizeof(n_vforks);
		r_vforks = sysctl(oid_v_vforks, oid_v_vforks_len,
				  &n_vforks, &len, NULL, 0);
		len = sizeof(n_rforks);
		r_rforks = sysctl(oid_v_rforks, oid_v_rforks_len,
				  &n_rforks, &len, NULL, 0);
		if (r_forks >= 0 && r_vforks >= 0 && r_rforks >= 0)
			proc.n_forks = n_forks + n_vforks + n_rforks;
		}
	else
		{
		/* workaround: Can I get total number of processes? */
		if (kvmd != NULL)
			{
			if (nl[0].n_type == 0)
				kvm_nlist(kvmd, nl);
			if (nl[0].n_type != 0 &&
			    kvm_read(kvmd, nl[N_NEXTPID].n_value,
				     (char *)&nextpid,
				     sizeof(nextpid)) == sizeof(nextpid))
				{
				if (curpid < 0)
					curpid = nextpid;
				if ((nforked = nextpid - curpid) < 0)
					n_forks += PID_MAX - 100;
				n_forks += nforked;
				curpid = nextpid;
				proc.n_forks = n_forks;
				}
			}
		}

#if __FreeBSD_version >= 400000
	if (sysctl(oid_proc, 3, NULL, &len, NULL, 0) >= 0)
		proc.n_processes = len / sizeof(struct kinfo_proc);
#else
	if (kvmd != NULL)
		kvm_getprocs(kvmd, KERN_PROC_ALL, 0, &proc.n_processes);
#endif

	if (!GK.five_second_tick || !proc.extra_info)
		return;
	if (stat(_PATH_UTMP, &s) != 0 || s.st_mtime == utmp_mtime)
		return;
	if ((ut = fopen(_PATH_UTMP, "r")) != NULL)
		{
		proc.n_users = 0;
		while (fread(&utmp, sizeof(utmp), 1, ut))
			{
			if (utmp.ut_name[0] == '\0')
				continue;
			(void)snprintf(ttybuf, sizeof(ttybuf), "%s/%s",
				       _PATH_DEV, utmp.ut_line);
			/* corrupted record */
			if (stat(ttybuf, &sb))
				continue;
			++proc.n_users;
			}
		(void)fclose(ut);
		}
	utmp_mtime = s.st_mtime;
	}
#endif


/* ------- Linux ----------------------------------------------------- */
#if defined(__linux__)
#include <utmp.h>
#include <paths.h>

#define	PROC_LOADAVG_FILE	"/proc/loadavg"

static void
read_linux_proc()
	{
	FILE			*f;
	gchar			buf[160], loadbuf[16];
	struct utmp		*ut;
	struct stat		s;
	static time_t	utmp_mtime;
	gint			n;

	if ((f = fopen(PROC_LOADAVG_FILE, "r")) != NULL)
		{
		/* sscanf(buf, "%f") might fail to convert because for some locales
		|  commas are used for decimal points.
		*/
		fgets(buf, sizeof(buf), f);
		n = sscanf(buf,"%f %*f %*f %d/%d %lu", &proc.fload,
					&proc.n_running, &proc.n_processes, &proc.n_forks);
		if (n != 4)
			{
			if (sscanf(buf,"%16s %*d.%*d %*d.%*d %d/%d %lu", loadbuf,
					&proc.n_running, &proc.n_processes, &proc.n_forks) == 4)
				proc.fload = locale_float_fix(loadbuf);
			}
		fclose(f);
		}
	if (GK.five_second_tick && proc.extra_info)
		{
		if (stat(_PATH_UTMP, &s) == 0 && s.st_mtime != utmp_mtime)
			{
			proc.n_users = 0;
			setutent();
			while ((ut = getutent()) != NULL)
				if (ut->ut_type == USER_PROCESS && ut->ut_name[0] != '\0')
					++proc.n_users;
			endutent();
			utmp_mtime = s.st_mtime;
			}
		}
	}
#endif	/* __linux__ */


/* ----- NetBSD --------------------------------------------------------- */

#if defined(__NetBSD__) || defined(__OpenBSD__)

#include <sys/proc.h>
#include <sys/sysctl.h>
#include <uvm/uvm_extern.h>
#include <kvm.h>

#include <utmp.h>

static struct nlist nl[] = {
#define X_UVM_EXP    0
   { "_uvmexp" },
   { NULL }
};

extern	kvm_t	*kvmd;

void
read_netbsd_proc()
{
   static int mib[] = { CTL_KERN, KERN_PROC, KERN_PROC_ALL };
   double avenrun;
   static time_t utmp_mtime;
   struct utmp utmp;
   struct stat s;
   struct uvmexp *uvmexp;
   int len, i;
   FILE *ut;

   if (sysctl(mib, 3, NULL, &len, NULL, 0) >= 0) {
      proc.n_processes = len / sizeof(struct kinfo_proc);
   }

   /* get name list if it is not done yet */
   if (kvmd == NULL) return;
   if (nl[0].n_type == 0) kvm_nlist(kvmd, nl);

   if (nl[0].n_type != 0) {
      uvmexp = (struct uvmexp *)nl[X_UVM_EXP].n_value;
      if (kvm_read(kvmd, (u_long)&uvmexp->forks, &i, sizeof(i)) == sizeof(i))
	 proc.n_forks = i;
   }

   if (!GK.second_tick)	/* Only one read per second */
      return;

   if (getloadavg(&avenrun, 1) > 0) proc.fload = avenrun;

   if (proc.extra_info) {
      if (stat(_PATH_UTMP, &s) == 0 && s.st_mtime != utmp_mtime) {
	 if ((ut = fopen(_PATH_UTMP, "r")) != NULL) {
	    proc.n_users = 0;
	    while (fread(&utmp, sizeof(utmp), 1, ut)) {
	       if (utmp.ut_name[0] == '\0') continue;
	       ++proc.n_users;
	    }
	    (void)fclose(ut);
	 }
	 utmp_mtime = s.st_mtime;
      }
   }
}

#endif /* __NetBSD__ || __OpenBSD__ */


/* ----- Others ------------------------------------------------------ */
#if defined(USE_LIBGTOP)

#include <glibtop/loadavg.h>

static void
read_glibtop_proc()
	{
	glibtop_loadavg		glt_loadavg;

	/* If this is expensive, may want to second_tick it
	*/
	glibtop_get_loadavg (&glt_loadavg);
	proc.fload = (float) glt_loadavg.loadavg[0];

	/* Not all systems fill in the rest of these
	*/
	proc.n_running = (gint) glt_loadavg.nr_running;
	proc.n_processes = (gint) glt_loadavg.nr_tasks;
	proc.n_forks = (gint) glt_loadavg.last_pid;

	proc.n_users = -1;
	}
#endif


/* ----- Pick a system interface ----------------------------------------- */
static gint
setup_proc_interface()
	{
#if defined(__FreeBSD__)
	init_freebsd_proc();
	read_system_proc_info = read_freebsd_proc;
#elif defined(__linux__)
	read_system_proc_info = read_linux_proc;
#elif defined(__NetBSD__) || defined(__OpenBSD__)
	read_system_proc_info = read_netbsd_proc;
#else
	read_system_proc_info = read_glibtop_proc;
#endif
	return TRUE;
	}

/* ======================================================================== */

static Launcher	proc_launch;

static gint		load_resolution;
static gint		ascent;
static gint		style_id;
static gint		mapped_sensors;
static gint		sensor_separate_mode;

static void
draw_proc_extra()
	{
	TextStyle	*ts, *ts_alt;
	gint		x, x1, y;
	gchar		buf[32];

	if (proc.chart == NULL)
		return;
	y = 4;
	x1 = 4;
	ts = gkrellm_chart_textstyle(style_id);
	ts_alt = gkrellm_chart_alt_textstyle(style_id);
	if (ascent == 0)
		ascent = gdk_char_height(ts->font, '8');
	if (proc.n_processes > 0)
		{
		y += ascent;
		snprintf(buf, sizeof(buf), "%d", proc.n_processes);
		x1 = gkrellm_draw_chart_label(proc.chart, ts, x1, y, buf);
		gkrellm_draw_chart_label(proc.chart, ts_alt, x1 + 2, y, _("procs"));
		y += 2;
		}
	if (proc.n_users > 0)
		{
		y += ascent;
		snprintf(buf, sizeof(buf), "%d", proc.n_users);
		x = gkrellm_draw_chart_label(proc.chart, ts, 4, y, buf);
		if (x < x1)
			x = x1;
		gkrellm_draw_chart_label(proc.chart, ts_alt, x + 2, y, _("users"));
		}
	}


static void
refresh_proc_chart(Chart *cp)
	{
	if (proc.enabled)
		{
		draw_proc_chart(cp);
		}
	if (proc.extra_info)
		draw_proc_extra();
	cp->need_redraw = FALSE;
	}

static void
draw_sensor_decals()
	{
	Panel	*p		= proc.chart->panel;
	gchar	*name	= "mb";
	gchar	units;
	gfloat	t, f;
	gint	toggle;

	if (   sensor_separate_mode
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& (mapped_sensors & SENSOR_FAN)
	   )
		{
		gkrellm_sensor_read_temperature(name, &t, &units);
		gkrellm_sensor_draw_temperature_decal(p, proc.sensor_decal, t, units);
		gkrellm_sensor_read_fan(name, &f);
		gkrellm_sensor_draw_fan_decal(p, proc.fan_decal, f);
		}
	else
		{
		toggle = time_now & 2;
		if (   (mapped_sensors & SENSOR_FAN)
			&& (toggle || !(mapped_sensors & SENSOR_TEMPERATURE))
		   )
			{
			gkrellm_sensor_read_fan(name, &f);
			gkrellm_sensor_draw_fan_decal(p, proc.sensor_decal, f);
			}
		else if (   (mapped_sensors & SENSOR_TEMPERATURE)
				 && (!toggle || !(mapped_sensors & SENSOR_FAN))
				)
			{
			gkrellm_sensor_read_temperature(name, &t, &units);
			gkrellm_sensor_draw_temperature_decal(p, proc.sensor_decal, t,
					units);
			}
		}
	}

static void
update_proc()
	{
	Chart	*cp;
	Panel	*p;
	gint	load;

	if (!proc.enabled)
		return;
	(*read_system_proc_info)();

	cp = proc.chart;
	p = cp->panel;
	gkrellm_update_krell(p, KRELL(p), proc.n_forks);
	gkrellm_draw_layers(p);

	if (GK.second_tick)
		{
		cp->prevOut = 0;

		/* Scale the forks number.  See setup_proc_scaling().
		*/
		load = (int) (100.0 * proc.fload);
		store_proc_chart_data(cp, load, cp->private * proc.n_forks);
		refresh_proc_chart(cp);
		}
	if (   (GK.two_second_tick && !sensor_separate_mode)
		|| (GK.five_second_tick && sensor_separate_mode)
	   )
		draw_sensor_decals();

	if (cp->need_redraw)
		refresh_proc_chart(cp);
	}


static gint
proc_expose_event(GtkWidget *widget, GdkEventExpose *ev)
	{
	Chart		*cp		= proc.chart;
	GdkPixmap	*pixmap	= NULL;

	if (cp)
		{
		if (widget == cp->drawing_area)
			pixmap = cp->pixmap;
		else if (widget == cp->panel->drawing_area)
			pixmap = cp->panel->pixmap;
		}
	if (pixmap)
		gdk_draw_pixmap(widget->window, GK.draw1_GC, pixmap,
				  ev->area.x, ev->area.y, ev->area.x, ev->area.y,
				  ev->area.width, ev->area.height);
	return FALSE;
	}


static gint
cb_proc_extra(GtkWidget *widget, GdkEventButton *event)
	{
	if (event->button == 1)
		{
		proc.extra_info = 1 - proc.extra_info;
		gkrellm_config_modified();
		}
	refresh_proc_chart(proc.chart);
	return TRUE;
	}

static void
setup_proc_scaling()
	{
	Chart	*cp		= proc.chart;
	gint	grids, fork_factor, i;

	if (cp == NULL)
		return;
	grids = UC.fixed_scale ? UC.fixed_scale : FULL_SCALE_GRIDS;
	cp->scale_min = load_resolution;

	/* Since scale_min is set for load, set krell_full_scale explicitely
	|  to get what I want, which is 10 forks full scale.
	|  Use private field to scale forks/sec to get hardwired 50 forks/sec
	|  max on the charts.
	*/
	KRELL(cp->panel)->full_scale = 10;
	fork_factor = grids * load_resolution / 50;

	/* If load resolution changed, scale all fork data to keep the
	|  constant 50 forks/sec.
	*/
	if (fork_factor != cp->private && cp->private > 0)
		{
		for (i = 0; i < cp->w; ++i)
			cp->pDataIn[i] = cp->pDataIn[i] * fork_factor / cp->private;
		cp->prevIn = cp->prevIn * fork_factor / cp->private;
		}
	cp->private = fork_factor;
	cp->scale_max = 0;
	cp->maxval = 0;
	}

static void
destroy_proc_monitor(void)
	{
	Chart	*cp		= proc.chart;

	if (proc_launch.button)
		gkrellm_destroy_button(proc_launch.button);
	proc_launch.button = NULL;

	g_free(cp->name);
	gkrellm_monitor_height_adjust( - proc.height);
	gkrellm_destroy_panel(cp->panel);
    proc_launch.tooltip = NULL;
	g_free(cp->panel);
	gkrellm_destroy_chart(cp);
	g_free(cp);
	proc.chart = NULL;
	proc.enabled = FALSE;
	}

  /* Next routine is same as in cpu.c - perhaps should make into one?*/
  /* How to decide when to make the sensor_decal and fan_decal visible.
  |  The sensor_decal can show temp values, fan values, or alternating
  |  temp/fan values.  The fan_decal only shows fan values when non
  |  alternating mode is selected. The sensors are mapped in sensors.c
  |
  |   Sensor and fan decal display truth table:
  |   |-----decals visible----||--sensors mapped--|   separate  |
  |	  |sensor_decal  fan_decal||   temp     fan   |    mode     |
  |   |-----------------------||--------------------------------|
  |   |     0           0     ||     0       0          0       |
  |   |     1           0     ||     1       0          0       |
  |   |     1           0     ||     0       1          0       |
  |   |     1           0     ||     1       1          0       |
  |   |     0           0     ||     0       0          1       |
  |   |     1           0     ||     1       0          1       |
  |   |     1           0     ||     0       1          1       |
  |   |     1           1     ||     1       1          1       |
  |   |----------------------------------------------------------
  */
static gint
adjust_sensors_display(gint force)
	{
	Panel	*p;
	Decal	*ds, *df;
	gint	position	= 0;

	ds = proc.sensor_decal;
	df = proc.fan_decal;
	if (!ds || !df)
		return FALSE;
	/* The test for state change is decal state vs success at reading
	|  a temperature.
	*/
	p = proc.chart->panel;
	mapped_sensors = 0;
	if (get_mapped_sensor("mb", SENSOR_TEMPERATURE) || GK.demo)
		mapped_sensors |= SENSOR_TEMPERATURE;
	if (get_mapped_sensor("mb", SENSOR_FAN) || GK.demo)
		mapped_sensors |= SENSOR_FAN;

	if (mapped_sensors & (SENSOR_TEMPERATURE | SENSOR_FAN))
		{
		if (! gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_visible(p, ds);
		position = 0;
		}
	else
		{
		if (gkrellm_is_decal_visible(ds) || force)
			gkrellm_make_decal_invisible(p, ds);
		position = proc.save_label_position;
		}
	if (   (mapped_sensors & SENSOR_FAN)
		&& (mapped_sensors & SENSOR_TEMPERATURE)
		&& sensor_separate_mode
	   )
		{
		if (! gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_visible(p, df);
		position = -1;
		}
	else
		{
		if (gkrellm_is_decal_visible(df) || force)
			gkrellm_make_decal_invisible(p, df);
		}
	if (position != p->label->position || force)
		{
		if (proc.save_label_position >= 0)	/* Reassign position only if the */
			p->label->position = position;	/* original label was visible.   */
		gkrellm_draw_panel_label(p, gkrellm_bg_panel_image(style_id));
		draw_sensor_decals();
		gkrellm_draw_layers(p);
		return TRUE;
		}
	return FALSE;
	}

  /* Proc monitor shows processes (forks) per second and load average.
  |  The scaling goes like this.  Load is read from /proc/loadavg as a
  |  float ranging from 0 - n where n may be 1, 2, 3, ...  Basically,
  |  something small.  So, I scale load by 100 and set scale_min to
  |  100.  Now grid lines have units of 1.
  */
static void
create_proc_monitor(GtkWidget *vbox, gint first_create)
	{
	Chart	*cp;
	Panel	*p;
	Style	*style;

	if (first_create)
		{
		proc.chart = gkrellm_chart_new0();
		proc.chart->panel = gkrellm_panel_new0();
		proc.chart->name = g_strdup(_("Proc"));
		}
	else
		{
		gkrellm_destroy_decal_list(proc.chart->panel);
		gkrellm_destroy_krell_list(proc.chart->panel);
		}
	cp = proc.chart;
	p = cp->panel;
	cp->scale_min = 100;		/* For load */

	style = gkrellm_panel_style(style_id);
	gkrellm_create_krell(p, gkrellm_krell_panel_image(style_id), style);

	cp->h = UC.chart_height[MON_PROC];

	gkrellm_create_chart(proc.vbox, cp, style_id);

	p->textstyle = gkrellm_panel_textstyle(style_id);

	/* I put motherboard temp on Proc panel (if temperature sensors found)
	*/
	sensor_create_decals(p, style_id,
					&proc.sensor_decal, &proc.fan_decal);

	gkrellm_configure_panel(p, cp->name, style);

	gkrellm_create_panel(proc.vbox, p, gkrellm_bg_panel_image(style_id));
	proc.height = cp->h + p->h;
	gkrellm_monitor_height_adjust(proc.height);

	proc.save_label_position = p->label->position;
	if (proc.sensor_decal)
		adjust_sensors_display(TRUE);

	gkrellm_alloc_chart_data(cp);
	cp->type = 1;
	setup_proc_scaling();

	if (first_create)
		{
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area), "expose_event",
					(GtkSignalFunc) proc_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(p->drawing_area), "expose_event",
					(GtkSignalFunc) proc_expose_event, NULL);
		gtk_signal_connect(GTK_OBJECT(cp->drawing_area),"button_press_event",
					(GtkSignalFunc) cb_proc_extra, NULL);
		}
	else
		refresh_proc_chart(cp);
	gkrellm_setup_launcher(p, &proc_launch, CHART_PANEL_TYPE, 4);
	}

static void
create_proc(GtkWidget *vbox, gint first_create)
	{
	if (first_create)
		{
		proc.vbox = gtk_vbox_new(FALSE, 0);
		gtk_container_add(GTK_CONTAINER(vbox), proc.vbox);
		if (proc.enabled)
			gtk_widget_show(proc.vbox);
		}
	ascent = 0;
	if (proc.enabled)
		create_proc_monitor(proc.vbox, first_create);
	}



#define	PROC_CONFIG_KEYWORD	"proc"

static void
save_proc_config(FILE *f)
	{
	fprintf(f, "%s resolution %d\n", PROC_CONFIG_KEYWORD, load_resolution);
	fprintf(f, "%s enable %d %d %d %d\n", PROC_CONFIG_KEYWORD,
				proc.enabled, UC.proc_load_bar_graph,
				UC.proc_clip_processes, proc.extra_info);
	fprintf(f, "%s launch %s\n", PROC_CONFIG_KEYWORD, proc_launch.command);
	fprintf(f, "%s tooltip_comment %s\n", PROC_CONFIG_KEYWORD,
				proc_launch.tooltip_comment);
	fprintf(f, "%s sensor_mode %d\n", PROC_CONFIG_KEYWORD,
				sensor_separate_mode);
	}

static void
load_proc_config(gchar *arg)
	{
	gchar	config[32], item[CFG_BUFSIZE];
	gint	n;

	n = sscanf(arg, "%31s %[^\n]", config, item);
	if (n == 2)
		{
		if (strcmp(config, "resolution") == 0)
			sscanf(item, "%d\n", &load_resolution);
		else if (strcmp(config, "enable") == 0)
			sscanf(item, "%d %d %d %d\n", &proc.enabled,
					&UC.proc_load_bar_graph,
					&UC.proc_clip_processes, &proc.extra_info);
		else if (strcmp(config, "launch") == 0)
			proc_launch.command = g_strdup(item);
		else if (strcmp(config, "tooltip_comment") == 0)
			proc_launch.tooltip_comment = g_strdup(item);
		else if (strcmp(config, "sensor_mode") == 0)
			sscanf(item, "%d\n", &sensor_separate_mode);
		}
	}

/* ---------------------------------------------------------------------- */
static GtkWidget	*load_spin_button;

static GtkWidget	*bar_graph_button,
					*clip_proc_button,
					*proc_enable_button,
					*sensor_separate_button;
static GtkWidget	*proc_launch_entry,
					*proc_tooltip_entry;

static void
apply_proc_config()
	{
	GtkSpinButton	*spin;
	gfloat			f;
	gint			pe;

	spin = GTK_SPIN_BUTTON(load_spin_button);
	f = gtk_spin_button_get_value_as_float(spin);
	load_resolution = (gint) (f * 100.0);

	UC.proc_load_bar_graph = GTK_TOGGLE_BUTTON(bar_graph_button)->active;
	UC.proc_clip_processes = GTK_TOGGLE_BUTTON(clip_proc_button)->active;
	if (sensor_separate_button)
		sensor_separate_mode =
				GTK_TOGGLE_BUTTON(sensor_separate_button)->active;

	pe = GTK_TOGGLE_BUTTON(proc_enable_button)->active;
	if (pe && ! proc.enabled)
		{
		gtk_widget_show(proc.vbox);
		create_proc_monitor(proc.vbox, TRUE);
		}
	else if (! pe && proc.enabled)
		{
		gtk_widget_hide(proc.vbox);
		destroy_proc_monitor();
		}
	proc.enabled = pe;
	if (proc.chart)
		setup_proc_scaling();

	if (proc.enabled)
		{
		gkrellm_apply_launcher(&proc_launch_entry, &proc_tooltip_entry,
					proc.chart->panel, &proc_launch, gkrellm_launch_button_cb);
		if (adjust_sensors_display(FALSE) && proc_launch.button)
			{
			gkrellm_destroy_button(proc_launch.button);
			proc_launch.button = 
				gkrellm_put_label_in_panel_button(proc.chart->panel,
					gkrellm_launch_button_cb, &proc_launch, proc_launch.pad);
			}
		}
	}

static void
create_proc_tab(GtkWidget *tab_vbox)
	{
	GtkWidget		*table;

	gkrellm_check_button(tab_vbox, &proc_enable_button, proc.enabled, TRUE, 0,
			_("Enable Proc chart"));

	gkrellm_check_button(tab_vbox, &bar_graph_button, UC.proc_load_bar_graph,
			FALSE, 2, _("Draw load data as solid bars instead of a line"));
	gkrellm_check_button(tab_vbox, &clip_proc_button, UC.proc_clip_processes,
			FALSE, 2,
			_("Clip process fork data (chart scaling uses load graph only)"));
	if (gkrellm_sensors_available())
		gkrellm_check_button(tab_vbox, &sensor_separate_button,
					sensor_separate_mode, FALSE, 0,
		_("Draw fan and temperature values separately (not alternating)."));

	gkrellm_spin_button(tab_vbox, &load_spin_button,
			(gfloat) load_resolution / 100, 0.5, 5.0, 0.5, 0.5, 1, 50,
			NULL, NULL, FALSE,
			_("Load resolution in average process load per minute per grid"));

	table = gkrellm_launcher_table_new(tab_vbox, 1);
	gkrellm_config_launcher(table, 0,  &proc_launch_entry, &proc_tooltip_entry,
			_("Proc"), &proc_launch);
	}

static Monitor	monitor_proc =
	{
	N_("Proc"),				/* Name, for config tab.	*/
	MON_PROC,			/* Id,  0 if a plugin		*/
	create_proc,		/* The create function		*/
	update_proc,		/* The update function		*/
	create_proc_tab,	/* The config tab create function	*/
	apply_proc_config,	/* Apply the config function		*/

	save_proc_config,	/* Save user conifg			*/
	load_proc_config,	/* Load user config			*/
	PROC_CONFIG_KEYWORD, /* config keyword			*/

	NULL,				/* Undef 2	*/
	NULL,				/* Undef 1	*/
	NULL,				/* Undef 0	*/

	0,					/* insert_before_id - place plugin before this mon */

	NULL,				/* Handle if a plugin, filled in by GKrellM		*/
	NULL				/* path if a plugin, filled in by GKrellM		*/
	};

Monitor *
init_proc_monitor(void)
	{
	monitor_proc.name = _(monitor_proc.name);
	proc.enabled = TRUE;
	load_resolution = 100;
	style_id = gkrellm_add_chart_style(&monitor_proc, PROC_STYLE_NAME);
	if (setup_proc_interface())
		return &monitor_proc;
	return NULL;
	}

