/*  $Id$
    Gengameng_LoadCompiledXPM.cc - Load a compiled XPM into an SDL surface.

    gengameng - Generic 2D Game Engine library
    Copyright (C) 2001 Pierre Sarrazin <sarrazip@iname.com>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
    02111-1307, USA.
*/

#include "SDL.h"
#include "SDL_image.h"
#include "SDL_rwops.h"

#include <stdlib.h>
#include <errno.h>


typedef struct {
    const char **compiled_xpm;
    size_t num_lines;  /* number of strings in compiled_xpm */
    size_t cur_line_index;  /* index of current line in compiled_xpm */
    int filler_no;  /* index in Gengameng_xpm_fillers */
    const char *reader;  /* pointer to the next character */
    int in_filler;  /* boolean: currently using filler (1) or XPM line (0) */
} Gengameng_CompiledXPMDesc;


static const char *Gengameng_xpm_fillers[3] = {
    "static char * bogus_name_xpm[] = {\n\"",
    "\",\n\"",
    "\"};\n"
};


static void Gengameng_CompiledXPMDesc_init(
		Gengameng_CompiledXPMDesc *desc, const char **compiled_xpm)
{
    char *rest;
    long width, height, num_colors;

    if (desc == NULL)
	return;
    desc->compiled_xpm = compiled_xpm;
    desc->num_lines = 0;  /* defined but invalid number */
    desc->cur_line_index = 0;
    desc->filler_no = 0;
    desc->reader = Gengameng_xpm_fillers[desc->filler_no];
    desc->in_filler = 1;


    /* Try to find the number of strings in compiled_xpm: */

    if (compiled_xpm == NULL || compiled_xpm[0] == NULL)
	return;

    errno = 0;
    width = strtol(compiled_xpm[0], &rest, 10);
    if (errno == ERANGE)
	return;
    height = strtol(rest, &rest, 10);
    if (errno == ERANGE || height <= 0)
	return;
    num_colors = strtol(rest, &rest, 10);
    if (errno == ERANGE || num_colors <= 0)
	return;

    desc->num_lines = 1 + num_colors + height;
}


static int Gengameng_CompiledXPMDesc_seek(
			struct SDL_RWops *context, int offset, int whence)
{
    return -1;  /* not needed */
}


static int Gengameng_CompiledXPMDesc_getc(Gengameng_CompiledXPMDesc *desc)
{
    if (desc == NULL)
	return EOF;
    if (*desc->reader != '\0')  /* if current string not finished */
	return *desc->reader++;
    
    if (desc->in_filler)
    {
	if (desc->filler_no == 2)  /* if this was the last filler string */
	    return EOF;

	/* start reading an XPM line */
	desc->in_filler = 0;
	desc->reader = desc->compiled_xpm[desc->cur_line_index];
	return *desc->reader++;
    }

    /* current string was XPM line; start reading a filler string */
    desc->in_filler = 1;
    if (desc->cur_line_index == desc->num_lines - 1)  /* if last XPM line */
	desc->filler_no = 2;  /* start reading ending filler string */
    else
    {
	desc->filler_no = 1;
	desc->cur_line_index++;
    }

    desc->reader = Gengameng_xpm_fillers[desc->filler_no];
    return *desc->reader++;
}


static int Gengameng_CompiledXPMDesc_read(
		    struct SDL_RWops *context, void *ptr, int size, int maxnum)
{
    unsigned char *buffer = (unsigned char *) ptr;
    int i;
    Gengameng_CompiledXPMDesc *desc;

    if (context == NULL || ptr == NULL) {
	IMG_SetError("Invalid parameters for Gengameng_CompiledXPMDesc_read");
	return -1;
    }
    if (size == 0 || maxnum == 0)  /* if nothing to do */
	return 0;  /* success */

    desc = (Gengameng_CompiledXPMDesc *) context->hidden.unknown.data1;
    if (desc == NULL) {
	IMG_SetError("Invalid data1 for Gengameng_CompiledXPMDesc_read");
	return -1;
    }
    for (i = 0; i < size * maxnum; i++)
    {
	int c = Gengameng_CompiledXPMDesc_getc(desc);
	if (c == EOF)
	    break;
	buffer[i] = (unsigned char) c;
    }
    return i / size;
}


static int Gengameng_CompiledXPMDesc_write(
		struct SDL_RWops *context, const void *ptr, int size, int num)
{
    return -1;  /* not needed */
}


static int Gengameng_CompiledXPMDesc_close(struct SDL_RWops *context)
{
    return 0;
	/* we let Gengameng_LoadCompiledXPM() free the allocated SDL_RWops */
}


SDL_Surface *Gengameng_LoadCompiledXPM(char **compiled_xpm)
{
    SDL_RWops *rwops;
    Gengameng_CompiledXPMDesc desc;
    SDL_Surface *image;

    if (compiled_xpm == NULL) {
	IMG_SetError("No image");
	return NULL;
    }

    Gengameng_CompiledXPMDesc_init(&desc, (const char **) compiled_xpm);

    rwops = SDL_AllocRW();
    if (rwops == NULL) {
	IMG_SetError("SDL_AllocRW failed");
	return NULL;
    }
    rwops->hidden.unknown.data1 = &desc;
    rwops->seek  = Gengameng_CompiledXPMDesc_seek;
    rwops->read  = Gengameng_CompiledXPMDesc_read;
    rwops->write = Gengameng_CompiledXPMDesc_write;
    rwops->close = Gengameng_CompiledXPMDesc_close;

    image = IMG_LoadXPM_RW(rwops);

    SDL_FreeRW(rwops);

    return image;
}
