/*
 * Copyright (c) 2005 Aleksander Piotrowski <aleksander.piotrowski@nic.com.pl>
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "internal.h"

#include "account.h"
#include "accountopt.h"
#include "debug.h" 
#include "notify.h" 
#include "request.h"
#include "server.h"
#include "util.h"
#include "version.h"

#include "icb.h"

static char  icb_input_buf[ICB_BUFSIZE+1];
static char *icb_input_pos = icb_input_buf;
static int   icb_input_fill = 0;

/* XXX Maybe this could be used?
 * gconv = gaim_find_conversation_with_account(zt->name, gc->account);
 */
static GaimConversation *
icb_get_current_group(GaimAccount *account)
{
	GList            *list = NULL;
	GaimConversation *conv = NULL;

	for (list = gaim_get_conversations(); list != NULL; list = list->next) {
		conv = (GaimConversation *) list->data;
               	if (conv->type == GAIM_CONV_CHAT && account == gaim_conversation_get_account(conv)) {
			return conv;
		}
	}

	return NULL;
}

#ifdef OLD_GAIM
static GList *
icb_status_types(GaimConnection *gc)
#else
static GList *
icb_status_types(GaimAccount *account)
#endif
{
	return NULL;
}

void
icb_dump_packet(IcbPacket *packet)
{
	int i;

	gaim_debug_info("icb", "-> icb_dump_packet\n");

	gaim_debug_info("icb", "length:  %d\n", packet->length);
	gaim_debug_info("icb", "command: %c\n", packet->command);
	for (i = 0; i < packet->nof; i++) {
		gaim_debug_info("icb", "field %d: %d \"%s\"\n", i,
		                strlen(packet->fields[i]), packet->fields[i]);
	} 

	gaim_debug_info("icb", "<- icb_dump_packet\n");
}

void
icb_free_packet(IcbPacket **packet)
{
	IcbPacket *tmp = *packet;
	int        i;

	for (i = 0; i < tmp->nof; i++) {
		free(tmp->fields[i]);
	} 
	free(tmp);
	*packet = NULL;
}

void
icb_dump_buf(char *buf, int len)
{
	int i;

	char *out = (char *) calloc(1, len+1);
	if (!out) {
		return;
	}

	for (i = 0; i < len; i++) {
		*(out+i) = *(buf+i) == ICB_SEPARATOR ? ',' : *(buf+i);
	}	

	/* Replace first char of buf (size) with X as it could have
	 * a non-printable value
	 */
	*out = 'X';

	gaim_debug_info("icb", "len= %d, buf=\"%s\"\n", len, out);
	free(out);
}

IcbPacket *
icb_parse_buf()
{
	char          *separator, *tmpbuf;
	unsigned char *size;
	IcbPacket     *packet = NULL;

	gaim_debug_info("icb", "-> icb_parse_buf\n");

	/* There has to be at least two chars in buffer: size and command */
	if (icb_input_fill < 2) {
		gaim_debug_info("icb", "Buffer is to short.  Only %d char(s)\n", icb_input_buf);
		return NULL;
	}

	icb_dump_buf(icb_input_pos, icb_input_fill);

	/* Check if there is a full packet in buffer.
	 * First char is a packet size, so let's check if
	 * buffer size isn't to small.
	 */
	if (*icb_input_pos > icb_input_fill) {
		gaim_debug_info("icb", "Looks like buffer is not filled with full packet\n");
		return NULL;
	}

	packet = (IcbPacket *) calloc(1, sizeof(IcbPacket));
	if (!packet) {
		gaim_debug_info("icb", "calloc(IcbPacket)\n");
		gaim_debug_info("icb", "<- icb_parse_buf\n");
		return NULL;
	}

	size = icb_input_pos; /* Size of the package */
	tmpbuf = icb_input_pos+1; /* Command sent in packet */

	packet->nof = 0;
	packet->fields = (char **) calloc(1, ICB_MAX_NO_FIELDS*sizeof(char *));
	packet->length = *size;
	packet->command = *tmpbuf++;

	separator = tmpbuf;
	while (tmpbuf - icb_input_pos < packet->length + 1) {
		/* new field */
		if ((*tmpbuf == ICB_SEPARATOR || *tmpbuf == '\0')
		    && separator != tmpbuf) {
			*tmpbuf = '\0';
			packet->fields[packet->nof++] = strdup(separator);
			separator = tmpbuf+1;
		}
		tmpbuf++;
	}

	icb_input_pos = tmpbuf;
	icb_input_fill -= packet->length + 1;
	
	gaim_debug_info("icb", "<- icb_parse_buf\n");

	return packet;
}

void
icb_login_cb(gpointer data, gint source, GaimInputCondition cond)
{
	GaimConnection *gc = data;
	IcbSession     *icb = gc->proto_data;

	gaim_debug(GAIM_DEBUG_INFO, "icb", "-> icb_login_cb\n");

	if (source < 0) {
		gaim_connection_error(gc, _("Couldn't connect to host"));
		return;
	}
	
	icb->fd = source;

	gaim_connection_update_progress(gc, _("Reading protocol packet"), 1,
	                                ICB_CONNECT_STEPS);

	icb_input_cb(data, source, cond);

	gaim_debug_info("icb", "<- icb_login_cb\n");
}

/* ICB packet is:
 *   size -> one byte
 *   command -> one byte
 *   data -> 253 bytes
 *     -> field 0
 *     -> seperator
 *     -> field 1
 *     -> [and so on]
 *     -> NUL
 */
int
icb_send(IcbSession *icb, char command, int params, ...)
{
	const char *field;
	char        packet[ICB_PACKET_SIZE];
	char       *pos = packet;
	va_list     arg;
	int         fieldlen, ret, size;

	gaim_debug_info("icb", "-> icb_send\n");
	if (icb->fd < 0) {
		gaim_debug_info("icb", "<- icb_send: icb->fd < 0");
		return -1;
	}

	bzero(packet, sizeof(packet));

	// size of packet (filled with dummy value for now) 
	*pos++ = '-';

	// command	
	*pos = command;

	// amount of date in buffer
	size = 2;

	// fields
	va_start(arg, params);
	while (params-- > 0) {
		field = va_arg(arg, const char *);
		fieldlen = strlen(field);

		// -1 to leave space for NUL at the end of buffer
		if (fieldlen + size > ICB_PACKET_SIZE -1) {
			gaim_debug_info("icb", "<- icb_send: too much data to write");
			va_end(arg);
			return -1;
		}

		// field data
		strncpy(&packet[size], field, fieldlen);
		size += fieldlen;

		// separator
		if (params != 0) {
			packet[size] = ICB_SEPARATOR;
			size++;
		}
	}
	va_end(arg);

	// packet size
	*packet = size;

	icb_dump_buf(packet, strlen(packet));

	ret = write(icb->fd, packet, size + 1);
	if (ret < 0) {
		gaim_debug_info("icb", "write(): %d, %s\n", errno, strerror(errno));
		gaim_connection_error(gaim_account_get_connection(icb->account),
			_("Server has disconnected"));
	}

	gaim_debug_info("icb", "<- icb_send %d byte(s)\n", ret);

	return ret;
}

static void
icb_keepalive(GaimConnection *gc)
{
	gaim_debug_info("icb", "icb_keepalive()");
}

static void
#ifdef OLD_GAIM
icb_login(GaimAccount *account)
#else
icb_login(GaimAccount *account, GaimStatus *status)
#endif
{
	GaimConnection  *gc;
	IcbSession      *icb;
	int              err;
	const char      *user;
	char           **userparts;

	gaim_debug_info("icb", "-> icb_login\n");

	gc = gaim_account_get_connection(account);
	gc->flags |= GAIM_CONNECTION_NO_NEWLINES;

	gc->proto_data = icb= g_new0(IcbSession, 1);

	icb->account = account;

	bzero(icb_input_buf, sizeof(icb_input_buf));
	icb_input_pos = icb_input_buf;
	icb_input_fill = 0;

	user = gaim_account_get_username(account);
	userparts = g_strsplit(user, "@", 2);
	gaim_connection_set_display_name(gc, userparts[0]);
	icb->server = g_strdup(userparts[1]);
	icb->user = g_strdup(userparts[0]);
	g_strfreev(userparts);

	icb->port = gaim_account_get_int(account, "port", ICB_DEFAULT_PORT);

	gaim_connection_update_progress(gc, _("Connecting"), 0,
	                                ICB_CONNECT_STEPS);

	err = gaim_proxy_connect(account, icb->server, icb->port, icb_login_cb,
	                         gc);

	gaim_debug_info("icb", "<- icb_login\n");

	if (err || !account->gc) {
		gaim_connection_error(gc, _("Couldn't create socket"));
		return;
	}
}

void
icb_input_cb(gpointer data, gint source, GaimInputCondition cond)
{
	GaimConnection *gc = data;
	IcbSession     *icb = gc->proto_data;
	IcbPacket      *packet = NULL;

	int             len, ret;
	static int      chat_id = 1;
	char           *tmp;

	gaim_debug_misc("icb", "-> icb_input_cb: fd=%d\n", icb->fd);

	gaim_debug_misc("icb", "icb_input_pos - icb_input_buf=%d\n", icb_input_pos - icb_input_buf);
	gaim_debug_misc("icb", "icb_input_fill=%d\n", icb_input_fill);

	if (icb->fd < 0) {
		gaim_debug_info("icb", "icb->fd < 0");
		return;
	}

	len = read(icb->fd, icb_input_pos + icb_input_fill,
	          ICB_BUFSIZE - (icb_input_pos - icb_input_buf + icb_input_fill));
	gaim_debug_info("icb", "Read() got %d chars\n", len);
	icb_input_fill += len;
	gaim_debug_info("icb", "Now buffer is filled with %d char(s)\n", icb_input_fill);
	
	if (len < 0) {
		gaim_connection_error(gc, _("Read error"));
		return;
	} else if (len == 0) {
		gaim_connection_error(gc, _("Server has disconnected"));
		return;
	}

	while (icb_input_fill > 0 && (packet = icb_parse_buf()) != NULL) {
		icb_dump_packet(packet);
		switch (packet->command) {
			case ICB_CMD_PROTO_VERSION:
				gaim_connection_update_progress(gc,
					_("Sending login information"), 2,
	                        	ICB_CONNECT_STEPS);
				if (gc->account->password &&
				    *gc->account->password) {
					ret = icb_send(icb, ICB_CMD_LOGIN, 5,
				        	icb->user, icb->user, "1",
						"login", gc->account->password);
				} else {
					ret = icb_send(icb, ICB_CMD_LOGIN, 4,
				               icb->user, icb->user,
				               "1", "login");
				}
				if (ret < 0) {
					gaim_connection_error(gc,
						_("Error sending login information"));
					icb_free_packet(&packet);
					return;
				}

				gc->inpa = gaim_input_add(icb->fd,
					GAIM_INPUT_READ, icb_input_cb, gc);

				break;
			case ICB_CMD_LOGIN:
				gaim_connection_set_state(gc, GAIM_CONNECTED);
				break;
			case ICB_CMD_ERROR:
				if (packet->nof == 1) {
					gaim_notify_warning(gc, NULL, _("Error message from server"), packet->fields[0]);
				}
				break;
			case ICB_CMD_EXIT:
				gaim_connection_error(gc, "Received Exit packet. Closing connection.");
				break;
			case ICB_CMD_PERSONAL_MSG:
				serv_got_im(gc, packet->fields[0], packet->nof == 1 ? "" : packet->fields[1], 0, time(NULL));
				break;
			case ICB_CMD_OPEN_MSG:
				if (packet->nof == 2) {
					GaimConversation *conv = NULL;
					
					conv = icb_get_current_group(gc->account);
					/* User has closed conversation, so let's reopen conversation window,
					 * put incomming message there and ask server about group members.
					 * We do this here as it seems to be impossible to remove `Close conversation'
					 * button.
					 */
					if (!conv) {
						serv_got_joined_chat(gc, chat_id++, icb->group);
						conv = icb_get_current_group(gc->account);
						icb_send(icb, ICB_CMD_COMMAND, 2, "w", ".");
					}

					serv_got_chat_in(gc, gaim_conv_chat_get_id(GAIM_CONV_CHAT(conv)),
					         packet->fields[0], 0, packet->fields[1], time(NULL));
				}
				break;
			case ICB_CMD_STATUS_MSG:
				/* Just joined a group */
				if (packet->nof == 2 && 
				    strncmp(packet->fields[1], ICB_STAT_JOIN, ICB_STAT_JOIN_LEN) == 0) {
					GaimConversation *conv = NULL;
					char              group[ICB_PACKET_SIZE], *name_end, *name_start;

					/* Looking for a group name */
					bzero(group, sizeof(group));

					/* +1 to skip space after ICB_STAT_JOIN */
					name_start = packet->fields[1] + ICB_STAT_JOIN_LEN + 1;
					name_end = name_start;
					while (((name_end - packet->fields[1]) < strlen(packet->fields[1])) && *name_end != ' ') {
						gaim_debug_info("icb", "end=%c\n", *name_end);
						name_end++;
					}
					strncpy(group, packet->fields[1] + ICB_STAT_JOIN_LEN + 1,
					        name_end - name_start);

					/* Kill other chats that are using this account */
					/* You can only join one group on ICB */
					conv = icb_get_current_group(gc->account);
					if (conv) {
						gaim_conversation_destroy(conv);
						free(icb->group);
					}

					/* Joining new group and getting user list */
					icb->group = strdup(group);
					serv_got_joined_chat(gc, chat_id++, group);
					icb_send(icb, ICB_CMD_COMMAND, 2, "w", ".");
				/* Somebody has joined current group */
				} else if (packet->nof == 2 &&
				           ((strncmp(packet->fields[0], ICB_STAT_ARRIVE, ICB_STAT_ARRIVE_LEN) == 0)
					   || (strncmp(packet->fields[0], ICB_STAT_SIGNON, ICB_STAT_SIGNON_LEN) == 0))) {
					GaimConversation *conv = NULL;
					char              user[ICB_PACKET_SIZE], *name_end, *name_start;
					int               len;

					/* Get current group */
					conv = icb_get_current_group(gc->account);
					if (!conv)
						break;
					
					/* Looking for a group name */
					bzero(user, sizeof(user));

					len = strlen(packet->fields[1]);
					name_start = packet->fields[1];
					name_end = name_start;
					while (((name_end - packet->fields[1]) < len) && *name_end != ' ') {
						name_end++;
					}
					strncpy(user, packet->fields[1], name_end - name_start);
	
					gaim_conv_chat_add_user(GAIM_CONV_CHAT(conv), user,
						NULL, GAIM_CBFLAGS_NONE, TRUE);
				/* Somebody has left current group */
				} else if (packet->nof == 2 &&
				           ((strncmp(packet->fields[0], ICB_STAT_DEPART, ICB_STAT_DEPART_LEN) == 0)
				           || (strncmp(packet->fields[0], ICB_STAT_SIGNOFF, ICB_STAT_SIGNOFF_LEN) == 0))) {
					GaimConversation *conv = NULL;
					char              user[ICB_PACKET_SIZE], *name_end, *name_start;
					int               len;

					/* Get current group */
					conv = icb_get_current_group(gc->account);
					if (!conv)
						break;
					
					/* Looking for a group name */
					bzero(user, sizeof(user));

					len = strlen(packet->fields[1]);
					name_start = packet->fields[1];
					name_end = name_start;
					while (((name_end - packet->fields[1]) < len) && *name_end != ' ') {
						name_end++;
					}
					strncpy(user, packet->fields[1], name_end - name_start);
	
					gaim_conv_chat_remove_user(GAIM_CONV_CHAT(conv), user, NULL);
				}
				break;
			case ICB_CMD_COMMAND_RESP:
				/* group listing: one wl entry for each member */
				if (packet->nof == 9 && (strncmp(packet->fields[0], "wl", 2) == 0)) {
					GaimConversation *conv;

					conv = icb_get_current_group(gc->account);
					if (!conv)
						break;

					gaim_conv_chat_add_user(GAIM_CONV_CHAT(conv), packet->fields[2],
						NULL, GAIM_CBFLAGS_NONE, FALSE);
				/* packet with Mod and Topic */
				} else if (packet->nof == 2 && (strncmp(packet->fields[0], "co", 2) == 0)
				           && ((tmp = strstr(packet->fields[1], ICB_TOPIC)) != NULL)) {
					GaimConversation *conv;

					gaim_debug_info("icb", "Topic jest: %s\n", tmp + strlen(ICB_TOPIC));

					conv = icb_get_current_group(gc->account);
					if (!conv)
						break;

					gaim_debug_info("icb", "za icb_get_current_group\n");
						
					gaim_conv_chat_set_topic(gaim_conversation_get_chat_data(conv), "(unknown)",
					                         tmp + strlen(ICB_TOPIC));
				}
			default:
				break;
		}
		icb_free_packet(&packet);
	}

	/* Move data left in buffer to the beginning */
	if (icb_input_fill > 0 && icb_input_pos != icb_input_buf) {
		memmove(icb_input_buf, icb_input_pos, icb_input_fill);
		*(icb_input_buf + icb_input_fill) = '\0';
	}

	icb_input_pos = icb_input_buf;

	gaim_debug_misc("icb", "<- icb_input_cb()\n");
}


static void
icb_close(GaimConnection *gc)
{
	IcbSession *icb = gc->proto_data;

	gaim_debug_info("icb", "-> icb_close\n");

	if (icb == NULL) {
		gaim_debug_info("icb", "<- icb_close\n");
		return;
	}

	if (gc->inpa)
		gaim_input_remove(gc->inpa);

        close(icb->fd);
        g_free(icb->server);
        g_free(icb->user);
        g_free(icb);		

	gaim_debug_info("icb", "<- icb_close\n");
}

static int
icb_send_chat(GaimConnection *gc, int id, const char *message)
{
	IcbSession *icb = gc->proto_data;
	int         r, len = strlen(message);
	char       *tmp, *pos, buf[ICB_MAX_DATA_SIZE+1];

	gaim_debug_info("icb", "icb_send_chat\n");
	gaim_debug_info("icb", "id=%d, len=%d, msg=\"%s\"\n", id, len, message);

	/* Split <message> into smaller chunks, as packed size is limited to
	 * ICB_MAX_DATA_SIZE bytes.
	 */
	pos = (char *) message;
	while (len > 0) {
		r = len < ICB_MAX_DATA_SIZE ? len : ICB_MAX_DATA_SIZE;
		memcpy(buf, pos, r);
		buf[r] = '\0';

		pos += r;
		len -= r;
	
		r = icb_send(icb, ICB_CMD_OPEN_MSG, 1, buf);
		if (r) {
			tmp = gaim_escape_html(buf);
			serv_got_chat_in(gc, id, gaim_connection_get_display_name(gc),
				0, tmp, time(NULL));
			free(tmp);
		}
	}

	gaim_debug_info("icb", "<- icb_send_chat\n");

	return 0;
}

static int
icb_send_im(GaimConnection *gc, const char *who, const char *msg, GaimConvImFlags flags)
{
	IcbSession       *icb = gc->proto_data;
	GaimConversation *conv;
	char              buf[ICB_PACKET_SIZE], *tmp, *pos, *appendpos;
	int               r, wholen, msglen, max_msg_size;

	wholen = strlen(who);
	msglen = strlen(msg);

	gaim_debug_info("icb", "icb_send_im\n");
	gaim_debug_info("icb", "who=\"%s\", len=%d, msg=\"%s\"\n", who, msglen, msg);

	/* max_msg_size is smaller than ICB_MAX_DATA_SIZE as IM packet looks like this:
	 *   -gm,username message sent by user*
	 * where:
	 *   - is packet size
	 *   , is ICB_SEPARATOR
	 *   * is NUL
	 * -3 is for "m", "," and spacebar after username.
	 */
	max_msg_size = ICB_MAX_DATA_SIZE - 3 - wholen;

	/* adding common "username " to buffer */
	memcpy(buf, who, wholen);
	buf[wholen] = ' ';

	appendpos = &buf[wholen + 1];
	pos = (char *) msg;
	while (msglen > 0) {
		r = msglen < max_msg_size ? msglen : max_msg_size;
		memcpy(appendpos, pos, r);
		*(appendpos + r) = '\0';

		pos += r;
		msglen -= r;

		r = icb_send(icb, ICB_CMD_COMMAND, 2, "m", buf);
		if (r > 0) {
#ifdef OLD_GAIM
			conv = gaim_find_conversation_with_account(who, gc->account);
#else
			conv = gaim_find_conversation_with_account(GAIM_CONV_IM, who, gc->account);
#endif
			if (conv) {
				tmp = gaim_escape_html(appendpos);
				gaim_conv_im_write(GAIM_CONV_IM(conv), gaim_connection_get_display_name(gc),
					tmp, 0, time(NULL));
				free(tmp);
			}
		} 
	}

	gaim_debug_info("icb", "<- icb_send_im\n");

	return 0;
}

static void
icb_get_info(GaimConnection *gc, const char *who)
{
	IcbSession *icb = gc->proto_data;
	int         ret;

	gaim_debug_info("icb", "-> icb_get_info: %s\n", who);

	ret = icb_send(icb, ICB_CMD_COMMAND, 2, "w", who);
	if (ret < 0) {
		gaim_connection_error(gc, _("Unable to access user profile."));
		return;
	}

	gaim_debug_info("icb", "<- icb_get_info\n");
}

static void
icb_join_chat(GaimConnection *gc, GHashTable *data)
{
	IcbSession *icb = gc->proto_data;
	char       *group;

	gaim_debug_info("icb", "-> icb_join_chat\n");

	group = g_hash_table_lookup(data, "group");

	gaim_debug_info("icb", "group %s\n", group);

	icb_send(icb, ICB_CMD_COMMAND, 2, "g", group);

	gaim_debug_info("icb", "<- icb_join_chat\n");
}

static void
icb_leave_chat(GaimConnection *gc, int id)
{
	//IcbSession *icb = gc->proto_data;

	gaim_debug_info("icb", "-> icb_leave_chat\n");

	//icb_send(icb, ICB_CMD_COMMAND, 2, "g", "1");

	gaim_debug_info("icb", "<- icb_leave_chat\n");
}

static const char *
icb_list_icon(GaimAccount *a, GaimBuddy *b)
{
	return "icb";
}

static GList *
icb_chat_info(GaimConnection *gc)
{
	GList                   *m = NULL;
	struct proto_chat_entry *pce;

	gaim_debug_misc("icb", "-> icb_chat_info\n");

	pce = g_new0(struct proto_chat_entry, 1);
	pce->label = _("_Group:");
	pce->identifier = "group";
	m = g_list_append(m, pce);

	gaim_debug_misc("icb", "<- icb_chat_info\n");

	return m;
}

GHashTable *
icb_chat_info_defaults(GaimConnection *gc, const char *chat_name)
{
	GHashTable *defaults;

	gaim_debug_misc("icb", "-> icb_chat_info_defaults\n");

        defaults = g_hash_table_new_full(g_str_hash, g_str_equal, NULL, g_free);
	gaim_debug_misc("icb", "defaults %p\n", defaults);

	gaim_debug_misc("icb", "%s\n", chat_name);

#if 0
        if (chat_name != NULL)
                g_hash_table_insert(defaults, "room", g_strdup(chat_name));
#endif

	gaim_debug_misc("icb", "<- icb_chat_info_defaults\n");

        return defaults;
}

static GaimPlugin *icb_plugin = NULL;

static GaimPluginProtocolInfo prpl_info =
{
	OPT_PROTO_CHAT_TOPIC | OPT_PROTO_PASSWORD_OPTIONAL,
	NULL,			/* user_splits */
	NULL,			/* protocol_options */
	NO_BUDDY_ICONS,		/* icon_spec */
	icb_list_icon,		/* list_icon */
	NULL,			/* list_emblems */
	NULL,			/* status_text */
	NULL,			/* tooltip_text */
	icb_status_types,	/* status_types */
	NULL,			/* blist_node_menu */
	icb_chat_info,		/* chat_info */
	icb_chat_info_defaults,	/* chat_info_defaults */
	icb_login,		/* login */
	icb_close,		/* close */
	icb_send_im,		/* send_im */
	NULL,			/* set_info */
	NULL,			/* send_typing */
	icb_get_info,		/* get_info */
	NULL,			/* set_away */
	NULL,			/* set_idle */
	NULL,			/* change_passwd */
	NULL,			/* add_buddy */
	NULL,			/* add_buddies */
	NULL,			/* remove_buddy */
	NULL,			/* remove_buddies */
	NULL,			/* add_permit */
	NULL,			/* add_deny */
	NULL,			/* rem_permit */
	NULL,			/* rem_deny */
	NULL,			/* set_permit_deny */
	NULL,			/* warn */
	icb_join_chat,		/* join_chat */
	NULL,			/* reject_chat */
	NULL,			/* get_chat_name */
	NULL,			/* chat_invite */
	icb_leave_chat,		/* chat_leave */
	NULL,			/* chat_whisper */
	icb_send_chat,		/* chat_send */
	icb_keepalive,		/* keepalive */
	NULL,			/* register_user */
	NULL,			/* get_cb_info */
	NULL,			/* get_cb_away */
	NULL,			/* alias_buddy */
	NULL,			/* group_buddy */
	NULL,			/* rename_group */
	NULL,			/* buddy_free */
	NULL,			/* convo_closed */
	NULL,			/* normalize */
	NULL,			/* set_buddy_icon */
	NULL,			/* remove_group */
	NULL,			/* get_cb_real_name */
	NULL,			/* set_chat_topic */
	NULL,			/* find_blist_chat */
	NULL,			/* roomlist_get_list */
	NULL,			/* roomlist_cancel */
	NULL,			/* roomlist_expand_category */
	NULL,			/* can_receive_file */
	NULL			/* send_file */
};

static GList *
icb_actions(GaimPlugin *plugin, gpointer context)
{
	GList *list = NULL;
	//GaimPluginAction *act = NULL;
/*
	act = gaim_plugin_action_new(_("View MOTD"), irc_view_motd);
	list = g_list_append(list, act);
 */
	return list;
}
static GaimPluginInfo info =
{
	GAIM_PLUGIN_MAGIC,
	GAIM_MAJOR_VERSION,
	GAIM_MINOR_VERSION,
	GAIM_PLUGIN_PROTOCOL,                        /**< type           */
	NULL,                                        /**< ui_requirement */
	0,                                           /**< flags          */
	NULL,                                        /**< dependencies   */
	GAIM_PRIORITY_DEFAULT,                       /**< priority       */

	"prpl-icb",		                     /**< id             */
	"ICB",                                       /**< name           */
	ICB_VERSION,                                 /**< version        */
	N_("ICB Protocol Plugin"),                   /**  summary        */
	N_("Internet Citizen's Band Protocol Plugin"), /**  description    */
	"Aleksander Piotrowski <aleksander.piotrowski@nic.com.pl>", /**< author         */
	GAIM_WEBSITE,                                /**< homepage       */

	NULL,                                        /**< load           */
	NULL,                                        /**< unload         */
	NULL,                                        /**< destroy        */

	NULL,                                        /**< ui_info        */
	&prpl_info,                                  /**< extra_info     */
	NULL,
	icb_actions
};

void
_init_plugin(GaimPlugin *plugin)
{
	GaimAccountUserSplit *split;
	GaimAccountOption    *option;

	split = gaim_account_user_split_new(_("Server"), ICB_DEFAULT_SERVER, '@');
	prpl_info.user_splits = g_list_append(prpl_info.user_splits, split);

	option = gaim_account_option_int_new(_("Port"), "port", ICB_DEFAULT_PORT);
	prpl_info.protocol_options = g_list_append(prpl_info.protocol_options, option);

	icb_plugin = plugin;
}

GAIM_INIT_PLUGIN(icb, _init_plugin, info)
