/*
 * station.c
 *
 * Routines to manage station presets.
 *
 * (C) 1997 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*      ******************** Include Files                ************** */

#include <stdio.h>
#include "tvdefines.h"
#include "glob.h"
#include "station.h"

/*      ******************** Local defines                ************** */
/*      ******************** Forward declarations         ************** */
/*      ******************** Private variables            ************** */
/*      ******************** Function Definitions         ************** */

/*  TVSTATIONLookup - Given the current driver state, lookup the      */
/*    the predefined station we're on (if applicable).  If not on a   */
/*    station, NULL is returned.                                      */
void TVSTATIONLookup(
         TV_DRIVER_STATE  *state,
         TV_STATION      **station )
{
    TV_PREFS        *p = &G_glob.prefs;
    TV_STATION      *st;
    TV_INT32         num_st,
                     i;

    *station = NULL;

    /*  Figure out which tuner input we're dealing with  */
    if ( p->tuner_mode == TV_TUNER_MODE_ANTENNA )
        st     = p->ant_station,
        num_st = p->ant_num_stations;
    else if ( p->tuner_mode == TV_TUNER_MODE_CABLE )
        st     = p->cable_station,
        num_st = p->cable_num_stations;
    else {
        fprintf( stderr, 
                 "TVUTILStationLookup: Unsupported tuner mode\n" );
        return;
    }

    /*  Ok, now look for a station that matches the current state  */
    for ( i = 0; i < num_st; i++ )
        if ( (( state->tuner_chan_active  &&  st[i].set_via_channel &&
              ( state->tuner_chan == st[i].channel ) )) ||
             (( !state->tuner_chan_active && !st[i].set_via_channel &&
                APPROX( state->tuner_freq, st[i].freq, 1.1/FREQFACTOR ) )) )
            break;

    *station = ( i < num_st ) ? &st[i] : NULL;
}
