/*
 * appear_dlg.c
 *
 * Code for the capture appearance parameters dialog
 *
 * (C) 1997 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*      ******************** Include Files                ************** */

#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/Box.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Form.h>
#include <X11/Xaw/Scrollbar.h>
#include "tvdefines.h"
#include "glob.h"
#include "xutil.h"
#include "appear_dlg.h"


/*      ******************** Local defines                ************** */

#define THUMB_LEN  0.1

typedef struct {
    TV_DRIVER_PARAM  param;
    char            *title_name;
    char            *value_name;
    char            *scroll_name;
    char            *min_name;
    char            *max_name;
    Widget           value_wgt;
    Widget           scroll_wgt;
    Widget           min_wgt;
    Widget           max_wgt;
} TV_APPEAR_CTL_DEF;

/*      ******************** Private variables            ************** */

static Widget Dialog_wgt = NULL;

static TV_APPEAR_CTL_DEF Appearctl_def[] = {

  { TV_PARAM_HUE     , "hueTitleLabel"   , "hueValueText"   , 
    "hueScrollbar"   , "hueMinLabel"     , "hueMaxLabel"     },
  { TV_PARAM_BRIGHT  , "brightTitleLabel", "brightValueText",
    "brightScrollbar", "brightMinLabel"  , "brightMaxLabel"  },
  { TV_PARAM_CONTRAST, "contrTitleLabel" , "contrValueText" ,
    "contrScrollbar" , "contrMinLabel"   , "contrMaxLabel"   },
  { TV_PARAM_SATU    , "satUTitleLabel"  , "satUValueText"  ,
    "satUScrollbar"  , "satUMinLabel"    , "satUMaxLabel"    },
  { TV_PARAM_SATV    , "satVTitleLabel"  , "satVValueText"  ,
    "satVScrollbar"  , "satVMinLabel"    , "satVMaxLabel"    }
};

/*      ******************** Forward declarations         ************** */
/*      ******************** Function Definitions         ************** */

/*  OkCmdCB - Dismiss the dialog  */
static void OkCmdCB( Widget w, XtPointer cl, XtPointer cb )
{
    if ( Dialog_wgt == NULL )
        return;
    XtPopdown( Dialog_wgt );
}

static void TextValUpdate( Widget text_wgt, double val )
{
#ifdef WHEN_WE_MAKE_IT_EDITABLE
    XawTextBlock     tblk;
    char             *old_str;
    int              old_len;
#endif
    char             str[ 80 ];

    assert( text_wgt != NULL );

    sprintf( str, "%.0f", val );

#ifdef WHEN_WE_MAKE_IT_EDITABLE
    memset( &tblk, '\0', sizeof( tblk ) );
    tblk.firstPos = 0;
    tblk.length   = strlen( str );
    tblk.ptr      = str;
    tblk.format   = XawFmt8Bit;

    XtVaGetValues( text_wgt, XtNstring, &old_str, 
                             NULL );
    old_len = (old_str == NULL) ? 0 : strlen( old_str );
    XawTextReplace( text_wgt, 0, old_len, &tblk );
#else
    XtVaSetValues( text_wgt, XtNstring, str,
                             NULL );
#endif
}


static void ScrollValUpdate( Widget scroll_wgt, double val, double lim[2] )
{
    assert( scroll_wgt != NULL );
    assert( !APPROX( lim[0], lim[1], DOUBLE_EPS ) );

    val = (val - lim[0]) / (lim[1] - lim[0]);
    val *= (1.0 - THUMB_LEN);
    XUTILXawScrollbarSetThumb ( scroll_wgt, val, THUMB_LEN );
}


static void ControlSliderJumpCB( Widget w, 
                                 XtPointer cl_data, 
                                 XtPointer cb_data )
{
    TV_CAPTURE        *c       = &G_glob.capture;
    TV_APPEAR_CTL_DEF *r       = (TV_APPEAR_CTL_DEF *) cl_data;
    float              percent = *(float *)cb_data;
    double             lim[2],
                       val;

    TVCAPTUREQueryParamLimits( c, r->param, lim );

    val = (percent / (1.0 - THUMB_LEN)) * (lim[1] - lim[0]) + lim[0];

    TVCAPTURESetAppearanceParam( c, r->param, val );
    TextValUpdate( r->value_wgt, val );
}


static void ControlSliderScrollCB( Widget w, 
                                   XtPointer cl_data, 
                                   XtPointer cb_data )
{
    TV_CAPTURE        *c       = &G_glob.capture;
    TV_APPEAR_CTL_DEF *r       = (TV_APPEAR_CTL_DEF *) cl_data;
    float              percent;
    int                pos     = (int)cb_data;
    Dimension          length;
    double             lim[2],
                       val;

    TVCAPTUREQueryParamLimits( c, r->param, lim );
    XtVaGetValues( w, XtNlength    , &length,
                      XtNtopOfThumb, &percent,
                      NULL );

    percent /= (1.0 - THUMB_LEN);

    percent += (double)pos / length;
    percent = MAX( 0.0, MIN( 1.0, percent ) );
    val = percent * (lim[1] - lim[0]) + lim[0];

    XUTILXawScrollbarSetThumb ( w, percent * (1.0 - THUMB_LEN), THUMB_LEN );
    TVCAPTURESetAppearanceParam( c, r->param, val );
    TextValUpdate( r->value_wgt, val );
}


static void TVAppearDialogBuild( Widget *dialog_wgt )
{
    Position  x, y;
    Dimension height;
    Widget    w, box;
    TV_INT32     i;

    /*  Create the dialog widgets  */
    *dialog_wgt = XtVaCreatePopupShell( "inputAppearDialog",
                      transientShellWidgetClass, TVTOPLEVEL,
                      NULL );

    box = XtVaCreateManagedWidget( "mainBox", boxWidgetClass, *dialog_wgt, 
                                   XtNorientation, XtorientVertical,
                                   NULL );

    for ( i = 0; i < XtNumber( Appearctl_def ); i++ ) {
        TV_APPEAR_CTL_DEF *r = &Appearctl_def[i];
        Widget             gbox, cbox;

        gbox = XtVaCreateManagedWidget( "groupBox", boxWidgetClass, box, 
                                       XtNorientation, XtorientVertical,
                                       NULL );
        
        cbox = XtVaCreateManagedWidget( "headerBox", boxWidgetClass, gbox, 
                                       XtNorientation, XtorientHorizontal,
                                       NULL );
        
        w = XtVaCreateManagedWidget( r->title_name, labelWidgetClass, cbox,
                                     NULL );

        w = XtVaCreateManagedWidget( r->value_name, asciiTextWidgetClass, 
                                     cbox,
                                 XtNtype            , XawAsciiString,
                                 XtNuseStringInPlace, False,
                                 XtNscrollHorizontal, XawtextScrollNever,
                                 XtNscrollVertical  , XawtextScrollNever,
                                 XtNdisplayCaret    , False,
                                 XtNeditType        , XawtextRead,
                                 XtNresize          , XawtextResizeNever,
                                 NULL );
        r->value_wgt = w;

        cbox = XtVaCreateManagedWidget( "scrollBox", boxWidgetClass, gbox, 
                                       XtNorientation, XtorientHorizontal,
                                       NULL );
        
        w = XtVaCreateManagedWidget( r->min_name, labelWidgetClass, cbox,
                                     XtNresize, False,
                                     XtNencoding, XawTextEncoding8bit,
                                     NULL );
        r->min_wgt = w;

        w = XtVaCreateManagedWidget( r->scroll_name, scrollbarWidgetClass, 
                                     cbox,
                                     XtNorientation, XtorientHorizontal,
                                     NULL );
        r->scroll_wgt = w;

        XtAddCallback( w, XtNjumpProc  , ControlSliderJumpCB  , r );
        XtAddCallback( w, XtNscrollProc, ControlSliderScrollCB, r );

        w = XtVaCreateManagedWidget( r->max_name, labelWidgetClass, cbox,
                                     XtNresize, False,
                                     XtNencoding, XawTextEncoding8bit,
                                     NULL );
        r->max_wgt = w;
    }

    w = XtVaCreateManagedWidget( "actionForm", formWidgetClass, box, 
                                 NULL );

    w = XtVaCreateManagedWidget( "okCmd", commandWidgetClass, w,
                                 NULL );
    XtAddCallback( w, XtNcallback, OkCmdCB, NULL );

    /*  By default, position dialog just below main TV window  */
    XtVaGetValues( TVTOPLEVEL, XtNheight , &height,
                               NULL);

    XtTranslateCoords( TVTOPLEVEL, 0, height, &x, &y );

    XtVaSetValues( *dialog_wgt, XtNx, x,
                                XtNy, y,
                                NULL );
}


void TVAPPEARDIALOGPopUp()
{
    if ( Dialog_wgt == NULL )
        TVAppearDialogBuild( &Dialog_wgt );

    TVAPPEARDIALOGResync();

    XUTILXtPopup( Dialog_wgt, XtGrabNone, TVTOPLEVEL );
}


void TVAPPEARDIALOGResync()
{
    TV_CAPTURE      *c       = &G_glob.capture;
    TV_DRIVER_STATE  s;
    TV_INT32            i;
    double           lim[2],
                     val;
    char             str[80];

    /*  FIXME:  Also install EnterNotify handler for this dialog to  */
    /*    resync values on entry of it's shell.                      */

    if ( Dialog_wgt == NULL )
        return;

    /*  Query capture driver's current values  */
    if ( !TVCAPTUREQueryDriverState( c, &s ) ) {
        fprintf( stderr, "TVCAPTUREQueryDriverState() failed\n" );
        exit(1);
    }

    /*  Update min/max labels, scrollbar position, and value text widget  */
    for ( i = 0; i < XtNumber( Appearctl_def ); i++ ) {
        TV_APPEAR_CTL_DEF *r = &Appearctl_def[i];

        TVCAPTUREQueryParamLimits( c, r->param, lim );

        assert( r->min_wgt != NULL );
        sprintf( str, "%.0f", lim[0] );
        XtVaSetValues( r->min_wgt, XtNlabel, str, 
                                   NULL );

        assert( r->max_wgt != NULL );
        sprintf( str, "%.0f", lim[1] );
        XtVaSetValues( r->max_wgt, XtNlabel, str, 
                                   NULL );

        switch ( r->param ) {
            case TV_PARAM_HUE      :  val = s.hue       ;  break;
            case TV_PARAM_BRIGHT   :  val = s.brightness;  break;
            case TV_PARAM_CONTRAST :  val = s.contrast  ;  break;
            case TV_PARAM_SATU     :  val = s.sat_u     ;  break;
            case TV_PARAM_SATV     :  val = s.sat_v     ;  break;
            default :
                fprintf( stderr, 
                   "TVAPPEARDIALOGResync: Unsupported param %d\n", r->param );
                exit(1);
        }

        TextValUpdate  ( r->value_wgt , val );
        ScrollValUpdate( r->scroll_wgt, val, lim );
    }
}

