/*
 * annot.h
 *
 * API for annot.c: routines to manage the display of annotation on top of 
 * the video window.
 *
 * (C) 1997 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef __ANNOT_H
#define __ANNOT_H

#include <stdio.h>
#include <sys/time.h>
#include <X11/Xlib.h>
#include <X11/Intrinsic.h>
#include "tvtypes.h"

typedef enum {
    TV_ANNOT_TYPE_STATION,
    TV_ANNOT_TYPE_TUNER_MODE,
    TV_ANNOT_TYPE_INPUT_DEV,
    TV_ANNOT_TYPE_VOLUME,
    TV_ANNOT_TYPE_MUTE
} TV_ANNOT_TYPE;
 

typedef struct {
    TV_INT32       delay;            /*  How long to display (in msec)    */
                                     /*    0=never; -1=always             */
    struct timeval start_time;       /*  When started (-1 = not running)  */  

    GC             gc;               /*  GC to use for drawing            */
    Pixel          fg_pixel;         /*  Foreground color                 */
    char          *font_pattern;     /*  Font pattern                     */
    TV_INT32       pixel_size;       /*  Current font pixel size          */
    XFontSet       fontset;          /*  Cached fontset                   */
    TV_GEOM        loc;              /*  Where to draw it                 */
} TV_ANNOT_PROP;

typedef struct {

    Display       *display;
    int            screen;
    XtAppContext   app_context;
    Drawable       drawable;         /*  Window or pixmap to annotate     */
    TV_GEOM        geom;             /*  Width & height of drawable       */

    TV_BOOL        auto_update;      /*  Whether auto-update mode is on   */
    XtIntervalId   timer;            /*  Used in auto update mode         */
    TV_BOOL        timer_set;      

    TV_ANNOT_PROP  station;          /*  Station                          */
    TV_ANNOT_PROP  tuner_mode;       /*  Tuner mode                       */
    TV_ANNOT_PROP  input_dev;        /*  Input device                     */
    TV_ANNOT_PROP  volume;           /*  Audio volume                     */
    TV_ANNOT_PROP  mute;             /*  Audio mute state                 */

} TV_ANNOT;


#ifdef __cplusplus
extern "C" {
#endif

void TVANNOTInit( 
         TV_ANNOT     *a,
         Display      *display,
         int           screen,
         XtAppContext  app_context );

void TVANNOTSetDrawable( 
         TV_ANNOT *a,   
         Drawable  drawable );

void TVANNOTSetDrawableSize(
         TV_ANNOT      *a,
         Dimension      width,
         Dimension      height );

void TVANNOTSignalPropChange(
         TV_ANNOT      *a,
         TV_ANNOT_TYPE  type );

TV_BOOL TVANNOTUpdate(
         TV_ANNOT *a );

void TVANNOTSetAutoUpdateMode(
         TV_ANNOT *a,
         TV_BOOL   enabled );

#ifdef __cplusplus
}  /* Close 'extern "C"' */
#endif

#endif




