/*
 * tvcapture.h
 *
 * API for determining basic capabilities of the video card on an
 *   X server screen.
 *
 * (C) 1997 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

#ifndef __TVCAPTURE_H
#define __TVCAPTURE_H

#include <X11/Intrinsic.h>
#ifdef __NetBSD__
# include <dev/pci/bt848var.h>
#else
# include <machine/ioctl_meteor.h>
#endif
#include "tvtypes.h"

#define TV_CHAN_MIN 1
#define TV_CHAN_MAX 99
#define TV_MAX_CHANNELS ( TV_CHAN_MAX - TV_CHAN_MIN + 1 )

/* scaling factor for tuner frequencies expressed as ints          */
/*   FIXME: From brooktree.c.  Should be defined in ioctl_bt848.h  */
#define FREQFACTOR              16

typedef enum {
            TV_FIELD_SUPPRESS,
            TV_FIELD_DISPLAY,
            TV_FIELD_DISK
        } TV_FIELD_TARGET;

typedef enum {
            TV_CAPTURE_SINGLE,
            TV_CAPTURE_CONTINUOUS
        } TV_CAPTURE_MODE;

typedef enum {
            TV_DEVICE_TUNER,
            TV_DEVICE_VIDEO,
            TV_DEVICE_SVIDEO,
            TV_DEVICE_CSVIDEO,
            TV_NUM_INPUT_DEVICES
        } TV_INPUT_DEVICE;

typedef enum {
            TV_AUDIO_INPUT_AUTO,
            TV_AUDIO_INPUT_TUNER,
            TV_AUDIO_INPUT_EXTERN,
            TV_AUDIO_INPUT_INTERN
        } TV_AUDIO_INPUT_DEVICE;

typedef enum {
            TV_INPUT_AUTO,
            TV_INPUT_NTSCM,
            TV_INPUT_NTSCJ,
            TV_INPUT_PALBDGHI,
            TV_INPUT_PALM,
            TV_INPUT_PALN,
            TV_INPUT_SECAM,
            TV_INPUT_PALNCOMB
        } TV_INPUT_FORMAT;

typedef int TV_FREQ_SET;

typedef enum {
            TV_PARAM_HUE,
            TV_PARAM_BRIGHT,
            TV_PARAM_CONTRAST,
            TV_PARAM_SATU,
            TV_PARAM_SATV,
        } TV_DRIVER_PARAM;

typedef enum {
            TV_PIXELTYPE_RGB,
            TV_PIXELTYPE_YUV
        } TV_PIXELTYPE;

typedef struct {
            TV_INPUT_DEVICE       input_dev;
            TV_INPUT_FORMAT       input_fmt;
            TV_AUDIO_INPUT_DEVICE audio_input_dev;
            TV_BOOL               audio_mute;

            double          brightness;
            double          contrast;
            double          hue;
            double          sat_u;
            double          sat_v;
            TV_UINT32       signal;
            TV_FREQ_SET     tuner_freq_set;
            TV_BOOL         tuner_chan_active;
            TV_UINT32       tuner_chan;
            double          tuner_freq;
            TV_UINT32       fps;
            struct {
                long        frames_captured;
            } stats;
    
        } TV_DRIVER_STATE;

typedef enum {
           TV_FRAME_PLANAR,                    /*  E.g. YYYYY..UUUUU...VVVVV */
           TV_FRAME_PACKED                     /*  E.g. YVUVYVUV...          */
        } TV_FRAME_PACKING;

typedef struct {
            TV_INT32         index;            /*  Drvr hdlr for pix fmt     */
            TV_PIXELTYPE     type;             /*  RGB, YUV                  */

            /* RGB-only attributes  */
            TV_UINT32        Bpp;              /*  Bytes per pixel           */
            TV_UINT32        mask[3];          /*  Pixel masks               */
            unsigned         swap_bytes  :1;   /*  Bytes in shorts swapped   */
            unsigned         swap_shorts :1;   /*  Shorts in longs swapped   */

            /* YUV-only attributes  */
            TV_UINT32        samp_size[3];     /*  [YUV] Bits per sample     */
            TV_UINT32        samp_int_h[3];    /*  [YUV] Horiz samp interval */
            TV_UINT32        samp_int_v[3];    /*  [YUV] Vert  samp interval */
            TV_FRAME_PACKING frame_packing;    /*  Order YUV data is stored  */
            char             comp_order[30];   /*  "YUYV", "YVU", etc.       */
            unsigned         order_t_to_b :1;  /*  Scanline order TtoB;BtoT  */
            unsigned         order_l_to_r :1;  /*  Column   order RtoL;LtoR  */
            unsigned         y_trans      :1;  /*  LSb Y = transparency?     */

        } TV_PIXEL_GEOM;

typedef struct {
            TV_INT8       *buf;                /*  Pixel data               */
            TV_GEOM        geom;               /*  Dimensional geometry     */
            TV_PIXEL_GEOM  pix_geom;           /*  Pixel geometry           */
        } TV_IMAGE;

typedef struct {

    int               fd;
    int               tfd;
    TV_UINT8         *drv_buf;

    TV_INT32              input_format;
    TV_INT32              input_dev;
    TV_AUDIO_INPUT_DEVICE audio_input_dev;

    TV_UINT32         fps;
    TV_INT32          bpp_format;
    TV_CAPTURE_MODE   cap_mode;
    TV_TRANSFER_MODE  xfer_mode;
    TV_PIXEL_GEOM    *pix_geom_list;
    TV_UINT32         pix_geom_list_len;
    TV_UINT32         pix_geom_idx;
    TV_BOOL           tuner_chan_active;      /* FIXME: this is a hack     */

    TV_FIELD_TARGET   field_targ[2];
    double            brightness;
    double            contrast;
    double            hue;
    double            sat_u;
    double            sat_v;
    TV_UINT32         addr;
    TV_GEOM           geom;

    void            (*frame_done_cb)( TV_IMAGE *img );
    TV_BOOL           frame_cb_enabled;

    TV_INT32          width_min , width_max , width_res;
    TV_INT32          height_min, height_max, height_res;
    TV_UINT32         fps_max;
                      
    TV_BOOL           contin_on;
    
} TV_CAPTURE;


#ifdef __cplusplus
extern "C" {
#endif

void TVCAPTUREInit( TV_CAPTURE *c );

/*  Primary capture parameters  */
void TVCAPTURESetCaptureMode ( TV_CAPTURE *c, TV_CAPTURE_MODE   cap_mode  );
void TVCAPTURESetTransferMode( TV_CAPTURE *c, TV_TRANSFER_MODE  xfer_mode );
void TVCAPTURESetRegionGeom  ( TV_CAPTURE *c, TV_GEOM          *reg_geom  );
void TVCAPTURESetPixelGeom   ( TV_CAPTURE *c, TV_PIXEL_GEOM    *pix_geom  );

/*  Capture control  */
TV_BOOL TVCAPTUREValidRegionGeom( TV_CAPTURE *c, TV_GEOM *reg_geom );
TV_BOOL TVCAPTUREConfigure( TV_CAPTURE    *c, char **fail_reason );
void    TVCAPTUREStart    ( TV_CAPTURE    *c );
void    TVCAPTUREStop     ( TV_CAPTURE    *c );

/*  Secondary capture parameters  */
void TVCAPTURESetBrightness  ( TV_CAPTURE *c, double brightness );
void TVCAPTURESetContrast    ( TV_CAPTURE *c, double contrast );
void TVCAPTURESetHue         ( TV_CAPTURE *c, double hue );
void TVCAPTURESetSatU        ( TV_CAPTURE *c, double sat_u );
void TVCAPTURESetSatV        ( TV_CAPTURE *c, double sat_v );
void TVCAPTURESetInputFormat ( TV_CAPTURE *c, TV_INPUT_FORMAT format );
void TVCAPTURESetInputDevice ( TV_CAPTURE *c, TV_INPUT_DEVICE dev );
void TVCAPTURESetAudioInputDevice( TV_CAPTURE *c, TV_AUDIO_INPUT_DEVICE dev );
void TVCAPTURESetTunerFreqSet( TV_CAPTURE *c, TV_FREQ_SET set );
void TVCAPTURESetTunerChannel( TV_CAPTURE *c, TV_INT32 chan_num );
void TVCAPTURESetTunerFreq   ( TV_CAPTURE *c, double freq );
char *TVCAPTUREGetTunerFreqSetName( TV_CAPTURE *c, TV_FREQ_SET set );
void TVCAPTURESetAfc         ( TV_CAPTURE *c, TV_BOOL afc );
void TVCAPTURESetAudioMute   ( TV_CAPTURE *c, TV_BOOL mute );
void TVCAPTURESetColorbars   ( TV_CAPTURE *c, TV_BOOL colorbars );
void TVCAPTURESetFPS         ( TV_CAPTURE *c, TV_UINT32 fps );

void TVCAPTURESetAppearanceParam( TV_CAPTURE *c, TV_DRIVER_PARAM p,
                                  double val );

TV_BOOL TVCAPTUREQueryDriverState( TV_CAPTURE *c, TV_DRIVER_STATE *s );
void TVCAPTUREQueryParamLimits( TV_CAPTURE *c, TV_DRIVER_PARAM p, 
                                double lim[2] );

void TVCAPTUREGetNumPixFmts( TV_CAPTURE *c, TV_UINT32 *num );
void TVCAPTUREGetNthPixFmt ( TV_CAPTURE *c, TV_UINT32 index,
                             TV_PIXEL_GEOM *geom );

void TVCAPTUREGetPixFmtByPixGeom( TV_CAPTURE *c, TV_PIXEL_GEOM *geom,
                                  TV_INT32 *index );

void TVCAPTUREGetFPSMax( TV_CAPTURE *c, TV_UINT32 *fps_max );

/*  Utility  */
void TVCAPTURESetFrameDoneCBEnabled( TV_CAPTURE *c, TV_BOOL enable );
void TVCAPTUREWorkProc();

void TVCAPTUREClearPendingFrames();
TV_BOOL TVCAPTUREGetPendingFrame( TV_IMAGE **img );

#ifdef __cplusplus
}  /* Close 'extern "C"' */
#endif

#endif
