/*
 * glob.c
 *
 * Global state data for the TV app.
 *
 * (C) 1997 Randall Hopper
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are
 * met: 1. Redistributions of source code must retain the above copyright
 * notice, this list of conditions and the following disclaimer. 2.
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE FOR
 * ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 */

/*      ******************** Include Files                ************** */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include "tvdefines.h"
#include "tvtypes.h"
#include "glob.h"

/*      ******************** Global variables             ************** */

TV_GLOB  G_glob;
int      G_debug = 0;
TV_BOOL  G_in_x_error = FALSE;

/*      ******************** Local defines                ************** */
/*      ******************** Forward declarations         ************** */
/*      ******************** Private variables            ************** */
/*      ******************** Function Definitions         ************** */

static void TVGLOBDestroyImage()
{
    TV_DISPLAY          *d     = &G_glob.display;

    /*  Clean up shared memory on the way out  */
    if (( d->ximage.ximg != NULL ) && d->ximage.is_shm ) {
        shmdt ( d->ximage.shm_info.shmaddr );
        shmctl( d->ximage.shm_info.shmid, IPC_RMID, 0 );
    }
}


void TVGLOBInitXImage( TV_XIMAGE *i )
{
    i->ximg        = NULL;

    atexit( TVGLOBDestroyImage );
}


void TVGLOBInitDisplay( 
         TV_GLOB *p, 
         Display *display,
         int      screen,
         Widget   shell_wgt )
{
    TV_GEOM       g;
    XtAppContext  app_context = XtWidgetToApplicationContext( shell_wgt );

    p->display.enabled          = True;
    p->display.zoom_on          = False;
    p->display.freeze_on        = False;
    p->display.aspect_lock      = False;
    p->display.cap_mode         = TV_CAPTURE_CONTINUOUS;
    p->display.geom             = p->capture.geom;
    p->display.geom.x           = 10;
    p->display.geom.y           = 10;

    memcpy( &p->display.refresh_geom, &p->display.geom, 
            sizeof(p->display.refresh_geom) );

    p->display.unzoomed.geom    = p->capture.geom;
    p->display.unzoomed.mode    = -1;
    p->display.unzoomed.viewp_x = -1;
    p->display.unzoomed.viewp_y = -1;
    p->display.pix_geom.Bpp     = 0;

    p->display.image.buf        = NULL;
    p->display.image.geom.w     = 0;
    p->display.image.geom.h     = 0;

    p->display.app_context      = app_context;
    p->display.shell_wgt        = shell_wgt;
    p->display.video_wgt        = NULL;
    p->display.win              = None;
    p->display.win_visibility   = VisibilityFullyObscured;

    TVGLOBInitXImage( &p->display.ximage );
    p->display.ximage_use_for_expose = FALSE;

    p->display.gc                 = NULL;
    p->display.waiting_for_resize = FALSE;

    TVANNOTInit( &p->display.annot, display, screen, app_context );

    p->display.cursor_dozeoff_enabled = FALSE;
    p->display.cursor_timer_set       = FALSE;
}

void TVGLOBInitPrefs( TV_PREFS *p )
{
    p->cable_station = NULL;
    p->ant_station   = NULL;
    p->cable_num_stations = 0;
    p->ant_num_stations   = 0;

    p->ant_freq_set   = TV_FREQ_SET_NABCST;
    p->cable_freq_set = TV_FREQ_SET_CABLEIRC;
    p->tuner_mode     = TV_TUNER_MODE_ANTENNA;
    p->afc_mode       = TRUE;
}


void TVGLOBInitDisk( TV_GLOB *p )
{
    TV_CAPTURE *c   = &G_glob.capture;

    p->disk.contin_save        = False;
    p->disk.fn_freeze_base[0]  = '\0';
    p->disk.fn_video_base[0]   = '\0';
    p->disk.fn_audio_base[0]   = '\0';
    p->disk.freeze_fmt         = TV_STILL_FMT_PPM;
    p->disk.freeze_use_suffix  = FALSE;
    p->disk.freeze_next_suffix = -1;
    p->disk.audio.file_fmt     = TV_AUDIO_FILE_FMT_AIFF;
    p->disk.audio.sample_fmt   = TV_AUDIO_SAMPLE_FMT_LIN_S16_LE;
    p->disk.audio.stereo       = TRUE;
    p->disk.audio.sample_rate  = 44100;
    p->disk.video.capture_on   = FALSE;
    p->disk.video.icap_fmt     = TV_ICAP_FMT_RGB16;
    p->disk.video.target       = TV_VIDEO_TARGET_MPEG;
    p->disk.video.geom.w       = p->capture.width_max  / 2;
    p->disk.video.geom.h       = p->capture.height_max / 2;
    p->disk.video.fps          = p->capture.fps_max;
    p->disk.video.cleanup_temp = TRUE;
    p->disk.video.cap_audio    = TRUE;
}


void TVGLOBInitPrinter( TV_GLOB *p )
{
    strcpy( p->printer.queue_name, "ps" );   /*  FIXME  */
}


void TVGLOBInit( Display *display, int screen, Widget shell )
{
    memset( &G_glob, '\0', sizeof( G_glob ) );

    TVCAPTUREInit    ( &G_glob.capture );
    TVGLOBInitDisplay( &G_glob, display, screen, shell );
    TVGLOBInitPrefs  ( &G_glob.prefs );
    TVGLOBInitDisk   ( &G_glob );
    TVAUDIOInit      ( &G_glob.audio );
    TVSCREENInit     ( &G_glob.x, display, screen );
    TVGLOBInitDisk   ( &G_glob );
    TVGLOBInitPrinter( &G_glob );
}

