/* snapshot.c: snapshot handling routines
   Copyright (c) 1999-2003 Philip Kendall

   $Id: snapshot.c,v 1.75 2003/11/07 14:07:33 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   E-mail: pak21-fuse@srcf.ucam.org
   Postal address: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

*/

#include <config.h>

#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <libspectrum.h>

#include "display.h"
#include "fuse.h"
#include "machine.h"
#include "scld.h"
#include "sound.h"
#include "snapshot.h"
#include "spec128.h"
#include "specplus2a.h"
#include "specplus3.h"
#include "spectrum.h"
#include "ui/ui.h"
#include "utils.h"
#include "z80/z80.h"
#include "z80/z80_macros.h"

int snapshot_read( const char *filename )
{
  utils_file file;
  libspectrum_snap *snap;
  int error;

  error = libspectrum_snap_alloc( &snap ); if( error ) return error;

  error = utils_read_file( filename, &file );
  if( error ) { libspectrum_snap_free( snap ); return error; }

  error = libspectrum_snap_read( snap, file.buffer, file.length,
				 LIBSPECTRUM_ID_UNKNOWN, filename );
  if( error ) {
    utils_close_file( &file ); libspectrum_snap_free( snap );
    return error;
  }

  if( utils_close_file( &file ) ) {
    libspectrum_snap_free( snap );
    return 1;
  }

  error = snapshot_copy_from( snap );
  if( error ) { libspectrum_snap_free( snap ); return error; }

  error = libspectrum_snap_free( snap ); if( error ) return error;

  return 0;
}

void snapshot_flush_slt (void)
{
  int i;
  for ( i=0; i<256; i++ ) {
    if( slt[i] ) free( slt[i] );
    slt[i] = NULL;
    slt_length[i] = 0;
  }
  slt_screen = NULL;
}

int
snapshot_read_buffer( const unsigned char *buffer, size_t length,
		      libspectrum_id_t type )
{
  libspectrum_snap *snap; int error;

  error = libspectrum_snap_alloc( &snap ); if( error ) return error;

  error = libspectrum_snap_read( snap, buffer, length, type, NULL );
  if( error ) { libspectrum_snap_free( snap ); return error; }
    
  error = snapshot_copy_from( snap );
  if( error ) { libspectrum_snap_free( snap ); return error; }

  error = libspectrum_snap_free( snap ); if( error ) return error;

  return 0;
}

int snapshot_copy_from( libspectrum_snap *snap )
{
  int i,j; int error;
  int capabilities;

  libspectrum_machine machine = libspectrum_snap_machine( snap );

  error = machine_select( machine );
  if( error ) {

    /* If we failed on a +3 snapshot, try falling back to +2A (in case
       we were compiled without lib765) */
    if( machine == LIBSPECTRUM_MACHINE_PLUS3 ) {
      error = machine_select( LIBSPECTRUM_MACHINE_PLUS2A );
      if( error ) {
	ui_error( UI_ERROR_ERROR,
		  "Loading a %s snapshot, but neither that nor %s available",
		  libspectrum_machine_name( machine ),
		  libspectrum_machine_name( LIBSPECTRUM_MACHINE_PLUS2A )    );
	return 1;
      } else {
	ui_error( UI_ERROR_INFO,
		  "Loading a %s snapshot, but that's not available. "
		  "Using %s instead",
		  libspectrum_machine_name( machine ),
		  libspectrum_machine_name( LIBSPECTRUM_MACHINE_PLUS2A )  );
      }
    } else {			/* Not trying a +3 snapshot */
      ui_error( UI_ERROR_ERROR,
		"Loading a %s snapshot, but that's not available",
		libspectrum_machine_name( machine ) );
    }
  }

  capabilities = libspectrum_machine_capabilities( machine_current->machine );

  A  = libspectrum_snap_a ( snap ); F  = libspectrum_snap_f ( snap );
  A_ = libspectrum_snap_a_( snap ); F_ = libspectrum_snap_f_( snap );

  BC  = libspectrum_snap_bc ( snap ); DE  = libspectrum_snap_de ( snap );
  HL  = libspectrum_snap_hl ( snap ); BC_ = libspectrum_snap_bc_( snap );
  DE_ = libspectrum_snap_de_( snap ); HL_ = libspectrum_snap_hl_( snap );

  IX = libspectrum_snap_ix( snap ); IY = libspectrum_snap_iy( snap );
  I  = libspectrum_snap_i ( snap ); R   = libspectrum_snap_r( snap );
  SP = libspectrum_snap_sp( snap ); PC = libspectrum_snap_pc( snap );

  IFF1 = libspectrum_snap_iff1( snap ); IFF2 = libspectrum_snap_iff2( snap );
  IM = libspectrum_snap_im( snap );

  z80.halted = libspectrum_snap_halted( snap );

  spectrum_ula_write( 0x00fe, libspectrum_snap_out_ula( snap ) );

  if( capabilities & LIBSPECTRUM_MACHINE_CAPABILITY_AY ) {
    ay_registerport_write( 0xfffd,
			   libspectrum_snap_out_ay_registerport( snap ) );
    for( i=0; i<16; i++ ) {
      machine_current->ay.registers[i] =
	libspectrum_snap_ay_registers( snap, i );
      sound_ay_write( i, machine_current->ay.registers[i], 0 );
    }
  }

  if( capabilities & LIBSPECTRUM_MACHINE_CAPABILITY_128_MEMORY )
    spec128_memoryport_write( 0x7ffd,
			      libspectrum_snap_out_128_memoryport( snap ) );

  if( capabilities & LIBSPECTRUM_MACHINE_CAPABILITY_PLUS3_MEMORY )
    specplus3_memoryport_write( 0x1ffd,
				libspectrum_snap_out_plus3_memoryport( snap ));

  if( capabilities & LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_MEMORY )
    scld_hsr_write( 0x00fd, libspectrum_snap_out_scld_hsr( snap ) );

  if( capabilities & LIBSPECTRUM_MACHINE_CAPABILITY_TIMEX_VIDEO )
    scld_dec_write( 0x00ff, libspectrum_snap_out_scld_dec( snap ) );

  tstates = libspectrum_snap_tstates( snap );

  z80.interrupts_enabled_at =
    libspectrum_snap_last_instruction_ei( snap ) ? tstates : -1;

  for( i=0; i<8; i++ ) {
    if( libspectrum_snap_pages( snap, i ) )
      memcpy( RAM[i], libspectrum_snap_pages( snap, i ), 0x4000 );
  }

  for( i=0; i<256; i++ ) {

    slt_length[i] = libspectrum_snap_slt_length( snap, i );

    if( slt_length[i] ) {

      slt[i] = malloc( slt_length[i] * sizeof( libspectrum_byte ) );
      if( slt[i] == NULL ) {
	for( j=0; j<i; j++ ) {
	  if( slt_length[j] ) { free( slt[j] ); slt_length[j] = 0; }
	  ui_error( UI_ERROR_ERROR, "Out of memory in snapshot_copy_from" );
	  return 1;
	}
      }

      memcpy( slt[i], libspectrum_snap_slt( snap, i ),
	      libspectrum_snap_slt_length( snap, i ) );
    }
  }

  if( libspectrum_snap_slt_screen( snap ) ) {

    slt_screen = malloc( 6912 * sizeof( libspectrum_byte ) );
    if( slt_screen == NULL ) {
      for( i=0; i<256; i++ ) {
	if( slt_length[i] ) { free( slt[i] ); slt_length[i] = 0; }
	ui_error( UI_ERROR_ERROR, "Out of memory in snapshot_copy_from" );
	return 1;
      }
    }

    memcpy( slt_screen, libspectrum_snap_slt_screen( snap ), 6912 );
    slt_screen_level = libspectrum_snap_slt_screen_level( snap );
  }
	  
  return 0;
}

int snapshot_write( const char *filename )
{
  libspectrum_snap *snap;
  unsigned char *buffer; size_t length;
  int flags;

  int error;

  error = libspectrum_snap_alloc( &snap ); if( error ) return error;

  error = snapshot_copy_to( snap ); if( error ) return error;

  flags = 0;
  length = 0;
  error = libspectrum_snap_write( &buffer, &length, &flags, snap,
				  LIBSPECTRUM_ID_SNAPSHOT_Z80, fuse_creator,
				  0 );
  if( error ) return error;

  if( flags & LIBSPECTRUM_FLAG_SNAPSHOT_MAJOR_INFO_LOSS ) {
    ui_error(
      UI_ERROR_INFO,
      "A large amount of information has been lost in conversion; the snapshot probably won't work"
    );
  } else if( flags & LIBSPECTRUM_FLAG_SNAPSHOT_MINOR_INFO_LOSS ) {
    ui_error(
      UI_ERROR_INFO,
      "Some information has been lost in conversion; the snapshot may not work"
    );
  }

  error = libspectrum_snap_free( snap );
  if( error ) { free( buffer ); return 1; }

  error = utils_write_file( filename, buffer, length );
  if( error ) { free( buffer ); return error; }

  return 0;

}

int snapshot_copy_to( libspectrum_snap *snap )
{
  int i,j;

  libspectrum_snap_set_machine( snap, machine_current->machine );

  libspectrum_snap_set_a  ( snap, A   ); libspectrum_snap_set_f  ( snap, F   );
  libspectrum_snap_set_a_ ( snap, A_  ); libspectrum_snap_set_f_ ( snap, F_  );

  libspectrum_snap_set_bc ( snap, BC  ); libspectrum_snap_set_de ( snap, DE  );
  libspectrum_snap_set_hl ( snap, HL  ); libspectrum_snap_set_bc_( snap, BC_ );
  libspectrum_snap_set_de_( snap, DE_ ); libspectrum_snap_set_hl_( snap, HL_ );

  libspectrum_snap_set_ix ( snap, IX  ); libspectrum_snap_set_iy ( snap, IY  );
  libspectrum_snap_set_i  ( snap, I   ); libspectrum_snap_set_r  ( snap, R   );
  libspectrum_snap_set_sp ( snap, SP  ); libspectrum_snap_set_pc ( snap, PC  );

  libspectrum_snap_set_iff1( snap, IFF1 );
  libspectrum_snap_set_iff2( snap, IFF2 );
  libspectrum_snap_set_im( snap, IM );

  libspectrum_snap_set_halted( snap, z80.halted );
  libspectrum_snap_set_last_instruction_ei(
    snap, z80.interrupts_enabled_at == tstates
  );

  libspectrum_snap_set_out_ula( snap, spectrum_last_ula );
  
  /* These won't necessarily be valid in some machine configurations, but
     this shouldn't cause anything to go wrong */
  libspectrum_snap_set_out_128_memoryport( snap,
					   machine_current->ram.last_byte );
  libspectrum_snap_set_out_ay_registerport(
    snap, machine_current->ay.current_register
  );

  for( i=0; i<16; i++ )
    libspectrum_snap_set_ay_registers(
      snap, i, machine_current->ay.registers[i]
    );

  libspectrum_snap_set_out_plus3_memoryport( snap,
					     machine_current->ram.last_byte2 );

  libspectrum_snap_set_out_scld_hsr( snap, scld_last_hsr );
  libspectrum_snap_set_out_scld_dec( snap, scld_last_dec.byte );

  libspectrum_snap_set_tstates( snap, tstates );

  for( i=0; i<8; i++ ) {
    if( RAM[i] != NULL ) {

      libspectrum_byte *buffer;

      buffer = malloc( 0x4000 * sizeof( libspectrum_byte ) );
      if( !buffer ) {
	for( j=0; j<i; j++ )
	  if( libspectrum_snap_pages( snap, j ) ) {
	    free( libspectrum_snap_pages( snap, j ) );
	    libspectrum_snap_set_pages( snap, j, NULL );
	  }
	ui_error( UI_ERROR_ERROR, "Out of memory in snapshot_copy_to" );
	return 1;
      }

      memcpy( buffer, RAM[i], 0x4000 );
      libspectrum_snap_set_pages( snap, i, buffer );
    }
  }

  for( i=0; i<256; i++ ) {

    libspectrum_snap_set_slt_length( snap, i, slt_length[i] );

    if( slt_length[i] ) {

      libspectrum_byte *buffer;

      buffer = malloc( slt_length[i] * sizeof(libspectrum_byte) );
      if( !buffer ) {

	for( j=0; j<8; j++ )
	  if( libspectrum_snap_pages( snap, j ) ) {
	    free( libspectrum_snap_pages( snap, j ) );
	    libspectrum_snap_set_pages( snap, j, NULL );
	  }
	for( j=0; j<i; j++ )
	  if( libspectrum_snap_slt( snap, j ) ) {
	    free( libspectrum_snap_slt( snap, j ) );
	    libspectrum_snap_set_slt_length( snap, j, 0 );
	  }
	ui_error( UI_ERROR_ERROR, "Out of memory in snapshot_copy_to" );
	return 1;

      }

      memcpy( buffer, slt[i], slt_length[i] );
      libspectrum_snap_set_slt( snap, i, buffer );
    }
  }

  if( slt_screen ) {
 
    libspectrum_byte *buffer;

    buffer = malloc( 6912 * sizeof( libspectrum_byte ) );

    if( !buffer ) {

      for( i=0; i<8; i++ )
	if( libspectrum_snap_pages( snap, i ) ) {
	  free( libspectrum_snap_pages( snap, i ) );
	  libspectrum_snap_set_pages( snap, i, NULL );
	}
      for( i=0; i<256; i++ )
	if( libspectrum_snap_slt( snap, i ) ) {
	  free( libspectrum_snap_slt( snap, i ) );
	  libspectrum_snap_set_slt_length( snap, i, 0 );
	}
      ui_error( UI_ERROR_ERROR, "Out of memory in snapshot_copy_to" );
      return 1;

    }

    memcpy( buffer, slt_screen, 6912 );
    libspectrum_snap_set_slt_screen( snap, buffer );
    libspectrum_snap_set_slt_screen_level( snap, slt_screen_level );
  }
    
  return 0;
}
