/* trdos.c: Routines for handling the Betadisk interface
   Copyright (c) 2002-2003 Dmitry Sanarin, Fredrick Meunier, Philip Kendall

   $Id: trdos.c,v 1.13 2003/07/11 11:03:25 pak21 Exp $

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

   Author contact information:

   Philip: pak21-fuse@srcf.ucam.org
     Post: 15 Crescent Road, Wokingham, Berks, RG40 2DB, England

   Dmitry: sdb386@hotmail.com
   Fred: fredm@spamcop.net
*/

#include <config.h>

#include <errno.h>
#include <stdlib.h>
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>
#include <limits.h>
#include <sys/stat.h>

#include <libspectrum.h>

#include "compat.h"
#include "event.h"
#include "machine.h"
#include "spectrum.h"
#include "trdos.h"
#include "ui/ui.h"
#include "utils.h"

#define TRDOS_DISC_SIZE 655360

typedef struct 
{
  int disc_ready;		/* Is this disk present? */

  char filename[PATH_MAX];	/* The filename we used to open this disk */
  int fd;			/* The fd will we use to access this disk */

  int ro;			/* True if we have read-only access to this
				   disk */

  libspectrum_byte trk;

} discs_type;

#ifdef WORDS_BIGENDIAN

typedef struct
{
  unsigned b7  : 1;  /* This bit reflects the status of the Motor On output */
  unsigned b6  : 1;  /* disk is write-protected */
  unsigned b5  : 1;  /* When set, this bit indicates that the Motor Spin-Up
                        sequence has completed (6 revolutions) on type 1
                        commands. Type 2 & 3 commands, this bit indicates
                        record Type. 0 = Data Mark, 1 = Deleted Data Mark. */
  unsigned b4  : 1;  /* When set, it indicates that the desired track, sector,
                        or side were not found. This bit is reset when
                        updated. */
  unsigned b3  : 1;  /* If this is set, an error is found in one or more ID
                        fields; otherwise it indicates error in data field.
                        This bit is reset when updated. */
  unsigned b2  : 1;  /* When set, it indicates the computer did not respond to
                        DRQ in one byte time. This bit is reset to zero when
                        update. On type 1 commands, this bit reflects the
                        status of the TRACK 00 pin. */
  unsigned b1  : 1;  /* This bit is a copy of the DRQ output. When set, it
                        indicates the DR is full on a Read Operation or the DR
                        is empty on a write operation. This bit is reset to
                        zero when updated. On type 1 commands, this bit
                        indicates the status of the index pin. */
  unsigned b0  : 1;  /* When set, command is under execution. When reset, no
                        command is under execution. */
} rs_type;

#else				/* #ifdef WORDS_BIGENDIAN */

typedef struct
{
  unsigned b0  : 1;  /* When set, command is under execution. When reset, no
                        command is under execution. */
  unsigned b1  : 1;  /* This bit is a copy of the DRQ output. When set, it
                        indicates the DR is full on a Read Operation or the DR
                        is empty on a write operation. This bit is reset to
                        zero when updated. On type 1 commands, this bit
                        indicates the status of the index pin. */
  unsigned b2  : 1;  /* When set, it indicates the computer did not respond to
                        DRQ in one byte time. This bit is reset to zero when
                        update. On type 1 commands, this bit reflects the
                        status of the TRACK 00 pin. */
  unsigned b3  : 1;  /* If this is set, an error is found in one or more ID
                        fields; otherwise it indicates error in data field.
                        This bit is reset when updated. */
  unsigned b4  : 1;  /* When set, it indicates that the desired track, sector,
                        or side were not found. This bit is reset when
                        updated. */
  unsigned b5  : 1;  /* When set, this bit indicates that the Motor Spin-Up
                        sequence has completed (6 revolutions) on type 1
                        commands. Type 2 & 3 commands, this bit indicates
                        record Type. 0 = Data Mark, 1 = Deleted Data Mark. */
  unsigned b6  : 1;  /* disk is write-protected */
  unsigned b7  : 1;  /* This bit reflects the status of the Motor On output */
} rs_type;

#endif				/* #ifdef WORDS_BIGENDIAN */

# define CurrentDiscNum (last_vg93_system & 3)
# define CurrentDisk discs[CurrentDiscNum]

static int busy = 0;
static int index_impulse = 0;

static int towrite = 0;
static int towriteaddr;

static int pointer;
static int side;

static libspectrum_byte track[ TRDOS_DISC_SIZE ];
static libspectrum_byte *toread;
static unsigned int toread_num = 0;
static unsigned int toread_position = 0;
static libspectrum_byte six_bytes[6];

static int vg_spin;
static int vg_direction; /* 0 - spindlewards 1 - rimwards */
static libspectrum_byte vg_reg_trk; /* The last byte sent to VG93 track reg */
static libspectrum_byte vg_reg_sec; /* The last byte sent to VG93 sector reg */
static libspectrum_byte vg_reg_dat; /* The last byte sent to VG93 data reg */
static libspectrum_byte vg_portFF_in;
static libspectrum_byte last_vg93_system = 0; /* Last byte sent to VG93 system
						 port */

union
{
  libspectrum_byte byte;
  rs_type bit;
} vg_rs; 

static discs_type discs[4];

int trdos_active=0;

/* The template used for naming the results of the SCL->TRD conversion */
#define SCL_TMP_FILE_TEMPLATE "fuse.scl.XXXXXX"

static int Scl2Trd( const char *oldname, int TRD );
static int insert_scl( trdos_drive_number which, const char *filename );
static int insert_trd( trdos_drive_number which, const char *filename );
static libspectrum_dword lsb2dw( libspectrum_byte *mem );
static void dw2lsb( libspectrum_byte *mem, libspectrum_dword value );

int
trdos_init( void )
{
  discs[0].disc_ready = 0;
  discs[1].disc_ready = 0;
  discs[2].disc_ready = 0;
  discs[3].disc_ready = 0;

  return 0;
}

void
trdos_reset( void )
{
  trdos_active = 0;

  trdos_event_index( 0 );

  /* We can eject disks only if they are currently present */
  ui_menu_activate_media_disk_eject( TRDOS_DRIVE_A,
				     discs[ TRDOS_DRIVE_A ].disc_ready );
  ui_menu_activate_media_disk_eject( TRDOS_DRIVE_B,
				     discs[ TRDOS_DRIVE_B ].disc_ready );
}

void
trdos_end( void )
{
  ;
}

static
void trdos_update_index_impulse( void )
{
  if ( vg_spin ) {
    vg_rs.bit.b1 = 0;
    if ( CurrentDisk.disc_ready && index_impulse ) {
      vg_rs.bit.b1 = 1;
    }
  }
}

libspectrum_byte
trdos_sr_read( libspectrum_word port GCC_UNUSED )
{
  trdos_update_index_impulse();

  if ( !CurrentDisk.disc_ready ) {
    return( 0x80 ); /* No disk in drive */
  }

  return( vg_rs.byte );
}

libspectrum_byte
trdos_tr_read( libspectrum_word port GCC_UNUSED )
{
  trdos_update_index_impulse();

  return( vg_reg_trk );
}

void
trdos_tr_write( libspectrum_word port GCC_UNUSED, libspectrum_byte b )
{
  vg_reg_trk = b;
}

libspectrum_byte
trdos_sec_read( libspectrum_word port GCC_UNUSED )
{
  trdos_update_index_impulse();

  return( vg_reg_sec );
}

void
trdos_sec_write( libspectrum_word port GCC_UNUSED, libspectrum_byte b )
{
  vg_reg_sec = b;
}

libspectrum_byte
trdos_dr_read( libspectrum_word port GCC_UNUSED )
{
  trdos_update_index_impulse();

  if ( toread_position >= toread_num )
    return ( vg_reg_dat );

  vg_reg_dat = toread[toread_position];

  if ( ( ( toread_position & 0x00ff ) == 0 ) && toread_position != 0 )
    vg_reg_sec++;

  if ( vg_reg_sec > 16 ) vg_reg_sec = 1;

  toread_position++;

  if ( toread_position == toread_num ) {
    vg_portFF_in = 0x80;

    vg_rs.byte = 0;
  } else {
    vg_portFF_in = 0x40;

    vg_rs.bit.b0 = 1; /*  Busy */
    vg_rs.bit.b1 = 1; /*  DRQ copy */
  }

  return( vg_reg_dat );
}

void
trdos_dr_write( libspectrum_word port GCC_UNUSED, libspectrum_byte b )
{
  vg_reg_dat = b;

  if( towrite == 0 ) return;

  if( lseek( CurrentDisk.fd, towriteaddr, SEEK_SET ) == -1 ) {
    towrite = 0;

    ui_error( UI_ERROR_ERROR,
              "trdos_dr_write: seeking in '%s' failed: %s",
	      CurrentDisk.filename, strerror( errno ) );
    return;
  }

  write( CurrentDisk.fd, &b, 1 );

  towrite--;
  towriteaddr++;

  if( towrite == 0 ) {
    vg_portFF_in = 0x80;
    vg_rs.byte = 0;
  } else {
    vg_portFF_in = 0x40;
  }
}

libspectrum_byte
trdos_sp_read( libspectrum_word port GCC_UNUSED )
{
  trdos_update_index_impulse();

  if ( busy == 1 ) {
    return( vg_portFF_in &~ 0x40 );
  }

  return( vg_portFF_in );
}

void
trdos_sp_write( libspectrum_word port GCC_UNUSED, libspectrum_byte b )
{
  last_vg93_system = b;
}

static int
insert_trd( trdos_drive_number which, const char *filename )
{
  int fil;
  libspectrum_byte *temp;

  discs[which].ro = 0;

  fil = open( filename, O_RDWR | O_BINARY );
  if( fil == -1 ) {

    /* If we couldn't open the file read-write, try opening it read-only */
    if( errno == EACCES ) {
      discs[which].ro = 1;
      fil = open( filename, O_RDONLY | O_BINARY );
    }
  }

  if( fil == -1 ) {

    ssize_t written;

    /* If we got an error other than 'file doesn't exist', don't try
       anything else */
    if( errno != ENOENT ) {
      ui_error( UI_ERROR_ERROR, "Couldn't open '%s': %s", filename,
		strerror( errno ) );
      return 1;
    }

    ui_error( UI_ERROR_INFO, "No such file '%s'; creating it", filename );
    discs[which].ro = 0;

    /* FIXME: NFS locking problems */
    fil = open( filename, O_CREAT | O_EXCL | O_RDWR | O_BINARY );
    if( fil == -1 ) {
      ui_error( UI_ERROR_ERROR, "Couldn't open '%s': %s", filename,
		strerror( errno ) );
      return 1;
    }

    temp = malloc( TRDOS_DISC_SIZE );
    if( !temp ) {
      ui_error( UI_ERROR_ERROR, "Out of memory at %s:%d", __FILE__, __LINE__ );
      close( fil );
      return 1;
    }

    memset( temp, TRDOS_DISC_SIZE, 0 );

    written = write( fil, temp, TRDOS_DISC_SIZE );
    if( written != TRDOS_DISC_SIZE ) {
      if( written == -1 ) {
	ui_error( UI_ERROR_ERROR, "Error writing to '%s': %s", filename,
		  strerror( errno ) );
      } else {
	ui_error( UI_ERROR_ERROR,
		  "Only wrote %lu bytes out of %d to '%s'",
		  (unsigned long)written, TRDOS_DISC_SIZE, filename );
      }
      close( fil );
      return 1;
    }

  }

  discs[which].fd = fil;
  strcpy( discs[which].filename, filename );
  discs[which].disc_ready = 1;

  return 0;
}

static int
insert_scl( trdos_drive_number which, const char *filename )
{
  const char *temp_path; size_t length;
  char* trd_template;
  int ret;

  temp_path = utils_get_temp_path();

  length = strlen( temp_path ) + strlen( SCL_TMP_FILE_TEMPLATE ) + 1;

  trd_template = malloc( length );
  if( !trd_template ) {
    ui_error( UI_ERROR_ERROR, "out of memory at %s:%d", __FILE__, __LINE__ );
    return 1;
  }

  snprintf( trd_template, length, "%s%s", temp_path, SCL_TMP_FILE_TEMPLATE );

  discs[ which ].disc_ready = 0;

  discs[ which ].fd = mkstemp( trd_template );
  if( discs[ which ].fd == -1 ) {
    ui_error( UI_ERROR_ERROR, "couldn't get a temporary filename: %s",
	      strerror( errno ) );
    free( trd_template );
    return 1;
  }

  /* Unlink the file so it will be removed when the fd is closed */
  unlink( trd_template );

  if( ( ret = Scl2Trd( filename, discs[ which ].fd ) ) ) {
    close( discs[ which ].fd );
    free( trd_template );
    return ret;
  }

  strcpy( discs[which].filename, trd_template );
  discs[which].disc_ready = 1;
  discs[which].ro = 1;

  free( trd_template );

  return 0;
}

int
trdos_disk_insert( trdos_drive_number which, const char *filename )
{
  int error;
  utils_file file;
  libspectrum_id_t type;
  libspectrum_class_t class;

  if( discs[ which ].disc_ready ) {
    if( trdos_disk_eject( which ) ) return 1;
  }

  if( utils_read_file( filename, &file ) ) return 1;

  if( libspectrum_identify_file( &type, filename, file.buffer, file.length ) ) {
    utils_close_file( &file );
    return 1;
  }

  error = libspectrum_identify_class( &class, type );
  if( error ) return 1;

  if( class != LIBSPECTRUM_CLASS_DISK_TRDOS ) {
    ui_error( UI_ERROR_ERROR,
              "trdos_disk_insert: file `%s' is not a TR-DOS disk", filename );
    return 1;
  }

  if( utils_close_file( &file ) ) return 1;

  switch( type ) {
  case LIBSPECTRUM_ID_DISK_SCL:
    error = insert_scl( which, filename );
    break;
  case LIBSPECTRUM_ID_DISK_TRD:
    error = insert_trd( which, filename );
    break;
  default:
    ui_error( UI_ERROR_ERROR,
              "trdos_disk_insert: file `%s' is an unsupported TR-DOS disk", filename );
    error = 1;
  }

  if( error ) return error;

  /* Set the `eject' item active */
  ui_menu_activate_media_disk_eject( which, 1 );

  return 0;
}

int
trdos_disk_eject( trdos_drive_number which )
{
  if( close( discs[which].fd ) == -1 ) {
    ui_error( UI_ERROR_ERROR, "Error closing '%s': %s", discs[which].filename,
	      strerror( errno ) );
    return 1;
  }

  discs[which].disc_ready = 0;

  /* Set the `eject' item inactive */
  ui_menu_activate_media_disk_eject( which, 0 );

  return 0;
}

int
trdos_event_cmd_done( libspectrum_dword last_tstates GCC_UNUSED )
{
  busy = 0;

  return 0;
}

int
trdos_event_index( libspectrum_dword last_tstates )
{
  int error;
  int next_tstates;
  static int num_calls = 0;

  if( num_calls == 0 ) {
    /* schedule next call in 20ms */
    next_tstates = 20 * machine_current->timings.processor_speed / 1000;
    index_impulse = 1;
    num_calls = 1;
  } else {
    /* schedule next call in 180ms */
    next_tstates = 180 * machine_current->timings.processor_speed / 1000;
    index_impulse = 0;
    num_calls = 0;
  }

  error = event_add( last_tstates + next_tstates, EVENT_TYPE_TRDOS_INDEX );
  if( error ) return error;

  return 0;
}

static void
vg_seek_delay( libspectrum_byte dst_track ) 
{
  int error;

  if ( ( dst_track - CurrentDisk.trk ) > 0 )
    vg_direction = 1;
  if ( ( dst_track - CurrentDisk.trk ) < 0 )
    vg_direction = 0;

  busy = 1;

  if ( !CurrentDisk.disc_ready ) vg_portFF_in = 0x80;
  else vg_portFF_in = 0x80;

  CurrentDisk.trk = dst_track;
  vg_reg_trk = dst_track;

  /* schedule event */
  error = event_add( tstates +
                     machine_current->timings.processor_speed / 1000 * 20 *
                     abs( dst_track - CurrentDisk.trk ),
                     EVENT_TYPE_TRDOS_CMD_DONE );
}

static void
vg_setFlagsSeeks( void )
{
  vg_rs.bit.b0 = 0;

  if ( CurrentDisk.trk == 0 )
    vg_rs.bit.b2 = 1;
  else
    vg_rs.bit.b2 = 0;

  vg_rs.bit.b3 = 0;
  vg_rs.bit.b4 = 0;
  vg_rs.bit.b6 = CurrentDisk.ro;
  vg_rs.bit.b7 = CurrentDisk.disc_ready;
}

void
trdos_cr_write( libspectrum_word port GCC_UNUSED, libspectrum_byte b )
{
  int error;

  if ( last_vg93_system & 0x10 )
    side = 0;
  else
    side = 1;


  if ( (b & 0xF0) == 0xD0 ) { /*  interrupt */
    vg_portFF_in = 0x80;
    vg_setFlagsSeeks();

    return;
  }	

  if ( (b & 0xF0) == 0x00 ) { /*  seek trk0 AKA Initialisation */
    vg_seek_delay( 0 );
    vg_rs.bit.b5 = b & 8 ? 1 : 0;
    vg_setFlagsSeeks();
    if ( b & 8 ) vg_spin = 1;

    return;
  }

  if ( (b & 0xF0) == 0x10 ) { /*  seek track */
    vg_seek_delay( vg_reg_dat );
    vg_rs.bit.b5 = b & 8 ? 1 : 0;
    vg_setFlagsSeeks();
    if ( b & 8 ) vg_spin = 1;

    return;
  }

  if ( (b & 0xE0) == 0x40 ) { /*  fwd */
    vg_direction = 1;
    b = 0x20; /*  step */
  }
  if ( (b & 0xE0) == 0x60 ) { /*  back */
    vg_direction = 0;
    b = 0x20; /*  step */
  }
  if ( (b & 0xE0) == 0x20 ) { /*  step */
    if ( vg_direction )
      vg_seek_delay( ++CurrentDisk.trk );
    else
      vg_seek_delay( --CurrentDisk.trk );

    vg_rs.bit.b5 = 1;
    vg_setFlagsSeeks();
    vg_spin = 1;

    return;
  }

  if ( (b & 0xE0) == 0x80 ) { /*  readsec */
    vg_rs.byte = 0x81;
    vg_portFF_in = 0x40;
    vg_spin = 0;

    if ( !CurrentDisk.disc_ready ) {
      vg_rs.byte = 0x90;
      vg_portFF_in = 0x80; 

      ui_error( UI_ERROR_INFO, "No disk in drive %c", 'A' + CurrentDiscNum );

      return;
    }

    if ( ( vg_reg_sec == 0 ) || ( vg_reg_sec > 16 ) ) {
      vg_rs.byte |= 0x10; /*  sector not found */
      vg_portFF_in = 0x80;

      ui_error( UI_ERROR_INFO, "((vg_reg_sec==0)||(vg_reg_sec>16))" );

      return;
    }

    pointer = ( CurrentDisk.trk * 2 + side ) * 256 * 16 + ( vg_reg_sec - 1 ) *
              256;

    if( lseek( CurrentDisk.fd, pointer, SEEK_SET) == -1 ) {
      ui_error( UI_ERROR_ERROR,
		"trdos_cr_write: seeking in '%s' failed: %s",
		CurrentDisk.filename, strerror( errno ) );

      vg_rs.byte |= 0x10; /*  sector not found */
      vg_portFF_in = 0x80;

      return;
    }

    if ( b & 0x10 ) {
      ui_error( UI_ERROR_ERROR,
                "Unimplemented multi sector read:read vg_reg_sec=%d",
                vg_reg_sec );

      read( CurrentDisk.fd, track, 256 * ( 17 - vg_reg_sec ) );

      lseek( CurrentDisk.fd,
	     -256 * ( ( 17 - vg_reg_sec ) + ( vg_reg_sec - 1 ) ),
             SEEK_CUR );
      read( CurrentDisk.fd, track, 256 * ( vg_reg_sec - 1 ) );
 
      toread = track;
      toread_num = 256 * ( 16 );
      toread_position =  0;

      /* vg_portFF_in=0x80; */
      /* todo : Eto proverit' !!! */
    } else {
      if( read( CurrentDisk.fd, track, 256 ) == 256 ) {

        toread = track;
        toread_num = 256;
        toread_position = 0;

        /* schedule event */
        busy = 1;
        error = event_add( tstates + machine_current->timings.processor_speed
                                     / 1000 * 30,
                           EVENT_TYPE_TRDOS_CMD_DONE );
      } else {

        vg_rs.byte |= 0x10; /*  sector not found */
        vg_portFF_in = 0x80;
        
        ui_error( UI_ERROR_ERROR, "Error reading from '%s'",
		  CurrentDisk.filename );
      }
    }

    return;
  }

  if ( (b & 0xFB) == 0xC0 ) { /*  read adr */
    vg_rs.byte = 0x81;
    vg_portFF_in = 0x40;

    six_bytes[0] = CurrentDisk.trk;
    six_bytes[1] = 0; 
    six_bytes[2] = vg_reg_sec;
    six_bytes[3] = 1;
    six_bytes[4] = 0; /*  todo : crc !!! */
    six_bytes[5] = 0; /*  todo : crc !!! */

    toread = six_bytes;
    toread_num = 6;
    toread_position = 0;

    /* schedule event */
    busy = 1;
    vg_spin = 0;

    error = event_add( tstates + machine_current->timings.processor_speed
                                 / 1000 * 30,
                       EVENT_TYPE_TRDOS_CMD_DONE );

    return;
  }	

  if ( (b & 0xE0) == 0xA0 ) { /*  writesec */
    vg_rs.byte = 0x81;
    vg_portFF_in = 0x40;
    vg_spin = 0;


    if ( CurrentDisk.ro ) {
      vg_rs.byte = 0x60;
      vg_portFF_in = 0x80;
      return;
    }

    if ( !CurrentDisk.disc_ready ) {
      vg_rs.byte = 0x90;
      vg_portFF_in = 0x80; 

      ui_error( UI_ERROR_ERROR, "No disk" );

      return;
    }

    if ( ( vg_reg_sec == 0 ) || ( vg_reg_sec > 16 ) ) {
      vg_rs.byte |= 0x10; /*  sector not found */
      vg_portFF_in = 0x80;

      ui_error( UI_ERROR_ERROR, "((vg_reg_sec==0)||(vg_reg_sec>16))" );

      return;
    }

    towriteaddr = ( CurrentDisk.trk * 2 + side ) * 256 * 16
                    + ( vg_reg_sec - 1 ) * 256;
    towrite = 256;

    vg_spin = 0;

    return;		
  }		
}

#define TRD_NAMEOFFSET 0x08F5
#define TRD_DIRSTART 0x08E2
#define TRD_DIRLEN 32
#define TRD_MAXNAMELENGTH 8
#define BLOCKSIZE 10240

typedef union {
#ifdef WORDS_BIGENDIAN
  struct { libspectrum_byte b3, b2, b1, b0; } b;
#else
  struct { libspectrum_byte b0, b1, b2, b3; } b;
#endif
  libspectrum_dword dword;
} lsb_dword;

static libspectrum_dword 
lsb2dw( libspectrum_byte *mem )
{
  return ( mem[0] + ( mem[1] * 256 ) + ( mem[2] * 256 * 256 )
          + ( mem[3] * 256 * 256 * 256 ) );
}

static void
dw2lsb( libspectrum_byte *mem, libspectrum_dword value )
{
  lsb_dword ret;

  ret.dword = value;

  mem[0] = ret.b.b0;
  mem[1] = ret.b.b1;
  mem[2] = ret.b.b2;
  mem[3] = ret.b.b3;
}

static int
Scl2Trd( const char *oldname, int TRD )
{
  int SCL, i;

  libspectrum_byte *TRDh;

  libspectrum_byte *trd_free;
  libspectrum_byte *trd_fsec;
  libspectrum_byte *trd_ftrk;
  libspectrum_byte *trd_files;
  libspectrum_byte size;

  char signature[8];
  libspectrum_byte blocks;
  libspectrum_byte headers[256][14];
  void *tmpscl;
  libspectrum_dword left;
  libspectrum_dword fptr;
  size_t x;

  ssize_t written;

  libspectrum_byte template[34] = { 
    0x01, 0x16, 0x00, 0xF0,
    0x09, 0x10, 0x00, 0x00,
    0x20, 0x20, 0x20, 0x20,
    0x20, 0x20, 0x20, 0x20,
    0x20, 0x00, 0x00, 0x64,
    0x69, 0x73, 0x6B, 0x6E,
    0x61, 0x6D, 0x65, 0x00,
    0x00, 0x00, 0x46, 0x55
  };

  libspectrum_byte *mem;

  mem = malloc( BLOCKSIZE );
  if( !mem ) {
    ui_error( UI_ERROR_ERROR, "Out of memory at %s:%d", __FILE__, __LINE__ );
    return 1;
  }

  memset( mem, 0, BLOCKSIZE );

  memcpy( &mem[TRD_DIRSTART], template, TRD_DIRLEN );
  strncpy( &mem[TRD_NAMEOFFSET], "Fuse", TRD_MAXNAMELENGTH );

  written = write( TRD, mem, BLOCKSIZE );
  if( written != BLOCKSIZE ) {
    ui_error( UI_ERROR_ERROR, "Error writing to temporary TRD file" );
    free( mem );
    return 1;
  }

  memset( mem, 0, BLOCKSIZE );

  for( i = 0; i < 63; i++ ) {

    written = write( TRD, mem, BLOCKSIZE );
    if( written != BLOCKSIZE ) {
      ui_error( UI_ERROR_ERROR, "Error writing to temporary TRD file" );
      free( mem );
      return 1;
    }
  }

  free( mem );

  if( lseek( TRD, 0, SEEK_SET ) == -1 ) {
    ui_error( UI_ERROR_ERROR, "Error seeking in temporary TRD file" );
    return 1;
  }
  
  TRDh = malloc( 4096 );
  if( !TRDh ) {
    ui_error( UI_ERROR_ERROR, "Out of memory at %s:%d", __FILE__, __LINE__ );
    return 1;
  }

  if( read( TRD, TRDh, 4096 ) != 4096 ) {
    ui_error( UI_ERROR_ERROR, "Error reading from temporary TRD file" );
    free( TRDh );
    return 1;
  }

  trd_free = TRDh + 0x8E5;
  trd_files = TRDh + 0x8E4;
  trd_fsec = TRDh + 0x8E1;
  trd_ftrk = TRDh + 0x8E2;

  if( ( SCL = open( oldname, O_RDONLY | O_BINARY ) ) == -1 ) {
    ui_error( UI_ERROR_ERROR, "Can't open SCL file '%s': %s", oldname,
	      strerror( errno ) );
    free( TRDh );
    return 1;
  }

  if( read( SCL, &signature, 8 ) != 8 ) {
    ui_error( UI_ERROR_ERROR, "Error reading from '%s'", oldname );
    close( SCL ); free( TRDh );
    return 1;
  }

  if( strncasecmp( signature, "SINCLAIR", 8 ) ) {
    ui_error( UI_ERROR_ERROR, "SCL file '%s' has the wrong signature",
	      oldname );
    close( SCL ); free( TRDh );
    return 1;
  }

  if( read( SCL, &blocks, 1 ) != 1 ) {
    ui_error( UI_ERROR_ERROR, "Error reading from '%s'", oldname );
    close( SCL ); free( TRDh );
    return 1;
  }

  for( x = 0; x < blocks; x++ ) {
    if( read( SCL, &(headers[x][0]), 14 ) != 14 ) {
      ui_error( UI_ERROR_ERROR, "Error reading from '%s'", oldname );
      close( SCL ); free( TRDh );
      return 1;
    }
  }

  for( x = 0; x < blocks; x++ ) {
    size = headers[x][13];
    if( lsb2dw(trd_free) < size ) {
      ui_error( UI_ERROR_ERROR,
                "File is too long to fit in the image *trd_free=%u < size=%u",
                lsb2dw( trd_free ), size );
      goto Finish;
    }

    if( *trd_files > 127 ) {
      ui_error( UI_ERROR_ERROR, "Image is full" );
      goto Finish;
    }

    memcpy( TRDh + *trd_files * 16, headers[x], 14 );

    memcpy( TRDh + *trd_files * 16 + 0x0E, trd_fsec, 2 );

    tmpscl = malloc( 32000 );
    if( !tmpscl ) {
      ui_error( UI_ERROR_ERROR, "Out of memory at %s:%d", __FILE__, __LINE__ );
      goto Finish;
    }

    left = ( headers[x][13] ) * (libspectrum_dword)256;
    fptr = (*trd_ftrk) * (libspectrum_dword)4096 +
           (*trd_fsec) * (libspectrum_dword)256;
    
    if( lseek( TRD, fptr, SEEK_SET ) == -1 ) {
      ui_error( UI_ERROR_ERROR, "Error seeking in temporary TRD file: %s",
		strerror( errno ) );
      goto Finish;
    }

    while ( left > 32000 ) {
      read( SCL, tmpscl, 32000 );
      write( TRD, tmpscl, 32000 );
      left -= 32000;
    }

    read( SCL, tmpscl, left );
    write( TRD, tmpscl, left );

    free( tmpscl );

    (*trd_files)++;

    dw2lsb( trd_free, lsb2dw(trd_free) - size );

    while ( size > 15 ) {
      (*trd_ftrk)++;
      size -= 16;
    }

    (*trd_fsec) += size;
    while ( (*trd_fsec) > 15 ) {
      (*trd_fsec) -= 16;
      (*trd_ftrk)++;
    }
  }

Finish:
  close( SCL );
  lseek( TRD, 0L, SEEK_SET );
  write( TRD, TRDh, 4096 );
  free( TRDh );

  return 0;
}
