/* 
   ipmi-lan-interface-udm.c - IPMI UDM LAN Interface

   Copyright (C) 2003, 2004, 2005 FreeIPMI Core Team

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.

*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#ifdef STDC_HEADERS
#include <string.h>
#endif /* STDC_HEADERS */
#include <sys/socket.h>
#include <netinet/in.h>
#if TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else /* !TIME_WITH_SYS_TIME */
#if HAVE_SYS_TIME_H
#include <sys/time.h>
#else /* !HAVE_SYS_TIME_H */
#include <time.h>
#endif /* !HAVE_SYS_TIME_H */
#endif  /* !TIME_WITH_SYS_TIME */
#if HAVE_UNISTD_H
#include <unistd.h>
#endif /* HAVE_UNISTD_H */
#include <assert.h>
#include <errno.h>

#include "freeipmi/udm/ipmi-messaging-support-cmds-udm.h"
#include "freeipmi/udm/ipmi-lan-interface-udm.h"
#include "freeipmi/ipmi-authentication-type-spec.h"
#include "freeipmi/ipmi-channel-spec.h"
#include "freeipmi/ipmi-debug.h"
#include "freeipmi/ipmi-lan.h"
#include "freeipmi/ipmi-lan-interface.h"
#include "freeipmi/ipmi-lan-utils.h"
#include "freeipmi/ipmi-privilege-level-spec.h"
#include "freeipmi/rmcp.h"

#include "ipmi-udm-device.h"

#include "freeipmi-portability.h"
#include "udm-err-wrappers.h"
#include "udm-fiid-wrappers.h"

fiid_template_t tmpl_lan_raw_hdr = 
  {
    {2, "lun", FIID_FIELD_REQUIRED | FIID_FIELD_LENGTH_FIXED},
    {6, "net_fn", FIID_FIELD_REQUIRED | FIID_FIELD_LENGTH_FIXED},
    {0, "", 0}
  };

fiid_template_t tmpl_lan_raw =
  {
    {8192, "raw_data", FIID_FIELD_OPTIONAL | FIID_FIELD_LENGTH_VARIABLE},
    {0, "", 0}
  };

#define IPMI_LAN_RQ_SEQ_INC(__rq_seq) ((__rq_seq) = (((__rq_seq) + 1) % (IPMI_LAN_REQUESTER_SEQUENCE_NUMBER_MAX + 1)))

#define IPMI_LAN_BACKOFF_COUNT 2

int8_t 
ipmi_lan_open_session (ipmi_device_t dev)
{
  struct ipmi_device local_dev;
  fiid_obj_t obj_cmd_rs = NULL;
  uint64_t supported_authentication_type = 0;
  uint64_t temp_session_id = 0;
  uint8_t challenge_string[IPMI_CHALLENGE_STRING_LENGTH];
  uint64_t temp_session_sequence_number = 0;
  uint32_t initial_outbound_sequence_number = 0;
  unsigned int seedp;
  int8_t rv = -1;
  uint64_t val;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_INVALID_PARAMETERS (IPMI_AUTHENTICATION_TYPE_VALID(dev->io.outofband.authentication_type));
  
  dev->io.outofband.rq_seq = 0;
  
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rs, tmpl_cmd_get_channel_authentication_capabilities_rs);
  
  /* Must create a local dev w/ a different authentication type, b/c
   * we are too early in the protocol.
   */
  memcpy(&local_dev, dev, sizeof(struct ipmi_device));
  local_dev.io.outofband.authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;

  dev->io.outofband.lan_session_state = IPMI_LAN_SESSION_STATE_GET_CHANNEL_AUTHENTICAPTION_CAPABILITIES;
  if (ipmi_cmd_get_channel_authentication_capabilities (&local_dev, 
                                                        IPMI_CHANNEL_NUMBER_CURRENT_CHANNEL,
                                                        dev->io.outofband.privilege_level,
                                                        obj_cmd_rs) < 0)
    {
      dev->errnum = local_dev.errnum;
      goto cleanup;
    }

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
			    "authentication_status.per_message_authentication",
			    &val);
  dev->io.outofband.per_msg_auth_disabled = val;

  switch (dev->io.outofband.authentication_type)
    {
    case IPMI_AUTHENTICATION_TYPE_NONE:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.none", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_MD2:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.md2", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_MD5:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.md5", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_STRAIGHT_PASSWORD_KEY:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.straight_password_key", 
				&supported_authentication_type);
      break;
    case IPMI_AUTHENTICATION_TYPE_OEM_PROP:
      UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
				"authentication_type.oem_prop", 
				&supported_authentication_type);
      break;
    }

  if (!supported_authentication_type)
    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_AUTHENTICATION_TYPE);

  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_get_session_challenge_rs);

  /* Must create a local dev w/ a different authentication type, b/c
   * we are too early in the protocol.
   */
  memcpy(&local_dev, dev, sizeof(struct ipmi_device));
  local_dev.io.outofband.authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;

  dev->io.outofband.lan_session_state = IPMI_LAN_SESSION_STATE_GET_SESSION_CHALLENGE;
  if (ipmi_cmd_get_session_challenge (&local_dev, 
                                      dev->io.outofband.authentication_type,
                                      dev->io.outofband.username,
                                      IPMI_MAX_USER_NAME_LENGTH,
                                      obj_cmd_rs) < 0)
    {
      dev->errnum = local_dev.errnum;
      if (dev->errnum == IPMI_ERR_BAD_COMPLETION_CODE
	  && (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_INVALID_USERNAME) == 1
	      || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NULL_USERNAME_NOT_ENABLED) == 1))
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_USERNAME);
      UDM_ERR_LOG_CLEANUP(0);
    }

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
			    "temp_session_id", 
			    &temp_session_id);
  
  dev->io.outofband.session_id = temp_session_id;
  UDM_FIID_OBJ_GET_DATA_CLEANUP (obj_cmd_rs, 
				 "challenge_string", 
				 challenge_string,
				 IPMI_CHALLENGE_STRING_LENGTH);

  memcpy (dev->io.outofband.challenge_string, 
	  challenge_string, 
	  IPMI_CHALLENGE_STRING_LENGTH);

  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_activate_session_rs);

  /* Random number generation */
  seedp = (unsigned int) clock () + (unsigned int) time (NULL);
  srand (seedp);
  initial_outbound_sequence_number = rand_r (&seedp);

  dev->io.outofband.lan_session_state = IPMI_LAN_SESSION_STATE_ACTIVATE_SESSION;
  if (ipmi_cmd_activate_session (dev, 
                                 dev->io.outofband.authentication_type,
                                 dev->io.outofband.privilege_level,
                                 dev->io.outofband.challenge_string,
                                 IPMI_CHALLENGE_STRING_LENGTH,
                                 initial_outbound_sequence_number,
                                 obj_cmd_rs) < 0)
    {
      if (dev->errnum == IPMI_ERR_BAD_COMPLETION_CODE)
	{
	  if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NO_SESSION_SLOT_AVAILABLE) == 1
	      || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NO_SLOT_AVAILABLE_FOR_GIVEN_USER) == 1
	      || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_NO_SLOT_AVAILABLE_TO_SUPPORT_USER) == 1)
	    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_BMC_BUSY);
	  else if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_EXCEEDS_PRIVILEGE_LEVEL) == 1)
	    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE);
	}
      if (dev->errnum == IPMI_ERR_SESSION_TIMEOUT)
        UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PASSWORD_VERIFICATION_TIMEOUT);
      UDM_ERR_LOG_CLEANUP(0);
    }

  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
			    "session_id", 
			    &temp_session_id);

  dev->io.outofband.session_id = temp_session_id;
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs, 
			    "initial_inbound_sequence_number", 
			    &temp_session_sequence_number);
  dev->io.outofband.session_sequence_number = temp_session_sequence_number;
  
  UDM_FIID_OBJ_GET_CLEANUP (obj_cmd_rs,
			    "authentication_type",
			    &val);

  /* achu: Make sure the BMC actually wants to do permsg auth */
  if (dev->io.outofband.per_msg_auth_disabled
      && val != IPMI_AUTHENTICATION_TYPE_NONE)
    dev->io.outofband.per_msg_auth_disabled = 0;

  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  UDM_FIID_OBJ_CREATE_CLEANUP (obj_cmd_rs, tmpl_cmd_set_session_privilege_level_rs);
  dev->io.outofband.lan_session_state = IPMI_LAN_SESSION_STATE_SET_SESSION_PRIVILEGE_LEVEL;
  if (ipmi_cmd_set_session_privilege_level (dev, 
                                            dev->io.outofband.privilege_level,
                                            obj_cmd_rs) < 0)
    {
      if (dev->errnum == IPMI_ERR_BAD_COMPLETION_CODE)
	{
	  if (ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_RQ_LEVEL_NOT_AVAILABLE_FOR_USER) == 1
	      || ipmi_check_completion_code(obj_cmd_rs, IPMI_COMP_CODE_RQ_LEVEL_EXCEEDS_USER_PRIVILEGE_LIMIT) == 1)
	    UDM_ERR_SET_ERRNUM_CLEANUP(IPMI_ERR_PRIVILEGE);
	}
      UDM_ERR_LOG_CLEANUP(0);
    }

  dev->io.outofband.lan_session_state = IPMI_LAN_SESSION_STATE_IN_SESSION;
  rv = 0;
 cleanup:
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  return (rv);
}

int8_t 
ipmi_lan_close_session (ipmi_device_t dev)
{
  fiid_obj_t obj_cmd_rs = NULL;
  int8_t rv = -1;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.local_sockfd); 

  UDM_FIID_OBJ_CREATE (obj_cmd_rs, tmpl_cmd_close_session_rs);

  if (ipmi_cmd_close_session(dev,
                             dev->io.outofband.session_id,
                             obj_cmd_rs) < 0)
    goto cleanup;
  
  rv = 0;
 cleanup:
  UDM_FIID_OBJ_DESTROY(obj_cmd_rs);
  return (rv);
}

static void
_ipmi_lan_dump (ipmi_device_t dev,
		fiid_obj_t obj_rmcp,
		fiid_obj_t obj_session,
		fiid_obj_t obj_msg,
		fiid_obj_t obj_cmd,
		fiid_obj_t obj_trlr)
{
  char *rmcp_hdr =
    "RMCP Header:\n"
    "------------";
  char *session_hdr =
    "IPMI Session Header:\n"
    "--------------------";
  char *msg_hdr =
    "IPMI Message Header:\n"
    "--------------------";
  char *cmd_hdr =
    "IPMI Command Data:\n"
    "------------------";
  char *trlr_hdr =
    "IPMI Trailer:\n"
    "--------------";

  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
	  && fiid_obj_valid(obj_rmcp)
	  && fiid_obj_valid(obj_session)
	  && fiid_obj_valid(obj_msg)
	  && fiid_obj_valid(obj_cmd));

  ipmi_obj_dump_perror(STDERR_FILENO,
		       NULL,
		       rmcp_hdr,
		       NULL,
		       obj_rmcp);
  ipmi_obj_dump_perror(STDERR_FILENO,
		       NULL,
		       session_hdr,
		       NULL,
		       obj_session);
  ipmi_obj_dump_perror(STDERR_FILENO,
		       NULL,
		       msg_hdr,
		       NULL,
		       obj_msg);
  ipmi_obj_dump_perror(STDERR_FILENO,
		       NULL,
		       cmd_hdr,
		       NULL,
		       obj_cmd);
  if (obj_trlr && fiid_obj_valid(obj_trlr))
    ipmi_obj_dump_perror(STDERR_FILENO,
			 NULL,
			 trlr_hdr,
			 NULL,
			 obj_trlr);
}

static void
_ipmi_lan_dump_rq (ipmi_device_t dev, fiid_obj_t obj_cmd_rq)
{
  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
	  && fiid_obj_valid(obj_cmd_rq));

  _ipmi_lan_dump (dev,
		  dev->io.outofband.rq.obj_rmcp_hdr,
		  dev->io.outofband.rq.obj_lan_session_hdr,
		  dev->io.outofband.rq.obj_lan_msg_hdr,
		  obj_cmd_rq,
		  NULL);   
}

static void
_ipmi_lan_dump_rs (ipmi_device_t dev, fiid_obj_t obj_cmd_rs)
{
  assert (dev
	  && dev->magic == IPMI_UDM_DEVICE_MAGIC
	  && (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
	  && fiid_obj_valid(obj_cmd_rs));

  _ipmi_lan_dump (dev,
		  dev->io.outofband.rs.obj_rmcp_hdr,
		  dev->io.outofband.rs.obj_lan_session_hdr,
		  dev->io.outofband.rs.obj_lan_msg_hdr,
		  obj_cmd_rs,
		  dev->io.outofband.rs.obj_lan_msg_trlr);   
}

static int8_t
_ipmi_lan_cmd_send (ipmi_device_t dev, fiid_obj_t obj_cmd_rq)
{
  uint8_t *pkt;
  int32_t pkt_len = 1024;
  int32_t send_len = 0;
  uint8_t authentication_type;

  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && dev->io.outofband.local_sockfd 
	 && fiid_obj_valid(obj_cmd_rq)
	 && fiid_obj_packet_valid(obj_cmd_rq));

  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_rmcp_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_lan_session_hdr);
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rq.obj_lan_msg_hdr);
    
  pkt = alloca (pkt_len);
  UDM_ERR (pkt);
  memset (pkt, 0, pkt_len);
    
  UDM_ERR (fill_rmcp_hdr_ipmi (dev->io.outofband.rq.obj_rmcp_hdr) != -1);
  UDM_ERR (fill_lan_msg_hdr (dev->net_fn,
			     dev->lun,
			     dev->io.outofband.rq_seq,
			     dev->io.outofband.rq.obj_lan_msg_hdr) != -1);

  if ((dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_SET_SESSION_PRIVILEGE_LEVEL
       || dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_IN_SESSION
       || dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_CLOSE_SESSION)
      && dev->io.outofband.per_msg_auth_disabled)
    authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;
  else
    authentication_type = dev->io.outofband.authentication_type;

  UDM_ERR (fill_lan_session_hdr (authentication_type,
				 dev->io.outofband.session_sequence_number,
				 dev->io.outofband.session_id,
				 dev->io.outofband.rq.obj_lan_session_hdr) != -1);
  UDM_ERR ((send_len = assemble_ipmi_lan_pkt (dev->io.outofband.rq.obj_rmcp_hdr,
					      dev->io.outofband.rq.obj_lan_session_hdr,
					      dev->io.outofband.rq.obj_lan_msg_hdr,
					      obj_cmd_rq,
					      dev->io.outofband.password,
					      IPMI_MAX_AUTHENTICATION_CODE_LENGTH,
					      pkt,
					      pkt_len)) != -1);

  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
    _ipmi_lan_dump_rq (dev, obj_cmd_rq);

  UDM_ERR (!(ipmi_lan_sendto (dev->io.outofband.local_sockfd, 
			      pkt, 
			      send_len, 
			      0, 
			      (struct sockaddr *)&(dev->io.outofband.remote_host), 
			      sizeof(struct sockaddr_in)) < 0));

  UDM_ERR (!(gettimeofday(&dev->io.outofband.last_send, NULL) < 0));
  
  return (0);
}

static int
_session_timed_out(ipmi_device_t dev)
{
  struct timeval current;
  struct timeval session_timeout;
  struct timeval session_timeout_len;

  assert(dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  session_timeout_len.tv_sec = dev->io.outofband.session_timeout / 1000;
  session_timeout_len.tv_usec = (dev->io.outofband.session_timeout - (session_timeout_len.tv_sec * 1000)) * 1000;
  timeradd(&dev->io.outofband.last_received, &session_timeout_len, &session_timeout);

  UDM_ERR (!(gettimeofday(&current, NULL) < 0));

  return timercmp(&current, &session_timeout, >);
}

static int
_calculate_timeout(ipmi_device_t dev, struct timeval *timeout)
{
  struct timeval current;
  struct timeval session_timeout;
  struct timeval session_timeout_len;
  struct timeval session_timeout_val;
  struct timeval retry_timeout;
  struct timeval retry_timeout_len;
  struct timeval retry_timeout_val;
  unsigned int retry_timeout_multiplier;

  assert(dev 
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && timeout);

  UDM_ERR (!(gettimeofday(&current, NULL) < 0));

  session_timeout_len.tv_sec = dev->io.outofband.session_timeout / 1000;
  session_timeout_len.tv_usec = (dev->io.outofband.session_timeout - (session_timeout_len.tv_sec * 1000)) * 1000;

  timeradd(&current, &session_timeout_len, &session_timeout);
  timersub(&session_timeout, &current, &session_timeout_val);

  retry_timeout_multiplier = (dev->io.outofband.retry_count / IPMI_LAN_BACKOFF_COUNT) + 1;
  
  retry_timeout_len.tv_sec = (retry_timeout_multiplier * dev->io.outofband.retry_timeout) / 1000;
  retry_timeout_len.tv_usec = ((retry_timeout_multiplier * dev->io.outofband.retry_timeout) - (retry_timeout_len.tv_sec * 1000)) * 1000;

  timeradd(&dev->io.outofband.last_send, &retry_timeout_len, &retry_timeout);
  timersub(&retry_timeout, &current, &retry_timeout_val);

  if (timercmp(&retry_timeout_val, &session_timeout_val, <))
    {
      timeout->tv_sec = retry_timeout_val.tv_sec;
      timeout->tv_usec = retry_timeout_val.tv_usec;
    }
  else
    {
      timeout->tv_sec = session_timeout_val.tv_sec;
      timeout->tv_usec = session_timeout_val.tv_usec;
    }

  return 0;
}

static int8_t
_ipmi_lan_cmd_recv (ipmi_device_t dev, fiid_obj_t obj_cmd_rs)
{
  struct sockaddr_in from;
  socklen_t fromlen = 0;
  struct timeval timeout;
    
  uint8_t *pkt = NULL;
  uint32_t pkt_len = 1024;
  int32_t recv_len = 0;
  
  fd_set read_set;
  int status = 0;
  
  assert(dev
	 && dev->magic == IPMI_UDM_DEVICE_MAGIC
	 && dev->io.outofband.local_sockfd 
	 && fiid_obj_valid(obj_cmd_rs));
  
  pkt = alloca (pkt_len);
  UDM_ERR (pkt);
  
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_rmcp_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_session_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_msg_hdr); 
  UDM_FIID_OBJ_CLEAR(dev->io.outofband.rs.obj_lan_msg_trlr); 
  
  memset (pkt, 0, pkt_len);
      
  if (dev->io.outofband.retry_timeout != 0)
    {
      FD_ZERO (&read_set);
      FD_SET (dev->io.outofband.local_sockfd, &read_set);
      
      UDM_ERR (!(_calculate_timeout(dev, &timeout) < 0));
      
      UDM_ERR (!((status = select ((dev->io.outofband.local_sockfd + 1), 
				   &read_set,
				   NULL,
				   NULL, 
				   &timeout)) < 0));
      if (status == 0)
        return (0); // resend the request
    }

  UDM_ERR (!((recv_len = ipmi_lan_recvfrom (dev->io.outofband.local_sockfd, 
					    pkt, 
					    pkt_len, 
					    0, 
					    (struct sockaddr *) &from, 
					    &fromlen)) < 0));
  
  UDM_ERR (unassemble_ipmi_lan_pkt(pkt,
				   recv_len,
				   dev->io.outofband.rs.obj_rmcp_hdr,
				   dev->io.outofband.rs.obj_lan_session_hdr,
				   dev->io.outofband.rs.obj_lan_msg_hdr,
				   obj_cmd_rs,
				   dev->io.outofband.rs.obj_lan_msg_trlr) != -1);

  if (dev->flags & IPMI_FLAGS_DEBUG_DUMP)
    _ipmi_lan_dump_rs (dev, obj_cmd_rs);

  return (recv_len);
}

int8_t 
ipmi_lan_cmd (ipmi_device_t dev, 
	      fiid_obj_t obj_cmd_rq,
	      fiid_obj_t obj_cmd_rs)
{
  int status = 0;
  int retval = -1;
  int ret;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.local_sockfd); 

  UDM_ERR_INVALID_PARAMETERS (fiid_obj_valid(obj_cmd_rq)
			      && fiid_obj_valid(obj_cmd_rs));
  
  UDM_FIID_OBJ_PACKET_VALID(obj_cmd_rq);
  
  if (!dev->io.outofband.last_received.tv_sec
      && !dev->io.outofband.last_received.tv_usec)
    UDM_ERR (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));

  dev->io.outofband.retry_count = 0;

  if (_ipmi_lan_cmd_send (dev, obj_cmd_rq) < 0)
    goto cleanup;

  while (1)
    {
      uint8_t authentication_type;

      if (_session_timed_out(dev))
        {
	  UDM_ERR_SET_ERRNUM (IPMI_ERR_SESSION_TIMEOUT);
          retval = -1;
          break;
        }
     
      if ((status = _ipmi_lan_cmd_recv (dev, obj_cmd_rs)) < 0)
        {
          retval = -1;
          break;
        }

      if (!status)
        {
          dev->io.outofband.session_sequence_number++;
          IPMI_LAN_RQ_SEQ_INC (dev->io.outofband.rq_seq);
          dev->io.outofband.retry_count++;

          if (_ipmi_lan_cmd_send (dev, obj_cmd_rq) < 0)
            goto cleanup;

          continue;
        }

      /* else received a packet */

      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_checksum (dev->io.outofband.rs.obj_lan_msg_hdr,
                                                         obj_cmd_rs,
                                                         dev->io.outofband.rs.obj_lan_msg_trlr)) < 0));
      
      if (!ret)
        continue;

      if ((dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_SET_SESSION_PRIVILEGE_LEVEL
           || dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_IN_SESSION
           || dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_CLOSE_SESSION)
          && dev->io.outofband.per_msg_auth_disabled)
        authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;
      else
        authentication_type = dev->io.outofband.authentication_type;
      
      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_session_authentication_code (dev->io.outofband.rs.obj_lan_session_hdr,
                                                                            dev->io.outofband.rs.obj_lan_msg_hdr,
                                                                            obj_cmd_rs,
                                                                            dev->io.outofband.rs.obj_lan_msg_trlr,
                                                                            authentication_type,
                                                                            dev->io.outofband.password,
                                                                            IPMI_MAX_AUTHENTICATION_CODE_LENGTH)) < 0));

      if (!ret)
        continue;
  
      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_rq_seq(dev->io.outofband.rs.obj_lan_msg_hdr, 
                                                      dev->io.outofband.rq_seq)) < 0));

      if (!ret)
        continue;

     
      UDM_ERR_CLEANUP (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));
      retval = status;
      break;
    }

 cleanup:
  dev->io.outofband.session_sequence_number++;
  IPMI_LAN_RQ_SEQ_INC (dev->io.outofband.rq_seq);
  return (retval);
}

int32_t 
ipmi_lan_cmd_raw (ipmi_device_t dev, 
		  uint8_t *buf_rq, 
		  size_t buf_rq_len, 
		  uint8_t *buf_rs, 
		  size_t buf_rs_len)
{
  fiid_obj_t obj_cmd_rq = NULL;
  fiid_obj_t obj_cmd_rs = NULL;
  int status = 0;
  int retval = -1;
  int32_t len;
  int ret;

  UDM_ERR_DEV_CHECK (dev && dev->magic == IPMI_UDM_DEVICE_MAGIC);

  UDM_ERR_DEVICE_NOT_OPEN(dev->type == IPMI_DEVICE_LAN);
  UDM_ERR_DEVICE_NOT_OPEN(dev->io.outofband.local_sockfd); 

  UDM_ERR_INVALID_PARAMETERS (buf_rq
			      && buf_rq_len > 0
			      && buf_rs
			      && buf_rs_len > 0);

  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rq, tmpl_lan_raw);
  UDM_FIID_OBJ_CREATE_CLEANUP(obj_cmd_rs, tmpl_lan_raw);

  UDM_FIID_OBJ_SET_ALL_CLEANUP (obj_cmd_rq, buf_rq, buf_rq_len);

  if (!dev->io.outofband.last_received.tv_sec
      && !dev->io.outofband.last_received.tv_usec)
    UDM_ERR (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));

  dev->io.outofband.retry_count = 0;

  if (_ipmi_lan_cmd_send (dev, obj_cmd_rq) < 0)
    goto cleanup;

  while (1)
    {
      uint8_t authentication_type;

      if (_session_timed_out(dev))
        {
	  UDM_ERR_SET_ERRNUM (IPMI_ERR_SESSION_TIMEOUT);
          retval = -1;
          break;
        }

      if ((status = _ipmi_lan_cmd_recv (dev, obj_cmd_rs)) < 0)
        {
          retval = -1;
          break;
        }

      if (!status)
        {
          dev->io.outofband.session_sequence_number++;
          IPMI_LAN_RQ_SEQ_INC (dev->io.outofband.rq_seq);
          dev->io.outofband.retry_count++;

          if (_ipmi_lan_cmd_send (dev, obj_cmd_rq) < 0)
            goto cleanup;

          continue;
        }

      /* else received a packet */

      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_checksum (dev->io.outofband.rs.obj_lan_msg_hdr,
                                                         obj_cmd_rs,
                                                         dev->io.outofband.rs.obj_lan_msg_trlr)) < 0));

      if (!ret)
        continue;

      if ((dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_SET_SESSION_PRIVILEGE_LEVEL
           || dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_IN_SESSION
           || dev->io.outofband.lan_session_state == IPMI_LAN_SESSION_STATE_CLOSE_SESSION)
          && dev->io.outofband.per_msg_auth_disabled)
        authentication_type = IPMI_AUTHENTICATION_TYPE_NONE;
      else
        authentication_type = dev->io.outofband.authentication_type;

      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_session_authentication_code (dev->io.outofband.rs.obj_lan_session_hdr,
                                                                            dev->io.outofband.rs.obj_lan_msg_hdr,
                                                                            obj_cmd_rs,
                                                                            dev->io.outofband.rs.obj_lan_msg_trlr,
                                                                            authentication_type,
                                                                            dev->io.outofband.password,
                                                                            IPMI_MAX_AUTHENTICATION_CODE_LENGTH)) < 0));

      if (!ret)
        continue;
  
      UDM_ERR_CLEANUP (!((ret = ipmi_lan_check_rq_seq(dev->io.outofband.rs.obj_lan_msg_hdr, 
                                                      dev->io.outofband.rq_seq)) < 0));

      if (!ret)
        continue;
      
      UDM_ERR_CLEANUP (!(gettimeofday(&dev->io.outofband.last_received, NULL) < 0));

      UDM_FIID_OBJ_GET_ALL_LEN_CLEANUP (len, obj_cmd_rs, buf_rs, buf_rs_len);
      retval = len;
      break;
    }

 cleanup:
  dev->io.outofband.session_sequence_number++;
  IPMI_LAN_RQ_SEQ_INC (dev->io.outofband.rq_seq);
  UDM_FIID_OBJ_DESTROY (obj_cmd_rq);
  UDM_FIID_OBJ_DESTROY (obj_cmd_rs);
  return (retval);
}
