/********************************************************************************
*                                                                               *
*                          X P M   I n p u t / O u t p u t                      *
*                                                                               *
*********************************************************************************
* Copyright (C) 2000,2003 by Jeroen van der Zijp.   All Rights Reserved.        *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************
* $Id: fxxpmio.cpp,v 1.31 2003/09/05 06:06:42 fox Exp $                         *
********************************************************************************/
#include "xincs.h"
#include "fxver.h"
#include "fxdefs.h"
#include "FXStream.h"
#include "fxpriv.h"



/*
  Notes:
  - The transparent color hopefully does not occur in the image.
  - If the image is rendered opaque, the transparent is close to white.
  - References:
      http://www-sop.inria.fr/koala/lehors/xpm.html
  - XPM reader/writer is tweaked so that an XPM written to FXStream
    can be read back in and read exactly as many bytes as were written.
  - There may be other comment blocks in the file
*/

#define MAXPRINTABLE    92
#define MAXVALUE        96
#define HASH1(x,n)      (((unsigned int)(x)*13)%(n))            // Number [0..n-1]
#define HASH2(x,n)      (1|(((unsigned int)(x)*17)%((n)-1)))    // Number [1..n-2]

using namespace FX;


/*******************************************************************************/

namespace FX {


// Little helper
static void readbuffer(FXStream& store,FXchar* buffer,FXuint size){
  register FXuint i=0;
  FXchar ch;
  while(!store.eof()){
    store >> ch;
    if(ch=='"') break;
    if(ch!='/') continue;
    store >> ch;
    if(ch!='*') continue;
    while(!store.eof()){
      store >> ch;
      if(ch!='*') continue;
      store >> ch;
      if(ch=='/') break;
      }
    }
  while(!store.eof() && i<size){
    store >> ch;
    if(ch=='"') break;
    buffer[i++]=ch;
    }
  while(!store.eof()){
    store >> ch;
    if(ch=='\n') break;
    }
  buffer[i]=0;
  }


// Load image from array of strings
FXbool fxloadXPM(const FXchar **pixels,FXColor*& data,FXint& width,FXint& height){
  FXuint    ncolortable,index,ncolors,cpp,c;
  FXColor  *colortable=NULL;
  FXint     ww,hh,i,j;
  FXchar    name[100];
  FXchar    type[10];
  const FXchar *ptr;
  FXColor  *pix;
  FXColor   color;

  // Null out
  data=NULL;
  width=0;
  height=0;

  // NULL pointer passed in
  if(!pixels) return FALSE;

  // Read pointer
  ptr=*pixels++;

  // No size description line
  if(!ptr) return FALSE;

  // Parse size description
  sscanf(ptr,"%d %d %u %u",&ww,&hh,&ncolors,&cpp);

  // Check size
  if(ww<1 || hh<1 || ww>16384 || hh>16384) return FALSE;

  // Check number of colors, and number of characters/color
  if(cpp<1 || cpp>2 || ncolors<1 || ncolors>9216) return FALSE;

  // Color lookup table
  ncolortable = (cpp==1) ? MAXVALUE : MAXVALUE*MAXVALUE;

  // Check if characters/color is consistent with number of colors
  if(ncolortable<ncolors) return FALSE;

  // Make color table
  if(!FXMALLOC(&colortable,FXColor,ncolortable)){
    return FALSE;
    }

  // Read the color table
  for(c=0; c<ncolors; c++){
    ptr=*pixels++;
    index=*ptr++ - ' ';
    if(cpp==2) index=index+MAXVALUE*(*ptr++ - ' ');
    if(index>ncolortable){
      FXFREE(&colortable);
      return FALSE;
      }
    sscanf(ptr,"%s %s",type,name);
    if(type[0]!='c') sscanf(ptr,"%*s %*s %s %s",type,name);
    color=fxcolorfromname(name);
    colortable[index]=color;
    }

  // Try allocate pixels
  if(!FXMALLOC(&data,FXColor,ww*hh)){
    FXFREE(&colortable);
    return FALSE;
    }

  // Read the pixels
  for(i=0,pix=data; i<hh; i++){
    ptr=*pixels++;
    for(j=0; j<ww; j++){
      index=*ptr++ - ' ';
      if(cpp==2) index=index+MAXVALUE*(*ptr++ - ' ');
      if(index>ncolortable){
        FXFREE(&colortable);
        FXFREE(&data);
        return FALSE;
        }
      *pix++=colortable[index];
      }
    }
  FXFREE(&colortable);
  width=ww;
  height=hh;
  return TRUE;
  }


/*******************************************************************************/

// Load image from stream
FXbool fxloadXPM(FXStream& store,FXColor*& data,FXint& width,FXint& height){
  FXuint    ncolortable,index,ncolors,cpp,c;
  FXColor  *colortable=NULL;
  FXint     ww,hh,i,j;
  FXchar    header[256];
  FXchar    name[100];
  FXchar    type[10];
  FXchar    ch;
  FXchar   *ptr;
  FXColor  *pix;
  FXColor   color;

  // Null out
  data=NULL;
  width=0;
  height=0;

  readbuffer(store,header,sizeof(header));
  if(store.eof()) return FALSE;

  // Parse size description
  sscanf(header,"%d %d %u %u",&ww,&hh,&ncolors,&cpp);

  // Check size
  if(ww<1 || hh<1 || ww>16384 || hh>16384) return FALSE;

  // Check number of colors, and number of characters/color
  if(cpp<1 || cpp>2 || ncolors<1 || ncolors>9216) return FALSE;

  // Color lookup table
  ncolortable = (cpp==1) ? MAXVALUE : MAXVALUE*MAXVALUE;

  FXTRACE((50,"fxloadXPM: width=%d height=%d ncolors=%d cpp=%d\n",width,height,ncolors,cpp));

  // Check if characters/color is consistent with number of colors
  if(ncolortable<ncolors) return FALSE;

  // Make color table
  if(!FXMALLOC(&colortable,FXColor,ncolortable)){
    return FALSE;
    }

  // Read the color table
  for(c=0; c<ncolors; c++){
    readbuffer(store,header,sizeof(header));
    if(store.eof()) return FALSE;
    ptr=header;
    index=*ptr++ - ' ';
    if(cpp==2) index=index+MAXVALUE*(*ptr++ - ' ');
    if(index>ncolortable){
      FXFREE(&colortable);
      return FALSE;
      }
    sscanf(ptr,"%s %s",type,name);
    if(type[0]!='c') sscanf(ptr,"%*s %*s %s %s",type,name);
    color=fxcolorfromname(name);
    colortable[index]=color;
    }

  // Try allocate pixels
  if(!FXMALLOC(&data,FXColor,ww*hh)){
    FXFREE(&colortable);
    return FALSE;
    }

  // Read the pixels
  for(i=0,pix=data; i<hh; i++){
    while(!store.eof() && (store>>ch,ch!='"'));
    for(j=0; j<ww; j++){
      store >> ch;
      index=ch-' ';
      if(cpp==2){ store >> ch; index=index+MAXVALUE*(ch-' '); }
      if(index>ncolortable){
        FXFREE(&colortable);
        FXFREE(&data);
        return FALSE;
        }
      *pix++=colortable[index];
      }
    while(!store.eof() && (store>>ch,ch!='\n'));
    if(store.eof()) return FALSE;
    }
  FXFREE(&colortable);
  width=ww;
  height=hh;
  return TRUE;
  }


/*******************************************************************************/


// Save image to a stream
FXbool fxsaveXPM(FXStream& store,const FXColor *data,FXint width,FXint height){
  const FXchar printable[]=" .XoO+@#$%&*=-;:>,<1234567890qwertyuipasdfghjklzxcvbnmMNBVCZASDFGHJKLPIUYTREWQ!~^/()_`'][{}|";
  const FXchar quote='"';
  const FXchar comma=',';
  const FXchar newline='\n';
  FXColor   colormap[256];
  FXint     numpixels=width*height;
  FXint     ncolors,cpp,len,i,j,c1,c2,ok;
  FXchar    buffer[200];
  FXColor   color;
  FXuchar  *pixels,*ptr,pix;

  // Must make sense
  if(!data || width<=0 || height<=0) return FALSE;

  // Allocate temp buffer for pixels
  if(!FXMALLOC(&pixels,FXuchar,numpixels)) return FALSE;

  // First, try EZ quantization, because it is exact; a previously
  // loaded XPM will be re-saved with exactly the same colors.
  ok=fxezquantize(pixels,data,colormap,ncolors,width,height,256);

  if(!ok){

    // Floyd-Steinberg quantize full 32 bpp to 256 colors, organized as 3:3:2
    fxfsquantize(pixels,data,colormap,ncolors,width,height,256);
    }

  FXASSERT(ncolors<=256);

  // How many characters needed to represent one pixel, characters per line
  cpp=(ncolors>MAXPRINTABLE)?2:1;

  // Save header
  store.save("/* XPM */\nstatic char * image[] = {\n",36);

  // Save values
  len=sprintf(buffer,"\"%d %d %d %d\",\n",width,height,ncolors,cpp);
  store.save(buffer,len);

  // Save the colors
  for(i=0; i<ncolors; i++){
    color=colormap[i];
    c1=printable[i%MAXPRINTABLE];
    c2=printable[i/MAXPRINTABLE];
    if(FXALPHAVAL(color)){
      len=sprintf(buffer,"\"%c%c c #%02x%02x%02x\",\n",c1,c2,FXREDVAL(color),FXGREENVAL(color),FXBLUEVAL(color));
      store.save(buffer,len);
      }
    else{
      len=sprintf(buffer,"\"%c%c c None\",\n",c1,c2);
      store.save(buffer,len);
      }
    }

  // Save the image
  ptr=pixels;
  for(i=0; i<height; i++){
    store << quote;
    for(j=0; j<width; j++){
      pix=*ptr++;
      if(cpp==1){
        store << printable[pix];
        }
      else{
        store << printable[pix%MAXPRINTABLE];
        store << printable[pix/MAXPRINTABLE];
        }
      }
    store << quote;
    if(i<height-1){ store << comma; store << newline; }
    }
  store.save("};\n",3);
  FXFREE(&pixels);
  return TRUE;
  }

}

