/*
 * USB Voip-Blaster driver - v1.10
 *
 * Originally written by David Stanhope
 * TODO:
 * 	Add ioctl to read usb descriptors
 *
 *
 * Based on USB Skeleton driver - 1.1
 *
 * Copyright (C) 2001-2003 Greg Kroah-Hartman (greg@kroah.com)
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License as
 *	published by the Free Software Foundation, version 2.
 *
 *
 * This driver is to be used as a skeleton driver to be able to create a
 * USB driver quickly.  The design of it is based on the usb-serial and
 * dc2xx drivers.
 *
 * Thanks to Oliver Neukum, David Brownell, and Alan Stern for their help
 * in debugging this driver.
 *
 *
 * History:
 *
 * 2004-10-28 - johngeeaswell
 *                 bring driver version numbers into sync and reflect
 *                 latest updates. Minor housekeeping .. 
 * 2004-10-07 - mdivya 
 *                 vb application needs 2 dev files vbcx/vbvx; in 2.6 usb
 *                 support it looks like interface can save only one minor
 *                 number so usb_find_interface does not work for 2 devices.
 *                 Instead we will use our locally maintained table to locate
 *                 the correct dev entry.
 * 2004-08-10 - csenthil
 *                 main port of 2.4 kernel driver to 2.6. Initial driver
 *                 was based on usb-skeleton 0.6 and should now be up to
 *                 1.1 standards
 *
 *
 * 2003-05-06 - 1.1 - changes due to usb core changes with usb_register_dev()
 * 2003-02-25 - 1.0 - fix races involving urb->status, unlink_urb(), and
 *			disconnect.  Fix transfer amount in read().  Use
 *			macros instead of magic numbers in probe().  Change
 *			size variables to size_t.  Show how to eliminate
 *			DMA bounce buffer.
 * 2002_12_12 - 0.9 - compile fixes and got rid of fixed minor array.
 * 2002_09_26 - 0.8 - changes due to USB core conversion to struct device
 *			driver.
 * 2002_02_12 - 0.7 - zero out dev in probe function for devices that do
 *			not have both a bulk in and bulk out endpoint.
 *			Thanks to Holger Waechtler for the fix.
 * 2001_11_05 - 0.6 - fix minor locking problem in skel_disconnect.
 *			Thanks to Pete Zaitcev for the fix.
 * 2001_09_04 - 0.5 - fix devfs bug in skel_disconnect. Thanks to wim delvaux
 * 2001_08_21 - 0.4 - more small bug fixes.
 * 2001_05_29 - 0.3 - more bug fixes based on review from linux-usb-devel
 * 2001_05_24 - 0.2 - bug fixes based on review from linux-usb-devel people
 * 2001_05_01 - 0.1 - first version
 *
 *
 * (Written with Tabs=4)
 */


#define BG_READ 1
#define CONFIG_USB_DEBUG

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/signal.h>
#include <linux/errno.h>
#include <linux/poll.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/fcntl.h>
#include <linux/module.h>
#include <linux/spinlock.h>
#include <linux/list.h>
#include <linux/smp_lock.h>
#include <linux/devfs_fs_kernel.h>
#include <linux/usbdevice_fs.h>
#include <linux/usb.h>

#ifdef CONFIG_USB_DEBUG
	static int debug = 1;
#else
	static int debug;
#endif

// Use our own dbg() macro
#undef dbg
#define dbg(format, arg...) if (debug) \
	printk(KERN_DEBUG __FILE__ ", %d: " format "\n" , __LINE__, ## arg); \

// Use our own err() macro
#undef err
#define err(format, arg...) \
	printk(KERN_ERR  __FILE__ ", %d : " format "\n" , __LINE__, ## arg); \

// Version Information
#define DRIVER_VERSION "v1.10"
#define DRIVER_AUTHOR  "David Stanhope (voip@fobbit.com)"
#define DRIVER_DESC    "USB Voip-Blaster Device Driver"

// Module paramaters
MODULE_PARM(debug, "i");
MODULE_PARM_DESC(debug, "Debug enabled or not");

// Define these values to match your device
#define USB_VB_VENDOR_ID	0x1292
#define USB_VB_PRODUCT_ID	0x0258

#ifdef MODULE_DEVICE_TABLE
// table of devices that work with this driver
static struct usb_device_id vb_table [] = {
	{ USB_DEVICE(USB_VB_VENDOR_ID, USB_VB_PRODUCT_ID) },
	{ }					// Terminating entry
};

MODULE_DEVICE_TABLE (usb, vb_table);
#endif

// Get a minor range for your devices from the usb maintainer
#define USB_VB_MINOR_BASE	200

// we can have up to this number of device plugged in at once
// (really only half since each device takes two device nodes
#define MAX_DEVICES			8

// various pipe sizes
#define VOUT_SIZE 64
#define VINP_SIZE 20
#define  CMD_SIZE  1
#define STAT_SIZE  1

#define RDR_TIMEOUT (10*HZ)
#define RDR_BUF_CNT 64
// each buffer: 1st byte is bytes-in-buffer, then actual buffer

// Structure to hold all of our device specific stuff
struct usb_vb {
	struct usb_device       *udev; // save off the usb device pointer
	struct usb_interface	*interface; // the interface for this device
	int                     minor; // starting minor number for this device
	dev_t			devfs; // devfs device node

	u_char  out_buffer[VOUT_SIZE]; // voice/command output buffer for urb
	u_char  inp_buffer[VINP_SIZE]; // voice/status  input  buffer for urb
#ifdef BG_READ
	u_char            *rdr_buffer; // read system call buffer
	int                rdr_head  ; // interface to read system call
	int                rdr_tail  ; // interface to read system call
	int                rdr_status; // interface to read system call
	wait_queue_head_t  rdr_wqh   ; // Processes waiting
	wait_queue_head_t  rdr_pwait ; // Processes waiting
	struct urb       *have_urb  ;
#endif

	int                  out_size; // max size for output pipe
	int                  inp_size; // max size for input  pipe

	int              out_endpoint; // output endpoint address
	int              inp_endpoint; // input  endpoint address

        atomic_t                write_busy;             /* true iff write urb is busy */
        struct completion       write_finished;         /* wait for the write to finish */

	int                open_count; // no. of times this port has been opened

	struct semaphore      out_sem; // locks this structure
	struct semaphore      inp_sem; // locks this structure
};

// the global devfs handle
extern dev_t usb_devfs_handle;
static struct file_operations vb_fops; // forward decl of fops for devfs use

// array of pointers to our devices that are currently connected
static struct usb_vb *minor_table[MAX_DEVICES];

#define VB_WRITE_TIMEOUT (HZ*10)

// lock to protect the minor_table structure
static DECLARE_MUTEX (minor_table_mutex);

static int vb_bulk_msg(struct usb_device *usb_dev, unsigned int pipe,
	 		void *data, int len, int *actual_length, int timeout);

static void vb_post_read_error(struct usb_vb *dev, int err);
static void vb_read_restart(struct urb *urb,struct pt_regs *regs);
static int vb_read_start(struct usb_vb *dev);


static ssize_t vb_read (struct file *file, char *buffer, size_t count, loff_t *ppos);
static ssize_t vb_write (struct file *file, const char *buffer, size_t count, loff_t *ppos);

#ifdef TODO
static int vb_ioctl(struct inode *inode, struct file *file, unsigned int cmd,unsigned long arg);
#endif
static int vb_open (struct inode *inode, struct file *file);
static int vb_release (struct inode *inode, struct file *file);

static int vb_probe(struct usb_interface *interface, const struct usb_device_id *id);

static void vb_disconnect(struct usb_interface *interface);

static void vb_write_bulk_callback(struct urb *urb,struct pt_regs *regs);

static unsigned int vb_poll(struct file *file, struct poll_table_struct *wait);
static struct usb_vb * vb_dev_add(struct usb_interface *interface, struct usb_device *udev, int minor, int pipe, int osize, int isize);


/*
 * File operations needed when we register this driver.
 * This assumes that this driver NEEDS file operations,
 * of course, which means that the driver is expected
 * to have a node in the /dev directory. If the USB
 * device were for a network interface then the driver
 * would use "struct net_driver" instead, and a serial
 * device would use "struct tty_driver".
 */
static struct file_operations vb_fops = {
	owner:		THIS_MODULE,
	read:		vb_read,
	write:		vb_write,
	open:		vb_open,
#ifdef BG_READ
	poll:		vb_poll,
#endif
	release:	vb_release,
#ifdef TODO
	ioctl:		vb_ioctl,
#endif
};

/* usb class driver info in order to get a minor number from the usb core,
 * and to have the device registered with devfs and the driver core
 */
static struct usb_class_driver vb_classc = {
        .name =         "usb/vbc%d",
        .fops =         &vb_fops,
        .mode =         S_IFCHR | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH,
        .minor_base =   USB_VB_MINOR_BASE,
};

static struct usb_class_driver vb_classv = {
        .name =         "usb/vbv%d",
        .fops =         &vb_fops,
        .mode =         S_IFCHR | S_IRUSR | S_IWUSR | S_IRGRP | S_IWGRP | S_IROTH,
        .minor_base =   USB_VB_MINOR_BASE + 4,
};

// usb specific object needed to register this driver with the usb subsystem
static struct usb_driver vb_driver = {
	owner:		THIS_MODULE,
	name:		"usbvb",
	probe:		vb_probe,
	disconnect:	vb_disconnect,
	id_table:	vb_table,
};


//
// usb_vb_debug_data
//
static inline void
usb_vb_debug_data (const char *function, int size, const unsigned char *data)
{
	int i;

	if (!debug)
		return;

        printk (KERN_DEBUG __FILE__": %s - length = %d, data = ",
                function, size);
	for (i = 0; i < size; ++i) {
		printk ("%.2x ", data[i]);
	}
	printk ("\n");
}

//
// vb_delete
//
static inline void
vb_delete (struct usb_vb *dev)
{
	minor_table[dev->minor] = NULL;

#ifdef BG_READ
	kfree (dev->rdr_buffer);
        usb_free_urb (dev->have_urb);
#endif
	kfree (dev);
}

//
// vb_open
//


static int
vb_open (struct inode *inode, struct file *file)
{
	struct usb_vb *dev = NULL;
	int subminor;
	int retval = 0;

	dbg("%s Entering open",__FUNCTION__);

        // locate the device in our table - no more usb_find_interface
	subminor = MINOR(inode->i_rdev) - USB_VB_MINOR_BASE;
	if ((subminor < 0) ||		//validate subminor
	    (subminor >= MAX_DEVICES))
	{
		return -ENODEV;
	}
	down (&minor_table_mutex);
	dev = minor_table[subminor];

	if (dev == NULL)
	{
		err("%s no device; minor=%d ",__FUNCTION__,subminor);
		up(&minor_table_mutex);
		return -ENODEV;
	}

	// lock this device, both directions
	down (&dev->out_sem);
	down (&dev->inp_sem);

#ifdef BG_READ
	if(dev->open_count == 0) // on first open, start reader on input pipe
	{
		init_waitqueue_head(&dev->rdr_wqh  );
		init_waitqueue_head(&dev->rdr_pwait);
		dev->rdr_head   = 0   ;
		dev->rdr_tail   = 0   ;
		dev->rdr_status = 0   ;
		dev->have_urb   = NULL;
		if ((retval = vb_read_start(dev)) != 0)
		{
			// unlock this device, both directions
			up (&dev->inp_sem);
			up (&dev->out_sem);
			up (&minor_table_mutex);
			return retval;
		}
	}
#endif

	// increment our usage count for the driver
	++dev->open_count;

	// save our object in the file's private structure
	file->private_data = dev;

	// unlock this device, both directions
	up (&dev->inp_sem);
	up (&dev->out_sem);

	up (&minor_table_mutex);

	return retval;
}

//
// vb_release
//
static int
vb_release (struct inode *inode, struct file *file)
{
	struct usb_vb *dev;
	int retval = 0;

	dev = (struct usb_vb *)file->private_data;
	if (dev == NULL)
	{
		dbg ("%s - object is NULL", __FUNCTION__);
		return -ENODEV;
	}

	dbg("%s - minor %d", __FUNCTION__, dev->minor);

	// lock our minor table
	down (&minor_table_mutex);

	// lock our device
	down (&dev->out_sem);
	down (&dev->inp_sem);

	if (dev->open_count <= 0)
	{
		dbg ("%s - device not opened", __FUNCTION__);
		retval = -ENODEV;
		goto exit_not_opened;
	}

	// decrement our usage count for the device
	--dev->open_count;
	if (dev->open_count <= 0)
	{
		dev->open_count = 0;
#ifdef BG_READ
		if(dev->have_urb)
		{
			// must unlock since the unlink will cause restart to be called
			// and will deadlock when it trys to lock
			up (&dev->inp_sem);
			up (&minor_table_mutex);
			usb_unlink_urb(dev->have_urb);  // remove urb safely
			dev->have_urb = NULL;
			// now can relock
			down (&minor_table_mutex);
			down (&dev->inp_sem);
		}
#endif

		if (dev->udev == NULL)
		{
			dbg ("%s - device unplugged", __FUNCTION__);
			// the device was unplugged before the file was released
			// so held off rest of 'vb_disconnect' till now
			up (&dev->inp_sem);
			up (&dev->out_sem);
			vb_delete (dev);
			up (&minor_table_mutex);
			return 0;
		}
	}

	// decrement our usage count for the module

exit_not_opened:
	up (&dev->inp_sem);
	up (&dev->out_sem);
	up (&minor_table_mutex);

	return retval;
}

//
// vb_read
//
static ssize_t
vb_read (struct file *file, char *buffer, size_t count, loff_t *ppos)
{
	struct usb_vb *dev;
	int retval = 0;
#ifdef BG_READ
	u_char *cp;
#endif

	dev = (struct usb_vb *) file->private_data;

	dbg("%s - minor %d, count = %d", __FUNCTION__, dev->minor, count);

	down (&dev->inp_sem); // lock this object

 	// verify that the device wasn't unplugged
	if (dev->udev == NULL)
	{
		up (&dev->inp_sem);
		return -ENODEV;
	}

	if (count == 0)
	{
		up (&dev->inp_sem);
		return 0;
	}

#ifdef BG_READ

	while(dev->rdr_head == dev->rdr_tail) // block while empty
	{
		if((retval = dev->rdr_status) != 0)
		{
			dbg("%s-dev(%d) error(%d)", __FUNCTION__, dev->minor, retval);
			up (&dev->inp_sem); // unlock the device
			return retval;
		}
		if(signal_pending(current))
		{
			dbg("%s-dev(%d) have signal-pending", __FUNCTION__,dev->minor);
			up (&dev->inp_sem); // unlock the device
			return -EINTR;
		}
		up (&dev->inp_sem); // unlock the device
		interruptible_sleep_on_timeout(&dev->rdr_wqh, RDR_TIMEOUT);
		down (&dev->inp_sem); // lock the device
	}

	// 'inp_sem' still locked at this point

	cp = dev->rdr_buffer + (dev->rdr_tail * (dev->inp_size + 1));

	if((retval = *cp++) > count) { retval = count; }

	if (copy_to_user (buffer, cp, retval))
	{
		dbg("%s- dev(%d) copy_to_user error", __FUNCTION__,dev->minor);
		retval = -EFAULT;
	}

	if((dev->rdr_tail = (dev->rdr_tail + 1)) >= RDR_BUF_CNT)
	{
		dev->rdr_tail = 0;
	}

#else // BG_READ

	// do an immediate read to get data from the device
	retval = vb_bulk_msg (dev->udev,
				usb_rcvbulkpipe (dev->udev, dev->inp_endpoint & 0x7f),
					dev->inp_buffer, dev->inp_size, &count, 0);

	// if the read was successful, copy the data to userspace
	if (!retval && count)
	{
		if (copy_to_user (buffer, dev->inp_buffer, count))
			retval = -EFAULT;
		else
			retval = count;
	}

#endif // BG_READ

	up (&dev->inp_sem); // unlock the device
	return retval;
}

/**
 *      vb_write
 *
 *      A device driver has to decide how to report I/O errors back to the
 *      user.  The safest course is to wait for the transfer to finish before
 *      returning so that any errors will be reported reliably.  skel_read()
 *      works like this.  But waiting for I/O is slow, so many drivers only
 *      check for errors during I/O initiation and do not report problems
 *      that occur during the actual transfer.  That's what we will do here.
 *
 *      A driver concerned with maximum I/O throughput would use double-
 *      buffering:  Two urbs would be devoted to write transfers, so that
 *      one urb could always be active while the other was waiting for the
 *      user to send more data.
 */
static ssize_t
vb_write (struct file *file, const char *buffer, size_t count, loff_t *ppos)
{
	struct usb_vb *dev;
	ssize_t write_count;
	int retval = 0;

	dev = (struct usb_vb *) file->private_data;

	dbg("%s - minor %d, count = %d", __FUNCTION__, dev->minor, count);

	// lock this object
	down (&dev->out_sem);

	// verify that the device wasn't unplugged
	if (dev->udev == NULL)
	{
		retval = -ENODEV;
		goto exit;
	}

	// verify that we actually have some data to write
	if (count == 0)
	{
		dbg("%s - write request of 0 bytes",__FUNCTION__);
		goto exit;
	}

	// we can only write as much as 1 urb will hold
	write_count = (count > dev->out_size) ?  dev->out_size : count;

	// copy the data from userspace into our urb
	if (copy_from_user(dev->out_buffer, buffer, write_count))
	{
		retval = -EFAULT;
		goto exit;
	}

	usb_vb_debug_data(__FUNCTION__,write_count,
				dev->have_urb->transfer_buffer);


	// send the data to the device
	retval = vb_bulk_msg (dev->udev,
				usb_sndbulkpipe (dev->udev, dev->out_endpoint & 0x7f),
				dev->out_buffer, write_count, &count, VB_WRITE_TIMEOUT);

	if (retval >= 0)
	{
		retval = write_count;;
	}

exit:
	up (&dev->out_sem); // unlock the device
	return retval;
}

#ifdef TODO
static int
vb_ioctl(struct inode *inode, struct file *file,
			unsigned int cmd, unsigned long arg)
{
	struct usb_vb *dev = (struct usb_vb *)file->private_data;
	int ret = -ENOIOCTLCMD;

// Count how much space the descriptors take
// return error if not enough, perhaps ioctl to get count
// copy descriptors

	return ret;
}
#endif


static void
vb_write_bulk_callback(struct urb *urb,struct pt_regs *regs)
{

	struct dev_state *ps = (struct dev_state *) urb->context;
        dbg("%s ", __FUNCTION__);

        /* sync/async unlink faults aren't errors */
        if (urb->status && !(urb->status == -ENOENT ||
                                urb->status == -ECONNRESET)) {
                dbg("%s - nonzero write bulk status received: %d",
                    __FUNCTION__, urb->status);
        }

	wmb();

	ps->dev = NULL;
	wake_up(&ps->wait);
}



static int
vb_bulk_msg(struct usb_device *usb_dev, unsigned int pipe,
	 void *data, int len, int *actual_length, int timeout)
{

	DECLARE_WAITQUEUE(wait, current);
	struct dev_state ps;
	int status, was_sig;
	struct urb *urb;

	if (len < 0)
	{
		return -EINVAL;
	}



	urb=usb_alloc_urb(0,SLAB_KERNEL);
	if (!urb)
	{
		return -ENOMEM;
	}

        usb_fill_bulk_urb(urb, usb_dev, pipe, data, len,
                    vb_write_bulk_callback, usb_dev);

	// start urb and wait for completion or timeout

	init_waitqueue_head(&ps.wait);


	set_current_state(TASK_UNINTERRUPTIBLE);
	add_wait_queue(&ps.wait, &wait);

	urb->context = &ps;
	ps.dev = usb_dev;
	status = usb_submit_urb(urb,SLAB_KERNEL);
	if (status)
	{
		// something went wrong
		usb_free_urb(urb);
		set_current_state(TASK_RUNNING);
		remove_wait_queue(&ps.wait, &wait);
		return status;
	}

	was_sig = 0; // assume not a signal
	while (ps.dev)
	{
		schedule_timeout(1);
		set_current_state(TASK_UNINTERRUPTIBLE);
		rmb();
		if(signal_pending(current))
		{
			was_sig = 1;
			timeout = 0; // so goes thru timeout code
			break      ;
		}
		if ((timeout) && (--timeout == 0))
		{
			break      ; // timeout occured
		}
	}

	set_current_state(TASK_RUNNING);
	remove_wait_queue(&ps.wait, &wait);

	if (!timeout && ps.dev)
	{
		if (urb->status != -EINPROGRESS) // No callback?!!
		{
			err("usb: raced timeout, pipe 0x%x status %d time left %d",
									    urb->pipe, urb->status, timeout);
			status = urb->status;
		}
		else
		{
			usb_unlink_urb(urb);  // remove urb safely
			if (was_sig)
			{
				status = -EINTR;
			}
			else
			{
				err("usb_control/bulk_msg timeout");
				status = -ETIMEDOUT;
			}
		}
	}
	else
	{
		status = urb->status;
	}

	if (actual_length)
	{
		*actual_length = urb->actual_length;
	}

	usb_free_urb(urb);
	return status;
}


// ------------------------------------------------------------------------
// END code largely ripped from 'drivers/usb/usb.c'
// ------------------------------------------------------------------------

#ifdef BG_READ

static int vb_read_start(struct usb_vb *dev);

// device still locked when called
// post error to 'vb_read'
static void
vb_post_read_error(struct usb_vb *dev, int err)
{
	dev->rdr_head   = 0  ; // flush input data
	dev->rdr_tail   = 0  ; // flush input data
	dev->rdr_status = err;
}

// gets called when read-urb completes
static void
vb_read_restart(struct urb *urb,struct pt_regs *regs)
{
	int status;
	int next_head;
	int minor = (int) (urb->context);
	struct usb_vb *dev;
	u_char *cp;

	dbg("%s-dev(%d) entry", __FUNCTION__,minor);



	// validate minor
	if ((minor < 0) || (minor >= MAX_DEVICES))
	{
		err("%s: invalid minor %d",__FUNCTION__,minor);
		return;
	}

	status = urb->status;

	down (&minor_table_mutex); // lock table
	if ((dev = minor_table[minor]) == NULL)
	{
		// disconnected and closed
		dbg("%s-disconnected and closed",__FUNCTION__);
		up (&minor_table_mutex); // unlock table
		usb_free_urb(urb);
		return;
	}
	up (&minor_table_mutex); // unlock table

	down (&dev->inp_sem); // lock this device object

	// verify that the device is still open
	if (dev->open_count <= 0)
	{
		// connected but closed
		dbg("%s-dev(%d) connected but closed", __FUNCTION__,minor);
		usb_free_urb(urb);
		dev->have_urb = NULL;
		up (&dev->inp_sem); // unlock the device
		return;
	}

	// verify that the device wasn't unplugged
	if (dev->udev == NULL)
	{
		// disconnected but still open
		dbg("%s-dev(%d) disconnected but open", __FUNCTION__,minor);
		usb_free_urb(urb);
		dev->have_urb = NULL;
		vb_post_read_error(dev, -ENODEV);
		up (&dev->inp_sem); // unlock the device
		return;
	}

	if(status) // has some kind of usb/device error
	{
		dbg("%s-dev(%d) status(%d)", __FUNCTION__,minor, status);
		usb_free_urb(urb);
		dev->have_urb = NULL;
		vb_post_read_error(dev, status);
		up (&dev->inp_sem); // unlock the device
		return;
	}

	if((next_head = (dev->rdr_head + 1)) >= RDR_BUF_CNT) { next_head = 0; }

	if(next_head == dev->rdr_tail) // would wrap and look empty if true
	{
		err("dev(%d) no-room, drop (%d) bytes", minor, urb->actual_length);
	}
	else if((urb->actual_length < 1) || (urb->actual_length > dev->inp_size))
	{
		err("dev(%d) bad-length(%d)", minor, urb->actual_length);
	}
	else
	{
		// if gets here then have room, so store data in the read queue

		cp = dev->rdr_buffer + (dev->rdr_head * (dev->inp_size + 1));

		*cp++ = urb->actual_length;

		memcpy(cp, dev->inp_buffer, urb->actual_length);

		dev->rdr_head = next_head;
	}

	// TODO: could re-use the urb instead
	usb_free_urb(urb);
	dev->have_urb = NULL;

	if ((status = vb_read_start(dev)) != 0)
	{
		dbg("%s-dev(%d) read-error(%d)", __FUNCTION__,minor, status);
		vb_post_read_error(dev, status);
	}

	// wakeup 'vb_read' if sleeping
	if (waitqueue_active(&dev->rdr_wqh))
		wake_up_interruptible(&dev->rdr_wqh);

	wake_up(&dev->rdr_pwait);

	// unlock the device
	up (&dev->inp_sem);

	dbg("%s-dev(%d) exit", __FUNCTION__,minor);
}

// should be called with the device locked
static int
vb_read_start(struct usb_vb *dev)
{
	int status;
	struct urb *urb;

	dbg("%s-dev(%d) entry", __FUNCTION__,dev->minor);

	if(dev->have_urb)
	{
		err("%s-dev(%d) already has urb", __FUNCTION__,dev->minor);
		return -EINVAL;
	}

	if ((urb = usb_alloc_urb(0,SLAB_KERNEL)) == NULL)
	{
		err("dev(%d) can't alloc urb", dev->minor);
		return -ENOMEM;
	}

	usb_fill_bulk_urb(urb, dev->udev,
		usb_rcvbulkpipe (dev->udev, dev->inp_endpoint & 0x7f),
		dev->inp_buffer, dev->inp_size, vb_read_restart, 0);

	urb->context = ((void *)(dev->minor));
	if ((status = usb_submit_urb(urb,SLAB_KERNEL)) != 0)
	{
		// something went wrong
		err("%s-dev(%d) submit error(%d)", __FUNCTION__,dev->minor, status);
		usb_free_urb(urb);
		return status;
	}
	dev->have_urb = urb;

	dbg("%s-dev(%d) exit", __FUNCTION__,dev->minor);

	return 0;
}

#endif // BG_READ


#ifdef BG_READ
static unsigned int
vb_poll(struct file *file, struct poll_table_struct *wait)
{
	struct usb_vb *dev = (struct usb_vb *) file->private_data;

	if(dev->rdr_head == dev->rdr_tail) // wait if empty
	{
		poll_wait(file, &dev->rdr_pwait, wait);
	}

	return ((dev->rdr_head == dev->rdr_tail) ? 0 : (POLLIN | POLLRDNORM));
}
#endif

static struct usb_vb *
vb_dev_add(struct usb_interface *interface, struct usb_device *udev, int minor, int pipe, int osize, int isize)
{
	struct usb_vb *dev;

	err("dev_add");
	// allocate memory for our device state and intialize it
	dev = kmalloc (sizeof(struct usb_vb), GFP_KERNEL);
	if (dev == NULL)
	{
		err("Out of memory(%d): struct usb_vb kmalloc", pipe);
		return NULL;
	}
	memset (dev, 0x00, sizeof (*dev));

#ifdef BG_READ
	dev->rdr_buffer = kmalloc ((isize + 1) * RDR_BUF_CNT, GFP_KERNEL);
	if (dev->rdr_buffer == NULL)
	{
		err("Out of memory(%d): rdr_buffer kmalloc", pipe);
		kfree(dev);
		return NULL;
	}
#endif

	init_MUTEX (&dev->out_sem)     ;
	init_MUTEX (&dev->inp_sem)     ;
	dev->udev         = udev       ;
	dev->interface	  = interface;
	dev->minor        = minor      ;
	dev->out_size     = osize      ;
	dev->inp_size     = isize      ;
	dev->out_endpoint = pipe       ;
	dev->inp_endpoint = pipe | 0x80;
	dev->open_count   = 0          ;
#ifdef BG_READ
	dev->rdr_head     = 0          ;
	dev->rdr_tail     = 0          ;
	dev->rdr_status   = 0          ;
	dev->have_urb     = NULL       ;
#endif

	minor_table[minor] = dev;

	return dev;
}

//
// vb_probe
//
// Called by the usb core when a new device is connected that it thinks
// this driver might be interested in.
//
static int 
vb_probe(struct usb_interface *interface, const struct usb_device_id *id)
{
	struct usb_device *udev = interface_to_usbdev(interface);
	struct usb_vb *c_dev = NULL;
	struct usb_vb *v_dev = NULL;
	int minor;
	int retval = 0;

	err("entry");
	// See if the device offered us matches what we can accept
	if ((udev->descriptor.idVendor  != USB_VB_VENDOR_ID ) ||
	    (udev->descriptor.idProduct != USB_VB_PRODUCT_ID))
	{
		return -ENODEV;
	}

	// select a "subminor" number (part of a minor number)
	// get two entries since have two bidirectoional pipes
	down (&minor_table_mutex);
	for (minor = 0; minor < (MAX_DEVICES/2); minor++)
	{
		if (minor_table[minor] == NULL)
			break;
	}
	if (minor >= (MAX_DEVICES/2))
	{
		err("Too many devices plugged in, can not handle this device.");
		goto done;
	}

	// allocate memory for our device state and intialize it (cmd/status)
	c_dev = vb_dev_add (interface, udev, minor, 1, CMD_SIZE, STAT_SIZE);
	if (c_dev == NULL)
	{
		goto done;
	}

	// allocate memory for our device state and intialize it (voice-inp/out)
	v_dev = vb_dev_add (interface, udev, minor + (MAX_DEVICES/2), 2, VOUT_SIZE, VINP_SIZE);
	if (v_dev == NULL)
	{
		vb_delete (c_dev);
		c_dev = NULL;
		goto done;
	}

	// initialize the devfs node for voice and control device and register them
	retval = usb_register_dev(interface,&vb_classv);

	// only set the control device to the interface.

	usb_set_intfdata(interface,c_dev);	

	retval = usb_register_dev(interface,&vb_classc);
        if (retval) {
                // something prevented us from registering this driver 
                err ("Not able to get a minor for this device.");
                usb_set_intfdata (interface, NULL);
                goto error;
        }

	// let the user know what node this device is now attached to
	info("USB Voip-Blaster (%d) connected, ifnum(%d)", minor, c_dev->minor);

done:
	up (&minor_table_mutex);
	return 0;

error:
	err ("Not able register this device.");
	usb_set_intfdata (interface, NULL);
	if(c_dev)
		vb_delete(c_dev);
	if(v_dev)
		vb_delete(v_dev);
	return retval;
}

//
// vb_disconnect
//
// Called by the usb core when the device is removed from the system.
//
static void
vb_disconnect(struct usb_interface *interface)
{
	struct usb_vb *c_dev;
	struct usb_vb *v_dev;
	int minor;

	c_dev = usb_get_intfdata(interface);
	usb_set_intfdata(interface,NULL);

	down (&minor_table_mutex);

	down (&c_dev->out_sem);
	down (&c_dev->inp_sem);

	minor = c_dev->minor; // get before do any cleanup


	// remove our control devfs node
	usb_deregister_dev(interface, &vb_classc);

	// if the device is not opened, then we clean up right now
	if (!c_dev->open_count) {
		up (&c_dev->inp_sem);
		up (&c_dev->out_sem);
		vb_delete (c_dev);
	} else {
		c_dev->udev = NULL; // let 'vb_release' do 'vb_delete' later
		up (&c_dev->inp_sem);
		up (&c_dev->out_sem);
	}

	// now do the second half of the device

	v_dev = minor_table[minor + (MAX_DEVICES/2)];

	down (&v_dev->out_sem);
	down (&v_dev->inp_sem);


	// remove our voice devfs node
	usb_deregister_dev(interface, &vb_classv);

	// if the device is not opened, then we clean up right now
	if (!v_dev->open_count)
	{
		up (&v_dev->out_sem);
		up (&v_dev->inp_sem);
		vb_delete (v_dev);
	}
	else
	{
		v_dev->udev = NULL; // let 'vb_release' do 'vb_delete' later
		up (&v_dev->out_sem);
		up (&v_dev->inp_sem);
	}
	info("USB Voip-Blaster (%d) disconnected", minor);
	up (&minor_table_mutex);
}

// ------------------------------------------------------------------------

// ------------------------------------------------------------------------

//
// usb_vb_init
//
static int __init
usb_vb_init(void)
{
	int result;

	// register this driver with the USB subsystem
	result = usb_register(&vb_driver);
	if (result < 0)
	{
		err("usb_register failed for the "__FILE__" driver. Error number %d",
		    result);
		return -1;
	}

	info(DRIVER_DESC " " DRIVER_VERSION);
	return 0;
}

//
// usb_vb_exit
//
static void __exit
usb_vb_exit(void)
{
	// deregister this driver with the USB subsystem
	usb_deregister(&vb_driver);
}

module_init (usb_vb_init);
module_exit (usb_vb_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);

MODULE_LICENSE("GPL");

//
// The End!
//
