/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-xlate.c,v 1.10 2003/02/13 02:39:00 maf Exp $
 */

#include "ftconfig.h"
#include "ftlib.h"

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>

#if HAVE_STRINGS_H
 #include <strings.h>
#endif

#if HAVE_STRING_H
  #include <string.h>
#endif

#include "ftbuild.h"

int debug;
void usage(void);

#define XLATE_AS0                    0x1
#define XLATE_TO_LEGACY_NETWORK      0x2
#define XLATE_TO_FLOW_NETWORK        0x4
#define XLATE_PRIVACY_MASK           0x8
#define XLATE_SCALE_FLOWS            0x10
#define XLATE_SCALE_OCTETS           0x20
#define XLATE_SCALE_PACKETS          0x40
#define XLATE_SCALE_DURATION         0x80
#define XLATE_SCALE_ALL              (0x10 | 0x20 | 0x40 | 0x80)
#define XLATE_VERSION                0x100
#define XLATE_SRC_TAG                0x200
#define XLATE_DST_TAG                0x400

struct xlate {
  struct ftipmask ftipmask;
  u_int32 flags;
  u_int32 scale;
  u_int32 privacy_mask;
  u_int16 as;
  u_int32 src_tag_mask, dst_tag_mask;
};

int main(int argc, char **argv)
{
  struct ftio ftio_in, ftio_out;
  struct ftprof ftp;
  struct ftver ftv_in, ftv_out;
  struct ftset ftset;
  struct fts3rec_all cur;
  struct xlate xlate;
  struct fts3rec_offsets fo;
  char *rec, out_rec[FT_IO_MAXREC];
  unsigned int v1, v2;
  int i, n, keep_input_time;
  u_int32 total_flows, cap_start, cap_end;
  u_int32 time_start, time_end;

  /* init fterr */
  fterr_setid(argv[0]);

  bzero(&ftv_in, sizeof ftv_in);
  bzero(&ftv_out, sizeof ftv_out);
  total_flows = 0;
  xlate.flags = 0;

  /* profile */
  ftprof_start (&ftp);

  /* defaults + no compression */
  ftset_init(&ftset, 0);

  keep_input_time = 0;

  while ((i = getopt(argc, argv, "0:b:C:d:fh?klm:s:t:T:V:z:")) != -1)

    switch (i) {

    case '0': /* AS */
      xlate.flags |= XLATE_AS0;
      xlate.as = atoi(optarg);
      break;

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        ftset.byte_order = FT_HEADER_BIG_ENDIAN;
      else
        fterr_errx(1, "expecting \"big\" or \"little\"");
      break;

    case 'C': /* comment field */
      ftset.comments = optarg;
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* flow netmask */
      xlate.flags |= XLATE_TO_FLOW_NETWORK;
      break;

    case 'k': /* keep the start/end time from the input */
      keep_input_time = 1;
      break;

    case 'l': /* legacy netmask */
      xlate.flags |= XLATE_TO_LEGACY_NETWORK;
      break;

    case 'm': /* privacy_mask */
      xlate.privacy_mask = scan_ip(optarg);
      xlate.flags |= XLATE_PRIVACY_MASK;
      break;

    case 's': /* scale factor */
      xlate.flags |= XLATE_SCALE_FLOWS | XLATE_SCALE_OCTETS |
        XLATE_SCALE_PACKETS | XLATE_SCALE_DURATION;
      xlate.scale = atoi(optarg);
      break;

    case 't': /* source tag */
      xlate.flags |= XLATE_SRC_TAG;
      xlate.src_tag_mask = strtoul(optarg, (char **)0L, 0);
      break;

    case 'T': /* destination tag */
      xlate.flags |= XLATE_DST_TAG;
      xlate.dst_tag_mask = strtoul(optarg, (char **)0L, 0);
      break;

    case 'V': /* version */
     n = sscanf(optarg, "%u.%u", &v1, &v2);
      if (n == 1) {
        ftv_out.s_version = FT_IO_SVERSION;
        ftv_out.d_version = v1;
        ftv_out.set = 1;
      } else if (n == 2) {
        ftv_out.s_version = FT_IO_SVERSION;
        ftv_out.d_version = v1;
        ftv_out.agg_method = v2;
        ftv_out.agg_version = 2;
        ftv_out.set = 1;
      } else {
        fterr_errx(1, "Version scan failed");
      }
      xlate.flags |= XLATE_VERSION;
      break;

    case 'z': /* compress level */
      ftset.z_level = atoi(optarg);
      if ((ftset.z_level < 0) || (ftset.z_level > 9))
        fterr_errx(1, "Compression level must be between 0 and 9");
      break;

    case 'h': /* help */
    case '?': 
    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (argc - optind)
    fterr_errx(1, "Extra arguments starting with %s.", argv[optind]);

  if (xlate.flags & XLATE_PRIVACY_MASK)
    /* setup for ftrec_mask_ip */
    ftrec_compute_mask(&xlate.ftipmask, xlate.privacy_mask, xlate.privacy_mask,
      ftset.byte_order);

  /* input is stdin */
  if (ftio_init(&ftio_in, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  /* output is stdout */
  if (ftio_init(&ftio_out, 1, FT_IO_FLAG_WRITE |
    ((ftset.z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  /* preserve start/end time from input stream? */
  if (keep_input_time) {

    time_start = ftio_get_cap_start(&ftio_in);
    time_end = ftio_get_cap_end(&ftio_in);
    
    if (time_start && time_end) {

      ftio_set_preloaded(&ftio_out, 1);
      ftio_set_cap_time(&ftio_out, time_start, time_end);

    } 
  
  } /* keep_input_time */

  ftio_get_ver(&ftio_in, &ftv_in);

  ftv_in.s_version = FT_IO_SVERSION;

  if (!ftv_out.set)
    bcopy(&ftv_in, &ftv_out, sizeof ftv_in);

  /* set the version information in the io stream */
  if (ftio_set_ver(&ftio_out, &ftv_out) < 0)
    fterr_errx(1, "ftio_set_ver(): failed");

  ftio_set_comment(&ftio_out, ftset.comments);
  ftio_set_byte_order(&ftio_out, ftset.byte_order);
  ftio_set_z_level(&ftio_out, ftset.z_level);
  ftio_set_streaming(&ftio_out, 1);
  ftio_set_debug(&ftio_out, debug);
  ftio_set_cap_time(&ftio_out, cap_start, cap_end);
  ftio_set_flows_count(&ftio_out, total_flows);

  if (ftio_write_header(&ftio_out) < 0)
    fterr_errx(1, "ftio_write_header(): failed");

  /* check sanity */
  if (xlate.flags & (XLATE_SRC_TAG)) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_SRC_TAG))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  if (xlate.flags & (XLATE_DST_TAG)) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_DST_TAG))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  if (xlate.flags & XLATE_TO_FLOW_NETWORK) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_SRCADDR | FT_XFIELD_DSTADDR
      | FT_XFIELD_SRC_MASK | FT_XFIELD_DST_MASK))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  if (xlate.flags & XLATE_AS0) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_SRC_AS | FT_XFIELD_DST_AS))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  if (xlate.flags & XLATE_PRIVACY_MASK) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_SRCADDR | FT_XFIELD_DSTADDR))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  if (xlate.flags & XLATE_SCALE_OCTETS) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_DOCTETS))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  if (xlate.flags & XLATE_SCALE_PACKETS) {
    if (ftio_check_xfield(&ftio_in, FT_XFIELD_DPKTS))
      fterr_errx(1, "Flow record missing required field for format.");
  }

  fts3rec_compute_offsets(&fo, &ftv_in);

  while ((rec = ftio_read(&ftio_in))) {

    ++total_flows;

    if (xlate.flags & XLATE_TO_LEGACY_NETWORK) {

      cur.dstaddr = ((u_int32*)(rec+fo.dstaddr));
      cur.srcaddr = ((u_int32*)(rec+fo.srcaddr));

      if ((*cur.srcaddr & 0x80000000) == 0)
        *cur.srcaddr &= 0xFF000000;
      else if ((*cur.srcaddr & 0xC0000000) == 0x80000000)
        *cur.srcaddr &= 0xFFFF0000;
      else if ((*cur.srcaddr & 0xC0000000) == 0xC0000000)
        *cur.srcaddr &= 0xFFFFFF00;

      if ((*cur.dstaddr & 0x80000000) == 0)
        *cur.dstaddr &= 0xFF000000;
      else if ((*cur.dstaddr & 0xC0000000) == 0x80000000)
        *cur.dstaddr &= 0xFFFF0000;
      else if ((*cur.dstaddr & 0xC0000000) == 0xC0000000)
        *cur.dstaddr &= 0xFFFFFF00;

    }

    if (xlate.flags & XLATE_TO_FLOW_NETWORK) {

      cur.dstaddr = ((u_int32*)(rec+fo.dstaddr));
      cur.srcaddr = ((u_int32*)(rec+fo.srcaddr));
      cur.src_mask = ((u_int8*)(rec+fo.src_mask));
      cur.dst_mask = ((u_int8*)(rec+fo.dst_mask));

      *cur.srcaddr &= ipv4_len2mask(*cur.src_mask);
      *cur.dstaddr &= ipv4_len2mask(*cur.dst_mask);

    }

    if (xlate.flags & XLATE_AS0) {

      cur.src_as = ((u_int16*)(rec+fo.src_as));
      cur.dst_as = ((u_int16*)(rec+fo.dst_as));

      if (*cur.src_as == 0)
        *cur.src_as = xlate.as;

      if (*cur.dst_as == 0)
        *cur.dst_as = xlate.as;

    }

    if (xlate.flags & XLATE_PRIVACY_MASK)
      ftrec_mask_ip(rec, &ftv_in, &xlate.ftipmask);

/* XXX split flow if overflow */
    if (xlate.flags & XLATE_SCALE_OCTETS) {
      cur.dOctets = ((u_int32*)(rec+fo.dOctets));
      *cur.dOctets *= xlate.scale;
    }

    if (xlate.flags & XLATE_SCALE_PACKETS) {
      cur.dPkts = ((u_int32*)(rec+fo.dPkts));
      *cur.dPkts *= xlate.scale;
    }

    if (xlate.flags & XLATE_SRC_TAG) {
      cur.src_tag = ((u_int32*)(rec+fo.src_tag));
      *cur.src_tag &= xlate.src_tag_mask;
    }

    if (xlate.flags & XLATE_DST_TAG) {
      cur.dst_tag = ((u_int32*)(rec+fo.dst_tag));
      *cur.dst_tag &= xlate.dst_tag_mask;
    }

    if (xlate.flags & XLATE_VERSION) {

      ftrec_xlate(rec, &ftv_in, &out_rec, &ftv_out);

      if (ftio_write(&ftio_out, &out_rec) < 0)
        fterr_errx(1, "ftio_write(): failed");

    } else {

      if (ftio_write(&ftio_out, rec) < 0)
        fterr_errx(1, "ftio_write(): failed");

    }

  } /* while */


  if (ftio_close(&ftio_out) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (ftio_close(&ftio_in) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (debug > 0) {
    ftprof_end (&ftp, total_flows);
    ftprof_print(&ftp, argv[0], stderr);
  }

  return 0;

} /* main */

void usage(void) {

  fprintf(stderr, "Usage: flow-xlate [-fhkl] [-0 AS0_substitution] [-b big|little] [-C comment]\n");
  fprintf(stderr, "       [-d debug_level] [-m privacy_mask] [-s scale] [-t src_tag_mask]\n");
  fprintf(stderr, "       [-T dst_tag_mask] [-V pdu_version] [-z z_level]\n");
  fprintf(stderr, "\n%s version %s: built by %s\n", PACKAGE, VERSION, FT_PROG_BUILD);

}

