/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-split.c,v 1.3 2001/06/12 16:28:07 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/types.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <fcntl.h>
#include "ftlib.h"
#include "support.h"

int debug;
void usage();

#define MAXPATHNAME 256

int main(int argc, char **argv)
{
  extern char *optarg;
  extern int optind, debug;
  struct ftio ftio_in, ftio_out;
  struct ftprof ftp;
  struct ftver ftv;
  struct ftset ftset;
  struct fts3rec_gen *rec;
  int i, out_fd, new_file, out_id;
  char *out_path, out_fname[MAXPATHNAME];
  u_int32 total_flows, time_start;
  u_int32 max_flows, max_time;
  u_int32 cap_start, cap_end;

  /* init fterr */
  fterr_setid(argv[0]);

  bzero(&ftv, sizeof ftv);
  total_flows = 0;
  out_path = "split.";
  out_fd = -1;
  out_id = 0;
  max_flows = max_time = 0;

  /* profile */
  ftprof_start (&ftp);

  /* defaults */
  ftset_init(&ftset, -1);

  while ((i = getopt(argc, argv, "b:C:d:h?N:o:T:z:")) != -1)

    switch (i) {

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        ftset.byte_order = FT_HEADER_BIG_ENDIAN;
      else
        fterr_errx(1, "expecting \"big\" or \"little\"");
      break;

    case 'C': /* comment field */
      ftset.comments = optarg;
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'N': /* max flows */
      max_flows = strtoul(optarg, (char **)0L, 10);
      break;

    case 'o': /* base output filename */
      out_path = optarg;
      if (strlen(out_path) > (MAXPATHNAME-25))
        fterr_errx(1, "base output too long");
      break;

    case 'T': /* max time */
      max_time = strtoul(optarg, (char **)0L, 10);
      break;

    case 'z': /* compress level */
      ftset.z_level = atoi(optarg);
      if ((ftset.z_level < 0) || (ftset.z_level > 9))
        fterr_errx(1, "Compression level must be between 0 and 9");
      break;

    case 'h': /* help */
    case '?': 
    default:
      usage();
      exit (1);
      break;

    } /* switch */


  i = optind;

  /* input is stdin */
  if (ftio_init(&ftio_in, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  ftio_get_ver(&ftio_in, &ftv);

  ftv.s_version = FT_IO_SVERSION;

  /* signal new file is needed */
  new_file = 1;

  while ((rec = ftio_read(&ftio_in))) {

    /* create new output file */
    if (new_file) {

      new_file = 0;

      if (out_fd != -1) {

        ftio_set_cap_time(&ftio_out, cap_start, cap_end);
        ftio_set_flows_count(&ftio_out, total_flows);
        ftio_set_streaming(&ftio_out, 0);

        if (ftio_write_header(&ftio_out) < 0)
          fterr_errx(1, "ftio_write_header(): failed");

        if (ftio_close(&ftio_out) < 0)
          fterr_errx(1, "ftio_close(): failed");

      } /* out_fd != -1 */

      sprintf(out_fname, "%s.%d", out_path, out_id);

      if ((out_fd = open(out_fname,  O_WRONLY|O_CREAT|O_TRUNC, 0644)) == -1)
        fterr_err(1, "open(%s)", out_fname);

      /* output to out_fd */    
      if (ftio_init(&ftio_out, out_fd, FT_IO_FLAG_WRITE |
        ((ftset.z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
        fterr_errx(1, "ftio_init(): failed");

      /* set the version information in the io stream */
      if (ftio_set_ver(&ftio_out, &ftv) < 0)
        fterr_errx(1, "ftio_set_ver(): failed");

      ftio_set_comment(&ftio_out, ftset.comments);
      ftio_set_byte_order(&ftio_out, ftset.byte_order);
      ftio_set_z_level(&ftio_out, ftset.z_level);
      ftio_set_streaming(&ftio_out, 1);
      ftio_set_debug(&ftio_out, debug);
      ftio_set_cap_time(&ftio_out, cap_start, cap_end);
      ftio_set_flows_count(&ftio_out, total_flows);

      if (ftio_write_header(&ftio_out) < 0)
        fterr_errx(1, "ftio_write_header(): failed");

      total_flows = 0;

      /* LC's not synched very well */
      if (rec->unix_secs > time_start) {
        time_start = rec->unix_secs;
        cap_start = time_start;
      }

      ++out_id;

    } /* new_file */

    ++total_flows;

    /* signal new file if total_flows >= max_flows */
    if (max_flows && (total_flows >= max_flows))
      new_file = 1;

    /* signal new file if time elapsed > max_time */
    if ((max_time && (rec->unix_secs > time_start) &&
      (rec->unix_secs - time_start) >= max_time)) {
      /* LC's not synch'd very well */
      time_start = rec->unix_secs;
      cap_start = time_start;
      new_file = 1;
    }

    cap_end = rec->unix_secs;

    if (ftio_write(&ftio_out, rec) < 0)
      fterr_errx(1, "ftio_write(): failed");

  }

  if (out_fd != -1) {

    ftio_set_cap_time(&ftio_out, cap_start, cap_end);
    ftio_set_flows_count(&ftio_out, total_flows);
    ftio_set_streaming(&ftio_out, 0);

    if (ftio_write_header(&ftio_out) < 0)
      fterr_errx(1, "ftio_write_header(): failed");

    if (ftio_close(&ftio_out) < 0)
      fterr_errx(1, "ftio_close(): failed");

  } /* out_fd */

  if (ftio_close(&ftio_in) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (debug > 0) {
    ftprof_end (&ftp, total_flows);
    ftprof_print(&ftp, argv[0], stderr);
  }

  return 0;

} /* main */

void usage() {

  fprintf(stderr, "flow-split:\n\n");
  fprintf(stderr, " -b s  set output byte order (little or big)\n");
  fprintf(stderr, " -C s  set comment field in output\n");
  fprintf(stderr, " -d #  set debug level.\n");
  fprintf(stderr, " -N #  max number of flows in split\n");
  fprintf(stderr, " -o s  output to file s instead of stdout\n");
  fprintf(stderr, " -T #  max seconds of flows in split\n");
  fprintf(stderr, " -z #  set compression level (0..9).\n");
  fprintf(stderr, "\n\n");
}

