/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-import.c,v 1.2 2001/07/06 14:26:08 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <time.h>
#include <fcntl.h>
#include <ftlib.h>
#include "support.h"
#include "fmt.h"
#include "cflowd.h"

struct options {
  struct ftver ftv;
  u_int64 ft_mask;
  int set_format;
  u_long records;
};

struct jump {
    int (*where)(struct ftio *ftio, struct options *opt);
};

int format0(struct ftio *ftio, struct options *opt);
int format1(struct ftio *ftio, struct options *opt);
int format2(struct ftio *ftio, struct options *opt);

void usage(void);

struct jump format[] = {{format0}, {format1}, {format2}};

#define NFORMATS 3 /* nformats + 1 */

int main(int argc, char **argv)
{
  extern char *optarg;
  extern struct jump format[];
  struct ftset ftset;
  struct ftio ftio;
  struct options opt;
  int i, n, format_index, ret;
  unsigned int v1, v2;
  int debug;

  /* init fterr */
  fterr_setid(argv[0]);

  /* defaults + no compression */
  ftset_init(&ftset, 0);
  
  debug = 0;
  format_index = 0;
  bzero(&opt, sizeof opt);

  opt.ft_mask = 0xFFFFFFFFFFFFFFFFLL;

  while ((i = getopt(argc, argv, "b:d:f:h?m:V:z:")) != -1)

    switch (i) {

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        ftset.byte_order = FT_HEADER_BIG_ENDIAN;
      else
        fterr_errx(1, "expecting \"big\" or \"little\"");
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* format */
      format_index = atoi(optarg);
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'm': /* mask */
      opt.ft_mask = strtouq(optarg, (char **)0L, 0);
      opt.set_format = 1;
      break;

   case 'V': /* PDU version */
      n = sscanf(optarg, "%u.%u", &v1, &v2);
      if (n == 1) {
        opt.ftv.s_version = FT_IO_SVERSION;
        opt.ftv.d_version = v1;
        opt.ftv.set = 1;
      } else if (n == 2) {
        opt.ftv.s_version = FT_IO_SVERSION;
        opt.ftv.d_version = v1;
        opt.ftv.agg_method = v2;
        opt.ftv.agg_version = 2;
        opt.ftv.set = 1;
      } else
        fterr_errx(1, "Version scan failed");
      break;

    case 'z': /* compress level */
      ftset.z_level = atoi(optarg);
      if ((ftset.z_level < 0) || (ftset.z_level > 9))
        fterr_errx(1, "Compression level must be between 0 and 9");
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (format_index >= NFORMATS)
    fterr_errx(1, "No such format, %d", format_index);

  if (!opt.ftv.set)
    fterr_errx(1, "Must specify export version to store flows");

  /* output to stdout */
  if (ftio_init(&ftio, 1, FT_IO_FLAG_WRITE |
    ((ftset.z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
    fterr_errx(1, "ftio_init(): failed");
    
  ftio_set_comment(&ftio, "flow-import");
  ftio_set_cap_hostname(&ftio, "flow-import");
  ftio_set_byte_order(&ftio, ftset.byte_order);
  ftio_set_z_level(&ftio, ftset.z_level);
  ftio_set_streaming(&ftio, 1);
  ftio_set_debug(&ftio, debug);

  if (ftio_set_ver(&ftio, &opt.ftv) < 0)
    fterr_errx(1, "ftio_set_ver(): failed");

  /* header first */
  if (ftio_write_header(&ftio) < 0)
    fterr_errx(1, "ftio_write_header(): failed");

  ret = format[format_index].where(&ftio, &opt);

  if (ftio_close(&ftio) < 0)
    fterr_errx(1, "ftio_close(): failed");

  fprintf(stderr, "%s: Imported %lu records\n", argv[0], opt.records);
      
  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-import:\n\n");
  fprintf(stderr, " -b [big|little] output byte order (little or big)\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -f # format\n");
  fprintf(stderr, "    0 cflowd\n");
  fprintf(stderr, "    1 reserved\n");
  fprintf(stderr, "    2 ASCII CSV\n");
  fprintf(stderr, " -m # fields\n");
  fprintf(stderr, " -V #  version of store as.  Use #.# for V8\n");
  fprintf(stderr, "   1   - no sequence #, engine id, or AS\n");
  fprintf(stderr, "   5   - 1+ sequence #, engine id, and AS\n");
  fprintf(stderr, "   6   - 5+ encapsulation size\n");
  fprintf(stderr, "   7   - 5+ router shortcut (Catalyst switches)\n");
  fprintf(stderr, "   8.1 - AS aggregation\n");
  fprintf(stderr, "   8.2 - Protocol Port aggregation\n");
  fprintf(stderr, "   8.3 - Source Prefix aggregation\n");
  fprintf(stderr, "   8.4 - Destination Prefix aggregation\n");
  fprintf(stderr, "   8.5 - Prefix aggregation\n");
  fprintf(stderr, "\n\n");

} /* usage */

/*
 * function: format0
 *
 * import from cflowd files
*/
int format0(struct ftio *ftio, struct options *opt)
{
  struct fts3rec_offsets fo;
  size_t rlen;
  u_char buf[FT_IO_MAXREC];
  char *rec;
  u_int32 ui32, Start;
  u_int16 ui16;
  u_int8 ui8;
  u_int32 cfdmask, index;
  int ret;

  rec = (char*)&buf;

  fts3rec_compute_offsets(&fo, &opt->ftv);

  switch (opt->ftv.d_version) {

    case 1:
      cfdmask = CF_INDEX_V1_MASK;
      break;

    case 5:
      cfdmask = CF_INDEX_V5_MASK;
      break;

    case 6:
      cfdmask = CF_INDEX_V6_MASK;
      break;

    case 7:
      cfdmask = CF_INDEX_V7_MASK;
      break;

    case 8:

      switch (opt->ftv.agg_method) {

        case 1:
          cfdmask = CF_INDEX_V8_1_MASK;
          break;

        case 2:
          cfdmask = CF_INDEX_V8_2_MASK;
          break;

        case 3:
          cfdmask = CF_INDEX_V8_3_MASK;
          break;

        case 4:
          cfdmask = CF_INDEX_V8_4_MASK;
          break;

        case 5:
          cfdmask = CF_INDEX_V8_5_MASK;
          break;

       } /* switch */
       break;

  } /* switch */


  while (!(feof(stdin))) {

    ret = -1;

    bzero(rec, FT_IO_MAXREC);
    Start = 0;

    if ((rlen = fread(&index, sizeof (index), 1, stdin) != 1))
      goto done;
    index = ntohl(index);

    if (index & CF_ROUTERMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_ROUTERMASK)
        *((u_int32*)(rec+fo.exaddr)) = ui32;
    }

    if (index & CF_SRCIPADDRMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_SRCIPADDRMASK)
        *((u_int32*)(rec+fo.srcaddr)) = ui32;
    }

    if (index & CF_DSTIPADDRMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_DSTIPADDRMASK)
        *((u_int32*)(rec+fo.dstaddr)) = ui32;
    }

    if (index & CF_INPUTIFINDEXMASK) {
      if ((rlen = fread(&ui16, sizeof (ui16), 1, stdin) != 1))
        goto done;
      ui16 = ntohs(ui16);
      if (cfdmask & CF_DSTIPADDRMASK)
        *((u_int16*)(rec+fo.input)) = ui16;
    }

    if (index & CF_OUTPUTIFINDEXMASK) {
      if ((rlen = fread(&ui16, sizeof (ui16), 1, stdin) != 1))
        goto done;
      ui16 = ntohs(ui16);
      if (cfdmask & CF_OUTPUTIFINDEXMASK)
        *((u_int16*)(rec+fo.output)) = ui16;
    }

    if (index & CF_SRCPORTMASK) {
      if ((rlen = fread(&ui16, sizeof (ui16), 1, stdin) != 1))
        goto done;
      ui16 = ntohs(ui16);
      if (cfdmask & CF_SRCPORTMASK)
        *((u_int16*)(rec+fo.srcport)) = ui16;
    }

    if (index & CF_DSTPORTMASK) {
      if ((rlen = fread(&ui16, sizeof (ui16), 1, stdin) != 1))
        goto done;
      ui16 = ntohs(ui16);
      if (cfdmask & CF_DSTPORTMASK)
        *((u_int16*)(rec+fo.dstport)) = ui16;
    }

    if (index & CF_PKTSMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_PKTSMASK)
        *((u_int32*)(rec+fo.dPkts)) = ui32;
    }

    if (index & CF_BYTESMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_BYTESMASK)
        *((u_int32*)(rec+fo.dOctets)) = ui32;
    }

    if (index & CF_IPNEXTHOPMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_IPNEXTHOPMASK)
        *((u_int32*)(rec+fo.nexthop)) = ui32;
    }

    if (index & CF_STARTTIMEMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      Start = ui32;
      if (cfdmask & CF_STARTTIMEMASK)
        *((u_int32*)(rec+fo.unix_secs)) = ui32;
    }

    if (index & CF_ENDTIMEMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_ENDTIMEMASK) {
        if (Start)
          *((u_int32*)(rec+fo.Last)) = (ui32 - Start) * 1000;
        else
          *((u_int32*)(rec+fo.unix_secs)) = ui32;
        
      }
    }

    if (index & CF_PROTOCOLMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_PROTOCOLMASK)
        *((u_int8*)(rec+fo.prot)) = ui8;
    }

    if (index & CF_TOSMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_TOSMASK)
        *((u_int8*)(rec+fo.tos)) = ui8;
    }

    if (index & CF_SRCASMASK) {
      if ((rlen = fread(&ui16, sizeof (ui16), 1, stdin) != 1))
        goto done;
      ui16 = ntohs(ui16);
      if (cfdmask & CF_SRCASMASK)
        *((u_int16*)(rec+fo.src_as)) = ui16;
    }

    if (index & CF_DSTASMASK) {
      if ((rlen = fread(&ui16, sizeof (ui16), 1, stdin) != 1))
        goto done;
      ui16 = ntohs(ui16);
      if (cfdmask & CF_DSTASMASK)
        *((u_int16*)(rec+fo.dst_as)) = ui16;
    }

    if (index & CF_SRCMASKLENMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_SRCMASKLENMASK)
        *((u_int8*)(rec+fo.src_mask)) = ui8;
    }

    if (index & CF_DSTMASKLENMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_DSTMASKLENMASK)
        *((u_int8*)(rec+fo.dst_mask)) = ui8;
    }

    if (index & CF_TCPFLAGSMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_TCPFLAGSMASK)
        *((u_int8*)(rec+fo.tcp_flags)) = ui8;
    }

    if (index & CF_INPUTENCAPMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_INPUTENCAPMASK)
        *((u_int8*)(rec+fo.in_encaps)) = ui8;
    }

    if (index & CF_OUTPUTENCAPMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_OUTPUTENCAPMASK)
        *((u_int8*)(rec+fo.out_encaps)) = ui8;
    }

    if (index & CF_PEERNEXTHOPMASK) {
      if ((rlen = fread(&ui32, sizeof (ui32), 1, stdin) != 1))
        goto done;
      ui32 = ntohl(ui32);
      if (cfdmask & CF_PEERNEXTHOPMASK)
        *((u_int32*)(rec+fo.peer_nexthop)) = ui32;
    }

    if (index & CF_ENGINETYPEMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_ENGINETYPEMASK)
        *((u_int8*)(rec+fo.engine_type)) = ui8;
    }

    if (index & CF_ENGINEIDMASK) {
      if ((rlen = fread(&ui8, sizeof (ui8), 1, stdin) != 1))
        goto done;
      if (cfdmask & CF_ENGINEIDMASK)
        *((u_int8*)(rec+fo.engine_id)) = ui8;
    }


    if (ftio_write(ftio, rec) < 0) {
      fterr_warnx("ftio_write(): failed");
      return -1;
    }

    ++opt->records;

    ret = 0;

  }

done:

  return ret;

} /* format0 */

int format1(struct ftio *ftio, struct options *opt)
{
  fterr_warnx("Not implemented");
  return -1;
} /* format1 */

/*
 * function: format1
 *
 * import from ASCII CSV format
*/
int format2(struct ftio *ftio, struct options *opt)
{
  struct fts3rec_offsets fo;
  u_char buf[FT_IO_MAXREC];
  char inbuf[1024], *inbufp, *field;
  char *rec;
  u_int64 dmask, inmask;
  int ret;

  rec = (char*)&buf;

  fts3rec_compute_offsets(&fo, &opt->ftv);

  switch (opt->ftv.d_version) {

    case 1:
      dmask = FT_XFIELD_V1_MASK;
      break;

    case 5:
      dmask = FT_XFIELD_V5_MASK;
      break;

    case 6:
      dmask = FT_XFIELD_V6_MASK;
      break;

    case 7:
      dmask = FT_XFIELD_V7_MASK;
      break;

    case 8:

      switch (opt->ftv.agg_method) {

        case 1:
          dmask = FT_XFIELD_V8_1_MASK;
          break;

        case 2:
          dmask = FT_XFIELD_V8_2_MASK;
          break;

        case 3:
          dmask = FT_XFIELD_V8_3_MASK;
          break;

        case 4:
          dmask = FT_XFIELD_V8_4_MASK;
          break;

        case 5:
          dmask = FT_XFIELD_V8_5_MASK;
          break;

       } /* switch */
       break;

  } /* switch */

  /* no user specified format then default to one appropriate for version */
  if (!opt->set_format)
    inmask = dmask;
  else
    inmask = opt->ft_mask;

  while (!(feof(stdin))) {

    ret = 0;

    if (!fgets(inbuf, 1024, stdin))
      goto done;

    ret = -1;

    bzero(rec, FT_IO_MAXREC);

    inbufp = inbuf;

    if (inmask & FT_XFIELD_UNIX_SECS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_UNIX_SECS))
        *((u_int32*)(rec+fo.unix_secs)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_UNIX_NSECS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_UNIX_NSECS))
        *((u_int32*)(rec+fo.unix_nsecs)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_SYSUPTIME) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_SYSUPTIME))
        *((u_int32*)(rec+fo.sysUpTime)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_EXADDR) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_EXADDR))
        *((u_int32*)(rec+fo.exaddr)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_DFLOWS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DFLOWS))
        *((u_int32*)(rec+fo.dFlows)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_DPKTS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DPKTS))
        *((u_int32*)(rec+fo.dPkts)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_DOCTETS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DOCTETS))
        *((u_int32*)(rec+fo.dOctets)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_FIRST) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_FIRST))
        *((u_int32*)(rec+fo.First)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_LAST) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_LAST))
        *((u_int32*)(rec+fo.Last)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_ENGINE_TYPE) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_ENGINE_TYPE))
        *((u_int8*)(rec+fo.engine_type)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_ENGINE_ID) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_ENGINE_ID))
        *((u_int8*)(rec+fo.engine_id)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_SRCADDR) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_SRCADDR))
        *((u_int32*)(rec+fo.srcaddr)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_DSTADDR) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DSTADDR))
        *((u_int32*)(rec+fo.dstaddr)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_SRC_PREFIX) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_SRC_PREFIX))
        *((u_int32*)(rec+fo.src_prefix)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_DST_PREFIX) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DST_PREFIX))
        *((u_int32*)(rec+fo.dst_prefix)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_NEXTHOP) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_NEXTHOP))
        *((u_int32*)(rec+fo.nexthop)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_INPUT) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_INPUT))
        *((u_int16*)(rec+fo.input)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_OUTPUT) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_OUTPUT))
        *((u_int16*)(rec+fo.output)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_SRCPORT) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_SRCPORT))
        *((u_int16*)(rec+fo.srcport)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_DSTPORT) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DSTPORT))
        *((u_int16*)(rec+fo.dstport)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_PROT) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_PROT))
        *((u_int8*)(rec+fo.prot)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_TOS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_TOS))
        *((u_int8*)(rec+fo.tos)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_TCP_FLAGS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_TCP_FLAGS))
        *((u_int8*)(rec+fo.tcp_flags)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_SRC_MASK) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_SRC_MASK))
        *((u_int8*)(rec+fo.src_mask)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_DST_MASK) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DST_MASK))
        *((u_int8*)(rec+fo.dst_mask)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_SRC_AS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_SRC_AS))
        *((u_int16*)(rec+fo.src_as)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_DST_AS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_DST_AS))
        *((u_int16*)(rec+fo.dst_as)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_IN_ENCAPS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_IN_ENCAPS))
        *((u_int8*)(rec+fo.in_encaps)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_OUT_ENCAPS) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_OUT_ENCAPS))
        *((u_int8*)(rec+fo.out_encaps)) = strtoul(field, (char **)0L, 0);
    }

    if (inmask & FT_XFIELD_PEER_NEXTHOP) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_PEER_NEXTHOP))
        *((u_int32*)(rec+fo.out_encaps)) = scan_ip(field);
    }

    if (inmask & FT_XFIELD_ROUTER_SC) {
      field = strsep(&inbufp, ",");
      if (field && (dmask & FT_XFIELD_ROUTER_SC))
        *((u_int32*)(rec+fo.router_sc)) = scan_ip(field);
    }
      
    if (ftio_write(ftio, rec) < 0) {
      fterr_warnx("ftio_write(): failed");
      goto done;
    }

    ++opt->records;

  } /* while */

done:

  return ret;

} /* format2 */
