/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-filter.c,v 1.17 2001/06/18 00:41:43 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <time.h>
#include <fcntl.h>
#include "ftlib.h"
#include "support.h"
#include "acl2.h"
#include "aclyacc.h"

/*
 * TODO
 * extended ACL's
 * update for v8
 */

int debug;
int ip_net_only;

struct acl_list acl_list;

int yyparse (void);

void usage();

int main(int argc, char **argv)
{
  extern char *optarg;
  extern struct acl_list acl_list;
  extern FILE *yyin;
  struct ftio ftio_in, ftio_out;
  struct ftset ftset;
  struct ftver ftv;
  struct ftprof ftp;
  struct fts3rec_v5 *rec_v5;
  struct fts3rec_gen *rec_gen;
  int i, ret;
  void *rec;
  char *acl_fname, *acl_std_src_name, *acl_std_dst_name;
  char *acl_ext_name, *str, *strm;
  int acl_std_src_index, acl_std_src_index2;
  int acl_std_dst_index, acl_std_dst_index2;
  int acl_ext_index, acl_ext_index2;
  struct acl_ip_ext_entry tmp_ext;
  int filter_input, filter_output, filter_srcport, filter_dstport;
  int filter_prot, filter_srcas, filter_dstas, filter_tos;
  char in_tbl[65536], out_tbl[65536], src_tbl[65536], dst_tbl[65536];
  char srcas_tbl[65536], dstas_tbl[65536], tos_tbl[65536];
  char prot_tbl[256];
  u_char tos_mask, tos;
  u_int64 total_flows;
  int as_present;

  /* init fterr */
  fterr_setid(argv[0]);

  /* profile */
  ftprof_start (&ftp);

  bzero(&ftv, sizeof ftv);

  /* defaults + no compression */
  ftset_init(&ftset, 0);

  /* init */
  bzero(&acl_list, sizeof acl_list);
  acl_fname = acl_std_src_name = acl_std_dst_name = (char*)0L;
  acl_ext_name = (char*)0L;
  acl_std_src_index = acl_std_dst_index = -1;
  acl_ext_index = -1;
  bzero(&tmp_ext, sizeof tmp_ext);
  total_flows = 0;
  tos_mask = 0xff;

  filter_input = filter_output = filter_srcport = filter_dstport = 0;
  filter_prot = filter_srcas = filter_dstas = filter_tos = 0;

  while ((i = getopt(argc, argv, "a:A:b:C:d:f:p:P:r:S:t:D:E:z:i:I:")) != -1)
    switch (i) {

    case 'a': /* src AS filter list */

      if (load_lookup(optarg, 65536, srcas_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_srcas = 1;
      break;

    case 'A': /* dst AS filter list */

      if (load_lookup(optarg, 65536, dstas_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_dstas = 1;
      break;

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        ftset.byte_order = FT_HEADER_BIG_ENDIAN;
      else
        fterr_errx(1, "expecting \"big\" or \"little\"");
      break;

    case 'C': /* comment field */
      ftset.comments = optarg;
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* acl file name */
      acl_fname = optarg;
      break;

    case 'i': /* input filter interface list */

      if (load_lookup(optarg, 65536, in_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_input = 1;
      break;

    case 'I': /* output filter interface list */

      if (load_lookup(optarg, 65536, out_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_output = 1;
      break;

    case 'P': /* filter dstport */

      if (load_lookup(optarg, 65536, dst_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_dstport = 1;
      break;

    case 'p': /* filter srcport */

      if (load_lookup(optarg, 65536, src_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_srcport = 1;
      break;

    case 'r': /* filter protocol */

      if (load_lookup(optarg, 256, prot_tbl))
        fterr_errx(1, "load_lookup(): failed");

      filter_prot = 1;
      break;

    case 'S': /* src ip standard access list filter */
      acl_std_src_name = optarg;
      break;

    case 't': /* ToS Filter */

      if (!(str = malloc(strlen(optarg+1))))
        fterr_err(1, "malloc()");

      strcpy(str, optarg);

      /* search for mask option */
      if ((strm = index(str, '/'))) {
        ++strm;
        tos_mask = (u_char)strtol(strm, (char**)0L, 0);
        --strm;
        *strm = 0;
      }

      if (load_lookup(str, 65536, tos_tbl)) {
        free(str);
        fterr_errx(1, "load_lookup(): failed");
      }

      free(str);
      filter_tos = 1;
      break;

    case 'D': /* dst ip standard access list filter */
      acl_std_dst_name = optarg;
      break;

    case 'E': /* extended access list filter */
      acl_ext_name = optarg;
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'z': /* compress level */
      ftset.z_level = atoi(optarg);
      if ((ftset.z_level < 0) || (ftset.z_level > 9))
        fterr_errx(1, "Compression level must be between 0 and 9");
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  /* input from stdin */
  if (ftio_init(&ftio_in, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  if (ftio_check_generic(&ftio_in) < 0)
    fterr_errx(1, "flow-filter does not yet support PDU format");

  ftio_get_ver(&ftio_in, &ftv);
  ftv.s_version = FT_IO_SVERSION;

  if ((ftv.d_version == 5) || (ftv.d_version == 6) || (ftv.d_version == 7))
    as_present = 1;

  /* output to stdout */
  if (ftio_init(&ftio_out, 1, FT_IO_FLAG_WRITE |
    ((ftset.z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  ftio_set_comment(&ftio_out, ftset.comments);
  ftio_set_byte_order(&ftio_out, ftset.byte_order);
  ftio_set_z_level(&ftio_out, ftset.z_level);
  ftio_set_streaming(&ftio_out, 1);
  ftio_set_debug(&ftio_out, debug);

  if (ftio_set_ver(&ftio_out, &ftv) < 0)
    fterr_errx(1, "ftio_set_ver(): failed");

  /*
   * load acl's
   * XXX add fname check and close
   */
  if ((yyin = fopen(acl_fname ? acl_fname : "flow.acl", "r")))
    while (!feof(yyin))
      yyparse();

  /*
   * normalize masks
   */
  /* XXX TODO */

  if (debug > 5) 
    acl_dump(acl_list);

  if (acl_std_src_name) {
    if ((acl_std_src_index = acl_find(acl_list, acl_std_src_name)) == -1)
      fterr_errx(1, "Couldn't find list %s\n", acl_std_src_name);

    acl_std_src_index2 = acl_list.names[acl_std_src_index].num;

  }

  if (acl_std_dst_name) {
    if ((acl_std_dst_index = acl_find(acl_list, acl_std_dst_name)) == -1)
      fterr_errx(1, "Couldn't find list %s\n", acl_std_dst_name);

    acl_std_dst_index2 = acl_list.names[acl_std_dst_index].num;
  }

  if (acl_ext_name) {
    if ((acl_ext_index = acl_find(acl_list, acl_ext_name)) == -1)
      fterr_errx(1, "Couldn't find list %s\n", acl_ext_name);

    acl_ext_index2 = acl_list.names[acl_ext_index].num;
  }

  /* header first */
  if (ftio_write_header(&ftio_out) < 0)
    fterr_errx(1, "ftio_write_header(): failed");


  /* grab 1 flow */
  while ((rec = ftio_read(&ftio_in))) {

    rec_v5 = rec;
    rec_gen = rec;

    ++ total_flows;

    /* filter on input interface */
    if (filter_input) 
      if (!in_tbl[rec_gen->input])
        continue;

    /* filter on output interface */
    if (filter_output)
      if (!out_tbl[rec_gen->output])
        continue;

    /* filter on src port */
    if (filter_srcport)
      if (!src_tbl[rec_gen->srcport])
        continue;

    /* filter on dst port */
    if (filter_dstport)
      if (!dst_tbl[rec_gen->dstport])
        continue;

    /* filter on protocol */
    if (filter_prot)
      if (!prot_tbl[rec_gen->prot])
        continue;

    /* filter on src AS */
    if (as_present) {

      if (filter_srcas)
        if (!srcas_tbl[rec_v5->src_as])
          continue;

    /* filter on src AS */
      if (filter_dstas)
        if (!dstas_tbl[rec_v5->dst_as])
          continue;

    /* filter on ToS */
      if (filter_tos) {
        tos = rec_gen->tos & tos_mask;
        if (!tos_tbl[tos])
          continue;
      }

    }

    /* eval flow src addr and source standard acl */
    if (acl_std_src_index != -1) {
      ret = acl_eval_std(acl_list, acl_std_src_index2, rec_gen->srcaddr);
      if (ret == 1)
        continue;
    }

    /* eval flow dst addr and destination standard acl */
    if (acl_std_dst_index != -1) {
      ret = acl_eval_std(acl_list, acl_std_dst_index2, rec_gen->dstaddr);
      if (ret == 1)
        continue;
    }

    /* eval flow and extended acl */
    if (acl_ext_index != -1) {

      tmp_ext.protocol = rec_gen->prot;
      tmp_ext.tos = rec_gen->tos;

      /* XXX */
      tmp_ext.type = 0;
      tmp_ext.type_code = 0;
      tmp_ext.message = 0;

      tmp_ext.src_addr = rec_gen->srcaddr;
      tmp_ext.src_port = rec_gen->srcport;


      tmp_ext.dst_addr = rec_gen->dstaddr;
      tmp_ext.dst_port = rec_gen->dstport;

      ret = acl_eval_ext(acl_list, acl_ext_index2, tmp_ext);
      if (ret == 1)
        continue;
    }

    /*
     * made it by the filters, write it
     */

    if (ftio_write(&ftio_out, rec) < 0)
      fterr_errx(1, "ftio_write(): failed");

  } /* while more flows to read */

  if (ftio_close(&ftio_in) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (ftio_close(&ftio_out) < 0)
    fterr_errx(1, "ftio_close(): failed");

  if (debug > 0) {
    ftprof_end (&ftp, total_flows);
    ftprof_print(&ftp, argv[0], stderr);
  }   

  if (debug > 1) {
    acl_dump_std(acl_list, acl_std_src_index);
    acl_dump_std(acl_list, acl_std_dst_index);
  }

  return 0;

} /* main */

void usage() {

  fprintf(stderr, "flow-filter:\n\n");
  fprintf(stderr, " -a [!](list) src AS list filter\n");
  fprintf(stderr, " -A [!](list) dst AS list filter\n");
  fprintf(stderr, " -b s set output byte order (little or big)\n");
  fprintf(stderr, " -C s set comment field in output\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -i [!](list) input interface list filter\n");
  fprintf(stderr, " -I [!](list) output interface list filter\n");
  fprintf(stderr, " -f access-list file (defaults to flow.acl).\n");
  fprintf(stderr, " -p [!](list) src port list filter\n");
  fprintf(stderr, " -P [!](list) dst port list filter\n");
  fprintf(stderr, " -r [!](list) IP protocol list filter\n");
  fprintf(stderr, " -S acl_name  specify src address filter.\n");
  fprintf(stderr, " -t [!](list)[/mask] ToS filter\n");
  fprintf(stderr, " -D acl_name  specify dst address filter.\n");
  fprintf(stderr, " -E acl_name  specify extended acl filter.\n");
  fprintf(stderr, "  ! to invert a list\n");
  fprintf(stderr, "  , to seperate list items\n");
  fprintf(stderr, "  - to define a list ranges\n");
  fprintf(stderr, "  ie 1-1024,6000\n");
  fprintf(stderr, "\n\n");

} /* usage */

void yyerror(const char *msg)
{
  extern char *yytext;
  fterr_warnx("%s at '%s'\n", msg, yytext);
}

