/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-export.c,v 1.9 2001/07/06 14:26:08 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <time.h>
#include <fcntl.h>
#include <ftlib.h>
#include "fmt.h"
#include "pcap.h"
#include "cflowd.h"

#define PRCOMMA\
  if (comma)\
    printf(",");\

struct options {
  u_int32 cflowd_mask;
  u_int64 ft_mask;
  u_long records;
};

struct jump {
    int (*where)(struct ftio *ftio, struct options *opt);
};

int format0(struct ftio *ftio, struct options *opt);
int format1(struct ftio *ftio, struct options *opt);
int format2(struct ftio *ftio, struct options *opt);

void usage(void);

struct jump format[] = {{format0}, {format1}, {format2}};

#define NFORMATS 3 /* nformats + 1 */

int main(int argc, char **argv)
{
  int i, format_index, ret;
  extern char *optarg;
  extern struct jump format[];
  struct ftio ftio;
  struct ftprof ftp;
  struct options opt;
  int debug;

  /* init fterr */
  fterr_setid(argv[0]);

  debug = 0;
  format_index = 0;
  bzero(&opt, sizeof opt);

  opt.cflowd_mask = 0xFFFFFFFFL;
  opt.ft_mask = 0xFFFFFFFFFFFFFFFFLL;

  /* profile */
  ftprof_start (&ftp);

  while ((i = getopt(argc, argv, "h?d:f:m:")) != -1)
    switch (i) {

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* format */
      format_index = atoi(optarg);
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'm': /* cflowd mask */
      opt.cflowd_mask = strtoul(optarg, (char **)0L, 0);
      opt.ft_mask = strtouq(optarg, (char **)0L, 0);
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (format_index >= NFORMATS)
    fterr_errx(1, "No such format, %d", format_index);

  /* read from stdin */
  if (ftio_init(&ftio, 0, FT_IO_FLAG_READ) < 0)
    fterr_errx(1, "ftio_init(): failed");

  ret = format[format_index].where(&ftio, &opt);
      
  if ((!ret) && (debug > 0)) {
    ftprof_end(&ftp, ftio_get_rec_total(&ftio));
    ftprof_print(&ftp, argv[0], stderr);
  }

  fprintf(stderr, "%s: Exported %lu records\n", argv[0], opt.records);

        
  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-export:\n\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -f # format\n");
  fprintf(stderr, "    0 cflowd\n");
  fprintf(stderr, "    1 pcap (tcpdump)\n");
  fprintf(stderr, "    2 ASCII CSV\n");
  fprintf(stderr, " -m # fields\n");
  fprintf(stderr, "\n\n");

} /* usage */

/*
 * function: format0
 *
 * export flows in cflowd format
*/
int format0(struct ftio *ftio, struct options *opt)
{
  struct fts3rec_offsets fo;
  struct ftver ftv;
  struct fttime ftt;
  char *rec;
  u_int32 ui32, index, sysUpTime, unix_secs, unix_nsecs, First, Last;
  u_int16 ui16;
  u_int8 ui8;

  ftio_get_ver(ftio, &ftv);

  if (((ftv.d_version != 1) && (ftv.d_version != 5) && (ftv.d_version != 6) &&
       (ftv.d_version != 7)) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 1))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 2))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 3))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 4))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 5)))) {
    fterr_warnx("Unsupported export version");
    return -1;
  }

  fts3rec_compute_offsets(&fo, &ftv);

  switch (ftv.d_version) {

    case 1:
      opt->cflowd_mask &= CF_INDEX_V1_MASK;
      break;

    case 5:
      opt->cflowd_mask &= CF_INDEX_V5_MASK;
      break;

    case 6:
      opt->cflowd_mask &= CF_INDEX_V6_MASK;
      break;

    case 7:
      opt->cflowd_mask &= CF_INDEX_V7_MASK;
      break;

    case 8:

      switch (ftv.agg_method) {

        case 1:
          opt->cflowd_mask &= CF_INDEX_V8_1_MASK;
          break;

        case 2:
          opt->cflowd_mask &= CF_INDEX_V8_2_MASK;
          break;

        case 3:
          opt->cflowd_mask &= CF_INDEX_V8_3_MASK;
          break;

        case 4:
          opt->cflowd_mask &= CF_INDEX_V8_4_MASK;
          break;

        case 5:
          opt->cflowd_mask &= CF_INDEX_V8_5_MASK;
          break;

       } /* switch */
       break;

  } /* switch */

  /* index */
  index = opt->cflowd_mask;
  index = htonl(index);

  while ((rec = ftio_read(ftio))) {

    fwrite(&index, sizeof (index), 1, stdout);

    if (opt->cflowd_mask & CF_ROUTERMASK) {
       ui32 = *((u_int32*)(rec+fo.exaddr));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_SRCIPADDRMASK) {
       ui32 = *((u_int32*)(rec+fo.srcaddr));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_DSTIPADDRMASK) {
       ui32 = *((u_int32*)(rec+fo.dstaddr));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_INPUTIFINDEXMASK) {
       ui16 = *((u_int16*)(rec+fo.input));
       ui16 = htons(ui16);
       fwrite(&ui16, sizeof (ui16), 1, stdout);
    }

    if (opt->cflowd_mask & CF_OUTPUTIFINDEXMASK) {
       ui16 = *((u_int16*)(rec+fo.output));
       ui16 = htons(ui16);
       fwrite(&ui16, sizeof (ui16), 1, stdout);
    }

    if (opt->cflowd_mask & CF_SRCPORTMASK) {
       ui16 = *((u_int16*)(rec+fo.srcport));
       ui16 = htons(ui16);
       fwrite(&ui16, sizeof (ui16), 1, stdout);
    }

    if (opt->cflowd_mask & CF_DSTPORTMASK) {
       ui16 = *((u_int16*)(rec+fo.dstport));
       ui16 = htons(ui16);
       fwrite(&ui16, sizeof (ui16), 1, stdout);
    }

    if (opt->cflowd_mask & CF_PKTSMASK) {
       ui32 = *((u_int32*)(rec+fo.dPkts));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_BYTESMASK) {
       ui32 = *((u_int32*)(rec+fo.dOctets));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_IPNEXTHOPMASK) {
       ui32 = *((u_int32*)(rec+fo.nexthop));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_STARTTIMEMASK) {
       sysUpTime = *((u_int32*)(rec+fo.sysUpTime));
       unix_secs = *((u_int32*)(rec+fo.unix_secs));
       unix_nsecs = *((u_int32*)(rec+fo.unix_nsecs));
       First = *((u_int32*)(rec+fo.First));
       ftt = ftltime(sysUpTime, unix_secs, unix_nsecs, First);
       ui32 = htonl(ftt.secs);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_ENDTIMEMASK) {
       sysUpTime = *((u_int32*)(rec+fo.sysUpTime));
       unix_secs = *((u_int32*)(rec+fo.unix_secs));
       unix_nsecs = *((u_int32*)(rec+fo.unix_nsecs));
       Last = *((u_int32*)(rec+fo.Last));
       ftt = ftltime(sysUpTime, unix_secs, unix_nsecs, Last);
       ui32 = htonl(ftt.secs);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_PROTOCOLMASK) {
       ui8 = *((u_int8*)(rec+fo.prot));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_TOSMASK) {
       ui8 = *((u_int8*)(rec+fo.tos));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_SRCASMASK) {
       ui16 = *((u_int16*)(rec+fo.src_as));
       ui16 = htons(ui16);
       fwrite(&ui16, sizeof (ui16), 1, stdout);
    }

    if (opt->cflowd_mask & CF_DSTASMASK) {
       ui16 = *((u_int16*)(rec+fo.dst_as));
       ui16 = htons(ui16);
       fwrite(&ui16, sizeof (ui16), 1, stdout);
    }

    if (opt->cflowd_mask & CF_SRCMASKLENMASK) {
       ui8 = *((u_int8*)(rec+fo.src_mask));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_DSTMASKLENMASK) {
       ui8 = *((u_int8*)(rec+fo.dst_mask));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_TCPFLAGSMASK) {
       ui8 = *((u_int8*)(rec+fo.tcp_flags));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_INPUTENCAPMASK) {
       ui8 = *((u_int8*)(rec+fo.in_encaps));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_OUTPUTENCAPMASK) {
       ui8 = *((u_int8*)(rec+fo.out_encaps));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_PEERNEXTHOPMASK) {
       ui32 = *((u_int32*)(rec+fo.peer_nexthop));
       ui32 = htonl(ui32);
       fwrite(&ui32, sizeof (ui32), 1, stdout);
    }

    if (opt->cflowd_mask & CF_ENGINETYPEMASK) {
       ui8 = *((u_int8*)(rec+fo.engine_type));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    if (opt->cflowd_mask & CF_ENGINEIDMASK) {
       ui8 = *((u_int8*)(rec+fo.engine_id));
       fwrite(&ui8, sizeof (ui8), 1, stdout);
    }

    ++opt->records;

  } /* while */

  return 0;

} /* format 0 */

/*
 * function: format1
 *
 * export flows in pcap format.  Hack to use tcpdump's packet matcher
*/
int format1(struct ftio *ftio, struct options *opt)
{
  struct fts3rec_gen *rec;
  struct pcap_file_header pfh;
  struct pcap_packet_header pph;
  struct pcap_data1 pd1;
  struct pcap_data2 pd2;
  struct pcap_data3 pd3;
  struct pcap_data4 pd4;
  int good;
  struct timeval now;
  struct timezone tz;
  long thiszone;
  char buf[1024];
  int bsize, bsize2;

  if (ftio_check_generic(ftio) < 0)
    return -1;

  if (gettimeofday(&now, &tz) < 0) {
    fterr_warnx("gettimeofday() failed");
    return -1;
  }

  bzero(&pfh, sizeof pfh);
  bzero(&pph, sizeof pph);
  bzero(&pd1, sizeof pd1);
  bzero(&pd2, sizeof pd2);
  bzero(&pd3, sizeof pd3);
  bzero(&pd4, sizeof pd4);
  bsize = 0;

  thiszone = tz.tz_minuteswest * -60;

  if (localtime((time_t *)&now.tv_sec)->tm_isdst)
    thiszone += 3600;

  pfh.magic = TCPDUMP_MAGIC;  
  pfh.version_major = TCPDUMP_VERSION_MAJOR;
  pfh.version_minor = TCPDUMP_VERSION_MINOR;
  pfh.thiszone = thiszone;
  pfh.sigfigs = 6;
  pfh.snaplen = 38; /* XXX TODO */
  pfh.linktype = 1;

  if (fwrite(&pfh, sizeof pfh, 1, stdout) != 1) {
    fterr_warnx("pcap header write failed");
    return -1;
  }

  pph.len = 58;
  pph.caplen = 58;

  pd1.eth_prot = 0x0008;
  pd2.version = 0x45;

  bcopy(&pph, buf, sizeof pph);
  bsize += sizeof pph;

  bcopy(&pd1, buf+bsize, sizeof pd1);
  bsize += sizeof pd1;

  while ((rec = ftio_read(ftio))) {

    pd2.tos = rec->tos;
    pd2.prot = rec->prot;
    pd2.srcaddr = rec->srcaddr;
    pd2.dstaddr = rec->dstaddr;

#if BYTE_ORDER == LITTLE_ENDIAN
    SWAPINT32(pd2.srcaddr);
    SWAPINT32(pd2.dstaddr);
#endif /* LITTLE_ENDIAN */

    good = 1;

    switch (pd2.prot) {

    case 6:

      pd3.srcport = rec->srcport;
      pd3.dstport = rec->dstport;

#if BYTE_ORDER == LITTLE_ENDIAN
      SWAPINT16(pd3.srcport);
      SWAPINT16(pd3.dstport);
#endif /* LITTLE_ENDIAN */

      bcopy(&pd2, buf+bsize, sizeof pd2);
      bcopy(&pd3, buf+bsize+sizeof pd2, sizeof pd3);
      bsize2 = bsize + sizeof pd2 + sizeof pd3;

      break;

    case 17:

      pd4.srcport = rec->srcport;
      pd4.dstport = rec->dstport;

#if BYTE_ORDER == LITTLE_ENDIAN
      SWAPINT16(pd4.srcport);
      SWAPINT16(pd4.dstport);
#endif /* LITTLE_ENDIAN */

      bcopy(&pd2, buf+bsize, sizeof pd2);
      bcopy(&pd4, buf+bsize+sizeof pd2, sizeof pd4);
      bsize2 = bsize + sizeof pd2 + sizeof pd4;

      break;

    default:
      good = 0;
      break;

    } /* switch */

    if (good) {
      if (fwrite(&buf, bsize2, 1, stdout) != 1) {
        fterr_warnx("pcap pkt write failed");
        return -1;
      }
    }

    ++opt->records;

  } /* while */

  return 0;
  
} /* format1 */

/*
 * function: format2
 *
 * export flows in ASCII CSV Format
 */
int format2(struct ftio *ftio, struct options *opt)
{
  struct fts3rec_offsets fo;
  struct ftver ftv;
  char fmt_buf[256];
  char *rec;
  int comma;

  ftio_get_ver(ftio, &ftv);

  if (((ftv.d_version != 1) && (ftv.d_version != 5) && (ftv.d_version != 6) &&
       (ftv.d_version != 7)) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 1))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 2))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 3))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 4))) &&
       (!((ftv.d_version == 8) && (ftv.agg_method == 5)))) {
    fterr_warnx("Unsupported export version");
    return -1;
  }

  /* remove invalid fields */

  switch (ftv.d_version) {

    case 1:
      opt->ft_mask &= FT_XFIELD_V1_MASK;
      break;

    case 5:
      opt->ft_mask &= FT_XFIELD_V5_MASK;
      break;

    case 6:
      opt->ft_mask &= FT_XFIELD_V6_MASK;
      break;

    case 7:
      opt->ft_mask &= FT_XFIELD_V7_MASK;
      break;

    case 8:

      switch (ftv.agg_method) {

        case 1:
          opt->ft_mask &= FT_XFIELD_V8_1_MASK;
          break;

        case 2:
          opt->ft_mask &= FT_XFIELD_V8_2_MASK;
          break;

        case 3:
          opt->ft_mask &= FT_XFIELD_V8_3_MASK;
          break;

        case 4:
          opt->ft_mask &= FT_XFIELD_V8_4_MASK;
          break;

        case 5:
          opt->ft_mask &= FT_XFIELD_V8_5_MASK;
          break;

       } /* switch */
       break;

  } /* switch */

  comma = 0;

  fts3rec_compute_offsets(&fo, &ftv);

  while ((rec = ftio_read(ftio))) {

    if (opt->ft_mask & FT_XFIELD_UNIX_SECS) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.unix_secs)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;

    }

    if (opt->ft_mask & FT_XFIELD_UNIX_NSECS) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.unix_nsecs)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_SYSUPTIME) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.sysUpTime)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_EXADDR) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.exaddr)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DFLOWS) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.dFlows)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DPKTS) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.dPkts)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DOCTETS) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.dOctets)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_FIRST) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.First)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_LAST) {
      PRCOMMA;
      fmt_uint32(fmt_buf, *((u_int32*)(rec+fo.Last)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_ENGINE_TYPE) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.engine_type)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_ENGINE_ID) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.engine_id)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_SRCADDR) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.srcaddr)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DSTADDR) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.dstaddr)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_SRC_PREFIX) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.src_prefix)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DST_PREFIX) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.dst_prefix)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_NEXTHOP) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.nexthop)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_INPUT) {
      PRCOMMA;
      fmt_uint16(fmt_buf, *((u_int16*)(rec+fo.input)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_OUTPUT) {
      PRCOMMA;
      fmt_uint16(fmt_buf, *((u_int16*)(rec+fo.output)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_SRCPORT) {
      PRCOMMA;
      fmt_uint16(fmt_buf, *((u_int16*)(rec+fo.srcport)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DSTPORT) {
      PRCOMMA;
      fmt_uint16(fmt_buf, *((u_int16*)(rec+fo.dstport)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_PROT) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.prot)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_TOS) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.tos)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_TCP_FLAGS) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.tcp_flags)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_SRC_MASK) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.src_mask)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DST_MASK) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.dst_mask)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_SRC_AS) {
      PRCOMMA;
      fmt_uint16(fmt_buf, *((u_int16*)(rec+fo.src_as)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_DST_AS) {
      PRCOMMA;
      fmt_uint16(fmt_buf, *((u_int16*)(rec+fo.dst_as)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_IN_ENCAPS) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.in_encaps)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_OUT_ENCAPS) {
      PRCOMMA;
      fmt_uint8(fmt_buf, *((u_int8*)(rec+fo.out_encaps)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_PEER_NEXTHOP) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.peer_nexthop)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (opt->ft_mask & FT_XFIELD_ROUTER_SC) {
      PRCOMMA;
      fmt_ipv4(fmt_buf, *((u_int32*)(rec+fo.router_sc)), FMT_JUST_LEFT);
      printf(fmt_buf);
      comma = 1;
    }

    if (comma)
      printf("\n");

    comma = 0;

    ++opt->records;

  } /* while */

  return 0;
 
} /* format2 */ 
