/*
 * Copyright (c) 2001 Mark Fullmer and The Ohio State University
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 *
 *      $Id: flow-capture.c,v 1.47 2001/07/17 04:13:37 maf Exp $
 */

#if HAVE_CONFIG_H
 #include <config.h>
#endif

#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/errno.h>
#include <sys/socket.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#if HAVE_STRINGS_H
 #include <strings.h>
#endif
#if HAVE_STRING_H
  #include <string.h>
#endif
#include <time.h>
#include <fcntl.h>
#include <syslog.h>
#include "ftlib.h"
#include "fmt.h"
#include "support.h"
#ifdef LIBWRAP
#include <tcpd.h>
#endif /* LIBWRAP */



/* TODO
 *
 *  call external program on file rotation
 *
 *  The TCP client write() needs to have userland buffering
 *
 *  fix the client
 *
*/

void fterr_exit_handler(int code);


#define CAPTURE_PIDFILE    "/var/run/flow-capture.pid"

#define CLIENT_VALID   1   /* entry in use? */
#define CLIENT_HEADER  2   /* sent stream header yet? */

#define CLIENT_MAXCON  32  /* max # of TCP connections to server */

#define SELECT_TIMEOUT 1   /* 1 second */

struct client_rec {
  int fd;
  struct sockaddr_in addr;
  int flags;
  u_int32 con_time;
#ifdef LIBWRAP
  struct request_info tcpd;
#endif /* LIBWRAP */
};

struct client {
  struct client_rec recs[CLIENT_MAXCON+1]; /* +1 is used for error recovery */
  struct sockaddr_in addr;
  int fd;
  int len;
  int num;
};

struct rotate {
  u_int32 next;    /* time of next rotation */
  int    cur;     /* current rotation # */
  int    n;       /* number per day */
};

struct file {
  int fd;                   /* file descriptor */
  char name[MAXPATHLEN+1];  /* name */
  char nname[MAXPATHLEN+1]; /* new name */
  time_t time;
  struct ftver ftv;
  off_t nbytes;
  u_int32 hdr_nflows;
  u_int32 hdr_flows_corrupt;
  u_int32 hdr_flows_lost;
  u_int32 hdr_flows_reset;
};

int debug;
int sig_pipe_flag, sig_quit_flag, sig_hup_flag;
void sig_pipe();
void sig_quit();
void sig_hup();
char *progname = "flow-capture\0";
pid_t pid;
struct ftnet ftnet;

void usage();
int calc_rotate (int next, u_int32 *trotate, int *cur);

int main(argc, argv)   
int argc;
char **argv;
{
  extern int errno;
  extern char *optarg;
#ifdef IP_ADD_MEMBERSHIP
  struct sockaddr_in tmp_addr;
  struct ip_mreq mr;
#ifdef IP_ADD_SOURCE_MEMBERSHIP
  struct ip_mreq_source mrs;
#endif
#endif   
  fd_set rfd;
  struct timeval tv;
  struct tm *tm;
  struct ftset ftset;
  struct ftfile_entries fte;
  struct client client;
  struct ftio ftio;
  struct ftpdu ftpdu;
  struct ftpeeri ftpi;
  struct ftver ftv;
  struct ftchash *ftch;
  struct ftchash_rec_exp ftch_recexp, *ftch_recexpp;
  struct rotate rot;
  struct file cap_file;
  struct ftipmask ftipmask;
  time_t now;
  char work_dir[MAXPATHLEN+1];
  int i, n, len, invalid, enable_unlink, offset, detach, nest, one;
  unsigned int v1, v2;
  u_int32 privacy_mask, hash;
  char fmt_src_ip[32], fmt_dst_ip[32], fmt_dst_port[32];
  char out_rec[FT_IO_MAXREC];
  int stat_interval, stat_next;

  bzero (&rot, sizeof rot);
  bzero (&cap_file, sizeof cap_file);
  bzero (&ftnet, sizeof ftnet);
  bzero (&tv, sizeof tv);
  bzero (&work_dir, sizeof work_dir);
  bzero (&fte, sizeof fte);
  bzero (&client, sizeof client);
  bzero (&ftpdu, sizeof ftpdu);
  bzero (&ftv, sizeof ftv);
  stat_interval = 0;
  stat_next = -1;

  /* init fterr */
  fterr_setid(argv[0]);
  fterr_setexit(fterr_exit_handler);

  /* defaults + default compression */
  ftset_init(&ftset, Z_DEFAULT_COMPRESSION);

  /* by default do not mask any src/dst ip addr bits */
  privacy_mask = 0xFFFFFFFF;

  /* default timeout waiting for an active fd */
  tv.tv_sec = SELECT_TIMEOUT;

  /* listen for PDU's */
  ftnet.loc_addr.sin_family = AF_INET;
  ftnet.loc_addr.sin_addr.s_addr = htonl(INADDR_ANY);
  ftnet.loc_addr.sin_port = htons(FT_PORT);

  /* listen for clients */
  client.addr.sin_family = AF_INET;
  client.addr.sin_addr.s_addr = htonl(INADDR_ANY);
  client.addr.sin_port = htons(FT_PORT);

  /* default detach from parent */
  detach = 1;

  /* default 95 rotations per day, or every 15 minutes */
  rot.n = 95;

  /* no files initially open */
  cap_file.fd = -1;

  /* year/month/day nesting */
  nest = 3;

  while ((i = getopt(argc, argv, "A:b:c:C:d:De:E:hm:n:N:S:V:w:z:")) != -1)
  
    switch (i) {

    case 'A': /* AS substitution */
      ftset.as_sub = atoi(optarg);
      break;
  
    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        ftset.byte_order = FT_HEADER_LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        ftset.byte_order = FT_HEADER_BIG_ENDIAN;
      else 
        fterr_errx(1, "expecting \"big\" or \"little\" at -b");
      break;

    case 'c': /* client enable */
      client.num = atoi(optarg);
      if (client.num > CLIENT_MAXCON)
        fterr_errx(1, "CLIENT_MAXCON (%d) exceeded", CLIENT_MAXCON);
      break;

    case 'C': /* comment field */
      ftset.comments = optarg;
      break;
  
    case 'D': /* daemonize */
      detach = 0;
      break;

    case 'e': /* expire */
      fte.max_files = atoi(optarg);
      break;
  
    case 'E': /* expire bytes */
      if ((fte.max_bytes = scan_size(optarg)) == -1)
        fterr_errx(1, "scan_size(): failed");
      break;
        
    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'h': /* help */
      usage();
      exit (0);
      break;

    case 'm': /* privacy mask */
      privacy_mask = scan_ip(optarg);
      break;

    case 'n': /* # rotations / day */
      rot.n = atoi(optarg);
      /* no more than 1 rotation per 5 minutes */
      if (rot.n > (288-1))
        fterr_errx(1, "rotations limited to every 5 minutes");
      break;

    case 'N': /* nesting level */
      nest = atoi(optarg);
      if ((nest < -3) || (nest > 3))
        fterr_errx(1, "-3 <= nesting level <= 3\n");
      break;

    case 'S': /* stat interval */
      stat_interval = atoi(optarg);
      if ((stat_interval < 0) || (stat_interval > 60))
        fterr_errx(1, "Stat interval must be between 0 and 60.");
      break;
      
    case 'V': /* PDU version */
      n = sscanf(optarg, "%u.%u", &v1, &v2);
      if (n == 1) {
        ftv.s_version = FT_IO_SVERSION;
        ftv.d_version = v1;
        ftv.set = 1;
      } else if (n == 2) {
        ftv.s_version = FT_IO_SVERSION;
        ftv.d_version = v1;
        ftv.agg_method = v2;
        ftv.agg_version = 2;
        ftv.set = 1;
      } else {
        fterr_errx(1, "Version scan failed");
      }
      break;
        
    case 'w': /* working directory */
      if (strlen(optarg) > (MAXPATHLEN))
        fterr_errx(1, "Pathname too long");
      strcpy(work_dir, optarg);
      break;

    case 'z': /* compress level */
      ftset.z_level = atoi(optarg);
      if ((ftset.z_level < 0) || (ftset.z_level > 9))
        fterr_errx(1, "Compression level must be between 0 and 9");
      break;

    default:
      usage();
      exit (1);
      break;
        
    } /* switch */

  if ((argc - optind) != 1)
    fterr_errx(1, "Specify localip/remoteip/port\n");

  ftpi = scan_peeri(argv[optind]);

  ftnet.rem_ip = ftpi.rem_ip;
  ftnet.loc_ip = ftpi.loc_ip;

  if (ftpi.dst_port)
    ftnet.dst_port = ftpi.dst_port;
  else
    ftnet.dst_port = FT_PORT;

  ftnet.loc_addr.sin_addr.s_addr = htonl(ftpi.loc_ip);
  ftnet.loc_addr.sin_port = htons(ftnet.dst_port);

  /* if debugging is enabled do not unlink any files when aging */
  if (debug) 
    enable_unlink = 0;
  else
    enable_unlink = 1;

  /* setup for ftrec_mask_ip */
  if (privacy_mask != 0xFFFFFFFF)
    ftrec_compute_mask(&ftipmask, privacy_mask, privacy_mask,
      ftset.byte_order);

 /* daemonize */
  if (detach) {
    if ((pid = fork()) == -1) {
      fterr_err(1, "fork()");
    } else if (pid) {
      write_pidfile(pid, CAPTURE_PIDFILE, ftnet.dst_port);
      exit (0); /* parent */
    }

    umask(0022);
    setsid();
    for (n = 0; n < 16; ++n) /* XXX dynamically get NOFILE */
      close (n);

    /* enable syslog */
    fterr_setsyslog(1, LOG_PID|LOG_NDELAY, LOG_LOCAL6);

    /* disable stderr */
    fterr_setfile(0, (void*)0L);

  } 

  /* if client attachments are enabled, setup a listener */
  if (client.num) {
    
    /* socket for listen */
    if ((client.fd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
      fterr_err(1, "socket()");
      
    if (bind(client.fd, (struct sockaddr*)&client.addr, sizeof (client.addr))
      < 0)
      fterr_err(1, "bind()");
      
    /* listen for new TCP connections */
    if (listen(client.fd, 5) < 0)
      fterr_err(1, "listen()");

    /* non blocking */
    if (fcntl(client.fd, F_SETFL, O_NONBLOCK) < 0)
      fterr_err(1, "fcntl()");

  } /* clients enabled */

  /*
   * configure signal handlers
   */

  if (signal(SIGPIPE, sig_pipe) < 0)
    fterr_err(1, "signal()");

  if (signal(SIGHUP, sig_hup) < 0)
    fterr_err(1, "signal()");

  if (signal(SIGQUIT, sig_quit) < 0)
    fterr_err(1, "signal()");

  /* rotations needs to divide the day evenly */
  if (rot.n && (86400 % (rot.n+1)))
    fterr_errx(1, "Rotations must divide the day evenly");

  /* sandbox */
  if (chdir(work_dir) == -1)
    fterr_err(1, "chdir(%s)", work_dir);

  /*
   * load directory entries into the file ager
   */
  if (ftfile_loaddir(&fte, ".", FT_FILE_SORT|FT_FILE_INIT|FT_FILE_CHECKNAMES))
    fterr_errx(1, "ftfile_scandir(): failed");

  /* debugging gets a dump of the ager */
  if (debug)
    ftfile_dump(&fte);

  /* run the ager once now */
  if (ftfile_expire(&fte, enable_unlink, (u_int32)0))
    fterr_errx(1, "ftfile_export(): failed");

  /* get hostname */
  if (gethostname((char*)&ftset.hnbuf, FT_HOSTNAME_LEN-1) == -1)
    fterr_err(1, "gethostname()");

  /* ensure null terminated */
  ftset.hnbuf[FT_HOSTNAME_LEN-1] = 0;

  /* socket to receive flow pdu exports */
  if ((ftnet.fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0)
    fterr_err(1, "socket()");

  if (bigsockbuf(ftnet.fd, SO_RCVBUF, 224*1024) < 0)
    fterr_err(1, "bigsockbuf()");

/* multicast capable? */
#ifdef IP_ADD_MEMBERSHIP
  
  if (IN_CLASSD(ftpi.rem_ip)) {

    /* source is the first arg now */
    ftnet.rem_ip = ftpi.loc_ip;
    ftnet.loc_ip = ftpi.rem_ip;

    /* socket API usually requires INADDR_ANY
     * and s/g/port identifier does not have a source interface field
     * to use here
     */
    bzero(&tmp_addr, sizeof tmp_addr);
    tmp_addr.sin_family = AF_INET;
    tmp_addr.sin_port = htons(ftnet.dst_port);

    one = 1;

    /* Multicast streams may have multiple receivers */
    if (setsockopt(ftnet.fd, SOL_SOCKET, SO_REUSEADDR, (char *)&one,
      sizeof(one)) < 0)
      fterr_err(1, "setsockopt(SO_REUSEADDR)");

    if (bind(ftnet.fd, (struct sockaddr*)&tmp_addr,
      sizeof(struct sockaddr)) < 0)
      fterr_err(1, "bind(mcast-rcv)");

#ifdef IP_ADD_SOURCE_MEMBERSHIP

    /* ssm address? */
    if (IN_CLASSD_SSM(ftpi.rem_ip)) {

      mrs.imr_sourceaddr.s_addr = htonl(ftpi.loc_ip);
      mrs.imr_multiaddr.s_addr = htonl(ftpi.rem_ip);
      mrs.imr_interface.s_addr = INADDR_ANY;

      if (setsockopt(ftnet.fd, IPPROTO_IP, IP_ADD_SOURCE_MEMBERSHIP,
        (char*)&mrs, sizeof(mrs)) < 0)
        fterr_err(1, "setsockopt(IP_ADD_SOURCE_MEMBERSHIP)");

    }

    goto mcast_done;
#endif /* IP_ADD_SOURCE_MEMBERSHIP */

    mr.imr_multiaddr.s_addr = htonl(ftpi.rem_ip);
    mr.imr_interface.s_addr = INADDR_ANY;
    
    if (setsockopt(ftnet.fd, IPPROTO_IP, IP_ADD_MEMBERSHIP,
      (char *)&mr, sizeof(mr)) < 0)
      fterr_err(1, "setsockopt(IP_ADD_MEMBERSHIP)");


  } else { /* is a multicast group */

    /* unicast bind -- multicast support */
    if (bind(ftnet.fd, (struct sockaddr*)&ftnet.loc_addr,
      sizeof(ftnet.loc_addr)) < 0)
      fterr_err(1, "bind()");

  } /* not multicast group */

#ifdef IP_ADD_SOURCE_MEMBERSHIP
mcast_done:
#endif /* IP_ADD_SOURCE_MEMBERSHIP */

#else /* IP_ADD_MEMBERSHIP */

  /* unicast bind -- no multicast support */
  if (bind(ftnet.fd, (struct sockaddr*)&ftnet.loc_addr,
    sizeof(ftnet.loc_addr)) < 0)
    fterr_err(1, "bind()");

#endif /* IP_ADD_MEMBERSHIP */

#ifdef IP_RECVDSTADDR
  one = 1;
  /* return the destination IP address */
  if (setsockopt(ftnet.fd, IPPROTO_IP, IP_RECVDSTADDR, (char *)&one,
    sizeof(one)) < 0)
    fterr_err(1, "setsockopt(IP_RECVDSTADDR)");
#else
#ifdef IP_PKTINFO
  one = 1;
  /* return the destination IP address */
  if (setsockopt(ftnet.fd, IPPROTO_IP, IP_PKTINFO, (char *)&one,
    sizeof(one)) < 0)
    fterr_err(1, "setsockopt(IP_PKTINFO)");
#endif /* else */
#endif /* IP_RECVDSTADDR */

  /* init hash table for demuxing exporters */
  if (!(ftch = ftchash_new(256, sizeof (struct ftchash_rec_exp), 12, 1)))
    fterr_errx(1, "ftchash_new(): failed");

  /* init msg block */
  ftnet.iov[0].iov_len = sizeof ftpdu.buf;
  ftnet.iov[0].iov_base = (char*)&ftpdu.buf;
  ftnet.msg.msg_iov = (struct iovec*)&ftnet.iov;
  ftnet.msg.msg_iovlen = 1;
  ftnet.msg.msg_name = &ftnet.rem_addr;
  ftnet.msg.msg_namelen = sizeof ftnet.rem_addr;
  ftnet.msg.msg_control = &ftnet.msgip;
  ftnet.msg.msg_controllen = sizeof ftnet.msgip;

  while (1) {

    FD_ZERO (&rfd);
    FD_SET (ftnet.fd, &rfd);

    if (client.num)
      FD_SET (client.fd, &rfd);

/* XXX shouldn't this be the max of the clients and ftnet? */
/* XXX shouldn't and all the client fd's too? */

    if (select (ftnet.fd+1, &rfd, (fd_set *)0, (fd_set *)0, &tv) < 0)  {
      if (errno == EINTR) {
        FD_ZERO (&rfd);
      } else {
        fterr_err(1, "select()");
      }
    }

    bzero (&tv, sizeof tv);
    tv.tv_sec = SELECT_TIMEOUT;

    now = time((time_t)0L); 

    if (stat_interval) {

      tm = localtime (&now);

      /*
       * note there is an obscure race condition here if this
       * code is not reached at least every stat_interval*60 seconds 
       * where up to 1 hour of STAT lines would not show up.
       * This is highly unlikely and not handled.
       */

      if ((tm->tm_min == stat_next) || (stat_next == -1)) {

        ftchash_first(ftch);

        while ((ftch_recexpp = ftchash_foreach(ftch))) {

          fmt_ipv4(fmt_src_ip, ftch_recexpp->src_ip, FMT_JUST_LEFT);
          fmt_ipv4(fmt_dst_ip, ftch_recexpp->dst_ip, FMT_JUST_LEFT);

          fterr_info(
            "STAT: now=%lu src_ip=%s dst_ip=%s d_ver=%d pkts=%lu flows=%lu lost=%lu reset=%lu",
            (unsigned long)now, fmt_src_ip, fmt_dst_ip,
            ftch_recexpp->d_version, (u_long)ftch_recexpp->packets,
            (u_long)ftch_recexpp->flows, (u_long)ftch_recexpp->lost,
            (u_long)ftch_recexpp->reset);

        }

        stat_next = (tm->tm_min + (stat_interval - tm->tm_min % stat_interval))
          % 60;

      }

    } /* stat_inverval */

    /* new client connection ? */ 
    if (client.num && (FD_ISSET(client.fd, &rfd))) {

      i = -1;
      invalid = 0;

      /* find an open client slot */
      for (n = 0; n < client.num; ++n) {
        if (!(client.recs[n].flags & CLIENT_VALID)) {
          i = n;
          break;
        }
      }

      /* if > max connections, accept it and exit */
      if (i == -1) {
        fterr_warnx("Too many clients");
        i = CLIENT_MAXCON;
        invalid = 1;
      }

      /* accept the connection */
      len = sizeof(client.recs[i].addr);
      client.recs[i].fd =
        accept(client.fd, (struct sockaddr*)&client.recs[i].addr, &len);

      if (client.recs[i].fd < 0)
        fterr_warn("accept()");
      else {
        fterr_info("new client i=%d IP=%s", i,
          inet_ntoa(client.recs[i].addr.sin_addr));
        client.recs[i].flags = CLIENT_VALID;
        client.recs[i].con_time = (u_int32)now;
      }

#ifdef LIBWRAP
      request_init(&client.tcpd, RQ_DAEMON, "flow-capture", RQ_FILE,
        client.recs[i].fd, NULL);

      fromhost(&client.tcpd);

      if (!hosts_access(&client.tcpd)) {
        fterr_warnx("refused by libwrap");
        close(client.recs[i].fd);
        client.recs[i].flags = 0;
        invalid = 0;
      }
#endif /* LIBWRAP */

      if (invalid) {
        close(client.recs[i].fd);
        client.recs[i].flags = 0;
      }

    } /* new client */

    /* flag for work later on */
    ftpdu.ftd.count = 0;
   
    /* PDU ready */
    if (FD_ISSET(ftnet.fd, &rfd)) {

      if ((ftpdu.bused = recvmsg(ftnet.fd, &ftnet.msg, 0)) == -1)
        fterr_err(1, "recvmsg()");

#ifdef IP_RECVDSTADDR
      /* got destination IP back? */
      if ((ftnet.msgip.hdr.cmsg_level == IPPROTO_IP) &&
          (ftnet.msgip.hdr.cmsg_type == IP_RECVDSTADDR)) {
          ftnet.loc_addr.sin_addr.s_addr = ftnet.msgip.ip.s_addr;
      } else {
        ftnet.loc_addr.sin_addr.s_addr = 0;
      }
#else
#ifdef IP_PKTINFO
      if ((ftnet.msgip.hdr.cmsg_level == IPPROTO_IP) &&
          (ftnet.msgip.hdr.cmsg_type == IP_PKTINFO)) {
          ftnet.loc_addr.sin_addr.s_addr = ftnet.msgip.pktinfo.ipi_addr.s_addr;
      } else {
        ftnet.loc_addr.sin_addr.s_addr = 0;
      }
#else
      ftnet.loc_addr.sin_addr.s_addr = 0;
#endif
#endif /* IP_RECVDSTADDR */

      /* fill in hash key */
      ftch_recexp.src_ip = htonl(ftnet.rem_addr.sin_addr.s_addr);
      ftch_recexp.dst_ip = htonl(ftnet.loc_addr.sin_addr.s_addr);
      ftch_recexp.dst_port = ftnet.dst_port;
     
      /* verify integrity, get version */
      if (ftpdu_verify(&ftpdu) < 0) {
        fmt_ipv4(fmt_src_ip, ftch_recexp.src_ip, FMT_JUST_LEFT);
        fterr_warnx("ftpdu_verify(): src_ip=%s failed.", fmt_src_ip);
        ++cap_file.hdr_flows_corrupt;
        goto skip_pdu_decode;
      }
    
      /* rest of hash key */
      ftch_recexp.d_version = ftpdu.ftv.d_version;

      /* if exporter src IP has been configured then make sure it matches */
      if (ftnet.rem_ip && (ftnet.rem_ip != ftch_recexp.src_ip)) {
        fmt_ipv4(fmt_src_ip, ftch_recexp.src_ip, FMT_JUST_LEFT);
        fterr_warnx("Unexpected PDU: src_ip=%s not configured", fmt_src_ip);
        ++cap_file.hdr_flows_corrupt;
        goto skip_pdu_decode;
      }

      /* first flow or no configured destination version? */
      if (!ftv.set) {
        
        /* copy to compare next time */
        bcopy(&ftpdu.ftv, &ftv, sizeof ftv);
 
        /* flag struct as configured */
        ftv.set = 1;
 
      } else {

        /* translation to/from v8 not possible */
        if (((ftv.d_version == 8) && (ftpdu.ftv.d_version != 8)) ||
            ((ftv.d_version != 8) && (ftpdu.ftv.d_version == 8))) {
          fmt_ipv4(fmt_src_ip, ftch_recexp.dst_ip, FMT_JUST_LEFT);
          fterr_warnx("Unexpected PDU: src_ip=%s no v8 translation",
            fmt_src_ip);
          cap_file.hdr_flows_corrupt ++;
          goto skip_pdu_decode;
        }
 
        /* translation among v8 aggregation methods not possible */
        if ((ftv.d_version == 8) && ((ftv.agg_method != ftpdu.ftv.agg_method)
          || (ftv.agg_version != ftpdu.ftv.agg_version))) {
          fmt_ipv4(fmt_src_ip, ftch_recexp.dst_ip, FMT_JUST_LEFT);
          fterr_warnx(
            "Unexpected PDU: src_ip=%s multi v8 oagg=%d agg=%d over=%d ver=%d",
            fmt_src_ip, (int)ftv.agg_method, (int)ftpdu.ftv.agg_method,
            (int)ftv.agg_version, (int)ftpdu.ftv.agg_version);
          cap_file.hdr_flows_corrupt ++;
          goto skip_pdu_decode;
        }
      } /* version processing */

      /* compute 8 bit hash */
      hash = (ftch_recexp.src_ip & 0xFF);
      hash ^= (ftch_recexp.src_ip>>24);
      hash ^= (ftch_recexp.dst_ip & 0xFF);
      hash ^= (ftch_recexp.dst_ip>>24);
      hash ^= (ftch_recexp.d_version & 0xFF);

      /* get/create hash table entry */
      if (!(ftch_recexpp = ftchash_update(ftch, &ftch_recexp, hash)))
        fterr_errx(1, "ftch_update(): failed");

      /* if the packet count is 0, then this is a new entry */
      if (ftch_recexpp->packets == 0) {

        fmt_ipv4(fmt_src_ip, ftch_recexp.src_ip, FMT_JUST_LEFT);
        fmt_ipv4(fmt_dst_ip, ftch_recexp.dst_ip, FMT_JUST_LEFT);
        fterr_info("New exporter: time=%lu src_ip=%s dst_ip=%s d_version=%d",
          (u_long)now, fmt_src_ip, fmt_dst_ip, (int)ftpdu.ftv.d_version);

        /* set translation function */
        if (ftch_recexp.d_version != ftv.d_version)
          ftch_recexpp->xlate = ftrec_xlate_func(&ftpdu.ftv, &ftv);

      }

      /* verify sequence number */
      if (ftpdu_check_seq(&ftpdu, &(ftch_recexpp->ftseq)) < 0) {
        fmt_ipv4(fmt_src_ip, ftch_recexp.src_ip, FMT_JUST_LEFT);
        fmt_ipv4(fmt_dst_ip, ftch_recexp.dst_ip, FMT_JUST_LEFT);
        fmt_uint16(fmt_dst_port, ftch_recexp.dst_port, FMT_JUST_LEFT);
        fterr_warnx(
          "ftpdu_seq_check(): src_ip=%s dst_ip=%s d_version=%d expecting=%lu received=%lu lost=%lu",
          fmt_src_ip, fmt_dst_ip, (int)ftpdu.ftv.d_version,
          (u_long)ftch_recexpp->ftseq.seq_exp,
          (u_long)ftch_recexpp->ftseq.seq_rcv,
          (u_long)ftch_recexpp->ftseq.seq_lost);

        /* only count these lost if "lost" is a reasonable number */
        if (ftch_recexpp->ftseq.seq_lost < FT_SEQ_RESET) {
          cap_file.hdr_flows_lost += ftch_recexpp->ftseq.seq_lost;
          ftch_recexpp->lost += ftch_recexpp->ftseq.seq_lost;
        } else {
          cap_file.hdr_flows_reset ++;
          ftch_recexpp->reset ++;
        }
      }

      /* decode the pdu */
      ftpdu.ftd.byte_order = ftset.byte_order;
      ftpdu.ftd.as_sub = ftset.as_sub;
      ftpdu.ftd.exporter_ip = ftnet.rem_ip;
      n = fts3rec_pdu_decode(&ftpdu);

      /* update the exporter stats */
      ftch_recexpp->packets ++;
      ftch_recexpp->flows += n;

    } /* PDU on receive buffer */

skip_pdu_decode:

    /* no current file and pdu version has been set -> create file */
    if ((cap_file.fd == -1) && (ftv.d_version)) {

      /* calculate the current rotation and next rotate time */
      if (calc_rotate(rot.n, &rot.next, &rot.cur) == -1)
        fterr_errx(1, "calc_rotate(): failed");

      /* remember when file was created */
      cap_file.time = (u_int32)now;

      /* remember the version encoded in the filename */
      bcopy(&ftv, &cap_file.ftv, sizeof cap_file.ftv);

      /* construct the capture file name */
      ftfile_pathname(cap_file.name, MAXPATHLEN, nest, cap_file.ftv, 0,
        cap_file.time);

      /* create directory path for file */
      if (ftfile_mkpath(cap_file.time, nest) < 0)
        fterr_err(1, "ftfile_mkpath(%s)", cap_file.name);

      /* create/open the capture file */
      if ((cap_file.fd = open(cap_file.name, O_WRONLY|O_CREAT|O_TRUNC, 0644))
        == -1)
        fterr_err(1, "open(%s)", cap_file.name);

      /* initialize the IO stream */
      if (ftio_init(&ftio, cap_file.fd, FT_IO_FLAG_NO_SWAP | FT_IO_FLAG_WRITE |
        ((ftset.z_level) ? FT_IO_FLAG_ZINIT : 0) ) < 0)
        fterr_errx(1, "ftio_init(): failed");

      /* set the version information in the io stream */
      if (ftio_set_ver(&ftio, &ftv) < 0)
        fterr_errx(1, "ftio_set_ver(): failed");

      ftio_set_comment(&ftio, ftset.comments);
      ftio_set_cap_hostname(&ftio, ftset.hnbuf);
      ftio_set_byte_order(&ftio, ftset.byte_order);
      ftio_set_z_level(&ftio, ftset.z_level);
      ftio_set_cap_time(&ftio, cap_file.time, 0);
      ftio_set_debug(&ftio, debug);
      ftio_set_xip(&ftio, ftnet.rem_ip);
      ftio_set_corrupt(&ftio, cap_file.hdr_flows_corrupt);
      ftio_set_lost(&ftio, cap_file.hdr_flows_lost);
      ftio_set_reset(&ftio, cap_file.hdr_flows_reset);
      ftio_set_flows_count(&ftio, cap_file.hdr_nflows);

/*      ftio_map_load(&ftio, FT_MAP_FILE, ftnet.rem_ip); */

      /* header first */
      if ((n = ftio_write_header(&ftio)) < 0)
        fterr_errx(1, "ftio_write_header(): failed");
      else
        cap_file.nbytes = n;

    } /* create capture file and init new io stream */

    /* if the decode buffer has entries write them out */

    for (i = 0, offset = 0; i < ftpdu.ftd.count;
      ++i, offset += ftpdu.ftd.rec_size) {

      /* simple data privacy */
      if (privacy_mask != 0xFFFFFFFF)
        ftrec_mask_ip(ftpdu.ftd.buf+offset, &ftpdu.ftv, &ftipmask);

        /* translate version? */
        if (ftch_recexpp->xlate) {
 
          ftch_recexpp->xlate(ftpdu.ftd.buf+offset, &out_rec);
 
          if ((n = ftio_write(&ftio, &out_rec)) < 0)
            fterr_errx(1, "ftio_write(): failed");

        } else {

          if ((n = ftio_write(&ftio, ftpdu.ftd.buf+offset)) < 0)
            fterr_errx(1, "ftio_write(): failed");

        }

      /* update # of bytes stored in capture file */
      cap_file.nbytes += n;

    }

    /* update # of flows stored in capture file */
    cap_file.hdr_nflows += ftpdu.ftd.count;

    /*
     * time for a new file ?
     */
    if ((now > rot.next) || sig_quit_flag || sig_hup_flag) {

      if (sig_hup_flag)
        fterr_info("SIGHUP");

      sig_hup_flag = 0; /* re-arm */
            
      if (cap_file.fd != -1) {

        ftio_set_cap_time(&ftio, cap_file.time, (u_int32)now);
        ftio_set_corrupt(&ftio, cap_file.hdr_flows_corrupt);
        ftio_set_lost(&ftio, cap_file.hdr_flows_lost);
        ftio_set_reset(&ftio, cap_file.hdr_flows_reset);
        ftio_set_flows_count(&ftio, cap_file.hdr_nflows);

        /* re-write header first */
        if (ftio_write_header(&ftio) < 0)
          fterr_errx(1, "ftio_write_header(): failed");

        if ((n = ftio_close(&ftio)) < 0)
          fterr_errx(1, "ftio_close(): failed");

        cap_file.nbytes += n;

        /* construct final version of capture filename */
        ftfile_pathname(cap_file.nname, MAXPATHLEN, nest, cap_file.ftv, 1,
          cap_file.time);
        
        /* rename working to final */
        if (rename(cap_file.name, cap_file.nname) == -1)
          fterr_err(1, "rename(%s,%s)", cap_file.name, cap_file.nname);

        /* add it to the ager */
        if (ftfile_add_tail(&fte, cap_file.nname, cap_file.nbytes,
          cap_file.time))
          fterr_errx(1, "ftfile_add_tail(%s): failed", cap_file.name);

        /* debugging gets a dump of the ager */
        if (debug)
          ftfile_dump(&fte);

        /* reset */
        bzero(&cap_file, sizeof cap_file);

        /* invalidate file descriptor */
        cap_file.fd = -1;

        /* had enough */
        if (sig_quit_flag)
          goto main_exit;
      } /* file open */

    } /* time for new file */

    /* also need to check sig_quit if no file has been processed yet */
    if (sig_quit_flag)
      goto main_exit;

    if (!(cap_file.hdr_nflows % 1000)) {

      if (ftfile_expire(&fte, enable_unlink, cap_file.nbytes))
        fterr_errx(1, "ftfile_expire(): failed");

    } /* ager run? */

  } /* while 1 */

main_exit:

  unlink_pidfile(pid, CAPTURE_PIDFILE, ftnet.dst_port);

  if (sig_quit_flag)
    fterr_info("SIGQUIT");

  /* free storage allocated to file entries */
  ftfile_free(&fte);

  return 0;

} /* main */

void usage() {

  fprintf(stderr, "flow-capture: [options] locip/remip/port\n");
  fprintf(stderr, " -A #  replace exports with AS 0 with #\n");
  fprintf(stderr, " -b s  byte order (little or big)\n");
  fprintf(stderr, " -C s  comment field\n");
  fprintf(stderr, " -c #  enable n flow clients\n");
  fprintf(stderr, " -d #  debug level.\n");
  fprintf(stderr, " -D    do not daemonize.\n");
  fprintf(stderr, " -e #  files to keep before expire. (0=disable)\n");
  fprintf(stderr, " -E #  bytes to keep before expire. (0=disable)\n");
  fprintf(stderr, " -m m  privacy mask, ie 255.255.255.0\n");
  fprintf(stderr, " -n #  file rotations per day.\n");
  fprintf(stderr, " -N #  directory nesting level.\n");
  fprintf(stderr, " -s #  size of files to keep before expire. (0=disable)\n"
);
  fprintf(stderr, "       use b,K,M,G as multipliers, ie 16M for 16*1024*1024\n");
  fprintf(stderr, " -S #  receiver stat interval in minutes\n");
  fprintf(stderr, " -V #  version of export to capture.  Use #.# for V8\n");
  fprintf(stderr, "   1   - no sequence #, engine id, or AS\n");
  fprintf(stderr, "   5   - 1+ sequence #, engine id, and AS\n");
  fprintf(stderr, "   6   - 5+ encapsulation size\n");
  fprintf(stderr, "   7   - 5+ router shortcut (Catalyst switches)\n");
  fprintf(stderr, "   8.1 - AS aggregation\n");
  fprintf(stderr, "   8.2 - Protocol Port aggregation\n");
  fprintf(stderr, "   8.3 - Source Prefix aggregation\n");
  fprintf(stderr, "   8.4 - Destination Prefix aggregation\n");
  fprintf(stderr, "   8.5 - Prefix aggregation\n");
  fprintf(stderr, " -w x  working directory pathname.\n");
  fprintf(stderr, " signals\n");
  fprintf(stderr, "   SIGHUP  - close and rotate current file\n");
  fprintf(stderr, "   SIGQUIT - close current file and exit\n");
  fprintf(stderr, "\n");
    
} /* usage */ 

void sig_pipe()
{
  sig_pipe_flag = 1;
}
    
void sig_hup()
{
  sig_hup_flag = 1;
}
    
void sig_quit()
{  
  sig_quit_flag = 1;
}

int calc_rotate (int next, u_int32 *trotate, int *cur)
{

  time_t now;       /* current time */
  u_long irotate;   /* interval of next in seconds */
  time_t morning;   /* start of the day */
  struct tm *tm1;
        
  irotate = 86400 / (next+1);
       
  *cur = 0;
  
  if ((now = time((time_t)0L)) == -1)
    return -1;
    
  /* calculate start of the day */
  if (!(tm1 = localtime (&now)))
    return -1;

  tm1->tm_sec = 0;
  tm1->tm_min = 0;
  tm1->tm_hour = 0;
 
  if ((morning = mktime(tm1)) == -1)
    return -1;
  
  if (next)
    *trotate = morning + irotate;
  else  
    *trotate = morning + 86400;
  
  while (now > *trotate) {
  
    ++ *cur;
    *trotate += irotate;
  
  }

  return 0;

} /* calc_rotate */

void fterr_exit_handler(int code)
{
  if (pid)
    unlink_pidfile(pid, CAPTURE_PIDFILE, ftnet.dst_port);
  exit (code);
} /* fterr_exit_handler */


