/* generic defines */
#define MAXLINE 1024
#define MAXENTRIES 512
#define FTIO_OUT_MAX 512

/* generic includes */
#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <unistd.h>
#include <stddef.h>
#include <ctype.h>
#include <time.h>
#include <fcntl.h>

/* includes for AF_INET stuff */
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

/* includes for flow tools */
#include <config.h>
#include "ftlib.h"
#include "sym.h"
#include "fmt.h"

/* definition of prefix_t and prefix4_t type */
typedef struct _prefix4_t {
    u_short family;             /* AF_INET | AF_INET6 */
    u_short bitlen;             /* same as mask? */
    int ref_count;              /* reference count */
    struct in_addr sin;
} prefix4_t;

typedef struct _prefix_t {
    u_short family;             /* AF_INET | AF_INET6 */
    u_short bitlen;             /* same as mask? */
    int ref_count;              /* reference count */
    union {
                struct in_addr sin;
    } add;
} prefix_t;

/* structure for holding the client table */
typedef struct _client_entry {
    prefix_t* prefix;
    void* data;
} client_entry;

/* function prototypes */
void usage();
int readfile(char *);
prefix_t* New_Prefix(int, void*, int, prefix_t*);
prefix_t* ascii2prefix (int, char *);
int my_inet_pton (int, const char *, void *);

/* global variables */
struct _client_entry client_table[MAXENTRIES]; /* table for client entries */
struct ftio ftio;
int client_count;
int others_fd;
struct ftio others_file;
struct ftio ftio_out[FTIO_OUT_MAX];
struct fts3rec_v5 *rec;
int ftio_out_count, i;
struct ftver ftv;
struct ftset ftset;

int main(int argc, char *argv[])
{
  extern char *optarg;
  char *inputfile; 

  if (argc <= 1) {
	usage();
	return(0);
  }

  while ((i = getopt(argc, argv, "f:vh")) != -1)
  switch (i) {
    case 'v': 
      printf("Version 0.01 Copyright 2001 William Emmanuel S. Yu\n");
      return(0);
      break;
    case 'f':
      inputfile = optarg;
      break;
    case 'h':
    default:
      usage();
      return(0);
      break;
  }

  /* generate the others.dump file for the unaccounted packets. */
  others_fd = open("others.dump", O_WRONLY | O_TRUNC | O_CREAT, 0644);

    if (others_fd < 0) {
      fprintf (stderr, "cannot open for writing: others.dump\n");
      return 1;
    }

    ftset_init(&ftset, 0);
    memset(&ftv, 0, sizeof(ftv));
    ftv.s_version = FT_IO_SVERSION;
    ftv.d_version = 5;
    ftv.agg_method = 1;
    ftv.agg_version = 2;

    if (ftio_init(&others_file, others_fd, FT_IO_FLAG_WRITE) < 0)
    {
      fprintf (stderr, "cannot initialize for writing: others.dump\n");
      return 1;
    }

    if (ftio_set_ver(&others_file, &ftv) < 0) {
      fprintf (stderr, "cannot set version: others.dump\n");
      return 1;
    }

    ftio_set_byte_order(&others_file, ftset.byte_order);
    ftio_set_streaming(&others_file, 1);

    if (ftio_write_header(&others_file) < 0) {
      fprintf (stderr, "cannot write header: others.dump\n");
      return 1;
    }
  /* end of segment to initialize others file */
 
  if (!readfile(inputfile)) {
     fprintf(stderr,"File not parsed properly and table not generated.\n");
     exit(-1);
  }

  /* start reading the flow file from standard input */
  /* read from stdin */
  if (ftio_init(&ftio, 0, FT_IO_FLAG_READ) < 0) {
    fprintf(stderr, "ftio_init(): failed\n");
    exit (1);
  }

  while ((rec = ftio_read(&ftio))) {
    prefix_t pref_tmp, pref_out;
    int found=0;

    /* the routine that does the table lookup */
    for (i=0; i < client_count; i++) {
	u_int net_masked;

        pref_tmp.add.sin.s_addr = rec->dstaddr;	
	pref_out = *(client_table[i].prefix);

        net_masked = (0xffffffff ^ ((1 << (32-pref_out.bitlen)) - 1));
	pref_tmp.add.sin.s_addr &= net_masked;

	pref_out.add.sin.s_addr = ntohl(pref_out.add.sin.s_addr);

	if (pref_tmp.add.sin.s_addr == pref_out.add.sin.s_addr) {
	   found = 1;
#ifdef DEBUG
	   printf(".");
#endif
	   break;
	}
    }
  
    /* writing to the respective file */
    if (found==1) { 
       if(!ftio_write(client_table[i].data, rec)) {
	 fprintf(stderr,"Error writing to file.\n");
	 exit(-1);
       }
    } else {
       if(!ftio_write(&others_file, rec)) {
         fprintf(stderr,"Error writing to file.\n");
         exit(-1);
       }
    }
  }

  /* We are done, close down. */
  for (i = 0; i < ftio_out_count; i++) {
    ftio_close(ftio_out + i);
  }
  ftio_close(&others_file);

  return(0);

} /* main() */



/** 
	readfile() - parses the configuration file 
	the format of the configuration file is as follows:

	filename prefix1 prefix2 prefix3
 */
int readfile(char *infilename)
{
  char buf[161];
  FILE *infile;
  client_count = 0;

  if (!(infile = fopen(infilename, "r"))) {
    perror(infilename);
    return 0;
  }

  while (fgets (buf, 160, infile) != NULL) {
    char *s, *s2;
    int fd;
    
    s = strchr(buf, '\n');
    *s = 0;

    s = strchr(buf, ' ');
    if (s == NULL) 
      break;

    /* Null terminate the filename and move to the first prefix */
    *s++ = 0; 

#ifdef DEBUG
    printf("client: %s\n", buf);
#endif DEBUG

    /* generate the file to place the flow dumps */
    fd = open(buf, O_WRONLY | O_TRUNC | O_CREAT, 0644);

    if (fd < 0) {
      fprintf (stderr, "cannot open for writing: %s\n", buf);
      return 1;
    }

    ftset_init(&ftset, 0);
    memset(&ftv, 0, sizeof(ftv));
    ftv.s_version = FT_IO_SVERSION;
    ftv.d_version = 5;
    ftv.agg_method = 1;
    ftv.agg_version = 2;

    if (ftio_init(ftio_out + ftio_out_count, fd, FT_IO_FLAG_WRITE) < 0) {
      fprintf (stderr, "cannot initialize for writing: %s\n", buf);
      return 1;
    }

    if (ftio_set_ver(ftio_out + ftio_out_count, &ftv) < 0) {
      fprintf (stderr, "cannot set version: %s\n", buf);
      return 1;
    }

    ftio_set_byte_order(ftio_out + ftio_out_count, ftset.byte_order);
    ftio_set_streaming(ftio_out + ftio_out_count, 1);

    if (ftio_write_header(ftio_out + ftio_out_count) < 0) {
      fprintf (stderr, "cannot write header: %s\n", buf);
      return 1;
    }
    /* end segment to initialize dump files per client */

    while (s != NULL) {
      prefix_t *prefix;
#ifdef DEBUG
      prefix_t pref_tmp;
#endif

      s2 = strchr(s,' ');
      if (s2 != NULL) { 
        *s2++=0;
      }

      if ((prefix = ascii2prefix(AF_INET, s)) == NULL) {
        fprintf (stderr, "Invalid prefix: %s\n", s);
        return 0;
      } 

      client_table[client_count].prefix = prefix;
      client_table[client_count].data = ftio_out + ftio_out_count;

#ifdef DEBUG
      pref_tmp = *(client_table[client_count].prefix);
      printf("ip address: %x ",pref_tmp.add.sin.s_addr);
      printf("prefix: %x\n",pref_tmp.bitlen);
#endif DEBUG

      client_count++;		/* increment client count */
      s = s2;
    }
   
    if (++ftio_out_count == FTIO_OUT_MAX) {
      fputs ("too many files, ignoring the rest.\n", stderr);
      break;
    }
  }
  fclose(infile);

  return 1;
} /* readfile() */

/** 
	usage() - displayed usage message 
 */
void usage()
{
  fprintf(stderr, "flow-split-non:\n\n");
  fprintf(stderr, " -f filename  input filename\n");
  fprintf(stderr, " -v           show version number\n");
  fprintf(stderr, " -h 		 help\n");
  fprintf(stderr, "\n");
} /* usage() */


/**
	my_inet_pton() - allows incomplete prefix coversions
 */
int my_inet_pton (int af, const char *src, void *dst)
{
    if (af == AF_INET) {
        int i, c, val;
        u_char xp[4] = {0, 0, 0, 0};

        for (i = 0; ; i++) {
            c = *src++;
            if (!isdigit (c))
                return (-1);
            val = 0;
            do {
                val = val * 10 + c - '0';
                if (val > 255)
                    return (0);
                c = *src++;
            } while (c && isdigit (c));
            xp[i] = val;
            if (c == '\0')
                break;
            if (c != '.')
                return (0);
            if (i >= 3)
                return (0);
        }
        memcpy (dst, xp, 4);
        return (1);
    } else {
        return -1;
    }
}

/**
	New_Prefix() - helper function for ascii2prefix
  */
prefix_t* New_Prefix(int family, void *dest, int bitlen, prefix_t *prefix)
{
    int dynamic_allocated = 0;
    int default_bitlen = 32;

    if (family == AF_INET) {
       if (prefix == NULL) {
          prefix = calloc(1, sizeof (prefix4_t));
          dynamic_allocated++;
       }
       memcpy (&prefix->add.sin, dest, 4);
    }
    else {
        return (NULL);
    }
    prefix->bitlen = (bitlen >= 0)? bitlen: default_bitlen;
    prefix->family = family;
    prefix->ref_count = 0;
    if (dynamic_allocated) {
        prefix->ref_count++;
    }
    return (prefix);
}

/** 
   ascii2prefix() - converts string netmask ipaddress pairs into prefix
	format
 */
prefix_t* ascii2prefix (int family, char *string)
{
    u_long bitlen, maxbitlen = 0;
    char *cp;
    struct in_addr sin;
    int result;
    char save[MAXLINE];

    if (string == NULL) return (NULL);

    /* easy way to handle both families */
    if (family == 0) {
       family = AF_INET;
    }

    if (family == AF_INET) {
       maxbitlen = 32;
    }

    if ((cp = strchr (string, '/')) != NULL) {
       bitlen = atol (cp + 1);
                
       assert (cp - string < MAXLINE);
       memcpy (save, string, cp - string);
       save[cp - string] = '\0';
       string = save;
       if (bitlen < 0 || bitlen > maxbitlen)
          bitlen = maxbitlen;
       }
    else {
       bitlen = maxbitlen;
    }

    if (family == AF_INET) {
       if ((result = my_inet_pton (AF_INET, string, &sin)) <= 0) 
          return (NULL);
       return (New_Prefix (AF_INET, &sin, bitlen, NULL));
    } else {
       return (NULL);
    }
}
