#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "hash_ip_s.h"
#include "fmt.h"

#define OPT_PERCENT 1 /* use percent's */


/* init data structures */
int hash_ip_s_init(ht)
struct hash_ip_s_table *ht;
{

  struct hash_ip_s_rec *rec;

  ht->head.seg = 1; /* 0 is for the end of chain marker */
  ht->head.off = 0;

  /* calculate the offset that the last record can go on */
  ht->lastoff = 65536 - (65536 % sizeof(struct hash_ip_s_rec)) -
    sizeof(struct hash_ip_s_rec);

  /* allocate the hash table */
  if (!(ht->table = (struct hash_ip_s_addr*)malloc(
    sizeof(struct hash_ip_s_addr)*65536))) {
    return -1;
  }

  bzero(ht->table, (65536*sizeof(struct hash_ip_s_addr)));

  /* the 0 index is wasted here, the address seg=0, off=0 is a marker */
  if (!(ht->data = (char**)malloc(sizeof(char*)*2))) {
    free (ht->table);
    return -1;
  }

  /* allocate the first block of data records */
  if (!(ht->data[1] = (char*)malloc(65536))) {
    free (ht->table);
    free (ht->data);
    return -1;
  }

  /* only need to make sure the first block doesn't have invalid
    next pointer */
  rec = (struct hash_ip_s_rec*)(ht->data[1]);
  rec->next.seg = 0;


  return 0;

} /* hash_ip_s_table */

/* free data structures */
int hash_ip_s_free(ht)
struct hash_ip_s_table *ht;
{
  int x;

  if (ht->table)
    free (ht->table);

  if (ht->data) {
    for (x = 0; x < ht->head.seg; ++x)
      free (ht->data[x]);
    free (ht->data);
  }

  return 0;

} /* hash_ip_free */

/* update or add entry */
/* 0  =  allready existed, updated
/* 1  =  collision, added new
/* 2  =  no collision, added new
/* <0 =  error
*/
int hash_ip_s_update(ht, newrec)
struct hash_ip_s_table *ht;
struct hash_ip_s_rec *newrec;
{

  u_long hash;
  struct hash_ip_s_rec *rec;
  struct hash_ip_s_addr addr;

  /* check if the record exists in the table */
  hash = (newrec->addr>>16) ^ (newrec->addr & 0xFFFF);

  if (ht->table[hash].seg) { /* hash key exists */

    addr.seg = ht->table[hash].seg;
    addr.off = ht->table[hash].off;

    while (1) {

      rec = (struct hash_ip_s_rec*)(ht->data[addr.seg] + addr.off);

      if (rec->addr == newrec->addr) { /* found it */
        rec->index = newrec->index;
        return 0;
      }

      if (rec->next.seg) { /* not the last record in the chain? */
        addr.seg = rec->next.seg;
        addr.off = rec->next.off;
      } else
        break;
    
    }

    /* didn't find it, allocate a new one and add it to the chain */
#ifdef HASH_IP_S_STAT
  ht->table[hash].hits ++;
#endif /* HASH_IP_S_STAT */

    rec->next.seg = ht->head.seg;
    rec->next.off = ht->head.off;

    rec = (struct hash_ip_s_rec*)(ht->data[ht->head.seg] + ht->head.off);
    rec->addr = newrec->addr;
    rec->index = newrec->index;
    rec->next.seg = 0;
    rec->next.off = 0;

    if (ht->head.off != ht->lastoff)
      ht->head.off += sizeof(struct hash_ip_s_rec);
    else {
      if (!(ht->data = (char**)realloc(ht->data,
        sizeof(char*)*(ht->head.seg+2))))
        return -1;
      if (!(ht->data[ht->head.seg+1] = (char*)malloc(65536))) 
        return -1;
      ++ ht->head.seg;
      ht->head.off = 0;
    }
    return 1;
  }

  /* wasn't in the hash table, allocate a block and link it in */
#ifdef HASH_IP_S_STAT
  ht->table[hash].hits ++;
#endif /* HASH_IP_S_STAT */

  ht->table[hash].seg = ht->head.seg;
  ht->table[hash].off = ht->head.off;

  rec = (struct hash_ip_s_rec*)(ht->data[ht->head.seg] + ht->head.off);
  rec->addr = newrec->addr;
  rec->index = newrec->index;
  rec->next.seg = 0;

  if (ht->head.off != ht->lastoff)
    ht->head.off += sizeof(struct hash_ip_s_rec);
  else {
    if (!(ht->data = (char**)realloc(ht->data,
      sizeof(char*)*(ht->head.seg+2))))
      return -1;
    if (!(ht->data[ht->head.seg+1] = (char*)malloc(65536))) 
      return -1;
    ++ ht->head.seg;
    ht->head.off = 0;
  }

  return 2;

} /* hash_ip_s_update */

hash_ip_s_dump (ht)
struct hash_ip_s_table *ht;
{
  struct hash_ip_s_rec *rec;
  int x, len;
  struct hash_ip_s_addr addr;
  char fmt_buf[256];

  for (x = 0; x < 65536; ++x) {

    if (ht->table[x].seg) { /* hash key exists */

      addr.seg = ht->table[x].seg;
      addr.off = ht->table[x].off;

      while (1) {

        rec = (struct hash_ip_s_rec*)(ht->data[addr.seg] + addr.off);

        len = fmt_ipv4(fmt_buf, rec->addr, FMT_PAD_RIGHT);
        fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';

        len += fmt_uint32(fmt_buf+len, rec->index,
          FMT_PAD_RIGHT);
        fmt_buf[len++] = ' '; fmt_buf[len++] = ' ';
        fmt_buf[len++] = 0;

        puts(fmt_buf);

        if (!rec->next.seg)
          break;

        addr.seg = rec->next.seg;
        addr.off = rec->next.off;

      } /* while */
    } /* if */
  } /* for */

  return 0;

} /* hash_ip_s_dump */

/* lookup entry */
/* 0  =  found it, rec updated
/* 1  =  didn't find it
/* <0 =  error
*/
int hash_ip_s_lookup(ht, newrec)
struct hash_ip_s_table *ht;
struct hash_ip_s_rec *newrec;
{

  u_long hash;
  struct hash_ip_s_rec *rec;
  struct hash_ip_s_addr addr;

  /* check if the record exists in the table */
  hash = (newrec->addr>>16) ^ (newrec->addr & 0xFFFF);

  if (ht->table[hash].seg) { /* hash key exists */

    addr.seg = ht->table[hash].seg;
    addr.off = ht->table[hash].off;

    while (1) {

      rec = (struct hash_ip_s_rec*)(ht->data[addr.seg] + addr.off);

      if (rec->addr == newrec->addr) { /* found it */
        newrec->index = rec->index;
        return 0;
      }

      if (rec->next.seg) { /* not the last record in the chain? */
        addr.seg = rec->next.seg;
        addr.off = rec->next.off;
      } else
        break;
    
    }
  }

  return 1;

} /* hash_ip_s_lookup */

#ifdef HASH_IP_S_STAT

int hash_ip_stat_dump(ht)
struct hash_ip_s_table *ht;
{
  int x;
  int maxdepth, m;
  struct hash_ip_s_rec *rec;
  struct hash_ip_s_addr addr;

  maxdepth = 0;

  for (x = 0; x < 65536; ++x) {

    if (ht->table[x].seg) { /* hash key exists */

      addr.seg = ht->table[x].seg;
      addr.off = ht->table[x].off;

      m = 1;

      while (1) {

        rec = (struct hash_ip_s_rec*)(ht->data[addr.seg] + addr.off);

        if (!rec->next.seg)
          break;

        addr.seg = rec->next.seg;
        addr.off = rec->next.off;

        ++m;
      }
      if (m > maxdepth)
        maxdepth = m;
    }
  }

  printf("# maxdepth = %d\n", maxdepth);

  for (x = 0; x < 65536; ++x) {

    if (!ht->table[x].hits)
      continue;

    printf("%-10d    %-10ld\n", x, ht->table[x].hits);
  }

} /* hash_ip_stat_dump */

#endif /* HASH_IP_S_STAT */
