#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include "flow.h"
#include "support.h"

int debug;

void usage();

int main(argc, argv)
int argc;
char **argv;
{

  int i, ret, tx_delay;
  extern char *optarg;
  extern int optind;
  struct flow_stream fs;
  struct flow_profile fp;
  u_int64 total_flows;
  char tmp_buf[256], *c;
  u_int32 ip_addr;
  int udp_port;
  int fpdu_version;

  /* profile */
  if (profile_start (&fp) == -1) {
    fprintf(stderr, "profile_start(): failed\n");
    exit (1);
  }

  bzero (&fs, sizeof fs);
  tx_delay = 1;
  fpdu_version == 0;

  while ((i = getopt(argc, argv, "d:h?V:x:")) != -1)
    switch (i) {

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'x': /* transmitter delay */
      tx_delay = atoi(optarg);
      break;

    case 'V': /* pdu version */
      fpdu_version = atoi(optarg);
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  /* expect IP address/port option */
  if ((argc - optind) != 1) {
    usage();
    exit (1);
  }

  if (strlen(argv[optind]) > 255) {
    fprintf(stderr, "ouch!\n");
    exit (1);
  }

  strcpy(tmp_buf, argv[optind]);

  /* skip to end or port seperator */
  for (c = tmp_buf; (*c != '/') && (*c); ++c);

  /* extract the port port */
  if (*c == '/') {
    *c = 0;
    ++c;
    udp_port = atoi(c);
  } else
    udp_port = FLOWPORT;

  /* extract the ip address part */
  ip_addr = scan_ip(tmp_buf);
  
  /* read from stdin */
  fs.fd = 0;

  if (flow_read_header(&fs) == -1) {
    fprintf(stderr, "flow_read_header(): failed.\n");
    exit (1);
  }

  /* if fpdu version is not set, use what's in the input stream */
  if (!fpdu_version)
    fpdu_version = fs.fh.d_version;

  /* else, default to 1 */
  if (!fpdu_version)
    fpdu_version = 1;

  if ((fpdu_version != 1) && (fpdu_version != 5)) {
    fprintf(stderr, "only understand v1 and v5 formats!\n");
    exit (1);
  }

  ret = flow_send(fs, &total_flows, ip_addr, udp_port, tx_delay,
    fpdu_version);

  fp.nflows = total_flows;

  if ((!ret) && (debug > 0))
    if (profile_end(argv[0], &fp) == -1) {
      fprintf(stderr, "profile_end(): failed\n");
      exit (1);
    }

  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-send:\n\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -x # transmit delay in microseconds.\n");
  fprintf(stderr, " -V pdu version.\n");
  fprintf(stderr, " A.B.C.D/port\n");
  fprintf(stderr, "\n\n");

} /* usage */

int flow_send(fs, total_flows, ip_addr, udp_port, tx_delay, fpdu_version)
struct flow_stream *fs;
u_int64 *total_flows;
u_int32 ip_addr;
int udp_port;
int tx_delay;
int fpdu_version;
{

  struct flow_pdu_v1 fpdu_v1;
  struct flow_pdu_v5 fpdu_v5;
  extern int errno;
  int i, n, udp_sock;
  u_int64 nflows;
  struct flow_data *fdata;
  struct sockaddr_in addr;
  int maxflows, header_size, pdu_size;
  char *tx_buf;

  /* create a socket for transmitting the flow pdu's */
  if ((udp_sock = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    fprintf(stderr, "socket(): %s\n", strerror(errno));
    return -1;
  }

  if (fpdu_version == 1) {
    maxflows = MAXFLOWS_V1;
    header_size = 16;
    pdu_size = sizeof (struct flow_rec_v1);
    tx_buf = (char*)&fpdu_v1;
  } else if (fpdu_version == 5) {
    maxflows = MAXFLOWS_V5;
    header_size = 24;
    pdu_size = sizeof (struct flow_rec_v5);
    tx_buf = (char*)&fpdu_v5;
  }


  bzero(&addr, sizeof (struct sockaddr_in));
  nflows = 0;
  i = 0;
  addr.sin_addr.s_addr = ip_addr;
  addr.sin_family = AF_INET;
  addr.sin_port = udp_port;
  

  fpdu_v1.version = fpdu_v5.version = fpdu_version;
  fpdu_v1.sysUpTime = fpdu_v5.sysUpTime = 0; /* XXX */
  fpdu_v1.unix_secs = fpdu_v5.unix_secs = 0; /* XXX */
  fpdu_v1.unix_nsecs = fpdu_v5.unix_nsecs = 0; /* XXX */
  fpdu_v5.flow_sequence = 0;

  while ((fdata = flow_read(fs))) {

    ++nflows;

    if (fpdu_version == 1) {

      fpdu_v1.records[i].srcaddr = fdata->srcaddr;
      fpdu_v1.records[i].dstaddr = fdata->dstaddr;
      fpdu_v1.records[i].nexthop = fdata->nexthop;
      fpdu_v1.records[i].input = fdata->input;
      fpdu_v1.records[i].output = fdata->output;

      fpdu_v1.records[i].dPkts = fdata->dPkts;
      fpdu_v1.records[i].dOctets = fdata->dOctets;
      fpdu_v1.records[i].First = fdata->First;
      fpdu_v1.records[i].Last = fdata->Last;

      fpdu_v1.records[i].srcport = fdata->srcport;
      fpdu_v1.records[i].dstport = fdata->dstport;
      fpdu_v1.records[i].pad = fdata->pad;
      fpdu_v1.records[i].prot = fdata->prot;
      fpdu_v1.records[i].tos = fdata->tos;

      fpdu_v1.records[i].flags = fdata->flags;
      fpdu_v1.records[i].tcp_retx_cnt = fdata->tcp_retx_cnt;
      fpdu_v1.records[i].tcp_retx_secs = fdata->tcp_retx_secs;
      fpdu_v1.records[i].tcp_misseq_cnt = fdata->tcp_misseq_cnt;

    } else if (fpdu_version == 5) {

      fpdu_v5.records[i].srcaddr = fdata->srcaddr;
      fpdu_v5.records[i].dstaddr = fdata->dstaddr;
      fpdu_v5.records[i].nexthop = fdata->nexthop;
      fpdu_v5.records[i].input = fdata->input;
      fpdu_v5.records[i].output = fdata->output;

      fpdu_v5.records[i].dPkts = fdata->dPkts;
      fpdu_v5.records[i].dOctets = fdata->dOctets;
      fpdu_v5.records[i].First = fdata->First;
      fpdu_v5.records[i].Last = fdata->Last;

      fpdu_v5.records[i].srcport = fdata->srcport;
      fpdu_v5.records[i].dstport = fdata->dstport;
      fpdu_v5.records[i].pad = fdata->pad;
      fpdu_v5.records[i].prot = fdata->prot;
      fpdu_v5.records[i].tos = fdata->tos;

      fpdu_v5.records[i].tcp_flags = fdata->flags;
      fpdu_v5.records[i].src_as = fdata->src_as;
      fpdu_v5.records[i].dst_as = fdata->dst_as;
      fpdu_v5.records[i].src_mask = fdata->src_mask;
      fpdu_v5.records[i].dst_mask = fdata->dst_mask;
      fpdu_v5.records[i].drops = fdata->drops;
    }

    ++i;

    /* transmit when # of flows stuffed in PDU is max */

    if (i == maxflows) {

      /* # of bytes to transmit */
      n = header_size;
      n += pdu_size * i;

      fpdu_v1.count = fpdu_v5.count = i;

      if (sendto(udp_sock, tx_buf, n, 0, (struct sockaddr*)&addr,
        sizeof (struct sockaddr_in)) == -1) {
        fprintf(stderr, "sendto(): %s\n", strerror(errno));
        return -1;
      }

      if (tx_delay)
        usleep((unsigned)tx_delay);

      i = 0;
    }


  } /* while */

  /* transmit any remaining flows? */
  if (i) {

    if (tx_delay)
      usleep((unsigned)tx_delay);

    /* # of bytes to transmit */
    n = header_size;
    n += pdu_size * i;

    fpdu_v1.count = fpdu_v5.count = i;

    if (sendto(udp_sock, tx_buf, n, 0, (struct sockaddr*)&addr,
      sizeof (struct sockaddr_in)) == -1) {
      fprintf(stderr, "sendto(): %s\n", strerror(errno));
      return -1;
    }
  }

  return 0;

} /* flow_send */

