#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <netinet/tcp.h>
#include "flow.h"
#include "fmt.h"
#include "hash_ip_s.h"
#include "hash_ip_s2.h"
#include "sym.h"
#include "flow-profile.h"


/*
 TODO: just input/output port totals (all 64K UDP/TCP + ip protocols)
 TODO: input to output ip address map + bitmap for used ports
 TODO: output to input ip address map + bitmap for used ports
 TODO: output in binary format
 TODO: output the other totals in percent form
 TODO: should the ranges have an input interface associated with them
  so this is more generalized?
 TODO: can't deal with > 65536 sorting 
 TODO: hash table sorting to deal with host profiling 
*/

int debug;

int sort_64uint64();
int tbl_out1();

int flow_profile0();

#define FLOW_PROFILE_FORMATS 5

void usage();
void print_3float();
void print_3float2();

char cc; /* comment character */

int main(argc, argv)
int argc;
char **argv;
{

  extern char *optarg;
  extern struct jump flow_profile[];
  struct flow_stream fs;
  extern char cc;
  int x, y, ret, i, format, tally;
  int sort_order, print_header;
  u_int64 total_flows;
  struct flow_profile fp;
  int options;
    char *cfgfile;
  FILE *CFG;
  char buf[1024], *p, *p2, fmt_buf[64], *hp_name;
  u_int16 in_tbl[65536], out_tbl[65536];
  unsigned int u;
  u_int32 range_high, range_low, range_i, missing;
  struct hash_ip_s_rec rec;
  struct hash_ip_s_table ht;
  struct string_list string_list;
  size_t range_index, st, hp_index;
  u_int32 *name_totals;

  /* profile */
  if (profile_start (&fp) == -1) {
    fprintf(stderr, "profile_start(): failed\n");
    exit (1);
  }

  options = 0;
  format = 0;
  print_header = 0;
  bzero(&fs, sizeof fs);
  cc = '#';
  sort_order = 0;
  tally = 0;
  cfgfile = hp_name = (char*)0L;
  bzero(&in_tbl, sizeof in_tbl);
  bzero(&out_tbl, sizeof out_tbl);
  bzero(&string_list, sizeof string_list);

  if (hash_ip_s_init(&ht)) {
    fprintf(stderr, "hash_ip_s_init() failed\n");
    return -1;
  }


  while ((i = getopt(argc, argv, "c:d:f:hi:?npPs:S:t:H:")) != -1)
    switch (i) {

    case 'c': /* comment character */
      cc = optarg[0];
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* format */
      format = atoi(optarg);
      break;

    case 'h': /* help */
    case '?': /* help */
      usage();
      exit (0);
      break;

    case 'H': /* host profile name */
      hp_name = (char*)optarg;
      break;

    case 'i': /* cfg file */
      cfgfile = (char*)optarg;
      break;

    case 'n': /* use names */
      options |= OPT_NAMES;
      break;

    case 'p': /* print header */
      print_header = 1;
      break;

    case 'P': /* percent's */
      options |= OPT_PERCENT;
      break;

    case 's': /* sort low to high on field n */
      sort_order = (atoi(optarg)+1) * -1;
      break;

    case 'S': /* sort high to low on field n */
      sort_order = (atoi(optarg)+1);
      break;

    case 't': /* tally output */
      tally = atoi(optarg);
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (format >= FLOW_PROFILE_FORMATS) {
    fprintf(stderr, "no such format, %d\n", format);
    exit (1);
  }

  if (!(CFG = fopen(cfgfile, "r"))) {
    fprintf(stderr, "error opening %s\n", cfgfile);
    exit (1);
  }


  while (!feof(CFG)) {

    fgets(buf, 1024, CFG);

    /* strip off the cr or lf that fgets puts on */
    for (p = buf; *p; ++p)
      if ((*p == '\n') || (*p == '\r'))
        *p = 0;

    /* ignore commented and blank lines */
    for (p = buf; isspace(*p); ++p);
      if ((*p == '#') || (*p == 0) || (feof(CFG)) ){
      continue;
    }

    if (!strncasecmp(p, "inside ", 7)) {

      p += 7;

      while (1) {

        u = atoi(p);

        if (u > 255 ) {
          fprintf(stderr, "interface is 0..255\n");
          exit (1);
        }

        in_tbl[u] = 1;

        while (*p && (*p != ' ') && (*p != '\t')) ++p;
        while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

        if (!*p)
          break;
      } /* 1 */
      continue;
    } /* inside */

    if (!strncasecmp(p, "outside ", 8)) {

      p += 8;

      while (1) {

        u = atoi(p);

        if (u > 255 ) {
          fprintf(stderr, "interface is 0..255\n");
          exit (1);
        }

        out_tbl[u] = 1;

        while (*p && (*p != ' ') && (*p != '\t')) ++p;
        while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

        if (!*p)
          break;
      } /* 1 */
      continue;
    } /* outside */

    if (!strncasecmp(p, "range ", 5)) {

      p += 5;

      /* skip white space up to ipaddr */
      while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

      range_low = scan_ip(p);

      /* skip to next white space */
      while (*p && (*p != ' ') && (*p != '\t')) ++p;

      /* skip white space up to ipaddr */
      while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

      range_high = scan_ip(p);

      /* skip to next white space */
      while (*p && (*p != ' ') && (*p != '\t')) ++p;

      /* skip white space up to name */
      while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

      if (string_add(&string_list, p, &range_index)) {  
        fprintf(stderr, "string_add() failed\n");
        exit (1);
      }

      for (range_i = range_low; range_i <= range_high; ++range_i) {

        rec.addr = range_i;
        rec.index = range_index;

        if ((ret = hash_ip_s_update(&ht, &rec)) == -1) {
          fprintf(stderr, "hash_ip_s_update() failed\n");
          exit (1);
        }

        if ((ret == 0) && (format == 1)) {
          x = fmt_ipv4(fmt_buf, rec.addr, FMT_PAD_RIGHT);
          fmt_buf[x++] = ' '; fmt_buf[x++] = 0;
          strcat(fmt_buf, "duplicate!\n");
          fputs(fmt_buf, stdout);
        }

      } /* for */
      continue;
    } /* range */

    if (!strncasecmp(p, "bound ", 6)) {

      missing = 0;

      if (!(name_totals = (u_int32*)malloc((size_t)(
        (sizeof (u_int32))*string_list.entries)))) {
        fprintf(stderr, "malloc() failed for name_totals\n");
        exit (1);
      }
      bzero(name_totals, (size_t)
        ((sizeof (u_int32))*string_list.entries));

      p += 6;

      /* skip white space up to ipaddr */
      while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

      range_low = scan_ip(p);

      /* skip to next white space */
      while (*p && (*p != ' ') && (*p != '\t')) ++p;

      /* skip white space up to ipaddr */
      while (*p && ((*p == ' ') || (*p == '\t'))) ++p;

      range_high = scan_ip(p);

      for (range_i = range_low; range_i <= range_high; ++range_i) {

        rec.addr = range_i;

        if ((ret = hash_ip_s_lookup(&ht, &rec)) == -1) {
          fprintf(stderr, "hash_ip_s_lookup() failed\n");
          exit (1);
        }

        if (ret == 0) 
          name_totals[rec.index]++;

        if ((ret == 1) && (format == 1)) {
          x = fmt_ipv4(fmt_buf, rec.addr, FMT_PAD_RIGHT);
          fmt_buf[x++] = ' '; fmt_buf[x++] = 0;
          strcat(fmt_buf, "missing!\n");
          fputs(fmt_buf, stdout);
          ++missing;
        }

      } /* for */

      /* output namespace totals for bound */
      if (format == 1)  {
        printf("\n# namespace totals for %s\n\n", buf);
        for (st = 0; st < string_list.entries; ++st) {
          if (name_totals[st]) {
            printf("%-30.30s  %lu\n", string_list.ptr[st],
              (unsigned long)name_totals[st]);
          }
        }

        free (name_totals);

        printf("total:   %lu\n", (u_long)range_high-range_low+1);
        printf("missing: %lu\n", (u_long)missing);

      } /* format */
      continue;
    } /* range */

  } /* while CFG */

  fclose(CFG);

  /* make sure an interface is not marked as both in and out */
  for (x = 0; x < 65536; ++x) {
    if (out_tbl[x] && in_tbl[x]) {
      fprintf(stderr, "interface %d can't be both in and out.\n", x);
      exit (1);
    }
  }

  /* debug, output the namelist */
  if (debug > 5) {
    for (st = 0; st < string_list.entries; ++st) {
      printf("name: %s\n", string_list.ptr[st]);
    }
  } /* debug */

  /* debug, output the hash tables */
  if (debug > 5) 
    hash_ip_s_dump (&ht);

  if (format == 1) {
    ret = 0;
    goto done;
  }

  if ((format == 3) || (format == 4)) {

    if (!hp_name) {
      fprintf(stderr, "need to define name firist\n");
      exit (1);
    }

    for (st = 0; st < string_list.entries; ++st) {
      if (!strcmp(string_list.ptr[st], hp_name))
        hp_index = st;
    }

  } /* host_profile */

  /* read from stdin */
  fs.fd = 0;

  if (flow_read_header(&fs) == -1) {
    fprintf(stderr, "flow_read_header(): failed.\n");
    exit (1);
  }

  /* print cmdl args */
  printf("%c\n%c output from: ", cc, cc);
  for (x = 0; x < argc; ++x) {
    for (y = 0; y < strlen(argv[x]); ++y) {
      if (isprint((int)argv[x][y])) putc(argv[x][y], stdout);
    }
    putc (' ', stdout);
  }
  putc ('\n', stdout);

  if (print_header) {
    flow_print_header(&fs.fh, cc);
  }

  if ((format == 2) || (format == 4))
    options |= OPT_LONG;

  ret = flow_profile0(&in_tbl, &out_tbl, &string_list, &ht, 
    &fs, sort_order, &total_flows, options, tally, format, hp_index);

  fp.nflows = total_flows;

  if ((!ret) && (debug > 0))
    if (profile_end(argv[0], &fp) == -1) {
      fprintf(stderr, "profile_end(): failed\n");
      exit (1);
    }

done:

  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-profile:\n\n");
  fprintf(stderr, " -dn  set debug level to n.\n");
  fprintf(stderr, " -fn  format n.\n");
  fprintf(stderr, "   0 in/out bytes and packets\n");
  fprintf(stderr, "   1 name totals for bounds (missing and dups)\n");
  fprintf(stderr, "   2 long version of 0\n");
  fprintf(stderr, "   3 host profile\n");
  fprintf(stderr, "   4 long version of 3\n");
  fprintf(stderr, " -if  input config file\n");
  fprintf(stderr, " -n   use symbolic names for ports and protocols\n");
  fprintf(stderr, " -p   print flow header.\n");
  fprintf(stderr, " -P   use percents instead of counters\n");
  fprintf(stderr, " -Sn  sort descending on field n.\n");
  fprintf(stderr, " -sn  sort ascending on field n.\n");
  fprintf(stderr, " -tn  tally percent totals every n output lines.\n");
  fprintf(stderr, "\n formats:\n");
  fprintf(stderr, "    0 Summary\n");
  fprintf(stderr, "\n\n");

} /* usage */


/*
/* function: flow_profile0
/*
/*  in/out byte totals per customer
/*
/* returns 0 for okay.
*/
int flow_profile0(in_tbl, out_tbl, sl, ht, fs, sort_order, total_flows,
   options, tally, format, hp_index)
struct flow_stream *fs;
int sort_order, options, tally, format;
u_int64 *total_flows;
struct hash_ip_s_table *ht;
struct string_list *sl;
u_int16 *in_tbl, *out_tbl;
size_t hp_index;
{
  struct flow_data *fdata;
  struct hash_ip_s_rec rec;
  struct hash_ip_s2_rec rec2;
  extern char cc;
  u_int64 nflows[2], noctets[2], npackets[2];
  u_int64 nflows2, noctets2, npackets2;
  int ret, x;
  register size_t i;
  register int dir;
  char fmt_buf[100];
  struct profile *prof;
  int host_profile;
  struct hash_ip_s2_table ht2;
  
  if (hash_ip_s2_init(&ht2)) {
    fprintf(stderr, "hash_ip_s2_init() failed\n");
    return -1;
  }

  bzero(&nflows, sizeof nflows);
  bzero(&noctets, sizeof noctets);
  bzero(&npackets, sizeof npackets);
  *total_flows = 0;
  nflows2 = noctets2 = npackets2 = 0;
  host_profile = 0; /* doing host profiling? */

  if ((format == 3) || (format == 4))
    host_profile = 1;

  /* allocate profile entries for each unique id */
  if (!(prof = malloc((size_t)(sizeof (struct profile) *
    (host_profile ? 1 : sl->entries))))) {
    fprintf(stderr, "malloc() failed for prof %d\n", (int)sl->entries);
    return -1;
  }

  bzero(prof, (size_t)(sizeof(struct profile) *
    (host_profile ? 1 : sl->entries)));

  i = 0;

  while ((fdata = flow_read(fs))) {

    nflows2 ++;
    noctets2 += fdata->dOctets;
    npackets2 += fdata->dPkts;

    /* packet outgoing? */
    if ((in_tbl[fdata->input]) && (out_tbl[fdata->output])) {
      rec.addr = fdata->srcaddr;
      rec2.addr = fdata->srcaddr;
      dir = DIR_OUT;
      goto skip1;
    }

    /* packet incoming? */
    if ((out_tbl[fdata->input]) && (in_tbl[fdata->output])) {
      rec.addr = fdata->dstaddr;
      rec2.addr = fdata->dstaddr;
      dir = DIR_IN;
      goto skip1;
    }

    /* XXX should note this somewhere, packets to the router */
    if ((!fdata->input) || (!fdata->output)) 
      continue;

    if (debug > 3) {
      fprintf(stderr, "in=%d, out=%d not defined\n", (int)fdata->input,
        (int)fdata->output);
      continue;
    }

    /* don't know the direction then don't count */
    continue;

skip1:

    if ((ret = hash_ip_s_lookup(ht, &rec)) == -1) {
      fprintf(stderr, "hash_ip_s_lookup() failed\n");
      goto skip2;
    }

    /* if didn't find hash */
    /* XXX should probably note this somewhere */
    if (ret != 0)
      continue;

    /* only do per host stats for the named group */
    if (host_profile && (rec.index != hp_index))
      continue;

    if (host_profile) {

      /* update either creates a new rec, or points us to one */
      if ((ret = hash_ip_s2_update(&ht2, &rec2)) == -1) {
        fprintf(stderr, "hash_ip_s2_update() failed\n");
        goto skip2;
      }

      rec2.prof->ip_prot_octets[dir][fdata->prot] += fdata->dOctets;
      rec2.prof->ip_prot_packets[dir][fdata->prot] += fdata->dPkts;
  
      if (fdata->prot == PROTO_UDP) {
        if (fdata->dstport < 1024) {
          rec2.prof->udp_port_packets[dir][fdata->dstport] += fdata->dPkts;
          rec2.prof->udp_port_octets[dir][fdata->dstport] += fdata->dOctets;
        }
      } else if (fdata->prot == PROT_TCP) {
        if (fdata->dstport < 1024) {
          rec2.prof->tcp_port_packets[dir][fdata->dstport] += fdata->dPkts;
          rec2.prof->tcp_port_octets[dir][fdata->dstport] += fdata->dOctets;
        }
        if (fdata->flags & TH_SYN) {
          rec2.prof->tcp_flag_syn[dir] ++;
          if (fdata->flags & TH_ACK) {
            rec2.prof->tcp_flag_syn_ack[dir] ++;
          } /* TH_ACK */
        } /* TH_SYN */
      } /* PROT_TCP */
    } 

    if (!host_profile)
      i = rec.index;

    prof[i].ip_prot_octets[dir][fdata->prot] += fdata->dOctets;
    prof[i].ip_prot_packets[dir][fdata->prot] += fdata->dPkts;

    noctets[dir] += fdata->dOctets;
    npackets[dir] += fdata->dPkts;
    nflows[dir] ++;

    if (fdata->prot == PROTO_UDP) {
      if (fdata->dstport < 1024) {
        prof[i].udp_port_packets[dir][fdata->dstport] += fdata->dPkts;
        prof[i].udp_port_octets[dir][fdata->dstport] += fdata->dOctets;
      }
    } else if (fdata->prot == PROT_TCP) {
      if (fdata->dstport < 1024) {
        prof[i].tcp_port_packets[dir][fdata->dstport] += fdata->dPkts;
        prof[i].tcp_port_octets[dir][fdata->dstport] += fdata->dOctets;
      }
      if (fdata->flags & TH_SYN) {
        prof[i].tcp_flag_syn[dir] ++;
        if (fdata->flags & TH_ACK) {
          prof[i].tcp_flag_syn_ack[dir] ++;
        } /* TH_ACK */
      } /* TH_SYN */
    } /* PROT_TCP */


  } /* while */

  *total_flows = nflows[DIR_IN] + nflows[DIR_OUT];

  dump_profile_ascii(sl, sort_order, &nflows, &noctets, &npackets, prof,
    options, tally, nflows2, noctets2, npackets2, &ht2, format, hp_index);


skip2:

  return 0;

} /* flow_profile0 */

string_add(list, s, index)
struct string_list *list;
char *s;
size_t *index;
{

  int n;
  size_t i;

  /* try to find it in the list */
  for (i = 0; i < list->entries; ++i) {
    if (!strcmp(list->ptr[i], s)) {
      *index = i;
      return 0;
    }
  }

  /* allocate storage for the list of pointers to the strings */
  if (!list->entries) {
    if (!(list->ptr = (char**)malloc((size_t)sizeof (char**)))) {
      fprintf(stderr, "malloc() failed for list->ptr\n");
      return -1;
    }
  } else {
    if (!(list->ptr = (char**)realloc(list->ptr, (size_t)(sizeof
      (char**)*(list->entries+1))))) {
      fprintf(stderr, "realloc() failed\n");
      return -1;
    }
  }

  n = strlen(s);

  /* allocate space for the string itself */
  if (!(list->ptr[list->entries] = (char*)malloc((size_t)(n+1)))) {
    fprintf(stderr, "malloc() failed for list->ptr[list->entries]\n");
    return -1;
  }
    
  strcpy(list->ptr[list->entries], s);

  *index = list->entries++;

  return 0;

} /* string add */

dump_profile_ascii(sl, sort_order, nflows, noctets, npackets, prof,
  options, tally, nflows2, noctets2, npackets2, ht2, format, hp_index)
int sort_order, options, format;
u_int64 *nflows, *noctets, *npackets;
u_int64 nflows2, noctets2, npackets2;
struct string_list *sl;
struct profile *prof;
struct hash_ip_s2_table *ht2;
size_t hp_index;
{

  char fmt_buf[256];
  int32 start, end, increment, x, i, i2;
  extern char cc;
  int s;
  u_int64 t_octets[2], t_packets[2];
  int ret, len;
  u_int64 sort_object[65536];
  u_int index[65536];
  int host_profile;
  size_t sl_start, sl_end, sl_entries;
  struct hash_ip_s2_addr addr2;
  struct hash_ip_s2_rec rec2, *prec2;

  host_profile = 0; /* doing host profiling? */

  /* initialize sort index */
  for (i = 0; i < 65536; ++i)
    index[i] = i;

  bzero(&sort_object, sizeof sort_object);
  bzero(&t_octets, sizeof t_octets);
  bzero(&t_packets, sizeof t_packets);

  s = abs(sort_order);
  x = 0;

  if (sl->entries > 65535) {
    fprintf(stderr, "warning, sorting not available.\n");
    s = 0;
  }

  if ((format == 3) || (format == 4)) {
    host_profile = 1;
    /* disable sorting if host profile since there is nothing to sort */
    sort_order = 0;
    sl_start = sl_end = hp_index;
    sl_entries = 1;
  } else
    sl_start = 0, sl_end = sl->entries, sl_entries = sl->entries;

  if (s == 0) /* no sorting */
    ;
  else if (s == 1) /* name (allready sorted) */
    s = 0;
  else if (s == 2)
    for (i = 0; i < sl_entries; ++i)
      for (x = 0; x < 256; ++x)
        sort_object[i] += prof[i].ip_prot_octets[DIR_OUT][x];
  else if (s == 3)
    for (i = 0; i < sl_entries; ++i)
      for (x = 0; x < 256; ++x)
        sort_object[i] += prof[i].ip_prot_packets[DIR_OUT][x];
  else if (s == 4)
    for (i = 0; i < sl_entries; ++i)
      for (x = 0; x < 256; ++x)
        sort_object[i] += prof[i].ip_prot_octets[DIR_IN][x];
  else if (s == 5) 
    for (i = 0; i < sl_entries; ++i)
      for (x = 0; x < 256; ++x)
        sort_object[i] += prof[i].ip_prot_packets[DIR_IN][x];

  if (s)
    sort_64uint64(&sort_object, &index, (u_int)sl_entries);

  if (sort_order >= 0)
    start = sl_end - 1, end = -1, increment = -1;
  else
    start = 0, end = sl_end - 1, increment = 1;

  printf("%c\n", cc);

  if (options & OPT_LONG) {

    len = fmt_uint64(fmt_buf, nflows2, FMT_JUST_LEFT);
    fmt_buf[len++] = '/';
    fmt_uint64(fmt_buf+len, nflows[DIR_IN]+nflows[DIR_OUT],
      FMT_JUST_LEFT);
    printf("%c flows processed/counted       %s %6.3f%%\n", cc, fmt_buf,
      (double)((nflows[DIR_IN]+nflows[DIR_OUT])/(double)nflows2)*100);

    len = fmt_uint64(fmt_buf, noctets2, FMT_JUST_LEFT);
    fmt_buf[len++] = '/';
    fmt_uint64(fmt_buf+len, noctets[DIR_IN]+noctets[DIR_OUT],
      FMT_JUST_LEFT);
    printf("%c octets processed/counted      %s %6.3f%%\n", cc, fmt_buf,
      (double)((noctets[DIR_IN]+noctets[DIR_OUT])/(double)noctets2)*100);

    len = fmt_uint64(fmt_buf, npackets2, FMT_JUST_LEFT);
    fmt_buf[len++] = '/';
    fmt_uint64(fmt_buf+len, npackets[DIR_IN]+npackets[DIR_OUT],
      FMT_JUST_LEFT);
    printf("%c packets processed/counted     %s %6.3f%%\n%c\n",
      cc, fmt_buf,
      (double)((npackets[DIR_IN]+npackets[DIR_OUT])/
        (double)npackets2)*100, cc);
  }

  /* header */
  if (options & OPT_PERCENT)
    printf("%cname                           octets out   packets out    octets in    packets in\n%c\n", cc, cc);
  else
    printf("%cname                           octets out            packets out             octets in             packets in\n%c\n", cc, cc);

  if (!host_profile) {
    for (i2 = 0, i = start; i != end; i += increment, ++i2) {

      profile_dump(&prof[index[i]], options, tally, sl->ptr[index[i]],
        noctets, nflows, npackets, &t_octets, &t_packets);

      /* print tally line ? */
      if (i2 && tally && (!((i2+1) % tally))) {
        printf(
          "%c\n%cSUB%6.3f%%                     %6.3f%%      %6.3f%%        %6.3f%%      %6.3f%%\n%c\n", 
          cc, cc, ((double)i2/(double)sl_entries)*100,
          ((double)t_octets[DIR_OUT]/(double)noctets[DIR_OUT])*100,
          ((double)t_packets[DIR_OUT]/(double)npackets[DIR_OUT])*100,
          ((double)t_octets[DIR_IN]/(double)noctets[DIR_IN])*100,
          ((double)t_packets[DIR_IN]/(double)npackets[DIR_IN])*100,
          cc);
      } /* if */
    } /* for */
  } else {

    profile_dump(&prof[index[0]], options, tally, sl->ptr[hp_index],
      noctets, nflows, npackets, &t_octets, &t_packets);

    for (x = 0; x < 65536; ++x) {
      if (ht2->table[x].seg) { /* hash key exists */

        addr2.seg = ht2->table[x].seg;
        addr2.off = ht2->table[x].off;

        while (1) {
          prec2 = (struct hash_ip_s2_rec*)(ht2->data[addr2.seg] +
            addr2.off);

          fmt_buf[0] = ' ';
          fmt_ipv4(fmt_buf+1, prec2->addr, FMT_PAD_RIGHT);

          profile_dump(prec2->prof, options, tally, fmt_buf,
            &noctets, &nflows, &npackets, &t_octets, &t_packets);

          if (!prec2->next.seg)
            break;

          addr2.seg = prec2->next.seg;
          addr2.off = prec2->next.off;
        } /* while */
      } /* if */
    } /* for */
  } /* host_profile */
} /* dump_profile_ascii */

