#include <sys/time.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include "flow.h"
#include "sym.h"
#include "fmt.h"

int debug;
int ip_net_only;

int flow_print0(), flow_print1(), flow_print2(), flow_print3(),
    flow_print4(), flow_print5();

struct jump flow_print[] = {{flow_print0}, {flow_print1}, {flow_print2},
          {flow_print3}, {flow_print4}, {flow_print5},};

void usage();

#define FLOW_PRINT_FORMATS  6 /* # of types of output + 1 */

char cc; /* comment character */

int main(argc, argv)
int argc;
char **argv;
{

  int i, format, ret;
  extern char *optarg;
  extern struct jump flow_print[];
  int print_header;
  struct flow_stream fs;
  struct flow_profile fp;
  int options;
  u_int64 total_flows;

  options = 0;

  /* profile */
  if (profile_start (&fp) == -1) {
    fprintf(stderr, "profile_start(): failed\n");
    exit (1);
  }

  format = 0;
  print_header = 0;
  bzero (&fs, sizeof fs);
  cc = '#';

  while ((i = getopt(argc, argv, "ph?d:f:c:ln")) != -1)
    switch (i) {

    case 'c': /* comment character */
      cc = optarg[0];
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'f': /* format */
      format = atoi(optarg);
      break;

    case 'h': /* help */
    case '?':
      usage();
      exit (0);
      break;

    case 'l': /* turn off buffered output */
      options |= OPT_NOBUF;
      break;

    case 'n': /* symbolic names */
      options |= OPT_NAMES;
      break;

    case 'p': /* print header */
      print_header = 1;
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  if (format >= FLOW_PRINT_FORMATS) {
    fprintf(stderr, "no such format, %d\n", format);
    exit (1);
  }

  /* read from stdin */
  fs.fd = 0;

  if (flow_read_header(&fs) == -1) {
    fprintf(stderr, "flow_read_header(): failed.\n");
    exit (1);
  }

  if (print_header) {
    flow_print_header(&fs.fh, cc);
  }

  ret = flow_print[format].where(&fs, &total_flows, options);

  fp.nflows = total_flows;

  if ((!ret) && (debug > 0))
    if (profile_end(argv[0], &fp) == -1) {
      fprintf(stderr, "profile_end(): failed\n");
      exit (1);
    }

  return ret;

} /* main */

void usage() {

  fprintf(stderr, "flow-print:\n\n");
  fprintf(stderr, " -d # set debug level.\n");
  fprintf(stderr, " -f # format\n");
  fprintf(stderr, "    0 1 line format\n");
  fprintf(stderr, "    1 2 line format (includes timing and flags)\n");
  fprintf(stderr, "    2 candidate TCP syn attack flows\n");
  fprintf(stderr, "    3 another 1 line format\n");
  fprintf(stderr, " -l no buffered output\n");
  fprintf(stderr, " -n use symbolic names\n");
  fprintf(stderr, " -p print header\n");
  fprintf(stderr, "\n\n");

} /* usage */


/*
/* function: flow_print0
/*
/* 1 line summary
*/
int flow_print0(fs, total_flows, options)
struct flow_stream *fs;
u_int64 *total_flows;
int options;
{

  char fmt_buf1[64], fmt_buf2[64];
  struct flow_data *fdata;
  u_int64 nflows;

  nflows = 0;

  puts("Sif  SrcIPaddress     Dif  DstIPaddress      Pr SrcP DstP  Pkts       Octets");


  while ((fdata = flow_read(fs))) {

    ++nflows;

    fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_PAD_RIGHT);

    printf("%4.4x %-15.15s  %4.4x %-15.15s   %2.2x %-4x %-4x  %-10lu %-10lu\n",
      (int)fdata->input, fmt_buf1, (int)fdata->output, fmt_buf2,
      (int)fdata->prot, (int)fdata->srcport, (int)fdata->dstport,
      (u_long)fdata->dPkts, (u_long)fdata->dOctets);

    if (options & OPT_NOBUF)
      fflush(stdout);

  } /* while */

  *total_flows = nflows;

  return 0;

} /* flow_print0 */

/*
/* function: flow_print1
/*
/* 2 line summary
*/
int flow_print1(fs, total_flows, options)
struct flow_stream *fs;
u_int64 *total_flows;
int options;
{
  char fmt_buf1[64], fmt_buf2[64];
  struct flow_data *fdata;
  u_int64 nflows;
  struct tm *tm;
  u_long start_secs, start_msecs;
  u_long end_secs, end_msecs;
  u_long active_secs, active_msecs;
  u_long bpp;

  nflows = 0;

  puts("Sif  SrcIPaddress     DIf  DstIPaddress      Pr SrcP DstP  Pkts  Octets");

  puts(" StartTime          EndTime             Active   B/Pk Ts Fl R1 CS MC\n");
  while ((fdata = flow_read(fs))) {

    ++nflows;

    fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_PAD_RIGHT);

    printf("%4.4x %-15.15s  %4.4x %-15.15s   %2.2x %-4x %-4x  %-10lu %-10lu\n",
      (int)fdata->input, fmt_buf1, (int)fdata->output, fmt_buf2,
      (int)fdata->prot, (int)fdata->srcport, (int)fdata->dstport,
      (u_long)fdata->dPkts, (u_long)fdata->dOctets);

    start_secs = fdata->unix_secs;
    start_msecs = fdata->unix_msecs;

    start_secs += fdata->First / 1000;
    start_msecs += fdata->First % 1000;

    if (start_msecs >= 1000) { 
      start_msecs -= 1000;
      start_secs += 1;
    }

    tm = localtime((time_t*)&start_secs);

    printf(" %-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)start_msecs);

    end_secs = fdata->unix_secs;
    end_msecs = fdata->unix_msecs;

    end_secs += fdata->Last / 1000;
    end_msecs += fdata->Last % 1000;

    if (end_msecs >= 1000) { 
      end_msecs -= 1000;
      end_secs += 1;
    }

    tm = localtime((time_t*)&end_secs);

    active_secs = (fdata->Last - fdata->First) / 1000;
    active_msecs = (fdata->Last - fdata->First) % 1000;

    bpp = fdata->dOctets / fdata->dPkts;

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  %5lu.%-3lu %-3lu %2.2x %2.2x %2.2x %2.2x %2.2x\n\n",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)end_msecs,
      active_secs, active_msecs, bpp, (int)fdata->tos,
      (int)fdata->flags, (int)fdata->tcp_retx_cnt,
      (int)fdata->tcp_retx_secs, (int)fdata->tcp_misseq_cnt);

    if (options & OPT_NOBUF)
      fflush(stdout);

  } /* while */

  *total_flows = nflows;

  return 0;

} /* flow_print1 */

/*
/* function: flow_print2
/*
/* only print flows that are TCP with only a SYN bit set and
/* a single packet
*/
int flow_print2(fs, total_flows, options)
struct flow_stream *fs;
u_int64 *total_flows;
int options;
{

  char fmt_buf1[64], fmt_buf2[64];
  struct flow_data *fdata;
  u_int64 nflows;
  struct tm *tm;
  u_long start_secs, start_msecs;
  u_long end_secs, end_msecs;
  u_long active_secs, active_msecs;
  u_long bpp;


  puts("Sif SrcIPaddress     DIf DstIPaddress    Pr SrcP DstP Pkts       Octets");
  puts(" StartTime          EndTime             Active   B/Pk Ts Fl R1 CS MC\n");

  while ((fdata = flow_read(fs))) {

    ++nflows;

    /* If it's not TCP */
    if (fdata->prot != PROT_TCP)
      continue;

    /* If more than the SYN bit is set */
    if (fdata->flags != 2)
      continue;

    /* many SYN bit only packets per flow are suspect */
    if (fdata->dPkts != 2)
      continue;

    /* 40 byte datagrams are the output of the current tool */
    if (fdata->dOctets != (fdata->dPkts * 40))
      continue;


    fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_PAD_RIGHT);

    printf("%2.2x  %-15.15s  %2.2x  %-15.15s %2.2x %-4x %-4x %-10lu %-10lu\n",
      (int)fdata->input, fmt_buf1, (int)fdata->output, fmt_buf2,
      (int)fdata->prot, (int)fdata->srcport, (int)fdata->dstport,
      (u_long)fdata->dPkts, (u_long)fdata->dOctets);

    start_secs = fdata->unix_secs;
    start_msecs = fdata->unix_msecs;

    start_secs += fdata->First / 1000;
    start_msecs += fdata->First % 1000;

    if (start_msecs >= 1000) { 
      start_msecs -= 1000;
      start_secs += 1;
    }

    tm = localtime((time_t*)&start_secs);

    printf(" %-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour, (int)tm->tm_min,
      (int)tm->tm_sec, (u_long)start_msecs);

    end_secs = fdata->unix_secs;
    end_msecs = fdata->unix_msecs;

    end_secs += fdata->Last / 1000;
    end_msecs += fdata->Last % 1000;

    if (end_msecs >= 1000) { 
      end_msecs -= 1000;
      end_secs += 1;
    }

    tm = localtime((time_t*)&end_secs);

    active_secs = (fdata->Last - fdata->First) / 1000;
    active_msecs = (fdata->Last - fdata->First) % 1000;

    bpp = fdata->dOctets / fdata->dPkts;

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu  %5lu.%-3lu %-3lu %2.2x %2.2x %2.2x %2.2x %2.2x\n\n",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)end_msecs, active_secs,
      active_msecs, bpp, (int)fdata->tos, (int)fdata->flags,
      (int)fdata->tcp_retx_cnt, (int)fdata->tcp_retx_secs,
      (int)fdata->tcp_misseq_cnt);

    if (options & OPT_NOBUF)
      fflush(stdout);

  } /* while */

  *total_flows = nflows;

  return 0;

} /* flow_print2 */

/*
/* function: flow_print3
/*
/* another 1 line format
*/
int flow_print3(fs, total_flows, options)
struct flow_stream *fs;
u_int64 *total_flows;
int options;
{

  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[64], fmt_buf5[64];
  struct flow_data *fdata;
  u_int64 nflows;
  extern struct sym_table sym_tcp[], sym_ipprot[];

  nflows = 0;

  puts("srcIP             dstIP           prot  srcPort  dstPort  octets      packets");

  while ((fdata = flow_read(fs))) {

    ++nflows;

    fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_PAD_RIGHT);
    strcpy(fmt_buf3, sym_pr_uint16(sym_ipprot, (u_int16)fdata->prot,
      FMT_JUST_LEFT, options));
    strcpy(fmt_buf4, sym_pr_uint16(sym_tcp, (u_int16)fdata->srcport,
      FMT_JUST_LEFT, options));
    strcpy(fmt_buf5, sym_pr_uint16(sym_tcp, (u_int16)fdata->dstport,
      FMT_JUST_LEFT, options));


    printf("%-15.15s  %-15.15s  %-4.4s  %-7.7s  %-7.7s  %-10lu  %-10lu\n",
      fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5,
      (u_long)fdata->dOctets, (u_long)fdata->dPkts);

    if (options & OPT_NOBUF)
      fflush(stdout);

  } /* while */

  *total_flows = nflows;

  return 0;

} /* flow_print3 */

/*
/* function: flow_print4
/*
/* another 1 line format
*/
int flow_print4(fs, total_flows, options)
struct flow_stream *fs;
u_int64 *total_flows;
int options;
{

  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[64], fmt_buf5[64];
  struct flow_data *fdata;
  u_int64 nflows;
  extern struct sym_table sym_tcp[], sym_ipprot[];
  char tmp[16];

  nflows = 0;

  puts("srcIP               dstIP               prot srcAS dstAS octets      packets");

  while ((fdata = flow_read(fs))) {

    ++nflows;

    fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_JUST_LEFT);
    fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_JUST_LEFT);
    strcpy(fmt_buf3, sym_pr_uint16(sym_ipprot, (u_int16)fdata->prot,
      FMT_JUST_LEFT, options));
    fmt_uint16(fmt_buf4, (u_int16)fdata->src_as, FMT_JUST_LEFT);
    fmt_uint16(fmt_buf5, (u_int16)fdata->dst_as, FMT_JUST_LEFT);

    sprintf(tmp, "/%d", (int)fdata->src_mask);
    strcat(fmt_buf1, tmp);

    sprintf(tmp, "/%d", (int)fdata->dst_mask);
    strcat(fmt_buf2, tmp);

    printf("%-18.18s  %-18.18s  %-2.2s   %-4.4s  %-4.4s  %-10lu  %-10lu\n",
      fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4, fmt_buf5,
      (u_long)fdata->dOctets, (u_long)fdata->dPkts);

    if (options & OPT_NOBUF)
      fflush(stdout);

  } /* while */

  *total_flows = nflows;

  return 0;

} /* flow_print4 */

/*
/* function: flow_print5
/*
/* 1 line summary, steve's favorite
*/
int flow_print5(fs, total_flows, options)
struct flow_stream *fs;
u_int64 *total_flows;
int options;
{
  char fmt_buf1[64], fmt_buf2[64];
  struct flow_data *fdata;
  u_int64 nflows;
  struct tm *tm;
  u_long start_secs, start_msecs;
  u_long end_secs, end_msecs;
  u_long active_secs, active_msecs;
  u_long bpp;

  nflows = 0;

  puts("Start             End                Sif SrcIPaddress      SrcP  DIf DstIPaddress      DstP   P Fl       Pkts     Octets\n");

  while ((fdata = flow_read(fs))) {

    ++nflows;

    /* start time */
    start_secs = fdata->unix_secs;
    start_msecs = fdata->unix_msecs;

    start_secs += fdata->First / 1000;
    start_msecs += fdata->First % 1000;

    if (start_msecs >= 1000) { 
      start_msecs -= 1000;
      start_secs += 1;
    }

    tm = localtime((time_t*)&start_secs);

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)start_msecs);

    /* end time */
    end_secs = fdata->unix_secs;
    end_msecs = fdata->unix_msecs;

    end_secs += fdata->Last / 1000;
    end_msecs += fdata->Last % 1000;

    if (end_msecs >= 1000) { 
      end_msecs -= 1000;
      end_secs += 1;
    }

    tm = localtime((time_t*)&end_secs);

    printf("%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
      (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
      (int)tm->tm_min, (int)tm->tm_sec, (u_long)end_msecs);

    /* other info */
    fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_PAD_RIGHT);
    fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_PAD_RIGHT);

    printf("%4d %-15.15s %6d %4d %-15.15s %6d %3d %2d %10lu %10lu\n",
           (int)fdata->input, fmt_buf1, (int)fdata->srcport, 
           (int)fdata->output, fmt_buf2, (int)fdata->dstport,
           (int)fdata->prot, 
           (int)fdata->flags & 0x7,
           (u_long)fdata->dPkts, 
           (u_long)fdata->dOctets);

    if (options & OPT_NOBUF)
      fflush(stdout);

  } /* while */

  *total_flows = nflows;

  return 0;

} /* flow_print5 */

