#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include <signal.h>
#include <syslog.h>
#include "flow.h"
#include "fmt.h"
#include "queue.h"
#include "flow-dscan.h"
#include "bit1024.h"

/*
/* Detect network scan's
*/
/* XXX TODO
/*
/* trigger alarm on # of flows/second
/*
/* need to max out on # of records allocated.  If hit, then run the ager
/* more often/more agressive timeout.
/*
/* FUTURE:
/*  take advantage of need to only allocate strcuture of the same
/*  size by having malloc allocate chunks at a time and maintaining
/*  a freelist.
/* 
/*  most of the dst ports are 53 and 80.  Could save a bunch of memory
/*  by only allocating a bit1024 struct when it's not a list of well
/*  known ports -- tradoff for a few cpu cycles..
/*
/*  possibly trigger the ager with an alarm instead of by flow traffic
/*
/*  if the dst ip list gets too long convert to a hash?
*/

int debug;
int sig_hup, sig_usr1;
void sig_hup_handler(), sig_usr1_handler();
void usage();
char *progname = "flow-dscan\0";
int detach;

int main(argc, argv)
int argc;
char **argv;
{

  extern char *optarg;
  extern int optind, debug, errno;
  int i, j, match, k;
  char *fname;
  struct flow_stream fs_in;
  struct flow_data *fdata;
  u_int64 total_flows;
  u_int32 total_flows32, i32;
  struct flow_profile fp;
  struct stat stat_buf;
  struct dscan_state ds;
  struct dscan_rec *drp, *odrp, *drp2;
  struct dscan_dst *ddp, *oddp, *ddp2;
  struct dscan_sup *dsp_src, *dsp_dst;
  u_long hash;
  char fmt_buf1[64], fmt_buf2[64], fmt_buf3[64], fmt_buf4[54];
  int do_dump, do_load;
  int filter_www, filter_mcast, filter_input, filter_output;
  char in_tbl[65536], out_tbl[65536];
  u_int32 trigger_time, trigger_packets, trigger_octets;
  pid_t pid;
  u_long start_secs;
  u_long start_msecs;
  struct tm *tm;

  total_flows = total_flows32 = 0;
  bzero (&fs_in, sizeof fs_in);
  fs_in.fd = 0;
  bzero(&ds, sizeof ds);
  ds.ager_timeout = DSCAN_AGER_TIMEOUT;
  ds.statefile = DSCAN_STATEFILE;
  ds.supfile = DSCAN_SUP_FILE;
  ds.dscan_ip_depth = DSCAN_IP_DEPTH;
  ds.dscan_port_trigger = DSCAN_PORT_TRIGGER;
  sig_hup = 0;
  sig_usr1 = 0;
  do_dump = 0;
  do_load = 0;
  filter_www = filter_mcast = filter_input = filter_output = 0;
  trigger_time = trigger_octets = trigger_packets = 0;
  detach = 0;

  while ((i = getopt(argc, argv, "d:i:I:Dh?lmps:t:wL:O:P:S:T:bBW")) != -1)

    switch (i) {

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'i': /* input filter interface list */

      if (load_lookup(optarg, 65536, &in_tbl)) {
        fprintf(stderr, "error parsing list\n");
        exit (1);
      }

      filter_input = 1;
      break;

    case 'l': /* load state */
      do_load = 1;
      break;

    case 'm': /* multicast filter */
      filter_mcast = 1;
      break;

    case 'I': /* output filter interface list */

      if (load_lookup(optarg, 65536, &out_tbl)) {
        fprintf(stderr, "error parsing list\n");
        exit (1);
      }

      filter_output = 1;
      break;

    case 'p': /* dump state on exit */
      do_dump = 1;
      break;

    case 's': /* statefile */
      ds.statefile = optarg;
      break;

    case 't': /* ager timeout */
      sscanf(optarg, "%u", &ds.ager_timeout);
      break;

    case 'w': /* filter www inbound (replies) */
      filter_www = 1;
      break;

    case 'b': /* no detach */
      detach = 1;
      break;

    case 'B': /* no detach, use syslog */
      detach = 2;
      break;

    case 'D':
      sscanf(optarg, "%u", &ds.dscan_ip_depth);
      break;

    case 'L': /* suppress file */
      ds.supfile = optarg;
      break;

    case 'O': /* excessive octets trigger */
      sscanf(optarg, "%u", (unsigned int*)&trigger_octets);
      break;

    case 'P': /* excessive packets trigger */
      sscanf(optarg, "%u", (unsigned int*)&trigger_packets);
      break;

    case 'S': /* port scan trigger */
      sscanf(optarg, "%u", &ds.dscan_port_trigger);
      break;

    case 'T': /* excessive flow time trigger */
      sscanf(optarg, "%u", (unsigned int*)&trigger_time);
      break;

    case 'W': /* filter www outbound */
      filter_www = 2;
      break;

    case 'h': /* help */
    case '?': 
    default:
      usage();
      exit (1);
      break;
    } /* switch */

  /* daemonize */
  if (!detach) {
    if ((pid = fork()) == -1) {
      fprintf(stderr, "fork(): %s\n", strerror(errno));
    } else if (pid)
      exit (0); /* parent */

    umask(0022);
    setsid();
    for (i = 1; i < 16; ++i) /* XXX dynamically get NOFILE */
      close (i);

    report_init(0);

  } else {
    if (detach == 1)
      report_init(1);
    else if (detach == 2)
      report_init(0);
  }

  /* profile */
  if (profile_start (&fp) == -1) {
    fprintf(stderr, "profile_start(): failed\n");
    exit (1);
  }

  /* configure handlers */
  signal(SIGHUP, sig_hup_handler);
  signal(SIGUSR1, sig_usr1_handler);

  /* initialize scanner hash table */
  for (i = 0; i < DSCAN_HASHSIZE; ++i) {
    SLIST_INIT(&ds.hash_scan[i]);
  }

  /* initialize suppress hash table */
  for (i = 0; i < DSCAN_HASHSIZE; ++i) {
    SLIST_INIT(&ds.hash_sup_src[i]);
  }

  for (i = 0; i < DSCAN_HASHSIZE; ++i) {
    SLIST_INIT(&ds.hash_sup_dst[i]);
  }

  /* load state file */
  if (do_load) {

    if (load_state(&ds)) {
      report(LOG_ERR, "load_state(): failed");
      exit (1);
    }
#ifdef 0
    ds.statefile = "/var/tmp/ds.state2";
    dump_state(&ds);
    exit(0);
#endif
  }

  if (load_suppress(&ds, 0)) {
    report(LOG_ERR, "load_suppress(): failed");
    exit (1);
  }

  if (load_suppress(&ds, 1)) {
    report(LOG_ERR, "load_suppress(): failed");
    exit (1);
  }

  if (flow_read_header(&fs_in) == -1) {
    report(LOG_ERR, "flow_read_header(): failed for %s", fname);
    exit (1);
  }

  while ((fdata = flow_read(&fs_in))) {

    if (sig_usr1) {
      if (debug)
        report(LOG_INFO, "signal: USR1");
      dump_state(&ds);
      sig_usr1 = 0;
    } /* sig_usr1 */

    if (sig_hup) {
      if (debug)
        report(LOG_INFO, "signal: HUP");
      clear_suppress(&ds, 0);
      clear_suppress(&ds, 1);
      load_suppress(&ds, 0);
      load_suppress(&ds, 1);
      sig_hup = 0;
    } /* sig_hup */

    ++total_flows;
    ++total_flows32;

    /* filter on input interface */
    if (filter_input) 
      if (!in_tbl[fdata->input])
        goto skip2;

    /* filter on output interface */
    if (filter_output)
      if (!out_tbl[fdata->output])
        goto skip2;

    /* filter multicast? */
    if (filter_mcast) {

      if ((fdata->srcaddr >= 0xe0000000) && (fdata->srcaddr <= 0xefffffff))
        goto skip2;

      if ((fdata->dstaddr >= 0xe0000000) && (fdata->dstaddr <= 0xefffffff))
        goto skip2;

    } /* filter_mcast */

    /* calculate hash based on src ip */
    hash = DSCAN_HASHFUNC(fdata->srcaddr);

    if ((filter_www == 1) && (fdata->srcport == 80) && (fdata->prot == 6) &&
      (fdata->dstport > 1023))
      goto skip2;

    if ((filter_www == 2) && (fdata->dstport == 80) && (fdata->prot == 6) &&
      (fdata->srcport > 1023))
      goto skip2;

    /* evaluate this src ip and the suppress list */
    match = 0;
    SLIST_FOREACH(dsp_src, &ds.hash_sup_src[hash], chain) {
      if (dsp_src->ip == fdata->srcaddr) {
        if (!(dsp_src->flags & DSCAN_SUP_SRCPORT))
          goto skipsrc;
        if (dsp_src->srcport != fdata->srcport)
          goto sup1;
skipsrc:
        if (!(dsp_src->flags & DSCAN_SUP_DSTPORT))
          goto skipdst;
        if (dsp_src->dstport != fdata->dstport)
          goto sup1;
skipdst:
        if (!(dsp_src->flags & DSCAN_SUP_PROTOCOL))
          goto skip2;
        if (dsp_src->protocol != fdata->prot)
          goto sup1;
        goto skip2;

      }
    }

sup1:

    /* evaluate this dst ip and the suppress list */
    match = 0;
    SLIST_FOREACH(dsp_dst, &ds.hash_sup_dst[hash], chain) {
      if (dsp_dst->ip == fdata->dstaddr) {
        if (!(dsp_dst->flags & DSCAN_SUP_SRCPORT))
          goto skipsrc1;
        if (dsp_dst->srcport != fdata->srcport)
          goto sup;
skipsrc1:
        if (!(dsp_dst->flags & DSCAN_SUP_DSTPORT))
          goto skipdst1;
        if (dsp_dst->dstport != fdata->dstport)
          goto sup;
skipdst1:
        if (!(dsp_dst->flags & DSCAN_SUP_PROTOCOL))
          goto skip2;
        if (dsp_dst->protocol != fdata->prot)
          goto sup;
        goto skip2;

      }
    }

sup:

    /* evaluate the triggers */
    if (trigger_octets && (fdata->dOctets > trigger_octets)) {
      flow_dump(fdata);
    }

    if (trigger_packets && (fdata->dPkts > trigger_packets)) {
      flow_dump(fdata);
    }

    if (trigger_time && ((fdata->Last - fdata->First) > trigger_time)) {
      flow_dump(fdata);
    }

    match = 0;
    SLIST_FOREACH(drp, &ds.hash_scan[hash], chain) {
      if (drp->ip_src == fdata->srcaddr) {
        match = 1;
        break;
      }
    }

    if (match) { /* src ip match */


      /* previous host or scan report then skip */
      if (drp->flags)
        goto skip2;

      /* if rec exists with this dest ip */
      STAILQ_FOREACH(ddp, &drp->dlhead, chain) {
        if (ddp->ip_dst == fdata->dstaddr) {
          ddp->ip_time = total_flows32;
          if (fdata->dstport < 1024) {
            bit1024_store((int)fdata->dstport, &ddp->portmap);
            if ((k = bit1024_count(&ddp->portmap)) >= ds.dscan_port_trigger) {
              fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_JUST_LEFT);
              fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_JUST_LEFT);
              fmt_uint32(fmt_buf3, start_secs, FMT_JUST_LEFT);

          start_secs = fdata->unix_secs;
          start_msecs = fdata->unix_msecs;

          start_secs += fdata->First / 1000;
          start_msecs += fdata->First % 1000;

          if (start_msecs >= 1000) { 
            start_msecs -= 1000;
            start_secs += 1;
          }

          tm = localtime((time_t*)&start_secs);

          snprintf(fmt_buf4, 64, "%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
               (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
               (int)tm->tm_min, (int)tm->tm_sec, (u_long)start_msecs);

              report(LOG_INFO, "port scan: src=%s dst=%s ts=%s start=%s",
                fmt_buf1, fmt_buf2, fmt_buf3, fmt_buf4);
              bzero(&ddp->portmap, sizeof ddp->portmap);
              drp->flags |= DSCAN_FLAGS_PORTSCAN;
            }
          }
          goto skip2;
        }
      }

      /* no rec exists with this dst ip */

      if (drp->depth >= ds.dscan_ip_depth) {
        fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_JUST_LEFT);
        fmt_uint32(fmt_buf3, start_secs, FMT_JUST_LEFT);

        start_secs = fdata->unix_secs;
        start_msecs = fdata->unix_msecs;

        start_secs += fdata->First / 1000;
        start_msecs += fdata->First % 1000;

        if (start_msecs >= 1000) { 
          start_msecs -= 1000;
          start_secs += 1;
        }

        tm = localtime((time_t*)&start_secs);

        snprintf(fmt_buf4, 64, "%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
         (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
         (int)tm->tm_min, (int)tm->tm_sec, (u_long)start_msecs);

        report(LOG_INFO, "host scan: ip=%s ts=%s start=%s", 
           fmt_buf1, fmt_buf3, fmt_buf4);
        drp->depth = 0;
        drp->flags |= DSCAN_FLAGS_HOSTSCAN;

        ddp = drp->dlhead.stqh_first;
        while (ddp != NULL) {
          ddp2 = ddp->chain.stqe_next;
          free(ddp);
          ++ds.stat_free;
          ++ds.stat_free_dst;
          ddp = ddp2;
        }
        STAILQ_INIT(&drp->dlhead);
      } else {
        if (!(ddp = malloc(sizeof (struct dscan_dst)))) {
          report(LOG_ERR, "malloc(): failed for dscan_dst");
          exit (1);
        }
        bzero(ddp, sizeof (struct dscan_dst));
        ds.stat_malloc++;
        ds.stat_malloc_dst++;
        drp->depth ++;

        STAILQ_INSERT_TAIL(&drp->dlhead, ddp, chain);

        ddp->ip_dst = fdata->dstaddr;
        ddp->ip_time = total_flows32;
        if (fdata->dstport < 1024) {
          bit1024_store((int)fdata->dstport, &ddp->portmap);
        }
      }

    } else { /* no src ip match */

      if (!(drp = malloc(sizeof (struct dscan_rec)))) {
        report(LOG_ERR, "malloc(): failed for dscan_rec");
        exit (1);
      }
      ds.stat_malloc++;
      ds.stat_malloc_rec++;
      bzero(drp, sizeof (struct dscan_rec));
      drp->ip_src = fdata->srcaddr;
      drp->depth = 1;
      SLIST_INSERT_HEAD(&ds.hash_scan[hash], drp, chain);

      STAILQ_INIT(&drp->dlhead);

      if (!(ddp = malloc(sizeof (struct dscan_dst)))) {
        report(LOG_ERR, "malloc(): failed for dscan_dst");
        exit (1);
      }
      bzero(ddp, sizeof (struct dscan_dst));
      ds.stat_malloc++;
      ds.stat_malloc_dst++;

      STAILQ_INSERT_TAIL(&drp->dlhead, ddp, chain);

      ddp->ip_dst = fdata->dstaddr;
      ddp->ip_time = total_flows32;
      if (fdata->dstport < 1024) {
        bit1024_store((int)fdata->dstport, &ddp->portmap);
      }

    }

    /*
    /* ager
    */

    if (ds.ager_timeout && (!(total_flows % 1000)))
      ager(&ds, total_flows32);

skip2:


  } /* while fdata */

  flow_read_end(&fs_in);

  fp.nflows = total_flows;

  if (debug > 0)
    if (profile_end(argv[0], &fp) == -1) {
      report(LOG_ERR, "profile_end(): failed");
      exit (1);
    }

  if (do_dump)
    dump_state(&ds);

  return 0;
    

} /* main */

int dump_state(ds)
  struct dscan_state *ds;
{

  FILE *FP;
  int i, j;
  char fmt_buf1[64];
  struct dscan_rec *drp;
  struct dscan_dst *ddp;

  if (!(FP = fopen(ds->statefile, "w"))) {
    report(LOG_ERR, "fopen(%s) failed", ds->statefile);
    return -1;
  }

  /* dump data structures */
  for (i = 0; i < DSCAN_HASHSIZE; ++i) {
    SLIST_FOREACH(drp, &ds->hash_scan[i], chain) {
      fmt_ipv4(fmt_buf1, drp->ip_src, FMT_JUST_LEFT);
      fprintf(FP, "H %d %d %d %s\n", i, (int)drp->depth, (int)drp->flags,
        fmt_buf1);
      STAILQ_FOREACH(ddp, &drp->dlhead, chain) {
        fmt_ipv4(fmt_buf1, ddp->ip_dst, FMT_JUST_LEFT);
        fprintf(FP, "I  %u %s\n", ddp->ip_time, fmt_buf1);
        bit1024_print(FP, &ddp->portmap); 
      }
    }
  }
  fprintf(FP, ".\n");

  if (fclose(FP)) {
    report(LOG_ERR, "fclose(%s) failed", ds->statefile);
    return -1;
  }

  return 0;

} /* dump_state */

int load_state(ds)
  struct dscan_state *ds;
{
  FILE *FP;
  int done;
  struct dscan_rec *drp;
  struct dscan_dst *ddp;
  char type;
  char buf[1024];
  u_int32 ip, hash, depth, flags, h, time, nports, i, j;

  if (!(FP = fopen(ds->statefile, "r"))) {
    report(LOG_ERR, "fopen(%s) failed", ds->statefile);
    exit (1);
  }

  done = 0;

  while (!done) {

    if (fscanf(FP, "%c", &type) == EOF) {
      report(LOG_WARNING, "fscanf(): EOF");
      break;
    }

    switch (type) {

      case '.':
        done = 1;
        break;

      case 'H': /* host - hash depth flags srcIP */
        fscanf(FP, "%u %u %u %64s", &hash, &depth, &flags, &buf);
        ip = scan_ip(buf);
        h = DSCAN_HASHFUNC(ip);
        /* create the record */

        if (!(drp = malloc(sizeof (struct dscan_rec)))) {
          report(LOG_ERR, "malloc(): failed for dscan_rec");
          return -1;
        }

        ds->stat_malloc++;
        ds->stat_malloc_rec++;
        bzero(drp, sizeof (struct dscan_rec));
        drp->ip_src = ip;
        drp->depth = depth;
        SLIST_INSERT_HEAD(&ds->hash_scan[hash], drp, chain);

        /* init destination list */
        STAILQ_INIT(&drp->dlhead);
        break;

      case 'I': /* include - time dstIP */
        fscanf(FP, "%u %15s", &time, &buf);
        ip = scan_ip(buf);
        if (!(ddp = malloc(sizeof (struct dscan_dst)))) {
          report(LOG_ERR, "malloc(): failed for dscan_dst");
          exit (1);
        }

        bzero(ddp, sizeof (struct dscan_dst));
        ds->stat_malloc++;
        ds->stat_malloc_dst++;

        STAILQ_INSERT_TAIL(&drp->dlhead, ddp, chain);

        ddp->ip_dst = ip;
        ddp->ip_time = time;
        break;

      case 'P': /* portmap - nports portlist */
        fscanf(FP, "%u", &nports);
        for (i = 0; i < nports; ++i) {
          fscanf(FP, "%u", &j);
          if (j < 1024) {
            bit1024_store((int)j, &ddp->portmap);
          }
        }
        break;

      case '\n': /* ignore */
        break;

      default:
        report(LOG_ERR, "unknown record type: %c", type);
        return -1;
        break;

    } /* switch */
  } /* while */

  fclose (FP);
  return 0;
} /* load_state */

void usage() {

  fprintf(stderr, "flow-dscan:\n\n");
  fprintf(stderr, " -d #  set debug level.\n");
  fprintf(stderr, " -b do not detach\n");
  fprintf(stderr, " -i [!](list) input interface list filter\n");
  fprintf(stderr, " -I [!](list) output interface list filter\n");
  fprintf(stderr, " -l load state\n");
  fprintf(stderr, " -m enable multicast address filter\n");
  fprintf(stderr, " -p dump state on exit\n");
  fprintf(stderr, " -s state file\n");
  fprintf(stderr, " -t ager timeout (0 to disable ager)\n");
  fprintf(stderr, " -w filter inbound www (prot=6, srcPort=80, dstPort>1023)\n");
  fprintf(stderr, " -B do not detach, leave syslog enabled\n");
  fprintf(stderr, " -D depth of dst ip list (max 255)\n");
  fprintf(stderr, " -L suppress list file\n");
  fprintf(stderr, " -S trigger for port scan (max 1024)\n");
  fprintf(stderr, " -O # trigger warning on excessive flow octets\n");
  fprintf(stderr, " -P # trigger warning on excessive flow packets\n");
  fprintf(stderr, " -T # trigger warning on excessive flow time (duration)\n");
  fprintf(stderr, " -W filter outbound www (prot=6, dstPort=80, srcPort>1023)\n");
  fprintf(stderr, " signals\n");
  fprintf(stderr, "   SIGHUP  - reload suppress list\n");
  fprintf(stderr, "   SIGUSR1 - dump state\n");

  fprintf(stderr, "\n\n");
}

void sig_usr1_handler(sig)
int sig;
{
  extern int sig_usr1;
  sig_usr1 = 1;
}

void sig_hup_handler(sig)
int sig;
{
  extern int sig_hup;
  sig_hup = 1;
}

ager(ds, total_flows32)
struct dscan_state *ds;
u_int32 total_flows32;
{

  static int ager_i;
  int i, work;
  struct dscan_rec *drp, *odrp, *drp2;
  struct dscan_dst *ddp, *oddp, *ddp2;
  u_int32 i32;

  work = 0;
  for (i = ager_i; i < DSCAN_HASHSIZE; ++i) {
    odrp = (void*)0L;
    drp = ds->hash_scan[i].slh_first;
    while (drp) {
      work += 2;
      oddp = (void*)0L;
      ddp = drp->dlhead.stqh_first;
      while (ddp) {
        ++work;
        if (ddp->ip_time > total_flows32)
          i32 = 4294967295U - ddp->ip_time + total_flows32 + 1;
        else 
          i32 = total_flows32 - ddp->ip_time;
        if (i32 > ds->ager_timeout) { 
          if (debug > 5)
            report(LOG_INFO, "ager: remove max=%u i32=%u",
              ds->ager_timeout, i32);
          --drp->depth;
          if (oddp) {
            oddp->chain.stqe_next = ddp->chain.stqe_next;
            if (!ddp->chain.stqe_next) /* tail update */
              drp->dlhead.stqh_last = &oddp->chain.stqe_next;
          }
          else {
            drp->dlhead.stqh_first = ddp->chain.stqe_next;
            if (!ddp->chain.stqe_next) /* tail update */
              drp->dlhead.stqh_last = &ddp->chain.stqe_next;
          }
          ddp2 = ddp;
          ddp = ddp->chain.stqe_next;
          free (ddp2);
          ++ds->stat_free;
          ++ds->stat_free_dst;
          ++ds->stat_aged_ip;
          continue;
        }
        oddp = ddp;
        ddp = ddp->chain.stqe_next;
      } /* while ddp */
            /*  if they're all gone, delete the record itself */
      if (!drp->depth) {
        if (debug > 5)
          report(LOG_INFO, "ager: remove record");
        if (odrp)
          odrp->chain.sle_next = drp->chain.sle_next;
        else
          ds->hash_scan[i].slh_first = drp->chain.sle_next;
        
        drp2 = drp;
        drp = drp->chain.sle_next;
        free (drp2);
        ++ds->stat_free;
        ++ds->stat_free_rec;
        ++ds->stat_aged_dsr;
        continue;
      } 
  
      odrp = drp;
      drp = drp->chain.sle_next;
  
    } /* while */
  
    /* simple work queue */
    if (++work > DSCAN_AGER_WORK) {
      if (debug > 2) {
        report(LOG_INFO,
          "ager: work=%d malloc=%u free=%u aged_ip=%u aged_dsr=%u",
          work, ds->stat_malloc, ds->stat_free, ds->stat_aged_ip,
          ds->stat_aged_dsr);
        report(LOG_INFO,
          "ager: malloc_dst=%u malloc_rec=%u free_dst=%u free_rec=%u",
          ds->stat_malloc_dst, ds->stat_malloc_rec, ds->stat_free_dst,
          ds->stat_free_rec);
      }
      ager_i = i++;
      goto skip3;
    }
  } /* for hash */
  ager_i = 0;
  report(LOG_INFO,"ager: reset hash run");

skip3:

} /* ager */

load_suppress(ds, sd)
struct dscan_state *ds;
int sd;
{
  char buf1[1024], *c1, *c2, *c3, *c4, *c5, *c;
  struct dscan_sup *dsp;
  int i, match;
  FILE *FP;
  u_int32 prot, srcport, dstport;
  u_int32 ip, hash;
  char *fname;

  report(LOG_INFO, "load_suppress %d", sd);

  if (!(fname = (char*)malloc(strlen(ds->supfile)+5))) {
    report(LOG_ERR, "malloc(): failed\n");
    return -1;
  }

  strcpy(fname, ds->supfile);

  if (sd == 0) {
    strcat(fname, ".src");
  } else if (sd == 1) {
    strcat(fname, ".dst");
  }

  if (!(FP = fopen(fname, "r"))) {
    report(LOG_ERR, "fopen(%s) failed", fname);
    return -1;
  }

  while (1) {

    if (!fgets(buf1, 1024, FP))
      break;

    /* skip whitespace */
    for (c = buf1; (*c == ' ') || (*c == '\t') && *c; ++c);
    c1 = c;

    if (*c1 == '#') 
      continue;

    if (*c1 == '\n')
      continue;

    for (; (*c != ' ') && (*c != '\t') && *c; ++c);
    for (; (*c == ' ') || (*c == '\t') && *c; ++c);
    c2 = c;

    for (; (*c != ' ') && (*c != '\t') && *c; ++c);
    for (; (*c == ' ') || (*c == '\t') && *c; ++c);
    c3 = c;

    for (; (*c != ' ') && (*c != '\t') && *c; ++c);
    for (; (*c == ' ') || (*c == '\t') && *c; ++c);
    c4 = c;

    if ((!*c1) || (!*c2) || (!*c3) || (!*c4)) {
      report(LOG_ERR, "suppress parser: syntax error: %s", buf1);
      continue;
    }

    if (debug > 5)
      report(LOG_INFO, "suppress parser: c1=%s c2=%s c3=%s c4=%s",
      c1, c2, c3, c4);

    ip = scan_ip(c1);
    sscanf(c2, "%u", &prot);
    sscanf(c3, "%u", &srcport);
    sscanf(c4, "%u", &dstport);

    hash = DSCAN_HASHFUNC(ip);


    if (sd == 0) {

      match = 0;
      SLIST_FOREACH(dsp, &ds->hash_sup_src[hash], chain) {
        if (dsp->ip == ip) {
          match = 1;
          break;
        }
      }
  
      if (match) {
        report(LOG_ERR, "suppress parser: dup ip: %s", buf1);
        continue;
      }
  
      if (*c1 == '-') {
        report(LOG_ERR, "suppress parser: no src ip wildcard");
        continue;
      }
  
      if (!(dsp = malloc(sizeof (struct dscan_sup)))) {
        report(LOG_ERR, "malloc(): failed for dscan_spp");
        fclose(FP);
        return -1;
      }
  
      bzero(dsp, sizeof (struct dscan_sup));
      dsp->ip = ip;
      dsp->srcport = srcport;
      dsp->dstport = dstport;
      dsp->protocol = prot;
  
      if (*c2 != '-')
        dsp->flags &= DSCAN_SUP_PROTOCOL;
      if (*c3 != '-')
        dsp->flags &= DSCAN_SUP_SRCPORT;
      if (*c4 != '-')
        dsp->flags &= DSCAN_SUP_DSTPORT;
  
      
      SLIST_INSERT_HEAD(&ds->hash_sup_src[hash], dsp, chain);

    } else if (sd == 1) {

      match = 0;
      SLIST_FOREACH(dsp, &ds->hash_sup_dst[hash], chain) {
        if (dsp->ip == ip) {
          match = 1;
          break;
        }
      }
  
      if (match) {
        report(LOG_ERR, "suppress parser: dup ip: %s", buf1);
        continue;
      }
  
      if (*c1 == '-') {
        report(LOG_ERR, "suppress parser: no src ip wildcard");
        continue;
      }
  
      if (!(dsp = malloc(sizeof (struct dscan_sup)))) {
        report(LOG_ERR, "malloc(): failed for dscan_spp");
        fclose(FP);
        return -1;
      }
  
      bzero(dsp, sizeof (struct dscan_sup));
      dsp->ip = ip;
      dsp->srcport = srcport;
      dsp->dstport = dstport;
      dsp->protocol = prot;
  
      if (*c2 != '-')
        dsp->flags &= DSCAN_SUP_PROTOCOL;
      if (*c3 != '-')
        dsp->flags &= DSCAN_SUP_SRCPORT;
      if (*c4 != '-')
        dsp->flags &= DSCAN_SUP_DSTPORT;
  
      
      SLIST_INSERT_HEAD(&ds->hash_sup_dst[hash], dsp, chain);

    }
      
  } /* while 1 */

  fclose(FP);
  free (fname);
  return 0;
} /* load_suppress */

clear_suppress(ds, sd)
struct dscan_state *ds;
int sd;
{
  int i;
  struct dscan_sup *dsp_src, *dsp_dst;

/* XXX this is wrong.  should be iterating over i */

  if (sd == 0) {
    while (ds->hash_sup_src[i].slh_first != NULL) {
      dsp_src = ds->hash_sup_src[i].slh_first;
      SLIST_REMOVE_HEAD(&ds->hash_sup_src[i], chain);
      free (dsp_src);
    } /* while */
  }

  if (sd == 1) {
    while (ds->hash_sup_dst[i].slh_first != NULL) {
      dsp_dst = ds->hash_sup_dst[i].slh_first;
      SLIST_REMOVE_HEAD(&ds->hash_sup_dst[i], chain);
      free (dsp_dst);
    } /* while */
  }

} /* clear_suppress */


int flow_dump(fdata)
struct flow_data *fdata;
{
  char fmt_buf1[64], fmt_buf2[64];
  struct tm *tm;
  u_long start_secs, start_msecs;
  u_long end_secs, end_msecs;
  static int header;

  if (!header) {
    report(LOG_INFO, "Start             End                Sif SrcIPaddress      SrcP  DIf DstIPaddress      DstP   P Fl       Pkts     Octets");
    header = 1;
  }

  /* start time */
  start_secs = fdata->unix_secs;
  start_msecs = fdata->unix_msecs;

  start_secs += fdata->First / 1000;
  start_msecs += fdata->First % 1000;

  if (start_msecs >= 1000) { 
    start_msecs -= 1000;
    start_secs += 1;
  }

  tm = localtime((time_t*)&start_secs);

  report(LOG_INFO, "%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
    (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
    (int)tm->tm_min, (int)tm->tm_sec, (u_long)start_msecs);

  /* end time */
  end_secs = fdata->unix_secs;
  end_msecs = fdata->unix_msecs;

  end_secs += fdata->Last / 1000;
  end_msecs += fdata->Last % 1000;

  if (end_msecs > 1000) { 
    end_msecs -= 1000;
    end_secs += 1;
  }

  tm = localtime((time_t*)&end_secs);

  report(LOG_INFO, "%-2.2d%-2.2d.%-2.2d:%-2.2d:%-2.2d.%-3lu ",
    (int)tm->tm_mon+1, (int)tm->tm_mday, (int)tm->tm_hour,
    (int)tm->tm_min, (int)tm->tm_sec, (u_long)end_msecs);

  /* other info */
  fmt_ipv4(fmt_buf1, fdata->srcaddr, FMT_PAD_RIGHT);
  fmt_ipv4(fmt_buf2, fdata->dstaddr, FMT_PAD_RIGHT);

  report(LOG_INFO, "%4d %-15.15s %6d %4d %-15.15s %6d %3d %2d %10lu %10lu",
    (int)fdata->input, fmt_buf1, (int)fdata->srcport, 
    (int)fdata->output, fmt_buf2, (int)fdata->dstport,
    (int)fdata->prot, 
    (int)fdata->flags & 0x7,
    (u_long)fdata->dPkts, 
    (u_long)fdata->dOctets);

} /* flow_dump */

