#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/uio.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <fcntl.h>
#include "flow.h"
#include "fmt.h"

int debug;
void usage();

int main(argc, argv)
int argc;
char **argv;
{

  extern char *optarg;
  extern int optind, debug, errno;
  int i, z_level, out_byte_order, out_fd, out_fd_plain;
  char *fname, *out_fname;
  struct flow_stream fs_in, fs_out;
  struct flow_data *fdata;
  struct flow_header fh_new;
  u_int64 total_flows;
  struct flow_profile fp;
  struct stat stat_buf;

  /* profile */
  if (profile_start (&fp) == -1) {
    fprintf(stderr, "profile_start(): failed\n");
    exit (1);
  }

  out_fd_plain = 0;
  out_fname = (char*)0L;
  out_fd = -1;
  z_level = 0;
  bzero (&fs_out, sizeof fs_out);
  bzero (&fh_new, sizeof fh_new);
  total_flows = 0;
#if BYTE_ORDER == BIG_ENDIAN
  out_byte_order = BIG_ENDIAN;
#endif /* BYTE_ORDER == BIG_ENDIAN */
#if BYTE_ORDER == LITTLE_ENDIAN
  out_byte_order = LITTLE_ENDIAN;
#endif /* BYTE_ORDER == LITTLE_ENDIAN */


  while ((i = getopt(argc, argv, "b:C:d:h?o:z:")) != -1)

    switch (i) {

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        out_byte_order = LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        out_byte_order = BIG_ENDIAN;
      else {
        fprintf(stderr, "expecting \"big\" or \"little\" at -b\n");
        exit (1);
      }
      break;

    case 'C': /* comment field */
      if (strlen(optarg) > (FH_COMMENTS_LEN-1)) {
        fprintf(stderr, "comments limited to %d bytes.\n", (FH_COMMENTS_LEN-1));
        exit (1);
      }
      strcpy((char*)&fh_new.comments, optarg);
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'o': /* output filename */
      out_fname = optarg;
      break;


    case 'z': /* compress level */
      z_level = atoi(optarg);
      if ((z_level < 0) || (z_level > 9)) {
        fprintf(stderr, "Compression level must be between 0 and 9\n");
        exit (1);
      }
      break;

    case 'h': /* help */
    case '?': 
    default:
      usage();
      exit (1);
      break;

    } /* switch */

  /* set compression level in output stream */
  if (z_level != Z_NO_COMPRESSION) {
    fs_out.flags = FS_FLAG_COMPRESS;
    fs_out.z_level = z_level;
    fh_new.flags = FF_FLAG_COMPRESS;
  } else
    fs_out.flags = 0;

  i = optind;

  fh_new.magic1 = FF_MAGIC1;
  fh_new.magic2 = FF_MAGIC2;
  fh_new.s_version = FSTREAM_VERSION;
  fh_new.d_version = FF_D_VERSION_UNKNOWN;

#if BYTE_ORDER == LITTLE_ENDIAN
  fh_new.byte_order = FF_LITTLE_ENDIAN;

  if (out_byte_order == BIG_ENDIAN)
    swap_flow_header(&fh_new);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == BIG_ENDIAN
  fh_new.byte_order = FF_BIG_ENDIAN;

  if (out_byte_order == LITTLE_ENDIAN)
    swap_flow_header(&fh_new);
#endif /* BYTE_ORDER == BIG_ENDIAN */


  /* if out_fname is not set, then use stdout */
  if (out_fname) {

    if ((out_fd = open(out_fname,  O_WRONLY|O_CREAT|O_TRUNC, 0644)) == -1) {
      fprintf(stderr, "open(): %s\n", strerror(errno));
      exit (1);
    }

    if (fstat(out_fd, &stat_buf) == -1) {
      fprintf(stderr, "stat(): %s\n", strerror(errno));
      exit (1);
    }

    /* is this a plain file? */
    if (!stat_buf.st_rdev)
      out_fd_plain = 1;

  } else 
    out_fd = 1;


  if ((writen(out_fd, &fh_new, sizeof fh_new)) == -1) {
    fprintf(stderr, "writen(): %s\n", strerror(errno));
    exit (1);
  }

  /* need header in host byte order later */
#if BYTE_ORDER == LITTLE_ENDIAN
  fh_new.byte_order = FF_LITTLE_ENDIAN;

  if (out_byte_order == BIG_ENDIAN)
    swap_flow_header(&fh_new);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == BIG_ENDIAN
  fh_new.byte_order = FF_BIG_ENDIAN;

  if (out_byte_order == LITTLE_ENDIAN)
    swap_flow_header(&fh_new);
#endif /* BYTE_ORDER == BIG_ENDIAN */

  /* write to stdout */
  fs_out.fd = out_fd;

  /* handle special case of only stdin */
  if (i >= argc) {
    fname = "-";
    goto skip1;
  }

  for (; i < argc; ++i) {

    fname = argv[i];
skip1:

    if (debug > 5)
      fprintf(stderr, "fname=%s\n", fname);

    bzero (&fs_in, sizeof fs_in);

    if (fname[0] == '-' && !fname[1]) {

      /* special case, stdin */
      fs_in.fd = 0;

    } else {

      if ((fs_in.fd = open(fname, O_RDONLY, 0)) == -1) {
        fprintf(stderr, "open (%s): %s\n", fname, strerror(errno));
        continue;
      }
    }

    if (flow_read_header(&fs_in) == -1) {
      fprintf(stderr, "flow_read_header(): failed for %s\n", fname);
      continue;
    }

    while ((fdata = flow_read(&fs_in))) {

      ++total_flows;

#if BYTE_ORDER == BIG_ENDIAN
      if (out_byte_order == LITTLE_ENDIAN)
        swap_flow_data(fdata);
#endif /* BYTE_ORDER == BIG_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
      if (out_byte_order == BIG_ENDIAN)
        swap_flow_data(fdata);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

      if (flow_write(fdata, &fs_out) == -1) {
        fprintf(stderr, "flow_write(): failed\n");
        exit (1);
      }
    }

    close(fs_in.fd);

    flow_read_end(&fs_in);

  } /* foreach filename */

  if (flow_flush(&fs_out) == -1) {
    fprintf(stderr, "flow_flush(): failed.\n");
    exit (1);
  }

  flow_write_end(&fs_out);

  fp.nflows = total_flows;

  /*
  /* if the output file descriptor was a real file, re-write the
  /* flow_header with the correct # of total flows
  */
  if (out_fd_plain) {

    if (lseek(out_fd, (off_t)0L, SEEK_SET) == -1) {
      fprintf(stderr, "lseek(): %s\n", strerror(errno));
      exit (1);
    }

    fh_new.flags |= FF_FLAG_DONE;
    fh_new.nflows = total_flows;

#if BYTE_ORDER == LITTLE_ENDIAN
    fh_new.byte_order = FF_LITTLE_ENDIAN;

    if (out_byte_order == BIG_ENDIAN)
      swap_flow_header(&fh_new);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == BIG_ENDIAN
    fh_new.byte_order = FF_BIG_ENDIAN;

    if (out_byte_order == LITTLE_ENDIAN)
      swap_flow_header(&fh_new);
#endif /* BYTE_ORDER == BIG_ENDIAN */

  
    if ((writen(out_fd, &fh_new, sizeof fh_new)) == -1) {
      fprintf(stderr, "writen(): %s\n", strerror(errno));
      exit (1);
    }

  } /* out_fd_plain */


  if (close(out_fd) == -1) {
    fprintf(stderr, "close(): %s\n", strerror(errno));
    exit (1);
  }


  if (debug > 0)
    if (profile_end(argv[0], &fp) == -1) {
      fprintf(stderr, "profile_end(): failed\n");
      exit (1);
    }

  return 0;
    

} /* main */

void usage() {

  fprintf(stderr, "flow-cat:\n\n");
  fprintf(stderr, " -b s  set output byte order (little or big)\n");
  fprintf(stderr, " -C s  set comment field in output\n");
  fprintf(stderr, " -d #  set debug level.\n");
  fprintf(stderr, " -o s  output to file s instead of stdout\n");
  fprintf(stderr, " -z #  set compression level (0..9).\n");
  fprintf(stderr, "\n\n");
}

