#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <sys/signal.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/uio.h>
#include <sys/errno.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <dirent.h>
#include <fcntl.h>
#include <zlib.h>
#include <string.h>
#include <syslog.h>
#include "flow.h"
#include "flow-capture.h"
#include "support.h"
#ifdef LIBWRAP
#include <tcpd.h>
#endif /* LIBWRAP */
#ifdef LIBWRAP /* XXX */
#include <sys/file.h>
#endif /* XX */

/* TODO
/*
/*  v1, v5, v6, v8 issues
/*  call external program on file rotation?
/*  stats file
/*   time   flows/second
/*   1111   1099
/*  unlink pidfile after SIGQUIT, or any other exit
/*
/*  The TCP client write() needs to have userland buffering
/*
*/

#ifdef TIMELOCAL_MKTIME
#define timelocal mktime
#endif /* TIMELOCAL_MKTIME */

int debug;
int sig_pipe_flag, sig_quit_flag, sig_hup_flag;
void sig_pipe();
void sig_quit();
void sig_hup();
char *progname = "flow-capture\0";
int detach;

char *calc_fname();
void usage();
int calc_rotate();

#ifdef LIBWRAP
int allow_severity = LOG_INFO;
int deny_severity = LOG_WARNING; 
#endif /* LIBWRAP */

int main(argc, argv)
int argc;
char **argv;
{
  time_t trotate; /* time of next rotation */
  int crotate;  /* current rotation */
  int nrotate;  /* # of rotations per day */
  int i, n, len, x, force_newfile, out_byte_order, sendbuf;
  extern char *optarg;
  struct flow_header fh, fc_fh;
  struct flow_pdu_v1 *fpdu_v1;
  struct flow_pdu_v5 *fpdu_v5;
  struct flow_data fdata;
  struct flow_stream fs;
  char cur_fname[MAXPATHNAME+1], fname[MAXPATHNAME+1], wpname[MAXPATHNAME+1];
  extern int debug, errno;
  int fport, fsockfd, ffd, z_level, doit;
  pid_t pid;
  time_t now, time_fname;
  fd_set rfd;
  struct sockaddr_in fsocaddr, fc_addr;
  u_long nflows, j, k;
  struct timeval tv;
  char hostname[MAXDNAME+1];
  u_int16 fpdu_version;
  char rcv_buf[FLOW_RECV_BUFSIZE];
  struct flow_client fc[MAXCONNECTIONS+1];
  int fc_sockfd, fc_len, fc_num;
#ifdef LIBWRAP
  struct request_info fc_tcpd;
#endif /* LIBWRAP */
  struct file_ager fa;
  u_int32 nbytes;

  nrotate = 95;  /* rotate every 15 minutes */
  fport = FLOWPORT;
  debug = 0;
  wpname[0] = 0;
  ffd = -1;
  bzero(&fh, sizeof fh);
  bzero(&tv, sizeof tv);
  nflows = 0;
  z_level = Z_DEFAULT_COMPRESSION;
  force_newfile = 0;
  tv.tv_sec = 5;
  bzero(&fc, sizeof fc);
  bzero(&fc_addr, sizeof fc_addr);
  bzero(&fsocaddr, sizeof(fsocaddr));
  fsocaddr.sin_family = AF_INET;
  fsocaddr.sin_addr.s_addr = htonl(INADDR_ANY);
  fc_addr.sin_family = AF_INET;
  fc_addr.sin_addr.s_addr = htonl(INADDR_ANY);
  fc_num = 0;
  detach = 1; /* daemonize by default */
  bzero(&fa, sizeof fa);

#if BYTE_ORDER == BIG_ENDIAN
  out_byte_order = BIG_ENDIAN;
#endif /* BYTE_ORDER == BIG_ENDIAN */
#if BYTE_ORDER == LITTLE_ENDIAN
  out_byte_order = LITTLE_ENDIAN;
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

  while ((i = getopt(argc, argv, "ab:c:C:d:De:E:n:p:w:z:")) != -1)

    switch (i) {

    case 'a': /* always create new file */
      force_newfile = 1;
      break;

    case 'b': /* output byte order */
      if (!strcasecmp(optarg, "little"))
        out_byte_order = LITTLE_ENDIAN;
      else if (!strcasecmp(optarg, "big"))
        out_byte_order = BIG_ENDIAN;
      else {
        fprintf(stderr, "expecting \"big\" or \"little\" at -b\n");
        exit (1);
      }
      break;

    case 'c': /* client enable */
      fc_num = atoi(optarg);
      if (fc_num > MAXCONNECTIONS) {
        fprintf(stderr, "MAXCONNECTIONS (%d) exceeded\n", MAXCONNECTIONS);
        exit (1);
      }
      break;

    case 'C': /* comment field */
      if (strlen(optarg) > (FH_COMMENTS_LEN-1)) {
        fprintf(stderr, "comments limited to %d bytes.\n",
          (FH_COMMENTS_LEN-1));
        exit (1);
      }
      strcpy((char*)&fh.comments, optarg);
      break;

    case 'D': /* daemonize */
      detach = 0;
      break;

    case 'e': /* expire */
      fa.max_files = atoi(optarg);
      break;

    case 'E': /* expire bytes */
      if ((fa.max_bytes = scan_size(optarg)) == -1) {
        fprintf(stderr, "error scanning size\n");
        exit (1);
      }
      break;

    case 'd': /* debug */
      debug = atoi(optarg);
      break;

    case 'n': /* # rotations / day */
      nrotate = atoi(optarg);
      /* no more than 1 rotation per 5 minutes */
      if (nrotate > (288-1)) {
        fprintf(stderr, "rotations limited to every 5 minutes\n");
        exit (1);
      }
      break;

    case 'p': /* listen port */
      fport = atoi(optarg);
      break;

    case 'w': /* working directory */
      if (strlen(optarg) > (MAXPATHNAME)) {
        fprintf(stderr, "Pathname too long\n");
        exit (1);
      }
      strcpy(wpname, optarg);
      break;

    case 'z': /* compress level */
      z_level = atoi(optarg);
      if ((z_level < 0) || (z_level > 9)) {
        fprintf(stderr, "Compression level must be between 0 and 9\n");
        exit (1);
      }
      break;

    default:
      usage();
      exit (1);
      break;

    } /* switch */

  /* TCP client port */
  fc_addr.sin_port = htons(fport);

  /* do work in the file ager? */
  if (debug)
    doit = 0;
  else
    doit = 1;

  fsocaddr.sin_port = htons(fport);

  /* daemonize */
  if (detach) {
    if ((pid = fork()) == -1) {
      fprintf(stderr, "fork(): %s\n", strerror(errno));
      exit (1);
    } else if (pid) {
      write_pid(pid, FLOW_CAPTURE_PIDFILE, fport);
      exit (0); /* parent */
    }

    umask(0022);
    setsid();
    for (n = 0; n < 16; ++n) /* XXX dynamically get NOFILE */
      close (n);

    report_init(0);

  } else {
    report_init(1);
  }

  if (fc_num) {

    /* socket for listen */
    if ((fc_sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
      report(LOG_ERR, "socket(): %s", strerror(errno));
      exit (1);
    }

    if (bind(fc_sockfd, (struct sockaddr*)&fc_addr, sizeof (fc_addr)) < 0) {
      report(LOG_ERR, "bind(): %s", strerror(errno));
      exit (1);
    }

    /* listen for new TCP connections */
    if (listen(fc_sockfd, 5) < 0) {
      report(LOG_ERR, "listen(): %s", strerror(errno));
      exit (1);
    }

    /* non blocking */
    if (fcntl(fc_sockfd, F_SETFL, FNDELAY) < 0) {
      report(LOG_ERR, "fcntl(): %s", strerror(errno));
      exit (1);
    }
  } /* fc_num */

  /* rotations needs to divide the day evenly */
  if (nrotate && (86400 % (nrotate+1))) {
    report(LOG_ERR, "rotations must divide the day evenly.");
    exit (1);
  }

  if (signal(SIGPIPE, sig_pipe) < 0) {
    report(LOG_ERR, "signal(): %s", strerror(errno));
    exit (1);
  }

  if (signal(SIGHUP, sig_hup) < 0) {
    report(LOG_ERR, "signal(): %s", strerror(errno));
    exit (1);
  }

  if (signal(SIGQUIT, sig_quit) < 0) {
    report(LOG_ERR, "signal(): %s", strerror(errno));
    exit (1);
  }

  now = time((time_t)0L);

  /* sandbox */
  if (chdir(wpname) == -1) {
    report(LOG_ERR, "chdir(): %s", strerror(errno));
    exit (1);
  }

  /*
  /* load directory entries into memory
  */
  if (file_scandir(&fa, ".")) {
    report(LOG_ERR, "file_scandir(): failed");
    exit (1);
  }

  if (debug)
    file_dump(&fa);
  
  /*
  /*  run the file ager
  */
  if (file_expire(&fa, doit, (u_int32)0)) {
    report(LOG_ERR, "file_expire(): failed");
    exit (1);
  }

  /* get hostname */
  if (gethostname((char*)&hostname, (int)MAXDNAME) == -1) {
    report(LOG_ERR, "gethostname(): %s", strerror(errno));
    exit (1);
  }

  bcopy(hostname, fh.hostname, FH_HOSTNAME_LEN);
  fh.hostname[FH_HOSTNAME_LEN-1] = 0; /* ensure null terminated */

  fh.magic1 = FF_MAGIC1;
  fh.magic2 = FF_MAGIC2;
  fh.s_version = FSTREAM_VERSION;

#if BYTE_ORDER == LITTLE_ENDIAN
  fh.byte_order = FF_LITTLE_ENDIAN;
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == BIG_ENDIAN
  fh.byte_order = FF_BIG_ENDIAN;
#endif /* BYTE_ORDER == BIG_ENDIAN */

  /* socket to receive flow pdu exports */
  if ((fsockfd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
    report(LOG_ERR, "socket(): %s", strerror(errno));
    exit(1);
  }

  /* socket buffer */
  sendbuf = MAX_SOCKETBUF;
  if (setsockopt(fsockfd, SOL_SOCKET, SO_RCVBUF, (char*)&sendbuf, 
    sizeof (sendbuf)) < 0) {
    report(LOG_ERR, "setsockopt(SO_RCVBUF=%d): %s", MAX_SOCKETBUF,
      strerror(errno));
    exit(1);
  }

  if (bind(fsockfd, (struct sockaddr*)&fsocaddr, sizeof(fsocaddr)) < 0) {
    report(LOG_ERR, "bind(): %s", strerror(errno));
    exit (1);
  }

  /* hint to create the first flow export file */ 
  trotate = 0;

  len = sizeof(fsocaddr);

  fpdu_v1 = (struct flow_pdu_v1*)rcv_buf;
  fpdu_v5 = (struct flow_pdu_v5*)rcv_buf;

  bcopy(&fh, &fc_fh, sizeof fh);
  fc_fh.flags = 0;

  while (1) {

    FD_ZERO (&rfd);
    FD_SET (fsockfd, &rfd);
    if (fc_num)
      FD_SET (fc_sockfd, &rfd);

    if (select (fsockfd+1, &rfd, (fd_set *)0, (fd_set *)0, &tv) < 0)  {
      if (errno == EINTR) {
        FD_ZERO (&rfd);
      } else {
        report(LOG_ERR, "select(): %s", strerror(errno));
        exit (1);
      }
    }

    now = time((time_t)0L);

    /* new client connection ? */
    if (fc_num && (FD_ISSET(fc_sockfd, &rfd))) {

      i = -1;
      x = 0;

      for (n = 0; n < fc_num; ++n) {
        if (!(fc[n].flags & FLOW_CLIENT_VALID)) {
          i = n;
          break;
        }
      }

      /* if > max connections, accept it and exit */
      if (i == -1) {
        report(LOG_WARNING, "too many clients");
        i = MAXCONNECTIONS;
        x = 1;
      }

      fc_len = sizeof(fc[i].addr);
      fc[i].sockfd = accept(fc_sockfd, (struct sockaddr*)&fc[i].addr, &fc_len);

      sendbuf = MAX_SOCKETBUF;
      if (setsockopt(fc[i].sockfd, SOL_SOCKET, SO_SNDBUF, (char*)&sendbuf, 
        sizeof (sendbuf)) < 0) {
        report(LOG_ERR, "setsockopt(SO_SNDBUF=%d): %s", MAX_SOCKETBUF,
          strerror(errno));
        exit(1);
      }

      if (fc[i].sockfd < 0) 
        report(LOG_ERR, "accept(): %s", strerror(errno));
      else {
        report(LOG_INFO, "new client i=%d IP=%s", i,
          inet_ntoa(fc[i].addr.sin_addr));
        fc[i].flags = FLOW_CLIENT_VALID;
        fc_fh.start = now;
      }

#ifdef LIBWRAP
      request_init(&fc_tcpd, RQ_DAEMON, "flow-capture", RQ_FILE, fc[i].sockfd,
        NULL);
      fromhost(&fc_tcpd);
      if (!hosts_access(&fc_tcpd)) {
        report(LOG_WARNING, "refused by libwrap");
        close(fc[i].sockfd);
        fc[i].flags = 0;
        x = 0;
      }
#endif /* LIBWRAP */

      if (x) {
        close(fc[i].sockfd);
      }

    } /* new client */


    /* need new flow file */
    if (ffd == -1) {

      /* calculate the current rotation and next rotate time */
      if (calc_rotate(nrotate, &trotate, &crotate) == -1) {
        report(LOG_ERR, "calc_rotate() failed");
        exit (1);
      }

      time_fname = now;
      strcpy(cur_fname, calc_fname(0, 5, time_fname));

      if ((ffd = open(cur_fname, O_WRONLY|O_CREAT|O_TRUNC, 0644)) == -1) {
        report(LOG_ERR, "open(%s): %s", cur_fname, strerror(errno));
        exit (1);
      }

      nbytes = sizeof (fh);

      /* init the new output stream */
      bzero(&fs, sizeof fs);

      fs.fd = ffd;

      /* is the flow data compressed? */
      if (z_level != Z_NO_COMPRESSION) {
        fh.flags = FF_FLAG_COMPRESS;
        fs.flags = FS_FLAG_COMPRESS;
        fs.z_level = z_level;
      } else
        fh.flags = 0;

      fh.start = now;
      fh.rotation = nrotate;

#if BYTE_ORDER == BIG_ENDIAN
      if (out_byte_order == LITTLE_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
      if (out_byte_order == BIG_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

      if (write(ffd, &fh, sizeof fh) == -1) {
        report(LOG_ERR, "write(): %s", strerror(errno));
        exit (1);
      }

      /* need header in host byte order later */

#if BYTE_ORDER == BIG_ENDIAN
      if (out_byte_order == LITTLE_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
      if (out_byte_order == BIG_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */


    } /* ffd == -1 */

    if (FD_ISSET(fsockfd, &rfd)) {

      if ((n = recvfrom(fsockfd, (char*)&rcv_buf, sizeof rcv_buf, 0,
        (struct sockaddr*)&fsocaddr, &len)) == -1) {
        report(LOG_ERR, "recvfrom(): %s", strerror(errno));
        exit (1);
      }

      /* the first 2 bytes are the flow version */
      bcopy(rcv_buf, (char*)&fpdu_version, 2);
      
#if BYTE_ORDER == LITTLE_ENDIAN
      SWAPINT16(fpdu_version);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

      /*
      /* if the d_version is allready set, then we've
      /* received multiple flow export versions
      */
      if (fh.d_version != fpdu_version) {

        if (fh.d_version) {

          report(LOG_WARNING, "multiple pdu versions!");

          fh.flags |= FF_FLAG_MULT_PDU;

          /* the clients are going to get confused, just reset them */
          for (i = 0; i < MAXCONNECTIONS; ++i) {

            if (!(fc[i].flags & FLOW_CLIENT_VALID)) 
              continue;

            close(fc[i].sockfd);
            fc[i].flags = 0;
          }
        }

        fh.d_version = fc_fh.d_version = fpdu_version;
      } /* fh.d_version */

      if (fh.d_version == 1) {

#if BYTE_ORDER == LITTLE_ENDIAN
        if (out_byte_order == LITTLE_ENDIAN)
          swap_flow_pdu_v1(fpdu_v1, BIG_ENDIAN);
        else if (out_byte_order == BIG_ENDIAN)
          swap_flow_pdu2_v1(fpdu_v1, BIG_ENDIAN);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

        nflows += fpdu_v1->count;
  
        /* convert the pdu to internal format */
  
        j = fpdu_v1->sysUpTime / 1000;
        k = fpdu_v1->sysUpTime % 1000;
  
        fdata.unix_secs = fpdu_v1->unix_secs;
        fdata.unix_msecs = fpdu_v1->unix_nsecs / 1000000;
  
        fdata.unix_secs -= j;
        if (k > fdata.unix_msecs) {
          -- fdata.unix_secs;
          fdata.unix_msecs += 1000;
        }
        fdata.unix_msecs -= k;
  
        for (x = 0; x < fpdu_v1->count; ++x) {
  
          fdata.srcaddr = fpdu_v1->records[x].srcaddr;
          fdata.dstaddr = fpdu_v1->records[x].dstaddr;
          fdata.nexthop = fpdu_v1->records[x].nexthop;
          fdata.input = fpdu_v1->records[x].input;
          fdata.output = fpdu_v1->records[x].output;
          fdata.dPkts = fpdu_v1->records[x].dPkts;
          fdata.dOctets = fpdu_v1->records[x].dOctets;
          fdata.First = fpdu_v1->records[x].First;
          fdata.Last = fpdu_v1->records[x].Last;
          fdata.dstport = fpdu_v1->records[x].dstport;
          fdata.srcport = fpdu_v1->records[x].srcport;
          fdata.prot = fpdu_v1->records[x].prot;
          fdata.tos = fpdu_v1->records[x].tos;
          fdata.flags = fpdu_v1->records[x].flags;
          fdata.tcp_retx_cnt = fpdu_v1->records[x].tcp_retx_cnt;
          fdata.tcp_retx_secs = fpdu_v1->records[x].tcp_retx_secs;
          fdata.tcp_misseq_cnt = fpdu_v1->records[x].tcp_misseq_cnt;
  
#if BYTE_ORDER == BIG_ENDIAN
          if (out_byte_order == LITTLE_ENDIAN)
            swap_flow_data(&fdata);
#endif /* BYTE_ORDER == BIG_ENDIAN */
  
          if ((n = flow_write(&fdata, &fs)) == -1) {
            report(LOG_ERR, "flow_write(): failed");
            exit (1);
          }

          nbytes += n;

          fanout(&fdata, &fc, &fc_fh, fc_num, out_byte_order);

        } /* for x */
  
      } else if (fh.d_version == 5) {

#if BYTE_ORDER == LITTLE_ENDIAN
        if (out_byte_order == LITTLE_ENDIAN)
          swap_flow_pdu_v5(fpdu_v5, BIG_ENDIAN);
        else if (out_byte_order == BIG_ENDIAN)
          swap_flow_pdu2_v5(fpdu_v5, BIG_ENDIAN);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */
  
        nflows += fpdu_v5->count;
  
        /* convert the pdu to internal format */
  
        j = fpdu_v5->sysUpTime / 1000;
        k = fpdu_v5->sysUpTime % 1000;
  
        fdata.unix_secs = fpdu_v5->unix_secs;
        fdata.unix_msecs = fpdu_v5->unix_nsecs / 1000000;
  
        fdata.unix_secs -= j;
        if (k > fdata.unix_msecs) {
          -- fdata.unix_secs;
          fdata.unix_msecs += 1000;
        }
        fdata.unix_msecs -= k;
  
        for (x = 0; x < fpdu_v5->count; ++x) {
  
          fdata.srcaddr = fpdu_v5->records[x].srcaddr;
          fdata.dstaddr = fpdu_v5->records[x].dstaddr;
          fdata.nexthop = fpdu_v5->records[x].nexthop;
          fdata.input = fpdu_v5->records[x].input;
          fdata.output = fpdu_v5->records[x].output;
          fdata.dPkts = fpdu_v5->records[x].dPkts;
          fdata.dOctets = fpdu_v5->records[x].dOctets;
          fdata.First = fpdu_v5->records[x].First;
          fdata.Last = fpdu_v5->records[x].Last;
          fdata.dstport = fpdu_v5->records[x].dstport;
          fdata.srcport = fpdu_v5->records[x].srcport;
          fdata.flags = fpdu_v5->records[x].tcp_flags;
          fdata.prot = fpdu_v5->records[x].prot;
          fdata.tos = fpdu_v5->records[x].tos;
          fdata.src_as = fpdu_v5->records[x].src_as;
          fdata.dst_as = fpdu_v5->records[x].dst_as;
          fdata.src_mask = fpdu_v5->records[x].src_mask;
          fdata.dst_mask = fpdu_v5->records[x].dst_mask;
          fdata.drops = fpdu_v5->records[x].drops;
  
#if BYTE_ORDER == BIG_ENDIAN
          if (out_byte_order == LITTLE_ENDIAN)
            swap_flow_data(&fdata);
#endif /* BYTE_ORDER == BIG_ENDIAN */
  
          if ((n = flow_write(&fdata, &fs)) == -1) {
            report(LOG_ERR, "flow_write(): failed");
            exit (1);
          }

          nbytes += n;

          fanout(&fdata, &fc, &fc_fh, fc_num, out_byte_order);
  
        } /* for x */

      } else {
        report(LOG_WARNING, "unknown flow pdu version (%u).",
          (unsigned int)fpdu_version);
        continue;
      }

    } /* if FD_ISSET */

    /*
    /* time for a new file ?
    */
    if ((now > trotate) || sig_quit_flag || sig_hup_flag) {

      if (ffd != -1) {

        sig_hup_flag = 0; /* re-arm */

        if ((n = flow_flush(&fs)) == -1) {
          report(LOG_ERR, "flow_flush(): failed");
          exit (1);
        }

        nbytes += n;

        if (flow_write_end(&fs) == -1) {
          report(LOG_ERR, "flow_write_end(): failed");
          exit (1);
        }

        if (lseek(ffd, (off_t)0L, SEEK_SET) == -1) {
          report(LOG_ERR, "lseek(): %s", strerror(errno));
          exit (1);
        }

        fh.nflows = nflows;
        fh.end = now;
        fh.flags |= FF_FLAG_DONE;

        nflows = 0;

#if BYTE_ORDER == BIG_ENDIAN
      if (out_byte_order == LITTLE_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
      if (out_byte_order == BIG_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

        if (write(ffd, &fh, sizeof fh) == -1) {
          report(LOG_ERR, "write(): %s", strerror(errno));
          exit (1);
        }

        /* need header in host byte order later */
#if BYTE_ORDER == BIG_ENDIAN
      if (out_byte_order == LITTLE_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
      if (out_byte_order == BIG_ENDIAN)
        swap_flow_header(&fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

        if (close(ffd) == -1) {
          report(LOG_ERR, "close(): %s", strerror(errno));
          exit (1);
        }


        /* rename current to done */
        strcpy(fname, calc_fname(1, 5, time_fname));

        if (rename(cur_fname, fname) == -1) {
          report(LOG_ERR, "rename(%s,%s): %s", cur_fname, fname,
            strerror(errno));
          exit (1);
        }

        if (file_add_tail(&fa, fname, (off_t)nbytes, fh.start)) {
          report(LOG_ERR, "file_add_tail(%s): failed\n", fname);
        }

        /* invalidate flow file descriptor */
        ffd = -1;

        /* reset for next file */
        nbytes = 0;

        /* had enough */
        if (sig_quit_flag)
          goto main_exit;

      } /* ffd != -1 */
    } /* rotate time */

   /* expire old flows */
    if (!(nflows % 1000)) {
      if (file_expire(&fa, doit, nbytes)) {
        report(LOG_ERR, "file_expire(): failed");
        exit (1);
      }
    } /* ager run? */
  } /* while 1 */

main_exit:

  return 0;
} /* main */

int calc_rotate (nrotate, trotate, crotate)
int nrotate;    /* # of rotations / day */
time_t *trotate;  /* time of next rotation */
int *crotate;   /* # of rotation */
{

  time_t now;     /* current time */
  u_long irotate;   /* interval of nrotate in seconds */
  time_t morning;   /* start of the day */
  struct tm *tm1;

  irotate = 86400 / (nrotate+1);

  *crotate = 0;

  if ((now = time((time_t)0L)) == -1)
    return -1;

  /* calculate start of the day */
  if (!(tm1 = localtime (&now))) 
    return -1;

  tm1->tm_sec = 0;
  tm1->tm_min = 0;
  tm1->tm_hour = 0;

  if ((morning = timelocal(tm1)) == -1) 
    return -1;

  if (nrotate)
    *trotate = morning + irotate;
  else
    *trotate = morning + 86400;


  while (now > *trotate) {

    ++ *crotate;
    *trotate += irotate;

  }
  return 0;
}

char *calc_fname(done, ver, now)
int done, ver;
time_t now;
{

  struct tm *tm;
  static char fname[FLOWFILELEN+1];

  if (!(tm = localtime (&now)))
    return "";

  /* cfN.YYYY-DD-MM.HHMMSS */
  if (done) 
    sprintf(fname, "cf%2.2d.%4.4d-%2.2d-%2.2d.%2.2d%2.2d%2.2d", ver,
      (int)tm->tm_year+1900, (int)tm->tm_mon+1, (int)tm->tm_mday,
      (int)tm->tm_hour, (int)tm->tm_min, (int)tm->tm_sec);
  else
    sprintf(fname, "tmp%2.2d.%4.4d-%2.2d-%2.2d.%2.2d%2.2d%2.2d", ver,
      (int)tm->tm_year+1900, (int)tm->tm_mon+1, (int)tm->tm_mday,
      (int)tm->tm_hour, (int)tm->tm_min, (int)tm->tm_sec);

  return fname;

} /* calc_fname */

void usage() {

  fprintf(stderr, "flow capture:\n\n");
  fprintf(stderr, " -b s  byte order (little or big)\n");
  fprintf(stderr, " -C s  comment field\n");
  fprintf(stderr, " -c #  enable n flow clients\n");
  fprintf(stderr, " -d #  debug level.\n");
  fprintf(stderr, " -D    do not daemonize.\n");
  fprintf(stderr, " -e #  files to keep before expire. (0=disable)\n");
  fprintf(stderr, " -E #  bytes to keep before expire. (0=disable)\n");
  fprintf(stderr, " -n #  file rotations per day.\n");
  fprintf(stderr, " -p #  UDP port to accept flows on.\n");
  fprintf(stderr, " -s #  size of flow files to keep before expire. (0=disable)\n");
  fprintf(stderr, "       use b,K,M,G as multipliers, ie 16M for 16*1024*1024\n");
  fprintf(stderr, " -w x  working directory pathname.\n");
  fprintf(stderr, " signals\n");
  fprintf(stderr, "   SIGHUP  - close and rotate current file\n");
  fprintf(stderr, "   SIGQUIT - close current file and exit\n");
  fprintf(stderr, "\n\n");

} /* usage */


fanout(fdata, fc, fc_fh, fc_num, out_byte_order)
struct flow_data *fdata;
struct flow_client *fc;
struct flow_header *fc_fh;
int out_byte_order;
{

  int i, n;
  extern int errno;

  /* fanout to the clients */
  for (i = 0; i < fc_num; ++i) {

    if (!(fc[i].flags & FLOW_CLIENT_VALID)) 
      continue;

    if (!(fc[i].flags & FLOW_CLIENT_HEADER)) {

      fc[i].flags |= FLOW_CLIENT_HEADER;

#if BYTE_ORDER == BIG_ENDIAN
    if (out_byte_order == LITTLE_ENDIAN)
      swap_flow_header(fc_fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
    if (out_byte_order == BIG_ENDIAN)
      swap_flow_header(fc_fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

      n = writen(fc[i].sockfd, (char*)fc_fh, sizeof (struct flow_header));

#if BYTE_ORDER == BIG_ENDIAN
    if (out_byte_order == LITTLE_ENDIAN)
      swap_flow_header(fc_fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

#if BYTE_ORDER == LITTLE_ENDIAN
    if (out_byte_order == BIG_ENDIAN)
      swap_flow_header(fc_fh);
#endif /* BYTE_ORDER == LITTLE_ENDIAN */

      if (n < 1) {
        report(LOG_ERR, "write (%d):%s", fc[i].sockfd, strerror(errno));
        close(fc[i].sockfd);
        fc[i].flags = 0;
        continue;
      } /* n */
    }

    n = writen(fc[i].sockfd, (char*)fdata, sizeof (struct flow_data));
    if (n < 1) {
      report(LOG_ERR, "write (%d):%s", fc[i].sockfd, strerror(errno));
      close(fc[i].sockfd);
      fc[i].flags = 0;
    } /* n */
  } /* i */
} /* fanout */

void sig_pipe()
{
  sig_pipe_flag = 1;
}

void sig_hup()
{
  sig_hup_flag = 1;
}

void sig_quit()
{
  sig_quit_flag = 1;
}

int write_pid (pid, file, port)
pid_t pid;
char *file;
int port;
{
  int fd, len;
  char str[16], *c;

  if (!(c = (char*)malloc(strlen(file)+16)))
    return -1;

  sprintf(c, "%s.%d", file, port);

  len = sprintf(str, "%u\n", (unsigned)pid);
  if ((fd = open(c, O_WRONLY|O_CREAT|O_TRUNC, 0644)) == -1){
    free (c);
    return -1;
  }
  free (c);
  if (write(fd, str, len) != len) {
    close (fd);
    return -1;
  }
  return close (fd);

} /* write_pid */


