/* flac - Command-line FLAC encoder/decoder
 * Copyright (C) 2000,2001  Josh Coalson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#if defined _WIN32 && !defined __CYGWIN__
/* where MSVC puts unlink() */
# include <io.h>
#else
# include <unistd.h>
#endif
#include <math.h> /* for floor() */
#include <stdio.h> /* for FILE et al. */
#include <string.h> /* for strcmp() */
#include "FLAC/all.h"
#include "decode.h"
#include "file.h"

typedef struct {
	const char *inbasefilename;
	FILE *fout;
	FLAC__bool abort_flag;
	FLAC__bool analysis_mode;
	analysis_options aopts;
	FLAC__bool test_only;
	FLAC__bool is_wave_out;
	FLAC__bool is_big_endian;
	FLAC__bool is_unsigned_samples;
	FLAC__uint64 total_samples;
	unsigned bps;
	unsigned channels;
	unsigned sample_rate;
	FLAC__bool verbose;
	FLAC__uint64 skip;
	FLAC__bool skip_count_too_high;
	FLAC__uint64 samples_processed;
	unsigned frame_counter;
} stream_info_struct;

static FLAC__FileDecoder *decoder;
static FLAC__bool is_big_endian_host;

/* local routines */
static FLAC__bool init(const char *infilename, stream_info_struct *stream_info);
static FLAC__bool write_little_endian_uint16(FILE *f, FLAC__uint16 val);
static FLAC__bool write_little_endian_uint32(FILE *f, FLAC__uint32 val);
static FLAC__StreamDecoderWriteStatus write_callback(const FLAC__FileDecoder *decoder, const FLAC__Frame *frame, const FLAC__int32 *buffer[], void *client_data);
static void metadata_callback(const FLAC__FileDecoder *decoder, const FLAC__StreamMetaData *metadata, void *client_data);
static void error_callback(const FLAC__FileDecoder *decoder, FLAC__StreamDecoderErrorStatus status, void *client_data);
static void print_stats(const stream_info_struct *stream_info);

int flac__decode_wav(const char *infile, const char *outfile, FLAC__bool analysis_mode, analysis_options aopts, FLAC__bool verbose, FLAC__uint64 skip)
{
	FLAC__bool md5_failure = false;
	stream_info_struct stream_info;

	decoder = 0;
	stream_info.abort_flag = false;
	stream_info.analysis_mode = analysis_mode;
	stream_info.aopts = aopts;
	stream_info.test_only = (outfile == 0);
	stream_info.is_wave_out = true;
	stream_info.verbose = verbose;
	stream_info.skip = skip;
	stream_info.skip_count_too_high = false;
	stream_info.samples_processed = 0;
	stream_info.frame_counter = 0;
	stream_info.inbasefilename = flac__file_get_basename(infile);
	stream_info.fout = 0; /* initialized with an open file later if necessary */

	FLAC__ASSERT(!(stream_info.test_only && stream_info.analysis_mode));

	if(!stream_info.test_only) {
		if(0 == strcmp(outfile, "-")) {
			stream_info.fout = stdout;
		}
		else {
			if(0 == (stream_info.fout = fopen(outfile, "wb"))) {
				fprintf(stderr, "%s: ERROR: can't open output file %s\n", stream_info.inbasefilename, outfile);
				return 1;
			}
		}
	}

	if(analysis_mode)
		flac__analyze_init(aopts);

	if(!init(infile, &stream_info))
		goto wav_abort_;

	if(skip > 0) {
		if(!FLAC__file_decoder_process_metadata(decoder)) {
			fprintf(stderr, "%s: ERROR while decoding metadata, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto wav_abort_;
		}
		if(stream_info.skip_count_too_high) {
			fprintf(stderr, "%s: ERROR trying to skip more samples than in stream\n", stream_info.inbasefilename);
			goto wav_abort_;
		}
		if(!FLAC__file_decoder_seek_absolute(decoder, skip)) {
			fprintf(stderr, "%s: ERROR seeking while skipping bytes, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto wav_abort_;
		}
		if(!FLAC__file_decoder_process_remaining_frames(decoder)) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR while decoding frames, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto wav_abort_;
		}
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_OK && FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_END_OF_FILE) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR during decoding, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto wav_abort_;
		}
	}
	else {
		if(!FLAC__file_decoder_process_whole_file(decoder)) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR while decoding data, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto wav_abort_;
		}
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_OK && FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_END_OF_FILE) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR during decoding, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto wav_abort_;
		}
	}

	if(decoder) {
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_UNINITIALIZED)
			md5_failure = !FLAC__file_decoder_finish(decoder);
		print_stats(&stream_info);
		FLAC__file_decoder_delete(decoder);
	}
	if(0 != stream_info.fout && stream_info.fout != stdout)
		fclose(stream_info.fout);
	if(analysis_mode)
		flac__analyze_finish(aopts);
	if(md5_failure) {
		fprintf(stderr, "\r%s: WARNING, MD5 signature mismatch\n", stream_info.inbasefilename);
	}
	else {
		if(verbose)
			fprintf(stderr, "\r%s: %s         \n", stream_info.inbasefilename, stream_info.test_only? "ok           ":analysis_mode?"done           ":"done");
	}
	return 0;
wav_abort_:
	if(decoder) {
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_UNINITIALIZED)
			FLAC__file_decoder_finish(decoder);
		FLAC__file_decoder_delete(decoder);
	}
	if(0 != stream_info.fout && stream_info.fout != stdout) {
		fclose(stream_info.fout);
		unlink(outfile);
	}
	if(analysis_mode)
		flac__analyze_finish(aopts);
	return 1;
}

int flac__decode_raw(const char *infile, const char *outfile, FLAC__bool analysis_mode, analysis_options aopts, FLAC__bool verbose, FLAC__uint64 skip, FLAC__bool is_big_endian, FLAC__bool is_unsigned_samples)
{
	FLAC__bool md5_failure = false;
	stream_info_struct stream_info;

	decoder = 0;
	stream_info.abort_flag = false;
	stream_info.analysis_mode = analysis_mode;
	stream_info.aopts = aopts;
	stream_info.test_only = (outfile == 0);
	stream_info.is_wave_out = false;
	stream_info.is_big_endian = is_big_endian;
	stream_info.is_unsigned_samples = is_unsigned_samples;
	stream_info.verbose = verbose;
	stream_info.skip = skip;
	stream_info.skip_count_too_high = false;
	stream_info.samples_processed = 0;
	stream_info.frame_counter = 0;
	stream_info.inbasefilename = flac__file_get_basename(infile);
	stream_info.fout = 0; /* initialized with an open file later if necessary */

	FLAC__ASSERT(!(stream_info.test_only && stream_info.analysis_mode));

	if(!stream_info.test_only) {
		if(0 == strcmp(outfile, "-")) {
			stream_info.fout = stdout;
		}
		else {
			if(0 == (stream_info.fout = fopen(outfile, "wb"))) {
				fprintf(stderr, "%s: ERROR: can't open output file %s\n", stream_info.inbasefilename, outfile);
				return 1;
			}
		}
	}

	if(analysis_mode)
		flac__analyze_init(aopts);

	if(!init(infile, &stream_info))
		goto raw_abort_;

	if(skip > 0) {
		if(!FLAC__file_decoder_process_metadata(decoder)) {
			fprintf(stderr, "%s: ERROR while decoding metadata, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto raw_abort_;
		}
		if(stream_info.skip_count_too_high) {
			fprintf(stderr, "%s: ERROR trying to skip more samples than in stream\n", stream_info.inbasefilename);
			goto raw_abort_;
		}
		if(!FLAC__file_decoder_seek_absolute(decoder, skip)) {
			fprintf(stderr, "%s: ERROR seeking while skipping bytes, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto raw_abort_;
		}
		if(!FLAC__file_decoder_process_remaining_frames(decoder)) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR while decoding frames, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto raw_abort_;
		}
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_OK && FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_END_OF_FILE) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR during decoding, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto raw_abort_;
		}
	}
	else {
		if(!FLAC__file_decoder_process_whole_file(decoder)) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR while decoding data, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto raw_abort_;
		}
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_OK && FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_END_OF_FILE) {
			if(verbose) fprintf(stderr, "\n");
			fprintf(stderr, "%s: ERROR during decoding, state=%d:%s\n", stream_info.inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
			goto raw_abort_;
		}
	}

	if(decoder) {
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_UNINITIALIZED)
			md5_failure = !FLAC__file_decoder_finish(decoder);
		print_stats(&stream_info);
		FLAC__file_decoder_delete(decoder);
	}
	if(0 != stream_info.fout && stream_info.fout != stdout)
		fclose(stream_info.fout);
	if(analysis_mode)
		flac__analyze_finish(aopts);
	if(md5_failure) {
		fprintf(stderr, "\r%s: WARNING, MD5 signature mismatch\n", stream_info.inbasefilename);
	}
	else {
		if(verbose)
			fprintf(stderr, "\r%s: %s         \n", stream_info.inbasefilename, stream_info.test_only? "ok           ":analysis_mode?"done           ":"done");
	}
	return 0;
raw_abort_:
	if(decoder) {
		if(FLAC__file_decoder_get_state(decoder) != FLAC__FILE_DECODER_UNINITIALIZED)
			FLAC__file_decoder_finish(decoder);
		FLAC__file_decoder_delete(decoder);
	}
	if(0 != stream_info.fout && stream_info.fout != stdout) {
		fclose(stream_info.fout);
		unlink(outfile);
	}
	if(analysis_mode)
		flac__analyze_finish(aopts);
	return 1;
}

FLAC__bool init(const char *infilename, stream_info_struct *stream_info)
{
	FLAC__uint32 test = 1;

	is_big_endian_host = (*((FLAC__byte*)(&test)))? false : true;

	decoder = FLAC__file_decoder_new();
	if(0 == decoder) {
		fprintf(stderr, "%s: ERROR creating the decoder instance\n", stream_info->inbasefilename);
		return false;
	}

	FLAC__file_decoder_set_md5_checking(decoder, true);
	FLAC__file_decoder_set_filename(decoder, infilename);
	FLAC__file_decoder_set_write_callback(decoder, write_callback);
	FLAC__file_decoder_set_metadata_callback(decoder, metadata_callback);
	FLAC__file_decoder_set_error_callback(decoder, error_callback);
	FLAC__file_decoder_set_client_data(decoder, stream_info);

	if(FLAC__file_decoder_init(decoder) != FLAC__FILE_DECODER_OK) {
		fprintf(stderr, "%s: ERROR initializing decoder, state=%d:%s\n", stream_info->inbasefilename, FLAC__file_decoder_get_state(decoder), FLAC__FileDecoderStateString[FLAC__file_decoder_get_state(decoder)]);
		return false;
	}

	return true;
}

FLAC__bool write_little_endian_uint16(FILE *f, FLAC__uint16 val)
{
	FLAC__byte *b = (FLAC__byte*)(&val);
	if(is_big_endian_host) {
		FLAC__byte tmp;
		tmp = b[1]; b[1] = b[0]; b[0] = tmp;
	}
	return fwrite(b, 1, 2, f) == 2;
}

FLAC__bool write_little_endian_uint32(FILE *f, FLAC__uint32 val)
{
	FLAC__byte *b = (FLAC__byte*)(&val);
	if(is_big_endian_host) {
		FLAC__byte tmp;
		tmp = b[3]; b[3] = b[0]; b[0] = tmp;
		tmp = b[2]; b[2] = b[1]; b[1] = tmp;
	}
	return fwrite(b, 1, 4, f) == 4;
}

FLAC__StreamDecoderWriteStatus write_callback(const FLAC__FileDecoder *decoder, const FLAC__Frame *frame, const FLAC__int32 *buffer[], void *client_data)
{
	stream_info_struct *stream_info = (stream_info_struct *)client_data;
	FILE *fout = stream_info->fout;
	unsigned bps = stream_info->bps, channels = stream_info->channels;
	FLAC__bool is_big_endian = (stream_info->is_wave_out? false : stream_info->is_big_endian);
	FLAC__bool is_unsigned_samples = (stream_info->is_wave_out? bps<=8 : stream_info->is_unsigned_samples);
	unsigned wide_samples = frame->header.blocksize, wide_sample, sample, channel, byte;
	static FLAC__int8 s8buffer[FLAC__MAX_BLOCK_SIZE * FLAC__MAX_CHANNELS * sizeof(FLAC__int32)]; /* WATCHOUT: can be up to 2 megs */
	/* WATCHOUT: we say 'sizeof(FLAC__int32)' above instead of '(FLAC__MAX_BITS_PER_SAMPLE+7)/8' because we have to use an array FLAC__int32 even for 24 bps */
	FLAC__uint8  *u8buffer  = (FLAC__uint8  *)s8buffer;
	FLAC__int16  *s16buffer = (FLAC__int16  *)s8buffer;
	FLAC__uint16 *u16buffer = (FLAC__uint16 *)s8buffer;
	FLAC__int32  *s32buffer = (FLAC__int32  *)s8buffer;
	FLAC__uint32 *u32buffer = (FLAC__uint32 *)s8buffer;

	(void)decoder;

	if(stream_info->abort_flag)
		return FLAC__STREAM_DECODER_WRITE_ABORT;

	stream_info->samples_processed += wide_samples;
	stream_info->frame_counter++;

	if(stream_info->verbose && !(stream_info->frame_counter & 0x7f))
		print_stats(stream_info);

	if(stream_info->analysis_mode) {
		flac__analyze_frame(frame, stream_info->frame_counter-1, stream_info->aopts, fout);
	}
	else if(!stream_info->test_only) {
		if(bps == 8) {
			if(is_unsigned_samples) {
				for(sample = wide_sample = 0; wide_sample < wide_samples; wide_sample++)
					for(channel = 0; channel < channels; channel++, sample++)
						u8buffer[sample] = (FLAC__uint8)(buffer[channel][wide_sample] + 0x80);
			}
			else {
				for(sample = wide_sample = 0; wide_sample < wide_samples; wide_sample++)
					for(channel = 0; channel < channels; channel++, sample++)
						s8buffer[sample] = (FLAC__int8)(buffer[channel][wide_sample]);
			}
			if(fwrite(u8buffer, 1, sample, fout) != sample)
				return FLAC__STREAM_DECODER_WRITE_ABORT;
		}
		else if(bps == 16) {
			if(is_unsigned_samples) {
				for(sample = wide_sample = 0; wide_sample < wide_samples; wide_sample++)
					for(channel = 0; channel < channels; channel++, sample++)
						u16buffer[sample] = (FLAC__uint16)(buffer[channel][wide_sample] + 0x8000);
			}
			else {
				for(sample = wide_sample = 0; wide_sample < wide_samples; wide_sample++)
					for(channel = 0; channel < channels; channel++, sample++)
						s16buffer[sample] = (FLAC__int16)(buffer[channel][wide_sample]);
			}
			if(is_big_endian != is_big_endian_host) {
				unsigned char tmp;
				const unsigned bytes = sample * 2;
				for(byte = 0; byte < bytes; byte += 2) {
					tmp = u8buffer[byte];
					u8buffer[byte] = u8buffer[byte+1];
					u8buffer[byte+1] = tmp;
				}
			}
			if(fwrite(u16buffer, 2, sample, fout) != sample)
				return FLAC__STREAM_DECODER_WRITE_ABORT;
		}
		else if(bps == 24) {
			if(is_unsigned_samples) {
				for(sample = wide_sample = 0; wide_sample < wide_samples; wide_sample++)
					for(channel = 0; channel < channels; channel++, sample++)
						u32buffer[sample] = buffer[channel][wide_sample] + 0x800000;
			}
			else {
				for(sample = wide_sample = 0; wide_sample < wide_samples; wide_sample++)
					for(channel = 0; channel < channels; channel++, sample++)
						s32buffer[sample] = buffer[channel][wide_sample];
			}
			if(is_big_endian != is_big_endian_host) {
				unsigned char tmp;
				const unsigned bytes = sample * 4;
				for(byte = 0; byte < bytes; byte += 4) {
					tmp = u8buffer[byte];
					u8buffer[byte] = u8buffer[byte+3];
					u8buffer[byte+3] = tmp;
					tmp = u8buffer[byte+1];
					u8buffer[byte+1] = u8buffer[byte+2];
					u8buffer[byte+2] = tmp;
				}
			}
			if(is_big_endian) {
				unsigned lbyte;
				const unsigned bytes = sample * 4;
				for(lbyte = byte = 0; byte < bytes; ) {
					byte++;
					u8buffer[lbyte++] = u8buffer[byte++];
					u8buffer[lbyte++] = u8buffer[byte++];
					u8buffer[lbyte++] = u8buffer[byte++];
				}
			}
			else {
				unsigned lbyte;
				const unsigned bytes = sample * 4;
				for(lbyte = byte = 0; byte < bytes; ) {
					u8buffer[lbyte++] = u8buffer[byte++];
					u8buffer[lbyte++] = u8buffer[byte++];
					u8buffer[lbyte++] = u8buffer[byte++];
					byte++;
				}
			}
			if(fwrite(u8buffer, 3, sample, fout) != sample)
				return FLAC__STREAM_DECODER_WRITE_ABORT;
		}
		else {
			FLAC__ASSERT(0);
		}
	}
	return FLAC__STREAM_DECODER_WRITE_CONTINUE;
}

void metadata_callback(const FLAC__FileDecoder *decoder, const FLAC__StreamMetaData *metadata, void *client_data)
{
	stream_info_struct *stream_info = (stream_info_struct *)client_data;
	(void)decoder;
	if(metadata->type == FLAC__METADATA_TYPE_STREAMINFO) {
		/* remember, metadata->data.stream_info.total_samples can be 0, meaning 'unknown' */
		if(metadata->data.stream_info.total_samples > 0 && stream_info->skip >= metadata->data.stream_info.total_samples) {
			stream_info->total_samples = 0;
			stream_info->skip_count_too_high = true;
		}
		else
			stream_info->total_samples = metadata->data.stream_info.total_samples - stream_info->skip;
		stream_info->bps = metadata->data.stream_info.bits_per_sample;
		stream_info->channels = metadata->data.stream_info.channels;
		stream_info->sample_rate = metadata->data.stream_info.sample_rate;

		if(stream_info->bps != 8 && stream_info->bps != 16 && stream_info->bps != 24) {
			fprintf(stderr, "%s: ERROR: bits per sample is not 8/16/24\n", stream_info->inbasefilename);
			stream_info->abort_flag = true;
			return;
		}

		/* write the WAVE headers if necessary */
		if(!stream_info->analysis_mode && !stream_info->test_only && stream_info->is_wave_out) {
			FLAC__uint64 data_size = stream_info->total_samples * stream_info->channels * ((stream_info->bps+7)/8);
			if(data_size >= 0xFFFFFFDC) {
				fprintf(stderr, "%s: ERROR: stream is too big to fit in a single WAVE file chunk\n", stream_info->inbasefilename);
				stream_info->abort_flag = true;
				return;
			}
			if(fwrite("RIFF", 1, 4, stream_info->fout) != 4) stream_info->abort_flag = true;
			if(!write_little_endian_uint32(stream_info->fout, (FLAC__uint32)(data_size+36))) stream_info->abort_flag = true; /* filesize-8 */
			if(fwrite("WAVEfmt ", 1, 8, stream_info->fout) != 8) stream_info->abort_flag = true;
			if(fwrite("\020\000\000\000", 1, 4, stream_info->fout) != 4) stream_info->abort_flag = true; /* chunk size = 16 */
			if(fwrite("\001\000", 1, 2, stream_info->fout) != 2) stream_info->abort_flag = true; /* compression code == 1 */
			if(!write_little_endian_uint16(stream_info->fout, (FLAC__uint16)(stream_info->channels))) stream_info->abort_flag = true;
			if(!write_little_endian_uint32(stream_info->fout, stream_info->sample_rate)) stream_info->abort_flag = true;
			if(!write_little_endian_uint32(stream_info->fout, stream_info->sample_rate * stream_info->channels * ((stream_info->bps+7) / 8))) stream_info->abort_flag = true; /* @@@ or is it (sample_rate*channels*bps) / 8 ??? */
			if(!write_little_endian_uint16(stream_info->fout, (FLAC__uint16)(stream_info->channels * ((stream_info->bps+7) / 8)))) stream_info->abort_flag = true; /* block align */
			if(!write_little_endian_uint16(stream_info->fout, (FLAC__uint16)(stream_info->bps))) stream_info->abort_flag = true; /* bits per sample */
			if(fwrite("data", 1, 4, stream_info->fout) != 4) stream_info->abort_flag = true;
			if(!write_little_endian_uint32(stream_info->fout, (FLAC__uint32)data_size)) stream_info->abort_flag = true; /* data size */
		}
	}
}

void error_callback(const FLAC__FileDecoder *decoder, FLAC__StreamDecoderErrorStatus status, void *client_data)
{
	stream_info_struct *stream_info = (stream_info_struct *)client_data;
	(void)decoder;
	fprintf(stderr, "%s: *** Got error code %d:%s\n", stream_info->inbasefilename, status, FLAC__StreamDecoderErrorStatusString[status]);
	stream_info->abort_flag = true;
}

void print_stats(const stream_info_struct *stream_info)
{
	if(stream_info->verbose) {
#ifdef _MSC_VER
		/* with VC++ you have to spoon feed it the casting */
		const double progress = (double)(FLAC__int64)stream_info->samples_processed / (double)(FLAC__int64)stream_info->total_samples * 100.0;
#else
		const double progress = (double)stream_info->samples_processed / (double)stream_info->total_samples * 100.0;
#endif
		if(stream_info->total_samples > 0) {
			fprintf(stderr, "\r%s: %s%u%% complete",
				stream_info->inbasefilename,
				stream_info->test_only? "testing, " : stream_info->analysis_mode? "analyzing, " : "",
				(unsigned)floor(progress + 0.5)
			);
		}
		else {
			fprintf(stderr, "\r%s: %s %u samples",
				stream_info->inbasefilename,
				stream_info->test_only? "tested" : stream_info->analysis_mode? "analyzed" : "wrote",
				(unsigned)stream_info->samples_processed
			);
		}
	}
}
