/* evilwm - Minimalist Window Manager for X
 * Copyright (C) 1999-2002 Ciaran Anscomb <evilwm@6809.org.uk>
 * see README for license and other details. */

#include <stdio.h>
#include "evilwm.h"

#ifdef VWM
# define HAS_HIDE 1
#endif
#ifdef VDESK
# define HAS_HIDE 1
#endif

void draw_outline(Client *c) {
	char buf[20];
	int width_inc = 1, height_inc = 1;

	XDrawRectangle(dpy, root, invert_gc,
		c->x - c->border, c->y - c->border,
		c->width + c->border, c->height + c->border);

	if (c->size->flags & PResizeInc) {
		width_inc = c->size->width_inc;
		height_inc = c->size->height_inc;
	}
	snprintf(buf, sizeof(buf), "%dx%d+%d+%d", c->width/width_inc,
		c->height/height_inc, c->x, c->y);
	XDrawString(dpy, root, invert_gc,
		c->x + c->width - XTextWidth(font, buf, strlen(buf)) - SPACE,
		c->y + c->height - SPACE,
		buf, strlen(buf));
}

#ifdef MOUSE
void get_mouse_position(int *x, int *y) {
	Window dw1, dw2;
	int t1, t2;
	unsigned int t3;

	XQueryPointer(dpy, root, &dw1, &dw2, x, y, &t1, &t2, &t3);
}
#endif

void recalculate_sweep(Client *c, int x1, int y1, int x2, int y2) {
	int basex, basey;

	c->width = abs(x1 - x2);
	c->height = abs(y1 - y2);

	if (c->size->flags & PResizeInc) {
		basex = (c->size->flags & PBaseSize) ? c->size->base_width :
			(c->size->flags & PMinSize) ? c->size->min_width : 0;
		basey = (c->size->flags & PBaseSize) ? c->size->base_height :
			(c->size->flags & PMinSize) ? c->size->min_height : 0;
		c->width -= (c->width - basex) % c->size->width_inc;
		c->height -= (c->height - basey) % c->size->height_inc;
	}

	if (c->size->flags & PMinSize) {
		if (c->width < c->size->min_width) c->width = c->size->min_width;
		if (c->height < c->size->min_height) c->height = c->size->min_height;
	}

	if (c->size->flags & PMaxSize) {
		if (c->width > c->size->max_width) c->width = c->size->max_width;
		if (c->height > c->size->max_height) c->height = c->size->max_height;
	}

	c->x = (x1 <= x2) ? x1 : x1 - c->width;
	c->y = (y1 <= y2) ? y1 : y1 - c->height;
}

#ifdef MOUSE
void sweep(Client *c) {
	XEvent ev;
	int old_cx = c->x;
	int old_cy = c->y;

	if (!grab(root, MouseMask, resize_curs)) return;
	XGrabServer(dpy);

	draw_outline(c);

	setmouse(c->window, c->width, c->height);
	for (;;) {
		XMaskEvent(dpy, MouseMask, &ev);
		switch (ev.type) {
			case MotionNotify:
				draw_outline(c); /* clear */
				recalculate_sweep(c, old_cx, old_cy, ev.xmotion.x, ev.xmotion.y);
				draw_outline(c);
				break;
			case ButtonRelease:
				draw_outline(c); /* clear */
				XUngrabServer(dpy);
				XUngrabPointer(dpy, CurrentTime);
				return;
		}
	}
}
#endif

void show_info(Client *c, KeySym key) {
	XEvent ev;

	XGrabServer(dpy);
	draw_outline(c);
	do {
		XMaskEvent(dpy, KeyReleaseMask, &ev);
	} while (XKeycodeToKeysym(dpy,ev.xkey.keycode,0) != key);
	draw_outline(c);
	XUngrabServer(dpy);
}

#ifdef MOUSE
void drag(Client *c) {
	XEvent ev;
	int x1, y1;
	int old_cx = c->x;
	int old_cy = c->y;
	int reallymove = 0;

	if (!grab(root, MouseMask, move_curs)) return;
	get_mouse_position(&x1, &y1);
#ifndef SOLIDDRAG
	XGrabServer(dpy);
	draw_outline(c);
#endif 
	for (;;) {
		XMaskEvent(dpy, MouseMask, &ev);
		switch (ev.type) {
			case MotionNotify:
#ifndef SOLIDDRAG
				draw_outline(c); /* clear */
#endif
				c->x = old_cx + (ev.xmotion.x - x1);
				c->y = old_cy + (ev.xmotion.y - y1);
#ifndef SOLIDDRAG
				draw_outline(c);
#else
				if ((abs(c->x - old_cx)>5) || (abs(c->y - old_cy)>5))
					reallymove = 1;
				if (reallymove) {
					XMoveWindow(dpy, c->parent, c->x - c->border,
						c->y - c->border);
					send_config(c);
				}
#endif 
				break;
			case ButtonRelease:
#ifndef SOLIDDRAG
				draw_outline(c); /* clear */
				XUngrabServer(dpy);
#endif
				XUngrabPointer(dpy, CurrentTime);
				if (!reallymove) {
					c->x = old_cx;
					c->y = old_cy;
				}
				return;
			default:
				break;
		}
	}
}
#endif /* def MOUSE */

void move(Client *c, int set) {
	if (c) {
		XRaiseWindow(dpy, c->parent);
#ifdef MOUSE
		if (!set)
			drag(c);
#endif
		XMoveWindow(dpy, c->parent, c->x - c->border, c->y - c->border);
		send_config(c);
	}
}

void resize(Client *c, int set) {
	if (c) {
		XRaiseWindow(dpy, c->parent);
#ifdef MOUSE
		if (!set)
			sweep(c);
#endif
		XMoveResizeWindow(dpy, c->parent, c->x - c->border,
				c->y - c->border, c->width + (c->border*2),
				c->height + (c->border*2));
		XMoveResizeWindow(dpy, c->window, c->border, c->border,
				c->width, c->height);
		send_config(c);
	}
}

void maximise_horiz(Client *c) {
#ifdef DEBUG
	fprintf(stderr, "SCREEN: maximise_horiz()\n");
#endif
	if (c->oldw) {
		c->x = c->oldx;
		c->width = c->oldw;
		c->oldw = 0;
	} else {
		c->oldx = c->x;
		c->oldw = c->width;
		recalculate_sweep(c, 0, c->y, DisplayWidth(dpy, screen),
				c->y + c->height);
	}
}

void maximise_vert(Client *c) {
#ifdef DEBUG
	fprintf(stderr, "SCREEN: maximise_vert()\n");
#endif
	if (c->oldh) {
		c->y = c->oldy;
		c->height = c->oldh;
		c->oldh = 0;
	} else {
		c->oldy = c->y;
		c->oldh = c->height;
		recalculate_sweep(c, c->x, 0, c->x + c->width,
				DisplayHeight(dpy, screen));
	}
}

#ifdef HAS_HIDE
void hide(Client *c) {
	if (c) {
		c->ignore_unmap = 2;
#ifdef XDEBUG
		fprintf(stderr, "screen:XUnmapWindow(parent); ");
#endif
		XUnmapWindow(dpy, c->parent);
#ifdef XDEBUG
		fprintf(stderr, "screen:XUnmapWindow(window); ");
#endif
		XUnmapWindow(dpy, c->window);
		set_wm_state(c, IconicState);
	}
}
#endif

void unhide(Client *c, int raise) {
	c->ignore_unmap = 0;
#ifdef XDEBUG
		fprintf(stderr, "screen:XMapWindow(window); ");
#endif
	XMapWindow(dpy, c->window);
#ifdef XDEBUG
		fprintf(stderr, "screen:XMapRaised(parent); ");
#endif
	raise ? XMapRaised(dpy, c->parent) : XMapWindow(dpy, c->parent);
	set_wm_state(c, NormalState);
}

void next(Client *c) {
	Client *newc = c;

	if (!newc)
		newc = head_client;
	if (newc) {
#ifdef VWM
		int wdesk;
#endif
#ifdef VDESK_BOTH
		int vdesk = GET_VDESK;
#endif
#ifdef VWM
		do {
#endif
			if (newc->next) {
				newc = newc->next;
			} else {
				newc = head_client;
			}
#ifdef VWM
			wdesk = RD_DESK(newc);
		} while (wdesk != vdesk && wdesk != STICKY && newc != c);
#endif
	}
	if (newc) {
		unhide(newc, RAISE);
		setmouse(newc->window, newc->width + newc->border - 1,
			newc->height + newc->border - 1);
	}
}

#ifdef VWM
void switch_vdesk(int v) {
	Client *c;
	int wdesk;
#ifdef VDESK_BOTH
	int vdesk = GET_VDESK;
#endif
	int warped = 0;

	if (v == vdesk)
		return;
#ifdef DEBUG
	fprintf(stderr, "\nscreen:switch_vdesk(%d); ", v);
#endif
	for (c = head_client; c; c = c->next) {
		wdesk = RD_DESK(c);
		if (wdesk == vdesk) {
			hide(c);
		} else if (wdesk == v) {
			unhide(c, NO_RAISE);
			if (!warped) {
				setmouse(c->window, c->width, c->height);
				warped = 1;
			}
		}
	}
	SET_VDESK(v);
}
#endif /* def VWM */
